/*************************************************************************/
/*                                                                       */
/*  Copyright (c) 1994 Stanford University                               */
/*                                                                       */
/*  All rights reserved.                                                 */
/*                                                                       */
/*  Permission is given to use, copy, and modify this software for any   */
/*  non-commercial purpose as long as this copyright notice is not       */
/*  removed.  All other uses, including redistribution in whole or in    */
/*  part, are forbidden without prior written permission.                */
/*                                                                       */
/*  This software is provided with absolutely no warranty and no         */
/*  support.                                                             */
/*                                                                       */
/* --------------------------------------------------------------------- */
/*                                                                       */
/*  Modifications of the original Barnes-Hut code (as taken from         */
/*  Stanford's SPLASH-2 distribution) to allow use on Alewife and        */
/*  with CRL are copyright:                                              */
/*                                                                       */
/*  Copyright (C) 1995 Massachusetts Institute of Technology             */
/*                                                                       */
/*************************************************************************/

#include "code.h"
#include <math.h>

/* alewife cpp seems to be leaving "unix" #define-d, so #undef it
 */
#if defined(ALEWIFE) && defined(unix)
#undef unix
#endif

#if defined(TCPUNIX) || defined(NULLCRL)
#include <sys/time.h>
#endif

#define MULT    (1103515245)
#define ADD     (12345)
#define MASK    (0x7FFFFFFF)
#define TWOTO31 (2147483648.0)

static int A = 1;
static int B = 0;
static int randx = 1;
static int lastrand;		/* the last random number */


/* return a random double in [0, 1.0)
 */
double prand(void)
{
  lastrand = randx;
  randx = (A*randx+B) & MASK;

  return ((double)lastrand/TWOTO31);
}


/* generate floating-point random number
 */
double xrand(double lo, double hi)
{
  return (lo + (hi - lo) * prand());
}


void pranset(int seed)
{
  A = 1;
  B = 0;
  randx = (A*seed+B) & MASK;
  A = (MULT * A) & MASK;
  B = (MULT*B + ADD) & MASK;
}


/* scream and die quickly
 */

void error(char *msg, char *a1, char *a2, char *a3, char *a4)
{
#if defined(CM5)
  CMMD_error(msg, a1, a2, a3, a4);
#else
  fprintf(stderr, msg, a1, a2, a3, a4);
  abort();
#endif
}


#if defined(ALEWIFE) || defined(TCPUNIX) || defined(NULLCRL)
static double _timer_start;
#endif

void timer_clear_and_start(void)
{
#if defined(CM5)
  CMMD_node_timer_clear(0);
  CMMD_sync_with_nodes();
  CMMD_node_timer_start(0);
#elif defined(ALEWIFE)
#if defined(USE_CRL)
  crl_prof_reset();
#endif
  mp_spin_barrier();
  _timer_start = get_time();
#else
  struct timeval tp;
  gettimeofday(&tp, NULL);
  _timer_start = tp.tv_sec + (tp.tv_usec * 1e-6);
#endif
}

void timer_stop_and_print(void)
{
  int    self;
  int    nprocs;
  double t1;
  double t2;
  double avg;
  double std;

#if defined(CM5)
  CMMD_node_timer_stop(0);
  t1 = CMMD_node_timer_elapsed(0); /* seconds */
#elif defined(ALEWIFE)
  t1 = get_time();
  t1 = (t1 - _timer_start) * 1e-6; /* Mcycles */
#else
  struct timeval tp;
  gettimeofday(&tp, NULL);
  t1  = tp.tv_sec + (tp.tv_usec * 1e-6);
  t1 -= _timer_start;		/* seconds */
#endif

#if defined(USE_CRL)
  self   = crl_self_addr;
  nprocs = crl_num_nodes;
#else
  self   = MY_PID;
  nprocs = NPROCS; 
#endif

  t2 = t1 * t1;
#if defined(USE_CRL)
  t1 = rgn_reduce_dadd(t1);
  t2 = rgn_reduce_dadd(t2);
#else
  t1 = shm_reduce_dadd(t1);
  t2 = shm_reduce_dadd(t2);
#endif

  avg = t1 / nprocs;
  std = sqrt((t2 / nprocs) - (avg * avg));

  if (self == 0)
  {
    printf(" %.3f +- %.3f  ", avg, std);
    printf(" %u %.6f %.6f\n", nprocs, t1, t2);
  }

#if defined(ALEWIFE) && defined(USE_CRL)
  crl_prof_print();
#endif
}
