/*************************************************************************/
/*                                                                       */
/*  Copyright (c) 1994 Stanford University                               */
/*                                                                       */
/*  All rights reserved.                                                 */
/*                                                                       */
/*  Permission is given to use, copy, and modify this software for any   */
/*  non-commercial purpose as long as this copyright notice is not       */
/*  removed.  All other uses, including redistribution in whole or in    */
/*  part, are forbidden without prior written permission.                */
/*                                                                       */
/*  This software is provided with absolutely no warranty and no         */
/*  support.                                                             */
/*                                                                       */
/* --------------------------------------------------------------------- */
/*                                                                       */
/*  Modifications of the original Barnes-Hut code (as taken from         */
/*  Stanford's SPLASH-2 distribution) to allow use on Alewife and        */
/*  with CRL are copyright:                                              */
/*                                                                       */
/*  Copyright (C) 1995 Massachusetts Institute of Technology             */
/*                                                                       */
/*************************************************************************/

/*
 * GRAV.C: 
 */

#include "code.h"
#include <math.h>

void hackwalk(unsigned);
void walksub(nodeptr, real, unsigned);
bool subdivp(nodeptr, real, unsigned);


/* evaluate grav field at a given particle
 *
 * (p is a mapped region with a write operation in progress; write gets
 * terminated, but new write gets started before returning)
 */

void hackgrav(bodyptr p, unsigned ProcessId)
{
  Local.pskip = p;

  SETV(Local.pos0, Pos(p));
#if defined(USE_CRL)
  rgn_end_write(p);
#endif

  Local.phi0 = 0.0;
  CLRV(Local.acc0);
  Local.myn2bterm = 0;
  Local.mynbcterm = 0;
  Local.skipself = FALSE;
  hackwalk(ProcessId);

#if defined(USE_CRL)
  rgn_start_write(p);
#endif
  Phi(p) = Local.phi0;
  SETV(Acc(p), Local.acc0);
#ifdef QUADPOLE
  Cost(p) = Local.myn2bterm + NDIM * Local.mynbcterm;
#else
  Cost(p) = Local.myn2bterm + Local.mynbcterm;
#endif
}


/* compute a single body-body or body-cell interaction
 *
 * (p is a mapped region with a read operation in progress)
 */

void gravsub(nodeptr p, unsigned ProcessId)
{
  real   drabs, phii, mor3;
  vector ai;
#ifdef QUADPOLE
  vector quaddr;
  real   dr5inv, phiquad, drquaddr;
#endif

  if (p != Local.pmem)
  {
    SUBV(Local.dr, Pos(p), Local.pos0);
    DOTVP(Local.drsq, Local.dr, Local.dr);
  }

  Local.drsq += epssq;
  drabs = sqrt((double) Local.drsq);
  phii = Mass(p) / drabs;
  Local.phi0 -= phii;
  mor3 = phii / Local.drsq;
  MULVS(ai, Local.dr, mor3);
  ADDV(Local.acc0, Local.acc0, ai); 

  if (Type(p) != BODY)
  {				/* a body-cell/leaf interaction */
    Local.mynbcterm += 1;
#ifdef QUADPOLE
    dr5inv = 1.0/(Local.drsq * Local.drsq * drabs);
    MULMV(quaddr, Quad(p), Local.dr);
    DOTVP(drquaddr, Local.dr, quaddr);
    phiquad = -0.5 * dr5inv * drquaddr;
    Local.phi0 += phiquad;
    phiquad = 5.0 * phiquad / Local.drsq;
    MULVS(ai, Local.dr, phiquad);
    SUBV(Local.acc0, Local.acc0, ai);
    MULVS(quaddr, quaddr, dr5inv);   
    SUBV(Local.acc0, Local.acc0, quaddr);
#endif
  }
  else
  {				/* a body-body interaction */
    Local.myn2bterm += 1;
  }
}


/* walk the tree opening cells too close to a given point
 */

void hackwalk(unsigned ProcessId)
{
  walksub((nodeptr) g_root, rsize * rsize, ProcessId);
}


/* recursive routine to do hackwalk operation
 *
 * (n is a mapped region with no operations in progress) 
 */

void walksub(nodeptr n, real dsq, unsigned ProcessId)
{
  int     i;
#if defined(USE_CRL)
  rid_t   nn_tmp;
#else
  nodeptr nn_tmp;
#endif
  nodeptr nn;
  leafptr l;
  bodyptr p;

#if defined(USE_CRL)
  rgn_start_read(n);
#endif

  if (subdivp(n, dsq, ProcessId))
  {
    if (Type(n) == CELL)
    {
      for (i=0; i<NSUB; i++)
      {
	nn_tmp = Subp(n)[i];
	if (nn_tmp != 0)
	{
#if defined(USE_CRL)
	  nn = (nodeptr) rgn_map(nn_tmp);
#else
	  nn = nn_tmp;
#endif
	  walksub(nn, dsq / 4.0, ProcessId);
#if defined(USE_CRL)
	  rgn_unmap(nn);
#endif
	}
      }
    }
    else
    {
      l = (leafptr) n;

      for (i=0; i<l->num_bodies; i++)
      {
#if defined(USE_CRL)
	p = (bodyptr) rgn_map(Bodyp(l)[i]);
#else
	p = Bodyp(l)[i];
#endif

	if (p != Local.pskip)
	{
#if defined(USE_CRL)
	  rgn_start_read(p);
#endif
	  gravsub((nodeptr) p, ProcessId);
#if defined(USE_CRL)
	  rgn_end_read(p);
#endif
	}
	else
	{
	  Local.skipself = TRUE;
	}

#if defined(USE_CRL)
	rgn_unmap(p);
#endif
      }
    }
  }
  else
  {
    gravsub(n, ProcessId);
  }

#if defined(USE_CRL)
  rgn_end_read(n);
#endif
}


/* SUBDIVP: decide if a node should be opened
 * side effects: sets pmem, dr, and drsq
 *
 * (p is a mapped region with a read operation in progress)
 */

bool subdivp(nodeptr p, real dsq, unsigned ProcessId)
{
  SUBV(Local.dr, Pos(p), Local.pos0);
  DOTVP(Local.drsq, Local.dr, Local.dr);
  Local.pmem = p;
  return (tolsq * Local.drsq < dsq);
}
