/*************************************************************************/
/*                                                                       */
/*  Copyright (c) 1994 Stanford University                               */
/*                                                                       */
/*  All rights reserved.                                                 */
/*                                                                       */
/*  Permission is given to use, copy, and modify this software for any   */
/*  non-commercial purpose as long as this copyright notice is not       */
/*  removed.  All other uses, including redistribution in whole or in    */
/*  part, are forbidden without prior written permission.                */
/*                                                                       */
/*  This software is provided with absolutely no warranty and no         */
/*  support.                                                             */
/*                                                                       */
/* --------------------------------------------------------------------- */
/*                                                                       */
/*  Modifications of the original Barnes-Hut code (as taken from         */
/*  Stanford's SPLASH-2 distribution) to allow use on Alewife and        */
/*  with CRL are copyright:                                              */
/*                                                                       */
/*  Copyright (C) 1995 Massachusetts Institute of Technology             */
/*                                                                       */
/*************************************************************************/

#ifndef _DEFS_H_
#define _DEFS_H_

#if defined(CM5)
#include <cm/cmmd.h>
#elif defined(ALEWIFE)
#include <parallel.h>
#endif

#if !defined(USE_CRL)
/* this is to get beng's funky reactive locks */
#include <synch/locks/reactive.h>
#endif

#if defined(CM5)
#define assert(x)                                  \
 do {                                              \
  if (!(x))                                        \
    CMMD_error("pn %d: failed assertion, %s:%d\n", \
	       crl_self_addr, __FILE__, __LINE__); \
 } while (0)
#else
#include <assert.h>
#endif

#if defined(USE_CRL)
#include "crl.h"
#endif

#include "stdinc.h"
#include "vectmath.h"

#define MAX_PROC            (128)
#define MAX_BODIES_PER_LEAF (10)

#define NSUB (1 << NDIM)	/* subcells per cell */

/* The more complicated 3D case
 */
#define NUM_DIRECTIONS (32)

#define BRC_FUC  (0)
#define BRC_FRA  (1)
#define BRA_FDA  (2)
#define BRA_FRC  (3)
#define BLC_FDC  (4)
#define BLC_FLA  (5)
#define BLA_FUA  (6)
#define BLA_FLC  (7)
#define BUC_FUA  (8)
#define BUC_FLC  (9)
#define BUA_FUC (10)
#define BUA_FRA (11)
#define BDC_FDA (12)
#define BDC_FRC (13)
#define BDA_FDC (14)
#define BDA_FLA (15)

#define FRC_BUC (16)
#define FRC_BRA (17)
#define FRA_BDA (18)
#define FRA_BRC (19)
#define FLC_BDC (20)
#define FLC_BLA (21)
#define FLA_BUA (22)
#define FLA_BLC (23)
#define FUC_BUA (24)
#define FUC_BLC (25)
#define FUA_BUC (26)
#define FUA_BRA (27)
#define FDC_BDA (28)
#define FDC_BRC (29)
#define FDA_BDC (30)
#define FDA_BLA (31)

extern int Child_Sequence[NUM_DIRECTIONS][NSUB];
extern int Direction_Sequence[NUM_DIRECTIONS][NSUB];


/*
 * BODY and CELL data structures are used to represent the tree:
 *
 *         +-----------------------------------------------------------+
 * root--> | CELL: mass, pos, cost, quad, /, o, /, /, /, /, o, /, done |
 *         +---------------------------------|--------------|----------+
 *                                           |              |
 *    +--------------------------------------+              |
 *    |                                                     |
 *    |    +--------------------------------------+         |
 *    +--> | BODY: mass, pos, cost, vel, acc, phi |         |
 *         +--------------------------------------+         |
 *                                                          |
 *    +-----------------------------------------------------+
 *    |
 *    |    +-----------------------------------------------------------+
 *    +--> | CELL: mass, pos, cost, quad, o, /, /, o, /, /, o, /, done |
 *         +------------------------------|--------|--------|----------+
 *                                       etc      etc      etc
 */


/*
 * NODE: data common to BODY and CELL structures.
 */

typedef struct _node
{
  short  type;			/* code for node type: body or cell */
  real   mass;			/* total mass of node */
  vector pos;			/* position of node */
  int    cost;			/* number of interactions computed */
  int    level;
#if defined(USE_CRL)
  rid_t  parent;		/* ptr to parent of this node in tree */
#else
  struct _node *parent;
#endif
  int    child_num;		/* Index that this node should be put
				   at in parent cell */
} node;

typedef node *nodeptr;

#define Type(x)     (((nodeptr) (x))->type)
#define Mass(x)     (((nodeptr) (x))->mass)
#define Pos(x)      (((nodeptr) (x))->pos)
#define Cost(x)     (((nodeptr) (x))->cost)
#define Level(x)    (((nodeptr) (x))->level)
#define Parent(x)   (((nodeptr) (x))->parent)
#define ChildNum(x) (((nodeptr) (x))->child_num)


/*
 * BODY: data structure used to represent particles.
 */

typedef struct _body *bodyptr;
typedef struct _leaf *leafptr;
typedef struct _cell *cellptr;

#define BODY (01)		/* type code for bodies */

typedef struct _body
{
  short  type;
  real   mass;			/* mass of body */
  vector pos;			/* position of body */
  int    cost;			/* number of interactions computed */
  int    level;
#if defined(USE_CRL)
  rid_t  parent;		
#else
  leafptr parent;
#endif
  int    child_num;		/* Index that this node should be put */
  vector vel;			/* velocity of body */
  vector acc;			/* acceleration of body */
  real   phi;			/* potential at body */
} body;

#define Vel(x) (((bodyptr) (x))->vel)
#define Acc(x) (((bodyptr) (x))->acc)
#define Phi(x) (((bodyptr) (x))->phi)


/*
 * CELL: structure used to represent internal nodes of tree.
 */

#define CELL (02)		/* type code for cells */

typedef struct _cell
{
  short    type;
  real     mass;		/* total mass of cell */
  vector   pos;			/* cm. position of cell */
  int      cost;		/* number of interactions computed */
  int      level;
#if defined(USE_CRL)
  rid_t    parent;		
#else
  cellptr  parent;
#endif
  int      child_num;		/* Index that this node should be put */
#ifdef QUADPOLE
  matrix  quad;			/* quad. moment of cell */
#endif
  volatile int done;		/* flag to tell when the c.of.m is ready */
#if defined(USE_CRL)
  rid_t   subp[NSUB];		/* descendents of cell */
#else
  nodeptr subp[NSUB];
  lock_t  cell_lock;		/* lock for cell */
#endif
} cell;

#define Subp(x) (((cellptr) (x))->subp)


/*
 * LEAF: structure used to represent leaf nodes of tree.
 */

#define LEAF (03)		/* type code for leaves */

typedef struct _leaf
{
  short    type;
  real     mass;		/* total mass of leaf */
  vector   pos;			/* cm. position of leaf */
  int      cost;		/* number of interactions computed */
  int      level;
#if defined(USE_CRL)
  rid_t    parent;		
#else
  cellptr  parent;
#endif
  int      child_num;		/* Index that this node should be put */
#ifdef QUADPOLE
  matrix   quad;		/* quad. moment of leaf */
#endif
  volatile int done;		/* flag to tell when the c.of.m is ready */
  unsigned num_bodies;
#if defined(USE_CRL)
  rid_t    bodyp[MAX_BODIES_PER_LEAF]; /* bodies of leaf */
#else
  bodyptr  bodyp[MAX_BODIES_PER_LEAF];
#endif
} leaf;

#define Bodyp(x)  (((leafptr) (x))->bodyp)

#ifdef QUADPOLE
#define Quad(x) (((cellptr) (x))->quad)
#endif
#define Done(x) (((cellptr) (x))->done)

/*
 * Integerized coordinates: used to mantain body-tree.
 */

#define MAXLEVEL (8*sizeof(int)-2)
#define IMAX  (1 << MAXLEVEL)    /* highest bit of int coord */

#endif
