/*************************************************************************/
/*                                                                       */
/*  Copyright (c) 1994 Stanford University                               */
/*                                                                       */
/*  All rights reserved.                                                 */
/*                                                                       */
/*  Permission is given to use, copy, and modify this software for any   */
/*  non-commercial purpose as long as this copyright notice is not       */
/*  removed.  All other uses, including redistribution in whole or in    */
/*  part, are forbidden without prior written permission.                */
/*                                                                       */
/*  This software is provided with absolutely no warranty and no         */
/*  support.                                                             */
/*                                                                       */
/* --------------------------------------------------------------------- */
/*                                                                       */
/*  Modifications of the original Barnes-Hut code (as taken from         */
/*  Stanford's SPLASH-2 distribution) to allow use on Alewife and        */
/*  with CRL are copyright:                                              */
/*                                                                       */
/*  Copyright (C) 1995 Massachusetts Institute of Technology             */
/*                                                                       */
/*************************************************************************/

/*
 * CODE_IO.C: 
 */

#include "code.h"

void out_int(stream, int);
void out_real(stream, real);
void out_vector(stream, vector);
void diagnostics(unsigned);


/* initialize output
 */

void initoutput(void)
{
  int nprocs;

#if defined(USE_CRL)
  nprocs = crl_num_nodes;
#else
  nprocs = NPROCS;
#endif

  printf("%10s%10s%10s%10s%10s%10s%10s%10s\n",
	 "nbody", "dtime", "eps", "tol", "dtout", "tstop","fcells","NPROC");
  printf("%10d%10.5f%10.4f%10.2f%10.3f%10.3f%10.2f%10d\n\n",
	 nbody, dtime, eps, tol, dtout, tstop, fcells, nprocs);
}


/* compute diagnostics and output data
 */

void output(unsigned ProcessId)
{
  int    nttot, nbavg, ncavg, k;
  vector tempv1, tempv2;

  if ((Local.tout - 0.01 * dtime) <= Local.tnow)
    Local.tout += dtout;

  diagnostics(ProcessId);

  if (Local.mymtot != 0)
  {
#if defined(USE_CRL)
    rgn_start_write(Global);
#else
    acquire_lock(&(Global->CountLock));
#endif

    Global->n2bcalc += Local.myn2bcalc;
    Global->nbccalc += Local.mynbccalc;
    Global->selfint += Local.myselfint;

    ADDM(Global->keten, Global->keten, Local.myketen);
    ADDM(Global->peten, Global->peten, Local.mypeten);

    for (k=0; k<3; k++)
      Global->etot[k] += Local.myetot[k];

    ADDV(Global->amvec, Global->amvec, Local.myamvec);

    MULVS(tempv1, Global->cmphase[0], Global->mtot);
    MULVS(tempv2, Local.mycmphase[0], Local.mymtot);
    ADDV(tempv1, tempv1, tempv2);
    DIVVS(Global->cmphase[0], tempv1, Global->mtot+Local.mymtot); 

    MULVS(tempv1, Global->cmphase[1], Global->mtot);
    MULVS(tempv2, Local.mycmphase[1], Local.mymtot);
    ADDV(tempv1, tempv1, tempv2);
    DIVVS(Global->cmphase[1], tempv1, Global->mtot+Local.mymtot);
    Global->mtot += Local.mymtot;

#if defined(USE_CRL)
    rgn_end_write(Global);
#else
    release_lock(&(Global->CountLock));    
#endif
  }

#if defined(USE_CRL)
  rgn_barrier();
#else
  shm_barrier();
#endif

  if (ProcessId == 0)
  {
#if defined(USE_CRL)
    rgn_start_read(Global);
#endif

    nttot = Global->n2bcalc + Global->nbccalc;
    nbavg = (int) ((real) Global->n2bcalc / (real) nbody);
    ncavg = (int) ((real) Global->nbccalc / (real) nbody);

#if defined(USE_CRL)
    rgn_end_read(Global);
#endif
  }
}


/* compute set of dynamical diagnostics
 */

void diagnostics(unsigned ProcessId)
{
  int     pidx;
  bodyptr p;
  real    velsq;
  vector  tmpv;
  matrix  tmpt;

  Local.mymtot    = 0.0;
  Local.myetot[1] = 0.0;
  Local.myetot[2] = 0.0;

  CLRM(Local.myketen);
  CLRM(Local.mypeten);
  CLRV(Local.mycmphase[0]);
  CLRV(Local.mycmphase[1]);
  CLRV(Local.myamvec);

  for (pidx=(Local.mynbody-1); pidx>=0; pidx--)
  {
#if defined(USE_CRL)
    p = (bodyptr) rgn_map(Local.mybodytab[pidx]);
    rgn_start_read(p);
#else
    p = Local.mybodytab[pidx];
#endif

    Local.mymtot += Mass(p);
    DOTVP(velsq, Vel(p), Vel(p));
    Local.myetot[1] += 0.5 * Mass(p) * velsq;
    Local.myetot[2] += 0.5 * Mass(p) * Phi(p);
    MULVS(tmpv, Vel(p), 0.5 * Mass(p));
    OUTVP(tmpt, tmpv, Vel(p));
    ADDM(Local.myketen, Local.myketen, tmpt);
    MULVS(tmpv, Pos(p), Mass(p));
    OUTVP(tmpt, tmpv, Acc(p));
    ADDM(Local.mypeten, Local.mypeten, tmpt);
    MULVS(tmpv, Pos(p), Mass(p));
    ADDV(Local.mycmphase[0], Local.mycmphase[0], tmpv);
    MULVS(tmpv, Vel(p), Mass(p));
    ADDV(Local.mycmphase[1], Local.mycmphase[1], tmpv);
    CROSSVP(tmpv, Pos(p), Vel(p));
    MULVS(tmpv, tmpv, Mass(p));
    ADDV(Local.myamvec, Local.myamvec, tmpv);

#if defined(USE_CRL)
    rgn_end_read(p);
    rgn_unmap(p);
#endif
  }

  Local.myetot[0] = Local.myetot[1] + Local.myetot[2];
  if (Local.mymtot != 0)
  {
    DIVVS(Local.mycmphase[0], Local.mycmphase[0], Local.mymtot);
    DIVVS(Local.mycmphase[1], Local.mycmphase[1], Local.mymtot);
  }
}


/* Low-level input and output operations.
 */

void out_int(stream str, int ival)
{
  fprintf(str, "  %d\n", ival);
}


void out_real(stream str, real rval)
{
  fprintf(str, " %21.14E\n", rval);
}


void out_vector(stream str, vector vec)
{
  fprintf(str, " %21.14E %21.14E", vec[0], vec[1]);
  fprintf(str, " %21.14E\n",vec[2]);
}
