//----------------------------------*-C++-*----------------------------------//
// SP.h
// Geoffrey Furnish
// 2 December 1994
//---------------------------------------------------------------------------//
// @> A "smart pointer" facility.
//
// $Id: SP.h,v 1.8 1995/11/11 01:51:09 furnish Exp $
//
// $Log: SP.h,v $
// Revision 1.8  1995/11/11  01:51:09  furnish
// Little nitty template specification errors detected by IBM xlC, but
// missed by every other compiler I've used to date.
//
// Revision 1.7  1995/07/18  03:57:40  furnish
// Class documentation for SPList<T>.
//
// Revision 1.6  1995/07/14  14:48:45  furnish
// Implemented a singly linked list and iterator especially for handling
// smart pointers, with extensive testing.  Still needs some
// documentation, which will have to come later.
//
// Revision 1.5  1995/04/27  16:01:59  furnish
// autodoc supression of SPrep.
//
// Revision 1.4  1995/01/11  22:09:22  furnish
// Remove automatic conversion to T *, there were too many ways it could
// be used accidentally.  Added conversion to int for testing whether it
// has been set or not.
//
// Revision 1.3  1995/01/10  20:28:00  furnish
// Provide a const overloading also.
//
// Revision 1.2  1994/12/12  15:42:15  furnish
// Convenience accessor.
//
// Revision 1.1  1994/12/06  19:22:39  furnish
// A smart pointer class.  Provides much safety, but beware the provided
// automatic conversions which can be abused to one's detriment.
//
//---------------------------------------------------------------------------//

#ifndef __SP_h__
#define __SP_h__

#include "Assert.h"

// autodoc: noprint SPrep

//===========================================================================//
// class SP<T> - A templated "smart pointer" class.

// This class provides an encapsulation of a pointer of a specified type,
// enabling an accurate determination of when the pointer is in use and when
// it can be freed.  Consider: A function new's an object and return the
// pointer to that object as its return value.  Now it is the caller's
// responsibility to free the object.  What if the caller passes the pointer
// to other objects or functions?  What if it is not known which will be
// deleted first or last?
//
// Instead the function can return a "smart pointer".  This SP class uses
// reference counting to determine the number of current users of a pointer.
// Each time an SP goes out of scope, the reference count is decremented.
// When the last user of a pointer is done, the pointer is freed.
//
// Note:  I am calling this a "smart pointer", not a "safe pointer".  There
// are clearly ways you can hose this.  In particular, when you bind an SP<T>
// to a T*, you yield all rights to the T*.  You'd better not squirrel the
// bare pointer away somewhere and expect to clandestinely use it in other
// ways or places--death will be sure to follow.  Consequently then, the
// safest way to use this smart pointer, is to bind it to the contained
// pointer and then always use the smart pointer.  Immediately returning the
// smart pointer as a return value, allowing the orriginal bare pointer to go
// out of scope never to be seen again, is one good example of how to use
// this. 
//
// In other words, as with other parts of DS++, the SP<T> class is intended to
// facilitate legitimate use.  I am not bending over backwards to make certain
// it is inviolable.  The user is supposed to abstain from heinous misuse.
//===========================================================================//

template<class T>
struct SPrep {
    T *p;
    int refs;

    SPrep( T *pp ) : p(pp), refs(1) {}
    ~SPrep();
};

template<class T>
class SP {

    SPrep<T> *s;

  public:
    SP();
    SP( T *pp );
    SP( const SP<T>& sp );
    ~SP();

    SP<T>& operator=( const SP<T>& sp );
    SP<T>& operator=( T *pp );

    T *operator->()
    {
	Insist( s, "This SP<T> has not been bound to a T*." );

	return s->p;
    }

    const T *operator->() const
    {
	Insist( s, "This SP<T> has not been bound to a T*." );

	return s->p;
    }

    operator int()
    {
	return (s != NULL && s->p != NULL) ? 1 : 0;
    }

// Unsafe stuff, nobody use but me :-).

//    operator T *() { return s->p; }   Too unsafe even for me :-{
    T *bare_ptr()  { return s->p; }
    T *bp()        { return s->p; }
    int refs()     { return s->refs; }
};

// autodoc: noprint SPLink

template<class T> class SPList;
template<class T> class SPList_iter;

template<class T>
class SPLink {
    SP<T> p;
    SPLink<T> *next;

    SPLink( const SPLink<T>& );
    SPLink<T>& operator=( const SPLink<T>& );

  public:
    SPLink( SP<T> _p ) : p(_p), next(0) {}
    SPLink( SP<T> _p, SPLink<T> *nx ) : p(_p), next(nx) {}
    ~SPLink();

    SP<T> data() { return p; }

    friend class SPList<T>;
    friend class SPList_iter<T>;
};

//===========================================================================//
// class SPLink<T> - Class for handling lists of smart pointers

// The Slist<T> class is not very robust.  It doesn't clean up after itself
// properly, and doesn't have sensible copy ctor and assignment operator.
// This class is a special type of list class made just for working with
// smart pointers.  By being very careful to use copy ctors, assignment
// operators, and destructurs, in the appropriate way, we are able to obtain
// a specialized list class, made just for smart pointers, which preserves
// the integrity of the contained smart pointer.
//===========================================================================//

template<class T>
class SPList {
    SPLink<T> *last;

  public:
    SPList() { last = 0; }
    SPList( const SPList<T>& spl );
    ~SPList() { clear(); }

    SPList<T>& operator=( const SPList<T>& spl );

    void clear();

    void insert( SP<T>& s );
    void append( SP<T>& s );

    friend class SPList_iter<T>;
};

template<class T>
class SPList_iter {
    SPLink<T> *ce;
    SPList<T> *spl;

  public:
    SPList_iter( SPList<T>& _spl );
// Compiler generated copy ctor, assignment operator, and dtor, are all okay
// for this object.

    void reset();

    operator int() const { return ce != 0; }

    SP<T> operator()();

    void operator++();
    void operator++(int);
};    

#define INSTANTIATE_SP(a) \
template class SP<a>; \
template class SPrep<a>;

#define INSTANTIATE_SPList(a) \
template class SPList<a>; \
template class SPList_iter<a>; \
template class SPLink<a>;

#endif                          // __SP_h__

//---------------------------------------------------------------------------//
//                              end of SP.h
//---------------------------------------------------------------------------//
