//----------------------------------*-C++-*----------------------------------//
// Map.h
// Geoffrey Furnish
// 28 January 1994
//---------------------------------------------------------------------------//
// @> A template Map class for making associative arrays.
//
// $Id: Map.h,v 1.12 1995/11/11 01:51:07 furnish Exp $
//
// $Log: Map.h,v $
// Revision 1.12  1995/11/11  01:51:07  furnish
// Little nitty template specification errors detected by IBM xlC, but
// missed by every other compiler I've used to date.
//
// Revision 1.11  1995/09/16  17:02:05  furnish
// Convert Mapiter to Map_iter for conformity with the list variants.
//
// Revision 1.10  1995/08/28  20:45:15  furnish
// Some cleanups induced by SGI C++.
//
// Revision 1.9  1995/07/18  03:56:59  furnish
// A tiny little change to make Map iterators work better.
//
// Revision 1.8  1994/12/02  17:46:16  furnish
// Formatting, documentation.  Switch from codoc to autodoc.
//
// Revision 1.7  1994/10/21  15:05:20  furnish
// Fixes to support explicit template instantiation.  Since these classes
// have auxiliary classes, added some macros to instantiate all of them
// in a single shot, which is much more convenient.
//
// Revision 1.6  1994/06/23  05:38:24  furnish
// Fix boundary handling problem, reimplement test case to be useful.
//
// Revision 1.5  1994/06/17  20:18:18  furnish
// Added new const operator[] indexing method so can perform
// non-autoexanding lookups on const Map's.  Internal doc improvements.
//
// Revision 1.4  1994/05/18  21:53:33  furnish
// Mods to support iteration on const Map's.
//
// Revision 1.3  1994/05/05  20:48:34  furnish
// Try to support iteration on const Map's, but it still doesn't seem to
// be quite right.  GCC warnings persist.
//
// Revision 1.2  1994/04/20  22:05:27  furnish
// Fashion police.
//
// Revision 1.1  1994/02/08  19:28:27  furnish
// Stroustrup's associative array class.
//
//---------------------------------------------------------------------------//

#ifndef __Map_h__
#define __Map_h__

template<class K, class V> class Map;
template<class K, class V> class Map_iter;

// autodoc: noprint Link

//===========================================================================//
// class Link<K,V> - Helper class for Map<K,V>

// Not for use by client code.  This is a class which is intended to support
// the Map<K,V> class.  There is no other use.
//===========================================================================//

template<class K, class V> class Link {

    friend class Map<K,V>;
    friend class Map_iter<K,V>;

  private:
    const K key;
    V value;

    Link<K,V> *pre;
    Link<K,V> *suc;

    Link( const K& k, const V& v ) : key(k), value(v) {}
    Link( const Link<K,V>& l ) : key(l.key), value(l.value) {
	pre = suc = 0;
    }

#ifdef __GNUC__
  public:
    void set_pre( Link<K,V> *s ) { pre = s; }
    K Key()   { return key; }
    V Value() { return value; }
    Link *Suc() { return suc; }
#endif

    ~Link();			// delete all Links recursively;    
};

//===========================================================================//
// class Map<K,V> - A templated associative array class

// The Map class provides an associative mapping between two types.
// Indexing with objects of the Map key type produces references to
// objects of the Map value type.
//
// This class is also based directly on Stroustrup's version in
// section 8.8 of  the 2nd edition.
//===========================================================================//

template<class K, class V> class Map {

    friend class Map_iter<K,V>;

    Link<K,V> *head;
    Link<K,V> *current;

    K def_key;
    V def_val;
    int sz;

    void init() { sz = 0; head = 0; current = 0; }

  public:

    Map() { init(); }
    Map( const K& k, const V& d ) : def_key(k), def_val(d) { init(); }
    ~Map() { delete head; }	// delete all links recursively

    Map( const Map<K,V>& m );
    Map<K,V>& operator=( const Map<K,V>& m );

    V& operator[]( const K& key );      // expansion/retrieval
    V operator[]( const K& key ) const; // Lookup, no expansion.

    int size() const { return sz; }
    void clear() { delete head; init(); }
    void remove( const K& key );

// iteration functions.

    Map_iter<K,V> element( const K& k );
    Map_iter<K,V> first();
    Map_iter<K,V> last();
};

//===========================================================================//
// class Map_iter<K,V> - An iterator for Map<K,V>

// This class provides a mechanism for iterating over the elements in a
// Map<K,V>.  It is provided as a seperate class, rather than a facility
// embedded in Map<K,V>, so that users can have multiple concurrent iterators
// active on a given object, if they so desire.
//
// Iterator provides a void * conversion so it can be tested directly in
// conditionals.  If true, more entries are still in the Map.
//
// This class is also based directly on Stroustrup's version in
// section 8.8 of  the 2nd edition.
//===========================================================================//

template<class K, class V> class Map_iter {

    friend class Map<K,V>;

    const Map<K,V> *m;
    Link<K,V> *p;

// Private ctor, used by Map<K,V> to produce an iterator preset to a specific
// element.

    Map_iter( const Map<K,V> *mm, Link<K,V> *pp ) : m(mm), p(pp) {}

  public:
    // Default ctor.  Not real useful.  Maybe should deprecate?
    Map_iter() : m(0) {
	p = 0;
    }
    Map_iter( const Map<K,V>& mm );

    operator void * () { return p; }

    const K& key();
    const V& value();		// If they want to modify the value,
				// they can do m[ miter.key() ] = ...

    void reset();

    Map_iter<K,V>& operator--();
    void operator--(int);
    Map_iter<K,V>& operator++();
    void operator++(int);
};

#define INSTANTIATE_Map(a,b) \
template class Map<a,b>; \
template class Link<a,b>; \
template class Map_iter<a,b>;

#endif				// __Map_h__

//---------------------------------------------------------------------------//
//                              end of Map.h
//---------------------------------------------------------------------------//
