//----------------------------------*-C++-*----------------------------------//
// Map.cc
// Geoffrey Furnish
// 28 January 1994
//---------------------------------------------------------------------------//
// @> A template Map class for making associative arrays.
//
// $Id: Map.cc,v 1.15 1995/10/13 19:53:56 furnish Exp $
//
// $Log: Map.cc,v $
// Revision 1.15  1995/10/13  19:53:56  furnish
// Reformulated logic to avoid a compile warning on PGI and SGI.
//
// Revision 1.14  1995/09/16  17:02:04  furnish
// Convert Mapiter to Map_iter for conformity with the list variants.
//
// Revision 1.13  1995/08/28  20:45:13  furnish
// Some cleanups induced by SGI C++.
//
// Revision 1.12  1995/08/18  04:46:25  furnish
// Dummy methods needed to perform instantiation on an SGI.
//
// Revision 1.11  1995/07/18  03:57:00  furnish
// A tiny little change to make Map iterators work better.
//
// Revision 1.10  1994/12/06  19:26:59  furnish
// Assert.h now in DS++.
//
// Revision 1.9  1994/12/02  17:46:15  furnish
// Formatting, documentation.  Switch from codoc to autodoc.
//
// Revision 1.8  1994/10/05  17:53:15  furnish
// Correct bug in removal of head element.
//
// Revision 1.7  1994/06/23  05:38:22  furnish
// Fix boundary handling problem, reimplement test case to be useful.
//
// Revision 1.6  1994/06/17  20:18:16  furnish
// Added new const operator[] indexing method so can perform
// non-autoexanding lookups on const Map's.  Internal doc improvements.
//
// Revision 1.5  1994/05/18  21:53:32  furnish
// Mods to support iteration on const Map's.
//
// Revision 1.4  1994/05/05  20:48:32  furnish
// Try to support iteration on const Map's, but it still doesn't seem to
// be quite right.  GCC warnings persist.
//
// Revision 1.3  1994/04/20  22:05:26  furnish
// Fashion police.
//
// Revision 1.2  1994/02/15  19:37:29  furnish
// Fix bug in size counting which trashed the copy ctor.
//
// Revision 1.1  1994/02/08  19:28:26  furnish
// Stroustrup's associative array class.
//
//---------------------------------------------------------------------------//

#include "Map.h"

#include "Assert.h"

template<class K, class V>
Link<K,V>::~Link()
{
    delete suc;
}

//===========================================================================//
// class Map<K,V> - An associative array class

// This templated associative array class performs lookups of values
// based on keys, where the type of each is user selectable.  This
// class is modelled extremely closely on Stroustrup's 2nd edition
// version.  Minor bug fixes, and small enhancements are the only
// distinguishing features at this time.
//===========================================================================//

//---------------------------------------------------------------------------//
// Construct a new Map<K,V>, initializing w/ data from an existing one.
//---------------------------------------------------------------------------//

template<class K, class V>
Map<K,V>::Map( const Map<K,V>& m )
{
    init();
    def_key = m.def_key;
    def_val = m.def_val;
    sz = m.sz;
    if (!sz) return;

// Copy the head element.

    Link<K,V> *mp = m.head;
    Link<K,V> *p  = head = new Link<K,V>( *mp );

// Copy following elements if any.

    for( mp = mp->suc; mp; mp = mp->suc ) {
	Link<K,V> *np  = new Link<K,V>( *mp );
	p->suc = np;
	np->pre = p;
	np->suc = 0;		// presume end of list.
	p = np;			// prepare for next iteration.
    }

    (void) operator[]( m.current->key ); // set current.
}

//---------------------------------------------------------------------------//
// Assign to an existing Map<K,V>, initializing w/ data from another one.
//---------------------------------------------------------------------------//

template<class K, class V>
Map<K,V>& Map<K,V>::operator=( const Map<K,V>& m )
{
    if (this == &m)
      return *this;

    clear();

// All this copied verbatim from the copy ctor above.  If you change
// either, change both!

    def_key = m.def_key;
    def_val = m.def_val;
    sz = m.sz;
    if (!sz) return *this;

// Copy the head element.

    Link<K,V> *mp = m.head;
    Link<K,V> *p  = head = new Link<K,V>( *mp );

// Copy following elements if any.

    for( mp = mp->suc; mp; mp = mp->suc ) {
	Link<K,V> *np  = new Link<K,V>( *mp );
	p->suc = np;
	np->pre = p;
	np->suc = 0;		// presume end of list.
	p = np;			// prepare for next iteration.
    }

    (void) operator[]( m.current->key ); // set current.

    return *this;
}

//---------------------------------------------------------------------------//
// Remove an entry w/ specified key from an existing Map<K,V>.
//---------------------------------------------------------------------------//

template<class K, class V>
void Map<K,V>::remove( const K& key )
{
    Link<K,V> *p = head;
    for(; p ; p = p->suc ) {
	if (key == p->key) {	// remove this one.
 	    sz--;
	    if (p == head) {
 		head = p->suc;
		if (head)	// Might've just deleted the only node.
		    head->pre = 0;
 		p->suc = 0;	// don't delete the whole list!
 		delete p;
 		return;
 	    } else {
		p->pre->suc = p->suc;
		if (p->suc)
		    p->suc->pre = p->pre;
		p->suc = 0;	// don't delete the whole list!
		delete p;
		return;
 	    }
	}
    }

// Should this be an exception?  Hmm.  Well, he doesn't want it in
// there, and it's not, so he should be happy, right?

    throw( "No entry with key in Map." );  // On second thought...
    return;			// not found.
}

//---------------------------------------------------------------------------//
// Lookup the value specified by key, and return a reference.  User
// must avoid problems with dangling references himself.  Perform
// expansion of the map if key is new, filling in with default value.
//---------------------------------------------------------------------------//

template<class K, class V>
V& Map<K,V>::operator[]( const K& key )
{
    if (head == 0) {
	current = head = new Link<K,V>( key, def_val );
	current->pre = current->suc = 0;
	sz = 1;
	return current->value;
    }

    Link<K,V> *p = head;
    for(;;) {
	if ( p->key == key ) {	// found!
	    current = p;
	    return current->value;
	}
	if ( key < p->key ) {	// insert before p
	    current = new Link<K,V>( key, def_val );
	    current->pre = p->pre;
	    current->suc = p;
	    if (p == head)
		head = current;
	    else
		p->pre->suc = current;
	    p->pre = current;
	    sz++;
	    return current->value;
	}

	Link<K,V> *s = p->suc;
	if (s == 0) {		// add to end
	    current = new Link<K,V>( key, def_val );
	    current->pre = p;
	    current->suc = 0;
	    p->suc = current;
	    sz++;
	    return current->value;
	}
	p = s;
    }
}

//---------------------------------------------------------------------------//
// Perform lookup on a const Map<K,V>.  If key not found, throw an
// exception. 
//---------------------------------------------------------------------------//

#ifdef __GNUC__
template<class K, class V>
V Map<K,V>::operator[]( const K& key ) const
{
    if (head == 0)
	throw( "const Map<K,V> not initialized!" );

    for( Link<K,V> *p = head; p; p=p->Suc()  ) {
	if ( p->Key() == key )
	    return p->Value();
    }

    throw( "No such entry in Map<K,V>" );
    //  We'll be dead by now, so just shutting up the compiler.
    return head->Value();	// grrrrrrrrrrrr.
}
#else
template<class K, class V>
V Map<K,V>::operator[]( const K& key ) const
{
    if (head == 0)
	throw( "const Map<K,V> not initialized!" );

    int found = 0;
    for( Link<K,V> *p = head; p; p=p->suc) {
	if ( p->key == key ) {
	    found = 1;
	    break;
	}
    }

    if (!found) {
	throw( "No such entry in Map<K,V>" );
    }

    return p->value;
}
#endif

// iteration functions.

//---------------------------------------------------------------------------//
// Obtain an iterator preset to the value with specified key.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V> Map<K,V>::element( const K& k )
{
    (void) operator[](k);	// move current to k
    return Map_iter<K,V>(this,current);
}

//---------------------------------------------------------------------------//
// Obtain an iterator preset to first item in the map.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V> Map<K,V>::first()
{
    return Map_iter<K,V>(this,head);
}

//---------------------------------------------------------------------------//
// Obtain an iterator preset to last item in the map.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V> Map<K,V>::last()
{
    Map_iter<K,V> m(this, head);

    for( int i=0; i < sz-1; i++ )
	m++;

    return m;
}

//===========================================================================//
// class Map_iter<K,V> - An iterator for the Map<K,V> associative array class

// This class allows you to iterate through the elements of a Map
// without accidentally inducing expansion, etc.  Since it is freestanding,
// you may have multiple iterators active on a given map at one time, and they
// can traverse the map at different rates, etc.
//===========================================================================//

//---------------------------------------------------------------------------//
// Constructor.  Initialize iterator to first element of Map.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V>::Map_iter( const Map<K,V>& mm )
    : m(&mm)
{
    p = m->head;
}

//---------------------------------------------------------------------------//
// Return the key for the current Map entry.
//---------------------------------------------------------------------------//

template<class K, class V> const K& Map_iter<K,V>::key()
{
    if (p)
	return p->key;
    else
	return m->def_key;
}

//---------------------------------------------------------------------------//
// Return the value of the current Map entry.  If you want to modify it, do
// m[miter.key()] = ... 
//---------------------------------------------------------------------------//

template<class K, class V> const V& Map_iter<K,V>::value()
{
    if (p)
	return p->value;
    else
	return m->def_val;
}

//---------------------------------------------------------------------------//
// Reset the iterator to the beginning, so we can reuse it.
//---------------------------------------------------------------------------//

template<class K, class V> void Map_iter<K,V>::reset()
{
    p = m->head;
}

//---------------------------------------------------------------------------//
// Prefix decrement the iterator.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V>& Map_iter<K,V>::operator--()
{
    if (p) p = p->pre;
    return *this;
}

//---------------------------------------------------------------------------//
// Postfix decrement the iterator.
//---------------------------------------------------------------------------//

template<class K, class V>
void Map_iter<K,V>::operator--(int)
{
    if (p) p = p->pre;
}

//---------------------------------------------------------------------------//
// Prefix increment the iterator.
//---------------------------------------------------------------------------//

template<class K, class V>
Map_iter<K,V>& Map_iter<K,V>::operator++()
{
    if (p) p = p->suc;
    return *this;
}

//---------------------------------------------------------------------------//
// Postfix increment the iterator.
//---------------------------------------------------------------------------//

template<class K, class V>
void Map_iter<K,V>::operator++(int)
{
    if (p) p = p->suc;
}

//---------------------------------------------------------------------------//
//                              end of Map.cc
//---------------------------------------------------------------------------//
