//----------------------------------*-C++-*----------------------------------//
// List.h
// Geoffrey Furnish
// 15 January 1994
//---------------------------------------------------------------------------//
// @> A class for managing lists of other things.
//
// $Id: List.h,v 1.8 1995/09/16 17:04:13 furnish Exp $
//
// $Log: List.h,v $
// Revision 1.8  1995/09/16  17:04:13  furnish
// Convert all list variants away from auto-incrementation behavior, and
// standardize on the explicit increment mode which was already in use
// with the SPlist.  All DS++ list and map classes now support uniform
// naming and operation conventions.
//
// Revision 1.7  1995/06/13  21:18:38  furnish
// Support resetting of list iterators.
//
// Revision 1.6  1994/12/02  17:46:13  furnish
// Formatting, documentation.  Switch from codoc to autodoc.
//
// Revision 1.5  1994/10/21  15:05:19  furnish
// Fixes to support explicit template instantiation.  Since these classes
// have auxiliary classes, added some macros to instantiate all of them
// in a single shot, which is much more convenient.
//
// Revision 1.4  1994/05/18  21:52:54  furnish
// Arcane bug exposed by Lucid C++.  There might be more like this.
//
// Revision 1.3  1994/02/15  19:39:30  furnish
// Formatting.
//
// Revision 1.2  1994/02/08  19:25:48  furnish
// Fix the iterator so it works...
//
// Revision 1.1  1994/02/01  19:20:34  furnish
// Stroustrup's Islist and Slist classes, which implement templated
// intrusive and nonintrusive singly linked lists, respectively.
//
//---------------------------------------------------------------------------//

#ifndef __List_h__
#define __List_h__

#include "slist.h"

template<class T>
class Islist_iter;

//===========================================================================//
// class Islist<T> - An intrusive singly linked list

// Ala Stroustrup, chapter 8.
//===========================================================================//

template<class T>
class Islist : private slist_base {
  public:
    Islist() : slist_base() {}
    Islist( T *a ) : slist_base(a) {}

    void insert( T *a ) {
	slist_base::insert(a);
    }
    void append( T *a ) {
	slist_base::append(a);
    }
    T *get() {
	return (T *) slist_base::get();
    }

    void clear() { slist_base::clear(); }

    friend class Islist_iter<T>;
};

//===========================================================================//
// class Islist_iter<T> - An iterator for an Islist<T>

// Ala Stroustrup, chapter 8.
//===========================================================================//

template<class T>
class Islist_iter : private slist_base_iter {
  public:
    Islist_iter( Islist<T>& s ) : slist_base_iter(s) {}

    operator void *() { return slist_base_iter::operator void *(); }

    T *operator()() {
	return (T *) slist_base_iter::operator()();
    }

    void reset() { slist_base_iter::reset(); }

    void operator++()    { slist_base_iter::operator++(); }
    void operator++(int) { slist_base_iter::operator++(); }
};

// autodoc: noprint Tlink

// And now, finally, the nonintrusive singly linked list.

template<class T>
class Slist_iter;

template<class T>
struct Tlink :public slink {
    T info;
    Tlink( const T& a ) : info(a) {}
};

//===========================================================================//
// class Slist<T> - A templated singly linked list class

// The Slist<T> class provides a way to make a list of any arbitrary object.
// An associated iterator class is available.  This code is derived from
// Stroustrup, 2nd, with corrections.
//===========================================================================//

template<class T>
class Slist : private slist_base {
  public:
    void insert( T& a ) {
	slist_base::insert( new Tlink<T>(a) );
    }
    void append( const T& a ) {
	slist_base::append( new Tlink<T>(a) );
    }
    T get() {
	Tlink<T> *lnk = (Tlink<T> *) slist_base::get();
	T i = lnk->info;
	delete lnk;
	return i;
    }

    void clear() { slist_base::clear(); }

    friend class Slist_iter<T>;
};

//===========================================================================//
// class Slist_iter<T> - An iterator for an Slist<T>

// A basic iterator for the Slist<T> class.  This is a free standing class, so
// you can have multiple iterators per list object.  This code is based on
// that presented in Stroustrup, 2nd, with corrections.  The iterator is
// autoadvancing in the sense that subscripting the iterator produces an
// element and advances to point to the next.  The iterator can be tested
// directly in a conditional via the provided operator void *, to determine if
// there are additional elements available on the list.
//===========================================================================//

template<class T>
class Slist_iter : private slist_base_iter {
  public:
    Slist_iter( Slist<T>& s ) : slist_base_iter(s) {}

    operator void *() { return slist_base_iter::operator void *(); }

    T& operator()() {
    // This has the severe disadvantage that if the list is empty,
    // the return result of this iterator is complete garbage.  I
    // don't see a way to fix this.  Should really throw an
    // exception. 
	return ((Tlink<T> *) slist_base_iter::operator()())->info;
    }

    void reset() { slist_base_iter::reset(); }

    void operator++()    { slist_base_iter::operator++(); }
    void operator++(int) { slist_base_iter::operator++(); }
};

//---------------------------------------------------------------------------//
// Handy macros for explicitly instantiating the list code, for compilers
// which support the new syntax.  GCC 2.6+ is one such compiler.
//---------------------------------------------------------------------------//

#define INSTANTIATE_Islist(a) \
template class Islist<a>; \
template class Islist_iter<a>;

#define INSTANTIATE_Slist(a) \
template class Tlink<a>; \
template class Slist<a>; \
template class Slist_iter<a>;

#endif				// __List_h__

//---------------------------------------------------------------------------//
//                              end of List.h
//---------------------------------------------------------------------------//
