//----------------------------------*-C++-*----------------------------------//
// DynTable.cc
// Geoffrey Furnish
// Tue Jan 24 11:44:14 1995
//---------------------------------------------------------------------------//
// @> An autoexpanding row/col container class.
//
// $Id: DynTable.cc,v 1.2 1995/07/20 17:24:33 lindberg Exp $
//
// $Log: DynTable.cc,v $
// Revision 1.2  1995/07/20  17:24:33  lindberg
// Expanded DynTable that incorporates added methods that will be used by
// tracer.
//
// Revision 1.1  1995/01/24  18:31:53  furnish
// New templated class for handling dynamic tables of data.
//
//---------------------------------------------------------------------------//

#include "DynTable.h"
#include "Assert.h"

//---------------------------------------------------------------------------//
// Autoexpanding index method.  Exports a reference.
//---------------------------------------------------------------------------//

template<class T>
T& DynTable<T>::operator()( int row, int col )
{
    return data[row][col];
}

//---------------------------------------------------------------------------//
// Index method.  Returns only the value, so use this one where practical.
//---------------------------------------------------------------------------//

template<class T>
T DynTable<T>::operator()( int row, int col ) const
{
    return data[row][col];
}

//---------------------------------------------------------------------------//
// Delete an entire row from the table.  Assumed that deleting a row
//   does not count as a reference, so low and high reference marks are set
//   such that the old values are retained (and possibly decremented by one).
//---------------------------------------------------------------------------//

template<class T>
void DynTable<T>::delete_row( int row )
{
    int row_base = data.Get_base();
    int row_size = data.Get_size();
    int row_low = data.low();
    int row_high = data.high();

    Assert( row >= row_base && row < row_base + row_size );

    for (int index = row; index < row_base + row_size - 1; index++) {
      data[index] = data[index+1];
    }

// Clear out the last data array element by setting it equal to an empty
//   DynArray.
    DynArray<T> empty;
    data[row_base+row_size-1] = empty;

// Decrement the highest and lowest row reference, if appropriate,
//   since a row has been deleted.
    if ( row < row_low ) row_low--;
    if ( row <= row_high ) row_high--;

// Reset the row low and high reference marks since the copying done in
//   the loop above and the emptying of the last array element caused
//   the reference marks to be reset.
    data.low(row_low);
    data.high(row_high);

}


//---------------------------------------------------------------------------//
//  Delete a range of columns in a particular row.  The surviving columns
//    are copied into a new row.  Since the new row would automatically be
//    properly sized for the smaller number of columns, although it is
//    necessary keep track of the low and high reference marks.
//---------------------------------------------------------------------------//
template<class T>
void DynTable<T>::del_col_range( int row, int begin_col, int end_col )
{
    int row_base = data.Get_base();
    int row_size = data.Get_size();

    Assert( row >= row_base && row < row_base + row_size );

// Save the current values for later use.
    int col_base = data[row].Get_base();
    int col_size = data[row].Get_size();
    int col_low = data[row].low();
    int col_high = data[row].high();

    Assert( begin_col >= col_base && end_col < col_base + col_size );

// Create a new row that will contain the row in question minus the
// the deleted columns.  The non-deleted columns are copied into new_row.
// New_row will have the correct size needed to contain the non-deleted
// columns, and not the size of the original row.
    DynArray<T> new_row;

// Copy in the non-deleted columns
    int column;
    for (column = 0; column < begin_col; column++ ) {
      new_row[column] = data[row][column];
    }

    int col_offset = end_col - begin_col + 1;
    for ( ; column < col_base+col_size-col_offset; column++) {
      new_row[column] = data[row][column+col_offset];
    }

    data[row] = new_row;

// Check to see if the low and high reference marks need to be reset.
    int old_col_low = col_low;
    int old_col_high = col_high;
    for (column = begin_col; column <= end_col; column++) {
      if (column < old_col_low) col_low--;
      if (column <= old_col_high) col_high--;
    }

// Reset the row low and high reference marks since the copying done in
//   the loop above caused the reference marks to be reset.
    data[row].low(col_low);
    data[row].high(col_high);
}

//---------------------------------------------------------------------------//
//                              end of DynTable.cc
//---------------------------------------------------------------------------//
