//----------------------------------*-C++-*----------------------------------//
// DynArray.h
// Geoffrey Furnish
// 28 January 1994
//---------------------------------------------------------------------------//
// @> A dynamicly growing array template class.
//
// $Id: DynArray.h,v 1.10 1995/05/26 05:17:47 furnish Exp $
//
// $Log: DynArray.h,v $
// Revision 1.10  1995/05/26  05:17:47  furnish
// Added equality and inequality operators.
//
// Revision 1.9  1995/01/27  20:08:43  furnish
// Public methods for manipulating the high and low water marks.
// Dangerous, make sure you know what you're doing before you muck with
// these.
//
// Revision 1.8  1995/01/09  21:29:16  furnish
// Improvements to usability and to expansion calculation code.
// Hopefully this heads off some possibility for bad behavior with
// pathological arguments.
//
// Revision 1.7  1994/12/02  17:48:21  furnish
// Improvements to the DynArray<T> class to facilitate use with the
// Perstalyzer.  Documentation, formatting improvements.  Support for
// explicit instantiation.  Etc.
//
// Revision 1.6  1994/10/21  14:29:38  furnish
// Convert to using -fno-implicit-templates with gcc.  Requires 2.6+.
//
// Revision 1.5  1994/08/08  17:08:22  furnish
// Fixed insidious bug.
//
// Revision 1.4  1994/06/17  20:10:26  furnish
// Some new methods, and lots comment improvements.
//
// Revision 1.3  1994/06/02  15:37:36  furnish
// Added an operator[]() const method, which was accidentally omitted.
//
// Revision 1.2  1994/04/20  22:04:54  furnish
// Bug fixes, better robustness more consistency checks, etc.
//
// Revision 1.1  1994/02/08  19:24:48  furnish
// Templated array and dynamically growable array classes.
//
//---------------------------------------------------------------------------//

#ifndef __DynArray_h__
#define __DynArray_h__

#include <iostream.h>

//===========================================================================//
// class DynArray<T> - A dynamically growable templated array class

// This class provides an array facility which expands dynamically on
// reference so that it is always big enough to hold what you want to put on
// it, but predimensioning is not required.  When a reference is made to an
// element which is out of bounds on either end, the array is resized,
// extending in the direction of the reference.  
//
// The default subscripting mechanism exports a reference to the internal
// datum.  This is obviously dangerous, and the user is hereby forewarned.
// Beware of dangling references!  You should assume that any reference may
// invalidate the previous result.  A const form of indexing allows retrieval
// by value without dynamic extension, if such semantics are desired for a
// particular situation.
//===========================================================================//

template<class T>
class DynArray {

    T *v;
    T defval;
    int base;
    int sz;
    float growthfactor;
    int lowref, hiref;

  public:

    DynArray( int _sz =1, int _base =0, float gf =1.5 );
    DynArray( int _sz, int _base, T dv, float gf );
    DynArray( const DynArray<T>& da );
    DynArray<T>& operator=( const DynArray<T>& da );

    ~DynArray() { v += base; delete[] v; }

    T Get_defval() const { return defval; }
    int Get_base() const { return base; }
    int Get_size() const { return sz; }
    float Get_growthfactor() const { return growthfactor; }
    int low()  const { return lowref; }
    int high() const { return hiref; }

    void low(  int l ) { lowref = l; }
    void high( int h ) { hiref  = h; }

// This is dangerous.  Some say you shouldn't even ever return a
// reference from a method if the memory could move underneath. ..
// Well, this is an array, and we need these semantics.  Just have to
// leave it to the user to be sure he doesn't give his code a dirty
// fill.

    T& operator[]( int n );
    T operator[]( int n ) const;

    int operator==( const DynArray<T>& da ) const;
    int operator!=( const DynArray<T>& da ) const;
};

template<class T>
ostream& operator<<( ostream& os, const DynArray<T>& d );

#define INSTANTIATE_DynArray(a) \
template class DynArray<a>; \
template ostream& ostream<<( ostream& os, const DynArray<a>& d );

#endif				// __DynArray_h__

//---------------------------------------------------------------------------//
//                              end of DynArray.h
//---------------------------------------------------------------------------//
