//----------------------------------*-C++-*----------------------------------//
// Array.h
// Geoffrey Furnish
// 28 January 1994
//---------------------------------------------------------------------------//
// @> An Array template class.
//
// $Id: Array.h,v 1.13 1995/05/19 08:04:16 mjl Exp $
//
// $Log: Array.h,v $
// Revision 1.13  1995/05/19  08:04:16  mjl
// Eliminated the Mat1 define.  Now Mat1 is a separate class, just for
// matrices holding numeric data.
//
// Revision 1.12  1995/05/05  04:44:54  furnish
// Constification.
//
// Revision 1.11  1995/05/04  05:31:45  mjl
// Made a #define so that a Mat1<T> is the same as an Array<T> -- good for
// orthogonality.  Defined () indexing operators, and moved [] indexing
// operators into header to be with the others.
//
// Revision 1.10  1995/03/27  23:22:35  mjl
// Member data that was previously private is now protected so that derived
// classes (such as DstArray) may directly access them.
//
// Revision 1.9  1994/12/02  17:46:11  furnish
// Formatting, documentation.  Switch from codoc to autodoc.
//
// Revision 1.8  1994/10/21  14:37:40  furnish
// Fix major aliasing bug which was corrupting the memory manager.
//
// Revision 1.7  1994/09/23  20:08:38  furnish
// Added a new aliasing constructor, which should probably have been in a
// long time ago anyway.  Needed right now by the grid orthogonalizer.
//
// Revision 1.6  1994/08/31  19:10:15  furnish
// New method.
//
// Revision 1.5  1994/06/17  20:09:50  furnish
// Some new methods, and lots comment improvements.
//
// Revision 1.4  1994/06/09  19:56:37  furnish
// Made some better accessors, but had to rename some class variables
// because of Cfront braindeath.
//
// Revision 1.3  1994/04/27  18:55:01  furnish
// New methods for returning state info.
//
// Revision 1.2  1994/04/20  22:03:59  furnish
// Massive flesh out.  Much more mature now.
//
// Revision 1.1  1994/02/08  19:24:46  furnish
// Templated array and dynamically growable array classes.
//
//---------------------------------------------------------------------------//

#ifndef __Array_h__
#define __Array_h__

//===========================================================================//
// class Array<T> - A bounds checked templated array class

// The Array template class is provided to allow making robust arrays
// of user objects.  This class implements the basic functionality
// which Stroustrup and others generally represent in their ``Vector''
// class examples.  Well, a vector is a mathematical object.  Array
// seems to be a more appropriate name for this.
//
// This Array class is intended to provide a robust linear container.
// As such, it checks for range violations on both ends.  The theory
// here is that if you want something efficient, you'll use a
// conventional pointer.  This class should catch any out of bounds
// index.
//
// The class allows the Array to have arbitrary lower bound--it
// doesn't have to be zero based.  The constructors take a size
// argument which is how many elements the Array has, and a base
// argument which defaults to zero for normal C/C++ style indexing.
// Supplying any integer value for the base argument makes the first
// item appear at that index.
//
// A special constructor is provided which allows specifying a default
// value for the elements.
//
// And finally, the class destructor deletes the entire array,
// including the elements which comprise it.  So any elements which
// are to be put on this list should be well behaved with respect to
// memory usage.  If they have internal pointers in their member data,
// they should ensure that these pointers are not abused.  Normally
// this means providing (correctly designed) copy constructors,
// assignment operators, and destructors, in any class which will be
// contained by this Array class.
//===========================================================================//

#include "Assert.h"

#define BUILD_BINARY_OP(a,b) \
Array<T>& operator a ( b t ) \
{ \
  for( int i=base_; i < base_ + sz; i++ ) \
    v[i] a t; \
  return *this; }

template<class T>
class Array {

  protected:

    T *v;
    int base_, sz;
    int am_alias;

  public:

    // So can allocate arrays of Array<T>'s.
    Array() : base_(0), sz(0), v(0), am_alias(0) {} 

    Array( int s, int b =0 );
    Array( int s, T defval, int b ); // Note 2.
    Array( const Array<T>& t );
    Array( T *pt, int s, int b =0 );

    ~Array();

    Array<T>& operator=( const Array<T>& t );

    int operator==( const Array<T>& t ) const;
    int operator!=( const Array<T>& t ) const;

// Return reference to element.
// Safe b/c this class doesn't do dynamic resizing, and hence exported
// references don't get shuffled around underneath the client's nose.
// Of course an exported reference won't be valid past the destruction
// of the containing Array.
 
    T& operator() ( int n ) {
	Assert( n >= base_ && n < base_ + sz );
	return v[ n ];
    }

    T operator() ( int n ) const {
	Assert( n >= base_ && n < base_ + sz );
	return v[ n ];
    }

    T& operator[]( int n )
    {
	Assert( n >= base_ && n < base_ + sz );
	return v[ n ];
    }

    T operator[]( int n ) const
    {
	Assert( n >= base_ && n < base_ + sz );
	return v[ n ];
    }

// Should these be deprecated?  Why should you have to say "Get_xxx"?
// Is this nonsense just because we don't want a method with the same
// name as a member?

    int Get_base() const { return base_; }
    int Get_size() const { return sz; }

    int base() const { return base_; }
    int size() const { return sz; }
    int top()  const { return base_ + sz - 1; }

// Now let's hit the operators.

    BUILD_BINARY_OP(=,T);

    BUILD_BINARY_OP(+=,T);
    BUILD_BINARY_OP(-=,T);
    BUILD_BINARY_OP(*=,T);
    BUILD_BINARY_OP(/=,T);

    Array<T> operator+( const Array<T>& x );
    Array<T> operator+( const T& x );
//    friend Array<T> operator+( const T& x, const Array<T>& y ); // Note 1

    Array<T>& operator+=( const Array<T>& x );

    Array<T> operator-( const Array<T>& x );
    Array<T> operator-( const T& x );
//   friend Array<T> operator-( const T& x, const Array<T>& y ); // Note 1

    Array<T> operator*( const T& x );
//   friend Array<T> operator*( const T& x, const Array<T>& y );

    Array<T> operator/( const T& x );

// Note 1:  These represent "promotion" of a T on the left to an
// Array<T>.  Does this seem sensible?  Looks okay for things like
// Array<float>, etc, but we'll have to keep an eye on this to make
// sure it doesn't lead to nasty situations.
//
// NOTE to Note 1:  This unfortunately doesn't work, due to blasted
// bugs in Cfront 3.x.  Abandoned until Cfront gets its act together. 

// Note 2:  It was desirable to have something like:
//     Array( int _sz, T defval, int _base =0 );
// but this caused problems if T happened to be int, since that made
// this ctor ambiguous with Array( int _sz, int _base =0 ).  That is,
// Cfront couldnt tell which one was to be invoked by code like:
//    Array<int> a( 4, 5 );
// Is that Array( int sz = 4, int base = 5 ), or is it 
// Array( int sz = 4, int defval = 5, int base = 0 );
// So this had to be abandoned as well.   grrrrrrrrrrrrr.  Instead we
// provide the super convenient ctor, and then if you want to specify
// the default value, well, you get to set the base too.  Oh well.

};

#undef BUILD_BINARY_OP

#endif				// __Array_h__

//---------------------------------------------------------------------------//
//                              end of Array.h
//---------------------------------------------------------------------------//
