//----------------------------------*-C++-*----------------------------------//
// tstSP.cc
// Geoffrey Furnish
// 3 December 1994
//---------------------------------------------------------------------------//
// @> Test program for the SP<T> class
//
// $Id: tstSP.cc,v 1.6 1995/09/16 17:04:53 furnish Exp $
//
// $Log: tstSP.cc,v $
// Revision 1.6  1995/09/16  17:04:53  furnish
// Upgrade to new iterator semantics.
//
// Revision 1.5  1995/07/18  03:58:31  furnish
// Embellished to show interaction of SP<T> with existing Map<K,V> class.
//
// Revision 1.4  1995/07/14  14:48:47  furnish
// Implemented a singly linked list and iterator especially for handling
// smart pointers, with extensive testing.  Still needs some
// documentation, which will have to come later.
//
// Revision 1.3  1995/04/27  15:45:27  furnish
// Prevent spurious manpage generation for silly test classes.
//
// Revision 1.2  1995/02/10  19:57:48  furnish
// Better support for GCC.
//
// Revision 1.1  1994/12/06  19:25:21  furnish
// Test code for the smart pointer class.
//
//---------------------------------------------------------------------------//

#include <iostream.h>

#include "SP.cc"
#include "List.cc"
#include "Map.cc"

// autodoc: noprint foo

int nfoos = 0;

class foo {

    int v;

  public:
    foo() { v=0; nfoos++; cout << "foo constructed.\n"; }
    foo( int i ) { v=i; nfoos++; cout << "foo constructed.\n";; }
    ~foo() { nfoos--; cout << "foo destroyed.\n"; }
    void method() { cout << "foo::method invoked.\n"; }
    int val() const { return v; }
};

#ifdef __GNUC__
INSTANTIATE_SP(foo);
INSTANTIATE_SPList(foo);
INSTANTIATE_Map(int,SP<foo>);
#endif

//---------------------------------------------------------------------------//
// Function to help us keep track of the foo pool.
//---------------------------------------------------------------------------//

int expect( int n ) {
    cout << "Expecting " << n << " foo's, ";
    if (nfoos == n) {
	cout << "good.\n";
	return 1;
    } else {
	cout << "but there are " << nfoos << ". <<<<<<<<<<<<<\n";
	return 0;
    }
}

SP<foo> getafoo()
{
    SP<foo> p = new foo;
    return p;
}

SP<foo> bar( SP<foo> s )
{
    s->method();

    SP<foo> p = new foo;

    p = s;

    cout << "a foo should've just been destroyed.\n";

    return p;
}

#ifdef __GNUC__
INSTANTIATE_Slist(SP<foo>);
#endif

void tst_splist();
void tst_map();

main()
{
    ios::sync_with_stdio();

    cout << "tstSP starting.\n";

    {
	expect(0);
	
	SP<foo> s = getafoo();

	expect(1);

	Slist<SP<foo> > spfl;

	bar( s );

	expect(1);
    }
    expect(0);

    tst_splist();

    tst_map();
}

//---------------------------------------------------------------------------//
// Function to test most basic behavior of SPList<T>.  Make sure we can build
// a list, and get things to go away automatically at the right time.
//---------------------------------------------------------------------------//

void t1()
{
    expect(0);
    {
	SPList<foo> s;
	{
	    SP<foo> spf = new foo;

	    expect(1);

	    s.append( spf );

	    expect(1);

	    cout << "An SP<foo> is now going to go out of scope, but since\n"
		 << "it is on an SPList, the foo won't be destroyed.\n";
	}
	expect(1);

	cout << "Now the list is going out of scope, so a foo should die.\n";
    }
    expect(0);
}

//---------------------------------------------------------------------------//
// More advanced tests of SPList<T>, checking the copy ctor, operator=,
// clear() method, etc.  Tests of having things on multiple lists, etc.
//---------------------------------------------------------------------------//

void t2()
{
    int i;

    cout << "\n\n Testing SPList copy ctor, etc.\n\n";
    expect(0);

    {
	SPList<foo> s1;

	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    s1.append( spf );
	}
	expect(5);

	{
	    cout << "Now going to make a new list using copy ctor.\n";
	    SPList<foo> s2( s1 );
	    expect(5);
	    cout << "Now s2 will go out of scope, but no foo's should die\n"
		 << "since they are also on s1.\n";
	}
	expect(5);

	{
	    cout << "Now going to make a new list using operator=().\n";
	    SPList<foo> s3;
	    s3 = s1;
	    expect(5);
	    cout << "Now s3 will go out of scope, but no foo's should die\n"
		 << "since they are also on s1.\n";
	}
	expect(5);

	cout << "Now going to clear the list, all foo's should die.\n";
	s1.clear();
	expect(0);

	cout << "Putting some new foo's on the cleared list.\n";
	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    s1.append( spf );
	}
	expect(5);

	{
	    cout << "Now going to make a new list with some new foo's.\n";
	    SPList<foo> s4;
	    for( i=0; i < 3; i++ ) {
		SP<foo> spf = new foo(i);
		s4.insert( spf );
	    }
	    expect(8);
	    cout << "Now going to assign s4 to s1, 5 foo's should die.\n";
	    s1 = s4;
	    expect(3);
	}

	cout << "Now s1 will go out of scope, 3 foo's should die.\n";
    }
    expect(0);
    cout << endl << endl;
}

//---------------------------------------------------------------------------//
// Tests of SPList_iter<T>, etc.
//---------------------------------------------------------------------------//

void t3()
{
    int i;
    cout << "\n\n Testing SPList_iter<T>.\n\n";
    expect(0);
    {
	SPList<foo> s1;

	cout << "Putting some foo's on a list.\n";
	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    s1.append( spf );
	}
	expect(5);

	cout << "Now going to walk the list.  Should see 0..4.\n";

	for( SPList_iter<foo> si(s1); si; si++ )
	    cout << si()->val() << ' ';
	cout << endl;

	cout << "Clearing the list.\n";
	s1.clear();
	expect(0);

	cout << "Putting some foo's on a list.\n";
	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    s1.insert( spf );
	}
	expect(5);

	cout << "Now going to walk the list.  Should see 4..0.\n";

	for( SPList_iter<foo> si2(s1); si2; si2++ )
	    cout << si2()->val() << ' ';
	cout << endl;

	cout << "Now resetting the iterator and rewalking the list.\n";

	for( si2.reset(); si2; si2++ )
	    cout << si2()->val() << ' ';
	cout << endl;

	cout << "Now initializing a new iter from an old one,\n"
	     << "should see 2..0.\n";

	si2.reset(); si2++; si2++;
	for( SPList_iter<foo> si3(si2); si3; si3++ )
	    cout << si3()->val() << ' ';
	cout << endl;

	expect(5);
    }
    expect(0);
}

void tst_splist()
{
    cout << "\n\n Testing the SPList<T> class.\n\n";

    t1();
    t2();
    t3();
}

//---------------------------------------------------------------------------//
// Test interaction of smart pointers with Maps.
//---------------------------------------------------------------------------//

void u1()
{
    int i;
    expect(0);
    {
	Map< int, SP<foo> > mif;

	cout << "Putting some foo's in a Map.\n";

	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    mif[i] = spf;
	}

	expect(5);

	cout << "Now the Map will go out of scope.\n";
    }
    expect(0);
    {
	Map< int, SP<foo> > mif;

	cout << "Putting some foo's in a new Map.\n";

	for( i=0; i < 5; i++ ) {
	    SP<foo> spf = new foo(i);
	    mif[i] = spf;
	}

	expect(5);

	cout << "Walking a Map.\n";
	for( Map_iter<int,SP<foo> > mi( mif ); mi; mi++ ) {
	    SP<foo> spf = mi.value();
	    cout << spf->val() << ' ';
	}
	cout << endl;

	{
	    cout << "Using copy ctor to make a new Map.\n";
	    Map<int,SP<foo> > mif2( mif );
	    expect(5);
	    cout << "Ready to let aliasing Map go out of scope.\n";
	}
	expect(5);

	SPList<foo> s;
	for( mi.reset(); mi; mi++ ) {
	    SP<foo> spf = mi.value();
	    s.insert( spf );
	}
	expect(5);

	cout << "Now the Map and the SPList will go out of scope.\n";
    }

    expect(0);
}

void tst_map()
{
    cout << "\n\n Testing SP<T> with Map<K,V>.\n\n";

    u1();
}

//---------------------------------------------------------------------------//
//                              end of tstSP.cc
//---------------------------------------------------------------------------//
