//----------------------------------*-C++-*----------------------------------//
// SP.cc
// Geoffrey Furnish
// 2 December 1994
//---------------------------------------------------------------------------//
// @> A "smart pointer" facility.
//
// $Id: SP.cc,v 1.3 1995/10/13 19:55:11 furnish Exp $
//
// $Log: SP.cc,v $
// Revision 1.3  1995/10/13  19:55:11  furnish
// Pedantic compiler issues.
//
// Revision 1.2  1995/07/14  14:48:44  furnish
// Implemented a singly linked list and iterator especially for handling
// smart pointers, with extensive testing.  Still needs some
// documentation, which will have to come later.
//
// Revision 1.1  1994/12/06  19:22:38  furnish
// A smart pointer class.  Provides much safety, but beware the provided
// automatic conversions which can be abused to one's detriment.
//
//---------------------------------------------------------------------------//

#include "SP.h"

#include "Assert.h"

template<class T>
SPrep<T>::~SPrep()
{
    delete p; 
}

//---------------------------------------------------------------------------//
// Default constructor.  Initializes the contained pointer to NULL.  The other
// methods below all check for this case, and refuse to do anything bad like
// dereference it, etc.
//---------------------------------------------------------------------------//

template<class T>
SP<T>::SP()
    : s(NULL) 
{
}

//---------------------------------------------------------------------------//
// Principle constructor.  Bind this smart pointer to the specified T *.  This
// is the usual way to instantiate an SP<T> object.
//---------------------------------------------------------------------------//

template<class T>
SP<T>::SP( T *pp )
{
    s = new SPrep<T>( pp );
}

//---------------------------------------------------------------------------//
// Copy constructor.  Alias one SP<T> to another.  Avoid diabolical behavior
// if the initializer was instantiated with its default constructor.
//---------------------------------------------------------------------------//

template<class T>
SP<T>::SP( const SP<T>& sp )
{
    s = sp.s;
    if (s)
	s->refs++;
}

//---------------------------------------------------------------------------//
// Destructor.  Disconnect from the contained T *.  If we were the last user
// of it, then delete it.
//---------------------------------------------------------------------------//

template<class T>
SP<T>::~SP()
{
    if (s) {
	s->refs--;
	if (!s->refs) delete s;
    }
}

//---------------------------------------------------------------------------//
// Assignment operator.  Cause this SP<T> to alias another.  In order to do
// this, we have to disconnect from the one we have now.  If we were the last
// user, we have to delete it (the old one).  Then alias to the new one, and
// adjust its reference count accordingly.
//---------------------------------------------------------------------------//

template<class T>
SP<T>& SP<T>::operator=( const SP<T>& sp )
{
// Avoid "assignment to self" bogosity.

    if (this == &sp) return *this;

// If we are not using a T * currently, just bind to the new one now.

    if (!s) {
	s = sp.s;
	if (s)
	    s->refs++;
	return *this;
    }

// If we are one of many users of our current pointer, disconnect and alias to
// the new one.

    if (s->refs > 1) {
	s->refs--;
	s = sp.s;
	if (s)
	    s->refs++;
    }

// If we are the last user of our current pointer, delete it, and then alias
// to the new one.

    else {
	delete s;
	s = sp.s;
	if (s)
	    s->refs++;
    }

    return *this;
}

//---------------------------------------------------------------------------//
// Assignment operator.  Bind an existing SP<T> to a T *.
//---------------------------------------------------------------------------//

template<class T>
SP<T>& SP<T>::operator=( T *pp )
{
    if (!s) {
	s = new SPrep<T>( pp );
	return *this;
    }

// Okay, we're already managing a T *, so disconnect from it.

    if (s->refs > 1)
	s->refs--;
    else
	delete s;

    s = new SPrep<T>( pp );
    return *this;
}

//---------------------------------------------------------------------------//
// Destroy an SPLink by deleting each member in the chain.
//---------------------------------------------------------------------------//

template<class T>
SPLink<T>::~SPLink()
{
    delete next;
}

//---------------------------------------------------------------------------//
// Construct a new SPList<T> from an old one.
//---------------------------------------------------------------------------//

template<class T>
SPList<T>::SPList( const SPList<T>& spl )
{
    last = 0;

    if ( spl.last ) {
	SPLink<T> *ohead = spl.last->next;
	SPLink<T> *ofirst = ohead;
	SPLink<T> *nhead = new SPLink<T>( ohead->data() );
	SPLink<T> *nel = nhead;

	while( ohead->next != ofirst ) {
	    ohead = ohead->next;
	    nel->next = new SPLink<T>( ohead->data() );
	    nel = nel->next;
	}

	last = nel;
	last->next = nhead;
    }
}

//---------------------------------------------------------------------------//
// Set this list to equal another list.
//---------------------------------------------------------------------------//

template<class T>
SPList<T>& SPList<T>::operator=( const SPList<T>& spl )
{
    clear();

    if ( spl.last ) {
	SPLink<T> *ohead = spl.last->next;
	SPLink<T> *ofirst = ohead;
	SPLink<T> *nhead = new SPLink<T>( ohead->data() );
	SPLink<T> *nel = nhead;

	while( ohead->next != ofirst ) {
	    ohead = ohead->next;
	    nel->next = new SPLink<T>( ohead->data() );
	    nel = nel->next;
	}

	last = nel;
	last->next = nhead;
    }

    return *this;
}

//---------------------------------------------------------------------------//
// Empty the list by deleting all the entries.
//---------------------------------------------------------------------------//

template<class T>
void SPList<T>::clear()
{
    if (last) {
	SPLink<T> *head = last->next;
	last->next = 0;
	delete head;
	last = 0;
    }
}

//---------------------------------------------------------------------------//
// Insert a new SP<T> at the head of the list.
//---------------------------------------------------------------------------//

template<class T>
void SPList<T>::insert( SP<T>& s )
{
    if (last)
	last->next = new SPLink<T>( s, last->next );
    else {
	last = new SPLink<T>( s );
	last->next = last;
    }
}

//---------------------------------------------------------------------------//
// Append a new SP<T> to the end of the list.
//---------------------------------------------------------------------------//

template<class T>
void SPList<T>::append( SP<T>& s )
{
    if (last) {
	last->next = new SPLink<T>( s, last->next );
	last = last->next;
    }
    else {
	last = new SPLink<T>( s );
	last->next = last;
    }
}

//---------------------------------------------------------------------------//
// Construct an iterator and bind to a particular SPList<T>.
//---------------------------------------------------------------------------//

template<class T>
SPList_iter<T>::SPList_iter( SPList<T>& _spl )
{
    spl = &_spl;
    if (spl->last)
	ce = spl->last->next;
    else
	ce = 0;
}

//---------------------------------------------------------------------------//
// Reset the iterator back to the beginning of the list.
//---------------------------------------------------------------------------//

template<class T>
void SPList_iter<T>::reset()
{
    if (spl->last)
	ce = spl->last->next;
    else
	ce = 0;
}

//---------------------------------------------------------------------------//
// Evaluate the iterator (get data from the list).
//---------------------------------------------------------------------------//

template<class T>
SP<T> SPList_iter<T>::operator()()
{
    if (!ce) {
	if (!spl->last) {
	    throw( "List has no elements." );
	} else { 
	    throw( "Iterator is out of bounds." );
	}
    }
    else
	return ce->data();
}

//---------------------------------------------------------------------------//
// Prefix increment the iterator.
//---------------------------------------------------------------------------//

template<class T>
void SPList_iter<T>::operator++()
{
    if (!ce)
	throw( "Iterator out of bounds." );

    if (ce == spl->last)
	ce = 0;
    else
	ce = ce->next;
}

//---------------------------------------------------------------------------//
// Postfix increment the iterator.
//---------------------------------------------------------------------------//

template<class T>
void SPList_iter<T>::operator++(int)
{
    if (!ce)
	throw( "Iterator out of bounds." );

    if (ce == spl->last)
	ce = 0;
    else
	ce = ce->next;
}

//---------------------------------------------------------------------------//
//                              end of SP.cc
//---------------------------------------------------------------------------//
