//----------------------------------*-C++-*----------------------------------//
// Array.cc
// Geoffrey Furnish
// 28 January 1994
//---------------------------------------------------------------------------//
// @> An Array template class.
//
// $Id: Array.cc,v 1.12 1995/05/05 04:44:55 furnish Exp $
//
// $Log: Array.cc,v $
// Revision 1.12  1995/05/05  04:44:55  furnish
// Constification.
//
// Revision 1.11  1995/05/04  05:34:05  mjl
// Eliminated extraneous header include.
//
// Revision 1.10  1995/05/04  05:29:23  mjl
// Moved indexing operators into the header file.
//
// Revision 1.9  1994/12/06  19:26:57  furnish
// Assert.h now in DS++.
//
// Revision 1.8  1994/12/02  17:46:09  furnish
// Formatting, documentation.  Switch from codoc to autodoc.
//
// Revision 1.7  1994/09/23  20:08:36  furnish
// Added a new aliasing constructor, which should probably have been in a
// long time ago anyway.  Needed right now by the grid orthogonalizer.
//
// Revision 1.6  1994/08/31  19:10:14  furnish
// New method.
//
// Revision 1.5  1994/07/06  18:57:40  furnish
// Bug fixes.
//
// Revision 1.4  1994/06/17  20:09:48  furnish
// Some new methods, and lots comment improvements.
//
// Revision 1.3  1994/06/09  19:56:36  furnish
// Made some better accessors, but had to rename some class variables
// because of Cfront braindeath.
//
// Revision 1.2  1994/04/20  22:03:58  furnish
// Massive flesh out.  Much more mature now.
//
// Revision 1.1  1994/02/08  19:24:44  furnish
// Templated array and dynamically growable array classes.
//
//---------------------------------------------------------------------------//

#include "Array.h"

//---------------------------------------------------------------------------//
// Construct an Array<T> with a given size and possibly with a given
// base (minimum index value).
//---------------------------------------------------------------------------//

template<class T>
Array<T>::Array( int s, int b /*=0*/ ) : sz(s), base_(b)
{
    if (sz < 1) sz = 1;

    v = new T[ sz ];
    v -= base_;

    am_alias = 0;
}

//---------------------------------------------------------------------------//
// Construct an Array<T> with given size, default value, and base.  No
// defaulting allowed here, to avoid problems with ambiguities.
//---------------------------------------------------------------------------//

template<class T>
Array<T>::Array( int sz_, T defval, int b ) : sz(sz_), base_(b)
{
    if (sz < 1) sz = 1;

    v = new T[sz];

    for( int i=0; i < sz; i++ )
      v[i] = defval;

    v -= base_;

    am_alias = 0;
}

//---------------------------------------------------------------------------//
// Construct to be identical to another Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T>::Array( const Array<T>& t )
{
    base_ = t.base_;
    sz = t.sz;

    v = new T[ sz ];

    v -= base_;

    for( int i=base_; i < base_ + sz; i++ )
	v[i] = t.v[i];

    am_alias = 0;
}

//---------------------------------------------------------------------------//
// Construct to alias memory from some other origin.
//---------------------------------------------------------------------------//

template<class T>
Array<T>::Array( T *pt, int s, int b /*=0*/ )
{
    base_ = b;
    sz = s;

    v = pt;

    v -= base_;
    am_alias = 1;
}

//---------------------------------------------------------------------------//
// Destroy an Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T>::~Array()
{
    if (am_alias) return;

    v += base_;			// Get back to what was actually
				// allocated. 

    delete[] v;			// Annihilate them all, one by one.
}

//---------------------------------------------------------------------------//
// Assign an Array<T> from another one.  If not conformal, reinit this one.
//---------------------------------------------------------------------------//

template<class T>
Array<T>& Array<T>::operator=( const Array<T>& t )
{
    Assert( t.v );  // 'at thang'd bettr've been initialized.

    if (base_ != t.base_ || sz != t.sz) {
	Assert( !am_alias );	// Don't free someone else's memory.

	// Does this guy know what he's doing?
	v += base_;
	delete[] v;		// Annihilate them all, one by one.
	base_ = t.base_;
	sz = t.sz;

	v = new T[ sz ];
	v -= base_;
    }

    for( int i=base_; i < base_ + sz ; i++ )
      v[i] = t.v[i];

    return *this;
}

//---------------------------------------------------------------------------//
// Equality operator.
//---------------------------------------------------------------------------//

template<class T>
int Array<T>::operator==( const Array<T>& t ) const
{
    if (base_ != t.base_) return 0;
    if (sz != t.sz) return 0;

    for( int i=base_; i < base_+sz; i++ )
	if (v[i] != t.v[i]) return 0;

    return 1;
}

//---------------------------------------------------------------------------//
// Inequality operator.
//---------------------------------------------------------------------------//

template<class T>
int Array<T>::operator!=( const Array<T>& t ) const
{
    return !operator==(t);
}

//===========================================================================//
// Now for the friend operators.

// Since the type T may have overloaded its operators in arbitrary
// ways, we are careful when combining Arrays, or Arrays and scalars,
// to use the same operators on the elements as specified in the
// equation.  The promotion should only make an Array from a scalar,
// but should not affect the choice of operator.  Stated in
// mathematical terms, we don't assume associativity or commutativity.
// We don't assume that z = x + y is the same as z = x; z += y; since
// the type T may well have overloaded both operator+ and operator+=.
// Nor do we assume that x + y == y + x;

//---------------------------------------------------------------------------//
// Add two Array<T>'s, producing a third.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator+( const Array<T>& x )
{
    Assert( x.sz == sz );
    Assert( x.base_ == base_ );	// Instead of throw(misaligned).

    Array<T> r( sz, base_ );
    
// It looks nice, but is it inefficient?

    for( int i=r.base_; i < r.base_ + r.sz; i++ )
	r[i] = v[i] + x.v[i];

    return r;
}

//---------------------------------------------------------------------------//
// Add a constant to an Array<T>, producing a new Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator+( const T& x )
{
    Array<T> r( sz, base_ );
    
    for( int i=r.base_; i < r.base_ + r.sz; i++ )
	r[i] = v[i] + x;
    
    return r;
}

// template<class T>
// Array<T> operator+( const T& x, const Array<T>& y )
// {
//     Array<T> r( y.sz, y.base_ );

//     for( int i=r.base_; i < r.base_ + r.sz; i++ )
//       r[i] = x + y.v[i];

//     return r;
// }

//---------------------------------------------------------------------------//
// Add an Array<T> into another.
//---------------------------------------------------------------------------//

template<class T>
Array<T>& Array<T>::operator+=( const Array<T>& x )
{
    Assert( x.sz == sz );
    Assert( x.base_ == base_ );

    for( int i=base_; i < base_ + sz; i++ )
	v[i] += x.v[i];

    return *this;
}

//---------------------------------------------------------------------------//
// Subtract an Array<T> from another Array<T>, producing a third.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator-( const Array<T>& x )
{
    Assert( x.sz == sz );
    Assert( x.base_ == base_ );	// Instead of throw(misaligned).

    Array<T> r( sz, base_ );

    for( int i=r.base_; i < r.base_ + r.sz; i++ )
	r[i] = v[i] - x.v[i];

    return r;
}

//---------------------------------------------------------------------------//
// Subract a constant from an Array<T>, producing a new Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator-( const T& x )
{
    Array<T> r( sz, base_ );

    for( int i=r.base_; i < r.base_ + r.sz; i++ )
      r[i] = v[i] - x;

    return r;
}

// template<class T>
// Array<T> operator-( const T& x, const Array<T>& y )
// {
//     Array<T> r( y.sz, y.base_ );

//     for( int i=r.base_; i < r.base_ + r.sz; i++ )
//       r[i] = x - y.v[i];

//     return r;
// }

//---------------------------------------------------------------------------//
// Multipley an Array<T> by a constant, producing a new Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator*( const T& x )
{
    Array<T> r( sz, base_ );

    for( int i=r.base_; i < r.base_ + r.sz; i++ )
      r[i] = v[i] * x;

    return r;
}

// template<class T>
// Array<T> operator*( const T& x, const Array<T>& y )
// {
//     Array<T> r( y.sz, y.base_ );

//     for( int i=r.base_; i < r.base_ + r.sz; i++ )
//       r[i] = x * y.v[i];

//     return r;
// }

//---------------------------------------------------------------------------//
// Divide an Array<T> by a constant, producing a new Array<T>.
//---------------------------------------------------------------------------//

template<class T>
Array<T> Array<T>::operator/( const T& x )
{
    Array<T> r( sz, base_ );
    Assert( x != 0. );

    for( int i=r.base_; i < r.base_ + r.sz; i++ )
	r[i] = v[i] / x;

    return r;
}

//---------------------------------------------------------------------------//
//                              end of Array.cc
//---------------------------------------------------------------------------//
