
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;;      File:     compile.el                                                 ;;
;;      Author:   Wolfgang S. Rupprecht <wolfgang@wsrcc.com>                 ;;
;;      Created:  March 1987                                                 ;;
;;      Contents: I did significant hacking to the error parser              ;;
;;              for next-error.  The parser now has a table that             ;;
;;              it scans for applicable rexexp templates.  If                ;;
;;              one of them fits, it uses that one to parse the              ;;
;;              line. If it doesn't fit, the scanner tries the               ;;
;;              next template. If all templates fail, the line               ;;
;;              is deemed a useless line and discarded.                      ;;
;;                                                                           ;;
;;      Copyright (c) 1989, 1987 Wolfgang Rupprecht.                         ;;
;;                                                                           ;;
;;      $Header$                                                             ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; changes: Lutz Prechelt, 92/06/03
;;            added scan-gmd and Gentle regexp

;; This file is not really part of GNU Emacs, but is (hopefully)
;; useful in conjunction with it.  It is meant as a patch to the
;; distributed GnuEmacs lisp file by the same name.

;; GNU Emacs and this compile.el is distributed in the hope that it
;; will be useful, but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General
;; Public License for full details.

;; Everyone is granted permission to copy, modify and redistribute GNU
;; Emacs, but only under the conditions described in the GNU Emacs
;; General Public License.  A copy of this license is supposed to have
;; been given to you along with GNU Emacs so you can know your rights
;; and responsibilities.  It should be in a file named COPYING.  Among
;; other things, the copyright notice and this notice must be
;; preserved on all copies.

;; First we load the real lib.  The original load-path should be stored
;; in the variable old-load-path.

;(let ((load-path old-load-path))
;  (require 'compile))

;; Now we patch it.

(define-key ctl-x-map "`" 'nxt-error)

;; let's hope that nobody is stupid enough to put a colon or
;; parenthesis in their filenames, (these regexps desperately need to
;; cue off of them) -wsr

;; for forced updating of the defvar definitions
;; (defmacro defvar-f (name val doc) (list 'setq name val))

;;========== Description of regexps
;;0: Gentle, Modula-2*, Mocka
;;      [Error in 'gelati.g']   (M2s: [messages for 'grpf.ms'])
;;      205 15: Error       syntax error
;;      205 15: Information expected tokens: DOT SMALLID LARGEID KEY 
;;1: 4.3bsd lint part2: inconsistant type warnings
;;	strcmp: variable # of args.      llib-lc(359)  ::  /users/wolfgang/foo.c(8)
;;	also sysV lint: from kamat@uceng.uc.edu
;;	seekdir      llib-lc(345) :: uuq.c?(73)
;;2: 4.3bsd lint part3: defined, but unused messages
;;	linthemorrhoids defined( /users/wolfgang/foo.c(4) ), but never used
;;	foo used( file.c(144) ), but not defined
;;3: 4.3bsd compiler
;;	"foo.c", line 18: cc_cardiac_arrest undefined
;;4: apollo cc warnings, yuk -wsr
;;5: as on a sun 3 under sunos 3.4
;;	(as) "spl.i", line 23:  Error:  syntax error.
;;6: m88kcc
;;	"./foo.h" , line 128: redeclaration of bar
;;	note the extra space before the comma (after filename) : grotty
;;7: Make
;;	Make: line 20: syntax error.  Stop.
;;	Make: Must be a separator on rules line 84.  Stop.
;;8: /bin/sh 
;;	./binshscript: syntax error at line 5: `newline or ;' unexpected
;;	filename can only be parsed correctly if it is a full pathname, or
;;	is relative to this directory.
;;9: sysV woes
;;	rcmd         cico.c?(243)
;;10: sysV lint - "Reach out and confuse someone."
;;	cico.c
;;	==============
;;	(88)  warning: alias unused in function main
;;	(656)  warning: main() returns random value to invocation environment
;;	cntrl.c:
;;	 
;;	uucpd.c
;;	==============
;;	warning: argument unused in function:
;;	    (48)  argc in main
;;	warning: possible pointer alignment problem
;;	    (145)            (246)           (329)  
;;	    (367)        
;;	note: This regexp has to be incredibly weak.  There just isn't much
;;	to get a toe-hold on here.  Better keep this one on the end. -wsr
;;11: perl 3.0
;;	syntax error in file findman at line 34, next 2 tokens ") {"
;;12: 4.3bsd grep, cc, lint(part1 warnings)
;;	/users/wolfgang/foo.c(8): warning: w may be used before set
;;13: C-Refine 3.0
;;      "crefine.cr", line 330, warning: refinement unused
;;14: Eli-generated compilers
;;      "backprop.nn", line 207:11 FATAL: Syntax error

(defvar error-parsing-regexp-list
'(
;;0:
    ("^ *\\([0-9]+\\),? *\\([0-9]+\\)[ :]" scan-gmd 1 nil nil 2)
;;14:
    ("^\"\\([^\"\n]+\\)\", line \\([0-9]+\\)[:,]\\([0-9]+\\)" 1 2 nil nil 3)
;;1:
;;    ("^[^\n]*[ \t]+\\([^:( \t\n]+\\)[:(]+[ \t]*\\([0-9]+\\)[:) \t]+\\([^:?( \t\n]+\\)\\??[:(]+[ \t]*\\([0-9]+\\)[:) \t]+$"  3 4 1 2)
;;    ("[^\n]*[ \t:]+\\([^:( \t\n]+\\)[ \t]*[:(]+[ \t]*\\([0-9]+\\)[:) \t]*$"     1 2)
  ("^oweiraldfqoeiruldkafjg$" 1 2)  ;;nothing
;;2:
    ("[^\n]*\\(defined\\|used\\)[ \t(]+\\([^:( \t\n]+\\)[:( \t]+\\([0-9]+\\)[:) \t]+"  2 3)
;;3:
    ("^[\* \t]*[Ll]ine[ \t]+\\([0-9]+\\)[ \t]+of[ \t]+\"\\([^\"\n]+\\)\":" 2 1)
;;4:
    ("^[\* \t]*\"\\([^\"\n]+\\)\",?[ \t]+[Ll]ine[ \t]+\\([0-9]+\\):" 1 2)
;;5:
    ("^(.+)[ \t]+\"\\([^\"\n]+\\)\",[ \t]+line[ \t]+\\([0-9]+\\):" 1 2)
;;6:
    ("^\\((.+)[ \t]+\\)?\"\\([^\"\n]+\\)\" ?,[ \t]+line[ \t]+\\([0-9]+\\):" 2 3)
;;7:
    ("^[\* \t]*[Mm]ake: [^\n]*[Ll]ine[ \t]+\\([0-9]+\\)[.:]"  scan-make 1)
;;8:
    ("^\\([^:\n]+\\):.*line[ \t]+\\([0-9]+\\):" 1 2)
;;9:
    ("^    [^: \t\n]+ +\t\\([^:?( \t\n]+\\)\\??(\\([0-9]+\\))$" 1 2)
;;10:
    ("^[ \t]*(\\([0-9]+\\))[ \t]" scan-s5lint 1)
;;11:
    ("^syntax error in file \\([^ \t\n]+\\) at line \\([0-9]+\\)" 1 2)
;;12:
  ("^\\([^:( \t\n]+\\)[:( \t]+\\([0-9]+\\)[:) \t]" 1 2)
;;13:
  ("^\"\\([^\"]\\)*\", line \\([0-9]+\\), .*[error|warning]: " 1 2)
;;14:
;;15:
;;16:
)
  "a list of lists consisting of:
\((rexexp filename-match-index linenum-match-index 
         [f2-m-idx l2-m-idx [column-match-index [c2-m-idx]]])(...)(...))
for parsing error messages
f2/l2 indexes are for a second file and may be nil")

(defun test-parse (pos)
  "Test the line parsing code, attempts to parse the current line for
filename and line number. Answer is returned in minibuffer."
  (interactive "d")
  (forward-line 0)
  (let (filename linenum filename-2 linenum-2) ; set by compilation-parse-line
    (let ((parserule (compilation-parse-line)))
      (if parserule
          (if filename-2
              (message "Parses as: '%s(%d)' and '%s(%d)' [rule %d]"
                       filename linenum
                       filename-2 linenum-2 parserule)
            (message "Parses as: '%s(%d)' [rule %d]"
                     filename linenum parserule))
        (message "Couldn't parse that line")))))

(defvar compilations-window-height 10 "*Height of compilations buffer window.")

;; parse error msgs, find file or files, and position cursor on the
;; appropriate lines.
;; The "primary" file is always at the bottom of the screen.
;; The *compilations* buffer is always at the top, and reduced to
;;  a smaller height.

;;----- the following variables are used to handle the compilation buffer:

(setq last-compilation-buffer nil)

(defvar last-error-buf nil
  "The last buffer that next-error used.")

(defvar last-compbuf-linenum nil
  "The last line in the compilation buffer that was parsed by next-error")

(defvar last-error-linenum nil
  "The last line no in an error message parsed by next-error")

(defvar last-error-filename nil
  "The last filename in an error message parsed by next-error")

;;-----

(defun nxt-error (&optional flag)
  "This is the *new* NEXT ERROR: Visit next compilation error message
and corresponding source code.  This operates on the output from the
\\[compile] command.  This command uses the line after current point
as the starting point of the next error search.

If optional FLAG is set (C-u for interactive), cause the current
buffer to be the new compilation buffer."
  (interactive "P")
  (if flag
      (progn
        (setq last-error-buf (current-buffer))
        (setq last-compbuf-line-no nil)
        (setq last-error-line-no nil)
        (setq last-error-filename nil)))
  (pop-to-buffer (or last-error-buf last-compilation-buffer
		     "*compilation*"))
  (let ((opoint (point))
        (pwd default-directory)
        filename linenum filename-2 linenum-2)
    ;; note: compilation-parse-line will set the above 4 variables
    ;; by side effect.
    (while (and (setq last-compbuf-point (point))
                (zerop (forward-line 1))
                (null (compilation-parse-line))))
    (setq last-error-filename filename)
    (setq last-error-linenum  linenum)
    (if (null filename)
	;; this error will leave on in the compilation buffer.
	;; usually this is of benefit - one can now move the up
	;; point back up to bet back to the last error of interest.
        (error (concat compilation-error-message
                       (if (and compilation-process
                                (eq (process-status compilation-process)
                                    'run))
                           " yet" "")))
      (recenter 0)
      (if (file-exists-p filename)
          (progn
            (delete-other-windows)
            (let ((wh (window-height)))
              (find-file-other-window filename)
              (shrink-window (- compilations-window-height (/ wh 2)))
              )
            (goto-line linenum)
            (and colnum (forward-char (- colnum 1)))
            (recenter (/ (window-height) 2))
            (if filename-2              ; a two file match
                (let ((default-directory pwd)) ; get the pwd right!
                  (if (file-exists-p filename-2)
                      (progn
                        (split-window-vertically nil)
                        (find-file filename-2)
                        (goto-line linenum-2)
                        (and colnum-2 (forward-char (- colnum-2 1)))
                        (recenter (/ (window-height) 2))
                        ;; now back to file # 1
                        (other-window 1)
                        ;; needed if both windows are on the same file
                        (recenter (/ (window-height) 2)))
                    (message "Can't find file '%s(%d)'"
                             filename-2 linenum-2)))))
        ;; try filename-2 ... suggested by kamat
        (if filename-2
            (if (file-exists-p filename-2)
                (progn
                  (message "Can't find file '%s(%d)'" filename linenum)
                  (delete-other-windows)
                  (let ((wh (window-height)))
                    (find-file-other-window filename-2)
                    (shrink-window (- compilations-window-height (/ wh 2))))
                  (goto-line linenum-2)
                  (and colnum-2 (forward-char (- colnum-2 1)))
                  (recenter (/ (window-height) 2)))
              (error "Can't find files '%s(%d)' or '%s(%d)'" 
                     filename linenum filename-2 linenum-2))
          (error "Can't find file '%s(%d)'" filename linenum))))))

(defun compilation-parse-line ()
  "Parse this line, setq-ing filename, linenum, colnum.
If the line is no error line, return nil"
  (let ((parse-list error-parsing-regexp-list)
        (rule-num 0))
    (while parse-list
      (let ((rule-list (car parse-list)))
	;;(message "%s" rule-list)  ;; for debugging purposes
        ;;(sit-for 5)  ;; look at potentially illegal regexps
        (if (looking-at (car rule-list))
            (let ((file-index (nth 1 rule-list))
                  (line-index (nth 2 rule-list))
                  (file-2-index (nth 3 rule-list))
                  (line-2-index (nth 4 rule-list))
                  (col-index    (nth 5 rule-list))
                  (col-2-index  (nth 6 rule-list)))
              (setq linenum (string-to-int
                             (buffer-substring (match-beginning line-index)
                                               (match-end line-index))))
              (if col-index
                (setq colnum (string-to-int
                               (buffer-substring
                                   (match-beginning col-index)
                                   (match-end col-index))))
                (setq colnum nil))
              (if col-2-index
                (setq colnum-2 (string-to-int
				 (buffer-substring
				    (match-beginning col-2-index)
                                    (match-end col-2-index))))
                (setq colnum-2 nil))
              (if file-2-index
                  (progn
                    (setq filename-2 (buffer-substring
                                      (match-beginning file-2-index)
                                      (match-end file-2-index)))
                    (setq linenum-2 (string-to-int
                                     (buffer-substring
                                      (match-beginning line-2-index)
                                      (match-end line-2-index))))))
              (setq filename
                    (cond ((integerp file-index)
                           (buffer-substring (match-beginning file-index)
                                             (match-end file-index)))
                          ;; careful! this next funcall may mash
                          ;; the match-data, so it must be done
                          ;; after all the line numbers and names have been
                          ;; extracted
                          ((symbolp file-index) (funcall file-index))
                          ((stringp file-index) file-index)
                          (t (error "Parsing error: unknown action type: %s"
                                    file-index))))
              (setq parse-list nil))    ;we're done
          (setq parse-list (cdr parse-list)
                rule-num (1+ rule-num)))))
    (if (and (equal last-error-filename filename)
             (equal last-error-linenum  linenum)
             (<=    last-compbuf-point  (point)))
       nil  ; if another error on same line: don't talk about it
       (and linenum filename rule-num)   ; return matching rule number
    )
  )
)

(defun scan-make ()
  "Attempt to find the name of the Makefile used by this make run.
This routine shouln't be used for anything drastic, since it just isn't
that robust."
  (cond ((save-excursion
           (re-search-backward "make[^\n]+-f[ \t]+\\(\\sw\\|\\s_\\)+" nil t))
         (buffer-substring (match-beginning 1)(match-end 1)))
        ((file-exists-p "makefile") "makefile")
        ((file-exists-p "Makefile") "Makefile")
        (t nil)
      ))

(defun scan-gmd ()
  "Attempt to find the name of the Gentle, Gelati, or Modula-2* file or 
other GMD-tool used by this run by searching backwards for
[Error in 'file.g'] or the like or for 
[messages for 'file.ms'] as used by Modula-2*"
  (cond ((save-excursion
           (re-search-backward 
              "^\\[\\(Error in\\|messages for\\) '\\([^']+\\)'\\]" nil t))
         (let ((name (buffer-substring (match-beginning 2)(match-end 2)))
               lname
              )
            (setq lname (concat name "l"))
            (cond ((file-exists-p lname) lname)
                  (t name))
         ))
        (t "<unknown_file>")
  ))

(defun scan-s5lint ()
  "Attempt to find the name of the file that lint was griping about on
this line.  This routine also has the side-effect of modifying the current
buffer.  The current line will have the first gripe of a multi-gripe line 
broken off onto a separate line."
  (let (retval)
    (if (save-excursion
          (re-search-backward "^\\(\\sw\\|\\s_\\|\\s.\\)+\n======+$" nil t))
        (progn
          (setq retval (buffer-substring (match-beginning 1)(match-end 1)))
          (save-excursion
            (if (re-search-forward ")[ \t]*("
                                   (save-excursion (end-of-line) (point)) t)
                (replace-match ")\n(")))))
  retval))

