/*
 * DebugFn.c
 *
 * Implementation of the debug functions for Modula-to-C programs
 * (Basic functions, without graphics)
 *
 * Stefan Haenssgen  10-Jul-92
 *
 * Updates:
 *
 *	10-Jul-92	First rough implementation:
 *			Implementation of DebugInitialize, DebugEnd,
 *			 NewNode, KillNode, KillTree
 *			Test witn own main()
 *			Wrote PrintTree() to get simple tree output
 *	11-Jul-92	Wrote PrintNode for better output
 *			Added TypeString[] and ModeString[]
 *			1st implementation of DebugFn{Start,Update,End}()
 *			Added Depth to nodes for better printing etc
 *			Test environment with nested loops / ifs  etc
 *			Defined ProfileNode to contain & collect
 *			 profiling information that lasts longer than
 *			 one instance of a DebugNode (e.g. total number
 *			 of calls etc)
 *			Wrote InitializeProfileTable, AddProfileNode,
 *			 KillProfileNode, KillProfileTable, PrintProfileTable
 *			 to handle profiling information
 *			Added calls to DebugFn{Start,End} to update profile
 *			 info (create p-node at start, if not already there,
 *			 and update info from d-node into p-node at end)
 *	12-Jul-92	Introduced context (sequential, parallel, synchronous)
 *	16-Jul-92	Added handling of parallel & synchronous contexts
 *			Wrote an extra test program in Modula and added
 *			 debug function calls by hand
 *	17-Jul-92	Dropped DN_IF
 *			Added Visualizer calls for visualization of activities
 *			 inside synchronous foralls (not much use for parallel
 *			 foralls since just one is active at a time in the
 *			 simulator)
 *			PrintTree(NULL) gives the whole tree
 *	21-Jul-92	Created ShowTree() to display the DebugNode tree
 *			 using X Windows
 *			Imported the Tree Widget
 *			Wrote ShowTreeAddNode() to recursively add DebugNodes
 *			 to the tree to be shown
 *			Wrote ShowTreeCallback() as callback function to pop
 *			 up further information when node is klicked
 *			Test with testm.c - works nicely!
 *	22-Jul-92	Implemented ShowTreeCallback, use popup window that
 *			 displays information and pops down when clicked
 *			Made topW global variable for access in S.T.Callback
 *			Positioned info widget directly over calling node
 *			 button (for fast click-up click-down)
 *			Optimized the layout / borders etc
 *			Quit ShowTree when Root is clicked
 *			Examine the event type to determine if done (i.e.
 *			 the top level widget is destroyed)
 *	23-Jul-92	Wrote GetProfileNode() to find a node containing
 *			 the given line & type
 *			Include the RunCount in debug node info popup
 *			Modified AddprofileNode to init the run count to
 *			 the one of the given debug node, not just 0
 *	30-Jul-92	Split ShowTree() into ShowTree() to create an
 *			 Application Context and do the Event Loop and
 *			 ShowTreeWidget() to create the Popup Widget and
 *			 display the Tree
 *			Wrote ShowProfileWidget() to display the profiling
 *			 information using an AsciiText Widget
 *			insert-string() doesn't work as it should, use own
 *			 tmp buffer for text building
 *			Introduced PROFILE_MAX_{WIDTH,HEIGHT} for sizing it
 *			Introduced maxProfileLine to remember the
 *			 actual line range inside the hash table
 *			Output information in correct sequence of lines
 *			 (not in the hashed form where different lines
 *			 are in the same chain of entries)
 *			Debugged AddprofileNode: Also watch out for type,
 *			 don not just compare line numbers!
 *	07-Aug-92	Print Debug Tree as list with PrintTreeAsList(),
 *			 no explicit pointers, all such info is already in
 *			 depth & sequence of the nodes
 *			Wrote BuildTreeFromList() to reassemble the tree
 *			Changed end condition of ShowTree: Depth == 0
 *			 instead of node == RootNode
 *			Use relative Widget addressing in ShowTreeCallback
 *			 (no more topW globally necessary!)
 *			End node list with line full of "0" for better
 *			 parsing (recognition of end-of-tree-list)
 *			 [because xdbx adds lines to the call]
 *			Print Profile with PrintProfileAsList(),
 *			 reassemble it with BuildProgileFromList()
 *			Added Quit button to Profile Widget, wrote
 *			 ShowProfileCallback() to handle it
 *	12-Aug-92	Added MColumn for the column in which the construct
 *			 starts, to be unique (line+type is not enough!)
 *			Added MToLine & MToColumn to denote the position of
 *			 the end of the Modula-2* construct in the source
 *			Also changed ProfileNode accordingly.
 *			Changed KillProfileNode to take line&column instead
 *			 of line&type. Also changed GetProfileNode
 *			Modified AddProfileNode: RunCounter values >0 are
 *			 summed up as usual, but values <0 overwrite the
 *			 former value with their absolute value (purpose:
 *			 reset counters without destroying the node)
 *			Changed BuildProfileFromList() to override the
 *			 old values using the mechanism above and not to
 *			 kill the whole profile before updating it
 *	14-Aug-92	Added {profile,tree}_active to mark number of
 *			 profiles/trees being displayed
 *			Modified init-function and popup/popdown to update
 *			 those counters
 *			Moved the profile table text output code to
 *			 MakeProfileText()
 *			Wrote UpdateTree() and UpdateProfile() to handle
 *			 updates of the Tree / Profile displays without
 *			 popping them up again
 *			Debugging... hmmm... keeps dumping core
 *			Also not nice: tree generated by UpdateTreeWidget()
 *			 seems to be flat / wrong layout
 *			Solution: kill the whole tree widget and place a
 *			 new one there. Also resize it to fit the new tree!
 *			Better layout using ResizeWidget / QueryGeomenty
 *			Use paned Widget instead of box for profile info
 *	19-Aug-92	Use goto (yuck) to skip small lines in
 *			 Build{Profile,Tree}FromList(), adjusted counters
 *			Further debugging... argh! wrong if (newroot==NULL !!)
 *	24-Aug-92	Introduced GetLastNode() to get the most recent
 *			 node of the given Debug Tree - use straight line
 *			 of children (yuck but ok for sequential list ;-)
 *			Debugged MakeProfile: Init string with 0 before use
 *			Debugged PrintProfileAsList: was wrong format :-/
 *	25-Aug-92	Resize tree info buttons automatically to display
 *			 the whole text (XtQueryPreferredGeometry etc)
 *			Resize the profile widget on updates
 *	02-Sep-92	Introduced a checksum that changes when the
 *			 Tree chenges, thus saving redraws when nothing
 *			 has happened in the meantime
 *			Wrote GetChecksum() to get the value of it
 *	04-Sep-92	Changed used Visualizer to VisualizeActivation()
 *	22-Oct-92	Moved ProfileTable into ADT ProfileTableType
 *			Wrote CreateProfileTable() and DestroyProfileTable()
 *			Removed InitializeProfileTable()
 *			Changed KillProfileTable() to ClearProfileTable()
 *			Added new KillProfileTable() to clear & deallocate it
 *			Changed BuildProfileFromList() to AddListToProfile()
 *			 (with possibility of adding or replacing counters)
 *			Added global Profile Table PTable (to let functions
 *			 like DebugInitialize() and all Graphics still work,
 *			 until all is done using the ADT) !!!!!!!
 *			Debugged GetProfileNode (would return if line OR
 *			 column were OK, not necessarily both)
 *	25-Oct-92	Added ProfileAsString() to return output of
 *			 PrintProfileAsList() as string
 *	26-Oct-92	Split into DebugFn.c (basic functions without graphics)
 *			 and DebugFnV.c (additional Visualizer / X11 stuff)
 *			Renamed TypeString[] to _D_TypeString as we have to
 *			 export it to DebugFnV.c - Dito _D_ContextString
 *	27-Oct-92	Packed DebugNodes into ADT DebugTree, made DTree
 *			 the global Debug Tree, moved global variables into it
 *			Replaced checksum by NumC (#node creations) and NumK
 *			 (#node kills) to be 100% reliable [NumC or NumK
 *			 different -> Tree has changed]
 *			Adapted functions accordingly
 *			Moved PrintTree() to PrintSubTree(), added new
 *			 PrintTree()
 *			Wrote CreateDebugTree() and KillDebugTree()
 *			All DebugFnXxx() calls use global tree "DTree" !!!
 *			Changed PrintTreeAsList() to TreeAsString(),
 *			 wrote new PrintTreeAsList()
 *			Added AddListToTree() ??? SEMANTICS ???
 *			Created KillSubTree(), new KillTree()
 *	28-Oct-92	Hunted some obscure bug that caused TreeAsList()
 *			 to duplicate some lines and forget others ...
 *			 replaced realloc() by malloc() & free() ...
 *			Wrote an iterative version of TreeAsList() using
 *			 the fact that node->Parent->Child == node if
 *			 we're back at the 1st (or only) child
 *	29-Oct-92	Debugged Profile Table (handle context correctly)
 *			Implemented semantics for AddListToTree:
 *			 tree given as string gets added to the given tree
 *			 as child of the given node (or root if none)
 *			Added start ID to BuildTreeFromList() to make
 *			 merging of trees easier (unique IDs !!)
 *			Wrote InsertSubTree() to insert a subtree (given
 *			 by its root node) into a given tree
 *	30-Oct-92	Improved ID handling when adding subtree
 *			Set tree to NULL when killed
 *			Adjust depth in TreeAsString to make depth of
 *			 subtree's root = 0, wrote AdjustDepths()
 *			Add an artificial root note in TreeAsString() if
 *			 the subtree doesn't start at the real root node
 *			 (to make conversion back to a tree easier)
 *			Added a call to GetLastNode() to BuildTreeFromList()
 *			 to have a valid last node entry
 *			Put additional info (ID,LastNode,NumK,NumC) into
 *			 tree-as-list-string
 *			When rebuilding in BuildTreeFromList(), look for
 *			 LastNode but take deepest node if LastNode not in
 *			 the given (sub)tree.
 *			Also output LastNode etc info in PrintTree()
 *	31-Oct-92	Replaced "List" in names with "String" (more appro-
 *			 priate), scratched "Type" from "ProfileTableType"
 *			Replaced DebugFnUpdate by macro (faster!)
 *	04-Nov-92	Wrote MergedTree() which merges two trees into
 *			 one until the 1st difference is encountered,
 *			 then appends the subtrees
 *			Wrote BuildAndMergeNodes() to merge subtrees (does
 *			 NOT check for permutations, nodes have to be in
 *			 same order to be matched correctly !!!)
 *			Wrote CopyNode() to copy an existing node as
 *			 new child of the given parent
 *			Also pass parent to BuildAndMergeNodes()
 *			Removed tree parameter from PrintNode()
 *	06-Nov-92	Wrote GroupTree() to join equivalent nodes in tree
 *			 (easier than MergedTree)
 *			Wrote GroupSubTree() for nodes
 *			Introduced macro EquivNodes(n1,n2)
 *			Wrote JoinNodes() to join brother nodes
 *	08-Nov-92	Added calls to handle the newly introduced NumEntries
 *			 in the Profile Table (inc if new, dec if kill, etc)
 *			Wrote MakeProfileStat() to make a sorted array of
 *			 Profile Nodes from a Profile Table
 *			Wrote ProfileStatAsString() to output this array
 *	25-Nov-92	Introduced NodeAsString() and BuildNodeFromString()
 *			 to convert single nodes to/from string
 *			Added PrintNodeAsString, which also frees the string
 *			Added GimmeData() to write data to process
 *			Added DumpToFile() to dump data into file (formerly
 *			 "generate_dump()")
 *	29-Nov-92	Introduced DebugNode->ProfCountP to directly update
 *			 profiling counts
 *			Modified AddProfileTable() to set ProfCountP in node
 *			Init pointer to NULL & check in macro if defined
 *			Don't update profile info in DebugFnEnd any more,
 *			 already done while running through the Updates
 *	10-Jan-93	Added JoinInfo to DebugNodes to remember infos
 *			 that would otherwise be lost when joining Nodes
 *			 (RunCount of every node, PE, etc)
 *			Added code to handle this info, added NewJoinInfo(),
 *			 AddJoinInfo() and AppendJoinInfo()
 *			[Don't handle things in BuildAndMerge (outdated)]
 *			Also remember ID of joined node, check for duplicate
 *			 entries in join list, allow adding to empty list
 *	13-Jan-93	Added PE initialization to DebugInitialize()
 *			Always inherit PE from patent, save PE in JoinInfo when
 *			 joining trees
 *			Modified all functions (merge, create,string,...)
 *	04-Feb-93	Also clear node->Joined in BuildTreeFromString()
 *	23-Feb-93	Renamed GroupTree() etc to AllGroupTree() etc, because
 *			 they also group nodes with different children
 *			Wrote new GroupTree() to group only nodes whose
 *			 children are (recursively) equivalent
 *			Wrote EquivSubTree() to recursively determine eq. trees
 *			Moved old PrintSubtree() to PrintSubtreeBrothers()
 *			 and introduced new PrintSubTree which doesn't print
 *			 the given nodes' brothers
 *			In JoinNodes(), take care that the 1st child of
 *			 a parent stays the 1st child (swap nodes if necess.)
 *			 [necessary because many recursive functions test
 *			  if node == node->parent->child to detect termination]
 *	24-Feb-93	New approach: Start at bottom of tree and work
 *			 upwards, joining equivalent brothers. Use
 *			 parent->Child for comparision to avoid cmp with
 *			 node that might have been "joined away"
 *			Fixed PrintSubTreeBrothers() to call iself, not P.S.T()
 *			Adapted JoinNodes() to recursively join all children
 *			 (since it's only called when both nodes are eq.
 *			  i.e. children match, too)
 *			Moved former JoinNodes() to AllJoinNodes() for use
 *			 with AllGroupTree etc
 *			New global flag TreePrintStyle (TPS_NORMAL,TPS_POINTER,
 *			 TPS_VERBOSE, TPS_SHORT) for PrintNode, wrote function
 *			 SetPrintStyle to change this style, set default=NORMAL
 *
 */

/* For GimmeData() -> ptrace() */

#include <sys/wait.h>
#include <sys/ptrace.h>
#include <errno.h>
extern char *sys_errlist[];
extern int  errno;

#include <stdio.h>
#include <string.h>
#include "DebugFn.h"


/* Maximum depth of tree (for BuildTreeFrom) */

#define MAXDEPTH	256


/* Global variables (yuck ;-) */

DebugTree		*DTree;		/* Global Debug Structure Tree	*/
ProfileTable	 	*PTable;	/* Global Profile Info Table	*/
int			 TreePrintStyle	/* Style for node output	*/
                         = TPS_NORMAL;


/* For Display stuff (but also initialized here!) */

int    			tree_active;	/* # of currently active tree views  */
int    			profile_active;	/* Dito # of profiles being displayed*/



/* Names of node-types (in same sequence as types in DebugFn.h)	*/

char *_D_TypeString[] = {
	"",				/*  0 */
	"Root",
	"","","","","","","","",
	"While",			/* 10 */
	"Repeat",
	"Loop",
	"For",
	"","","","","","",
	"ForallSyn",			/* 20 */
	"ForallPar",
	"","","","","","","","",
	"",				/* 30 */
	"Then",
	"Else",
	"Case",
	"Of",
	"","","","","",
	"Funct"				/* 40 */
};


/* Names of contexts (in same sequence as contexts in DebugFn.h)	*/

char *_D_ContextString[] = {
	"",				/*  0 */
	"SEQ",
	"PAR",
	"SYN"
};


/*
 * DebugNode *NewNode(tree,type,parent,
 *                    mline,mcolumn,mtoline,mtocolumn,
 *                    actvalue,actarray,
 *                    rangevar,rangelo,rangehi)
 *
 * Creates a new Node and fills in the given information.
 * Also assigns a new and unique (for that tree) ID to that node.
 * Returns a pointer to the new node.
 * Updates the LastNode variable of the tree.
 *
 * NOTES: - of course a new node has no children (yet)
 *        - the unique ID is automagically generated
 *        - sets depth of node = depth of parent + 1 (if exists, else 0)
 *        - the new node is inserted to the right of the 1st child (if any)
 *
 */
DebugNode *NewNode(tree,type,parent,
                   mline,mcolumn,mtoline,mtocolumn,
                   actvalue,actarray,
                   rangevar,rangelo,rangehi)
DebugTree	*tree;		/* Tree for this node			*/
int		type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
DebugNode	*parent;	/* Parent of this node			*/
long		mline;		/* Line number on Modula-2* Source	*/
long		mcolumn;	/* Dito Column				*/
long		mtoline;	/* End position of current construct	*/
long		mtocolumn;	/* ...dito				*/
long		actvalue;	/* Current activation value		*/
void		*actarray;	/* Array of activation values		*/
void		*rangevar;	/* Variable e.g. of FOR or FORALL	*/
long		rangelo;	/* Range of Variable e.g. for FORALL	*/
long		rangehi;	/* Dito					*/
{
    DebugNode *node;

    node = (DebugNode*) malloc (sizeof(DebugNode));

    node->Type = type;
    node->ID   = tree->ID;		/* Get unique ID		*/

    (tree->ID)++;
    (tree->NumC)++;			/* Update #created (checksum)	*/

    node->Parent  = parent;

    if (parent == NULL) {		/* e.g. RootNode has parent NULL*/

      node->LBrother = node;
      node->RBrother = node;
      node->Depth = 0;
      node->Context = DN_SEQ_CONTEXT;	/* Start in sequential context	*/
      node->PE = 0;			/* PE has to be set externally!!*/

    } else {				/* We do have a parent node	*/

      if (parent->Child == NULL) {	/* We're the 1st child!		*/

        parent->Child  = node;		/* (and thus our own brother :)	*/
        node->LBrother = node;
        node->RBrother = node;

      } else {				/* We already have brothers	*/

        DebugNode *rb = parent->Child;	/* Get our right brother	*/
        DebugNode *lb = rb->LBrother;	/* Get our new left brother	*/
        rb->LBrother   = node;
        node->RBrother = rb;		/* Link to right brother...	*/
        lb->RBrother   = node;
        node->LBrother = lb;		/* ...and to left one		*/

      };

      node->Depth = parent->Depth +1;	/* Set new depth		*/
      node->Context = parent->Context;	/* Copy context	from parent	*/
      node->PE = parent->PE;		/* Also copy PE			*/

    };
    node->Child = NULL;			/* No children (yet)		*/

    node->Joined = NULL;		/* No join info (yet)		*/

    node->MLine    = mline;		/* Copy data			*/
    node->MColumn  = mcolumn;
    node->MToLine  = mtoline;
    node->MToColumn= mtocolumn;
    node->ActValue = actvalue;
    node->ActArray = actarray;

    node->RangeVar = rangevar;
    node->RangeLo  = rangelo;
    node->RangeHi  = rangehi;

    node->RunCount = 0;			/* Init run count for profiles	*/

    node->ProfCountP = (long*) NULL;	/* Init pointer to profile ctr	*/

    
    /* If FORALL, change the context (otherwise keep the parent's one)	*/

    if (type == DN_PAR_FORALL)
        node->Context = DN_PAR_CONTEXT;	/* Set parallel context		*/
    if (type == DN_SYN_FORALL)
        node->Context = DN_SYN_CONTEXT;	/* Set synchronous context	*/


    tree->LastNode = node;		/* Remeber most recent node	*/
    return(node);
}


/*
 * JoinInfo *NewJoinInfo(long ID, long RunCount, int PE)
 *
 * Generates a new JoinInfo node containing the given information
 *
 */
JoinInfo *NewJoinInfo(ID,RunCount,PE)
long ID;
long RunCount;
int PE;
{
    JoinInfo *ji = (JoinInfo*)malloc(sizeof(JoinInfo));

    ji->ID	 = ID;
    ji->RunCount = RunCount;
    ji->PE       = PE;
    ji->Next     = NULL;		/* No successor yet		*/

    return(ji);
}


/*
 * void AddJoinInfo(long ID, long RunCount, int PE, JoinInfo **chain)
 *
 * Adds a JoinInfo node containing the given information to the existing
 * JoinInfo node chain starting at "chain"
 * Don't add node if ID is already in chain!
 * NOTE: JoinInfo ** !! i.e. call with "&(DNode->Joined)" etc!
 *
 */
void AddJoinInfo(ID,RunCount,PE,chain)
long	  ID;
long	  RunCount;
int	  PE;
JoinInfo**chain;
{

    if (chain==NULL)			/* Check for bad input		*/
      return;

    if (*chain==NULL) {			/* 1st in chain? Create new!	*/
	*chain = NewJoinInfo(ID,RunCount,PE); /* Remember new info	*/
    } else {
	JoinInfo *ji = *chain;		/* Go to end of chain, checking	*/
					/*  if ID is already there	*/
	while (ji) {
	    if (ji->ID == ID)		/* Don't add if ID already there*/
	      return;
	    ji = ji->Next;
	};
	ji->Next = NewJoinInfo(ID,RunCount,PE); /* Add new info to end	*/
    };
}


/*
 * void *AppendJoinInfo(JoinInfo *j1, JoinInfo *j2)
 *
 * Appends JoinInfo chain starting at "j2" to end of chain "j1"
 *
 * NOTE: Same node ID CANNOT be in two chains at once, so don't check
 *       check for that case...
 *
 */
void *AppendJoinInfo(j1,j2)
JoinInfo *j1,*j2;
{
    if ((j1==NULL) || (j2==NULL))
      return;				/* Don't join empty chains	*/

    while (j1->Next)
      j1 = j1->Next;			/* Go to end of chain		*/

    j1->Next = j2;			/* Append other chain		*/
}


/*
 * DebugNode *CopyNode(DebugTree *tree, DebugNode *parent, *node)
 *
 * Copy the contents (not children etc) of an existing node into a
 * new node in the given tree with the given parent, resulting in
 * a new child for this parent (brothers are adjusted, if any).
 * The new node is returned.
 * Note: The LastNode pointer of the tree is NOT changed (but the
 *       tree's ID is)!
 */
DebugNode *CopyNode(tree,parent,node)
DebugTree *tree;
DebugNode *parent;
DebugNode *node;
{
    DebugNode *last;			/* Remember old LastNode	*/
    DebugNode *newnode;			/* Newly created Node		*/

    if ((tree == NULL) || (parent == NULL) || (node == NULL))
      return(node);

    last = tree->LastNode;
    newnode = NewNode(tree,			/* Add node below parent*/
		      node->Type, parent,
		      node->MLine, node->MColumn,
		      node->MToLine, node->MToColumn,
		      node->ActValue, node->ActArray,
		      node->RangeVar, node->RangeLo, node->RangeHi);
    newnode->RunCount = node->RunCount;		/* Copy Counter + Context */
    newnode->Context = node->Context;		/* (depth is computed,OK) */
    newnode->PE = node->PE;			/* Also copy PE		  */
    tree->LastNode = last;			/* Restore last node	  */
    return(newnode);

}


/*
 * void PrintJoinInfo(JoinInfo *ji)
 *
 * Output list of join infos, if any
 *
 */
void PrintJoinInfo(ji)
JoinInfo *ji;
{
    while (ji) {
	printf(" ID%d,c%d,P%d",ji->ID,ji->RunCount,ji->PE);
	ji=ji->Next;
    };

}


/*
 * void SetPrintStyle(int style)
 *
 * Sets print style to any of the TPS_xxx constants
 *
 */
void SetPrintStyle(style)
int style;
{

    TreePrintStyle = style;	/* No error checking, PrintNode handles it */
}


/*
 * void PrintNode(DebugNode *node)
 *
 * Prints the given node, indented by the depth given in the node
 * using the globally set print style
 *
 */
void PrintNode(node)
DebugNode *node;
{
    int       i;

    if (node==NULL)
      return;
    
    for (i=0; i<node->Depth; i++)
        printf("      ");

    switch (TreePrintStyle) {
    case TPS_NORMAL:
	printf("%s %s (ID=%d l,c=%d,%d..%d,%d count=%d, PE=%d)",
	       _D_ContextString[node->Context],
	       _D_TypeString[node->Type],
	       node->ID, node->MLine, node->MColumn,
	       node->MToLine, node->MToColumn,
	       node->RunCount,node->PE);
	PrintJoinInfo(node->Joined);
	break;
    case TPS_POINTER:
	printf("%s %s (ID=%d l,c=%d,%d cnt=%d, PE=%d)\n",
	       _D_ContextString[node->Context],
	       _D_TypeString[node->Type],
	       node->ID, node->MLine, node->MColumn,
	       node->RunCount,node->PE);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	printf("[This @ %d, RB @ %d, LB @ %d]\n",
	       node, node->RBrother, node->LBrother);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	printf("[Par @ %d, Ch @ %d]",
	       node->Parent, node->Child);
	for (i=0; i<node->Depth; i++)	PrintJoinInfo(node->Joined);
	break;
    case TPS_VERBOSE:
	printf("%s %s [ID=%d lin,col=%d,%d..%d,%d count=%d, PE=%d]\n",
	       _D_ContextString[node->Context],
	       _D_TypeString[node->Type],
	       node->ID, node->MLine, node->MColumn,
	       node->MToLine, node->MToColumn,
	       node->RunCount,node->PE);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	printf("[Depth=%d, ActValue = %d, ActArray @ %d]\n",
	       node->Depth, node->ActValue, node->ActArray);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	printf("[RangeVar @ %d, RangeLo = %d, RangeHi = %d, ProfCount @ %d]\n",
	       node->RangeVar,node->RangeLo,node->RangeHi,node->ProfCountP);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	printf("[This @ %d, RB @ %d, LB @ %d, Par @ %d, Ch @ %d]\n",
	       node, node->RBrother, node->LBrother, node->Parent,
	       node->Child);
	for (i=0; i<node->Depth; i++)
	  printf("      ");
	PrintJoinInfo(node->Joined);
	break;
    case TPS_SHORT:
    default:
	printf("%s %s (ID=%d l,c=%d,%d cnt=%d, PE=%d ",
	       _D_ContextString[node->Context],
	       _D_TypeString[node->Type],
	       node->ID, node->MLine, node->MColumn,
	       node->RunCount,node->PE);
	if (node->Joined)
	  printf("join");
	printf(")");
	break;
    };      
    printf("\n");
}



/*
 * void PrintSubTreeBrothers(DebugTree *tree, DebugNode *node)
 *
 * Prints the Tree starting at node including brothers
 * If given node is NULL, prints the whole tree starting at RootNode
 *
 */
void PrintSubTreeBrothers(tree,node)
DebugTree *tree;
DebugNode *node;
{
    DebugNode *p;			/* Pointer to parent		*/


    if (node == NULL)
        node = tree->RootNode;

    p = node->Parent;
    do {				/* For all brothers:		*/
        PrintNode(node);
        if (node->Child != NULL)	/* Print children 		*/
            PrintSubTreeBrothers(tree,node->Child);
        node = node->RBrother;		/* Goto brother			*/
    } while ((p!=NULL) && (p->Child!=node)); /* Until at start node	*/


}


/*
 * void PrintSubTree(DebugTree *tree, DebugNode *node)
 *
 * Prints the Tree starting at node EXCLUDING brothers
 * If given node is NULL, prints the whole tree starting at RootNode
 *
 */
void PrintSubTree(tree,node)
DebugTree *tree;
DebugNode *node;
{

    if (node == NULL)
        node = tree->RootNode;			/* Set default		*/

    PrintNode(node);				/* Output this node	*/

    if (node->Child)				/* Output child & its	*/
      PrintSubTreeBrothers(tree,node->Child);	/*  brothers		*/

}



/*
 * void PrintTree(DebugTree *tree)
 *
 * Prints the whole given Tree
 *
 */
void PrintTree(tree)
DebugTree *tree;
{

    if (tree == NULL)			/* Check for Errors		*/
        return;

    /* Output additional information */
    
    printf("Tree @ %d: ID %d, LastNode %d, NumC %d, NumK %d\n",
	   tree, tree->ID, tree->LastNode->ID, tree->NumC, tree->NumK);
    
    PrintSubTree(tree,tree->RootNode);	/* Print it all			*/

}


/*
 * DebugNode *GetLastNode(DebugTree *tree)
 *
 * Returns the most recent node of the given Debug Node Tree
 *
 * CURRENTLY NOT USED - MOST RECENT IS tree->LastNode
 *
 */
DebugNode *GetLastNode(tree)
DebugTree *tree;
{
    DebugNode *node = tree->RootNode;

    /*!!!!!!! HOW DO WE GET THE *REAL* LAST NODE IN A TREE, not list ?	*/

    if (node) {			/* Simply traverse line of 1st children	*/
        while (node->Child)
            node=node->Child;
        return(node);		/* Return deepest child			*/
    } else {
        return(NULL);
    };
}


/*
 * long GetChecksum(DebugTree *tree)
 *
 * Returns a global checksum over all nodes, guranteed to change if
 * tree changes.
 *
 * (NOTE: currently implemented as the sum of the node-kill-counter and
 *        the node-creation-counter, shifted by 16bit. Should do)
 *
 */
long GetChecksum(tree)
DebugTree *tree;
{
    return( ((tree->NumC) & 65535)  + ((tree->NumK) << 16) );
}



/*
 * char *NodeAsString(DebugNode *node)
 *
 * Produces a string containing all the node's data. If node == NULL,
 * return a string containing zeroes. Does no depth adjusting.
 * Note: Result must be deallocated again using free()
 *
 */
char *NodeAsString(node)
DebugNode *node;
{
    char  s[BUFSIZ];			/* Temporary buffer		*/
    char *r;				/* Return string		*/

    if (node == NULL)
      sprintf(s,"0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 \n");
    else
      sprintf(s,"%d %d %ld %d %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %d \n",
	      node->Type, node->Context, node->ID, node->Depth,
	      node->MLine, node->MColumn, node->MToLine, node->MToColumn,
	      node->ActValue, node->ActArray,
	      node->RangeVar, node->RangeLo, node->RangeHi,
	      node->RunCount, node->PE);
    r = (char*) malloc(strlen(s)+1);
    strcpy(r,s);			/* Copy generated string	*/
    
    return(r);
    
}



/*
 * void PrintNodeAsString(DebugNode *node)
 *
 * Prints the node (using the output of NodeAsString(node) ).
 *
 */
void PrintNodeAsString(node)
DebugNode *node;
{
    char *s;

    s=NodeAsString(node);	/* Simply output the resulting string	*/
    printf("%s\n",s);
    free(s);

}



/*
 * char *TreeAsString(DebugTree *tree, DebugNode *node)
 *
 * Converts the given Tree to a string, starting at the given node.
 * If node is NULL, the whole tree is used (starting at the root node).
 * If the given node is not the root node, the subtree starting at this
 * node is converted to a string and an artificial root note is inserted
 * at the beginning (to make conversion back to tree happy).
 * The depth of all nodes of the resulting tree is adjusted to start with "0"
 * at the tree's sub-root node (i.e. the real/artificial root node).
 * NumK, numC, LastNode's ID and the tree ID are also included in the output
 * (LastNode even if it's not within the subtree! Beware!)
 *
 * Purpose: Simple transfer of information to debugger
 *
 * Note: The resulting string must be freed later using free() !
 *
 */
char *TreeAsString(tree,node)
DebugTree *tree;
DebugNode *node;
{
    DebugNode	*p;			/* Pointer to parent		*/
    DebugNode	*orig;			/* Original start node		*/
    char     	 line[BUFSIZ];
    int		 sdepth;		/* Depth of start node (subtract)*/
    char    	*s;
    char 	*s1;


    if (tree == NULL)
        return(NULL);

    if (node == NULL)			/* Nothing given: Use root node	*/
        node=tree->RootNode;
    orig = node;			/* Remember start node		*/

    s= (char*) malloc(strlen("")+1);
    strcpy(s,"");

    sprintf(line,"%ld %ld %ld %ld \n",tree->ID, tree->LastNode->ID,
	    tree->NumC, tree->NumK);
    s1 = (char*) malloc(strlen(s)+strlen(line)+2);
    sprintf(s1,"%s%s",s,line);		/* Add tree's ID counter, last	*/
    free(s);				/*  node's ID & checksums	*/
    s=s1;
    
    if (node != tree->RootNode) { 	/* If not at root node:		*/
	sdepth = node->Depth-1;		/* Remember initial depth	*/
	sprintf(line,"1 1 0 0 0 0 0 0 0 0 0 0 0 0 %d \n",node->PE);
        s1 = (char*) malloc(strlen(s)+strlen(line)+2);
	sprintf(s1,"%s%s",s,line);	/* Add artificial root node	*/
	free(s);
	s=s1;
    } else {
	sdepth = 0;			/* Root node: don't change depth*/
    };
	
    
    do {				/* Output this node & adjust depth */

	sprintf(line,
		"%d %d %ld %d %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %d \n",
		node->Type, node->Context, node->ID, node->Depth - sdepth,
		node->MLine, node->MColumn, node->MToLine, node->MToColumn,
		node->ActValue, node->ActArray,
		node->RangeVar, node->RangeLo, node->RangeHi,
		node->RunCount, node->PE);
        s1 = (char*) malloc(strlen(s)+strlen(line)+2);
	/* DOESN'T WORK: realloc(s,strlen(s)+strlen(line)+2);*/
	
	sprintf(s1,"%s%s",s,line);
        free(s);
        s=s1;

        if (node->Child != NULL) {	/* Visit Child, if any 		*/

	    node = node->Child;
	    
	} else {			/* This node has no Child:	*/

	    node = node->RBrother;	/* Visit the brother		*/

	    /* Visit brother of parent if we're done with all brothers here */

            while ((node->Parent) &&		    /* End if root	    */
		   (node == node->Parent->Child) && /* or different brother */
		   (node->LBrother != orig)) {	    /* or back at start lvl */
		
 		node = node->Parent->RBrother;	    /* Go up & right in tree*/
		
	    };
	    
	};
	
    /* End if at root node or at start (directly or left, from going up) */

    } while ((node) && (node!=orig) && (node->LBrother != orig));


    sprintf(line,"0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 \n");
    /* DOESN'T WORK: realloc(s,strlen(s)+strlen(line)+2);*/
    s1 = (char*)malloc(s,strlen(s)+strlen(line)+2);
    sprintf(s1,"%s%s",s,line);		/* Add final line		*/
    free(s);
    s=s1;
    
    return(s);
}




/*
 * OLD RECURSIVE VERSION
 *
 * char *TreeAsStringXXX(DebugTree *tree, DebugNode *node)
 *
 * Converts the given Tree to a string, starting at the given node.
 * If node is NULL, the whole tree is used (starting at the root node).
 *
 * Purpose: Simple transfer of information to debugger
 *
 * Note: The resulting string must be freed later using free() !
 *
 */
char *TreeAsStringXXX(tree,node)
DebugTree *tree;
DebugNode *node;
{
    DebugNode	*p;			/* Pointer to parent		 */
    char     	 line[BUFSIZ];
    char    	*s;
    char 	*s1;
    char 	*s2;


    if (tree == NULL)
        return(NULL);

    if (node == NULL)			/* Nothing given: Use root node	*/
        node=tree->RootNode;

    s= (char*) malloc(strlen("")+1);
    strcpy(s,"");
    p = node->Parent;
    do {				/* For all brothers:		*/
					/* Output information		*/
        sprintf(line,"%d %d %ld %d %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld \n",
		node->Type, node->Context, node->ID, node->Depth,
		node->MLine, node->MColumn, node->MToLine, node->MToColumn,
		node->ActValue, node->ActArray,
		node->RangeVar, node->RangeLo, node->RangeHi,
		node->RunCount);
        s1 = (char*)malloc(strlen(s)+strlen(line)+2);
	/* DOESN'T WORK: realloc(s,strlen(s)+strlen(line)+2);*/
	sprintf(s1,"%s%s",s,line);
        free(s);
        s=s1;

        if (node->Child != NULL) {	/* Print children 		*/
	    s1  = TreeAsString(tree,node->Child);
	    s2 = (char*)malloc(strlen(s)+strlen(s1)+2);
	    free(s1);
	    free(s);
	    s=s2;
	};

        node = node->RBrother;		/* Goto brother			*/
	
    } while ((p!=NULL) && (p->Child!=node)); /* Until at start node	*/

    if (node->Depth == 0) {		/* End listing with line of 0's	*/
        sprintf(line,"0 0 0 0 0 0 0 0 0 0 0 0 0 0 \n");
	/* DOESN'T WORK: realloc(s,strlen(s)+strlen(line)+2);*/
	s1 = (char*)malloc(s,strlen(s)+strlen(line)+2);
	sprintf(s1,"%s%s",s,line);
	free(s);
	s=s1;
    };
    
    return(s);
}



/*
 * void PrintTreeAsString(DebugTreee *tree, DebugNode *node)
 *
 * Prints the Tree starting at node including brothers using a string
 * representation instead of pointers. No explicit information about
 * Brothers etc needed since tree traversal & depth of node give this
 * information implicitly.
 * Purpose: Simple transfer of information to debugger
 * If given node is NULL, prints the whole tree starting at RootNode
 *
 */
void PrintTreeAsString(tree,node)
DebugTree *tree;
DebugNode *node;
{
    char *s;

    s=TreeAsString(tree,node);	/* Simply output the resulting string	*/
    printf("%s",s);
    free(s);

}


/*
 * DebugNode *BuildNodeFromString(char *in)
 *
 * Scans the given string (preferably one line) and creates a DebugNode
 * containing the data from the string.
 * Does no error checking of the data (except that a null string results
 * in a null pointer)
 * Note: The node must be deallocated using free() if no longer needed
 *
 */
DebugNode *BuildNodeFromString(in)
char *in;
{
    DebugNode *node;

    if (in == NULL)			/* Check for bad input	*/
      return((DebugNode*)NULL);
    
    node = (DebugNode*) malloc(sizeof(DebugNode));

    sscanf(in,"%d %d %ld %d %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %d \n",
	   &(node->Type),&(node->Context),&(node->ID),&(node->Depth),
	   &(node->MLine),&(node->MColumn),&(node->MToLine),
	   &(node->MToColumn),&(node->ActValue),&(node->ActArray),
	   &(node->RangeVar),&(node->RangeLo),&(node->RangeHi),
	   &(node->RunCount),&(node->PE));
    return(node);
}


/*
 * DebugTree *BuildTreeFromString(char *in, long startID)
 *
 * Builds a new Debug Tree from the tree given in the string "in"
 * and returns the result. "startID" is added to all IDs in the tree,
 * ensuring that no lower ID is used (for merging trees later!).
 *
 * Purpose: Simple transfer of information to debugger
 *
 * (can't use NewNode() because of unique IDs etc)
 *
 * This relies on a correct tree, e.g. only one root (with depth 0) etc.
 * which makes things easier, e.g. there always has to be a brother node
 * if the depth of the current node is less than the last node's depth,
 * because we descend the depths 1 by 1, etc.
 *
 * When rebuilding, look for LastNode (using its ID passed in the string)
 * but take the deepest node if LastNode is not in the given (sub)tree.
 *
 * !!!!!!! MAXDEPTH is POISON for recursive calls :-/
 * !!!!!!! FATAL IF STRING WITH TREE INFO TRUNCATED
 *
 * !!!!!!! can also handle a subtree IF the root has depth 0 and is the only
 * !!!!!!!  node with depth 0 (e.g. as TreeAsString() does it)
 *
 */
DebugTree *BuildTreeFromString(in,startID)
char	*in;
long	 startID;
{
    DebugTree	*tree;			/* New debug tree		*/
    DebugNode	*node;			/* Pointer to current node	*/
    DebugNode   *newroot;		/* New root node		*/
    DebugNode   *lastbrother[MAXDEPTH];	/* Last brother node for depths	*/
    DebugNode	*lb;			/* Last Brother Pointer		*/
    DebugNode	*deepest;		/* Deepest node found yet	*/
    int		lastdepth;		/* Remember last depth encountrd*/
    long	ngot;			/* # of read characters		*/
    char	s[512];			/* One line to read		*/
    char	*inp;			/* Current position in input	*/
    long	lastnodeid;		/* ID of LastNode		*/
    int		i;


    if ( (inp = in) == NULL) 		/* Check for bad input		*/
        return(NULL);

    tree = CreateDebugTree(0);		/* Create new tree, PE=0	*/
    
    for (i=0; i<MAXDEPTH; i++)		/* Init array 			*/
        lastbrother[i] = NULL;
    lastdepth = 0;
    newroot = NULL;
    tree->LastNode = NULL;		/* Init last node etc		*/
    deepest = NULL;


    /* Get Tree's ID counter, ID of last node and the checksum counters	*/
    
    sscanf(inp,"%ld %ld %ld %ld \n",&(tree->ID), &(lastnodeid),
	    &(tree->NumC), &(tree->NumK));
    inp = strchr(inp,(int)'\n');	/* Get next line		*/
    if (inp) inp++;			/* Skip \n			*/

    
    while (inp != NULL) {
        node = (DebugNode*) malloc(sizeof(DebugNode));

        if (sscanf(inp,
		"%d %d %ld %d %ld %ld %ld %ld %ld %ld %ld %ld %ld %ld %d \n",
		&(node->Type),&(node->Context),&(node->ID),&(node->Depth),
		&(node->MLine),&(node->MColumn),&(node->MToLine),
		&(node->MToColumn),&(node->ActValue),&(node->ActArray),
		&(node->RangeVar),&(node->RangeLo),&(node->RangeHi),
		&(node->RunCount),&(node->PE))     <  15) {
            goto btfl_cont;		/* skip defect lines		*/
        };
        if (!( node->Type     || node->Context  || node->ID  || node->Depth ||
              node->MLine     || node->MColumn  || node->MToLine  ||
              node->MToColumn || node->ActValue || node->ActArray ||
              node->RangeVar  || node->RangeLo  || node->RangeHi  ||
              node->RunCount  || node->PE) ) {	/* All zeroes?		*/
            inp=NULL;
            break;			/* Must be last line, end input	*/
        };

	if ((deepest == NULL) ||	/* Remember deepest node (or	*/
	    (node->Depth > deepest->Depth)) {  /*  just take this one	*/
	    deepest = node;		       /*  if no deepest yet)	*/
	};

	if (node->ID == lastnodeid) {	/* Remember LastNode if seen	*/
	    tree->LastNode = node;	/* (if not, it's handled later)	*/
	};
	
	node->ID = node->ID + startID;	/* Shift ID range up		*/
	node->Joined = NULL;		/* Nothing joined here		*/
	
        lb = lastbrother[lastdepth];	/* i.e. the last node added	*/

        if (node->Depth == 0) {		/* Found root node		*/
            if (newroot == NULL) {	/* But ignore duplicates!	*/
		free(tree->RootNode);
		tree->RootNode = node;	/* REPLACE OLD ROOT (from creation)*/
                node->RBrother = node->LBrother = node;	/*  (of new tree  )*/
                node->Parent = NULL;
                newroot = node;
                node->Child = NULL;
		/*!!!!!!! JUST THROWING AWAY THE OLD ROOT NODE IS MAYBE	*/
		/*!!!!!!! DANGEROUS (but it should be empty anyway)	*/
            };

        } else if (node->Depth == lastdepth) {	/* Next in a chain of bro's */
            node->Parent = lb->Parent;		/* Same parent		*/
            node->RBrother = lb->RBrother;	/* Relink chain		*/
            node->LBrother = lb;
            lb->RBrother = node;
            (node->RBrother)->LBrother = node;
            node->Child = NULL;

        } else if (node->Depth > lastdepth) {	/* Child of last node	*/
            node->Parent = lb; /* parent = most recent node at depth -1 */
            node->RBrother = node->LBrother = node; /* Only child now	*/
            lb->Child = node;			/* Link to parent	*/
            node->Child = NULL;

        } else {			     /* Up the tree again	*/
            lb = lastbrother[node->Depth];   /* Brother with same depth */
            node->Parent = lb->Parent;		/* Same parent		*/
            node->RBrother = lb->RBrother;	/* Relink chain		*/
            node->LBrother = lb;
            lb->RBrother = node;
            (node->RBrother)->LBrother = node;
            node->Child = NULL;
        };

        lastdepth = node->Depth;	/* Update last brother info	*/
        lastbrother[lastdepth] = node;

    btfl_cont:
        inp = strchr(inp,(int)'\n');	/* Get next line		*/
        if (inp) inp++;			/* Skip \n			*/
    };

    if (tree->LastNode == NULL) {	/* Take deepest node as LastNode*/
	tree->LastNode = deepest;	/*  if no LastNode yet		*/
    };
    
    return(tree);
}




/*
 * DebugNode *AdjustDepths(DebugNode *node, int sdepth)
 *
 * Recursively adjusts the depths of all nodes in the given tree "node"
 * starting with "sdepth" for "node" itself.
 * Returns its result, modifies the given tree.
 * Also adjusts depths if start depth == 0 (for reconstruct a tree where
 * the depths are not set or wrong)
 *
 */
DebugNode *AdjustDepths(node,sdepth)
DebugNode *node;
int	   sdepth;
{
    DebugNode *p;
    
    if (node == NULL)
        return(node);				/* Nothing to do	 */

    p=node;
    do {					
	node->Depth = sdepth;			/* Correct depth	 */
	if (node->Child) 			/* Adjust children	 */
	    AdjustDepths(node->Child,sdepth+1);	/*  if any (save fn call)*/
	node = node->RBrother;			/* Adjust brothers	 */
    } while (node != p);			/* Until we got all bros */

    return(node);
}




/*
 * DebugTree *InsertSubtree(DebugTree *tree,
 *                          DebugNode *node, DebugNode *subtree)
 *
 * Inserts the given subtree (or just node) "subtree" as child of the
 * node "node" in the tree "tree". Rearranges node's children if any.
 * If node == NULL, the tree's RootNode is assumed.
 * The depth of all nodes of the subtree is adjusted.
 * The subtree's brothers are also integrated if any (shouldn't be any!)
 * Also modifies the subtree (parent pointer) !
 *
 */
DebugTree *InsertSubtree(tree,node,subtree)
DebugTree *tree;
DebugNode *node;
DebugNode *subtree;
{
    DebugNode *p;
    DebugNode *q;

    if ((tree == NULL) || (subtree == NULL))	/* No changes if	*/
      return(tree);				/*  nothing to do	*/

    if (node == NULL)
      node = tree->RootNode;		/* Take root node if none given	*/

    if (node->Child == NULL) {		/* Just add subtree if		*/
	node->Child = subtree;		/*  no child yet		*/

    } else {
    
        node = node->Child;		/* Link new brothers into list	*/
        p = node->LBrother;
        q = subtree->LBrother;
        q->RBrother	  = node;
        node->LBrother	  = q;
        p->RBrother	  = subtree;
        subtree->LBrother = p;
        node = node->Parent;
	
    };
    p=subtree;
    
    do {				/* Set parent pointers		*/
	p->Parent = node;
	p=p->RBrother;
    } while (p!=subtree);

    AdjustDepths(subtree,node->Depth+1);/* Re-compute all depths in node*/
    
    return(tree);
}



/*
 * int KillNode(DebugTree *tree, Debugnode *node)
 *
 * Destroys the given node in the given tree without checking for children
 * but with correcting of brothers.
 * Returns 1 if successful.
 *
 */
int KillNode(tree,node)
DebugTree *tree;
DebugNode *node;
{
    DebugNode *rb; 			/* Right Brother of this node	*/
    DebugNode *p;			/* Parent of this node		*/

    if ((node == NULL) || (tree == NULL)) {
        return (0);			/* Nothing to kill		*/
    } else {

        p  = node->Parent;
        rb = node->RBrother;		/* Correct brothers before kill:*/
        if (node == rb) {		/* We're the only child		*/
            if (p != NULL) {		/* Set children of parent to 0	*/
                p->Child = NULL;
		if (tree->LastNode == node)
		  tree->LastNode = p;	/* Parent now = most recent node*/
		
		/*!!!!!!! STRANGE COMPUTATION OF LAST NODE, ALWAYS LAST	*/
		/*!!!!!!! IN A LINEAR LIST				*/
            };
        } else {			/* Brothers left, update them	*/
	    
            rb->LBrother = node->LBrother;
            (node->LBrother)->RBrother = rb;
	    
            if ((p != NULL) &&
		(p->Child == node))	/* Set new child for parent	*/
	      p->Child = rb;
	    
	    if (tree->LastNode == node)
	      tree->LastNode = rb;	/* Brother now = most recent	*/

        };
	(tree->NumK)++;			/* Update counter (checksum)	*/

        free(node);/*!!!!!!! because "return(free(node))" -> complains?!? */
        return(1);			/* Free it & return result (1=OK) */
    };
}



/*
 * DebugTree *CreateDebugTree(int PE)
 *
 * Creates a new Debug Tree, adds a root node, initializes IDs etc,
 * Sets PE number for this tree to "PE"
 *
 */
DebugTree *CreateDebugTree(PE)
int PE;
{
    DebugTree *tree = (DebugTree*) malloc(sizeof(DebugTree));
    
    tree->ID   = 0;			/* Init global ID		*/
    tree->NumC = 0;			/* Init checksums (counters)	*/
    tree->NumK = 0;
    tree->RootNode = NewNode(tree, 	/* Init root node		*/
		       DN_ROOT, NULL,
                       0, 0, 0, 0,
		       0, NULL,
                       NULL, 0, 0);
    tree->RootNode->PE = PE;		/* Set PE number		*/
    return(tree);
}



/*
 * int KillSubTree(DebugTree *tree, DebugNode *node)
 *
 * Recursively destroys the subtree of the given tree, starting at node.
 * Does also kill brothers.
 * Returns 1 if successful.
 *
 */
int KillSubTree(tree,node)
DebugTree *tree;
DebugNode *node;
{
    DebugNode *n,*p;			/* Work pointer to brothers	*/


    if ((node == NULL) || (tree == NULL))
        return(0);
    
    p = node->Parent;
    do {				/* For all brothers:		*/
        KillSubTree(tree,node->Child);
        n = node->RBrother;
        KillNode(tree,node);
        node=n;
    } while ((p!=NULL) && (p->Child!=NULL));

    return(1);
    
    /* Note: KillNode takes care of correcting brother's pointers!	*/

}



/*
 * int KillTree(DebugTree *tree)
 *
 * Kills the whole given Debug Tree recursively and deallocates it.
 * Returns 1 if successful.
 *
 */
int KillTree(tree)
DebugTree *tree;
{

    if (tree==NULL)			/* Check for non-existant tree	*/
      return(0);
    if (KillSubTree(tree,tree->RootNode)) {
	free(tree);
	tree=NULL;			/* Deallocate tree and set	*/
	return(1);			/*  pointer to NULL to show it	*/
    } else {
	return(0);			/* Got some problem...		*/
    };

}



/*
 * DebugTree *AddStringToTree(DebugTree *tree, char *in, DebugNode *snode)
 *
 * Adds the tree given in the string "in" to the given tree as child of
 * the given node. Returns the resulting tree (and modifies "tree").
 *
 * Purpose: Pasting of subtrees into existing tree
 *
 */
DebugTree *AddStringToTree(tree,in,snode)
DebugTree *tree;
char	  *in;
DebugNode *snode;
{
    DebugTree	*newtree;		/* Tree built from given string	*/


    if (tree == NULL)
        return(NULL);
    
    if (in == NULL)	 		/* Check for bad input		*/
        return(tree);

    if (snode==NULL)			/* Take root if none given	*/
        snode = tree->RootNode;
    
    /* Build a new tree from the given string with IDs above the current one */
    
    newtree = BuildTreeFromString(in,tree->ID+1);
    tree->ID = newtree->ID+1;		/* Update ID range again	*/
    
    InsertSubtree(tree,snode,newtree->RootNode->Child); /* Do it	*/

    newtree->RootNode->Child = NULL;	/* Kill remainder of new tree	*/
    KillTree(newtree);

    return(tree);
}




/*
 * void BuildAndMergeNodes(DebugTree *tree, DebugNode *parent, *n1, *n2)
 *
 * Merges the given trees into a new tree, joining information of
 * identical nodes (i.e. same type, context, line, column) until
 * the 1st difference is encountered, then just appends the
 * remaining subtrees. The new node(s) is(are) appended to the node
 * "parent" of the given tree "tree" (so tree->LastNode doesn't matter).
 *
 * Purpose: Merging of information from two trees
 *
 * (algorithm: check if one node nil, just add other node's subtree if yes.
 *             check if both nodes match,
 *		   merge them & add children if yes
 *		   otherwise add the two nodes' subtrees as two new children
 *	       advance to the nodes' brothers
 *	       if one or both nodes don't have any more brothers, add
 *	        the other node's remaining brothers (if any) to the tree
 *	       otherwise loop to the next check for matching
 * )
 * !!!!!!! PROBLEM: NODES HAVE TO MATCH IN THE SAME ORDER, NO PERMUTATIONS
 * !!!!!!!          ARE RECOGNIZED !!!
 *
 * !!!!!!! DOESN'T GENERATE JOIN INFOS
 */
void BuildAndMergeNodes(tree,parent,n1,n2)
DebugTree *tree;
DebugNode *parent;
DebugNode *n1;
DebugNode *n2;
{
    DebugNode	*node;			/* Inspected Node		*/
    DebugNode	*newnode;		/* Merged Node			*/

    if ((n1 == NULL) && (n2 == NULL))	/* Both null -> do nothing	*/
      return;

    node = NULL;
    if (n1 == NULL)			/* Set node to a non-null node	*/
      node = n2;			/*  (if n1 or n2 is null!)	*/
    if (n2 == NULL)
      node = n1;

    if (node != NULL) {			/* One node was zero: Just add..*/
        do {				/* ..the other's subtree+bro's	*/
	    DebugNode *new;

	    new = CopyNode(tree,parent,node);		/* Add node	*/
	    BuildAndMergeNodes(tree,new,
			       node->Child,NULL);	/* Add children	*/
	    node = node->RBrother;			/* Add brothers	*/
	} while  ((node->Parent) && (node != node->Parent->Child));
	return;
    };

    /* From here: n1 and n2 non-NULL, check for matching attributes	*/

    do {
	DebugNode *new;
	
	if ( (n1->Type    != n2->Type)	||	/* If nodes are different, */
	    (n1->Context != n2->Context)||	/*  add the subtrees as	   */
	    (n1->MLine   != n2->MLine)	||	/*  two different children */
	    (n1->MColumn != n2->MColumn) ) {
	    
	    new = CopyNode(tree,parent,n1);
	    BuildAndMergeNodes(tree,new,n1->Child,NULL);/* Add 1s' children*/
	    new = CopyNode(tree,parent,n2);
	    BuildAndMergeNodes(tree,new,n2->Child,NULL);/* Add 2s' children*/
	    
	} else {    				/* Nodes match, merge them */

	    new = CopyNode(tree,parent,n1); /* Copy one node into new tree */
	    new->RunCount = new->RunCount + /* Add Profile Counters	   */
	                     n2->RunCount;
            BuildAndMergeNodes(tree,new,    /* Recursively merge children  */
				 n1->Child,n2->Child);
	};
	n1 = n1->RBrother;
	n2 = n2->RBrother;
	
    } while ((n1->Parent) &&			/* End if one or both	*/
	     (n1->Parent->Child != n1) &&	/*  nodes are through	*/
	     (n2->Parent) &&			/*  with their brothers	*/
	     (n2->Parent->Child != n2) );

    node = NULL;				    /* Find left over	*/
    if ((n1->Parent) && (n1->Parent->Child != n1))  /*  brothers (only	*/
      node = n1;				    /*  one node can	*/
    if ((n2->Parent) && (n2->Parent->Child != n2))  /*  still have	*/
      node = n2;				    /*  some !!)	*/
    
    if (node != NULL) {		/* Add left over brothers to new tree	*/
	do {
	    DebugNode *new;

	    new = CopyNode(tree,parent,node);		/* Add node	*/
	    BuildAndMergeNodes(tree,new,
			       node->Child,NULL);	/* Add children	*/
	    node = node->RBrother;			/* Add brothers	*/
	} while  ((node->Parent) && (node != node->Parent->Child));
    };		

}




/*
 * DebugTree *MergedTree(DebugTree *tree1, *tree2)
 *
 * Merges the given trees into a new tree, joining information of
 * identical nodes (i.e. same type, context, line, column) until
 * the 1st difference is encountered, then just appends the
 * remaining subtrees.
 *
 * Purpose: Merging of information from two trees
 *
 * !!!!!!! DOESN'T GENERATE JOIN INFOS (use GroupTree instead...)
 */
DebugTree *MergedTree(tree1,tree2)
DebugTree *tree1;
DebugTree *tree2;
{
    DebugTree	*newtree;		/* Merged Tree			*/


    if (tree1 == NULL)			/* Check for trivial cases	*/
        return(tree2);
    if (tree2 == NULL)
        return(tree1);
    
    /* Start new tree for the result, using PE of tree1 */
    
    newtree = CreateDebugTree(tree1->RootNode->PE);
    
    if (tree1->ID > tree2->ID)
      newtree->ID = tree1->ID+1;	/* Take max of both tree IDs	*/
    else
      newtree->ID = tree2->ID+1;

    BuildAndMergeNodes(newtree,newtree->RootNode,
		       tree1->RootNode->Child,tree2->RootNode->Child);

    newtree->LastNode = GetLastNode(newtree);	/* CONSTRUCT last node	*/
    return(newtree);
}



/*
 * void AllJoinNodes(DebugNode *n1,*n2)
 *
 * Joins the given nodes (which must be brothers!! i.e. same parent, same
 * depth), adds RunCounts and kills "n2", setting it to "n1"
 * Remembers info about single nodes in JoinInfo
 * Takes care that a parent's 1st child stays that way!
 * NOTE: PE number of joined node is one of the PEs of the two nodes (arbitr.)
 *
 */
void AllJoinNodes(n1,n2)
DebugNode *n1;
DebugNode *n2;
{

    if ((n1 == NULL) || (n2 == NULL) || (n1 == n2))
      return;

    
    /* Remember information of single nodes when joining */
    
    AddJoinInfo(n1->ID,n1->RunCount,n1->PE,/* Append own info (if new) to n1 */
		&(n1->Joined));
    AddJoinInfo(n2->ID,n2->RunCount,n2->PE,/* Append own info (if new) to n2 */
		&(n2->Joined));
    AppendJoinInfo(n1->Joined,n2->Joined); /* Merge the info chains	     */
    n2->Joined = NULL;
	
	  
    n1->RunCount = n1->RunCount + n2->RunCount;	/* Add Profile Info	*/
    
    if (n2->Child != NULL) {			/* Move n2's children	*/
	DebugNode *p = n2->Child;		/*  to n1		*/

	if (n1->Child == NULL) {		/* n1 has no children	*/
	    DebugNode *c;			/*  yet, just copy n2's	*/
	    n1->Child = n2->Child;		/*  children & correct	*/
	    c = n1->Child;			/*  their parent	*/
	    do {				/*  pointers		*/
		c->Parent = n1;
		c = c->RBrother;
	    } while (c!=n1->Child);
	} else {				/* n1 has children,	*/
	    DebugNode *p,*q;			/*  join them with n2's	*/

	    p = n1->Child->LBrother;		/* Join the two lists	*/
	    q = n2->Child->LBrother;		/*  of children		*/
	    q->RBrother	       = n1->Child;
	    n1->Child->LBrother= q;
	    p->RBrother	       = n2->Child;
	    n2->Child->LBrother= p;

	    p = n2->Child;
	    q = n1->Child;
	    do {
		p->Parent = n1;			/* Set new parent n1	*/
		p = p->RBrother;		/*  for n2's children	*/
	    } while (p != q);			/* (not for n1's again)	*/

	};
    };
    n2->RBrother->LBrother = n2->LBrother;	/* Remove n2 from list	*/
    n2->LBrother->RBrother = n2->RBrother;

    if (n2->Parent->Child == n2) {		/* Take care that 1st	*/
	n2->Parent->Child = n1;			/*  child pointer OK	*/
    };
    
    free(n2);
    
    n2 = n1;					/* n1 and n2 are now eq.*/

}



/*
 * Macro EquivNodes(DebugNode *n1,*n2)
 *
 * Evaluates to True if the nodes are equivalent (i.e. same type etc)
 * (does NOT check if children are equivalent!!)
 *
 */
#define EquivNodes(n1,n2) ( (n1->Type    == n2->Type)   &&	\
			    (n1->Context == n2->Context)&&	\
			    (n1->MLine   == n2->MLine)  &&	\
			    (n1->MColumn == n2->MColumn)	\
		          )



/*
 * void AllGroupSubTree(DebugNode *node)
 *
 * Recursively joins equivalent nodes (i.e. same type, line, column etc)
 * Ignores children (i.e. equivalent nodes with non-equ. children ARE grouped)
 *
 * !!!!!!! CAN KILL LASTNODE WITHOUT NOTICING IT !!!
 */
void AllGroupSubTree(node)
DebugNode *node;
{
    DebugNode *p;

    if (node == NULL)
      return;

    p = node;				/* Examine all brothers of the	*/
    do {				/*  node			*/
	DebugNode *q;			/* (even if it doesn't have any,    */
					/*  we want to group its children!) */
	q = p->RBrother;

	if (p!=q) {			/* At least one brother?	*/
	    
	    do {			/* Examine all brothers of p:	*/
                if (EquivNodes(p,q)) {	/* Join nodes if equivalent	*/
		    JoinNodes(p,q);	/*  (updates q as well)		*/
	        };
                q = q->RBrother;
            } while (q != p);		/* Until all examined		*/
	};

	if (p->Child != NULL)		/* Recursively group children	*/
	  AllGroupSubTree(p->Child);
	
	p = p->RBrother;    		/* Next joining round		*/
	
    } while (p->RBrother != node);
    
}



/*
 * void AllGroupTree(DebugTree *tree)
 *
 * Recursively joins equivalent nodes (i.e. same type, line, column etc)
 * in tree
 * Ignores children (i.e. equivalent nodes with non-equ. children ARE grouped)
 *
 * !!!!!!! CAN KILL LASTNODE WITHOUT NOTICING IT !!!
 */
void AllGroupTree(tree)
DebugTree *tree;
{
    if (tree == NULL)
      return;
    
    AllGroupSubTree(tree->RootNode);
    
}




/*
 * void JoinNodes(DebugNode *n1,*n2)
 *
 * Joins the given nodes (which must be brothers!! i.e. same parent, same
 * depth), adds RunCounts and kills "n2", setting it to "n1"
 * RECURSIVELY joins children of the nodes, too (since the nodes are
 * equivalent!)
 * Nodes HAVE TO have same number and kind of children in same order, else
 * they wouldn't be equivalent and deserve to be joined!
 * Remembers info about single nodes in JoinInfo
 * Takes care that a parent's 1st child stays that way (n1 or n2, same parent)!
 * NOTE: PE number of joined node is one of the PEs of the two nodes (arbitr.)
 *
 */
void JoinNodes(n1,n2)
DebugNode *n1;
DebugNode *n2;
{

    if ((n1 == NULL) || (n2 == NULL) || (n1 == n2))
      return;

    
    /* Remember information of single nodes when joining */
    
    AddJoinInfo(n1->ID,n1->RunCount,n1->PE,/* Append own info (if new) to n1 */
		&(n1->Joined));
    AddJoinInfo(n2->ID,n2->RunCount,n2->PE,/* Append own info (if new) to n2 */
		&(n2->Joined));
    AppendJoinInfo(n1->Joined,n2->Joined); /* Merge the info chains	     */
    n2->Joined = NULL;
	
    if ((n2->Parent) &&				/* If n2 is 1st child,	*/
	(n2 == n2->Parent->Child))		/*  set n1 as new child	*/
      n2->Parent->Child = n1;			/* (keep child valid!)	*/
    
    n1->RunCount = n1->RunCount + n2->RunCount;	/* Add Profile Info	*/

    n2->RBrother->LBrother = n2->LBrother;	/* Remove n2 from list	*/
    n2->LBrother->RBrother = n2->RBrother;	/*  of brothers		*/

    /* Now recursively group the children - since the nodes are		*/
    /*  equivalent, they have the same number & kind of children in	*/
    /*  the same order. Makes things easier.				*/
    
    if (n1->Child != NULL) {			/* Move n2's children	*/
	DebugNode *p = n1->Child;		/*  to n1 by joining	*/
	DebugNode *q = n2->Child;		/*  them pair-wise	*/

	p->LBrother->RBrother = q;		/* Merge the two child	*/
	q->LBrother->RBrother = p;		/*  lists into one	*/
	p->LBrother           = q;
	q->LBrother           = p;

	do {					/* Join eq. node pairs	*/
	    DebugNode *qrb = q->RBrother;
	    
	    q->Parent = n1;			/* Set new parent	*/
	    if (EquivNodes(p,q)) {		/* SHOULD be equ. here	*/
		JoinNodes(p,q);			/* anyway! Join the two	*/
	    };
	    p = p->RBrother;
	    q = qrb;
	} while ((p != n1->Child) && (q != n1->Child));
	
    };
    
    free(n2);
    n2 = n1;					/* n1 and n2 are now eq.*/

}


/*
 * char EquivSubTree(DebugNode *n1, DebugNode *n2)
 *
 * Returns 1 if the subtrees "n1" and "n2" are equivalent, i.e.
 * all nodes and (recursively) their children are equivalent
 * NOTE: Does not try to match nodes, just takes them in the same order
 *       as they appear in the list of children! I.e. equivalent nodes
 *	 are not found if their order is different in the two trees!!!
 *
 */
char EquivSubTree(n1,n2)
DebugNode *n1;
DebugNode *n2;
{
    DebugNode	*c1;			/* Children			*/
    DebugNode	*c2;

    if ((n1==NULL) && (n2==NULL))	/* No nodes -> eq.		*/
      return((char)1);
    
    if ((n1==NULL) || (n2==NULL))	/* One node null -> not eq.	*/
      return((char)0);

    if (!EquivNodes(n1,n2))		/* Nodes not eq -> Tree,neither	*/
      return((char)0);

    c1=n1->Child;			/* Nodes equivalent, now examine*/
    c2=n2->Child;			/*  the children		*/
    
    if ((c1==NULL) && (c2==NULL))	/* Nodes eq. & no children -> eq*/
      return((char)1);

    if ((c1==NULL) || (c2==NULL))	/* Only one has children -> nope*/
      return((char)0);

    do {				/* Test equivalence of children	*/
	if (!EquivSubTree(c1,c2))	/* Children not eq -> nope	*/
	  return((char)0);
	
	c1 = c1->RBrother;		/* Now look at brothers		*/
	c2 = c2->RBrother;
	
    } while ((c1 != n1->Child) &&	/* Repeat until at 1st child,	*/
	     (c2 != n2->Child));	/*  again			*/

    if ((c1 == n1->Child) &&		/* Both child lists done at same*/
	(c2 == n2->Child))		/*  time -> OK, equivalent	*/
      return((char)1);

    return((char)0);			/* One list shorter, not equiv.	*/
    
}



/*
 * void GroupSubTree(DebugNode *node)
 *
 * Recursively joins equivalent nodes (i.e. same type, line, column etc)
 * Does NOT group nodes which are equivalent but have non-eq. children
 * Uses IDs instead of node pointers to check if node is the same
 * Also groups the node's brothers!
 *
 * !!!!!!! CAN KILL LASTNODE WITHOUT NOTICING IT !!!
 */
void GroupSubTree(node)
DebugNode *node;
{
    DebugNode *p;
    DebugNode *parent;
    char       next_round;
    char       new_change;

    if (node == NULL)
      return;

    p = node;				/* Group all children of all	*/
    do {				/*  brothers of this node	*/
	if (p->Child)			/*  (including the node itself)	*/
	  GroupSubTree(p->Child);
	p = p->RBrother;
    } while (p != node);

    if (node->Parent == NULL)		/* If we're the root node, done	*/
      return;				/* (has no brothers, anyway)	*/

    /* From now on, nodes are joined, so remember parent to have one	*/
    /*  fixed reference point for iteration (pointer to node may	*/
    /*  become obsolete when note is joined with other node! But the	*/
    /*  parent's child is updated when a join occurs)			*/
    
    parent = node->Parent;
    p = parent->Child;	
    do {
	DebugNode *q = p->RBrother;
	
	/* Inner loop: Examine all of p's right brothers (until 1st	*/
	/*  child is reached) and join them with p if equivalent	*/
	
	while (q != parent->Child) {	/* Remember q's RBrother for	*/
	    DebugNode *qrb =q->RBrother;/*  continuing after join	*/
	    
	    if ((p != q) &&		/* Must not be the same		*/
		(EquivSubTree(p,q))) {	/* Equivalent -> join them	*/
		JoinNodes(p,q);		/*  (joins recursively!)	*/
	    };
	    q = qrb;			/* Examine next brother		*/
	};
	
	p = p->RBrother;		/* Next p to examine		*/
	
    } while (p != parent->Child);	/* Until p is at 1st child again*/
    
}



/*
 * void GroupTree(DebugTree *tree)
 *
 * Recursively joins equivalent nodes (i.e. same type, line, column etc)
 * in tree
 * Does NOT group nodes which are equivalent but have non-eq. children
 *
 * !!!!!!! CAN KILL LASTNODE WITHOUT NOTICING IT !!!
 */
void GroupTree(tree)
DebugTree *tree;
{
    if (tree == NULL)
      return;
    
    GroupSubTree(tree->RootNode);
    
}





/*
 * ProfileTable *CreateProfileTable()
 *
 * Creates a new profile information hash table and initializes it
 *
 */
ProfileTable *CreateProfileTable()
{
    ProfileTable *pt;
    register int i;

    pt = (ProfileTable*)		/* Get memory for a new table	*/
          malloc (sizeof(ProfileTable));
    
    for (i=0; i<P_TABLE_SIZE; i++)	/* Init profile info table	*/
        pt->ProfileTable[i] = NULL;
    pt->maxProfileLine = 0;		/* Init line info		*/
    pt->NumEntries = 0;			/* Init number of entries	*/

    return(pt);				/* Return the new table		*/

}



/*
 * void AddProfileNode(ProfileTable *pt, DebugNode *dnode)
 *
 * Adds the information contained in the given debug node
 *  (i.e. line number, run count etc) to the profile table.
 * If new node, init run count to dnode's count (was called in a DebugFnStart).
 * Else, add run count to current value (was called in a DebugFnEnd
 *  or at next call of a DebugFnStart).
 * Run counters >= 0 are summed up in the Profile's counter, Values
 *  <0 overwrite the old value with their absolute value
 * Sets the ProfCountP pointer in the DebugNode to point to the RunCount
 *  of the corresponding Profile entry
 * 
 */
void AddProfileNode(pt,dnode)
ProfileTable *pt;
DebugNode *dnode;
{
    ProfileNode *p,*op;
    int		 type;
    long         line,column,toline,tocolumn;

    if (dnode == NULL)
        return;

    type    = dnode->Type;
    line    = dnode->MLine;
    column  = dnode->MColumn;
    toline  = dnode->MToLine;
    tocolumn= dnode->MToColumn;
    p = pt->ProfileTable[phash(line)];	/* Get hash line number	in table*/


    while ((p != NULL) && ( (p->MLine != line) || (p->MColumn != column) ) ) {
        op= p;
        p = p->NextNode;		/* Look if node already exists	*/
    };


    if (p != NULL) {			/* Inc count for existing node	*/
        long d = dnode->RunCount;

        if (d<0) {			/* <0 : take absolute value	*/
            p->RunCount = - dnode->RunCount;
        }else {				/* >=0: sum up all values	*/
            p->RunCount = p->RunCount + dnode->RunCount;
        };

    } else {				/* Else: create new node	*/

        p = (ProfileNode*) malloc (sizeof(ProfileNode));
        p->RunCount = dnode->RunCount>0?dnode->RunCount:-dnode->RunCount;
        p->MLine    = dnode->MLine;	/* Insert information		*/
        p->MColumn  = dnode->MColumn;
        p->MToLine  = dnode->MToLine;
        p->MToColumn= dnode->MToColumn;
        p->Type     = dnode->Type;
        p->Context  = dnode->Context;
        p->NextNode = NULL;

        if (pt->ProfileTable[phash(line)] == NULL) {
            pt->ProfileTable[phash(line)] = p;/* First node: add to table*/
        } else {
            op->NextNode = p;		/* End of chain: add to end	*/
        };

	pt->NumEntries++;		/* One more entry		*/
    };

    if (line > pt->maxProfileLine)
        pt->maxProfileLine = line;	/* Update line range if larger	*/

    dnode->ProfCountP = &(p->RunCount);	/* Give pointer to PT-Counter	*/
    
}



/*
 * void KillProfileNode(ProfileTable *pt, long line, long column)
 *
 * Clears & deallocates the ProfileNode that corresponds to the given line
 * and column.
 * Corrects the corresponding ProfileTable entry.
 *
 */
void KillProfileNode(pt,line,column)
ProfileTable *pt;
long line,column;
{
    ProfileNode *p, *op;
    long         l;

    l = phash(line);
    p = pt->ProfileTable[l];			/* Get 1st hash entry	*/
    op= p;
    while ((p != NULL) && (p->MLine != line) &&
           (p->MColumn != column) ) {		/* Look for node in list*/
        op= p;
        p = p->NextNode;
    };
    if (p != NULL) {				/* Found matching line	*/
        if (op != pt->ProfileTable[l]) {	/* If inside a chain:	*/
            op->NextNode = p->NextNode;		/* Relink list		*/
        } else {				/* If at start of chain:*/
            pt->ProfileTable[l] = p->NextNode;	/* Relink table entry	*/
        };
        free(p);
	pt->NumEntries--;			/* One entry less	*/
    };

    /*!!!!!!!! should also update the new maxProfileLine... not easy	*/
}



/*
 * void ClearProfileTable(ProfileTable *pt)
 *
 * Clears the whole profile table, deallocates all Nodes (but not the
 * table itself)
 *
 */
void ClearProfileTable(pt)
ProfileTable *pt;
{
    register int i;

    for (i=0; i<P_TABLE_SIZE; i++)		/* Kill all nodes	*/
        while (pt->ProfileTable[i] != NULL)
            KillProfileNode(pt,
			    pt->ProfileTable[i]->MLine,
			    pt->ProfileTable[i]->MColumn);
    pt->maxProfileLine = 0;
    pt->NumEntries = 0;				/* Clear other info, too*/
}

    

/*
 * void PrintProfileTable(ProfileTable *pt)
 *
 * Prints all entries of the profile table.
 *
 */
void PrintProfileTable(pt)
ProfileTable *pt;
{
    register int i;

    if (pt == NULL)
      return;
    
    printf("  Profiling information table (%d entries):\n",pt->NumEntries);

    for (i=0; i<P_TABLE_SIZE; i++) {
        ProfileNode *p = pt->ProfileTable[i];

        if (p!=NULL) {			/* Print only non-empty entries	*/
            printf("  PT[%4d] : ",i);
            while (p != NULL) {
                printf("(%s lines=%d..%d count=%d) ",_D_TypeString[p->Type],
                                          p->MLine,p->MToLine,p->RunCount);
                p = p->NextNode;	/* Output whole chain (if any)	*/
            };
            printf("\n");
        };
    };
}




/*
 * char *ProfileAsString(ProfileTable *pt)
 *
 * Putputs the contents of the Profile Table one by one in no special order,
 * ignoring empty lines. Returns a string (which has to be free()ed again)
 * Purpose: Simple transfer of information to debugger
 *
 */
char *ProfileAsString(pt)
ProfileTable *pt;
{
    register int i;
    char     line[BUFSIZ];
    char    *s = (char*) malloc(strlen("")+1);
    char    *s1;

    if (pt==NULL)
      return(NULL);
    
    strcpy(s,"");
    for (i=0; i<P_TABLE_SIZE; i++) {
        ProfileNode *p = pt->ProfileTable[i];

        if (p!=NULL) {			/* Print only non-empty entries	*/
            while (p != NULL) {		/* One entry per line		*/
                sprintf(line,
			"%d %d %ld %ld %ld %ld %ld \n",
			p->Type, p->Context, p->MLine, p->MColumn,
			p->MToLine, p->MToColumn, p->RunCount);
                p = p->NextNode;	/* Output whole chain (if any)	*/
		s = (char*) realloc(s,strlen(s)+strlen(line)+2);
		strcat(s,line);
            }
        }
    }
    strcpy(line,"0 0 0 0 0 0 0 \n");	/* Set end marker		*/
    realloc(s,strlen(s)+strlen(line)+2);
    strcat(s,line);

    return(s);
}




/*
 * void PrintProfileAsString(ProfileTable *pt)
 *
 * Prints the contents of the Profile Table one by one in no special order,
 * ignoring empty lines.
 * Purpose: Simple transfer of information to debugger
 *
 */
void PrintProfileAsString(pt)
ProfileTable *pt;
{
    char *s;

    s = ProfileAsString(pt);
    printf("%s",s);
    free(s);

}



/*
 * void AddStringToProfile(ProfileTable *pt, char *in, char replace)
 *
 * Adds the contents of the Profile Table given in string "in" to the
 * current Profile Table, summing counters in case of duplicates (if
 * "replace" is false) or replacing counters (if "replace" is true).
 *
 * Purpose: Joining Profile Strings (no replace) / info transfer to
 *          Debugger (with replace)
 *
 */
void AddStringToProfile(pt,in,replace)
ProfileTable *pt;
char *in;
char replace;
{
    char	*inp;			/* Current position in input	*/
    DebugNode	dnode;			/* For transfer of profile info	*/
    int		i;


    if (pt == NULL)
        return;
    
    if ( (inp = in) == NULL) 		/* Check for bad input		*/
        return;

    while (inp != NULL) {
        if (sscanf(inp,"%d %d %ld %ld %ld %ld %ld \n",
		&(dnode.Type),&(dnode.Context),&(dnode.MLine),&(dnode.MColumn),
                &(dnode.MToLine),&(dnode.MToColumn),		
                &(dnode.RunCount))      <  7) {
            goto bpfl_cont;		/* skip defect lines		*/
        };
        if (!( dnode.Type      || dnode.Context ||	/* All zeroes?	*/
               dnode.MLine     || dnode.MColumn || dnode.MToLine ||
               dnode.MToColumn || dnode.RunCount) ) {
            inp=NULL;
            break;			/* Must be last line, end input	*/
        };

	if (replace)			/* Override old value if wanted	*/
	  dnode.RunCount = -dnode.RunCount;
        AddProfileNode(pt,&dnode);	/* Add info to Table		*/

    bpfl_cont:
        inp = strchr(inp,(int)'\n');	/* Get next line		*/
        if (inp) inp++;			/* Skip \n			*/
    };

}




/*
 * ProfileNode *GetProfileNode(ProfileTable* pt, long line, long column)
 *
 * Finds the profile node with the same line number & column as requested.
 * Returns NULL if not found.
 *
 */
ProfileNode *GetProfileNode(pt,line,column)
ProfileTable* pt;
long line, column;
{
    ProfileNode *p, *op;
    long         l;

    l = phash(line);
    p = pt->ProfileTable[l];			/* Get 1st hash entry	*/
    op= p;
    while ((p != NULL) && ((p->MLine != line) ||/* (line AND col must match!)*/
           (p->MColumn != column))) {		/* Look for node in list*/
        op= p;
        p = p->NextNode;
    };
    return(p);
}


/*
 * char *ProfileStatAsString(ProfileNode **pstat, int psize)
 *
 * Returns a string containing the profile statistics given in "pstat"
 * (with "psize" entries).
 * Note: The string must be freed again using free() when no longer needed
 *
 */
char *ProfileStatAsString(pstat,psize)
ProfileNode **pstat;
int psize;
{
    char *s;		/* Output string	*/
    int	  p;		/* Offset in s		*/
    int   i;
    int   l = strlen("cccccccc lllll ttttttttttt\n"); /* Length of one entry */

    if ((pstat == NULL) || (psize == 0))	/* Check for bad input	*/
      return(NULL);

    s = (char*)malloc((l+1)*(psize+1));

    sprintf(s,"   Count  Line Type       \n");	/* Header line		*/
    p = l;
    
    for (i=0;i<psize;i++) {			/* Output all entries	*/
	sprintf( (s + p),
		"%8d %5d %-11s\n",
		pstat[i]->RunCount,pstat[i]->MLine,
		_D_TypeString[pstat[i]->Type]);
	p = p + l;				/* Advance in buffer s	*/
    };
    
    return(s);

}


/*
 * ProfileNode **MakeProfileStat(ProfileTable *ptable)
 *
 * Generates array of profile node references sorted after decreasing
 * RunCount
 * Node: Array most be freed using free() when no longer needed
 *
 */
ProfileNode **MakeProfileStat(ptable)
ProfileTable *ptable;
{
    ProfileNode **pstat;		/* Statistics array		*/
    long	  psize;		/* Current size of array	*/
    int		  i;
    

    if (ptable == NULL)			/* Check for bad input		*/
      return(NULL);

    psize = 0;				/* Create clear profile array	*/
    pstat = (ProfileNode**)
                     malloc ((ptable->NumEntries+1)*sizeof(ProfileNode*));
    memset((char*)pstat, (ptable->NumEntries+1)*sizeof(ProfileNode*), 0);

    
    for (i=0; i<P_TABLE_SIZE; i++) {	/* Examine all profile entries	*/
        ProfileNode *p = ptable->ProfileTable[i];
	
	while (p != NULL) {		/* Walk through chain of nodes	*/
	    int j=0;
	    int k=psize;

	    while ( (j<psize) &&	/* Look for insertion point	*/
		    (pstat[j]->RunCount >= p->RunCount) )
	        j++;
	    while (k>=j) {		/* Shift rest up		*/
	        pstat[k] = pstat[k-1];
		k--;
	    };
	    pstat[j] = p;		/* Insert node			*/
	    
	    psize++;
	    p = p->NextNode;		/* Look at next node		*/
	};
    };

    return(pstat);
}



/*
 * void KillProfileTable(ProfileTable *pt)
 *
 * Clears the Profile Table and deallocates it
 *
 */
void KillProfileTable(pt)
ProfileTable *pt;
{

    ClearProfileTable(pt);			/* Clear contents	*/
    free(pt);					/* Free memory		*/
    
}



/*
 * int GimmeData(int pid, long buffer, long adr, long len)
 *
 * Use ptrace() to transfer data from the process in which it is called
 * (starting at "adr", "len" bytes)  to the process with the given PID
 * into its given buffer.
 * Returns 1 if successful, 0 if failure
 *
 * HACK: TRANSFER DATA IN BLOCKS (problems with more than 4096 bytes at once)
 *
 */
int GimmeData(pid,buffer,adr,len)
int  pid;
long buffer;
long adr;
long len;
{
    int  ret    = 1;			/* Return value "OK"		*/
    long remain = len;			/* Remaining bytes to be sent	*/
    long offset = 0;			/* Offset in both buffers	*/
    long this   = 4000;			/* Size of this transfer block	*/


/*fprintf(stderr,"GimmeData(%d,%d,%d,%d) called\n",pid,buffer,adr,len);
*/
    errno=0;				/* Attach to given process	*/
    ptrace(PTRACE_ATTACH,pid);
/*fprintf(stderr,"GimmeData: attach to pid %d -> errno %d (%s)\n",
  pid,errno,sys_errlist[errno]);
*/
    
    if (errno!=0)
      return(0);			/* Couldn't attach		*/
    errno=0;
    wait(NULL);				/* Wait for process to stop	*/
    
    while (remain > 0) {
	
	if (this>remain)		/* Don't send more than wanted	*/
	  this = remain;
	
	errno=0;			/* Write data to Runner		*/
	ptrace(PTRACE_WRITEDATA,
	       pid,(char*)(buffer+offset),
	       this,(char*)(adr+offset));
/*fprintf(stderr,"GimmeData: write %d bytes from %d to %d -> errno %d (%s)\n",
       this,adr,buffer,errno,sys_errlist[errno]);
*/	
	remain = remain - this;		/* Update offsets & rest count	*/
	offset = offset + this;

	if (errno!=0) {
	    remain = 0;			/* Cancel if error		*/
	    ret    = 0;
	};
	
    };

    ptrace(PTRACE_DETACH,pid,(int*)1,NULL);	/* Let process continue	*/
/*fprintf(stderr,"GimmeData: detached -> errno %d (%s)\n",
  errno,sys_errlist[errno]);
*/

    return(ret);
    
}


/*
 * DumpToFile(char *adr, int len, char *filename)
 *
 * Dumps "len" bytes of memory starting at "adr" to the file "filename"
 * Returns 1 on success, 0 otherwise.
 */
int DumpToFile(adr,len,filename)
char *adr;
int  len;
char *filename;
{
    FILE *file;
    char *p = adr;
    int   wlen;

    if (!(file = fopen(filename,"w"))) {
	printf("DumpToFile: could not open '%s'\n",filename);
        return(0);
    };
    if ( (wlen=fwrite(p,1,len,file)) < len) {
	printf("DumpToFile: wrote only %d bytes (%d wanted)\n",wlen,len);
	return(0);
    };
    fclose(file);
    printf("DumpToFile: wrote %d bytes to '%s'\n",wlen,filename);
    return(1);
}


/*
 * void DebugInitialize(PE)
 *
 * Initialize ID and create first node of debug tree, with given PE, 
 * also create new (global) profile table
 *
 */
void DebugInitialize(PE)
int PE;
{

    DTree = CreateDebugTree(PE);	/* Create new global Debug Tree	   */

    PTable = CreateProfileTable();	/* Create new global Profile Table */

    profile_active = tree_active = 0;	/* Nothing being shown yet	   */

}



/*
 * void DebugEnd()
 *
 * Destroy and deallocate all nodes & the profile table
 *
 */
void DebugEnd()
{
    KillTree(DTree);
    KillProfileTable(PTable);

}



/*
 * void DebugFnStart(type,
 *                   mline,mcolumn, mtoline, mtocolumn,
 *                   actvalue,actarray,
 *                   rangevar,rangelo,rangehi)
 *
 * Creates a new debugging info node.		!!!!!! USES GLOBAL TREE
 * Parent is the currently active node.		!!!!!! "DTree"
 *
 */
void DebugFnStart(type,mline,mcolumn,mtoline,mtocolumn,
                  actvalue,actarray,
                  rangevar,rangelo,rangehi)
int		type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
long		mline;		/* Nine number on Modula-2* Source	*/
long		mcolumn;	/* Dito Column				*/
long		mtoline;	/* End position of current construct	*/
long		mtocolumn;	/* ...dito				*/
long		actvalue;	/* Current activation value		*/
void		*actarray;	/* Array of activation values		*/
void		*rangevar;	/* Variable e.g. of FOR or FORALL	*/
long		rangelo;	/* Range of Variable e.g. for FORALL	*/
long		rangehi;	/* Dito					*/
{
    DebugNode *ActiveNode, *node;


    ActiveNode = DTree->LastNode;	/* !!!!!!! GET ACTIVE NODE HERE  */
                                        /* more complicated if parallel! */

    node = NewNode(DTree,type,ActiveNode,mline,mcolumn,mtoline,mtocolumn,
                   actvalue,actarray,
                   rangevar,rangelo,rangehi);

    AddProfileNode(PTable,node);	/* Add to profile information	*/
}



/*
 * void DebugFnUpdateOLD(...)		 IS NOW A MACRO !!! 
 *
 * Updates profiling information of currently active node.
 *
 */
void DebugFnUpdateOLD(type,mline,mcolumn,mtoline,mtocolumn,
                  actvalue,actarray,
                  rangevar,rangelo,rangehi)
int		type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
long		mline;		/* Nine number on Modula-2* Source	*/
long		mcolumn;	/* Dito Column				*/
long		mtoline;	/* End position of current construct	*/
long		mtocolumn;	/* ...dito				*/
long		actvalue;	/* Current activation value		*/
void		*actarray;	/* Array of activation values		*/
void		*rangevar;	/* Variable e.g. of FOR or FORALL	*/
long		rangelo;	/* Range of Variable e.g. for FORALL	*/
long		rangehi;	/* Dito					*/
{
    DebugNode *ActiveNode;


    ActiveNode = DTree->LastNode;	/* !!!!!!! GET ACTIVE NODE HERE  */
                                        /* more complicated if parallel! */
				        /* DEPENDING ON MODULA INFO?	 */
    (ActiveNode->RunCount)++;

}



/*
 * void DebugFnEnd(...)
 *
 * Deactivate currenty active node.
 *
 */
void DebugFnEnd(type,mline,mcolumn,mtoline,mtocolumn,
                  actvalue,actarray,
                  rangevar,rangelo,rangehi)
int		type;		/* Type, e.g. DN_WHILE, DN_IF, ...	*/
long		mline;		/* Nine number on Modula-2* Source	*/
long		mcolumn;	/* Dito Column				*/
long		mtoline;	/* End position of current construct	*/
long		mtocolumn;	/* ...dito				*/
long		actvalue;	/* Current activation value		*/
void		*actarray;	/* Array of activation values		*/
void		*rangevar;	/* Variable e.g. of FOR or FORALL	*/
long		rangelo;	/* Range of Variable e.g. for FORALL	*/
long		rangehi;	/* Dito					*/
{
    DebugNode *ActiveNode;


    ActiveNode = DTree->LastNode;	/* !!!!!!! GET ACTIVE NODE HERE  */
                                        /* more complicated if parallel! */


/* ALREADY IN UPDATE MACRO    AddProfileNode(PTable,ActiveNode);	/* Update profile information	*/

    (void)KillNode(DTree,ActiveNode);	/* Now kill the useless node	*/
}


