/*
 *  swindow.c
 *
 *  Handle all additional features that come with multiple source
 *  code windows, e.g. des/activation of a window etc
 *
 *    CreateActiveButton():	Create activation button for SWindow
 *    ActiveButtonCall():	Callback procedure for activation button
 *    ActivateSWindow():	Activate one SWindow, desactivate others
 *    CreateSWindow():		Create & initialize new SWindow
 *    DeleteSWindow():		Close and remove SWindow
 *    PopupSWindow():		Create new SWindow and pop it up
 *
 *  Moved here from windows.c :
 *
 *    CreateFileLabel() :	Create file label in file window.
 *    CreateLineLabel() :	Create line label in file window.
 *    CreateFileWindow() :	Create file window.
 *
 *
 *  Stefan Haenssgen 08-01-91
 *
 *  Changes:
 *
 *  08-01-91	Wrote CreateActiveButton, ActiveButtonCall,
 *		 ActivateSWindow
 *		Moved CreateFileLabel, CreateLineLabel, CreateFileWindow
 *		 here from windows.c
 *		Wrote CreateSWindow to create & initialize a new SWindow
 *  08-02-91	Added optional close-Button
 *		Wrote CreateCloseButton, CloseButtonCall
 *		Enhanced CreateSWindow (added parameter "closeable")
 *  08-06-91	Added further comments
 *  08-07-91	Added initialization for line positions etc
 *		Wrote AddSWindowToFile() and RemoveSWindowFromFile()
 *		 to make file/swindow management easier
 *		Call SaveDisplayedFileInfo() when deleting SWindow
 *  08-09-91	Activate an SWindow that is popped up automagically (because
 *		 the user usually wants to load a file next)
 *  08-16-91	Also initialize the newly included struct elements arrow_i,
 *		 updown_i and bomb_i
 *
 *  14-aug-92	Initialize SWType and OtherSW (for later use for Modula-List's)
 *  24-aug-92	Added flag to PopupSW to make the new SWindow in/active
 *  17-nov-92	Removed arrow_i
 *  19-nov-92	Added mlines & clines (for line mapping)
 *  27-nov-92	Added parameters closable,x,y to PopupSWindow
 *		Re-enabled activation on click
 *  04-dec-92	Renamed to "msdb"
 *
 */


#include "global.h"

/* Labels for activation button */

#define IS_ACTIVE_STRING	" ACTIVE "
#define ACTIVATE_STRING		"inactive"
#define CLOSE_STRING		"X"

/* global variables */

SWindow  *swindows[MAX_SWINDOWS];	/* Array of SWindows */
Cardinal swindows_num;			/* Number of created SWindows */

void ActiveButtonCall();		/* Callback for activation button */
void CloseButtonCall();			/* Callback for close button */


/*
 * Three Routines moved here from windows.c:
 *   CreateFileLabel, CreateLineLabel, CreateFileWindow
 *
 * New routines that also use the file window:
 *   CreateActiveButton, CreateCloseButton
 */


static void CreateLineLabel(parent, swindow)
Widget parent;
SWindow *swindow;
{
    Arg 	args[MAXARGS];
    Cardinal 	n;

    n = 0;
    XtSetArg(args[n], XtNlabel, (XtArgVal) "");           		n++;
    XtSetArg(args[n], XtNborderWidth, (XtArgVal) 0);           		n++;
    XtSetArg(args[n], XtNfromHoriz, (XtArgVal) swindow->fileLabel);     n++;
    XtSetArg(args[n], XtNhorizDistance, (XtArgVal) 0);          	n++;
    swindow->lineLabel = XtCreateManagedWidget("lineLabel", labelWidgetClass, 
				      parent, args, n);
}


static void CreateFileLabel(parent, swindow)
Widget parent;
SWindow *swindow;
{
    Arg 	args[MAXARGS];
    Cardinal 	n;

    n = 0;
    XtSetArg(args[n], XtNlabel, (XtArgVal) "No Source File");           n++;
    XtSetArg(args[n], XtNborderWidth, (XtArgVal) 0);           		n++;
    XtSetArg(args[n], XtNfromHoriz, (XtArgVal) swindow->activeButton);	n++;
    XtSetArg(args[n], XtNhorizDistance, (XtArgVal) 0);          	n++;
    swindow->fileLabel = XtCreateManagedWidget("fileLabel", labelWidgetClass, 
				      parent, args, n);
}


/*
 *  void CreateActiveButton(Widget parent, SWindow *swindow)
 *
 *  Creates a command button widget and associates a callback procedure
 *  with it, using "swindow" as call parameter.
 *  Configures some geometry and highlighting behaviour.
 */
void CreateActiveButton(parent, swindow)
Widget parent;
SWindow *swindow;
{
    Arg         a[7];
    Cardinal    n;
    static String actionTranslations =	  /* We do our own set/highlight! */
		"<LeaveWindow>: unhighlight() \n\
		 <Btn1Up>:	notify()";

    /* Place button left of the file label */
    n=0;
    XtSetArg(a[n], XtNlabel, (XtArgVal) ACTIVATE_STRING); n++;
    XtSetArg(a[n], XtNresizable, TRUE); n++;
    XtSetArg(a[n], XtNborderWidth, (XtArgVal) 1); n++;
    swindow->activeButton = XtCreateManagedWidget("activeButton",
                  commandWidgetClass, parent, a, n);

    /* Pass swindow as client data */
    XtAddCallback(swindow->activeButton, XtNcallback, ActiveButtonCall,
                  swindow);

    /* Do our own hightlighting/setting, not the default one */
    XtOverrideTranslations(swindow->activeButton,
                   XtParseTranslationTable(actionTranslations));

    
}


/*
 *  void CreateCloseButton(Widget parent, SWindow *swindow)
 *
 *  Creates a button to allow closing / popping down the SWindow.
 *  Configures some geometry.
 */
void CreateCloseButton(parent, swindow)
Widget parent;
SWindow *swindow;
{
    Arg         a[8];
    Cardinal    n;

    /* Place button right of the line label */
    n=0;
    XtSetArg(a[n], XtNlabel, (XtArgVal) CLOSE_STRING); n++;
    XtSetArg(a[n], XtNborderWidth, (XtArgVal) 0); n++;
    XtSetArg(a[n], XtNinternalHeight, (XtArgVal) 0); n++;  /* no border..    */
    XtSetArg(a[n], XtNinternalWidth, (XtArgVal) 0); n++;   /* ..around label */
    XtSetArg(a[n], XtNfromHoriz, (XtArgVal) swindow->lineLabel); n++;
    XtSetArg(a[n], XtNhorizDistance, (XtArgVal) 0); n++;
    XtSetArg(a[n], XtNresizable, TRUE); n++;
    swindow->closeButton = XtCreateManagedWidget("closeButton",
                  commandWidgetClass, parent, a, n);

    /* Pass swindow as client data */
    XtAddCallback(swindow->closeButton, XtNcallback, CloseButtonCall,
                  swindow);
}


static void CreateFileWindow(parent, swindow, closeable)
Widget parent;
SWindow *swindow;
Boolean closeable;
{
    Arg 	args[MAXARGS];
    Cardinal 	n;

    n = 0;
    XtSetArg(args[n], XtNshowGrip, (XtArgVal) False);	n++;
    XtSetArg(args[n], XtNdefaultDistance, 2);		n++; /* smaller! */
    swindow->fileWindow = XtCreateManagedWidget("fileWindow", formWidgetClass, 
				       parent, args, n);
    CreateActiveButton(swindow->fileWindow, swindow);
    CreateFileLabel(swindow->fileWindow, swindow);
    CreateLineLabel(swindow->fileWindow, swindow);


    /* Also add a close-button if desired */

    if (closeable) {
        CreateCloseButton(swindow->fileWindow, swindow);
    };

    XawFormDoLayout(swindow->fileWindow, TRUE); /* Re-layout the form widget */
}



/*
 *  void ActiveButtonCall(Widget w, XtPointer swindow, XtPointer dummy)
 *
 *  Callback for activeButton - activates the given SWindow by calling
 *  ActivateSWindow
 */
void ActiveButtonCall(w, swindow, dummy)
Widget w;
SWindow *swindow;
XtPointer dummy;
{
    ActivateSWindow(swindow);
}


/*
 *  void CloseButtonCall(Widget w, XtPointer swindow, XtPointer dummy)
 *
 *  Callback for closeButton - close the SWindow / pop it down
 */
void CloseButtonCall(w, swindow, dummy)
Widget w;
SWindow *swindow;
XtPointer dummy;
{

    /* Activate first SWindow if this is the active one */

    if (swindow == a_swindow) {
        ActivateSWindow(swindows[0]);
    };
    XtPopdown(swindow->popupShell);
    DeleteSWindow(swindow);
}


/*
 *  void ActivateSWindow(SWindow *swindow)
 *
 *  Activates the given SWindow and desactivates all other ones
 */
void ActivateSWindow(swindow)
SWindow *swindow;
{
    Arg		a[2];
    Cardinal	i,n;

    /* Set new String (this is the active button) and highlight button */
    n=0;
    XtSetArg(a[n], XtNlabel, (XtArgVal) IS_ACTIVE_STRING);n++;
    XtSetArg(a[n], XtNresizable, (XtArgVal) TRUE);n++;
    XtSetValues(swindow->activeButton, a, n);
    XtCallActionProc(swindow->activeButton, "set",NULL,NULL,ZERO);

    /* This is now the active window */
    a_swindow = swindow;

    /* Mark other windows as inactive */

    XtSetArg(a[0], XtNlabel, (XtArgVal) ACTIVATE_STRING);
    for (i=0; i<swindows_num; i++) {
        if (i != swindow->my_number) {
             XtSetValues(swindows[i]->activeButton, a, 1);
             XtCallActionProc(swindows[i]->activeButton,
                              "unset",NULL,NULL,ZERO);
        }
    }

}


/*
 *  Swindow *CreateSWindow(Widget parent, Boolean closeable)
 *
 *  Allocate memory for an SWindow struct, create the Widgets
 *  for File, Label, Button and Window.
 *  If closeable is true, also add a button to close/popdown the SWindow.
 *  Initialize SWindow and enter it into the global swindows[] array.
 */

SWindow *CreateSWindow(parent, closeable)
Widget parent;
Boolean closeable;
{
    SWindow *swindow;
    int	i;


    /* Check if space left for an SWindow */

    if (swindows_num+1 >= MAX_SWINDOWS) {
        printf(stderr,"Maximum number of source windows exceeded!\n");
        return;
    };

    /* Allocate the space for a swindow and create the subwindows */

    swindow = (SWindow *) XtMalloc(sizeof(SWindow));
    CreateFileWindow(parent, swindow, closeable);
    CreateSourceWindow(parent, swindow);

    /* Init the popup shell info as "not popped up" (may be changed later) */
    /* and set file info to "nothing displayed" */
    /* Init all date structures for signs etc */

    swindow->popupShell = NULL;
    swindow->popupPaned = NULL;
    swindow->displayedFile = NULL;
/* NONONO this would also reset all signs ->    signs_init(swindow); */
    swindow->lines = 0;
    swindow->currentline = 1;
    swindow->topline = 1;
    swindow->bottomline = 0;
    swindow->topPosition = 0;

    /* Init the sign positions in the text */

    swindow->updown_i = 0;
    swindow->bomb_i = 0;

    /* Init the signs (widgets for the signs) */

    for (i=0; i<MAXSIGNS; i++) {
	swindow->bombsign[i].w = NULL;
	swindow->bombsign[i].mapped = FALSE;
	swindow->updownsign[i].w = NULL;
	swindow->updownsign[i].mapped = FALSE;
	swindow->arrowsign[i].w = NULL;
	swindow->arrowsign[i].mapped = FALSE;
	swindow->stopsign[i].w = NULL;
	swindow->stopsign[i].mapped = FALSE;
    }

    /* Init the "related-to" parts */

    swindow->SWType = SWTYPE_C;		/* It's a C listing by default	*/
    swindow->OtherSW= NULL;		/* Nothing related yet		*/


    /* Enter the new SWindow into the global SWindows list */

    swindows[swindows_num] = swindow;
    swindow->my_number = swindows_num;
    swindows_num++;    

    return(swindow);
}


/*
 *  void DeleteSWindow(SWindow *swindow)
 *
 *  Close the SWindow, deallocate its memory and remove it from the
 *  swindows[] array
 */
void DeleteSWindow(swindow)
SWindow *swindow;
{
    int i,n;

    /* Check if user wants to kill the basic SWindow of xdbx */
    n = swindow->my_number;
    if (n==0) {
        fprintf(stderr, "Cannot delete the basic source window!\n");
        return;
    };

    /* Remember cursor positiones etc for the displayed file */

    SaveDisplayedFileInfo(swindow);

    /* Destroy the widgets recursively and free memory */
    /* Also destroy popup shell if SWindow was popped up */

    XtDestroyWidget(swindow->sourceForm);
    XtDestroyWidget(swindow->fileWindow);
    if (swindow->popupShell) {
        XtDestroyWidget(swindow->popupShell);
    };
    XtFree(swindow);
    swindows[n] = NULL;

    /* Shift remaining windows down in array and clear last one */
    for (i=n+1; i<=swindows_num-1; i++) {
        swindows[i-1] = swindows[i];
        swindows[i-1]->my_number = i-1;
    };
    swindows[swindows_num] = NULL;
    swindows_num--;
    
}


/*
 *  SWindow *PopupSWindow(Boolean activate, Boolean closeable,
 *			  int x, int y)
 *
 *  Creates and initializes SWindow, then pops it up in a window
 *  of its own. Makes the new Window the active one if "activate" set.
 *  If "closable" ist true, the Window gets a close button.
 *  "x" and "y" determine the starting coordinates (or let the user
 *  position the window if x = y = -1)
 */
SWindow *PopupSWindow(activate, closable, x, y)
Boolean activate;
Boolean closable;
int x,y;
{
    SWindow	*swindow;
    Widget	SWpopup;	/* Popup shell for SWindow */
    Widget	SWpaned;	/* Paned Widget to contain the child-widgets */
    Arg		a[12];
    Cardinal	n;
    Dimension	w, h_s, h_f;	/* For computing the size of SWindow */
    char	s[128];
    

    /* If an active SWindow exists, make the new one just as big */

    if (a_swindow) {
        n=0;					/* Get size of source window*/
        XtSetArg(a[n], XtNwidth, &w); n++;
        XtSetArg(a[n], XtNheight, &h_s); n++;
        XtGetValues(a_swindow->sourceWindow, a, n);
        n=0;					/* Get size of file window */
        XtSetArg(a[n], XtNheight, &h_f); n++;
        XtGetValues(a_swindow->fileWindow, a, n);

        n=0;					/* Now set size of new SW. */
        XtSetArg(a[n], XtNwidth, w); n++;
        XtSetArg(a[n], XtNheight, h_s+1+h_f); n++;
    } else {
        n=0;					/* No active one, let it be */
    };
    if ((x>=0) && (y>=0)) {
	sprintf(s,"+%d+%d",x,y);
	XtSetArg(a[n], XtNgeometry,s);n++;	/* Set coordinates if given */
    };

    /* Create a popup widget and a paned widget to hold the SWindow */
    /* (use TopLevelShell to get window manager borders, placement etc) */

    XtSetArg(a[n], XtNinput, True); n++;
    XtSetArg(a[n], XtNallowShellResize, True); n++;
    SWpopup = XtCreatePopupShell("MSDB Source Code Window",
                 topLevelShellWidgetClass, toplevel, a, n);
    n=0;
    SWpaned = XtCreateManagedWidget("SWpaned", panedWidgetClass, SWpopup, a,n);


    /* Create the SWindow, init it and pop it all up, not grabbing any input */

    swindow = CreateSWindow(SWpaned, closable);
    swindow->popupShell = SWpopup;
    swindow->popupPaned = SWpaned;
    XtPopup(SWpopup, XtGrabNone);

    /* Activate the newly popped up SWindow if wanted */

    if (activate) {
        ActivateSWindow(swindow);
    } else {
    };

    return(swindow);

}


/*
 * void AddSWindowToFile(SWindow *swindow, FileRec *file)
 *
 * Add the swindow to the array of "where am I displayed"-swindows
 * that is kept in the file struct. Do it only once for each swindow.
 */
void AddSWindowToFile(swindow, file)
SWindow *swindow;
FileRec *file;
{
    int i;

    /* Check if this swindow is already entered here. Don't enter it twice */

    for (i=0; i<file->nsws; i++) {
        if (swindow == file->sws[i]) {
            return;
        };
    };

    file->sws[file->nsws] = swindow;	/* New entry, add it */
    file->nsws++;

}


/*
 * void RemoveSWindowFromFile(SWindow *swindow, FileRec *file)
 *
 * Remove the swindow from the array of "where am I displayed"-swindows
 * that is kept in the file struct.
 */
void RemoveSWindowFromFile(swindow, file)
SWindow *swindow;
FileRec *file;
{

    int i;
    int p=(-1);

    /* look for the entry "swindow" and remember its position */

    for (i=0; i<file->nsws; i++) {
        if (swindow == file->sws[i]) {
            p=i;
            break;
        };
    };
    if (p<0) return;	/* nothing found, do nothing */

    /* shift everything up, deleting the entry for swindow */

    for (i=p; i<file->nsws-1; i++) {
        file->sws[i] = file->sws[i+1];
    };
    file->nsws--;
    file->sws[file->nsws] = NULL;

}
