/*****************************************************************************
 *
 *  xdbx - X Window System interface to the dbx debugger
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation, and that the name of The University of Texas
 *  and Microelectronics and Computer Technology Corporation (MCC) not be 
 *  used in advertising or publicity pertaining to distribution of
 *  the software without specific, written prior permission.  The
 *  University of Texas and MCC makes no representations about the 
 *  suitability of this software for any purpose.  It is provided "as is" 
 *  without express or implied warranty.
 *
 *  THE UNIVERSITY OF TEXAS AND MCC DISCLAIMS ALL WARRANTIES WITH REGARD TO
 *  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TEXAS OR MCC BE LIABLE FOR
 *  ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 *  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 *  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  Author:  	Po Cheung
 *  Created:   	March 10, 1989
 *
 *****************************************************************************/

/*  source.c
 *
 *    Create the source window and handle display of file.
 *
 *    source_init(): 	Initialization routine.
 *    Update():		Action proc to update source window on scrollbar action
 *    NotifyResize():	Action proc to update source window on resize.
 *    CreateSourceWindow(): Create the source window.
 *    BuildLinePos():	Build an array of starting text position of each line.
 *    LookUpFileTable():Check out source file info from a file table.
 *    SaveDisplayedFileInfo(): records displayed file info into file table.
 *    DisplayFile():	Display a file on the source window
 *    LoadFile():	Search for a file and open it for display.
 *
 *
 *  Changes by Stefan Haenssgen:
 *
 *  07-19-91	Rewrote things to use a SWindow struct instead of a single
 *		 window, replacing sourceForm, sourceWindow, CurrentFile,
 *		 displayedFile
 *  07-30-91	Changed action procedures Update() and NotifyResize()
 *		 to take the actual SWindow as parameter; added this
 *		 in CreateSourceWindow()
 *		Pass swindow as parameter to CreateSourceWindow()
 *		Also pass it in translations for Btn1Up etc (override old)
 *  07-31-91    Use a_swindow for globally active one and pass
 *               swindow as parameter for calls that affect just one
 *               certain window (such as action procedures etc)
 *		Modified Update, NotifyResize, CreateSourceWindow,
 *		 LookupFile, SaveDisplayedInfo, DisplayFile, LoadFile,
 *		 LoadCurrentFile  accordingly (use globally or as parameter)
 *  08-07-91	Changed lines, topline, etc from file->* to swindow->*
 *		Added initialization of swindow->line etc to LookUpFileTable
 *		Changed DisplayFile to also get a swindow as parameter
 *		Changed DisplayFile to call AddSWindowToFile
 *		Changed SaveDisplayedFileInfo to call RemoveSWindowFromFile 
 *  08-09-91	Get cursor position from other SWindow displaying
 *		 the same file when opening
 *  08-18-91	Update all signs in LoadFile() every time it is called
 *		 to make screen updates etc consistently (e.g. to display
 *		 an existing stop when the file is loaded into a new window)
 *		 by calling Update() directly
 *		Activate the SWindow automagically when Update() or
 *		 NotifyResize() is called, i.e. the mouse is clicked in it
 *
 *  24-aug-92	Disabled the Activation on Update/Resize (for Modula SW's)
 *  17-nov-92	Changed UpdateArrow() to UpdateArrows()
 *		Introduced CToModulaLine() and ModulaToCLine()
 *  19-nov-92	Rewrote CToModulaLine() and ModulaToCLine()
 *		Wrote BuildLineInfos() to construct line mappings C<->M2* src,
 *		 stepping through lines of C source and parsing information
 *		 left there by the compiler
 *		Wrote PrintLineInfos() for debugging output
 *		Check if mappings already exist (and don't rebuild if yes)
 *		Initialize OtherFile & FileType for files
 *  26-nov-92	Enhanced BuildLineInfos() to also look for DebugFnStart/End
 *		 and remember the corresponding line range etc
 *		Introduced ADT LineStack with functions push/pop/create etc
 *		 to keep track of DebugFn-Calls
 *		Parse "DebugFnXxx" calls using LineParseDF()
 *  27-nov-92	Added Context (par/syn/norm) to line infos, parse & use it
 *		Wrote LSPrint() to print line stack
 *		Changed LS functions - current stack now returned
 *		Added c_swindow & m_swindow for c/modula source
 *		Re-enabled automatic activation on click
 *		Check for corresponding Modula file in LoadFile() (looking
 *		 at corr. SWindows etc)
 *		Smoother mapping from C->Modula
 *		Mapping from Modula->C by following the references from
 *		 C to Modula and filling in the rest
 *  28-nov-92	Improved mapping
 *  29-nov-92	Heuristic to smoothen the line mapping: Ignore all references
 *		 to the start of a Sync Forall when inside one (avoids a lot
 *		 of jumps because of initializations)
 *		Another one: When reverse mapping from Modula to C, check
 *		 for the type and ignore the line info if no match (i.e.
 *		 when in a DebugFnStart, check if this (from) line already
 *		 referenced and overwrite it if wrong type)
 *  30-nov-92	Also parse variable name info into VarNameInfo* table (one
 *		 for each C/Modula file pair)
 *		Renamed BuildLineInfos() to BuildMappingInfos()
 *		Introduced PrintVarInfos() to putput new varname mapping
 *		Use XtMalloc to get space
 *  01-dec-92	More "ironing out" of C/Modula line mappings:
 *		Delete forward-references to line containing a DebugFnStart
 *		Make a second pass over the C->Modula mapping to get
 *		 the final Modula->C mapping
 *		Check for missing infos (e.g. line #0 in DebugFn etc)
 *  10-mar-93	Correct line info for DebugInitialize() (that's where
 *		 the program starts, so it's nice to see the right
 *		 point at least there ;-)
 *		Ignore empty line infos
 *  31-aug-93	Include "DebugFn/DebugFn*"
 *		Use new Modula-2* directory structure:
 *		 C Files in ./imp/SUN4/x.SUN4.c
 *		 M Files in ./msm/x.msm (main program)
 *		     and in ./msi/x.msi (modules)
 *		  Binary in ./bin/SUN4/x
 *		Rewrote MFileName() accordingly
 *
 */

#include <X11/Xos.h>
#include <sys/stat.h>
#include <pwd.h>
#include "DebugFn/DebugFn.h"
#include "global.h"

#define	MAXDIRS	256			/* max number of dirs in dirList */

SWindow		*a_swindow;		/* Contains information on source */
					/* like widgets, files etc. GLOBAL*/
SWindow		*c_swindow;		/* C Source			  */
SWindow		*m_swindow;		/* Corresponding Modula source	  */


static FileRec	**fileTable;		/* table of file records */
static int	fileTableSize;		/* size of file table */
static char 	*dirList[MAXDIRS];	/* list of dirs for searching files */

void source_init()
{
    dirList[0] = NULL;
}

/*
 *  Update topline, bottomline, arrow sign, updown sign, stop signs, and
 *  line label.
 */
/* ARGSUSED */
void Update(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition     pos;
    int			topline;
    FileRec 		*file;

    /* Get the address of our SWindow from the parameters of the action call */
    SWindow		*sw = (SWindow*)(atol(params[0]));

#ifdef DEBUG_OUTPUT
printf("source.c: Update(%d) called\n",sw);
#endif

    if (sw != a_swindow) {
        ActivateSWindow(sw);		/* Activate this window */
    };
    if (sw->displayedFile) {
    	file = sw->displayedFile;
	pos = XawTextTopPosition(sw->sourceWindow);
	sw->topPosition = pos;
	topline = TextPositionToLine(pos,sw);
	/* Update the symbols only if the text scrolls */
	if (sw->topline != topline) {
	    sw->topline = topline;
	    sw->bottomline = MIN (sw->topline + sw->lines - 1, 
				    sw->displayedFile->lastline);
	    XawTextSetInsertionPoint(sw->sourceWindow,
                                     file->linepos[sw->topline]);
	    UpdateLineLabel(sw->topline, sw);
    	    UpdateStops(file);
    	    UpdateArrows(file);
    	    UpdateUpdown(file);
    	    UpdateBomb(file);
	}
	else {/* Update caret position only */
	    pos = XawTextGetInsertionPoint(sw->sourceWindow);
	    UpdateLineLabel(TextPositionToLine(pos,sw),sw);
	}
    }
}

/*
 *  Update bottomline, arrow sign, updown sign and stop signs on resize.
 *  Invoked by ConfigureNotify event.
 */
/* ARGSUSED */
static void NotifyResize(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition pos;
    FileRec	*file;

    /* Get the address of our SWindow from the parameters of the action call */
    SWindow		*sw = (SWindow*)(atol(params[0]));

    TextWidget  ctx = (TextWidget) sw->sourceWindow;


    if (sw != a_swindow) {
        ActivateSWindow(sw);		/* Activate this window */
    };

#ifdef DEBUG_OUTPUT
printf("source.c: NotifyResize(%d) called\n",sw);
#endif

    if (file = sw->displayedFile) {
	sw->lines = ctx->text.lt.lines;
	pos = XawTextTopPosition(sw->sourceWindow);
	sw->topline = TextPositionToLine(pos,sw);
        sw->bottomline = MIN (sw->topline + sw->lines - 1, 
			sw->displayedFile->lastline);
        UpdateStops(file);
        UpdateArrows(file);
        UpdateUpdown(file);
        UpdateBomb(file);
    }
}

/*  Update the position of the caret */
/*  ARGSUSED */
/* DUMMY !!!! NEVER CALLED !!! */
#ifdef notdef
void UpdateLine(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition pos;
    int	line;

    pos = XawTextGetInsertionPoint(w);
    line = TextPositionToLine(pos,a_swindow);
    UpdateLineLabel(line,a_swindow);
}
#endif

/*  My select-start routine that cancels the effect of automatic scrolling
 *  near the bottom of an Athena text widget window.
 */
/*  ARGSUSED */
void SelectStart(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition topPosition;

    /* remember the top display position before automatic scrolling */
    /* displayedFile->topPosition = XawTextTopPosition(w); */
    topPosition = XawTextTopPosition(w);

    XtCallActionProc(w, "select-start", event, params, num_params);

    /* reset to remembered position if top position changed */
    /* if (XawTextTopPosition(w) != swindow->topPosition)
    	TextSetTopPosition(w, swindow->topPosition); */
    if (XawTextTopPosition(w) != topPosition)
    	TextSetTopPosition(w, topPosition);
}

/*  My select-end routine to store the text selection into both the PRIMARY
 *  selection and cut buffer 0. 
 */
/*  ARGSUSED */
void SelectEnd(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition begin, end, start;
    Widget textsrc;
    XawTextBlock buffer;
    char s[10000];
    int nchars;

    XawTextGetSelectionPos(w, &begin, &end);
    XawTextSetSelection(w, begin, end);
    if (begin == end) return;
    textsrc = XawTextGetSource(w);
    strcpy(s, "");
    for (start=begin, nchars=end-begin; nchars > 0; 
	start=begin+buffer.length, nchars-=buffer.length) {
    	XawTextSourceRead(textsrc, start, &buffer, nchars);
	strncat(s, buffer.ptr, buffer.length);
    }
    XStoreBytes(display, s, strlen(s));
}

/*  This is my own select word routine to replace the standard action
 *  procedure provided by the Text widget.
 *  It selects a word delimited by DELIMITERS, not whitespace.
 */
/* ARGSUSED */
void SelectWord(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    XawTextPosition pos, left, right, start;
    XawTextBlock buffer;
    Widget	textsrc;
    char 	s[LINESIZ];
    char 	*p, *ls, *rs;
    int		nchars;

    pos = XawTextGetInsertionPoint(w);
    textsrc = XawTextGetSource(w);

    XawTextSourceRead(textsrc, pos, &buffer, 1);
    if (buffer.length == 0 || (buffer.length == 1 &&
	strchr(app_resources.delimiters, (int)*(buffer.ptr)) != NULL)) {
	XStoreBytes(display, NULL, 0);
	return;
    }

    left = XawTextSourceScan(textsrc, pos+1, XawstWhiteSpace, XawsdLeft, 1,
                             FALSE);
    right = XawTextSourceScan(textsrc, left, XawstWhiteSpace, XawsdRight, 1,
                              FALSE);
    
    strcpy(s, "");
    for (start=left, nchars=right-left; nchars > 0; 
	start=left+buffer.length, nchars-=buffer.length) {
    	XawTextSourceRead(textsrc, start, &buffer, nchars);
	strncat(s, buffer.ptr, buffer.length);
    }

    if (!strcmp(s, "")) return;
    p = s+pos-left;
    ls = (char *) strtok(s, app_resources.delimiters);
    rs = (char *) strtok(NULL, app_resources.delimiters);
    if (!ls) return;
    while (rs<=p && rs!=NULL) {
	ls = rs;
	rs = (char *) strtok(NULL, app_resources.delimiters);
    }
    left = left + ls - s;
    right = left + strlen(ls) - 1; 

    XawTextUnsetSelection(w);
    XStoreBytes(display, ls, strlen(ls));
    XawTextSetSelection(w, left, right+1);
}

/*  Print the value of the expression  in cut buffer 0. */
/*  ARGSUSED */
void PrintSelection(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    char command[LINESIZ];
    char *string;
    int nbytes;

    string = XFetchBytes(display, &nbytes);
    if (nbytes == 0) {
        UpdateMessageWindow(PRINT_HELP, NULL);
        bell(0);
        return;
    }
    sprintf(command, "print %s\n", string);
    send_command(command);
    AppendDialogText(command);
}

/* 
 *  On top of a form widget, we have a text widget with scrollbar, label
 *  widgets for the stop sign, arrow sign, and updown signs.
 */
void CreateSourceWindow(parent,swindow)
Widget parent;
SWindow *swindow;
{
    TextWidget ctx;
    Arg args[MAXARGS];
    Cardinal n;
    char *sbarTranslations;	/* not static! we need a new one */
				/* each time this is called!    */

    static XtActionsRec sbar_actions[] = {
        {"NotifyResize",   NotifyResize},
        {"Update", 	   Update},
        {NULL, NULL}
    };

    /* Pass the address of the actual SWindow as parameter */
    sbarTranslations = (char *)XtMalloc(1024);
    sprintf(sbarTranslations,
       "<Configure>:    NotifyResize(%d) \n\
        <Btn2Down>:     StartScroll(Continuous) MoveThumb() NotifyThumb() \
                        Update(%d) \n\
        <Btn2Motion>:   MoveThumb() NotifyThumb() Update(%d) \n\
        <BtnUp>:        NotifyScroll(Proportional) EndScroll() Update(%d)",
        swindow,swindow,swindow,swindow);

    n = 0;
    XtSetArg(args[n], XtNdefaultDistance, 0);                           n++;
    swindow->sourceForm = XtCreateManagedWidget("sourceForm", formWidgetClass, 
					 parent, args, n);

    n = 0;
    XtSetArg(args[n], XtNborderWidth, 0);				n++;
    XtSetArg(args[n], XtNtype, (XtArgVal)XawAsciiFile);			n++;
    XtSetArg(args[n], XtNstring, (XtArgVal)"/dev/null");		n++;
    XtSetArg(args[n], XtNscrollVertical, (XtArgVal) XawtextScrollAlways);n++;
    swindow->sourceWindow = XtCreateManagedWidget("sourceWindow", asciiTextWidgetClass,
					  swindow->sourceForm, args, n);

    ctx = (TextWidget) swindow->sourceWindow;
    if (ctx->text.vbar)
    	XtOverrideTranslations(ctx->text.vbar, 
				XtParseTranslationTable(sbarTranslations));
    XtAppAddActions(app_context, sbar_actions, XtNumber(sbar_actions));

    /* also give parameters to Update()-calls in the SourceWindow! */

    sprintf(sbarTranslations,
       "Shift<Btn1Up>:          Update(%d) SelectEnd() PrintSelection() \n\
        <Btn1Up>:               Update(%d) SelectEnd() \n",
        swindow,swindow);
    XtOverrideTranslations(ctx,
                           XtParseTranslationTable(sbarTranslations));


    XtFree(sbarTranslations);		/* no longer needed */
}


/*
 *  Build the array which gives the starting text position of each line.
 *  > Estimate the number of lines in the file and allocate memory buffer.
 *  > Starting position of line #1 is 0, and is stored in linepos[1].
 *  > Search for '\n' till end of buffer.
 */
static void BuildLinePos(file)
FileRec *file;
{
    char *p;
    int	 line, nlines;

    nlines = MAX(1, file->filesize/CHARS_PER_LINE);
    file->linepos = (XawTextPosition *)
		    XtMalloc ((nlines+2) * sizeof(XawTextPosition));
    p = file->buf;
    line = 0;
    file->linepos[line++] = 0;
    file->linepos[line++] = 0;
    while (*p) {
	if (*p++ == '\n') {
	    if (line == nlines) { 	/* buffer full, need more memory */
                file->linepos = (XawTextPosition *) XtRealloc (file->linepos, 
			  (nlines + ADD_LINES) * sizeof(XawTextPosition));
		nlines += ADD_LINES;
            }
            file->linepos[line++] = p - file->buf;
	}
    }
    file->lastline = line - 2;
    file->linepos = (XawTextPosition *) XtRealloc 	/* shrink to min size*/
			(file->linepos, line * sizeof(XawTextPosition));
}


/*
 * Look up the file table for an entry with "filename"
 * If not found, create an entry and initialize proper fields,
 * else, return pointer to entry found.
 */
static LookUpFileTable(pathname, filename, file, swindow)
char *pathname, *filename;
FileRec **file;
SWindow *swindow;
{
    struct stat fileinfo;
    int  	fd;
    int 	i, j, n;

    for (i=0; fileTable && fileTable[i] && i<fileTableSize; i++) {
	if (strcmp(fileTable[i]->pathname, pathname) == NULL) { /* file found*/
	    if (stat(pathname, &fileinfo) == -1) {
		UpdateMessageWindow("Error: cannot stat file %s", pathname);
	        *file = fileTable[i];
		return 0;
	    }
	    if (fileinfo.st_mtime > fileTable[i]->mtime) { /* file modified */
		XtFree((char *)fileTable[i]->buf);
		XtFree((char *)fileTable[i]->linepos);
		XtFree((char *)fileTable[i]);
		a_swindow->displayedFile = NULL;
		break;
	    }
	    else if (swindow->displayedFile &&  /* same as displayed file */
		     strcmp(pathname, swindow->displayedFile->pathname) == 0){
		*file = NULL;
		return 0;
	    }
	    else {
	    	*file = fileTable[i];
		return 0;
	    }
	}
    }

    /* Record file into file table */

    if (i == fileTableSize) {		/* file table full, enlarge it */
	fileTableSize += ADD_SIZE;
	fileTable = (FileRec **) 
		     XtRealloc (fileTable, fileTableSize * sizeof(FileRec *));
	for (j=i; j<fileTableSize; j++)
	    fileTable[j] = NULL;
    }
    if ((fd = open(pathname, O_RDONLY)) == -1) {
	UpdateMessageWindow("Error: cannot open file %s", pathname);
	return -1;
    }
    if (fstat(fd, &fileinfo) == -1) {
	UpdateMessageWindow("Error: cannot fstat file %s", pathname);
	close(fd);
	return -1;
    }
    fileTable[i] = (FileRec *) XtMalloc (sizeof(FileRec));
    fileTable[i]->filesize = fileinfo.st_size + 1;
    fileTable[i]->mtime = fileinfo.st_mtime;
    fileTable[i]->buf = XtMalloc((int)fileTable[i]->filesize);
    if ((n = read(fd, fileTable[i]->buf, (int) fileTable[i]->filesize)) == -1){
	UpdateMessageWindow("Error: cannot read file %s", pathname);
	XtFree(fileTable[i]->buf);
	XtFree(fileTable[i]);
	fileTable[i] = NULL;
	close(fd);
	return -1;
    }
    fileTable[i]->buf[n] = '\0';
    fileTable[i]->pathname = XtNewString(pathname);
    fileTable[i]->filename = XtNewString(filename);
    fileTable[i]->old_currentline = 1;
    fileTable[i]->old_topPosition = 0;
    fileTable[i]->nsws = 0;		/* No source windows yet */
    
    fileTable[i]->clines = NULL;	/* No line mapping yet	*/
    fileTable[i]->mlines = NULL;

    fileTable[i]->varinfos = NULL;	/* No variable name mapping yet */
    fileTable[i]->varinfosnum = 0;
    
    fileTable[i]->OtherFile = NULL;
    fileTable[i]->FileType  = SWTYPE_C;	/* Default: C file, no corr.Modula */
    
    swindow->currentline = 1;
    swindow->topline = 1;
    swindow->bottomline = 0;
    swindow->topPosition = 0;
    BuildLinePos(fileTable[i]);
    close(fd);
    *file = fileTable[i];
    for (i=0; i<MAX_SWINDOWS; i++)
        ((*file)->sws)[i] = NULL;	/* clear swindow-infos */
    return 0;
}

/*  
 *  Remember file position and current line before closing.
 */
void SaveDisplayedFileInfo(swindow)
SWindow *swindow;
{
    XawTextPosition pos;

    if (swindow->displayedFile) {
    	swindow->displayedFile->old_topPosition =
                                XawTextTopPosition(swindow->sourceWindow);
	pos = XawTextGetInsertionPoint(swindow->sourceWindow);
	swindow->displayedFile->old_currentline =
				TextPositionToLine(pos,swindow);
        RemoveSWindowFromFile(swindow, swindow->displayedFile);
    }
}


/*   DisplayFile() displays the file onto the source window.  It
 *     uses topPosition to remember where it was last opened.  But it
 *     must recalculate bottomline because the window size might be
 *     different.
 */
static void DisplayFile(swindow,file)
SWindow *swindow;
FileRec *file;
{
    Arg 	args[MAXARGS];
    Cardinal 	n;
    TextWidget 	ctx = (TextWidget) swindow->sourceWindow;
    XawTextPosition	top;
    int			cur;

    /* enter into array of swindows that are displaying this file*/
    AddSWindowToFile(swindow, file);

    /* Get cursor position from other SWindow that displays this file, if any*/
    /* (can't refer to a_swindow sensibly because files are loaded only */
    /*  into the active SWindow, and that's this one itself...) */

    if (file->nsws > 1) {
        XawTextPosition pos;

        top = XawTextTopPosition(
                     ((SWindow *)(file->sws[(file->nsws)-2]))->sourceWindow);
	pos = XawTextGetInsertionPoint(
                     ((SWindow *)(file->sws[(file->nsws)-2]))->sourceWindow);
        cur = TextPositionToLine(pos,(SWindow *)(file->sws[(file->nsws)-2]));
    } else {
        top = file->old_topPosition;
        cur = file->old_currentline;
    };
    n = 0;
    XtSetArg(args[n], XtNdisplayPosition, (XtArgVal) top);		n++;
    XtSetArg(args[n], XtNstring, (XtArgVal) file->pathname);		n++;
    XtSetArg(args[n], XtNeditType, (XtArgVal) XawtextRead);		n++;
    XtSetValues(swindow->sourceWindow, args, n);
    swindow->lines = ctx->text.lt.lines;
    swindow->bottomline = MIN (swindow->topline + swindow->lines - 1,
				 file->lastline);
    swindow->topPosition = top;
    swindow->currentline = cur;
}


/*  Given a filename starting with a tilde (`~'), it expands ~[user] to
 *  the home directory of that user, or to the login home directory if user
 *  is not specified.
 */
static char *expand(filename)
char *filename;
{
    struct passwd *pwd;
    char 	  *string, *name, newfile[MAXNAME];

    string = XtNewString(filename+1);
    if (*string == '\0' || *string == '/')
	name = (char *) getlogin();
    else
    	name = (char *) strtok(string, "/");
    if (name == NULL)
	return filename;
    pwd = (struct passwd *) getpwnam(name);
    if (pwd && pwd->pw_dir) {
    	sprintf(newfile, "%s%s", pwd->pw_dir, filename+strlen(name)+1);
    	return XtNewString(newfile);
    }
    else
	return filename;
}


/*  Create a list of directories for searching source files.
 *  It reads the list of directories specified by the user, adding
 *  the current directory into the list if it is not already there.
 *
 *  With fix from Dave Gagne (daveg@fs1.ee.ubc.ca) 7/30/90
 */
void MakeDirList(output)
char *output;
{
    char *s, list[BUFSIZ], command[LINESIZ];
    int  i, use_cwd;

    for (i=0; dirList[i]; i++)			/* remove old list */
	XtFree(dirList[i]);
    i = 0;
    use_cwd = TRUE;
    if (output) {                                        /* create list */
        s = (char *) strtok(output, " \n");
        while (s) {
            dirList[i] = XtNewString(s);

            if (dirList[i][0] == '~')                   /* expand '~' */
                dirList[i] = expand(dirList[i]);
            if (LASTCH(dirList[i]) == '/')              /* remove last '/' */
                LASTCH(dirList[i]) = '\0';
            if (strcmp(dirList[i], ".") == NULL)        /* watch for "." */
                use_cwd = FALSE;

            ++i;
            s = (char *) strtok(NULL, " \n");
        }
        dirList[i] = NULL;
    }

    if (use_cwd) {				/* include current dir */
	dirList[i++] = XtNewString(".");		
    	dirList[i] = NULL;
    }
    strcpy(list, "");				/* tell dbx our new list */
    for (i=0; dirList[i]; i++) {
	strcat(list, dirList[i]);
	strcat(list, " ");
    }
    sprintf(command, "use %s\n", list);
    Parse = False;
    query_dbx(command);
}


/*  Returns the full pathname of a given file.
 *  It searches for the file from a list of directories.
 */
char *GetPathname(filename)
char *filename;
{
    char	pathname[LINESIZ];
    int 	i;

    if (filename == NULL || strcmp(filename, "") == NULL)
	return NULL;
    for (i=0; dirList[i]; i++) {
	if (*filename == '/' && access(filename, R_OK) == -1) { 
	    /* this handles the exceptional case of sun4 dbx output */
	    strcpy(filename, &filename[1]);
	}
	if (*filename == '/' || *filename == '~')
	     strcpy(pathname, filename);
	else if (strcmp(dirList[i], ".") == NULL)
	     sprintf(pathname, "%s/%s", cwd, filename);
	else if (*dirList[i] == '/' || *dirList[i] == '~')
	     sprintf(pathname, "%s/%s", dirList[i], filename);
	else
	     sprintf(pathname, "%s/%s/%s", cwd, dirList[i], filename);
	if (access(pathname, R_OK) == 0)
	    return XtNewString(pathname);
    }
    UpdateMessageWindow("File not found: %s", filename);
    return NULL;
}



/*
 * char *MFileName(char *cname)
 *
 * Generate name of Modula source file from given C name
 * (obeying new Modula-2* directory structure).
 * Also checks if this file exists and return NULL if not (or not ".c" suffix)
 *
 */
char *MFileName(cname)
char *cname;
{
    int  clen   = strlen(cname);
    char *mname = NULL;
    struct stat sbuf;

    /* Check if name is "*.c" - if yes then generate Modula name */

    if ((clen>1) && (cname[clen-2] == '.') && (cname[clen-1] == 'c')) {
	char tmpname[BUFSIZ];
	char filename[BUFSIZ];
	char *pos;

	if (pos = strrchr(cname,'/')) {	/* Get file name without .SUN4.c */
	    char *pos2=strstr(pos,".SUN4.c");
	    int i=0;
	    if (pos2==NULL) {
/*fprintf(stderr,"MFileName: %s doesn't contain '.SUN4.c'\n",cname);*/
                return(NULL);
	    };
	    pos++;
	    while (pos<pos2)		/* Copy name		*/
	      filename[i++]=*pos++;	/*  without .SUN4.c	*/
	    filename[i]='\0';

	} else {
fprintf(stderr,"MFileName: couldn't get file name from %s\n",cname);
            return(NULL);
        };
/*fprintf(stderr,"MFileName: file name = %s (from %s)\n",filename,cname);*/

	/* Now, check if corresp. msi or msd file exists */
	
        strcpy(tmpname,cname);
	if (pos = strstr(tmpname,"imp/SUN4/")) {
	    sprintf(pos,"msi/%s.msi",filename);	/* Add filename +".msi"	    */
	    if (stat(tmpname,&sbuf) != 0) {	/* Check if file exists	    */
		sprintf(pos,"msm/%s.msm",filename); 	/* Try ".msm" 	    */
		if (stat(tmpname,&sbuf) != 0) {	
/*		    sprintf(pos,"msd/%s.msd",filename); /* Try ".msd"       */
/*		    if (stat(tmpname,&sbuf) != 0) {*/
/*fprintf(stderr,"MFileName: neither '.msi' nor '.msm' nor '.msd' file found\nfor %s\n",cname);*/
		        return(NULL);
/*                    }*/
                }
	    }
/*fprintf(stderr,"MFileName: found file %s\n",tmpname);*/
	} else {
fprintf(stderr,"MFileName: couldn't get 'imp' path name from %s\n",mname);
            return(NULL);
        };

	mname = (char*) XtMalloc(strlen(tmpname)+1);
	sprintf(mname,"%s",tmpname);
	
    } else {
fprintf(stderr,"MFileName: %s is no *.c file\n",cname);
    };
    return(mname);
}


/*
 * Given a file name, LoadFile attempts to open it and displays it onto
 * the source window:
 *   1. get the full pathname of the file
 *   2. LookUpFileTable() returns a pointer to the file's entry if it's
 *      already in the table; else, creates an entry and return a pointer.
 *   3. save the current displayedFile info
 *   4. display the file
 *   5. update the file label and the various signs on the source window.
 *  LoadFile returns 0 upon successful completion, -1 otherwise.
 *
 * Also tries to load the corresponding Modula source, if swindow is
 * a C Window and has a corr. Modula Window
 *
 */
int LoadFile(filename, swindow)
char *filename;
SWindow *swindow;
{
    FileRec 	*file;
    char	*pathname;

    pathname = GetPathname(filename);
    if (pathname == NULL) { 
fprintf(stderr,"ERROR WITH PATH OF FILE %s\n",filename);
	return -1;
    }
    if (LookUpFileTable(pathname, filename, &file, swindow) != -1) {
	if (file) {	/* load new file */
            char *s = (char *)XtMalloc(512);
            Cardinal x = 1;

/*fprintf(stderr,"LOADING FILE %s (PATH %s)\n",filename,pathname);*/
	    SaveDisplayedFileInfo(swindow);
	    swindow->displayedFile = file;
	    DisplayFile(swindow,file);
	    UpdateFileLabel(pathname,swindow);
	    XawTextUnsetSelection(swindow->sourceWindow);
	    XawTextSetInsertionPoint(swindow->sourceWindow,
                                     file->linepos[swindow->currentline]);
            sprintf(s,"%d",swindow);
            Update(swindow->sourceForm,NULL,&s,&x); /* Update all signs! */
            XtFree(s);
/*	    UpdateLineLabel(swindow->currentline,swindow);
	    UpdateStops(file);
	    UpdateArrows(file);
	    UpdateUpdown(file);
	    UpdateBomb(file);
*/
        } else {	/* also update windows if file is same as disp.File */
/*            file = swindow->displayedFile;
	    SaveDisplayedFileInfo(swindow);
	    swindow->displayedFile = file;
	    DisplayFile(swindow,file);
	    UpdateFileLabel(pathname,swindow);
	    XawTextUnsetSelection(swindow->sourceWindow);
	    XawTextSetInsertionPoint(swindow->sourceWindow,
                                     file->linepos[swindow->currentline]);
	    UpdateLineLabel(swindow->currentline,swindow);
	    UpdateStops(file);
	    UpdateArrows(file);
	    UpdateUpdown(file);
	    UpdateBomb(file);
*/
        }
	
	/* Also load corresponding Modula source, if any */
	
	if ((swindow->SWType == SWTYPE_C) && (swindow->OtherSW) &&
	    (swindow->OtherSW->SWType == SWTYPE_MODULA)) {

	    SWindow	*mwindow = swindow->OtherSW;
	    char	*mname;
	    FileRec	*mf, *cf;

	    if (mname = MFileName(filename)) {
		
		LoadFile(mname,mwindow);	/* Load the file */
		cf = swindow->displayedFile;
		mf = mwindow->displayedFile;
		
		cf->OtherFile = mf;		/* Init corresp's & types    */
		mf->OtherFile = cf;
		mf->FileType  = SWTYPE_MODULA;
		BuildMappingInfos(mf,cf);	/* Build var/line mappings   */

	    };

	};
    	return 0;
    }
    else {		/* LookUpFileTable() fails */
    	return -1;
    }
}

int LoadCurrentFile(swindow)
SWindow *swindow;
{
    query_dbx("file\n");
    return LoadFile(swindow->CurrentFile, swindow);
}


/*
 * CToModulaLine(SWindow *sw, int line)
 *
 * Given a line number in the C source, returns the corresponding line
 * in the equivalent Modula-2* source
 *
 */
int CToModulaLine(sw, line)
SWindow *sw;
int line;
{
    
    if ((sw == NULL) || (sw->displayedFile == NULL) ||
	(sw->displayedFile->mlines == NULL))
      return(0);				  /* No info, ret. default   */

    if ((line < 1) || (line > sw->displayedFile->lastline))
      return(0);				  /* Out of range	     */
    
    return(sw->displayedFile->mlines[line]->Line);/* Give corresponding line */
    
}



/*
 * ModulaToCLine(SWindow *sw, int line)
 *
 * Given a line number in the Modula source, returns the corresponding line
 * in the equivalent C source
 *
 */
int ModulaToCLine(sw, line)
SWindow *sw;
int line;
{
    
    if ((sw == NULL) || (sw->displayedFile == NULL) ||
	(sw->displayedFile->clines == NULL))
      return(0);				  /* No info, ret. default   */

    if ((line < 1) || (line > sw->displayedFile->lastline))
      return(0);				  /* Out of range	     */
    
    return(sw->displayedFile->clines[line]->Line);/* Give corresponding line */
    
}


LineStack* LSCreate()		/* Create new line stack ADT		*/
{
    LineStack *ls =  (LineStack*) XtMalloc(sizeof(LineStack));

    ls->Type	= DN_ROOT;
    ls->Context	= DN_SEQ_CONTEXT;
    ls->LineFrom= 0;
    ls->LineTo	= 0;
    ls->Up	= NULL;
    ls->Down	= 0;

    return(ls);
};

LineStack* LSPush(ls,type,context,from,to)/* Add info at top of line stack */
LineStack *ls;
int type,context,from,to;
{
    LineStack* newls;
    
    if (ls==NULL) return(ls);

    newls = (LineStack*) XtMalloc(sizeof(LineStack));
    newls->Type     = type; 
    newls->Context  = context;
    newls->LineFrom = from;
    newls->LineTo   = to;
    newls->Up       = NULL;
    newls->Down     = ls;
    ls   ->Up       = newls;
    ls = newls;
    return(newls);
}

LineStack* LSPop(ls,type,context,from,to)/* Remove topmost element */
LineStack *ls;
int *type, *context, *from, *to;
{
    LineStack *prevls;
    
    if (ls==NULL) return(ls);

    *type   = ls->Type;
    *context= ls->Context;
    *from   = ls->LineFrom;
    *to     = ls->LineTo;
    prevls  = ls;
    if (ls->Down != NULL) {	/* Don't ever remove last element	*/
	ls = ls->Down;
	ls->Up = NULL;
	XtFree(prevls);
    };
    return(ls);

}

LineStack* LSTop(ls,type,context,from,to) /* Return values of topmost element*/
LineStack *ls;
int *type, *context, *from, *to;
{
    if (ls==NULL)
      return(ls);
    
    *type   = ls->Type;
    *context= ls->Context;
    *from   = ls->LineFrom;
    *to     = ls->LineTo;

    return(ls);
}

Boolean LSEmpty(ls)		/* True if no more elements (just root)	*/
LineStack *ls;
{
    return((ls==NULL)||(ls->Down==NULL));	/* At bottom */
}


void LSDestroy(ls)		/* Destroy complete line stack		*/
LineStack *ls;
{
    int dummy;
    
    while(!LSEmpty(ls))
      LSPop(ls,&dummy,&dummy,&dummy,&dummy);
    if (ls)
      XtFree(ls);
}

void LSPrint(ls)		/* Output whole line stack		*/
LineStack *ls;
{
    LineStack *l2;

    if (ls==NULL)
      return;
    
    l2 = ls;
    while (l2->Down)	/* Goto bottom */
      l2 = l2->Down;
    while (l2) {	/* Print all */
/*	fprintf(stderr,"T%2d C%2d F%2d-T%2d / ",
		l2->Type, l2->Context, l2->LineFrom, l2->LineTo);*/
	l2 = l2->Up;
    };
/*    fprintf(stderr,"\n");*/

}


/*
 * char *LineParseDF(char *str, int *type, *context, *from, *to)
 *
 * Parses the given string (which points to the beginning of "DebugFn"),
 * placing the results for type, from-line and to-line in the variables.
 * Assigns a new context if FORALL encoutered, doesn't change it otherwise.
 * Returns a pointer to the "," after the MToColumn
 */
char *LineParseDF(str,type,context,from,to)
char *str;
int *type,*context,*from,*to;
{
    char *q;
    char *r;
    char  s[128];

    if (str==NULL)
      return(NULL);
    
    q = (char*)strchr(str,'('); q++;
    if (strncmp(q,"DN_ROOT",7) == 0) {		/* Get type from string	*/
	*type = DN_ROOT;
    } else if (strncmp(q,"DN_WHILE",8) == 0) {
	*type = DN_WHILE;
    } else if (strncmp(q,"DN_REPEAT",9) == 0) {
	*type = DN_REPEAT;
    } else if (strncmp(q,"DN_LOOP",7) == 0) {
	*type = DN_LOOP;
    } else if (strncmp(q,"DN_FOR",6) == 0) {
	*type = DN_FOR;
    } else if (strncmp(q,"DN_SYN_FORALL",13) == 0) {
	*type = DN_SYN_FORALL;
	*context = DN_SYN_CONTEXT;
    } else if (strncmp(q,"DN_PAR_FORALL",13) == 0) {
	*type = DN_PAR_FORALL;
	*context = DN_PAR_CONTEXT;
    } else if (strncmp(q,"DN_THEN",7) == 0) {
	*type = DN_THEN;
    } else if (strncmp(q,"DN_ELSE",7) == 0) {
	*type = DN_ELSE;
    } else if (strncmp(q,"DN_CASE",7) == 0) {
	*type = DN_CASE;
    } else if (strncmp(q,"DN_OF",5) == 0) {
	*type = DN_OF;
    } else {
	*type = 0;
    };
		
    q = strchr(q,','); q++;
    r = strchr(q,',');
    strncpy(s,q,r-q);
    s[r-q]='\0';
    *from = atol(s);			/* Get from-line	*/
    r++;
    q = strchr(r,','); q++;
    q = strchr(q,','); q++;		/* Skip from-column	*/
    r = strchr(q,',');
    strncpy(s,q,r-q);
    s[r-q]='\0';			/* Get to-line		*/
    *to = atol(s);
    q = strchr(r,','); q++;		/* Skip to-column	*/

    return(q);				/* Return prt to rest	*/
}



/*
 *  void BuildMappingInfos(FileRec *mfile, FileRec *cfile)
 *
 *  Build arrays to map Modula- to C line numbers, based on information found
 *  in the msc-generated C source. Fills in the "mline" and "cline" arrays
 *  in both files. "mfile" is the Modula-2* source file, "cfile" the C source.
 *  Also generates variable name mapping infos "varinfos"
 *
 */
void BuildMappingInfos(mfile, cfile)
FileRec *mfile;
FileRec *cfile;
{
    char     *p;		/* Pointer to search through C source	*/
    LineInfo **mlines;		/* Line mapping C->Modula		*/
    LineInfo **clines;		/* Line mapping Moduka->C		*/
    LineInfo *li;		/* One line information entry		*/
    LineInfo *lastli;		/* Most recent line info		*/
    int       l;		/* Counter through C source lines	*/
    char      filename[BUFSIZ];	/* Name of modula file			*/
    char      s[BUFSIZ];	/* Tmp buffer				*/
    int       line;		/* Current line				*/
    LineStack*lstack;		/* Line info stack for structuring	*/
    int	      type;		/* Type of construct (DN_WHILE etc)	*/
    int       from, to;		/* Line range of construct		*/
    int       context;		/* Context of construct (sync/par/norm)	*/
    VarNameInfo **vni;		/* Mapping of variable names		*/
    int           vninum;	/* Number of entries			*/
    

    /* Check for bad parameters */
    
    if ((mfile == NULL) || (cfile == NULL) ||
	(mfile->buf == NULL) || (cfile->buf == NULL))
      return;

    /* Check if line mapping information already exists */
    
    if ((mfile->clines != NULL) && (mfile->mlines != NULL) &&
	(cfile->clines != NULL) && (cfile->mlines != NULL))
      return;

    
    /* Allocate pointer arrays for line infos & init to zero	*/
    /* Also init var-info table (there cannot be more variables	*/
    /* than Modula source lines :-} .. shrink later)		*/
    
    mlines = (LineInfo**) XtMalloc((cfile->lastline+1)*sizeof(LineInfo*));
    memset((char*)mlines,0,(cfile->lastline+1)*sizeof(LineInfo*));
    clines = (LineInfo**) XtMalloc((mfile->lastline+1)*sizeof(LineInfo*));
    memset((char*)clines,0,(mfile->lastline+1)*sizeof(LineInfo*));

    vni = (VarNameInfo**) XtMalloc((mfile->lastline+1)*sizeof(VarNameInfo*));
    memset((char*)vni,0,(mfile->lastline+1)*sizeof(VarNameInfo*));
    vninum = 0;

    lastli = (LineInfo*) XtMalloc(sizeof(LineInfo));
    lastli->Line     = 1;
    lastli->Function = NULL;			/* Init first entry	*/
    lastli->File     = NULL;
    lastli->Type     = DN_ROOT;
    lastli->Context  = DN_SEQ_CONTEXT;

    lstack = LSCreate();

    
    /* Go through C source and look for Modula-source correspondences	*/

    
    type   = DN_ROOT;
    context= DN_SEQ_CONTEXT;			/* Init context etc	*/

    for (l=1; l<=cfile->lastline; l++) {
	char *q;
	
	p = cfile->buf + cfile->linepos[l];


        /* Look for DebugFnXxx calls and generate ranges of lines */

        q=p;
        while ( (*q == ' ') || (*q == '\t') ) q++;  /* Skip spaces etc	*/

	if ( strncmp(q,"DebugFnStart(",13) == 0) {/* "Start" in this line? */
	    int i;

	    q=q+12;
	    q = LineParseDF(q,&type,&context,	/* Parse infos & advance q */
			    &from,&to);
	    lstack=LSPush(lstack,type,context,from,to);/* Put on stack	*/

	    /* Overwrite any forward references to this DebugFnStart 	*/
	    /* as they lead to confusing mappings back from Modula to C	*/
	    /* (replace them by their preceeding line, instead)		*/
	    
	    for (i=1; i<l; i++) {
		if (mlines[i]->Line == from) {
		    *mlines[i] = *mlines[i-1];	/* Replace w/last (gives*/
		};				/*  dupicates, well...)	*/
	    };/* from==0 no problem, since no Line==0 */
	    
	    /* Also enter DebugFnStart-info into line infos */

	    if (from>0) {	/* Only if Modula-lines known */
		li = (LineInfo*) XtMalloc(sizeof(LineInfo));/* New entry*/
		li->File = strdup(filename);	/*!!!!!!! OVERHEAD! */
		li->Line    = from;
		li->Function= NULL;		/*!!!!!!! FIND OUT, TOO */
		li->Type    = type;
		li->Context = context;
		mlines[l] = li;			/* Remember new info	*/
		lastli = li;
	    }
	    
	} else if (strncmp(q,"DebugFnEnd(",11) == 0) {/*"Start" in this line?*/

    	    lstack=LSPop(lstack,&type,&context,	/* Remove from stack,	*/
		  &from,&to);
	    LSTop(lstack,&type,&context,	/*  remember next	*/
		  &from,&to);
	    
	};

	
	/* Look for direct line mapping info			*/
	/* Format:						*/
	/* #&!("<modula file name",<c line>,<modula line>)	*/

	
	p = cfile->buf + cfile->linepos[l];

	if ( (strncmp(p,"/* #&!(",6) == 0) &&   /* Line info found?     */
	    !(strncmp(p,"/* #&!() */",11)==0) ){/*  and not empty?      */
	    char *f; /* file name */
	    /*!!!!!!! LINE INFO MUST BE AT START OF LINE */

	    /* Enter in C-to-Modula table */
	    
	    p = p + 7;
	    f = p;				/* Scan for file name	*/
	    while (*p != ',')
	      p++;
	    p++;
	    strncpy(filename, f, p-f);		/* Copy file name	*/
	    filename[p-f]='\0';
	    f = p;
	    while (*p != ',')			/* Scan for line number	*/
	      p++;
	    strncpy(s,f,p-f);
	    s[p-f]='\0';			/* Convert line number	*/
	    line = atoi(s);
	    if ((line>0) &&
		((type != DN_SYN_FORALL) || (line > lastli->Line))
		){ /* When in sync forall, ignore jumps back to start of SFA */
		    

		li = (LineInfo*) XtMalloc(sizeof(LineInfo));/* New entry*/
		li->File = strdup(filename);/*!!!!!!! OVERHEAD, ONE COPY ENOUGH!!*/
		li->Line    = line;
		li->Function= NULL;		/*!!!!!!! FIND OUT, TOO */
		li->Type    = type;
		li->Context = context;
		mlines[l] = li;			/* Remember new info	*/
		lastli = li;
		
		/* Special case for DebugInitialize(), which represents */
		/*  the start of the program... there are no line infos */
		/*  before it, so we change them now. Looks bad if we   */
		/*  start a program and the PCs are not at the beginning*/
		
		if (l>2) {
		    char *pp = cfile->buf + cfile->linepos[l-2];
		    if (strncmp(pp,"DebugInitialize(",15) == 0) {
			mlines[l-2] = mlines[l-1] = li;
		    };
		};
		
		  
	    } else {
		mlines[l] = lastli;
	    };
	    
	} else {			/* Nothing found, copy old */
	    
	    mlines[l] = lastli;
	};


	/* Look for variable name mapping info	  */
	/* Format:				  */
	/* #&v(<modula name>,<c name>,"<c type>") */

	/*!!!!!!! OR THE OTHER WAY ROUND - C THEN MODULA ??? */
	
	if (strncmp(p,"/* #&v",6) == 0) {	/* Line info found?	*/
	    char *v;	/* Temp. variable name */
	    /*!!!!!!! VARNAME INFO MUST BE AT START OF LINE */
	    
	    vni[vninum] = (VarNameInfo*)XtMalloc(sizeof(VarNameInfo));

	    /* Enter in VarName table */
	    
	    p = p + 7;
	    v = p;				/* Scan for Modula name	*/
	    while (*p != ',')
	      p++;
	    *p='\0';
	    vni[vninum]->MName=strdup(v);	/* Copy M name		*/
	    *p=',';/*quickhack:end string artificially, replace "," again*/
	    p++;
	    v = p;				/* Scan for C name	*/
	    while (*p != ',')
	      p++;
	    *p='\0';
	    vni[vninum]->CName = strdup(v);	/* Copy C name		*/
	    *p=',';
	    p++;
	    p++;
	    v=p;
	    while (*p != '"')			/* Scan for c type name	*/
	      p++;
	    *p='\0';
	    vni[vninum]->CType = strdup(v);	/* Copy type name	*/
	    *p='"';

	    vni[vninum]->MType = NULL;		/* Not provided yet	*/
	    vni[vninum]->CLine = l-1;		/* Def. on previous line*/
	    vni[vninum]->MLine = mlines[l-1]->Line; /* Corresp. Modula	*/

	    vninum++;
	};
	
    };

    
    /* Shrink var info buffer to minimum size needed */

    vni = (VarNameInfo**) XtRealloc(vni,(vninum+1)*sizeof(VarNameInfo*));
    
    
    /* Now the Modula-to-C mapping, based on the reverse pointers	*/
    /* previously found in the C source					*/

    /* Copy info from C Source */
    
    for (l=1; l<=cfile->lastline; l++) {/* Copy all infos from C	*/
	line = mlines[l]->Line;
	if (clines[line] == NULL) {
	    li = (LineInfo*) XtMalloc(sizeof(LineInfo));/* New entry */
	    li->File = strdup(cfile->filename);/*!!!!!!! OVERHEAD*/
	    li->Line    = l;
	    li->Function= NULL;	/*!!!!!!! FIND OUT, TOO */
	    li->Type    = type;
	    li->Context = context;
	    clines[line] = li;
	};
    };

    /* Fill in gaps */
    
    line   = 1;
    type   = DN_ROOT;
    context= DN_SEQ_CONTEXT;			/* Init context etc	*/
    li = (LineInfo*) XtMalloc(sizeof(LineInfo));
    li->File    = NULL;
    li->Line    = 1;				/* Default line for	*/
    li->Function= NULL;				/*  start of program	*/
    li->Type    = type;
    li->Context = context;
    
    for (l=1; l <= mfile->lastline; l++) {	/* Copy last known line	*/
	if (clines[l] == NULL) {		/*  to unknown ones	*/
	    clines[l] = li;
	} else {
	    li = clines[l];			/* More recent, remember*/
	};
    };

    /* Done mapping, remember results */
    
    cfile->mlines = mlines;			/* Remember results	*/
    cfile->clines = clines;
    mfile->mlines = mlines;
    mfile->clines = clines;
    cfile->varinfos    = vni;
    mfile->varinfos    = vni;
    cfile->varinfosnum = vninum;
    mfile->varinfosnum = vninum;

    LSDestroy(lstack);				/* Free line stack	*/

/*    PrintLineInfos(cfile);/**/
/*    PrintLineInfos(mfile);/**/
    PrintVarInfos(mfile);/**/
}


void PrintLineInfos(file)
FileRec *file;
{
    int		  l;
    char	 *p;
    LineInfo	**li;

    if ((file == NULL) || (file->mlines == NULL) || (file->clines == NULL))
      return;

/*    fprintf(stderr,"\nLine Infos for file <%s> (%s):\n", file->filename,
	    (file->FileType==SWTYPE_MODULA) ? "MODULA" : "C");*/

    li = (file->FileType==SWTYPE_MODULA) ? file->clines : file->mlines;
    
    for (l=1; l<=file->lastline; l++) {
	char *q;
	
	p = file->buf + file->linepos[l];

	if (q=strchr(p,'\n')){
	    if (q-p > 55)
	      q=p+55;
	    *q='\0';
/*	    fprintf(stderr,"%5d: %-55s | ",l,p);*/
	    *q='/n';
	};
/*	if (li[l]) 
	  fprintf(stderr,"T%2d C%2d -> %3d\n",
		  li[l]->Type, li[l]->Context,li[l]->Line);
	else
	  fprintf(stderr,"<NO INFO>\n");*/
	

    };

};


void PrintVarInfos(file)
FileRec *file;
{
    int		   l;
    char	  *p;
    VarNameInfo	 **vni;

    if ((file == NULL) || (file->varinfos == NULL) || (file->varinfosnum==0))
      return;

/*    fprintf(stderr,"\n%d Variable Mapping Infos for file '%s' (%s):\n",
	    file->varinfosnum,file->filename,
	    (file->FileType==SWTYPE_MODULA) ? "MODULA" : "C");*/

    vni = file->varinfos;

/*    fprintf(stderr,"%-16s    %-16s (%-22s) %6s %6s\n",
	    "Modula name","C Name","C Type","MLine","CLine");*/
    for (l=0; l<file->varinfosnum; l++) {
/*	fprintf(stderr, "%-16s -> %-16s (%-22s) %6d %6d\n",
		vni[l]->MName,vni[l]->CName,vni[l]->CType,
		vni[l]->MLine,vni[l]->CLine);*/
    };

};
