/*****************************************************************************
 *
 *  xdbx - X Window System interface to the dbx debugger
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation, and that the name of The University of Texas
 *  and Microelectronics and Computer Technology Corporation (MCC) not be 
 *  used in advertising or publicity pertaining to distribution of
 *  the software without specific, written prior permission.  The
 *  University of Texas and MCC makes no representations about the 
 *  suitability of this software for any purpose.  It is provided "as is" 
 *  without express or implied warranty.
 *
 *  THE UNIVERSITY OF TEXAS AND MCC DISCLAIMS ALL WARRANTIES WITH REGARD TO
 *  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TEXAS OR MCC BE LIABLE FOR
 *  ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 *  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 *  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  Author:  	Po Cheung
 *  Created:   	March 10, 1989
 *
 *****************************************************************************/

/*  signs.c
 *
 *  This file contains all the routines for the creation and manipulation of 
 *  symbols used in xdbx.  There are 3 different signs:
 *    arrow  - a solid right arrow to indicate the current execution point.
 *    updown - an outlined right arrow to indicate position in stack trace.
 *    stop   - a stop hand symbol to indicate a breakpoint is set.
 *    bomb  - a bomb symbol to indicate the point of segmentation fault.
 *
 *  To display a sign on a given line in the source window, it is first
 *  created and mapped.  To undisplay it, the sign is unmapped.  It can
 *  be mapped again when the sign is needed.  Note that the sign is never
 *  moved, so that there can be as many signs created (but not mapped) as
 *  the number of lines in the source window.
 *  For arrow and updown, there can be at most one of each mapped at a time.
 *  For stop, there can be more than one mapped at the same time.
 *
 *
 *  Changes by Stefan Haenssgen:
 *
 *  07-19-91    Modified to use SWindow instead of sourceWindow etc
 *  07-30-91	Modified to use SWindow instead of arrow, bomb, updown,
 *		 stops, nstops
 *  07-31-91    Use a_swindow for globally active one and pass
 *               swindow as parameter for calls that affect just one
 *               certain window (such as action procedures etc)
 *		Modified signs_init, CreateSign, DisplayStop, UpdateStops,
 *		 RemoveStops, ClearStops, UpdateArrow, UpdateUpdown,
 *		 UpdateBomb  accordingly (global / parameter)
 *		Get swindow-info from file parameter where necessary
 *  08-07-91	Made stops, nstops, arrow, updown, bomb  global again
 *		Changed lines, topline, etc from file->* to swindow->*
 *		Changed Update*  to use file->sws[] and cycle through
 *		 all affected SWindows
 *		Included code from DisplayStop() in UpdateStops()
 *		 to avoid double loop through sws[] array
 *		Changed swindow->stops[] etc to stops again
 *  08-09-91	Moved widgets (Arrow-,Updown-,Stop-,Bombsign) into the
 *		 SWindow struct
 *  08-16-91	Optimized and debugged display part of UpdateStops() 
 *		Modified RemoveStop() to affect all SWindows that are
 *		 displaying the same file
 *		Added check for NULL file to UpdateBomb()
 *
 *  17-nov-92	Introduced multiple PC arrows of different sizes to
 *		 visualize different threads of execution (with different
 *		 numbers of processes in them):
 *		Changed arrow to arrows[], added narrows
 *		Modified UpdateArrows() [formerly UpdateArrow()] to examine
 *		 all existing arrows
 *		Modified UpdateUpdown() to use arrows[0] as default (UGLY)
 *		Use new "signbitmaps.h" instead of "bitmaps.h"
 *		Removed swindow->arrow_i, loop through all arrowsigns[]
 *		 to display stuff in UpdateArrows()
 *  28-nov-92	Modified UpdateBomb() to also update a bomb in the
 *		 corresponding Modula source [there is just one bomb
 *		 at a time, so no reason to proceed like with the stops
 *		 or the arrows]
 *		Added field "mline" to all signs, for the corresponding
 *		 Modula line
 *		Also create signs in Modula window (if any) in DisplayXxx
 *		Changed all routines to take corr. Modula windows in
 *		 account
 *  29-nov-92	Added debug infos (which windows/files when accessed etc)
 *		 to enhance performance later
 *  11-feb-93	Also update signs when called with modula-window!
 *		 (i.e. in UpdateArrows & UpdateStops, also work when
 *		 swindow is Modula-Window, not just swindow=C and mwindow=M)
 *  11-mar-93	Even better updating of Modula-source (e.g. RemoveStop())
 *
 */

#include "global.h"
#include "signbitmaps.h"


Arrow 		arrows[MAXARROWS];	/* Array of PC arrows	*/
Cardinal	narrows;		/* Number of arrows	*/
Updown 		updown;
Bomb 		bomb;
Stops		stops[MAXSTOPS];	/* array of stops */
Cardinal	nstops;			/* number of stops */


/* Initialize data structures */
/* (not the widgets, they get initialized in CreateSWindow() ! */
void signs_init()
{
    narrows = 1;
    arrows[0].line = 0;
    arrows[0].mline = 0;
    arrows[0].file = NULL;
    strcpy(arrows[0].func, "");		/* Init 1st in array of arrows	*/
    updown.line = 0;
    updown.mline = 0;
    strcpy(updown.file, "");
    nstops = 0;
    bomb.line = 0;
    bomb.mline = 0;
    strcpy(bomb.file, "");
}


/*  Create an arrow symbol, updown symbol or stop symbol:
 *    calculate the position of the symbol based on i, the number of lines
 *    from the top line.
 *    create the pixmap of the symbol
 *    display the symbol as a bitmap in a label widget.
 *  the "size" parameter is for different PC arrows
 */
/* get affected swindow as parameter */
static Widget CreateSign(parent, sign, i, swindow, size)
    Widget	parent;
    char	*sign;
    Cardinal 	i;
    SWindow	*swindow;
    int		size;
{
    TextWidget 	ctx = (TextWidget) swindow->sourceWindow;
    Arg 	args[15];
    Cardinal 	n;
    Dimension 	source_height, height, width; 
    char	*bits;
    Pixel       fg, bg;
    int 	horizDistance, vertDistance, height_per_line;
    int         screen;
    Dimension	vbar_width = 0;
    Dimension	border_width = 0;

    if (swindow->displayedFile == NULL) return NULL;

    /* Get height and background pixel values of parent window */
    n = 0;
    XtSetArg(args[n], XtNheight, &source_height);			n++;
    XtSetArg(args[n], XtNbackground, &bg);				n++;
    XtGetValues(parent, args, n);

    height_per_line = source_height/swindow->lines;
    vertDistance = OFFSET + (i * height_per_line); 

    screen = DefaultScreen(display);

    if (sign && !strcmp(sign, "arrow")) {
	bits = arrow_bits[size];
	width = arrow_width[size];
	height = arrow_height[size];
	horizDistance = 0;
	fg = app_resources.arrow_color;
    }
    else if (sign && !strcmp(sign, "updown")) {
	bits = updown_bits;
	width = updown_width;
	height = updown_height;
	horizDistance = 0;
	fg = app_resources.updown_color;
    }
    else if (sign && !strcmp(sign, "stop")) {
	bits = stop_bits;
	width = stop_width;
	height = stop_height;
	horizDistance = arrow_width[0];
	fg = app_resources.stop_color;
    }
    else if (sign && !strcmp(sign, "bomb")) {
	bits = bomb_bits;
	width = bomb_width;
	height = bomb_height;
	horizDistance = 0;
	fg = app_resources.bomb_color;
    };

    if( ctx->text.vbar != NULL )
    {
	    n = 0;
	    XtSetArg(args[n], XtNwidth, &vbar_width); 			n++;
	    XtSetArg(args[n], XtNborderWidth, &border_width);		n++;
	    XtGetValues(ctx->text.vbar, args, n);
	    vbar_width += (border_width * 2);
    }
    
    n = 0;
    XtSetArg(args[n], XtNborderWidth, 0);				n++;
    XtSetArg(args[n], XtNwidth, (XtArgVal) width);			n++;
    XtSetArg(args[n], XtNheight, (XtArgVal) height);			n++;
    XtSetArg(args[n], XtNresize, (XtArgVal) False);			n++;
    XtSetArg(args[n], XtNmappedWhenManaged, (XtArgVal) False);		n++;
    XtSetArg(args[n], XtNbitmap, XCreatePixmapFromBitmapData (
        display, DefaultRootWindow(display), bits, width, height,
        fg, bg, DefaultDepth(display, screen)));			n++;

    XtSetArg(args[n], XtNfromVert, (XtArgVal) NULL);			n++;
    XtSetArg(args[n], XtNfromHoriz, (XtArgVal) NULL);			n++;
    XtSetArg(args[n], XtNhorizDistance, (XtArgVal) horizDistance+vbar_width);
									n++;
    XtSetArg(args[n], XtNvertDistance, (XtArgVal) vertDistance);	n++;
    XtSetArg(args[n], XtNtop, (XtArgVal) XawChainTop);			n++;
    XtSetArg(args[n], XtNleft, (XtArgVal) XawChainLeft);		n++;
    XtSetArg(args[n], XtNbottom, (XtArgVal) XawChainTop);		n++;
    XtSetArg(args[n], XtNright, (XtArgVal) XawChainLeft);		n++;

    return XtCreateManagedWidget(sign, labelWidgetClass, parent, args, n);
}


/*
 *  Given a line number, displays a stop sign if that line is viewable.
 *  If the stop widget for that line does not exist, create one and map it.
 *  If the stop widget exists but not mapped, map it.
 *  Also display the sign in the corresp. Modula file, if any
 */
/* get affected swindow from file parameter */
void DisplayStop(file, line)
FileRec *file;
int	line;
{
    Cardinal n,i;


    /* Go through all windows where this file is displayed */

  for (n=0; n<file->nsws; n++) {

    SWindow *swindow = file->sws[n];	/* A window where file is displayed */


    if (line >= swindow->topline && line <= swindow->bottomline) {
	i = line - swindow->topline;
	if (swindow->stopsign[i].w == NULL) {	/* widget does not exist */
	    swindow->stopsign[i].w = CreateSign(swindow->sourceForm, "stop",
						i,swindow);
	    XtMapWidget(swindow->stopsign[i].w);
	    swindow->stopsign[i].mapped = 1;
	}
	else if (!swindow->stopsign[i].mapped) { /* widget not mapped */
	    XtMapWidget(swindow->stopsign[i].w);
	    swindow->stopsign[i].mapped = 1;
	}
    }
  }

    /* Also display stop in corr. Modula file */
    
    if ((file->FileType == SWTYPE_C) && (file->OtherFile) &&
	(file->mlines) && (file->lastline>=line)) {
	DisplayStop(file->OtherFile, file->mlines[line]->Line);
    };
	
}


/*
 *  Unmap all stop signs and then display only those stops that are viewable.
 */
/* get affected swindow from the file parameter */
void UpdateStops(file)
FileRec *file;
{
    Cardinal n,i;
    int	 line;

    if (file == NULL) return;


  for (n=0; n<file->nsws; n++) {

    SWindow *swindow = file->sws[n];	/* A window where file is displayed */
    SWindow *mwindow = NULL;

    for (i=0; i<swindow->lines; i++)	/* Unmap all old ones		*/
	if (swindow->stopsign[i].w && swindow->stopsign[i].mapped) {
	    XtUnmapWidget(swindow->stopsign[i].w);
	    swindow->stopsign[i].mapped = 0;
	}
    
    /* Also unmap stops in corr. modula window */
    
    if ((swindow->SWType == SWTYPE_C) && (mwindow = swindow->OtherSW)) {
      for (i=0; i<mwindow->lines; i++)	/* Unmap all old ones		*/
	if (mwindow->stopsign[i].w && mwindow->stopsign[i].mapped) {
	    XtUnmapWidget(mwindow->stopsign[i].w);
	    mwindow->stopsign[i].mapped = 0;
	}
    };
    
    
    /* Look for stops for this file and display them */
    
    for (i=1; i<=nstops; i++)
	if ( ( (file->FileType == SWTYPE_C) &&		/* C and name matches*/
	       (stops[i].file) &&
	       (!strcmp(stops[i].file,file->pathname)) &&
	       (line=stops[i].line)
	     ) || (					/* Or:		     */
	       (file->FileType == SWTYPE_MODULA) &&	/* M and corr.name ok*/
/*	       (stops[i].mfile) &&*/
/*	       (!strcmp(stops[i].mfile,file->pathname)) &&*/
	       (line=stops[i].mline)
	     )
	    ) {

	    /* Visible in C source? */
	    /*!!!!!!! can also be the modula-source !!! */
	    
	    if (line >= swindow->topline &&
		line <= swindow->bottomline) {

/* Slightly modified code from DisplayStop() inserted here!! */
/* (to avoid double loop that would occur if DisplayStop() was  */
/*  used, as DisplayStop cycles through all sws[] itself, too!) */

		int j;

		j = line - swindow->topline;
		if (swindow->stopsign[j].w == NULL) { /* widget doesn't exist*/
		    swindow->stopsign[j].w = CreateSign(swindow->sourceForm,
							"stop",j,swindow,0);
		    XtMapWidget(swindow->stopsign[j].w);
		    swindow->stopsign[j].mapped = 1;
		}
		else if (!swindow->stopsign[j].mapped) {/* widget not mapped */
		    XtMapWidget(swindow->stopsign[j].w);
		    swindow->stopsign[j].mapped = 1;
		}
	    }

	    /* Visible in Modula source? [find line pos. in modula in stop!] */

	    if (mwindow && (line = stops[i].mline) &&
		(line >= mwindow->topline &&
		line <= mwindow->bottomline)) {

		int j;

		j = line - mwindow->topline;
		if (mwindow->stopsign[j].w == NULL) { /* widget doesn't exist*/
		    mwindow->stopsign[j].w = CreateSign(mwindow->sourceForm,
							"stop",j,mwindow,0);
		    XtMapWidget(mwindow->stopsign[j].w);
		    mwindow->stopsign[j].mapped = 1;
		}
		else if (!mwindow->stopsign[j].mapped) {/* widget not mapped */
		    XtMapWidget(mwindow->stopsign[j].w);
		    mwindow->stopsign[j].mapped = 1;
		}
	    }

/* end of DisplayStop() [duplicated for modula] */

	}
  }
	
}

/*
 * Given a line number, unmap the stop sign associated with that line.
 */
/* get affected swindow as parameter */
/* Affect all SWindows where the file displayed in THIS SWindow is shown, too*/
void RemoveStop(line, mline, swindow)
int line;
int mline;
SWindow *swindow;
{
    Cardinal i,n;
    FileRec *file = swindow->displayedFile;

    if (file==NULL)
      return;
    
  for (n=0; n<file->nsws; n++) {

    SWindow *sw = file->sws[n];    /* A window where file is displayed */
    SWindow *mwindow = NULL;

    if (sw->displayedFile && line >= sw->topline && line <= sw->bottomline) {
	i = line - sw->topline;
	if (sw->stopsign[i].w && sw->stopsign[i].mapped) {
	    XtUnmapWidget(sw->stopsign[i].w);
	    sw->stopsign[i].mapped = 0;
	}
    }
    
    /* Also remove corr. Modula */
    
    if ((swindow->SWType == SWTYPE_C) && (mwindow = swindow->OtherSW) &&
	(file->OtherFile) && mline &&
	(mline >= mwindow->topline) &&
	(mline <= mwindow->bottomline)) {

	i = mline - mwindow->topline;
	if (mwindow->stopsign[i].w && mwindow->stopsign[i].mapped) {
	    XtUnmapWidget(mwindow->stopsign[i].w);
	    mwindow->stopsign[i].mapped = 0;
	}
    }
  }
}

void ClearStops()
/* affects all stop signs */
{
    int i;

    for (i=1; i<=nstops; i++) {
	stops[i].file  = NULL;
	stops[i].line  = 0;
	stops[i].mline = 0;
    }
}


/*
 *  Given a line number, displays an arrow if that line is viewable.
 *  If the arrow widget for that line does not exist, create one and map it.
 *  If the arrow widget exists but not mapped, map it.
 */
/* get affected swindow from file parameter */
void DisplayArrow(file, line)
FileRec *file;
int	line;
{
    Cardinal n,i;


    /* Go through all windows where this file is displayed */

  for (n=0; n<file->nsws; n++) {

    SWindow *swindow = file->sws[n];	/* A window where file is displayed */

    if (line >= swindow->topline && line <= swindow->bottomline) {
	i = line - swindow->topline;
	if (swindow->arrowsign[i].w == NULL) {	/* widget does not exist */
	    swindow->arrowsign[i].w = CreateSign(swindow->sourceForm, "arrow",
						i,swindow,0);/* DEFAULT */
	    XtMapWidget(swindow->arrowsign[i].w);
	    swindow->arrowsign[i].mapped = 1;
	}
	else if (!swindow->arrowsign[i].mapped) { /* widget not mapped */
	    XtMapWidget(swindow->arrowsign[i].w);
	    swindow->arrowsign[i].mapped = 1;
	}
    }
  }

    /* Also display arrow in corr. Modula file */
    
    if ((file->FileType == SWTYPE_C) && (file->OtherFile) &&
	(file->mlines)) {
	DisplayArrow(file->OtherFile, file->mlines[line]->Line);
    };
	
    
}


/*  Unmap the current arrow signs.
 *  Display new arrow signs if they are viewable.
 *  Do this for all arrows in this file.
 */
void UpdateArrows(file)
FileRec *file;
{
    Cardinal n,i,j;
    int	     line;

    if (file == NULL) return;


    /* Go through all windows where this file is displayed */

  for (n=0; n<file->nsws; n++) {
    SWindow *swindow = file->sws[n];	/* A window where file is displayed */
    SWindow *mwindow = NULL;

    for (i=0; i<swindow->lines; i++)
	if (swindow->arrowsign[i].w && swindow->arrowsign[i].mapped) {
	    XtUnmapWidget(swindow->arrowsign[i].w);
	    swindow->arrowsign[i].mapped = 0;
	}
    
    /* Also unmap arrows in corr. modula window */

    mwindow = swindow->OtherSW;
    if (mwindow->SWType == SWTYPE_C)	/* Ignore other window if C     */
      mwindow = NULL;			/*  (i.e. swindow is already M) */
    
    if ((swindow->SWType == SWTYPE_C) && (mwindow)) {
      for (i=0; i<mwindow->lines; i++)	/* Unmap all old ones		*/
	if (mwindow->arrowsign[i].w && mwindow->arrowsign[i].mapped) {
	    XtUnmapWidget(mwindow->arrowsign[i].w);
	    mwindow->arrowsign[i].mapped = 0;
	}
    }
    
    /* Look for arrows for this file and display them */
    
    for (j=0; j<narrows; j++) {

	if(((swindow->SWType == SWTYPE_C) &&
	    (arrows[j].file) &&
	    !strcmp(arrows[j].file, file->pathname) &&
	    (line = arrows[j].line)
           ) || (
	     (swindow->SWType == SWTYPE_MODULA) &&
	     (file->OtherFile) &&
	     (arrows[j].file) &&
	     !strcmp(arrows[j].file,file->OtherFile->pathname) &&
	     (line=arrows[j].mline)
           ) ) {


/*fprintf(stderr,"UpdateArrows()... line %d, top %d, bot %d (mline %d)\n",*/
/*line,swindow->topline,swindow->bottomline,arrows[j].mline);*/
	    /* Visible in C source? */
	    
	    if (line >= swindow->topline &&
		line <= swindow->bottomline) {

		i = line - swindow->topline;
		if (swindow->arrowsign[i].w == NULL) {
		    swindow->arrowsign[i].w = CreateSign(swindow->sourceForm,
				     "arrow", i,swindow,0); /* DEFAULT SIZE */
		    XtMapWidget(swindow->arrowsign[i].w);
		    swindow->arrowsign[i].mapped = TRUE;
		}
		else if (!swindow->arrowsign[i].mapped) {
		    XtMapWidget(swindow->arrowsign[i].w);
		    swindow->arrowsign[i].mapped = TRUE;
		}
	    }

	    /* Visible in Modula source? [find line pos. in modula in arrow!]*/

            line = arrows[j].mline;
	    if (mwindow && line &&
		(line >= mwindow->topline) &&
		(line <= mwindow->bottomline)) {
		    
		i = line - mwindow->topline;
		if (mwindow->arrowsign[i].w == NULL) {
		    mwindow->arrowsign[i].w = CreateSign(mwindow->sourceForm,
				     "arrow", i,mwindow,0); /* DEFAULT SIZE */
		    XtMapWidget(mwindow->arrowsign[i].w);
		    mwindow->arrowsign[i].mapped = TRUE;
		}
		else if (!mwindow->arrowsign[i].mapped) {
		    XtMapWidget(mwindow->arrowsign[i].w);
		    mwindow->arrowsign[i].mapped = TRUE;
		}
	    }
	}
    }
  }
}


/*
 * Given a line number, unmap the arrow sign associated with that line.
 */
/* get affected swindow as parameter */
/* Affect all SWindows where the file displayed in THIS SWindow is shown, too*/
void RemoveArrow(line, swindow)
int line;
SWindow *swindow;
{
    Cardinal i,n;
    FileRec *file = swindow->displayedFile;

  for (n=0; n<file->nsws; n++) {

    SWindow *sw = file->sws[n];    /* A window where file is displayed */
    SWindow *mwindow = NULL;

    if (sw->displayedFile && line >= sw->topline && line <= sw->bottomline) {
	i = line - sw->topline;
	if (sw->arrowsign[i].w && sw->arrowsign[i].mapped) {
	    XtUnmapWidget(sw->arrowsign[i].w);
	    sw->arrowsign[i].mapped = 0;
	}
    }

    /* Also remove corr. Modula */
    
    if ((swindow->SWType == SWTYPE_C) && (mwindow = swindow->OtherSW) &&
	(file->OtherFile) && line &&
	(line = CToModulaLine(mwindow,line)) >= mwindow->topline &&
	line <= mwindow->bottomline) {
	i = line - mwindow->topline;
	if (mwindow->arrowsign[i].w && mwindow->arrowsign[i].mapped) {
	    XtUnmapWidget(mwindow->arrowsign[i].w);
	    mwindow->arrowsign[i].mapped = 0;
	}
    }
  }
}

void ClearArrows()
/* affects all arrows signs */
{
    int i;

    for (i=0; i<narrows; i++) {
	arrows[i].file  = NULL;
	arrows[i].line  = 0;
	arrows[i].mline = 0;
    }
}



/*  If the new updown is on the same line as the arrow, remove the updown.
 *  Unmap current updown sign.
 *  Display the updown if it is viewable.
 */
/* get affected swindow from the file parameter */
void UpdateUpdown(file)
FileRec *file;
{
    Cardinal n,i;
    int	     line;

    if (file == NULL) return;


    /* Go through all windows where this file is displayed */

  for (n=0; n<file->nsws; n++) {

    SWindow *swindow = file->sws[n];	/* A window where file is displayed */

    if (updown.file && arrows[0].file && !strcmp(updown.file, arrows[0].file) && 
	!strcmp(updown.func, arrows[0].func)) {
	updown.line = 0;
	strcpy(updown.file, "");
    }
    i = swindow->updown_i;
    if (i>=0 && i<swindow->lines)
	if (swindow->updownsign[i].w && swindow->updownsign[i].mapped) {
	    XtUnmapWidget(swindow->updownsign[i].w);
	    swindow->updownsign[i].mapped = 0;
	}
    line = updown.line;
    if (updown.file &&
        !strcmp(updown.file, file->pathname) &&
    	line >= swindow->topline && line <= swindow->bottomline) {
        i = line - swindow->topline;
	swindow->updown_i = i;
	if (swindow->updownsign[i].w == NULL) {
	    swindow->updownsign[i].w = CreateSign(swindow->sourceForm,
                                         "updown", i,swindow);
	    XtMapWidget(swindow->updownsign[i].w);
	    swindow->updownsign[i].mapped = TRUE;
	}
	else if (!swindow->updownsign[i].mapped) {
	    XtMapWidget(swindow->updownsign[i].w);
	    swindow->updownsign[i].mapped = TRUE;
	}
    }
  }
}


/*
 * Display a bomb sign (if visible), also update corresp. Modula source
 *
 */
void DoUpdateBomb(file,swindow)
FileRec *file;
SWindow *swindow;
{
    SWindow *mwindow=NULL;
    FileRec *mfile=NULL;
    int      line;
    int	     mline=0;

    if ((file == NULL) || (swindow == NULL))	/* Catch bad input	*/
      return;

    if (swindow->SWType == SWTYPE_C)		/* Find corresp. M-Src	*/
      mwindow = swindow->OtherSW;
    else
      mwindow = NULL;

    line = swindow->bomb_i;
    
    if (line>=0 && line<swindow->lines) {	/* Erase old bomb	*/
	if (swindow->bombsign[line].w &&
	     swindow->bombsign[line].mapped) {
	    XtUnmapWidget(swindow->bombsign[line].w);
	    swindow->bombsign[line].mapped = 0;
	};
    };
    if (mwindow && (mline = mwindow->bomb_i) &&
	mline>=0 && mline<mwindow->lines) {
	if (mwindow->bombsign[mline].w &&	/* Also in M-Src*/
	     mwindow->bombsign[mline].mapped) {
	    XtUnmapWidget(mwindow->bombsign[mline].w);
	    mwindow->bombsign[mline].mapped = 0;
	};
    };

    /* Draw Bomb in C source (if visible) */
    
    line = bomb.line;
    if (line==0)
      return;
    
    if (bomb.file && !strcmp(bomb.file, file->pathname) &&
    	line >= swindow->topline && line <= swindow->bottomline) {
	
	int sline = line - swindow->topline;	/* Line inside window	*/
	
	swindow->bomb_i = sline;
	if (swindow->bombsign[sline].w == NULL) {
	    swindow->bombsign[sline].w = CreateSign(swindow->sourceForm,
						   "bomb", sline,swindow);
	    XtMapWidget(swindow->bombsign[sline].w);
	    swindow->bombsign[sline].mapped = TRUE;
	}
	else if (!swindow->bombsign[sline].mapped) {
	    XtMapWidget(swindow->bombsign[sline].w);
	    swindow->bombsign[sline].mapped = TRUE;
	}

    }

    /* Also display in corresponding M-Window, if any */
    
    if ((mwindow==NULL) || (mwindow->displayedFile == NULL) || (mline==0))
      return;

    /*!!!!!!! THIS ASSUMES that the modula file indeed corresponds to	*/
    /*!!!!!!! the c file (which it should, because it's the OtherSW!)	*/
    
    if (mline >= mwindow->topline && mline <= mwindow->bottomline) {

	int sline = mline - swindow->topline;	/* Line inside window	*/

	mwindow->bomb_i = sline;
	if (mwindow->bombsign[sline].w == NULL) {
	    mwindow->bombsign[sline].w = CreateSign(mwindow->sourceForm,
						   "bomb", sline,mwindow);
	    XtMapWidget(mwindow->bombsign[sline].w);
	    mwindow->bombsign[sline].mapped = TRUE;
	}
	else if (!mwindow->bombsign[sline].mapped) {
	    XtMapWidget(mwindow->bombsign[sline].w);
	    mwindow->bombsign[sline].mapped = TRUE;
	}

    }

}

/*  Unmap the current bomb sign, if any.
 *  Display a new bomb sign.
 */
/* get affected swindow from the file parameter */
void UpdateBomb(file)
FileRec *file;
{
    Cardinal n;


    if (file == NULL) return;


    /* Go through all windows where this file is displayed */

    for (n=0; n<file->nsws; n++) {
	DoUpdateBomb(file,file->sws[n]);
    }
}

