/*****************************************************************************
 *
 *  xdbx - X Window System interface to the dbx debugger
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation, and that the name of The University of Texas
 *  and Microelectronics and Computer Technology Corporation (MCC) not be 
 *  used in advertising or publicity pertaining to distribution of
 *  the software without specific, written prior permission.  The
 *  University of Texas and MCC makes no representations about the 
 *  suitability of this software for any purpose.  It is provided "as is" 
 *  without express or implied warranty.
 *
 *  THE UNIVERSITY OF TEXAS AND MCC DISCLAIMS ALL WARRANTIES WITH REGARD TO
 *  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TEXAS OR MCC BE LIABLE FOR
 *  ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 *  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 *  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  Author:  	Po Cheung
 *  Created:   	March 10, 1989
 *
 *****************************************************************************/

/*  xdbx.c
 *
 *    Contain main program and initialization, command line options handling,
 *    and resource database management.
 *
 *    Syntax():		Print an error message if xdbx is invoked with an
 *			incorrect number of arguments.
 *    main_init():	Initialization routine.
 *    dbxoptions():	Construct command line arguments for dbx.
 *    main():		Main program.
 *
 *
 *  Changes by Stefan Haenssgen:
 *
 *  07-30-91	Removed some translations for SourceWindow 
 *		 which are now defined in CreateSourceWindow() in source.c
 *  		Modified to use SWindow instead of arrow, bomb, updown,
 *		 stops, nstops
 *  08-01-91	Call ActivateSWindow to activate first (default) SWindow
 *  08-02-91    Init a_swindow and swindows[] first
 *		Use signs_init for a_swindow
 *  08-09-91	Added "[sth]" to title of window (to make it clear that
 *		 it's no standard xdbx any more ;-)
 *		Call signs_init no more with a_swindow as parameter
 *		 (signs-widgets are in SWindow now)
 *
 *  07-aug-92	Added DebugFn & ProfileTable initialization
 *  02-sep-92	Added treeChecksum initialization
 *  17-nov-92	Use new "iconbitmaps.h" instead of "bitmaps.h"
 *  26-nov-92	Create second Window (for C), use 1st for Modula,
 *		 connect the two (via OtherSW)
 *  04-dec-92	Renamed to "msdb"
 *  31-aug-93	Cleanup, Version 1.1
 *
 */

#include "global.h"
#include "iconbitmaps.h"
#include "patchlevel.h"
#include "DebugFn/DebugFn.h"

#define VERSION	"1.1"

#define Offset(field) (XtOffset(XdbxResources *, field))

XtAppContext  	app_context; 		/* application context */
Widget  	toplevel; 		/* top level widget */
Display		*display;		/* connection to X server */
Cursor		watch;			/* XC_watch cursor */
XdbxResources 	app_resources;		/* application resources of xdbx */
char 		xdbxinit[LINESIZ];	/* initialization file name */
Boolean		Tstartup = False;	/* if True, remove xdbxinit */
Boolean		debug = False;		/* debug mode for xdbx */


static XtResource resources[] = {
    {"bell", "Bell", XtRBoolean, sizeof(Boolean), 
	Offset(bell), XtRImmediate, (caddr_t)False},
    {"displayWindow", "DisplayWindow", XtRBoolean, sizeof(Boolean), 
	Offset(displayWindow), XtRImmediate, (caddr_t)False},
    {"prompt", "Prompt", XtRString, sizeof(char *), 
	Offset(prompt), XtRImmediate, (caddr_t)NULL},
    {"delimiters", "Delimiters", XtRString, sizeof(char *), 
	Offset(delimiters), XtRImmediate, (caddr_t)NULL},
    {"stop_color", "StopColor", XtRPixel, sizeof(Pixel), 
	Offset(stop_color), XtRString, "Red"},
    {"arrow_color", "ArrowColor", XtRPixel, sizeof(Pixel), 
	Offset(arrow_color), XtRString, "Blue"},
    {"updown_color", "UpdownColor", XtRPixel, sizeof(Pixel), 
	Offset(updown_color), XtRString, "Blue"},
    {"bomb_color", "bombColor", XtRPixel, sizeof(Pixel), 
	Offset(bomb_color), XtRString, "Red"},
    {"dataDpyMaxHeight", "DataDpyMaxHeight", XtRDimension, sizeof(Dimension), 
	Offset(dataDpyMaxHeight), XtRString, "300"},
    {"dataDpyMaxWidth", "DataDpyMaxWidth", XtRDimension, sizeof(Dimension), 
	Offset(dataDpyMaxWidth), XtRString, "600"},
    {"bigicon", "Msdboptions", XtRBoolean, sizeof(Boolean), 
	Offset(bigicon), XtRImmediate, (caddr_t)False},
    {"debug", "Msdboptions", XtRBoolean, sizeof(Boolean), 
	Offset(debug), XtRImmediate, (caddr_t)False},
    {"dbxopt_r", "Dbxoptions", XtRBoolean, sizeof(Boolean), 
	Offset(dbxopt_r), XtRImmediate, (caddr_t)False},
    {"dbxopt_i", "Dbxoptions", XtRBoolean, sizeof(Boolean), 
	Offset(dbxopt_i), XtRImmediate, (caddr_t)False},
    {"includeDir", "Dbxoptions", XtRString, sizeof(char *), 
	Offset(includeDir), XtRImmediate, (caddr_t)NULL},
    {"dbxopt_k", "Dbxoptions", XtRBoolean, sizeof(Boolean), 
	Offset(dbxopt_k), XtRImmediate, (caddr_t)False},
    {"cfile", "Dbxoptions", XtRString, sizeof(char *), 
	Offset(cfile), XtRImmediate, (caddr_t)NULL},
    {"dbxopt_kbd", "Dbxoptions", XtRBoolean, sizeof(Boolean), 
	Offset(dbxopt_kbd), XtRImmediate, (caddr_t)False},
    {"fcount", "Dbxoptions", XtRString, sizeof(char *), 
	Offset(fcount), XtRImmediate, (caddr_t)NULL},
    {"startup", "Dbxoptions", XtRString, sizeof(char *), 
	Offset(startup), XtRImmediate, (caddr_t)NULL},
    {"tstartup", "Dbxoptions", XtRString, sizeof(char *), 
	Offset(tstartup), XtRImmediate, (caddr_t)NULL},
};


String fallback_resources[] = {
    "*allowShellResize:                 True",
    "*borderWidth:			1",
    "*font:                             fixed",
    "*vpane.width:                      550",
    "*fileWindow*font:     		variable",
    "*fileLabel.width:     		500",
    "*lineLabel.width:     		50",
    "*sourceForm.preferredPaneSize:     320",
    "*sourceWindow.leftMargin:          35",
    "*sourceWindow.scrollHorizontal:	whenNeeded",
    "*sourceWindow.translations:	#override \\n\
        <Btn1Down>:             SelectStart() SelectWord() \\n",
/* Update() must have a valid address !!! [sth] 		*/
/* so define it later in CreateSourceWindow() in source.c !	*/
/*        \
        Shift<Btn1Up>:          Update() SelectEnd() PrintSelection() \\n\
        <Btn1Up>:               Update() SelectEnd() \\n",*/
    "*messageWindow*font:  		variable",
    "*messageWindow.min:  		30",
    "*messageWindow.max:  		30",
    "*dialogWindow.preferredPaneSize:	200",
    "*dialogWindow.resizeToPreferred:	True",
    "*dialogWindow.translations:	#override \\n\
        <Btn1Down>:     SelectStart() SelectWord() \\n\
        Shift<Btn1Up>:  SelectEnd() PrintSelection() \\n\
        <Btn1Up>:       SelectEnd() \\n",
    "*commandWindow.preferredPaneSize:  106",
    "*commandWindow.skipAdjust:		True",
    "*commandWindow.hSpace:		14",
    "*commandWindow.vSpace:		10",
    "*Command.height:                   20",
    "*Command.width:                    60",
    "*List.columnSpacing:               10",
    "*displayWindow.preferredPaneSize:  50",
    "*displayWindow.skipAdjust:         True",
    "*displayWindow.scrollVertical:	whenNeeded",
    "*displayWindow.scrollHorizontal:	whenNeeded",
    "*displayWindow.translations:	#override \\n\
        <Btn1Down>:             SelectStart() SelectWord() \\n\
        Shift<Btn1Up>:          SelectEnd() PrintSelection() \\n\
        <Btn1Up>:               SelectEnd() \\n",
    "*popup*showGrip:  			False",
    NULL,
};

static XrmOptionDescRec options[] = {
    {"-bigicon","bigicon",	XrmoptionNoArg, "True"},
    {"-debug",	"debug",	XrmoptionNoArg, "True"},
    {"-r",	"dbxopt_r",	XrmoptionNoArg, "True"},
    {"-i",	"dbxopt_i",	XrmoptionNoArg, "True"},
    {"-I",	"includeDir",	XrmoptionSepArg, NULL},
    {"-k",	"dbxopt_k",	XrmoptionNoArg, "True"},
#ifdef BSD   /* Berkeley dbx */
    {"-c",	"cfile",	XrmoptionSepArg, NULL},
#else	     /* Sun dbx */
    {"-kbd",	"dbxopt_kbd",	XrmoptionNoArg, "True"},
    {"-f",	"fcount",	XrmoptionSepArg, NULL},
    {"-s",	"startup",	XrmoptionSepArg, NULL},
    {"-sr",	"tstartup",	XrmoptionSepArg, NULL},
#endif
#ifdef MIPS  /* Mips dbx */
    {"-pixie",  "pixie",	XrmoptionNoArg, "True"},
#endif
};

XtActionsRec xdbx_actions[] = {
    {"SelectStart",	(XtActionProc) SelectStart},
    {"SelectEnd",	(XtActionProc) SelectEnd},
    {"SelectWord",	(XtActionProc) SelectWord},
    {"PrintSelection",	(XtActionProc) PrintSelection},
    {"Update",		(XtActionProc) Update},
    {"DeleteWord",	(XtActionProc) DeleteWord},
    {"DeleteLine",	(XtActionProc) DeleteLine},
    {NULL, NULL}
};

static void Syntax(call)
char *call;
{
    fprintf(stderr,
	    "Usage: %s [-toolkitoptions] [-dbxoptions] [objfile [corefile]]\n",
	    call);
    exit(1);
}

/*  Set window manager hints to indicate display accepts input.
 *  Initialize routines in source.c, signs.c and parser.c.
 *  Disable window resize of fileWindow.
 *  Get the name of the dbx command initialization file.
 */
static void main_init()
{
    XWMHints	wmhints;
    char	title[100];

    display = XtDisplay(toplevel);
    watch = XCreateFontCursor(display, XC_watch);

    sprintf(title, "MSDB %sp%d", VERSION, PATCHLEVEL);
    XStoreName(display, XtWindow(toplevel), title);
    XSetIconName(display, XtWindow(toplevel), "msdb");
    wmhints.input = True;
    if (app_resources.bigicon)
	wmhints.icon_pixmap= XCreateBitmapFromData(display, XtWindow(toplevel),
	    xdbx64_bits, xdbx64_width, xdbx64_height);
    else
	wmhints.icon_pixmap= XCreateBitmapFromData(display, XtWindow(toplevel),
	    xdbx48_bits, xdbx48_width, xdbx48_height);
    wmhints.flags = IconPixmapHint | InputHint;
    XSetWMHints(display, XtWindow(toplevel), &wmhints);

    if (!app_resources.delimiters || 
	strcmp(app_resources.delimiters, "") == NULL)
	app_resources.delimiters = XtNewString(DELIMITERS);
    if (app_resources.prompt && strcmp(app_resources.prompt, "") != NULL)
	xdbxprompt = app_resources.prompt;
    else
	xdbxprompt = XtNewString(XDBXPROMPT);
    debug = app_resources.debug;
    

    strcpy(xdbxinit, ".dbxinit");
    if (access(xdbxinit, R_OK) == -1) {
    	sprintf(xdbxinit, "%s/%s", (char *) getenv("HOME"), ".dbxinit");
    	if (access(xdbxinit, R_OK) == -1) {
	    strcpy(xdbxinit, "");
	}
    }
    DebugInitialize();	/* Init Debug Functions & Profile Table */
    treeChecksum = -2;	/* Init the checksum to "different"	*/
    source_init();
    signs_init();	/* init all sign positions (not the widgets) */
    parser_init();

}


/*  Reconstruct command line arguments for calling dbx.
 *  Return the argument list for dbx and new value of argc.
 */
static char **dbxoptions(argc, argv, app_resources)
    int  *argc;
    char **argv;
    XdbxResources *app_resources;
{
    char **dbxargv;
    char *temp = "msdb.XXXXXX";
    int  i=0;

    dbxargv = (char **) XtMalloc (MAXARGS * sizeof(char *));
    for (i=0; i < *argc; i++)
	dbxargv[i] = argv[i];

    if (app_resources->dbxopt_r)
	dbxargv[i++] = "-r";
    if (app_resources->dbxopt_i)
	dbxargv[i++] = "-i";
    if (app_resources->includeDir) {
	dbxargv[i++] = "-I";
	dbxargv[i++] = app_resources->includeDir;
    }
    if (app_resources->dbxopt_k)
	dbxargv[i++] = "-k";
    if (app_resources->cfile) {
	dbxargv[i++] = "-c";
	dbxargv[i++] = app_resources->cfile;
    }
    if (app_resources->dbxopt_kbd)
	dbxargv[i++] = "-kbd";
    if (app_resources->fcount) {
	dbxargv[i++] = "-f";
	dbxargv[i++] = app_resources->fcount;
    }
    /*  If .dbxinit exists in the local or home directory, include the option
     *  -c (Berkeley dbx) or -s (Sun dbx) and a dummy filename as the option 
     *	argument.  This will prevent dbx from reading the user's command
     *	initialization file.  Xdbx will read each line and pass it to dbx
     *  instead.
     */
    if (strcmp(xdbxinit, "")) {		/* .dbxinit or ~/.dbxinit exists */
#ifdef BSD
	dbxargv[i++] = "-c";
#else
	dbxargv[i++] = "-s";
#endif
	dbxargv[i++] = (char *) mktemp(temp);
    }
    if (app_resources->startup) {	/* overwrites dbxinit */
	Tstartup = False;
	strcpy(xdbxinit, app_resources->startup);
    }
    if (app_resources->tstartup) {	/* overwrites dbxinit */
	Tstartup = True;
	strcpy(xdbxinit, app_resources->tstartup);
    }
#ifdef MIPS
    if (app_resources->pixie) {		/* pixie output */
	dbxargv[i++] = "-pixie";
    }
#endif
    dbxargv[i] = NULL;
    *argc = i;
    return dbxargv;
}


void main(argc, argv)
int argc;
char **argv;
{
    char 	**dbxargv;
    SWindow	 *sw;

    trap_signals();

    toplevel=XtAppInitialize(&app_context, "MSdb", options, XtNumber(options),
			       &argc, argv, fallback_resources, NULL, 0);
    if (argc > 3) Syntax(argv[0]);
    XtGetApplicationResources(toplevel, &app_resources, resources,
                              XtNumber(resources), NULL, 0);
    XtAppAddActions(app_context, xdbx_actions, XtNumber(xdbx_actions));
    swindows_num = 0;			/* No SWindows in list yet */
    a_swindow = NULL;			/* No active SWindow yet */
    CreateSubWindows(toplevel);
    XtRealizeWidget(toplevel);
    DisableWindowResize(a_swindow->fileWindow);
    ActivateSWindow(a_swindow);		/* Activate 1st source code window */
    m_swindow = a_swindow;
    m_swindow->SWType = SWTYPE_MODULA;	/* (for Modula source) */
    
    PopupSWindow(True,False,-1,-1);	/* Create second SW (for C)	*/

    c_swindow = a_swindow;		/* C source = newly popped up	*/
    m_swindow->OtherSW = c_swindow;
    c_swindow->OtherSW = m_swindow;	/* Connect C & Modula source	*/
    a_swindow = m_swindow;

    main_init();
    dbxargv = dbxoptions(&argc, argv, &app_resources);
    calldbx(argc, dbxargv);

    XtAppMainLoop(app_context);
}
