/*****************************************************************************
 *
 *  xdbx - X Window System interface to the dbx debugger
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation, and that the name of The University of Texas
 *  and Microelectronics and Computer Technology Corporation (MCC) not be 
 *  used in advertising or publicity pertaining to distribution of
 *  the software without specific, written prior permission.  The
 *  University of Texas and MCC makes no representations about the 
 *  suitability of this software for any purpose.  It is provided "as is" 
 *  without express or implied warranty.
 *
 *  THE UNIVERSITY OF TEXAS AND MCC DISCLAIMS ALL WARRANTIES WITH REGARD TO
 *  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TEXAS OR MCC BE LIABLE FOR
 *  ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 *  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 *  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  Author:  	Po Cheung
 *  Created:   	March 10, 1989
 *
 *****************************************************************************/

/*  handler.c
 *
 *    Contain action handlers for the parser to invoke upon a dbx command.
 *
 *    TextSetTopPosition():	Set top character position of text displayed
 *    AdjustText():		Adjust the portion of text displayed.
 *    HighlightText():		Highlight part of the source & adjust it
 *    exec_handler():		Update file, line label, arrow position.
 *    done_handler():		Progrm execution completed, clear breakpoints
 *    stop_at_handler():	Place stop sign on line specified.
 *    stop_in_handler():	Place stop sign on function specified.
 *    updown_handler():		Update file, line label, updown arrow position.
 *    delete_handler():		Remove stop sign.
 *    func_handler():		Display function, if specified.
 *    file_handler():		Display file, if specified.
 *    debug_handler():		Check directory use list, display source file.
 *    cd_handler():		Record current working directory.
 *    use_handler():		Record directory paths.
 *    search_handler():		Adjust source file to display matched line.
 *    list_handler();		Adjust source file to display result.
 *    display_handler():	Display results in display window.
 *
 *
 *  Changes by Stefan Haenssgen:
 *
 *  07-19-91    Modified to use Swindow instead of sourceWindow etc
 *  07-30-91	Modified to use SWindow instead of arrow, bomb, updown,
 *		 stops, nstops
 *  07-31-91    Use a_swindow for globally active one and pass
 *               swindow as parameter for calls that affect just one
 *               certain window (such as action procedures etc)
 *		Modified exec_handler, done_handler,
 *		 stop_at_handler, stop_in_handler, updown_handler,
 *		 delete_handler, func_handler, debug_handler to
 *		 use a_swindow and give it as parameter where needed
 *		Modified AdjustText to take swindow as parameter
 *  08-07-91	Changed lines, topline, etc from file->* to swindow->*
 *		Changed swindow->stops[] etc to stops again
 *		Call UpdateLineLabel with swindow, not file->sw
 *  08-09-91	Modified AdjustText() to go through all SWindows that
 *		 are displaying the file
 *  08-16-91	Removed the modifications to AdjustText() - only the
 *		 active SWindow is to scroll
 *		Added (as a test) Update-calls to exec_handler and file_handler
 *		 to do screen updates (arrows etc) correctly
 *		Removed those again, because they had no effect
 *
 *  14-aug-92	Wrote HighlightText() to highlight parts of the source
 *		Modified exec_handler() to also update tree structure
 *		 and profile window (if any active) using
 *		 Update{Tree,Profile}Widget()
 *  19-aug-92	Use UpdateTree() instead of just updating the widget (also
 *		 generates new tree!)
 *  24-aug-92	Use UpdateProfile() 
 *		In exec_handler(), also load corresponding *.ms file for *.c
 *		 and create new SWindow for the Modula Source if needed
 *		Wrote MFileName() to get Modula file name from C name, check
 *		 if M-File exists
 *		Also modified updown_handler() according to exec_handler
 *		Enhanced HighlightText() to also use columns (not just lines)
 *  02-sep-92	Use treeChecksum to tell if structure tree has changed,
 *		 saves many redrawing efforts
 *  01-nov-92	Adapted to new DebugFnV calls & changes (Tree as ADT etc)
 *		Use new global DTree & PTable, deallocate old tree/profile
 *		 before building a new one
 *  17-nov-92	Use multiple arrows, use arrows[0] as default, UpdateArrows()
 *		Assign pointer to file instead of strcpy with arrows[].file
 *  19-nov-92	Rewrote exec_handler to build line mapping infos when
 *		 loading a corresponding modula source
 *		Modified highlighting code to use line mapping instead
 *		 of debug tree info (construct highlighting)
 *  20-nov-92	Clear all arrows (not just #0) in done_handler() &
 *		 debug_handler()
 *		Abandoned HighlightText(), introduced second arrow for
 *		 Modula-Source
 *		Handle diaplaying the 2nd arrow in the corresp. Modula line
 *  25-nov-92	Wrote UpdateActivations() which uptates the activation
 *		 values Visualizer (and pops it down if no longer in
 *		 the right context etc)
 *  26-nov-92	Enhanced UpdateActivations(), more checks for consistent
 *		 array values
 *  27-nov-92	Use ranges of activation variable
 *		Start in BEGIN_MODULE (not main)
 *		Moved MFileName() to source.c
 *		Changed a_swindow to c_swindow where necessary
 *		Also adjust corr. M-Window in AdjustText
 *  28-nov-92	Also update corresponding Modula line "mline" in all handlers
 *		Adjusted calls accordingly
 *  29-nov-92	Debugging, really show signs in Modula source as soon as in C
 *		Don't call complete AdjustText for M-src, split off
 *		 DoAdjustText() instead. UpdateXxx handle M-src themselves
 *		In exec_handler(), execute "next" if inside M-src "next"
 *		 command (i.e. if still same M-src line)
 *  01-dec-92	Wrote UpdateVisualizers() to update all currently active
 *		 Visualizers (and recognize popdowns etc)
 *		Call UpdateVisualizers() in corresp. handlers
 *  10-mar-93	Use "0" as active value in par.Forall Act.Visualizer
 *  11-mar-93	Pass mline to RemoveStop()
 *  12-mar-93	Check if inside DebugFn source, refuse to update tree if
 *		 yes (tree might be in inconsistent state, kabooom!)
 *  15-mar-93	Stop in DebugEnd() to preserve profiling information
 *  31-aug-93	Include "DebugFn/DebugFn*"
 *
 */

#include <ctype.h>
#include "DebugFn/DebugFnV.h"
#include "VisP.h"			/* Need some dirty tricks	*/
#include "Vis.h"
#include "global.h"

#ifdef BSD
#define	BRACKET	"[%d]"
#else
#define	BRACKET	"(%d)"
#endif

Boolean		Echo = True;		/* display dbx output if true */
static Boolean	Skip_func_handler = False;


/*  Display text starting from the top position specified by pos */

void TextSetTopPosition(w, pos)
    Widget w;
    XawTextPosition pos;
{
    Arg args[MAXARGS];
    Cardinal n;

    n = 0;
    XtSetArg(args[n], XtNdisplayPosition, (XtArgVal) pos);               n++;
    XtSetValues(w, args, n);
}


/*
 * void DoAdjustText(line,swindow)
 *
 * Does the text adjustment in itself (split from AdjustText)
 *
 */
void DoAdjustText(line,swindow)
int	 line;
SWindow	*swindow;
{
    FileRec 		*file;
    int	    		nlines = 0;
    int			i;
    XawTextPosition 	pos;

    
    if ((file = swindow->displayedFile) == NULL || line <= 0) return;

    swindow->currentline = line;

    if (line < swindow->topline || line > swindow->bottomline ) {
	/* Position line about 30% from the top */
	nlines = swindow->lines*0.3;
	if (line < nlines)			   /* near top */
	    swindow->topline = 1;
	else if (line > file->lastline - nlines)  /* near bottom */
	    swindow->topline = MAX(file->lastline - swindow->lines + 1, 1);
	else
	    swindow->topline = line - nlines;
	swindow->bottomline = MIN(swindow->topline + swindow->lines - 1,
				  file->lastline);
	TextSetTopPosition(swindow->sourceWindow,
                           file->linepos[swindow->topline]);
	swindow->topPosition = file->linepos[swindow->topline];
    }
    XawTextSetInsertionPoint(swindow->sourceWindow, file->linepos[line]);

    /* Text window might have scrolled, check topline & bottomline */
    pos = XawTextTopPosition(swindow->sourceWindow);
    for (i=1; pos >= file->linepos[i]; i++);
    if (swindow->topline != i-1) {
	swindow->topline = i-1;
	swindow->bottomline = MIN (swindow->topline + swindow->lines - 1,
				file->lastline);
    }
    UpdateLineLabel(line, swindow);

};


/*
 *  Adjust text so that 'line' will fall into the viewable part of the
 *  source window.
 *  Arrows, stop signs, and line label are updated accordingly.
 *
 *  Also affect corresponding Modula window!
 */
void AdjustText(line,swindow)
    int	   	line;
    SWindow	*swindow;
{
    FileRec 		*file;
    int	    		nlines = 0;
    int			i;
    XawTextPosition 	pos;

    if ((file = swindow->displayedFile) == NULL) return;
    
    DoAdjustText(line,swindow);	/* Adjust text region		*/
    
    if ((swindow->SWType == SWTYPE_C) && (swindow->OtherSW) &&
	(swindow->OtherSW->SWType == SWTYPE_MODULA)) {		/* Also Msrc */

	DoAdjustText(CToModulaLine(swindow,line),swindow->OtherSW);
    };

    UpdateStops(file);		/* These functions update the	*/
    UpdateArrows(file);		/*  corr. Modula window, too	*/
    UpdateUpdown(file);
    UpdateBomb(file);

	

}
    

/*
 *  UpdateTree()
 *
 *  Updates the currently shown tree structure widget by getting (by dbx)
 *  the debug info tree as string, re-converting it to a new tree and
 *  updating the shown tree widget.
 *  Note: treeW is the global tree widget! DTree->RootNode is global
 *        debugnode root
 *  Deallocates the old tree and builds a new one.
 *
 */
static void UpdateTree()
{
    char	*treelist;	/* Debug Tree as string			*/
    char	*csumstring;	/* Checksum to tell if new tree		*/
    long	csum = -1;
    char	*cp;
    char	 s[512];	/* General purpose string		*/


    if (tree_active <= 0) {	/* Check if tree exists			*/
/*        fprintf(stderr,"UpdateTree() called and no tree active!\n");*/
        return;
    };

        
    /* Check if we're inside a DebugFn (indicating that the tree might	*/
    /* be in an inconsistent state), refuse to update if yes		*/
    
    if ((c_swindow->displayedFile) &&
	(strstr(c_swindow->displayedFile->filename,
		DEBUGFNSOURCE) != NULL)) {
/*	fprintf(stderr,"Inside tree manipulation - tree not updated!\n");*/
	return;
    };


    /* Get the actual tree's checksum and test if it's the same	*/
    /* as before. If yes, we need not bother with redrawing!	*/

    csumstring = DirectlyQueryDbx("print GetChecksum(DTree)\n");
    if (strstr(csumstring,"not defined") != NULL) {
	XtFree(csumstring);
        UpdateMessageWindow(TREE_HELP,NULL);
        bell(0);
        return;
    };
    cp = strrchr(csumstring,'=');	/* Look for value for "atol"	*/
    if (cp) {
	cp++;
	csum = atol(cp);		/* Get checksum from string	*/
    } else {
	csum = -1;			/* No checksum -> default	*/
    };
    
/*fprintf(stderr,"CHECK:old %d new %d (string '%s', cp '%s')\n",treeChecksum,csum,csumstring,cp);*/
    
    XtFree(csumstring);
    if (csum == treeChecksum) {		/* Old one? Yes -> nothing to do*/
	return;
    };
    treeChecksum = csum;		/* Else remember new one	*/


    /* Get the whole tree as list and re-convert it to a tree	*/
    /* (Necessary because seperate address spaces!)		*/
    /* Complain if not defined, i.e. no such information found  */

    treelist = DirectlyQueryDbx("call PrintTreeAsString(DTree,0)\n");
    if (strstr(treelist,"not defined") != NULL) {
	XtFree(treelist);
        UpdateMessageWindow(TREE_HELP,NULL);
        bell(0);
        return;
    };
    KillTree(DTree);				/* Deallocate old tree	*/
    DTree = BuildTreeFromString(treelist,0);	/* Build new one	*/
    XtFree(treelist);


    /* Update the display Widget using the results */

    UpdateTreeWidget(treeW);


}

    

/*
 *  UpdateProfile()
 *
 *  Updates the currently shown profile widget by getting (by dbx)
 *  the profile table as string, re-converting it to a new profile and
 *  updating the shown profile widget.
 *  Note: profileW is the global Profile widget!
 *  Also clears the old profile before finning in new information
 *
 */
static void UpdateProfile()
{
    char	*profilelist;	/* Profile Table as string		*/
    char	 s[512];	/* General purpose string		*/


    if (profile_active <= 0) {	/* Check if profile exists		*/
/*        fprintf(stderr,"UpdateProfile() called and no profile active!\n");*/
        return;
    };


    /* Get the whole profile as list and re-convert it to a profile	*/
    /* table (Necessary because seperate address spaces!)		*/
    /* Complain if not defined, i.e. no such information found		*/

    profilelist = DirectlyQueryDbx("call PrintProfileAsString(PTable)\n");
    if (strstr(profilelist,"not defined") != NULL) {
        UpdateMessageWindow(PROFILE_HELP,NULL);
        bell(0);
        return;
    };
    ClearProfileTable(PTable);			/* Clear old entries	*/
    AddStringToProfile(PTable,profilelist,1);	/* Fill in new profile	*/


    /* Update the display Widget using the results */

    UpdateProfileWidget(profileW);


    if (profilelist)		/* Free resources again		*/
        XtFree(profilelist);        
}

    

/*
 *  UpdateActivations()
 *
 *  Updates the currently shown activation value array by getting the
 *  new array. Also checks if context is no longer syncronous, exits
 *  the Visualizer if yes (or if different array).
 *  Note: actVIS is the global Activation Visualizer
 *  (does some direct manipulation of the Visualizer's internal data...)
 *
 */
static void UpdateActivations()
{
    char	*actlist;	/* dbx output containing activations	*/
    char	 s[512];	/* General purpose string		*/
    DebugNode	*dnode;		/* Most recent node			*/
    long	*ab;		/* Test for act.buffer			*/

    
    if ((actVIS==NULL) || (actVIS->VISvis==NULL))
      return;

    actlist = DirectlyQueryDbx("call PrintNodeAsString(DTree->LastNode)\n");
    /*!!!!!!! NO PROBLEMS SHOULD OCCUR HERE (already got one node
    /*!!!!!!! so it should be OK to assume that we can get another one */
    
    dnode = BuildNodeFromString(actlist);
    XtFree(actlist);
    if (dnode->Context != act_context) {
	UpdateMessageWindow("Context has changed - closing activation",NULL);
	bell(0);
	free(dnode);
	VQuitWidget(actVIS->VISparent,		/* Destroy display	*/
		    actVIS->VISvis, actVIS);
	return;
    };

    
    /* Depending on context, fetch new activation data (sync)		*/
    /* or generate new array (par)					*/

    act_index=0;			/* Get value of act.index	*/
    if ((long)dnode->RangeVar) {
	sprintf(s,"*(long*)(%d)",(long)dnode->RangeVar);
	actlist = GetValue(s);
	if (actlist) {
	    act_index = atol(actlist);
	    XtFree(actlist);
	};
    };
    act_low	= (long)dnode->RangeLo;
    act_high	= (long)dnode->RangeHi;
    act_size    = (long)act_high-act_low;
    act_context = dnode->Context;
    
    if (act_context == DN_SYN_CONTEXT) {
    
	/* Collect new information. Complain if act.array address in	*/
	/* debuggee is different from former one. If it isn't, go on	*/
	/* (act_size shouldn't have changed, then, too)			*/
	
	if (act_address != (long) dnode->ActArray) { /* Address changed ?!? */
	    UpdateMessageWindow("Different activation array! Closing display",
				NULL);
	    bell(0);
	    free(dnode);
	    VQuitWidget(actVIS->VISparent,          /* Destroy display      */
			actVIS->VISvis, actVIS);
	    return;
	};
	
	act_compare = dnode->ActValue;		/* Get new active value	*/
	free(dnode);
	ab = (long*)FetchData((char*)act_address,
			      act_size*sizeof(long),
			      act_buffer);	/* Reuse old buffer	*/
	
	if (ab == NULL) {		/* Problem getting array, return */
	    UpdateMessageWindow(
			"Couldn't get activation array! Closing Display",NULL);
	    bell(0);
	    XtFree(act_buffer);			/* Free old buffer, too	*/
	    VQuitWidget(actVIS->VISparent,	/* Destroy display	*/
			actVIS->VISvis, actVIS);
	    return;
	};
	
	/* Set new active value for activations [array has changed	*/
	/* automatically since we read the new data into the old buffer]*/
	
	actVIS->VISvis->vcomparison = &act_compare;
	
	
    } else if (act_context == DN_PAR_CONTEXT) {

	memset(act_buffer,1,act_size*sizeof(long));	/* Update active */
	if ((act_index>=act_low) && (act_index<=act_high))
	 memset(act_buffer+act_index-act_low,0,
                 (act_high-act_index+1)*sizeof(long));

	/*  act_buffer[act_index]=1;			/* (using index) */
	
    };

    
    /* Update shown info in the Visualization */
    
    UpdateVisWidget(actVIS);
    

}

    

/*
 *  UpdateVisualizers()
 *
 *  Updates all currently shown array Visualizers by getting the
 *  new arrays. Also checks if Visualizer has been popped down in the
 *  mreantime or if data no longer available.
 *
 */
static void UpdateVisualizers()
{
    char	*newarray;	/* Most recently fetched array contents	*/
    int		 i;

    
    if (visnum==0)
      return;

    /* Handle all active Visualizers in "vislist" */
    
    for (i=0; i<visnum; i++) {

	while ((vislist[i].vis == NULL) ||	/* Has been popped down,*/
	    (vislist[i].vis->VISvis == NULL)) {	/*  erase from list	*/
	    int j=i;

	    XtFree(vislist[i].vis);
/*fprintf(stderr,"vis %d was popped down, erasing entry\n",i);*/
	    visnum--;
	    while (j<visnum) {			/* Shift others down	*/
		vislist[j] = vislist[j+1];
		j++;
	    };
	    if (i>= visnum) {			/* Was the last one, done */
/*fprintf(stderr,"was last one, returning from update\n");*/
                return;
            };
	    /* If Visualizer left, treat it next, instead of the erased one */
	}

	/* Fetch array data into old buffer "array" */
	
	newarray = FetchData((char*)vislist[i].adr,
			     vislist[i].len,
			     vislist[i].array);

	
	/* Problems getting data -> free buffer & destroy Visualizer     */
	/* NOTE: it will be detected at the next call of UpdateVisualzers*/
	/*       that this Visualizer is now down, so don't bother	 */
	/*	 erasing its record now					 */
	
	if (newarray==NULL) {
	    UpdateMessageWindow(
		"Couldn't get visualized array, closing display %d",i);
	    bell(0);
	    XtFree(vislist[i].array);
	    VQuitWidget(vislist[i].vis->VISparent,	/* Pop down and	*/
			vislist[i].vis->VISvis,		/*  destroy	*/
			vislist[i].vis);
	} else {
	    
	    /* Update shown info in the Visualization */
    
/*fprintf(stderr,"updating Visualizer %d\n",i);*/
	    UpdateVisWidget(vislist[i].vis);
	};
	
    };

}



/*
 * void HighlightText(int froml,fromc,tol,toc, SWindow *swindow)
 *
 *  Highlights the text in the given SWindow between the two given
 *  lines/columns, adjusting it as needed.
 */
void HighlightText(froml,fromc,tol,toc,swindow)
    int	   	froml,fromc,tol,toc;
    SWindow	*swindow;
{
    FileRec 		*file;

    if ((file = swindow->displayedFile) == NULL ||
         froml <= 0 || tol < froml)
       return;

    AdjustText(froml,swindow);

    XawTextSetSelection(swindow->sourceWindow,
                        file->linepos[froml]+fromc-1, file->linepos[tol]+toc);
    
}


/*
 * void UpdateHighlights(SWindow *swindow)
 *
 * Updates the Highlights in the given Modula-2* SWindow according
 * to the line information in the most recent DebugTree leaf.
 *
 */
void UpdateHighlights(swindow)
SWindow *swindow;
{
    DebugNode *last;

    if (swindow == NULL)
      return;
    
  if (0) { /* USE LINE NUMBERS */
    if (DTree->RootNode!=NULL) {	/* Use info from most recent node */
        last = DTree->LastNode;		/* was: GetLastNode(RootNode);	  */
        if (last!=NULL) {
            HighlightText(last->MLine, last->MColumn,
                          last->MToLine, last->MToColumn,
                          swindow);
        };
    } else {
/*fprintf(stderr,"UpdateHighlights: No tree active!\n");*/
    };
    
  } else {	/* Hightlight the corresponding line */

      if (swindow->OtherSW) {
	  int line = CToModulaLine(swindow,arrows[0].line);

	  if (line>0) 
	    HighlightText(line,0,line,10,swindow);
      };
  };

}

    

/*  Handle dbx output of run, cont, next, step, return commands.
 *  Result of output parsing is returned in a set of tokens.
 */
/* use globally active window a_swindow */
void exec_handler()
{
    int	 line, status;
    char *func, *mesg;
    char *segv = "signal SEGV";
    char *segfault = "Segmentation fault";


    /* Print "stopped in ..." line in message window 
     * Adjust text displayed
     */

    if (Token.func == NULL || Token.line == 0) 
	return; 
    UpdateMessageWindow(Token.mesg);
    line = Token.line;
    func = XtNewString(Token.func);
    mesg = XtNewString(Token.mesg);
#ifdef MIPS
    status = LoadCurrentFile(c_swindow);	/*!!!!!!! FOR MIPS ALSO */
#else
    if (Token.file) {

        /* Load new file. Check if corresponding Modula-2* Source exists  */
        /* and load it into the "OtherSW" source window (generate a new	  */
	/* one if necessary)						  */

	status = LoadFile(Token.file,c_swindow);
    }
#endif
    arrows[0].line = line;	/* update arrow sign position */
    strcpy(arrows[0].func, func);
    updown.line = 0;		/* remove updown, if any */
    if (c_swindow->displayedFile) {
    	arrows[0].file = c_swindow->displayedFile->pathname;
    }

    /* Display bomb sign if segmentation fault occurs in source code */
	
    if (status != -1 && (strncmp(mesg, segv, strlen(segv)) == NULL ||
	strncmp(mesg, segfault, strlen(segfault)) == NULL)) {
	arrows[0].line = 0;
	bomb.line = line;
	strcpy(bomb.func, func);
    	if (c_swindow->displayedFile)
            strcpy(bomb.file, c_swindow->displayedFile->pathname);
    }
    else
	bomb.line = 0;

    if ((c_swindow->SWType == SWTYPE_C) && (c_swindow->OtherSW)) {
	
	arrows[0].mline = CToModulaLine(c_swindow,arrows[0].line);
	/* line mapping returns 0 if line=0 so it's ok even if bomb occured! */
	if (bomb.line)
	  bomb.mline = CToModulaLine(c_swindow,bomb.line);
    } else {
	arrows[0].mline = 0;
	bomb.mline = 0;
    };
    
    AdjustText(line,c_swindow);		/* (!) also adjusts M-Source	*/
#ifndef BSD
    display_handler();
#endif


    /* When executing a "next" in Modula, check if M-Line has	*/
    /* already changed and send next "next" if not		*/
    
    if ((m_swindow) && (mline_next)) {
	if (m_swindow->currentline == mline_next) {
	    send_command("next\n");	/* Still same, next step	*/
	} else {
	    mline_next = 0;		/* New line, no more stepping	*/
	};
    };

    /* Update additional info windows, if not stepping */
    
    if (mline_next == 0) {
	
	if (tree_active)		/* Update structure tree info... */
	  UpdateTree();
	if (profile_active)		/* .. and profile, if any active */
	  UpdateProfile();
	if ((actVIS != NULL) &&
	    (actVIS->VISvis != NULL)) 	/* .. and activation values	 */
	  UpdateActivations();	
	if (visnum > 0)
	  UpdateVisualizers();		/* .. and all Visualizers	 */
    };

    XtFree(func);
    XtFree(mesg);
	    
}


/*  Remove all the arrow and updown signs, print message, then 
 *  change the file variable to the file name displayed.
 */
/* use globally active window a_swindow */
void done_handler()
{
    char command[LINESIZ];

    ClearArrows();
    updown.line = 0;
    updown.mline = 0;
    UpdateArrows(c_swindow->displayedFile);
    UpdateUpdown(c_swindow->displayedFile);
    UpdateMessageWindow("Ready for execution");
    if (c_swindow->displayedFile == NULL) return;
#ifdef MIPS
    sprintf(command, "file %s\n", c_swindow->displayedFile->filename);
#else
    sprintf(command, "file %s\n", c_swindow->displayedFile->pathname);
#endif
    Parse = False;
    query_dbx(command);
}


/*  Place a stop sign next to the line specified on the source file window 
 *  if it is to be viewable.
 */
/* use globally active window a_swindow */
void stop_at_handler()
{
    if (Token.stop == 0 || Token.line == 0 || c_swindow->displayedFile == NULL)
	return;
    if (Token.file == NULL)
	stops[Token.stop].file = c_swindow->displayedFile->pathname;
    else
	stops[Token.stop].file = GetPathname(Token.file);
    DisplayStop(c_swindow->displayedFile, Token.line);
    stops[Token.stop].line = Token.line;
    stops[Token.stop].tag = 0;
    nstops = Token.stop;

    /* Also mark stop in in corresp. Modula window, if any */
    
    if ((c_swindow->SWType == SWTYPE_C) && (c_swindow->OtherSW)) {
	stops[Token.stop].mline = CToModulaLine(c_swindow,Token.line);
    } else {
	stops[Token.stop].mline = 0;
    };

}


/*
 *  Place a stop sign next to the function routine, getting the line number 
 *  by "list <func>", (or "func <func>" on a MIPS), and resetting the file 
 *  variable properly.
 */
/* use globally active window a_swindow */
void stop_in_handler()
{
    char command[LINESIZ], *file;
    int  stop;
    int	 line;

    if (Token.stop == 0 || Token.func == NULL ||
         c_swindow->displayedFile == NULL)
	return;
    stop = Token.stop;
#ifdef MIPS
    /* For mips dbx, need to use func command to locate the function */
    Skip_func_handler = True;
    sprintf(command, "func %s\n", Token.func);
    query_dbx(command);
#else
#ifdef BSD
    sprintf(command, "list %s\n", Token.func);
    query_dbx(command);
#else
    sprintf(command, "list %s\n", Token.func);
    query_dbx(command);
    if (Token.line <= 0) 
	return;
    else 
	Token.line += 5;
#endif
#endif

    stops[stop].line = Token.line;
    nstops = stop;
    line = Token.line;


    /* Also mark stop in in corresp. Modula window, if any */
    
    if ((c_swindow->SWType == SWTYPE_C) && (c_swindow->OtherSW)) {
	stops[stop].mline = CToModulaLine(c_swindow,Token.line);
    } else {
	stops[stop].mline = 0;
    };

    /* Check the name of the file containing Token.func */
    
    query_dbx("file\n");
    if ((file = GetPathname(c_swindow->CurrentFile)) && 
        strcmp(file, c_swindow->displayedFile->pathname)) {   /* new file, record stop */
	stops[nstops].file = file;
#ifdef MIPS
	sprintf(command, "file %s\n", c_swindow->displayedFile->filename);
#else
	sprintf(command, "file %s\n", c_swindow->displayedFile->pathname);
#endif
	Parse = False;
	query_dbx(command);
    }
    else { 					   /* same file, display stop*/
	stops[nstops].file =
               c_swindow->displayedFile->pathname;
	DisplayStop(c_swindow->displayedFile, line);
	if (stops[nstops].mline)
	  DisplayStop(m_swindow->displayedFile, stops[nstops].mline);
	  
    }
}


/*  
 *  Display an outlined arrow to locate the calling routine in a stack
 *  frame.  BSD and SUN dbx have slightly different output semantics here.
 *  The appropriate file with the calling routine is displayed and the
 *  file variable is set accordingly.
 */
/* use globally active window a_swindow */
void updown_handler()
{
    char command[LINESIZ], *func, *file;
    int	 line;


    line = Token.line;
    func = XtNewString(Token.func);
#ifdef MIPS
    LoadCurrentFile(c_swindow);
#endif
#ifdef BSD
    file = GetPathname(Token.file);
#else
    if (line <= 0) line = 1;
    LoadCurrentFile(c_swindow);
    if (c_swindow->displayedFile)
	file = c_swindow->displayedFile->pathname;
#endif

    if (line <= 0 || func == NULL || file == NULL) 
	return;
    if (c_swindow->displayedFile &&
         strcmp(file, c_swindow->displayedFile->pathname)) {

        /* Load new file. Check if corresponding Modula-2* Source exists  */
        /* and load it into the "OtherSW" source window (generate a new	  */
	/* one if necessary)						  */

	LoadFile(Token.file,c_swindow);
	

	/* set dbx file variable to file */
#ifdef MIPS
	sprintf(command, "file %s\n", c_swindow->displayedFile->filename);
#else
	sprintf(command, "file %s\n", c_swindow->displayedFile->pathname);
#endif
	Parse = False;
	query_dbx(command);
    };
    updown.line = line;
    strcpy(updown.func, func);
    if (c_swindow->displayedFile)
    	strcpy(updown.file, c_swindow->displayedFile->pathname);
    AdjustText(line,c_swindow);
    if ((c_swindow->SWType == SWTYPE_C) &&	/* Also update M-source	 */
        (c_swindow->OtherSW)) {
	/*!!!!!!!*/
	AdjustText(CToModulaLine(c_swindow,line),m_swindow);
    }
    XtFree(func);
}


/*
 *  Delete handler: remove the stop specified and undisplay the stopsign
 *  if it's visible.
 *  It calls the dbx status command to find out what stops are left, and
 *  then update the array of stops accordingly.
 */
/* ARGSUSED */
/* use globally active window a_swindow */
/* but affect all windows that are displaying the same file! */
void delete_handler()
{
    char s[LINESIZ];
    int  i; 
    int	 line,mline;

    write_dbx("status\n");
    while (fgets(s, LINESIZ, dbxfp) == NULL);		/* Sync, fetch all   */
    do {						/*  stop line numbers*/
	if (strcmp(s, dbxprompt) || strcmp(s, "")) {
	    sscanf(s, BRACKET, &i);
	    if (i > 0 && 
                i <= nstops && stops[i].line > 0) 
	    	stops[i].tag = 1;
	}
    } while (fgets(s, LINESIZ, dbxfp));

    for (i=1; i<=nstops; i++)
	if (stops[i].line > 0) {
	    if (stops[i].tag)
		stops[i].tag = 0;
	    else {
		line = stops[i].line;
		mline= stops[i].mline;
		stops[i].line = 0;
		stops[i].file = NULL;
		if (LineToStop_no(line,c_swindow) == 0) {
		    RemoveStop(line,mline,	/* Also removes M-src stop */
			       c_swindow);
		}
	    }
	}
}

/*
 *  This handler displays the function routine on the source window.
 *  It locates the function by sending the dbx command "list <func>",
 *  and loads the appropriate file accordingly.
 */
/* use globally active window a_swindow */
void func_handler()
{
    int	 line;
    char command[LINESIZ];

    if (Token.func && !Skip_func_handler) {
#ifdef MIPS
	line = Token.line;
#else
	sprintf(command, "list %s\n", Token.func);
	query_dbx(command);
	line = Token.line + 5;
#endif
	LoadCurrentFile(c_swindow);
	AdjustText(line,c_swindow);
    }
    Skip_func_handler = False;
}


/*  File handler first queries the current file set by the user command,
 *  and then loads the file.
 */
/* ARGSUSED */
void file_handler() 	/* Command was 'file' */
{
    if (Token.file)
	strcpy(c_swindow->CurrentFile, Token.file);
    else
	strcpy(c_swindow->CurrentFile, "");
}

/* ARGSUSED */
/* use globally active window a_swindow */
void debug_handler()
{
    query_dbx("use\n");
    c_swindow->displayedFile = NULL;	/* force reloading of source file */
    if (LoadCurrentFile(c_swindow) == 0) {
	ClearArrows();			/* clear arrow signs */
	updown.line = 0;		/* clear updown sign */
	updown.mline= 0;
	bomb.line = 0;			/* clear bomb sign */
	bomb.mline= 0;
	ClearStops();
	query_dbx("stop in DebugEnd\n");	/* Preserve profile info! */
	query_dbx("func BEGIN_MODULE\n");
	UpdateArrows(c_swindow->displayedFile);
	UpdateUpdown(c_swindow->displayedFile);
	UpdateBomb(c_swindow->displayedFile);
	UpdateStops(c_swindow->displayedFile);
        UpdateMessageWindow("Ready for execution");
#ifndef BSD
	query_dbx("display\n");		/* clear display window */
#endif
    }
}

/* ARGSUSED */
void cd_handler()
{
    query_dbx("pwd\n");
}

/* ARGSUSED */
void pwd_handler(s)
char *s;
{
    strcpy(cwd, (char *)strtok(s, "\n"));
}

/* ARGSUSED */
void use_handler(output)
char *output;
{
    if (strcmp(output, "") == NULL)
	query_dbx("use\n");
    else
    	MakeDirList(output);
}

/* ARGSUSED */
void search_handler()
{
    AdjustText(Token.line,c_swindow);
}

/* ARGSUSED */
void list_handler()
{
    int	 line;

    if (Echo) {
	line = Token.line;
	LoadCurrentFile(c_swindow);
    	AdjustText(line,c_swindow);
    }
}


/* ARGSUSED */
/*  Show output on the display window.
 *  If output is null but the display window is managed, replace contents of
 *  the display window with the null string.
 */
void display_handler()
{
    Arg		args[MAXARGS];
    Cardinal	n;

    if (!Token.display || strcmp(Token.display, "") == NULL) {
	if (!XtIsManaged(displayWindow))
	    return;
	else {
	    XtFree(Token.display);
	    Token.display = XtNewString("");
	}
    }
    if (!XtIsManaged(displayWindow)) {
	XtManageChild(separator);
	XtManageChild(displayWindow);
    }
    n = 0;
    XtSetArg(args[n], XtNstring, (XtArgVal) Token.display);		n++;
    XtSetValues(displayWindow, args, n);
    XtFree(Token.display);
}




