/*****************************************************************************
 *
 *  xdbx - X Window System interface to the dbx debugger
 *
 *  Copyright 1989 The University of Texas at Austin
 *  Copyright 1990 Microelectronics and Computer Technology Corporation
 *
 *  Permission to use, copy, modify, and distribute this software and its
 *  documentation for any purpose and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation, and that the name of The University of Texas
 *  and Microelectronics and Computer Technology Corporation (MCC) not be 
 *  used in advertising or publicity pertaining to distribution of
 *  the software without specific, written prior permission.  The
 *  University of Texas and MCC makes no representations about the 
 *  suitability of this software for any purpose.  It is provided "as is" 
 *  without express or implied warranty.
 *
 *  THE UNIVERSITY OF TEXAS AND MCC DISCLAIMS ALL WARRANTIES WITH REGARD TO
 *  THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS, IN NO EVENT SHALL THE UNIVERSITY OF TEXAS OR MCC BE LIABLE FOR
 *  ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 *  CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 *  CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 *  Author:  	Po Cheung
 *  Created:   	March 10, 1989
 *
 *****************************************************************************/

/*  command.c
 *
 *    Create the command window, the command buttons and their callbacks.
 *
 *    CreateCommandPanel() : 	Create a window with command buttons
 *    CreateButtons() :		Create command buttons in panel
 *    AddButton() :		Add a command button into the command window
 *    ButtonSet() :		Action proc for command button translation
 *
 *    Command callbacks for the command buttons:
 *
 *    forwardSearch() :		forward string search
 *    reverseSearch() :		reverse string search
 *    Search() :		call either forwardSearch() or reverseSearch()
 *    PopupSearch() :		command callback for search button
 *    DoneSearch() :		command callback for DONE btn in search panel
 *    CreateSearchPopup() :	create search panel
 *    PopupSW() :		pop up a new SWindow
 *
 *    Command queue manipulation routines:
 *	send_command():		send a command to dbx and record in the queue
 *	get_command():		read command off head of queue
 *	insert_command():	insert command at the head of queue
 *	delete_command():	delete command from head of queue
 *
 *    Special features:
 *
 *	DirectlyQueryDbx():	get result from dbx without extra handlers
 *	DisplayVisualizer():	pop up a visualizer showing the selected array
 *	DisplayTree():		pop up the call structure tree of the program
 *	DisplayProfile():	pop up profiling information
 *	DisplayActivations():	pop up activation value visualizer
 *	FetchData():		get data from debugged program
 *	GetAddress():		find out address of variable in debuggee
 *	GetValue():		find out value of var/expr in debuggee
 *	
 *
 *
 *  Changes by Stefan Haenssgen:
 *
 *  07-19-91	Modified to use SWindow instead of sourceWindow etc
 *  07-31-91	Use a_swindow for globally active one and pass
 *		 swindow as parameter for calls that affect just one
 *		 certain window (such as action procedures etc)
 *		Modified Stop_at, Delete, PopupSearch, Search to use
 *		 global a_swindow
 *		Modified Search to give a_swindow as parameter to AdjustText
 *  08-02-91	Added button to pop up new SWindow
 *	 	Wrote PopupSW to handle this action
 *  02-04-92	Added code to use the Visualizer: VisualizeIt, corresponding
 *		 button etc
 *  02-09-92	Implemented the rest of the code for Visualization
 *		Copied code from dbx.c and parse.c to do our own query
 *		 to dbx and parse the result on our own
 *	        Get information like size and number of dimensions from the
 *		 information file "<file>.arrayinfo" 
 *  04-14-92	Generate dump of array to file by calling a dump routine
 *		 that has to be linked to the user program
 *		Read the dump file and visualize the data directly
 *		Introduced DirectlyQueryDbx() to get output as result of
 *		 dbx command without additional interference from xdbx
 *		Get name of program being debugged via "debug"-output
 *  05-24-92	Use Grayscale Visualizer if right mouse button pressed:
 *		Added VisualizeValues() to VisualizeIt()
 *		Modified AddButton() to report the right button to VisualizeIt
 *
 *  07-aug-92	Introduced new buttons: "structure" to show the call tree
 *		 of the program, "profile" to show profiling information
 *		Included DebugFn.h for DebugNode & ProfileTable etc
 *		Wrote callbacks "StructureIt" and "ProfileIt":
 *		- check for presence of debugging information
 *		- use PrintTreeAsList() and MakeTreeFromList() to pass
 *		   debug tree information from program to xdbx
 *		- dito PrintProfileAsList() and MakeProfileFromList()
 *  12-aug-92	Modified above functions to use the new DebugFn modifications
 *		 (line AND column, ending line & column supplied).
 *  14-aug-92	Ensured that no more than one structure/profile is active
 *		 at once ({structure,profile}_active)
 *		Renamed "structure" to "tree" for call structure tree
 *		Made profileW and treeW global (just one active at once anyway)
 *		Use global RootNode for debug info Tree
 *  19-aug-92	Don't check 1st if Print..AsList() defined, test after calling
 *		 dbx to get the result, instead
 *  02-sep-92	Finally found the "why is my output incomplete" bug in
 *		 DirectlyQueryDbx(): concat kills the 1st string if
 *		 the 2ns one is null!
 *		Fixed this bug - all OK now! ha!
 *		Use treeChecksum to tell if structure tree has changed,
 *		 saves many redrawing efforts
 *  01-nov-92	Adapted to new DebugFnV calls & changes (Tree as ADT etc)
 *		Use new global DTree & PTable, deallocate old tree/profile
 *		 before building a new one
 *  24-nov-92	Renamed TreeIt etc to DisplayTree etc
 *		Introduced DisplayActivations for activation value arrays
 *		Use new Visualizer Widgets for activation display
 *		 (global Visualizer struct actVIS)
 *		Added ActInfoPopup and ActInfoQuit to handle popup infos
 *		 for activation array display
 *  25-nov-92	Use new NodeAsString() and BuildNodeFromString() for
 *		 getting activation infos
 *		Wrote new function FetchData() to get data from debugged
 *		 program to the debugger (by copying through a file,
 *		 rather un-nicely, but does it for the moment)
 *		Wrote GetAddress() to get address of variable
 *		Experimented with ptrace() to get data from debuggee
 *		Use errno & sys_errlist[]
 *		New dumping strategy: 1st try ptrace() based GimmeData()
 *		 in debuggee. If that fails, use file based generate_dump()
 *		Included generate_dump() into DebugFns, renamed to
 *		 DumpToFile()
 *		Regard actVIS as free if == NULL (is 1st call) or
 *		 actVIS->VISvis == NULL (last one was popped down)
 *  26-nov-92	Rewrote FetchData() to take additional buffer parameter
 *		 and use this buffer, create new one if NULL, return NULL
 *		 on failure (but leave buffer intact if not self created)
 *		Introduced global act_buffer and act_address (to remember
 *		 address of activation array in debuggee, notice changes!)
 *  27-nov-92	Use newly included activation value ranges
 *		Added act_counter, act_low, act_high  for better activation
 *		 display
 *		Added act_context
 *		Also visualize parallel (async) foralls, as empty with just
 *		 the current index active
 *		Disabled the new-windows-popup
 *		When stopping at a line, compute the corresponding C-line
 *		 if in Modula source
 *  28-nov-92	Also use/update corresponding Modula line "mline"
 *		Transform Modula to C line when deleting breakpoint
 *  29-nov-92	Introduced DoItNext to handle the "next" command for M-src
 *		 by setting a global flag and stepping until M-line changes
 *		Entered "activations" into button-3-extras
 *  30-nov-92	Introduced GetValue() to return value of variable/expression
 *		 in debuggee (as string!)
 *		Completely rewrote DisplayVisualizer(), use name mapping
 *		 for Modula names
 *		Get size and type etc from dbx
 *		Use Visualizer Widgets instead of standalone-popup
 *		Allocate memory for from/to range (not local to DispVis!)
 *  01-dec-92	Also check if candidate for Visualization is an array
 *		 at all (inside a struct)
 *		Parse for dimensions & number of dimensions
 *		Use one of three possible Visualizer types, depending on
 *		 mouse button pressed
 *		Enter Visualizer in global list of active Visualizers "vislist"
 *		 and update "visnum"
 *		Expanded button range to 1,2,3
 *  10-mar-93	Show "0" as active in async.Forall Act.Visualizer
 *		 (memset uses bytes...)
 *		Also set last pixel to "active" (was "inactive", +1 now)
 *  12-mar-93	Add filename to "stop at" command, to make sure we're setting
 *		 the Breakpoint in the right file (not e.g. DebugFn.c)
 *  		Check if inside DebugFn source, refuse to display tree if
 *		 yes (tree might be in inconsistent state, kabooom!)
 *		Disabled buttons which are not useful for MSDB (e.g.
 *		 up, down, file)
 *  31-aug-93	Include "DebugFn/DebugFn*"
 *
 */
 
#include <signal.h>
#include <ctype.h>
#include <sys/ptrace.h>
#include <sys/wait.h>
#include <errno.h>

#include "DebugFn/DebugFnV.h"
#include "Vis.h"
#include "global.h"		/* Must be after Vis.h (for VIStype) !!! */

#define	 REVERSE	0
#define	 FORWARD	1

extern char	*sys_errlist[];
extern int	errno;

Widget		commandWindow;			/* command panel with buttons*/
Boolean		PopupMode = False;
static int	Button;
static Widget	searchPopupShell, searchPopup;
static Widget	AddButton();
static Widget	button[30];
static char	SearchString[BUFSIZ] = "";	/* search string buffer */
static char	command[LINESIZ];
static CommandRec *commandQueue = NULL;
#ifdef BSD
static char	savedCommand[LINESIZ] = ""; 
#endif

Widget		 treeW, profileW;		/* Widgets for tree/profile */
long		 treeChecksum = -1;		/* Checksum "anything new?" */

VIStype		*actVIS = NULL;			/* Activation Visualizer    */
long		 act_compare = 0;		/* Activation value	    */
int		 act_size    = 1;		/* Size of act.array	    */
long		*act_buffer  = NULL;		/* Buffer for act.array	    */
long		 act_address = 0;		/* Address in debuggee	    */
long		 act_low     = 0;		/* Activation index range   */
long		 act_high    = 0;
long		 act_index   = 0;		/* Index in array	    */
int		 act_context = 0;		/* Context (par/sync)	    */

void		 ActInfoPopup();		/* Popup for Act.Infos	    */
void		 ActInfoQuit();			/* Callback for popup	    */
char		*FetchData();			/* Get data from debuggee   */
long		 GetAddress();			/* Get address of variable  */

int		 mline_next=0;			/* Flag/modula line to chnge*/

VisRec		 vislist[MAX_VISNUM];		/* List of active Visualizers*/
int		 visnum=0;			/* Number of dito	     */


/*
 * char *DirectlyQueryDbx(command)
 *
 *   Query dbx and get output, all by hand (Not interferring with
 *   existing global structures of xdbx)
 *   [Adapted from query_dbx(), write_dbx() and read_dbx()  (in dbx.c) ]
 *
 *   The result should be deallocated via XtFree() when no longer needed
 *
 */
char *DirectlyQueryDbx(command)
char *command;
{
    char   *output = NULL;	/* Buffer for dbx output		*/
    char   *next_string = NULL;	/* String after dbx prompt		*/
    char   *string = NULL;	/* YAB (yet another buffer)		*/
    char    s[LINESIZ];		/* YAB					*/
    Boolean more;		/* True if more output from dbx pending	*/
    Boolean Prompt=False;	/* True if dbx prompt seen		*/


    fputs(command, dbxfp);	/* Give command to dbx for execution	*/
    fflush(dbxfp);
    while (!Prompt) {		/* Read dbx output until dbx prompt..	*/
      more = True;		/* arrives or nothing more to read	*/
      while (more) {
        Prompt = False;
        while (more = fgets(s, LINESIZ, dbxfp) && !Prompt) {
            if (strncmp(s, dbxprompt, strlen(dbxprompt)) == NULL) {
                Prompt = True;	/* Got the dbx prompt, remember any..	*/
                if (s[strlen(dbxprompt)])	/* stuff behind it, too	*/
                    next_string = XtNewString(s+strlen(dbxprompt));
                strcpy(s, "");
            };
            string = concat(string, s);
            strcpy(s, "");
        };

        /* Now we've got the output from dbx	*/

        output = concat2(output, string);		/* USE CONCAT2!	*/
        if (string) {			/* Free strings			*/
            XtFree(string);
            string = NULL;
        };
        if (next_string) {		/* Remember additional stuff	*/
            string = concat(string, next_string);
            XtFree(next_string);
            next_string = NULL;
        };
      };
    };

    if (string) {
        XtFree(string);
        string = NULL;
    };
    return (output);

};


/* ARGSUSED */
static void ButtonSet(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    Button = atoi(params[0]);
}


/* ARGSUSED */
/*  Execute the dbx command specifed in client_data
 */
static void DoIt (w, command, call_data)
    Widget w;
    XtPointer command;
    XtPointer call_data;
{
    /* run, cont, next, step, where, up, down, status */
    send_command(command);
    AppendDialogText(command);
}

/*
 * Handler for "next" - sets global flag to step until M-source changes
 *
 */
static void DoItNext (w, command, call_data)
    Widget w;
    XtPointer command;
    XtPointer call_data;
{
    Boolean OnlyC;

    /* Just a normal C-step if right button used or no M-src	*/
    
    OnlyC = ((Button == 3) || (m_swindow == NULL));

    if (!OnlyC) {
	mline_next = m_swindow->currentline;	/* Set step-until-changed */
    } else {
	mline_next = 0;				/* Set normal step	  */
    };
    send_command("next\n");
    AppendDialogText("next\n");
}


/* ARGSUSED */
static void Return (w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *funcname;
    int  nbytes;

    funcname = XFetchBytes(display, &nbytes);	/* from CUT_BUFFER0 */
    if (nbytes == 0)
    	strcpy(command, "return\n");
    else
    	sprintf(command, "return %s\n", funcname);
    send_command(command);
    AppendDialogText(command);
}


/* ARGSUSED */
/* use the active swindow */
/* may affect other windows that display the same file! */
static void Stop_at(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    XawTextPosition pos;
    int   line, c_line;
    char *filename;

    if (a_swindow->displayedFile == NULL) {
	UpdateMessageWindow(STOP_AT_HELP, NULL);
	bell(0);
	return;
    }
    pos = XawTextGetInsertionPoint(a_swindow->sourceWindow);
    line = TextPositionToLine(pos,a_swindow);
    
    if (a_swindow->SWType == SWTYPE_MODULA) {	/* Modula window:	*/
	FileRec *otherfile = a_swindow->displayedFile->OtherFile;
	
	c_line  = ModulaToCLine(a_swindow,line);/*  Use corresp. C line	*/
	if (otherfile)
	  filename= otherfile->filename;	/* Name of corr. C file	*/
	else
	  filename=NULL;
    } else {
	c_line   = line;			/* C Window: directly	*/
	filename = a_swindow->displayedFile->	/* Remember name	*/
	            filename;
    };

    if (filename)
      sprintf(command, "stop at \"%s\":%d\n", filename, c_line);
    else
      sprintf(command, "stop at %d\n", c_line);
      
    send_command(command);
    AppendDialogText(command);
	
	
}


/* ARGSUSED */
static void Stop_in(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *funcname;
    int  nbytes;

    funcname = XFetchBytes(display, &nbytes);	/* from CUT_BUFFER0 */
    if (nbytes == 0) {
	UpdateMessageWindow(STOP_IN_HELP, NULL);
	bell(0);
	return;
    }
    sprintf(command, "stop in %s\n", funcname);
    send_command(command);
    AppendDialogText(command);
}


/*  Delete removes the stop_no associated with a given line number.
 *  RemoveStop() is called to undisplay the stop sign only when there
 *  are no more stop_no's associated with that line number.
 *  (if in modula window, use corresponding c window!)
 */
/* use the active swindow */
/* may affect other windows that display the same file! */

/* ARGSUSED */
static void Delete(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    XawTextPosition pos;
    char	    *string;
    int		    stop_no, line, nbytes;

    string = XFetchBytes(display, &nbytes);
    if (nbytes > 0 && (stop_no = atoi(string)) > 0) {
    	sprintf(command, "delete %d\n", stop_no);
	send_command(command);
	AppendDialogText(command);
	return;
    }
    else if (a_swindow->displayedFile) {
	SWindow *cwindow = NULL;
	
	pos = XawTextGetInsertionPoint(a_swindow->sourceWindow);
	line = TextPositionToLine(pos,a_swindow);

	/* Transform to C line if in Modula window */
	
	if ((a_swindow->SWType == SWTYPE_MODULA) && (a_swindow->OtherSW)) {
	    int cline = 0;
	    int i;
	    
	    /* Here, we can't use ModulaToCLine() because that might	*/
	    /* point to a comment line etc, and the real stop would	*/
	    /* be below that line. Search in stop infos, instead	*/
	    
	    for (i=1; i<=nstops; i++)
	      if ((stops[i].mline == line) && (stops[i].line))
		cline=stops[i].line;
	    line = cline;
	    cwindow = a_swindow->OtherSW;
	    
	} else {
	    
	    cwindow = a_swindow;
	    
	};
	if (stop_no = LineToStop_no(line,cwindow)) {
	    sprintf(command, "delete %d\n", stop_no);
	    send_command(command);
	    AppendDialogText(command);
	    return;
	}
    }
    UpdateMessageWindow(DELETE_HELP, NULL);
    bell(0);
}

/* ARGSUSED */
static void Print(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *string;
    int nbytes;

    if (Button == 3) PopupMode = True;

    string = XFetchBytes(display, &nbytes);
    if (nbytes == 0) {
	UpdateMessageWindow(PRINT_HELP, NULL);
	bell(0);
	return;
    }
    if (client_data == (XtPointer)0)
	sprintf(command, "print %s\n", string);
    else if (client_data == (XtPointer)1)
	sprintf(command, "print *%s\n", string);
    send_command(command);
    AppendDialogText(command);
}

/* ARGSUSED */
static void Func(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *funcname;
    int nbytes;

    funcname = XFetchBytes(display, &nbytes);
    if (nbytes == 0)
    	strcpy(command, "func\n");
    else
    	sprintf(command, "func %s\n", funcname);
    send_command(command);
    AppendDialogText(command);
}

/* ARGSUSED */
static void Quit(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    union wait status;

    write_dbx("quit\n");
    XtDestroyApplicationContext(app_context);
    kill(dbxpid, SIGKILL);
    wait3(&status, WNOHANG, NULL);
    exit(0);
}


/* ARGSUSED */
static void Display_(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *string;
    int nbytes;

    string = XFetchBytes(display, &nbytes);
    sprintf(command, "display %s\n", string);
    send_command(command);
    AppendDialogText(command);
}


/* ARGSUSED */
static void Undisplay(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *string;
    int	 stop_no, nbytes;

    string = XFetchBytes(display, &nbytes);
    if (nbytes == 0) {
	UpdateMessageWindow(UNDISPLAY_HELP, NULL);
	bell(0);
	return;
    }
    if ((stop_no = atoi(string)) > 0)
	sprintf(command, "undisplay %d\n", stop_no);
    else
    	sprintf(command, "undisplay %s\n", string);
    send_command(command);
    AppendDialogText(command);
}


/* ARGSUSED */
static void Dump(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    char *funcname;
    int nbytes;

    funcname = XFetchBytes(display, &nbytes);
    if (nbytes == 0)
    	strcpy(command, "dump\n");
    else
    	sprintf(command, "dump %s\n", funcname);
    send_command(command);
    AppendDialogText(command);
}


/*  Beginning from startpos, this routine searches text forward for 
 *  searchstring, and returns 1 if searchstring is found, also returning 
 *  the left and right positions of the matched string in left and right; 
 *  else 0 is returned.
 *  It also does wrap-around search.
 */
static forwardSearch(text, startpos, searchstring, left, right)
    char *text;
    int  startpos;
    char *searchstring;
    XawTextPosition  *left, *right;
{
    int  searchlength, searchsize, i, n=0;
    char *s1, *s2;

    searchlength = strlen(searchstring);
    searchsize = strlen(text) - searchlength;
    for (i=startpos; i < searchsize; i++) {
	n = searchlength;
	s1 = &text[i];
	s2 = searchstring;
	while (--n >= 0 && *++s1 == *s2++);
	if (n < 0) break;
    }
    if (n < 0) {
    	*left = i+1;
    	*right = i+1+searchlength;
    	return 1;
    }
    else {
	for (i=0; i < startpos; i++) {
	    n = searchlength;
	    s1 = &text[i];
	    s2 = searchstring;
	    while (--n >= 0 && *++s1 == *s2++);
	    if (n < 0) break;
	}
	if (n < 0) {
	    *left = i+1;
	    *right = i+1+searchlength;
	    return 1;
	}
	return 0;
    }
}
	

/*  Similar to forwardSearch(), except that it does a reverse search
 */
static reverseSearch(text, startpos, searchstring, left, right)
    char 	    *text;
    XawTextPosition  startpos;
    char 	    *searchstring;
    XawTextPosition  *left, *right;
{
    int  searchlength, i, n=0;
    char *s1, *s2;

    searchlength = strlen(searchstring);
    for (i=startpos; i > searchlength; i--) {
	n = searchlength;
	s1 = &text[i];
	s2 = &searchstring[searchlength-1];
	while (--n >= 0 && *--s1 == *s2--);
	if (n < 0) break;
    }
    if (n < 0) {
    	*right = i;
    	*left = *right-searchlength;
    	return 1;
    }
    else {
	for (i=strlen(text)-1; i > startpos; i--) {
	    n = searchlength;
	    s1 = &text[i];
	    s2 = &searchstring[searchlength-1];
	    while (--n >= 0 && *--s1 == *s2--);
	    if (n < 0) break;
	}
	if (n < 0) {
            *right = i;
            *left = *right-searchlength;
	    return 1;
	}
	return 0;
    }
}

/* ARGSUSED */
/* use the active swindow */
static void PopupSearch(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    Arg 	args[MAXARGS];
    Cardinal 	n;
    Dimension	popup_width, dialog_width;
    Position	x, y;

    if (!a_swindow->displayedFile) {
	UpdateMessageWindow(SEARCH_HELP, NULL);
	bell(0);
    }
    else {
	XtRealizeWidget(searchPopupShell);
	n = 0;
	XtSetArg(args[n], XtNwidth, &popup_width);			n++;
	XtGetValues(searchPopupShell, args, n);
	n = 0;
	XtSetArg(args[n], XtNwidth, &dialog_width);			n++;
	XtGetValues(dialogWindow, args, n);
	XtTranslateCoords(dialogWindow, 
			  (Position)(dialog_width - popup_width)/2, 10, &x, &y);
	n = 0;
	XtSetArg(args[n], XtNx, x);					n++;
	XtSetArg(args[n], XtNy, y);					n++;
	XtSetValues(searchPopupShell, args, n);
	XtPopup(searchPopupShell, XtGrabNone);
    }
}


/* ARGSUSED */
/*  This routine handles both forward and reverse text search.
 *  If no text has been entered, the contents of the cut buffer are used
 *  for searching.
 */ 
/* use the active swindow */
static void Search(w, direction, call_data)
    Widget w;
    XtPointer direction;
    XtPointer call_data;
{
    XawTextBlock    	textblock;
    XawTextPosition	pos, left, right;
    char		*searchString;

    searchString = XawDialogGetValueString(searchPopup);
    if (strlen(searchString) == 0) {
	textblock.ptr = XFetchBytes(display, &textblock.length);
	if (!textblock.ptr) {
	    UpdateMessageWindow("No search string selected", NULL);
	    bell(0);
	    return;
	}
	searchString = textblock.ptr;
    }
    pos = XawTextGetInsertionPoint(a_swindow->sourceWindow);
    if ((direction == (XtPointer)FORWARD && 
	forwardSearch(a_swindow->displayedFile->buf, pos, searchString,
	 &left, &right)) ||
        (direction == (XtPointer)REVERSE && 
	reverseSearch(a_swindow->displayedFile->buf, pos, searchString,
	 &left, &right))) {
        AdjustText(TextPositionToLine(left,a_swindow), a_swindow);
        XawTextSetSelection(a_swindow->sourceWindow, left, right);
        XawTextSetInsertionPoint(a_swindow->sourceWindow, left);
    }
    else {
        if (direction == (XtPointer)FORWARD)
            UpdateMessageWindow("String not found", NULL);
        else if (direction == (XtPointer)REVERSE)
            UpdateMessageWindow("String not found", NULL);
        else
            UpdateMessageWindow("xdbx error: illegal search direction", NULL);
        bell(0);
    }
}

/* ARGSUSED */
static void DoneSearch(w, client_data, call_data)
    Widget w;
    XtPointer client_data;
    XtPointer call_data;
{
    XtPopdown(client_data);
}

/* ARGSUSED */
    
    
static void Activate(w, event, params, num_params)
    Widget w;
    XEvent *event;
    String *params;
    Cardinal *num_params;
{
    Search(w, (XtPointer)FORWARD, NULL);
    DoneSearch(w, (XtPointer)searchPopupShell, NULL);
}


static void CreateSearchPopup()
{
    Widget	dialogValue;
    Arg 	args[MAXARGS];
    Cardinal 	n;

    static XtActionsRec search_actions[] = {
        {"Activate", Activate},
        {NULL, NULL}
    };

    static String translations = "#override\n\
        <Key>Return:         Activate() \n\
    ";

    n = 0;
    XtSetArg(args[n], XtNinput, True);					n++;
    XtSetArg(args[n], XtNallowShellResize, True);			n++;
    searchPopupShell = XtCreatePopupShell("Search", transientShellWidgetClass, 
	toplevel, args, n);

    n = 0;
    XtSetArg(args[n], XtNlabel, "Enter search string :");		n++;
    XtSetArg(args[n], XtNvalue, SearchString);				n++;
    searchPopup = XtCreateManagedWidget("searchPopup", dialogWidgetClass, 
	searchPopupShell, args, n);
    
    AddButton(searchPopup, "<<", Search, (XtPointer) REVERSE);
    AddButton(searchPopup, ">>", Search, (XtPointer) FORWARD);
    AddButton(searchPopup, "DONE", DoneSearch, (XtPointer)searchPopupShell);

    dialogValue = XtNameToWidget(searchPopup, "value");
    XtOverrideTranslations(dialogValue, XtParseTranslationTable(translations));
    XtAppAddActions(app_context, search_actions, XtNumber(search_actions));
}


/*
 *  PopupSW(w, client_data, call_data)
 *
 *  Callback for creation of a new SWindow
 */
static void PopupSW(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    PopupSWindow(True,True,-1,-1);/* Activate new Window when popped up */
}



/*
 * char *FetchData(char *adr, long len, char *buffer)
 *
 * Copies a part of the debuggee's memory (starting at "adr", "len" bytes)
 * to the debugger and returns the address of the resulting buffer.
 * Creates new buffer if "buffer"==NULL, else uses "buffer".
 * First tries using a ptrace() based dump routine in the Debuggee,
 * then a file based one if ptrace() fails (e.g. debugger being debugged).
 * Returns NULL if failure (but leaves "buffer" intact if it exists).
 * Note: Result must be freed if no longer used (XtFree!)
 * Note: Needs the "DumpToFile()" function compiled into the target
 *       program (and the GimmeData() function, too)
 *
 */
char *FetchData(adr,len,buffer)
char *adr;
long len;
char *buffer;
{
    char   *reply;
    char    s[BUFSIZ];
    char    command[LINESIZ];		/* Buffer for dbx command	*/
    FILE   *dumpfile;			/* File with array data dump	*/
    char    dumpfilename[LINESIZ];	/* Name of array dump file	*/
    int     rlen;			/* # of bytes read		*/
    Boolean newbuffer;			/* True if self-made buffer	*/


    /* Call function in debuggee which puts the required data	*/
    /* directly into the debugger using ptrace()		*/

    if (buffer == NULL) {
	buffer    = (char*)XtMalloc(len+1);	/* Create buffer ourself*/
	newbuffer = TRUE;
    } else {
	newbuffer = FALSE;			/* Use existing buffer	*/
    };

    sprintf(s,"print GimmeData(%d,%d,%d,%d)\n",
	    getpid(),buffer,adr,len);

/*fprintf(stderr,"FetchData: called '%s'\n",s);*/
    reply = DirectlyQueryDbx(s);
    if (strstr(reply,"is not defined") == NULL) {
	if (strstr(reply,") = 1")) {
	
	    XtFree(reply);		/* Successful transfer, done	*/
	    return(buffer);
	};
    };
    XtFree(reply);

    
    /* Failure (GimmeData() not defined or not successful) 	*/
    /* -> try with transfer via file				*/
	
    
    /* Make dbx call the dump generation routine to put the array	*/
    /* data into a temporary file					*/
    
    sprintf(dumpfilename,"/tmp/dbg%d.%d",getpid(),adr);
    sprintf(command,"call DumpToFile(%d,%d,\"%s\")\n",
	    adr,len,dumpfilename);
    
/*fprintf(stderr,"FetchData: ptrace() failed - new command = '%s'\n",
command);*/
    
    reply = DirectlyQueryDbx(command);
    
    
    /* Check if error occured during dump */
    
    if (strstr(reply,"is not defined")) {

	UpdateMessageWindow("No dump function in program!",NULL);
	bell(0);
	XtFree(reply);
	return(NULL);
    };
    XtFree(reply);
    
    
    /* Load the array into a buffer of our own, removing the	*/
    /* temporary file afterwards				*/
    
    if (!(dumpfile=fopen(dumpfilename,"r"))) {
	UpdateMessageWindow("Can't open dump file %s !",dumpfilename);
	bell(0);
	return(NULL);
    };

    rlen = fread(buffer,1,len,dumpfile);
    if (rlen < len) {
	sprintf(s,"dump file %s length %d (should be %d) !",
		dumpfilename,rlen,len);
	UpdateMessageWindow(s,NULL);
	bell(0);
	fclose(dumpfile);
	unlink(dumpfilename);
	if (newbuffer)
	  XtFree(buffer);		/* Free buffer if self-allocated */
	return(NULL);
    };
    
    fclose(dumpfile);
    unlink(dumpfilename);
    
    return(buffer);			/* Success, return buffer	 */

}

    

/*
 * long GetAddress(char *varname)
 *
 * Asks the debugger about the address of the given variable
 *
 */
long GetAddress(varname)
char *varname;
{
    char    command[LINESIZ];		/* Buffer for dbx command	*/
    char   *reply;			/* Reply of dbx			*/
    char   *p;
    long    adr;
    
    
    /* Ask dbx about address */

    sprintf(command,"print &%s\n",varname);

    reply = DirectlyQueryDbx(command);
    
/*fprintf(stderr,"GetAddress: sent '%s', got '%s'\n",command,reply);*/


    if (strstr(reply,"is not defined")) {
	XtFree(reply);
	return(NULL);		/* Check if variable was defined at all */
    };

    if ( (p = (char*)strchr(reply,'=')) == NULL) {
	XtFree(reply);
	return(NULL);		/* Check where answer stands		*/
    };
    p++;

    adr = strtol(p,NULL,0);
    XtFree(reply);

    return(adr);
    
}

    

/*
 * char *GetValue(char *expr)
 *
 * Asks the debugger about the value of the given expression
 *
 */
char *GetValue(expr)
char *expr;
{
    char    command[LINESIZ];		/* Buffer for dbx command	*/
    char   *reply;			/* Reply of dbx			*/
    char   *p;
    char   *value;			/* Value (as string)		*/
    
    
    /* Ask dbx about address */

    sprintf(command,"print %s\n",expr);

    reply = DirectlyQueryDbx(command);
    
/*fprintf(stderr,"GetValue: sent '%s', got '%s'\n",command,reply);*/


    if (strstr(reply,"is not defined")) {
	XtFree(reply);
	return(NULL);		/* Check if variable was defined at all */
    };

    if ( (p = (char*)strchr(reply,'=')) == NULL) {
	XtFree(reply);
	return(NULL);		/* Check where answer stands		*/
    };
    p++;

    value=(char*)XtMalloc(strlen(p)+1);
    strcpy(value,p);
    XtFree(reply);
      
    return(value);
    
}

    


/*
 *  DisplayVisualizer(w, client_data, call_data)
 *
 *  Create a Visualization of array data
 *  Gets the address of the selected array from dbx (using an interface
 *  routine of our own), gets array information (how many dimensions,
 *  how large is each dimension etc) by asking dbx about them, then displays
 *  the array.
 */
static void DisplayVisualizer(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char   *arrayname;		/* Name of array to be visualized	*/
    long    arrayadr;		/* Address of array			*/
    int     arraytype;		/* Type of array (V_INT etc)		*/
    int     arraylen;		/* Length of array in bytes		*/
    char   *array;		/* Local copy of array			*/
    int     numdims;		/* Number of dimensions in array	*/
    int     dims[20];		/* Sizes of each dimension		*/
    int     nbytes;		/* Number of Bytes fetched from Xdbx	*/
    int     i;			/* Counter				*/
    char    s[LINESIZ];		/* Yet Another Buffer			*/
    char   *cname;		/* Name of Modula variable in C		*/
    char   *ctype;		/* Type of variable in C		*/
    char   *typelist;		/* Dbx reply for type			*/
    char   *lenlist;		/* Dbx reply for length			*/
    char   *innertype;		/* Type of array inside struct		*/
    Widget  VisVIS;		/* Widget for Visualizer		*/
    void   *cmp;		/* Comparison value for Visualizer	*/
    void   *from,*to;		/* Range values for Visualizer		*/


    /* Get name of array from X Windows (current selection) */

    arrayname = XFetchBytes(display, &nbytes);	/* Get name of array	*/
    if (nbytes == 0) {				/* Complain if none	*/
	UpdateMessageWindow(VIS_HELP, NULL);
	bell(0);
	return;
    };


    /* Check if we have a Visualizer left		*/
    
    if (visnum+1 >= MAX_VISNUM) {
	UpdateMessageWindow("No more Visualizers allowed!",NULL);
	bell(0);
	return;
    };
    
    /* If in Modula source, look for corresponding C variable and	*/
    /* get the type of the variable from the information in the source	*/
    
    if ((a_swindow->SWType == SWTYPE_MODULA) && (a_swindow->displayedFile)) {
	char *p,*q,*r;
	char  c;
	
	ctype = NULL;
	cname = NULL;
	i=0;
	while ((ctype == NULL) &&	/* Look through var entries	*/
	       (i < a_swindow->displayedFile->varinfosnum)) {
	    char *mn = a_swindow->displayedFile->varinfos[i]->MName;
	    
	    if ((strlen(arrayname)>=strlen(mn)) &&
		(strncmp(mn,arrayname,strlen(mn))==0)) {/* Found name?	   */
		    cname = a_swindow->displayedFile->	/* Remember C name */
		             varinfos[i]->CName;	/*  and c type	   */
		    ctype = a_swindow->displayedFile->
		             varinfos[i]->CType;
	    };
	    i++;
	};

	if ((ctype==NULL) || (cname==NULL)) {
	    UpdateMessageWindow("'%s' not found in Modula names!",arrayname);
	    bell(0);
	    return;
	};

	/* Ask the debugger what's behind the C Type (e.g. "stuct xx" etc) */
	
	sprintf(s,"whatis %s\n",ctype);
	typelist = DirectlyQueryDbx(s);
/*fprintf(stderr,"typelist = '%s'\n",typelist);*/
	if (strstr(typelist,"not defined") != NULL) {
	    UpdateMessageWindow("type '%s' not defined!",ctype);
	    bell(0);
	    XtFree(typelist);
	    return;
	};

	p = strchr(typelist,'[');	/* Look for array brackets	*/
	if (p==NULL) {
	    sprintf(s,"'%s' (type %s) is no array!",arrayname,ctype);
	    UpdateMessageWindow(s,NULL);
	    bell(0);
	    XtFree(typelist);
	    return;
	};
	q=p+1;				/* Remember 1st array index pos	*/
	while ((*p != ' ') && (*p != '\t') && (p>typelist))
	  p--;				/* skip inner name		*/
	while (((*p == ' ') || (*p == '\t')) && (p>typelist))
	  p--;				/* skip spaces			*/
	r=p+1;				/* Remember end of type		*/
	while ((*p != ' ') && (*p != '\t') && (p>typelist))
	  p--;				/* look for type name		*/
	p++;
	c=*r;
	*r='\0';
	innertype=strdup(p);		/* Remember name of inner type	*/
	*r=c;
	
/*fprintf(stderr,"inner type '%s'\n",innertype);*/

	/* Get type of array from inner type name			*/
	/* Also initialize comparison values & ranges for Visualizers	*/
	
	if ((strlen(innertype)>=4) &&
	    (strncmp("char",innertype,4)==0)) {		/* char		*/
	    arraytype = V_CHAR;
	    cmp  = (char*)XtMalloc(sizeof(char));
	    from = (char*)XtMalloc(sizeof(char));
	    to   = (char*)XtMalloc(sizeof(char));
	    *(char*)cmp=0;
	    *(char*)from=0;
	    *(char*)to=10;
	} else if ((strlen(innertype)>=3) &&		/* int		*/
	    (strncmp("int",innertype,3)==0)) {
	    cmp  = (int*)XtMalloc(sizeof(int));
	    from = (int*)XtMalloc(sizeof(int));
	    to   = (int*)XtMalloc(sizeof(int));
	    *(int*)cmp=0;
	    *(int*)from=0;
	    *(int*)to=10;
	    arraytype = V_INT;
	} else if ((strlen(innertype)>=4) &&		/* long		*/
	    (strncmp("long",innertype,4)==0)) {
	    cmp  = (long*)XtMalloc(sizeof(long));
	    from = (long*)XtMalloc(sizeof(long));
	    to   = (long*)XtMalloc(sizeof(long));
	    *(long*)cmp=0;
	    *(long*)from=0;
	    *(long*)to=10;
	    arraytype = V_LONG;
	} else if ((strlen(innertype)>=5) &&		/* float	*/
	    (strncmp("float",innertype,5)==0)) {
	    cmp  = (float*)XtMalloc(sizeof(float));
	    from = (float*)XtMalloc(sizeof(float));
	    to   = (float*)XtMalloc(sizeof(float));
	    *(float*)cmp=0.0;
	    *(float*)from=0.0;
	    *(float*)to=10.0;
	    arraytype = V_FLOAT;
	} else if ((strlen(innertype)>=6) &&
	    (strncmp("double",innertype,6)==0)) {	/* double	*/
	    cmp  = (double*)XtMalloc(sizeof(double));
	    from = (double*)XtMalloc(sizeof(double));
	    to   = (double*)XtMalloc(sizeof(double));
	    *(double*)cmp=0.0;
	    *(double*)from=0.0;
	    *(double*)to=10.0;
	    arraytype = V_DOUBLE;
	} else {
	    sprintf(s,"unknown inner type '%s' for array %s",
		    innertype,arrayname);
	    UpdateMessageWindow(s,NULL);
	    bell(0);
	    XtFree(typelist);
	    free(innertype);
	    return;
	};
/*fprintf(stderr,"-> type %d \n",arraytype);*/
	

	numdims=0;			/* Remember 1st array index pos	*/
	r=q;
	while (r) {
	    p=r;			/* Look for dimension		*/
	    while ((*p) && (*p!=']'))
	      p++;
	    if (*p) {			/* Found end of bracket -> get size */
		c=*p;
		*p='\0';
		dims[numdims]=atoi(q);
/*fprintf(stderr,"dims[%d] = '%s' = %d\n",numdims,q,dims[numdims]);*/
		numdims++;
		*p=c;
		p++;
		r=strchr(p,'[');	/* Look for next bracket/dimension */
	    } else {
		r=NULL;
	    };
	};
	
    };

    /* Determine array address (in debuggee) and size,	*/
    /* then fetch the whole array			*/

    arrayadr = GetAddress(cname);
    if (arrayadr==0) {
	UpdateMessageWindow("Couldn't get address of '%s'!",cname);
	bell(0);
	return;
    };
    sprintf(s,"sizeof(%s)",ctype);
    lenlist = GetValue(s);
    if (lenlist==NULL) {
	UpdateMessageWindow("Couldn't get size of '%s'!",cname);
	bell(0);
	return;
    };
    arraylen = atoi(lenlist);
/*fprintf(stderr,"DisplayVisualizer: adr = %d, size -> '%s' = %d\n",arrayadr,lenlist,arraylen);*/
    XtFree(lenlist);
    
    array = FetchData((char*)arrayadr,arraylen,NULL); /* Get new buffer */

    
    if (array == NULL) {
        UpdateMessageWindow("Error fetching array '%s' !",arrayname);
        bell(0);
        return;
    };


    /* Start Visualizer using normal interface if button 1,	*/
    /* use ranges Visualizer when button 2 and			*/
    /* use Grayscale Display if right button pressed		*/

    if (Button == 1) {

	VisVIS = VisWidget(toplevel,arrayname,0,(void*)array,
			   dims,numdims,arraytype, V_GREATER,cmp);

    } else if (Button == 2) {

	VisVIS = VisRangeWidget(toplevel,arrayname,0,(void*)array,
                       dims,numdims,arraytype, from, to);

    } else {

        VisVIS = VisValuesWidget(toplevel,arrayname,0,(void*)array,
                        dims,numdims,arraytype);
    };

    /* Enter the Visualizer in the list of active Visualizers */

    vislist[visnum].vis  = VisVIS;
    vislist[visnum].name = arrayname;
    vislist[visnum].adr  = arrayadr;
    vislist[visnum].len  = arraylen;
    vislist[visnum].array= array;
    visnum++;
    
}


/*
 *  DisplayTree(w, client_data, call_data)
 *
 *  Callback for Display of dynamic call tree.
 *  Calls a function in the debugged program using dbx, gets as a result
 *  the debug info tree as string, re-converts it to a new tree and
 *  displays this tree using a popup window
 *  Note: treeW is the global tree widget! DTree->RootNode is global
 *        debugnode root
 *  Deallocates the old tree and builds a new one (there IS an old tree
 *  because DebugInitialize creates one!)
 *
 */
static void DisplayTree(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char	*treelist;	/* Debug Tree as string			*/


    if (tree_active > 0) {		/* Not more than one tree at once */
        UpdateMessageWindow("Tree Structure already being shown",NULL);
        bell(0);		/* Note: number of trees is updated in..  */
        return;			/* .. the ShowTreeWidget / Popdown calls! */
    };

    
    /* Check if we're inside a DebugFn (indicating that the tree might	*/
    /* be in an inconsistent state), refuse to show it if yes		*/
    
    if ((c_swindow->displayedFile) &&
	(strstr(c_swindow->displayedFile->filename,
		DEBUGFNSOURCE) != NULL)) {
	UpdateMessageWindow("Inside tree manipulation - try again later!",
			    NULL);	
	bell(0);
	return;
    };

    
    /* Testing the Checksum of the Tree here isn't useful because here	*/
    /* we do the 1st popup etc anyway so we have to build a tree no	*/
    /* matter what the checksum is					*/

    /* Get the whole tree as list and re-convert it to a tree	*/
    /* (Necessary because seperate address spaces!)		*/
    /* Complain if not defined, i.e. no such information found  */

    treelist = DirectlyQueryDbx("call PrintTreeAsString(DTree,0)\n");
    if (strstr(treelist,"not defined") != NULL) {
        UpdateMessageWindow(TREE_HELP,NULL);
        bell(0);
        return;
    };
    KillTree(DTree);				/* Deallocate old tree	*/
    DTree = BuildTreeFromString(treelist,0);	/* Build new one	*/


    /* Display the results in a Widget */

    treeW = ShowTreeWidget(toplevel);		/* (takes global DTree!	*/


    if (treelist)		/* Free resources again		*/
        XtFree(treelist);        
}


/*
 *  DisplayProfile(w, client_data, call_data)
 *
 *  Callback for Display of profile information.
 *  Gets the ProfileTable by calling the dump function in the debugged
 *  program and reassembling the table within xdbx. Displays it using
 *  a Widget.
 *  Note: profileW is the gloabl profile info widget!
 *  Also clears the old profile before finning in new information
 *
 */
static void DisplayProfile(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char	*profilelist;	/* dbx output containing profile as list*/


    if (profile_active > 0) {		/* Not more than one profile at once */
        UpdateMessageWindow("Profile already being shown",NULL);
        bell(0);		/* Note: number of profiles is updated in..  */
        return;			/* .. the ShowProfileWidget / Popdown calls! */
    };


    /* Get the whole profile as list and re-convert it to a	  */
    /* profile table (Necessary because seperate address spaces!) */
    /* Complain if not defined, i.e. no such information found    */

    profilelist = DirectlyQueryDbx("call PrintProfileAsString(PTable)\n");
    if (strstr(profilelist,"not defined") != NULL) {
        UpdateMessageWindow(PROFILE_HELP,NULL);
        bell(0);
        return;
    };
    ClearProfileTable(PTable);			/* Clear old entries	*/
    AddStringToProfile(PTable,profilelist,1);	/* Fill in new profile	*/


    /* Display the results in a Widget */

    profileW = ShowProfileWidget(toplevel);


    if (profilelist)		/* Free resources */
        XtFree(profilelist);        

}


/*
 *  DisplayActivations(w, client_data, call_data)
 *
 *  Callback for Display of activation value information.
 *  Gets the current activation value array & active value by looking at
 *  the most recent node of the tree.
 *
 */
static void DisplayActivations(w, client_data, call_data)
Widget w;
XtPointer client_data;
XtPointer call_data;
{
    char	*actlist;	/* dbx output containing activations	*/
    char	 s[512];	/* General purpose string		*/
    DebugNode	*dnode;		/* Most recent node			*/
    long	 asz;		/* Size for visualizer (act_size+1)	*/


    if ((actVIS!=NULL) && (actVIS->VISvis!=NULL)) {	/* Not more than one */
        UpdateMessageWindow("Activation already being shown",NULL);
        bell(0);					 /* Act. at once     */
        return;
    };
    if (actVIS!=NULL) {			/* Free remains of old ActVis	*/
	XtFree(actVIS);
	actVIS = NULL;
    };
    if (act_buffer != NULL) {		/* Free old act.buffer if one	*/
	XtFree(act_buffer);
	act_buffer = NULL;
    };


    /* Check context of most recent node - must be FORALL IN SYNC/PAR ! */
    
    actlist = DirectlyQueryDbx("call PrintNodeAsString(DTree->LastNode)\n");
    if (strstr(actlist,"is not defined")) {
	XtFree(actlist);
	UpdateMessageWindow("No debug functions defined!",NULL);
	bell(0);
	return;
    };
    
    dnode = BuildNodeFromString(actlist);
    XtFree(actlist);

    /* Collect information for popup */

    act_index=0;			/* Get value of act.index	*/
    if ((long)dnode->RangeVar) {
	sprintf(s,"*(long*)(%d)",(long)dnode->RangeVar);
	actlist = GetValue(s);
	if (actlist) {
	    act_index = atol(actlist);
	    XtFree(actlist);
	};
    };
    act_low	= (long)dnode->RangeLo;
    act_high	= (long)dnode->RangeHi;
    act_size    = act_high - act_low + 1;
    act_context = dnode->Context;


    /* Sync context -> fetch data & display it				*/
    /* Par(ascync)  -> create array with just the current index active	*/
    /* Else         -> complain & return				*/

    if (act_context == DN_SYN_CONTEXT) {
    
	act_compare = dnode->ActValue;
	act_address = (long)dnode->ActArray;	/* Fetch data into new buffer*/
	act_buffer  = (long*)FetchData((char*)act_address,
				       act_size*sizeof(long),
				       NULL);
	
    } else if (act_context == DN_PAR_CONTEXT) {

	act_compare = 0;			/* Create buffer & set	*/
	act_address = 0;			/*  the index as active	*/
	act_buffer  = (long*)XtMalloc(act_size*sizeof(long));
	memset(act_buffer,1,act_size*sizeof(long));
	if ((act_index>=act_low) && (act_index<=act_high))
	 memset(act_buffer+act_index-act_low,0,
                 (act_high-act_index+1)*sizeof(long));

	/*  act_buffer[act_index]=1;	/*!!!!!!! check for offset? */
	
    } else {
	free(dnode);
	UpdateMessageWindow("Not inside FORALL!",NULL);
	bell(0);
	return;
    };
    free(dnode);
    
    
    if (act_buffer == NULL) {		/* Problem getting array, return */
	UpdateMessageWindow("Problem getting activation values!",NULL);
	bell(0);
	return;
    };
    
    
    /* Create activation Visualizer, prepare info popup */

    asz = act_size + 1;			/* Number of elements (0..n-1) */
    actVIS = VisActivationWidget(toplevel,"activation",0,
				 act_buffer,
				 &act_size,
				 &act_compare,
				 ActInfoPopup);
    
}



/*
 * void ActInfoQuit()
 *
 * Callback for ActInfo Popup - destroy the popup window again
 *
 */
void ActInfoQuit(w,popupW,dummy)
Widget w;
Widget popupW;
XtPointer dummy;
{
    XtPopdown(popupW);			/* Pop the widget down..	*/
    XtDestroyWidget(popupW);		/* ..and destroy it		*/
}



/*
 * void ActInfoPopup()
 *
 * Popup a window showing information about the selected
 * activation arrayelement
 *
 */
void ActInfoPopup(VIS,pos)
VIStype *VIS;
long pos;
{
    char        s[256];
    Arg         a[10];		/* Arguments for Widget creation	*/
    int         n;
    Widget      popupW;		/* Popup Widget to appear		*/
    Widget      clickW;		/* Quit button				*/
    Position    x,y;		/* To obtain coordinates for popup	*/
    XtWidgetGeometry prefgeo;	/* To resize it to sensible size	*/
#define ACT_X 100
#define ACT_Y  70		/* Approx. size of popup		*/
    
    
    /* Position the info widget above the parent */

    {
	Window          tmp_r, tmp_c;
	int             rx, ry, wx, wy;
	unsigned int    mask;
	    
	XQueryPointer(XtDisplay(VIS->VISparent), XtWindow(VIS->VISparent),
		     &tmp_r, &tmp_c, &rx, &ry, &wx, &wy, &mask);
	wx = wx - (ACT_X/2);
	wy = wy - (ACT_Y/2);
	XtTranslateCoords(VIS->VISparent,
			  (Position)wx,(Position)wy,
			  &x,&y);
    };

    
    
    /* Generate information popup window using Visualier's parent entry */
    /* as parent Widget                                                 */
    
    n=0;
    XtSetArg(a[n], XtNx, x); n++;
    XtSetArg(a[n], XtNy, y); n++;
    popupW = XtCreatePopupShell("ActInfoPopup",transientShellWidgetClass,
				VIS->VISparent,a,n);
    n=0;
    sprintf(s,"Virtual PE %d\non real CPU 1\nrange [%d..%d]\ncur.index = %d\n(click to exit)",
	    pos,act_low,act_high,act_index);
    XtSetArg(a[n], XtNlabel, s); n++;
    clickW = XtCreateManagedWidget("ActInfo",commandWidgetClass,
				   popupW,a,n);
    XtAddCallback(clickW, XtNcallback, ActInfoQuit, popupW);
    
    
    
    /* Pop up the Info Widget and resize it to preferred geometry	*/
    /* (xdbx app-defaults tend to make buttons too small)		*/
    
    XtPopup(popupW,XtGrabExclusive);

    XtQueryGeometry(clickW,NULL,&prefgeo);
    XtResizeWidget(popupW,prefgeo.width,prefgeo.height,
		   prefgeo.border_width);
    
}


static Widget AddButton(parent, name, function, client_data)
Widget parent;
char *name;
void (*function) ();
XtPointer client_data;		/* callback registered data */
{
    Widget 	button;
    Arg 	args[MAXARGS];
    Cardinal 	n;

    static XtActionsRec command_actions[] = {
	{"ButtonSet", (XtActionProc) ButtonSet},
        {NULL, NULL}
    };

    static String translations = "\
	<EnterWindow>:	highlight() \n\
	<LeaveWindow>:	reset() \n\
	<Btn1Down>:	set()\n\
	<Btn1Up>:	ButtonSet(1) notify() unset() \n\
	<Btn2Down>:	set()\n\
	<Btn2Up>:	ButtonSet(2) notify() unset() \n\
	<Btn3Down>:	set()\n\
	<Btn3Up>:	ButtonSet(3) notify() unset()\n\
    ";

    n = 0;
    XtSetArg(args[n], XtNresize, (XtArgVal) False);			n++;
    if (strcmp(name, "print")      == NULL ||	/* Add extra button 3..	*/
        strcmp(name, "print *")    == NULL ||	/* ..recognition	*/
        strcmp(name, "visualize")  == NULL ||
        strcmp(name, "activations")== NULL) {
	XtSetArg(args[n], XtNtranslations, 
	    XtParseTranslationTable(translations)); 			n++;
    }
    button = XtCreateManagedWidget(name, commandWidgetClass, parent, args, n);
    XtAddCallback(button, XtNcallback, function, client_data);
    XtAppAddActions(app_context, command_actions, XtNumber(command_actions));
    return (button);
}


static void CreateButtons (parent)
Widget parent;
{
    int i=0;

    button[i++] = AddButton (parent, "run", DoIt, "run\n");
    button[i++] = AddButton (parent, "cont", DoIt, "cont\n");
    button[i++] = AddButton (parent, "next", DoItNext, NULL);
    button[i++] = AddButton (parent, "step", DoIt, "step\n");
#ifdef BSD
    button[i++] = AddButton (parent, "return", Return, "return\n");
#endif
    button[i++] = AddButton (parent, "stop at", Stop_at, NULL);
    button[i++] = AddButton (parent, "stop in", Stop_in, NULL);
    button[i++] = AddButton (parent, "delete", Delete, NULL);
    button[i++] = AddButton (parent, "where", DoIt, "where\n");
/*    button[i++] = AddButton (parent, "up", DoIt, "up\n");*/
/*    button[i++] = AddButton (parent, "down", DoIt, "down\n");*/
    button[i++] = AddButton (parent, "print", Print, (XtPointer)0);
    button[i++] = AddButton (parent, "print *", Print, (XtPointer)1);
/*    button[i++] = AddButton (parent, "func", Func, NULL);*/
/*    button[i++] = AddButton (parent, "file", File, NULL);*/
    button[i++] = AddButton (parent, "status", DoIt, "status\n");
#ifndef BSD
    button[i++] = AddButton (parent, "display", Display_, NULL);
    button[i++] = AddButton (parent, "undisplay", Undisplay, NULL);
#endif
    button[i++] = AddButton (parent, "dump", Dump, NULL);
    button[i++] = AddButton (parent, "search", PopupSearch, NULL);
/*    button[i++] = AddButton (parent, "popup",
                             PopupSW, NULL);		/* popup an SWindow */
    button[i++] = AddButton (parent, "visualize",
                             DisplayVisualizer, NULL);	/* visualize array  */
    button[i++] = AddButton (parent, "tree view",
                             DisplayTree, NULL);	/* show call tree   */
    button[i++] = AddButton (parent, "profile",
                             DisplayProfile, NULL);	/* show profile info*/
    button[i++] = AddButton (parent, "activations",
                             DisplayActivations, NULL);	/* show activations */
    button[i++] = AddButton (parent, "quit", Quit, NULL);
    button[i++] = NULL;
    CreateSearchPopup();
}


/*  Create a command widget, and the buttons.  */

void CreateCommandPanel(parent)
Widget parent;
{
    Arg args[10];
    Cardinal n;

    n = 0;
    commandWindow = XtCreateManagedWidget("commandWindow", boxWidgetClass, 
					  parent, args, n);
    CreateButtons(commandWindow);
    getwd(cwd);
}

/**************************************************************************
 *
 *  Command queue functions
 *
 **************************************************************************/

/*  Append command to end of the command queue and send the command to dbx */

void send_command(command)
char *command;
{
    CommandRec *p, *q, *r;

#ifdef BSD 
    /* Save the command if it is not a blank command; else use the 
       last saved command instead */
    if (strcspn(command, " \n"))
	strcpy(savedCommand, command);
    else
	strcpy(command, savedCommand);
#endif

    p = (CommandRec *)XtNew(CommandRec);
    p->command = XtNewString(command);
    p->next = NULL;
    if (!commandQueue)
	commandQueue = p;
    else {
	q = commandQueue;
	while (r = q->next)
	    q = r;
	q->next = p;
    }
    write_dbx(command);
}

/*  Read command at the head of the command queue */

char *get_command()
{
    if (commandQueue) {
	return (commandQueue->command);
    }
    else
	return NULL;
}

/*  Delete command from the head of the command queue */

void delete_command()
{
    CommandRec *p;

    if (p = commandQueue) {
	commandQueue = p->next;
	XtFree(p->command);
	XtFree(p);
    }
}

/*  Insert command into head of queue */

void insert_command(command)
char *command;
{
    CommandRec *p;

    p = (CommandRec *)XtNew(CommandRec);
    p->command = XtNewString(command);
    p->next = NULL;
    if (!commandQueue)
	commandQueue = p;
    else {
	p->next = commandQueue;
	commandQueue = p;
    }
}


