/*
 * visual.c
 *
 * Multi dimensional Array Visualization
 *
 *
 * Stefan Haenssgen 28-Nov-91
 *
 * Updates :	28-Nov-91  Basic Implementation
 *			   "Quick Hack" adapted from testa.c
 *		20-Dec-91  Debugging and added functionality
 *		28-Dec-91  Complete structured redesign
 *			   Split visual.c into independant part (visual.c)
 *			    and test program (testv.c)
 *			   Use ADT "visualizer" instead of global variables
 *			   Added array "SizeOfType[]" for type sizes
 *			   Wrote macros to access an element of a multi-
 *			    dimensional array
 *			   Wrote access functions for each element type as
 *			    better way of accessing it all cleanly
 *			   Convey visualizer information via newly created
 *			    variable "userdata" in display structure
 *			   Debugging (offsets, display <-> values mapping etc)
 *			   Added function "update_shown_data" to show actual
 *			    positon & value even when mouse is not in array
 *			    (via counter settings)
 *			   Added comparison operators (less,equal,greater etc)
 *			    and functions to implement them (compare_xx)
 *			   Introduced function "update_shown_condition" to
 *			    support the new features
 *		30-Dec-91  Further enhancement: User editable selection
 *			    criteria via Widget "visselect" and Subwidgets
 *			    "vs_{label,gtbtn,eqbtn,ltbtn,value,type}"
 *			   Added Array "NameOfType" for names of types
 *			   Copied action procedure for text entry handling
 *			    from counter.c
 *			   Wrote Callback procedures "V{gt,eq,lt}ButtonCall"
 *			    and "VvalueCall" to handle the comparison entry
 *			    events
 *			   Added data "vactive{1,2}" for easier handling of
 *			    redisplay (no more search for active layers)
 *			   Reformatted the code for several functions,
 *			    added comments
 *			   Ensured that dimensions are always displayed
 *			    in sorted order (e.g. (x1,x3) not (x3,x1) )
 *		09-Jan-92  Changed interface:
 *			   Just ONE call, build our own parent widget,
 *			    no more separation of initialization and
 *			    execution!
 *			   -> removed "{init,do}_visualization", changed to
 *			    "visualize", unified the formerly seperated
 *			    functions
 *		10-Jan-92  Added generic comparison function "vcmpfn()" and
 *			    array access function "vaccfn()"
 *			   Changed call of compare functions to use just
 *			    pointers to elements
 *			   Added new interface "visualize_userfn()"
 *			   Changed "visualize()" to use this new interface
 *			   Added "do_visualization()" to do the final init-
 *			    ialization (common to both visualizer calls)
 *			   Debugged a bit...
 *			   Added an EventLoop of its own instead of
 *			    XtAppMainLoop()
 *			   Introduced "open_visualization()" and "close_v..()"
 *			    to get a common application context, using the
 *			    new "VIStype" struct (main parent widget).
 *		14-Jan-92  Call "destroy_{counter,display}()" on exit
 *		16-Jan-92  Renamed functions for easier Modula interfacing:
 *				open_visualization()-> OpenVisualizer()
 *				visualize()	   -> Visualize()
 *				visualize_userfn() -> VisualizeUserFn()
 *				close_visualizer   -> CloseVisualizer()
 *		19-Jan-92  Added Widgets for "from-to range": vftomto, vft_*
 *			   Added Values for range: vfromval, vtoval
 *			   Wrote new call function "VisualizeRange()"
 *			   Wrote new range comparison functions
 *			    "compare_range_<type>" with a trick (use cmp-value
 *			    as pointer to vis struct)
 *			   Modified initialization & condition display to
 *			    support the new features
 *		28-Jan-92  Added sockets and the possibility to run the
 *			    visualizer in the background:
 *			   Introduced new function UpdateForkedVisualizer()
 *			    to give new data to a subprocess using sockets
 *			   Implemented the initialization of the new data
 *			    in the VIS struct (VISvis and vforked,vsocketname
 *			    and vwritesocket)
 *			   Added error checking (already forked, not init'ed..)
 *			    to calls
 *			   Added forkflag to calls
 *			   Wrote setup_{read,write}() for socket initialization
 *			   Added handle_sockets() for data reading
 *		29-Jan-92  Debugging
 *			   Added killing of child & cleaning up to CloseVis.()
 *			   Made child call exit() instead of returning (argh!)
 *		30-Jan-92  Implemented UpdateForkedVisualizer() using a simple
 *			    protocol (magic number, length of data, data)
 *			   Implemented handle_sockets() to read the given data
 *			   Added timeout to selection in handle_socket()
 *			    (thus making it work again ;-) -> polling, no block
 *			   Fixed a bug in the accept routine that caused the
 *			    read socket to be set to 0 (argh!)
 *			   Tested the interaction in UpdateForkedVisualizer()
 *			    and disabled the magic-check
 *			   Added a timeout routine to ensure that handle_socket
 *			    is called at least once a second (VTimestep())
 *		31-Jan-92  Unmap main widget when Quit()ting (looks better)
 *		04-Feb-92  Use  Vis.h  and  VisP.h  (instead of visual.h)
 *			   Added typecasts to VIS->VISvis (now void* !)
 *			   Use vis->vreadoffset for better socket reading
 *			    (wait until data complete before displaying it)
 *		05-Feb-92  Utilize new button callback from array display:
 *			   Added VButtonClick() to handle click
 *			   Added new sockets for handling commands
 *			    and renamed existing ones. New names:
 *			    v{data,cmd}{bind,read,write}socket,
 *			    v{data,cmd}socketname
 *			   Introduced command sockets and their initialization
 *			   Updated all socket related functions to support the
 *			    command socket, too
 *			   Changed the debugging output to reflect parent/
 *			    child relationship (parent marked "++", child "**")
 *			   Implemented receiving of command data (format
 *			    described in the comments to handle_socket)
 *			   Added function ChangeUserFn() to supply new data
 *			    to the command socket
 *		06-Feb-92  Changed definitions of forkflag to Boolean,
 *			    added syncflag for determining if full synchroni-
 *			    zation wanted (i.e. wait for user click on sync-
 *			    button in Visualizer before going on with the
 *			    foreground work after an UpdateForkedVisualzier)
 *			   Added synchronization to UpdateForkedVisualizer(),
 *			    ChangeUserFn() [receive sync] and VClickButton()
 *			    [send sync if waiting]
 *			   Do additional cleanup when closing the Visualzier
 *			    (close sockets, wait for child termination etc)
 *			   Made socket names more unique by also using the
 *			    time (in usec) in the name
 *			   Output more information (VIS address etc)
 *			   Introduced venvp and venvlen for user environment
 *			    and -length for user defined compare function
 *			   Changed the parameters of VisualizeUserFn()
 *			    to include these
 *			   Added code to handle the modified environment data
 *		08-Feb-92  Use coordinates in OpenVisualizer() to avoid
 *			    user positioning of Visualizer Windows (if wanted)
 *			    (semi dirty trick using variable X Resources)
 *			   Introduced handling of one-dimensional arrays
 *			    by displaying them as nearly-square 2d-array
 *			   Introduced vonewidth for the length of one side
 *			    of this new 2d-array 
 *			   Updated Access macros and access function calls
 *			    etc to support one-dimensional arrays, too
 *		09-Feb-92  Show name and dimensions of array in window title
 *		10-Feb-92  Added vsyncchfn, renamed vsynced to vsyncupdate
 *			    (select synchronization for update / function
 *			    change seperately)
 *			   Changed calls accordingly
 *		16-Apr-92  Introduced different display modes (b&w, grayscale,
 *			    color) based on new display.c
 *			   Use b&w as default with the established Visualizers
 *			   Use grayscale with the new VisualizeValues() call
 *			   Added statistics (min,max,average) for value-Vis.
 *			   Put mapping of values to grayscale in special
 *			    compare functions value_{char,int,long}
 *		24-May-92  Set initial zoom to 6 (looks better with gray!)
 *			   Recalculate statistics if Value Visualizer used
 *			    (introduced do_statistics() )
 *			   Use Product(i=0,numdims-1,dims[i]+1) instead
 *			    of (...,...,dims[i]) only! Results in complete
 *			    transmission of array data, nothing left over!
 *			    (hehe, gotcha you old bug :-)
 *		01-Jul-92  Added computation of data variation and used it
 *			    to eliminate values that are far out of range
 *			    (which would distort the grayscale mapping)
 *			   Rewrote average calculation etc for more speed
 *			   Use variance to map out distorting values
 *			    (median +/- 2*variance covers 95%)
 *		02-Jul-92  Enclosed output of debug-info in #ifdef DEBUGINFO
 *		03-Jul-92  Added V_FLOAT and V_DOUBLE
 *			   Added floating point support:
 *			   - wrote coed to handle stuff (in places where
 *			      already V_INT etc were used)
 *			   - new functions {access,compare,compare_range,value}
 *			      _{float,double}()
 *			   - modified update_shown_data() to also accept floats
 *			   Made debug code nicer to look at by using a #define
 *			   Changed SIGQUIT to SIGKILL (we want no coredump!)
 *			   Added string length to each string in the call
 *			    interface for Modula-2* compatibility
 *		07-Jul-92  Changed numdims to unsigned long for M-2*
 *			   dito with envlen and access-fn's (numdims)
 *		22-Jul-92  Improved CloseVisualier: Better event handling,
 *			    wait until parent gets DestroyWindowEvent
 *			   Process remaining events for child process
 *			    only if it existed when we tried to kill it,
 *			    otherwise ignore such events (the user has
 *			    quit the child visualizer himself)
 *		27-Aug-92  Added VisualizeActivation() for special visuali-
 *			    zation of activation value arrays: call function
 *			    on mouse click, show as square, no data entry etc.
 *			   Rely on newly added set_xrest() and ACT_DISPLAY
 *			    from display module
 *			   Wrote documentation for VisualizeActivation()
 *			   Modified code to skip counter useage if Act.Display
 *			   Use venvp instead to keep track of the position
 *			   Call callfn of Activation Display in VMouseClick()
 *			    if it's defined
 *			   Check for overflow of position (last line stuffed!)
 *			   Include pointer to VIStype in "vVIS"
 *			   Some debugging of corners
 *		28-Aug-92  Last part (xrest) can be longer than one line!
 *			   Therefore, changed it to "restl" giving the length
 *			    of the array which is OK to access
 *		24-Oct-92  Better splitting of 1-dim arrays: given [0..n-1],
 *			    start with x=sqrt(n), y=n/x until x*y=n
 *			    then use [0..x-1] and [0..y-1] as 2d-ranges
 *			   Set vis->vonewidth = n if range [0..n-1] and 1dim!
 *			   Unified calls - always size n for [0..n-1] !
 *			   Also adapted handling of vis->vonwidth
 *			   Also fixed access for >= 3 dimensions
 *		02-Nov-92  Automagically select pixelsize for Visualizer
 *			    based on dimensions of array, introduced
 *			    screen[xy] and maxdim
 *		24-Nov-92  New concept: Make Visualizer available as Widget!
 *			    No background, no sync update, no extra AppLoop,
 *			    simply useable from programs which already have
 *			    an X environment of their own
 *			   Added internal functions PrepareVis{Range,...}()
 *			    for initialization of specific variables (code
 *			    moved there from Visualize{Range,...}), packed
 *			    common calls into PrepareCommon()
 *			    [Some Visualizers need their own access/compare
 *			    functions, so they overwrite the P.C.() defaults]
 *			   Moved all widget initialization stuff from
 *			    do_visualization() into newly created
 *			    do_vis_widgets()
 *			   Added new functions VisRangeWidget() etc to
 *			    supply programmer with Visualizer Widgets
 *			   Use quit function VQuitWidget() as callback
 *			    if widget-visualizer used. Deallocates whole
 *			    Visualizer.
 *			   Create own popup window for Visualizer as Widget
 *			   Wrote UpdateVisWidget() to update shown display
 *			    if data have changed
 *		25-Nov-92  Disable grips for Visualizer Widgets
 *		01-Dec-92  Set title "Visualizer" for Vis.Widgets
 *		02-Sep-93  Also set new value when leaving entry widget
 *
 *
 */


/* Include private and public definitions */

#include "VisP.h"
#include "Vis.h"


/* Un/Define output of debugging info */

#ifdef DEBUGINFO
#  define debuginfo(x) printf(x)
#else
#  define debuginfo(x)
#endif


/* Callbacks */

void	VQuit();		/* Guess				*/
void	VQuitWidget();		/* Quit callback for Visualizer Widgets	*/
void	VShowPos();		/* Show current mouse position in array */
void	VLeavePos();		/* Cancel shown position as mouse leaves*/
void	VMouseClick();		/* Mouse button clicked in array	*/
void	VButtonClick();		/* Display button clicked		*/
void	VChangedValue();	/* Value of counter changed		*/
void	VChangedState();	/* State of counter changed		*/
void	VgtButtonCall();	/* Buttons for comparison entry: ">"	*/
void	VeqButtonCall();	/*                               "="	*/
void	VltButtonCall();	/*                               "<"	*/
void	VvalueCall();		/* Comparison value changed (ActionProc)*/
void	VfromCall();		/* Range "from" changed (ActionProc)	*/
void	VtoCall();		/*       "to"	dito			*/
void	VTimestep();		/* Called every second when in backgnd	*/


/* Forward declarations */

void	*access_char();		/* Basic array access methods		*/
void	*access_int();
void	*access_long();
void	*access_float();
void	*access_double();
Boolean	compare_char();		/* Basic Array element comparison	*/
Boolean	compare_int();
Boolean	compare_long();
Boolean	compare_float();
Boolean	compare_double();
Boolean	compare_range_char();	/* Array element comparison for ranges	*/
Boolean	compare_range_int();
Boolean	compare_range_long();
Boolean	compare_range_float();
Boolean	compare_range_double();
char	value_char();		/* Array element to display value conversion */
char	value_int();
char	value_long();
char	value_float();
char	value_double();
void	update_shown_data();	/* Update display of position & value	*/
void	update_shown_condition();    /* Update display of condition	*/
void	display_layer();	/* Display a section along 2 dimensions	*/
void	do_visualization();	/* Final initialization & display	*/
void	setup_read();		/* Socket initialization		*/
void	setup_write();
void	setup_accept();
void	handle_socket();	/* Handle reading of new array data	*/
void	do_statistics();	/* Compute min/max/average		*/


/* Sizes of array elements */

static int SizeOfType[] = {
    sizeof(char),			/* (Elements in same order as	*/
    sizeof(int),			/*  listend in Vis.h !!!)	*/
    sizeof(long),
    sizeof(float),
    sizeof(double)
};


/* Name of of array element types */

static String NameOfType[] = {
    "char",				/* (Elements in same order as	*/
    "int",				/*  listend in Vis.h !!!)	*/
    "long",
    "float",
    "double"
};


/* Macros for array access */

#define ACCESS1(x0,d0) x0
#define ACCESS2(x0,d0,x1,d1) x0*d1+x1
#define ACCESS3(x0,d0,x1,d1,x2,d2) (x0*d1+x1)*d2+x2
#define ACCESS4(x0,d0,x1,d1,x2,d2,x3,d3) ((x0*d1+x1)*d2+x2)*d3+x3


/*
 *
 *  VIStype *OpenVisualizer(int x, int y)
 *
 *  Initializes the X Window basics to make further visualization possible.
 *  The return value is later needed for visualization and finally for
 *  "CloseVisualizer()"
 *  "x" and "y" specify the coordinates of the upper left corner of the
 *  Visualizer Window when it appears. If "x" or "y" are less than zero,
 *  the Visualizer will be positioned with user interaction (unless the
 *  user's window manager places windows automatically, anyway).
 *
 */
VIStype *OpenVisualizer(x,y)
int x,y;
{
    char    *sp[4];
    String resources[10];		/* X Resources		*/
    int      i;				/* Tmp counter / argc	*/
    char     a[128];
    char     r[128];
    VIStype *VIS = (VIStype*)XtMalloc(sizeof(VIStype));


    /* Set up resources for X, add positioning if x & y given	*/

   i=0;
   resources[i]="*showGrip:             off";  i++;
   resources[i]="*allowShellResize:     True"; i++;
   if ( (x>=0) && (y>=0) ) {			/* Set Position		*/
       sprintf(r,"Visualizer*geometry: +%d+%d", x,y);
       resources[i]=r; i++;
   };
   resources[i]=NULL;

    /* Create dummy argc & argv and a new X Window for the Visualizer	*/

    sprintf(a,"Visualizer");
    sp[0]=&(a[0]);				/* Create a dummy argv	*/
    sp[1]=NULL;					/* Terminate dummy argv	*/
    i=1;					/* Dummy argc		*/

    VIS->VISparent = XtAppInitialize(&(VIS->VISapp_con), "Visualizer",
                                     NULL, ZERO,
                                     &i, sp, resources, NULL, ZERO);

    /* Mark as not yet being used by a visualizer */

    VIS->VISvis = NULL;
    debuginfo(("VISUALIZER: VIS @ %d initialized\n",VIS));
    return(VIS);

};


/*
 *
 *  void CloseVisualizer(VIStype *VIS)
 *
 *  Destroys the X Window basics connected with VIS.
 *  Also kill off background process if there is one.
 *
 */
void CloseVisualizer(VIS)
VIStype *VIS;
{
    XEvent		event;
    XDestroyWindowEvent	*ep;
    visualizertype	*vis = (visualizertype *)VIS->VISvis;
    int			done;


    /* Do cleanup */

    if (vis) {/* Visualizer active*/
        if (vis->vforked) {

            /* Background Visualizer running -> close & kill		*/

            debuginfo(("++ CloseVisualizer: Killing child process %d\n", vis->vchildproc));

            /* Make Visualizer disappear (handle events until unmap)	*/

            close(vis->vdatawritesock);	/* Close sockets (we want no	*/
            close(vis->vcmdwritesock);	/* "broken pipe" signals)	*/
/*printf("%d CLOSE: KILLE KIND\n",VIS);*/
            done = FALSE;
            if (!kill(vis->vchildproc, SIGKILL)) {
                wait(NULL);		/* Wait for child to terminate	*/
            } else {
                if (errno == ESRCH)
                    done=TRUE;		/* Process didn't exist any more  */
					/* i.e. was quit by user already  */
					/* so don't try to process events */
					/* for it!			  */
            };
/*printf("%d CLOSE: KILL OK, DONE = %d\n",VIS,done);*/
            /*!!!!!!! deallocate data of vis? */

            XtDestroyWidget(VIS->VISparent);
/*printf("%d CLOSE: WARTE AUF DESTROY\n",VIS);*/
            while (done == FALSE) {
                XtAppNextEvent(VIS->VISapp_con,&event);
                XtDispatchEvent(&event);
                ep = (XDestroyWindowEvent*) &event;
                if ((ep->type == DestroyNotify) &&
                    (ep->window == XtWindow(VIS->VISparent)) )
                   done=TRUE;
            };
/*printf("%d CLOSE: DESTROY FERTIG\n",VIS);*/

/*            XtDestroyApplicationContext(VIS->VISapp_con);*/


            unlink(vis->vdatasockname);	/* Remove socket files (binding)*/
            unlink(vis->vcmdsockname);
       
        } else {

            /* Foreground Visualizer running ... hardly possible here	*/
            /* or this function couldn't have been called		*/
            debuginfo(("fooo!\n"));
        };
    } else {/* No Visualizer active */

            /* User has already closed the Visualizer Window (quit)	   */
            /* (be it in fore- or background)				   */
            /* Close the X Windows connection and handle remaining events  */

            debuginfo(("++ CloseVisualizer: clean up remaining events\n"));

/* !!!!!!!! sometimes dumps core in XtDestroyApplicationContext() !!!!!!!*/
/*            XtDestroyApplicationContext(VIS->VISapp_con); */

/*printf("%d CLOSE: CLEAN UP\n",VIS);*/
            XtDestroyWidget(VIS->VISparent);
            done = FALSE;
/*printf("%d CLOSE: WARTE AUF REST\n",VIS);*/
            while ( (done == FALSE) && (XtAppPending(VIS->VISapp_con)) ) {
                XtAppNextEvent(VIS->VISapp_con,&event);
                XtDispatchEvent(&event);
                ep = (XDestroyWindowEvent*) &event;	/*Handle final events*/
                if ((ep->type == DestroyNotify) &&
                    (ep->window == XtWindow(VIS->VISparent)) )
                   done=TRUE;
            };
/*printf("%d CLOSE: REST DA\n",VIS);*/

    };

    XtFree(VIS);
    debuginfo(("VISUALIZER: VIS @ %d closed and destroyed\n",VIS));

    return;
};



/****************** VISUALIZER PREPARATION (inits) **********************/



/*
 * void PrepareCommon(VIS, name, namelen, adr, dims, numdims, type)
 *
 * Prepares information common to all Visualizers
 *
 */
void PrepareCommon(VIS, name, namelen, adr, dims, numdims,type)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
{
    visualizertype *vis;
    int i;


    /* Create a new visualizer data structure */


    vis = (visualizertype *) XtMalloc(sizeof(visualizertype));


    /* Select the comparison and access functions according to		*/
    /* the array type	(not useful for all visualizers, but most)	*/

    switch (type) {
    case V_CHAR: vis->vaccfn = access_char;
                 vis->vcmpfn = compare_char;
                 break;
    case V_INT:  vis->vaccfn = access_int;
                 vis->vcmpfn = compare_int;
                 break;
    case V_LONG: vis->vaccfn = access_long;
                 vis->vcmpfn = compare_long;
                 break;
    case V_FLOAT:vis->vaccfn = access_float;
                 vis->vcmpfn = compare_float;
                 break;
    case V_DOUBLE:vis->vaccfn= access_double;
                 vis->vcmpfn = compare_double;
                 break;
    };


    /* Copy the values of the dimensions, also filling the nullvector	*/
    /* and the vector of maximum elements for each dimension (for use	*/
    /* with ADT counter							*/


    vis->vnumdims = (int)numdims;
    for (i = 0; i < numdims; i++) {
        vis->vdims[i] = dims[i];
    };
    vis->vtype = type;
    vis->vadr  = adr;
    if (namelen < 1) {		/* Get real length of string..	*/
        namelen = strlen(name);	/* ..if not given		*/
    };
    strncpy(vis->vname, name,namelen);
    (vis->vname)[namelen]='\0';	/* Zero-terminate string	*/

    VIS->VISvis      = vis;		/* Mark VIS in use		*/
    vis->vVIS        = VIS;		/* Remember parent		*/

}


/*
 * void PrepareVis(VIS, name, namelen, adr, dims, numdims,
 *                 type, condition, comparison, forkflag, syncupdateflag)
 *
 * Fills in the information necessary for a normal Visualizer.
 * Purpose: Initialization useable for "normal" Visualizer creation
 *          and Visualizer Widgets
 *
 */
void PrepareVis(VIS, name, namelen, adr, dims, numdims,
               type, condition, comparison,
               forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
int      condition;		/* e.g. V_GREATER			*/
void    *comparison;		/* Value to be compared to		*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    visualizertype *vis;

    
    /* Initialize common structures */
    
    PrepareCommon(VIS, name, namelen, adr, dims, numdims,type);

    vis = VIS->VISvis;		/* Freshly initialized		*/

    vis->vcondition = condition;
    vis->vcomparison = comparison;

    vis->vusefn = FALSE;	/* No user defined functions	*/
    vis->vuserange = FALSE;	/* No ranges used		*/
    vis->vcallfn = NULL;	/* No button click activation	*/
    vis->vcmptext[0] = '\0';	/* No comparison text		*/

    vis->vforked     = forkflag;	/* Set flag for forking		*/
    vis->vsyncupdate = syncupdateflag;	/* Set flag for synchronizing	*/
    vis->vsyncchfn   = FALSE;		/* No user function, so no sync	*/
    vis->venvp       = NULL;		/* No user environment		*/
    vis->venvlen     = 0;
    vis->vdispmode   = BW_DISPLAY;	/* Display mode b&w (display.h) */

}


/*
 * void PrepareVisRange(VIS, name, namelen, adr, dims, numdims,
 *                      type, condition, comparison, forkflag, syncupdateflag)
 *
 * Fills in the information necessary for a Range Visualizer.
 * Purpose: Initialization useable for "normal" Visualizer creation
 *          and Visualizer Widgets
 *
 */
void PrepareVisRange(VIS, name, namelen, adr, dims, numdims,
                    type, fromvalue, tovalue, forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
void    *fromvalue;		/* Range "from"				*/
void    *tovalue;		/* Range "to"				*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    visualizertype *vis;

    
    /* Initialize common structures */

    PrepareCommon(VIS, name, namelen, adr, dims, numdims,type);

    vis = VIS->VISvis;          /* Freshly initialized          */

    
    /* Use our own access/compare functions, not the ones provided	*/
    /*  by PrepareCommon()						*/

    switch (type) {
    case V_CHAR:  vis->vaccfn = access_char;
	          vis->vcmpfn = compare_range_char;
	          break;
    case V_INT:   vis->vaccfn = access_int;
	          vis->vcmpfn = compare_range_int;
	          break;
    case V_LONG:  vis->vaccfn = access_long;
	          vis->vcmpfn = compare_range_long;
	          break;
    case V_FLOAT: vis->vaccfn = access_float;
	          vis->vcmpfn = compare_range_float;
	          break;
    case V_DOUBLE:vis->vaccfn= access_double;
	          vis->vcmpfn = compare_range_double;
	          break;
    };
    
    vis->vcondition = V_EQUAL;	/* Dummy values for unused condition	    */
    vis->vcomparison = vis;	/* Pass the vis struct in the cmp-value !!! */
				/* (trick to get info in compare_range_xxx) */

    vis->vfromval = fromvalue;	/* Set the range, no infinites	*/
    vis->vtoval = tovalue;
    vis->vfrominfinite = FALSE;
    vis->vtoinfinite = FALSE;

    vis->vusefn = FALSE;	/* No user defined functions	*/
    vis->vuserange = TRUE;	/* We're using ranges		*/
    vis->vcallfn = NULL;	/* No button click activation	*/
    vis->vcmptext[0] = '\0';	/* No comparison text		*/

    vis->vforked     = forkflag;	/* Set flag for forking		*/
    vis->vsyncupdate = syncupdateflag;	/* Set flag for synchronizing	*/
    vis->vsyncchfn   = FALSE;		/* No user function, so no sync	*/
    vis->venvp       = NULL;		/* No user environment		*/
    vis->venvlen     = 0;
    vis->vdispmode   = BW_DISPLAY;	/* Display mode b&w (display.h) */

}



/*
 * void PrepareVisUserFn(VIS, name, namelen, adr, dims, numdims, type,
 *                       cmptext, cmptextlen, compare, access, envp, envlen,
 *                       forkflag, syncupdateflag, syncfunctionflag)
 *
 * Fills in the information necessary for a User Function Visualizer.
 * Purpose: Initialization useable for "normal" Visualizer creation
 *          and Visualizer Widgets
 *
 */
void PrepareVisUserFn(VIS, name, namelen, adr, dims, numdims, type,
                     cmptext, cmptextlen, compare, access, envp, envlen,
                     forkflag, syncupdateflag, syncfunctionflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
char     *cmptext;		/* Text for comparison description	*/
ulong    cmptextlen;		/* Length of this text			*/
Boolean  (*compare)();		/* Comparison function			*/
void     *(*access)();		/* Access function			*/
void     *envp;			/* Pointer to user data for function	*/
ulong    envlen;		/* Length of user data			*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
Boolean  syncfunctionflag;	/* Sync function change calls		*/
{
    visualizertype *vis;

    
    /* Initialize common structures */

    PrepareCommon(VIS, name, namelen, adr, dims, numdims,type);

    vis = VIS->VISvis;		        /* Freshly initialized          */
    
    
    vis->vcondition  = V_EQUAL;		/* Dummies for compare & cond.	*/
    vis->vcomparison = ZERO;

    vis->vuserange = FALSE;		/* No ranges used		*/
    vis->vusefn = TRUE;			/* Yes, using user def'ed fn's	*/
    vis->vcallfn = NULL;		/* No button click activation	*/
    vis->vaccfn = access;		/* Use the given functions	*/
    vis->vcmpfn  = compare;		/* (overwrite init by Prep.Com.)*/
    if (cmptextlen < 1) {		/* Get real length of string..	*/
        cmptextlen = strlen(cmptext);	/* ..if not given		*/
    };					/* Text for comparison display	*/
    strncpy(vis->vcmptext,cmptext,cmptextlen);
    (vis->vcmptext)[cmptextlen]='\0'; 	/* Zero-terminate string	*/


    vis->vforked     = forkflag;	/* Set flag for forking		*/
    vis->vsyncupdate = syncupdateflag;	/* Set flag for synchronizing	*/
    vis->vsyncchfn   = syncfunctionflag;/* Set flag for function sync.	*/


    /* Allocate environment space and copy user data into it		*/
    /* (if there is any)						*/

    if (envp) {
        vis->venvp   = (void*)XtMalloc((long)envlen);	/* Get buffer	*/
        bcopy(envp,vis->venvp,(long)envlen);		/* Copy data	*/
        vis->venvlen = (long)envlen;			/* Remember len	*/
    } else {
        vis->venvp   = NULL;				/* No buffer	*/
        vis->venvlen = (long)0;
    };
    vis->vdispmode   = BW_DISPLAY;	/* Display mode b&w (display.h) */

};



/*
 * void PrepareVisValues(VIS, name, namelen, adr, dims, numdims,type,
 *                       forkflag, syncupdateflag)
 *
 * Fills in the information necessary for a Value Visualizer.
 * Purpose: Initialization useable for "normal" Visualizer creation
 *          and Visualizer Widgets
 *
 */
void PrepareVisValues(VIS, name, namelen, adr, dims, numdims, type,
		      forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    visualizertype *vis;

    
    /* Initialize common structures */

    PrepareCommon(VIS, name, namelen, adr, dims, numdims,type);

    vis = VIS->VISvis;		        /* Freshly initialized          */
    
    
    /* Here, the comparison functions serve as translators      */
    /* from array element value to display pixel value          */

    switch (type) {
    case V_CHAR:  vis->vaccfn = access_char;
		  vis->vcmpfn = value_char;
		  break;
    case V_INT:   vis->vaccfn = access_int;
		  vis->vcmpfn = value_int;
		  break;
    case V_LONG:  vis->vaccfn = access_long;
		  vis->vcmpfn = value_long;
		  break;
    case V_FLOAT: vis->vaccfn = access_float;
		  vis->vcmpfn = value_float;
		  break;
    case V_DOUBLE:vis->vaccfn= access_double;
		  vis->vcmpfn = value_double;
		  break;
    };
    
    vis->vcondition  = V_EQUAL;		/* Init with dummy values	*/
    vis->vcomparison = ZERO;

    vis->vusefn      = FALSE;		/* No user defined functions	*/
    vis->vuserange   = FALSE;		/* No ranges used		*/
    vis->vcallfn     = NULL;		/* No button click activation	*/
    vis->vcmptext[0] = '\0';		/* No comparison text		*/

    vis->vforked     = forkflag;	/* Set flag for forking		*/
    vis->vsyncupdate = syncupdateflag;	/* Set flag for synchronizing	*/
    vis->vsyncchfn   = FALSE;		/* No user function, so no sync	*/
    vis->venvp       = NULL;		/* No user environment		*/
    vis->venvlen     = (long)0;
    vis->vdispmode   = GRAY_DISPLAY;	/* Display mode grayscale	*/

}



/*
 * void PrepareVisActivation(VIS, name, namelen, adr, dims, comparison, callfn,
 *                           forkflag, syncupdateflag)
 *
 * Fills in the information necessary for aa Activation Visualizer.
 * Purpose: Initialization useable for "normal" Visualizer creation
 *          and Visualizer Widgets
 *
 */
void PrepareVisActivation(VIS, name, namelen, adr, dims, comparison, callfn,
			  forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
void    *comparison;		/* Value to be compared to		*/
void	(*callfn)();		/* Function to call when clicked	*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    visualizertype *vis;
    int type    = V_LONG;		/* Standard values for...	*/
    int numdims = 1;			/* ...activation display	*/

    
    /* Initialize common structures */

    PrepareCommon(VIS, name, namelen, adr, dims, numdims,type);

    vis = VIS->VISvis;		        /* Freshly initialized          */
    

    vis->vcondition  = V_EQUAL;
    vis->vcomparison = comparison;

    vis->vusefn = FALSE;	/* No user defined functions	*/
    vis->vuserange = FALSE;	/* No ranges used		*/
    vis->vcallfn = callfn;	/* BUT a button click activation*/
    vis->vcmptext[0] = '\0';	/* No comparison text		*/

    vis->vforked     = forkflag;	/* Set flag for forking		*/
    vis->vsyncupdate = syncupdateflag;	/* Set flag for synchronizing	*/
    vis->vsyncchfn   = FALSE;		/* No user function, so no sync	*/
    vis->venvp       = (long*)XtMalloc(sizeof(long)); /* Position in array*/
    *((long*)vis->venvp) = 0;		/* (Instead of counters)	*/
    vis->venvlen     = 0;
    vis->vdispmode   = ACT_DISPLAY;	/* Activation mode (display.h)	*/

}




/****************** VISUALIZER USING "NORMAL" INTERFACE *****************/




/*
 *
 *  void  Visualize(  VIStype *VIS,
 *                    char    *name,
 *                    ulong    namelen,
 *                    void    *adr,
 *                    int     *dims,
 *                    ulong    numdims,
 *                    int      type,
 *                    int      condition,
 *                    void    *comparison,
 *                    Boolean  forkflag,
 *                    Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements fulfilling the comparison defined by "condition"
 *    and "comparison" are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    Visualize(VIS,"a", 1, a, dims, numdims,
 *              V_INT, V_GREATER, &value, FALSE, FALSE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements that are greater than "value", running in the foreground
 *    (i.e. main program being suspended until Visualizer is terminated)
 *  Of course the sync-flag is not set here, because there is no background
 *    process at all.
 *
 */
void Visualize(VIS, name, namelen, adr, dims, numdims,
               type, condition, comparison,
               forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
int      condition;		/* e.g. V_GREATER			*/
void    *comparison;		/* Value to be compared to		*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{

    /* Check if VIS already being used */

    if (VIS->VISvis) {
        printf(/*stderr,*/
              "VISUALIZER: VIS already being used for display! \n");
        exit(-1);
    };

    
    /* Create a new visualizer data structure and fill in the necessary	*/
    /* information							*/

    PrepareVis(VIS, name, namelen, adr, dims, numdims,
               type, condition, comparison,
               forkflag, syncupdateflag);

    
    /* Do the final initialization work and display the Visualizer	*/
    
    do_visualization(VIS);

}


/*
 *
 *  void  VisualizeRange(  VIStype *VIS,
 *                         char    *name,
 *                         ulong    namelen,
 *                         void    *adr,
 *                         int     *dims,
 *                         ulong    numdims,
 *                         int      type,
 *                         void    *fromvalue,
 *                         void    *tovalue,
 *		           Boolean  forkflag,
 *                         Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements in the range "fromvalue" to "tovalue"
 *    are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *  Example call:
 *
 *    VisualizeRange(VIS,"a", 0, a, dims, numdims, V_INT, &i, &j,
 *                   TRUE, FALSE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements in the range [i..j] running in the background (i.e. the
 *    main program continues and is able to update the array using
 *    UpdateForkedVisualizer(VIS) ), not synchronizing the update calls.
 *    Then length of the name (i.e. "a") is computed internally since the
 *    value given (i.e. 0) is less than 1.
 *
 */
void VisualizeRange(VIS, name, namelen, adr, dims, numdims,
                    type, fromvalue, tovalue, forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
void    *fromvalue;		/* Range "from"				*/
void    *tovalue;		/* Range "to"				*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{


    /* Check if VIS already being used */

    if (VIS->VISvis) {
        printf(/*stderr,*/
              "VISUALIZER: VIS already being used for display! \n");
        exit(-1);
    };


    /* Create a new visualizer data structure and fill in the necessary	*/
    /* information							*/

    PrepareVisRange(VIS, name, namelen, adr, dims, numdims,
                    type, fromvalue, tovalue, forkflag, syncupdateflag);

    
    /* Do the final initialization work and display the Visualizer	*/

    do_visualization(VIS);

}


/*
 *
 *  void  VisualizeUserFn(  VIStype *VIS,
 *                          char    *name,
 *                          ulong    namelen,
 *                          void    *adr,
 *                          int     *dims,
 *                          ulong    numdims,
 *                          int      type,
 *                          char    *cmptext,
 *                          ulong    cmptextlen,
 *                          Boolean  (*compare)(),
 *                          void    *(*access)(),
 *                          void    *envp,
 *                          ulong    envlen,
 *			    Boolean  forkflag,
 *                          Boolean  syncupdateflag,
 *                          Boolean  syncfunctionflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *    (the same applies to "cmptextlen")
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  "envp" points to some user data that may be used in the compare
 *    function later on. The length of the data is given in "envlen".
 *  The user data is newly allocated and copied so changing it after-
 *    wards has no effect. Use ChangeUserFn() to alter it.
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *  If "syncfunctionflag" is TRUE, a call to ChangeUserFn() will block.
 *
 *  The "cmptext" describes the comparison. It has to contain one
 *    occurence of "%s" which is automagically replaced by the array name
 *    and thea ctual coordinates.
 *  The user defined comparison and array access functions are used as
 *    follows:
 *
 *  Boolean compare(void *a, int cond, void *envp)
 *
 *      Compares the value of the array element (given in "a") to some
 *        other value, possibly given by the environment pointer "envp"
 *        (which is passed at the 1st call to VisualizeUserFn() and may
 *        be changed using ChangeUserFn() ). The comparison condition
 *        (V_EQUAL, V_LESS etc) is passed in "cond".
 *      The user function is free to ignore "cond" as well as "envp"
 *        and perform whatever operation it likes.
 *
 *  void   *access(void *adr, int *x, int *dims, ulong numdims)
 *
 *      Returns a pointer to the array element at adr[x[0],x[1],...]
 *        where the number of dimensions is in numdims and the size of
 *        each dimension is given in dims[].
 *
 *  All array elements fulfilling the comparison defined by the user
 *    supplied functions are displayed in black, the rest in white.
 *
 *  Example call:
 *
 *    VisualizeUserFn(VIS, "a", 0, a, dims, numdims, V_INT,
 *                    "10 <= %s <= 20", 0, my_compare, my_access,
 *                    &data, sizeof(data), TRUE, TRUE, TRUE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements that fulfill the comparison function "my_compare"
 *    using the data in "data" and are accessed via "my_access",
 *    running in the background (i.e. the main program continues and is able
 *    to update the array using UpdateForkedVisualizer(VIS))
 *  The Visualizer is synchronized with the foreground process for both
 *    UpdateForkedVisualizer() and ChangeUserFn() calls.
 *  The length of both the name of the array and the string describing
 *    the user function are computed internally since they are both <1.
 *
 */
void VisualizeUserFn(VIS, name, namelen, adr, dims, numdims, type,
                     cmptext, cmptextlen, compare, access, envp, envlen,
                     forkflag, syncupdateflag, syncfunctionflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
char     *cmptext;		/* Text for comparison description	*/
ulong    cmptextlen;		/* Length of this text			*/
Boolean  (*compare)();		/* Comparison function			*/
void     *(*access)();		/* Access function			*/
void     *envp;			/* Pointer to user data for function	*/
ulong    envlen;		/* Length of user data			*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
Boolean  syncfunctionflag;	/* Sync function change calls		*/
{
    visualizertype *vis;
    int i;


    /* Check if VIS already being used */

    if (VIS->VISvis) {
        printf(/*stderr,*/
              "VISUALIZER: VIS already being used for display! \n");
        exit(-1);
    };



    /* Create a new visualizer data structure and fill in the necessary	*/
    /* information							*/

    PrepareVisUserFn(VIS, name, namelen, adr, dims, numdims, type,
                     cmptext, cmptextlen, compare, access, envp, envlen,
                     forkflag, syncupdateflag, syncfunctionflag);

    
    /* Do the final initialization work and display the Visualizer	*/

    do_visualization(VIS);

    return;                     
};



/*
 *
 *  void  VisualizeValues(  VIStype *VIS,
 *                          char    *name,
 *                          ulong    namelen,
 *                          void    *adr,
 *                          int     *dims,
 *                          long     numdims,
 *                          int      type,
 *                          Boolean  forkflag,
 *                          Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  The elements of the array are displayed in using grayscale bitmaps,
 *    thus giving a quick overview of the distribution of values in
 *    the array. However, no boolean operators are used.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    VisualizeValues(VIS,"a", a, dims, numdims, V_INT, FALSE, FALSE)
 *
 *  will display the visualization of the integer array a and display
 *    all elements as different shades of gray according to their value.
 *    No background operation (and thus no sync) is initiated.
 *
 */
void VisualizeValues(VIS, name, namelen, adr, dims, numdims,type,
               forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    void *(*access)();			/* Access method for array	*/
    Boolean (*compare)();		/* Comparison method for array	*/
    visualizertype *vis;
    int i;


    /* Check if VIS already being used */

    if (VIS->VISvis) {
        printf(/*stderr,*/
              "VISUALIZER: VIS already being used for display! \n");
        exit(-1);
    };



    /* Create a new visualizer data structure and fill in the necessary	*/
    /* information							*/

    PrepareVisValues(VIS, name, namelen, adr, dims, numdims,type,
		     forkflag, syncupdateflag);

    
    /* Do the final initialization work and display the Visualizer	*/

    do_visualization(VIS);

    return;                     
}


/*
 *
 *  void  VisualizeActivation(  VIStype *VIS,
 *                              char    *name,
 *                              ulong    namelen,
 *                              void    *adr,
 *                              int     *dims,
 *                              void    *comparison,
 *                              void     (*callfn)(),
 *                              Boolean  forkflag,
 *                              Boolean  syncupdateflag)
 *
 *  Creates a visualization of the 1-dimensional array of activation values
 *    (longs) with given starting address "adr" and name "name" of the
 *    size [0..dims[0]-1]. The length of the array is adjusted so that
 *    it fits into a square (but the additional elements are never referenced
 *    so no need to worry :-).
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements equal to the long value passed in "*comparison"
 *    are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  "callfn" is a function that gets called whenever the user presses
 *    the middle mouse button inside the Visualizer. If NULL is given,
 *    nothing is called. The function has the following syntax:
 *
 *        callfn(VIStype *VIS, long nr)  [where nr = array index clicked on]
 *
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    VisualizeActivation(VIS,"activation for a", 0, act_bits, &size,
 *              &act_value, NULL, FALSE, FALSE)
 *
 *  will display the visualization of the activation array a and highlight
 *    all elements that are active (i.e. equal to "act_value"), running
 *    in the foreground (i.e. main program being suspended until Visualizer
 *    is terminated). When the mouse is clicked, nothing will be done.
 *  Of course the sync-flag is not set here, because there is no background
 *    process at all.
 *
 */
void VisualizeActivation(VIS, name, namelen, adr, dims, comparison, callfn,
               forkflag, syncupdateflag)
VIStype *VIS;			/* Parent Application Context & Widget	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
void    *comparison;		/* Value to be compared to		*/
void	(*callfn)();		/* Function to call when clicked	*/
Boolean  forkflag;		/* Run in background if TRUE		*/
Boolean  syncupdateflag;	/* Sync update calls when in background	*/
{
    void *(*access)();			/* Access method for array	*/
    Boolean (*compare)();		/* Comparison method for array	*/
    visualizertype *vis;
    int i;
    int type    = V_LONG;		/* Standard values for...	*/
    int numdims = 1;			/* ...activation display	*/


    /* Check if VIS already being used */

    if (VIS->VISvis) {
        printf(/*stderr,*/
              "VISUALIZER: VIS already being used for display! \n");
        exit(-1);
    };



    /* Create a new visualizer data structure and fill in the necessary	*/
    /* information							*/

    PrepareVisActivation(VIS, name, namelen, adr, dims, comparison, callfn,
			 forkflag, syncupdateflag);

    
    /* Do the final initialization work and display the Visualizer	*/

    do_visualization(VIS);

}



/********************* THE WIDGET CREATION AND MAIN LOOP ****************/



/*
 * void do_vis_widgets(VIStype *VIS, Boolean widget_only)
 *
 * Initializes all widgets of the Visualizer, according to the Visualizer's
 * type.
 * If "widget_only" is True, the callback for the Quit button is initialized
 * appropriately (destroys parent etc, otherwise, in normal Vis.useage!)
 *
 */
void do_vis_widgets(VIS,widget_only)
VIStype        *VIS;
Boolean		widget_only;
{
    int			i;
    char		s[512];
    int			nullvector[MAXDIMS]; /* Minima for counters	 */
    int			maxvector[MAXDIMS];  /* Upper boundaries	 */
    Arg			args[10];	     /* Arg's for Widget creation*/
    XtActionsRec	arec;	/* Action Record for callback addresses  */
    visualizertype	*vis = (visualizertype *)VIS->VISvis;
    int			restl = 0;	/* For Activation value Visual.	 */
    int			zoom = 6;	/* Zoom factor for Array Display */
    int			screenx;	/* Size of display screen	 */
    int			screeny;
    int			maxdim;		/* Size of larges dimension	 */


    /* Check if call was OK */


    if ( (vis->vnumdims < 1) || (vis->vnumdims > MAXDIMS)) {
        printf(/*stderr,*/
              "VISUALIZER: Number of dimensions must be between 1 and %d !\n",
               MAXDIMS);
        exit(-1);
    };


    /* If one-dimensional, try to find width (vonewidth) close to	*/
    /* sqrt(size_of_first_dimension) to display the array as		*/
    /* 2-dim: (vonewidth) x (size_of_first_dimension / vonewidth)	*/
    /* Exception: for Activation Display, use next square!		*/

    if (vis->vdispmode == ACT_DISPLAY) {
        int l = vis->vdims[0];	/* Length of array we're trying to fit in */
        int x;			/* Length of one side			  */

        x = (int) (sqrt( (float)l ));
        if (x*x < l) x++;	/* Round up to next square		*/
        vis->vonewidth = x;	/* Remember suitable width		*/
        restl = l-1;

        printf("VISUALIZER: Showing Activation (size 0..%d) as 2-dim (%d x %d), restl=%d\n",vis->vdims[0]-1, x, x, restl);

    } else if (vis->vnumdims == 1) {
        int x,y;		/* X and Y of new display		  */
        int l = vis->vdims[0];	/* Length of array we're trying to fit in */
				/* (use n for array [0..n-1] !!		  */
	
        y = (int) (sqrt( (float) l) );
        x = l / y;

	while (x*y != l) {	/* Look for suitable x & y until x*y==l	*/
            y--;		/* Terminates if y==1 and x==l		*/
            x = l / y;		/* (worst case of course (if l prime) ;)*/
        };
        vis->vonewidth = x;	/* Remember suitable width (n for 0..n-1)*/
	x--; y--;		/* Compensate (0..n-1 = n elements)	 */

        printf("VISUALIZER: Showing 1-dim array (size 0..%d) as 2-dim (0..%d,0..%d)\n",l-1, x, y);

    } else {
        vis->vonewidth = 0;	/* More than one dimension, don't care	*/
    };


    /* Compute a sensible size for the display window pixels */

    /*!!!!!!! THESE VALUES SHOULD BE COMPUTED from the real screen size	  */
    /*!!!!!!! and the size of the different widgets (scrollers, counters, */
    /*!!!!!!! quit etc) 						  */
    screenx = 600;		/*!!!!!!! SHOULD BE DisplayWidth(dsp,scrn); */
    screeny = 600;		/*!!!!!!! SHOULD BE DisplayHeight(dsp,scrn);*/
    
    if (vis->vnumdims == 1) {	/* Get size of largest dimension	*/
	int x = vis->vonewidth;
	int y = (vis->vdims[0])/x;
	maxdim = (x>y)?x:y;
	debuginfo(("1-DIM VISUALIZER SIZE MAXDIM %d\n",maxdim));
    } else {
	int i;
	maxdim=0;
	for (i=0; i<vis->vnumdims; i++)
	  if (vis->vdims[i]>maxdim)
	    maxdim=vis->vdims[i];
	debuginfo(("%d-DIM VISUALIZER SIZE MAXDIM %d\n",vis->vnumdims,maxdim));
    };
    zoom = screenx/maxdim;	/* Adapt zoom factor to max array size	*/
    if (zoom<1)
      zoom=1;
    if (zoom>10)
      zoom=10;
    debuginfo(("ZOOM SET TO %d\n",zoom));
    
	
    /* Create a null-vector and the upper boundaries for the counters */

    for (i = 0; i < vis->vnumdims; i++) {
        nullvector[i] = 0;
	maxvector[i]  = vis->vdims[i]-1;
    };


    /* Create children Widgets for data display */

    vis->vparent = VIS->VISparent;		  /* Basic parent widget */

    i=0;
    XtSetArg(args[i],XtNpreferredPaneSize, True);i++;
    XtSetArg(args[i],XtNshowGrip, False);i++;
    vis->vispaned = XtCreateManagedWidget("vispaned", panedWidgetClass,
                               vis->vparent, args, i);
    /* Quit button */
    i=0;
    XtSetArg(args[i],XtNlabel,"Quit visualization");i++;
    vis->visquit = XtCreateManagedWidget("visquit", commandWidgetClass,
                                   vis->vispaned, args,i);
    /* Position & value display */
    i=0;
    XtSetArg(args[i],XtNlabel,"Pointer not in Array");i++;
    vis->visshow = XtCreateManagedWidget("visshow",labelWidgetClass,
                               vis->vispaned, args,i);
    /* Condition display */
    i=0;
    XtSetArg(args[i],XtNlabel,"Condition: <name> <oper> <val>");i++;
    vis->viscond = XtCreateManagedWidget("viscond",labelWidgetClass,
                               vis->vispaned, args,i);


    /* Create the condition entry line if functions not user defined	*/
    /* and we're not using a value Visualizer (i.e. grayscale)		*/

    if (!(vis->vusefn || vis->vuserange) && (vis->vdispmode==BW_DISPLAY)) {
 
     /* Main form widget */
     i=0;
     XtSetArg(args[i],XtNshowGrip,False);i++;
     vis->visselect = XtCreateManagedWidget("visselect",formWidgetClass,
                                vis->vispaned, args,i);
     /* Label for selection */
     i=0;
     XtSetArg(args[i],XtNlabel,"Select Condition: ");i++;
     XtSetArg(args[i],XtNborderWidth,0);i++;
     vis->vs_label = XtCreateManagedWidget("vs_label",labelWidgetClass,
                                vis->visselect, args,i);
     /* Buttons for ">" "=" "<" */
     i=0;
     XtSetArg(args[i],XtNlabel,">");i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vs_label);i++;
     XtSetArg(args[i],XtNhorizDistance,1);i++;
     XtSetArg(args[i],XtNborderWidth,1);i++;
     vis->vs_gtbtn = XtCreateManagedWidget("vs_gtbtn",commandWidgetClass,
                                vis->visselect, args,i);
     i=0;
     XtSetArg(args[i],XtNlabel,"=");i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vs_gtbtn);i++;
     XtSetArg(args[i],XtNhorizDistance,2);i++;
     XtSetArg(args[i],XtNborderWidth,1);i++;
     vis->vs_eqbtn = XtCreateManagedWidget("vs_eqbtn",commandWidgetClass,
                                vis->visselect, args,i);
     i=0;
     XtSetArg(args[i],XtNlabel,"<");i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vs_eqbtn);i++;
     XtSetArg(args[i],XtNhorizDistance,2);i++;
     XtSetArg(args[i],XtNborderWidth,1);i++;
     vis->vs_ltbtn = XtCreateManagedWidget("vs_ltbtn",commandWidgetClass,
                                vis->visselect, args,i);
     /* Type display */
     i=0;
     sprintf(s,"(%s)",NameOfType[vis->vtype]);	/* Get name of used type */
     XtSetArg(args[i],XtNlabel,s);i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vs_ltbtn);i++;
     XtSetArg(args[i],XtNhorizDistance,5);i++;
     XtSetArg(args[i],XtNborderWidth,0);i++;
     vis->vs_type = XtCreateManagedWidget("vs_type",labelWidgetClass,
                                vis->visselect, args,i);
     /* Comparison value entry */
     i=0;
     switch (vis->vtype) {
     case V_CHAR: sprintf(s,"%d",*((char*)(vis->vcomparison)));
                  break;
     case V_INT:  sprintf(s,"%d",*((int*) (vis->vcomparison)));
                  break;
     case V_LONG: sprintf(s,"%d",*((long*)(vis->vcomparison)));
                  break;
     case V_FLOAT: sprintf(s,"%g",*((float*)(vis->vcomparison)));
                  break;
     case V_DOUBLE: sprintf(s,"%g",*((double*)(vis->vcomparison)));
                  break;
     };
     XtSetArg(args[i],XtNstring,s);i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vs_type);i++;
     XtSetArg(args[i],XtNhorizDistance,4);i++;
     XtSetArg(args[i],XtNborderWidth,0);i++;
     XtSetArg(args[i],XtNeditType,"edit");i++;
     XtSetArg(args[i],XtNdisplayCaret,False);i++;
     vis->vs_value = XtCreateManagedWidget("vs_value",asciiTextWidgetClass,
                               vis->visselect, args,i);

    }; /* end of if (for condition line) */


    /* Create range entry line if ranges are used */

    if (vis->vuserange) {
     /* Main form widget */
     i=0;
     XtSetArg(args[i],XtNshowGrip,False);i++;
     vis->visfromto = XtCreateManagedWidget("visfromto",formWidgetClass,
                                vis->vispaned, args,i);
     /* 1st label for selection, including name of used type */
     i=0;
     sprintf(s,"Setect (%s) range from",NameOfType[vis->vtype]);
     XtSetArg(args[i],XtNlabel,s);i++;
     XtSetArg(args[i],XtNborderWidth,0);i++;
     vis->vft_1 = XtCreateManagedWidget("vft_1",labelWidgetClass,
                                vis->visfromto, args,i);
     /* Value entry "from" */
     i=0;
     XtSetArg(args[i],XtNlabel,"0000000");i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vft_1);i++;
     XtSetArg(args[i],XtNhorizDistance,1);i++;
     XtSetArg(args[i],XtNborderWidth,1);i++;
     XtSetArg(args[i],XtNeditType,"edit");i++;
     XtSetArg(args[i],XtNdisplayCaret,False);i++;
     vis->vft_from = XtCreateManagedWidget("vft_from",asciiTextWidgetClass,
                               vis->visfromto, args,i);
     /* 2nd label for selection */
     i=0;
     XtSetArg(args[i],XtNlabel," to  ");i++;
     XtSetArg(args[i],XtNborderWidth,0);i++;
     XtSetArg(args[i],XtNfromHoriz,vis->vft_from);i++;
     XtSetArg(args[i],XtNhorizDistance,1);i++;
     vis->vft_2 = XtCreateManagedWidget("vft_2",labelWidgetClass,
                                vis->visfromto, args,i);
     /* Value entry "to" */
     i=0;
     XtSetArg(args[i],XtNlabel,"00000");i++;	/* Get some free space for */
     XtSetArg(args[i],XtNfromHoriz,vis->vft_2);i++; /* the number entry    */
     XtSetArg(args[i],XtNhorizDistance,1);i++;
     XtSetArg(args[i],XtNborderWidth,1);i++;
     XtSetArg(args[i],XtNeditType,"edit");i++;
     XtSetArg(args[i],XtNdisplayCaret,False);i++;
     vis->vft_to = XtCreateManagedWidget("vft_to",asciiTextWidgetClass,
                               vis->visfromto, args,i);
    }; /* end of if(vis->userange) */


    /* Set the currently active layers of the array	*/
    /* (if one-dimensional, there's just one)		*/

    vis->vactive1 = 0;
    if (vis->vnumdims == 1) {
        vis->vactive2 = 0;
    } else {
        vis->vactive2 = 1;
    };

    /* Create a counter paned widget using counter.c		*/
    /* Also remember the pointer to the visualizer structure as	*/
    /* user data in the counter data structure			*/
    /* Don't do anything if ACT_DISPLAY selected!		*/

    if (vis->vdispmode != ACT_DISPLAY) {
        vis->vcounter = create_counter(vis->vnumdims,nullvector,maxvector,
                                       vis->vispaned);
        set_counter_userdata(vis->vcounter,(void*)vis);
    };


    /* Create a viewport widget using display.c			*/
    /* displaying a section through the 1st and 2nd dimension	*/
    /* with zoom factor 6					*/
    /* If one-dimensional, use the computed "dummy" dimensions	*/

    if (vis->vnumdims == 1) {
        if (vis->vdispmode != ACT_DISPLAY) {	/* Make normal size (rect.) */
            vis->vdisplay = create_display(vis->vonewidth,
                                        vis->vdims[0]/vis->vonewidth,
                                        zoom,vis->vispaned,
                                        vis->vdispmode);
        } else {				/* Make square		    */
            vis->vdisplay = create_display(vis->vonewidth,
                                           vis->vonewidth,
                                           zoom,vis->vispaned,
                                           vis->vdispmode);
        };
    } else {
        vis->vdisplay = create_display(vis->vdims[vis->vactive1],
                                       vis->vdims[vis->vactive2],
                                       zoom,vis->vispaned,
                                       vis->vdispmode);
    };
    /* Also remember the visualizer pointer			*/
    set_display_userdata(vis->vdisplay,(void*)vis);
    /* Update restl for Activation Value display		*/
    if (vis->vdispmode == ACT_DISPLAY)
        set_restl(vis->vdisplay,restl);


    /* Realize the parent Widget and set a meaningful title */

    XtRealizeWidget(vis->vparent);
    if (vis->vdispmode == ACT_DISPLAY) {
        sprintf(s,"Activations for %s[0..%d]",
                vis->vname, vis->vdims[0]-1);
    } else {
        sprintf(s,"Visualize %s",vis->vname);
        for (i=0; i<vis->vnumdims; i++)  /* Indicate numer of dimensions */
            strcat(s,"[]");		 /* as "[]" for each dimension	 */
    };
    XStoreName(XtDisplay(vis->vparent), XtWindow(vis->vparent), s);


    /* If it's a value Visualizer, compute the statistics	*/
    /* Else, just leave them blank				*/

    if (vis->vdispmode == GRAY_DISPLAY) {
        do_statistics(vis);
    } else {
        vis->vminval = vis->vmaxval = vis->vadr;
        vis->vavval = vis->vfactorval = 0.0;
    };


    /* Now that all is realized, we can init the display and the counter */
    /* and also init all bitmaps, event functions etc pp		 */

    init_display(vis->vdisplay);
    set_display_functions(vis->vdisplay,
                         VShowPos, NULL, VLeavePos, VMouseClick, VButtonClick);


    /* Also init the counter's bitmaps and the event functions	*/
    /* (if not visualizing activation values!)			*/

    if (vis->vdispmode != ACT_DISPLAY) {    
        init_counter(vis->vcounter);
        set_counter_functions(vis->vcounter,VChangedValue, VChangedState);
        for (i=0; i<vis->vnumdims; i++) {
            char s[512];
            sprintf(s,"Dim %2d",i);
            set_counter_text(vis->vcounter,i,s);
        };
    };


    /* If normal standalone Visualizer used, perform dirty quit.	*/
    /* Otherwise exit gracefully					*/

    if (widget_only == FALSE) {
	XtAddCallback(vis->visquit, XtNcallback, VQuit, (XtPointer) vis);
    } else {
	XtAddCallback(vis->visquit, XtNcallback, VQuitWidget, (XtPointer) vis);
    };

    /* Add callbacks for buttons (comparison only if no user defined fn's) */
 
    if (!(vis->vusefn || vis->vuserange) && (vis->vdispmode==BW_DISPLAY)) {
     XtAddCallback(vis->vs_gtbtn, XtNcallback, VgtButtonCall, (XtPointer) vis);
     XtAddCallback(vis->vs_eqbtn, XtNcallback, VeqButtonCall, (XtPointer) vis);
     XtAddCallback(vis->vs_ltbtn, XtNcallback, VltButtonCall, (XtPointer) vis);


     /* Create our own action record for VvalueCall and add it to the	*/
     /* existing action table						*/

     arec.string = "VvalueCall";
     arec.proc   = VvalueCall;
     XtAppAddActions(XtWidgetToApplicationContext(vis->vparent), &arec, 1);

     /* Make <Return> go to end of line instead creating new line	*/
     /* and call VvalueCall(vis)					*/
     /* Later, we get the address of "vis" from the parameter and can	*/
     /* take appropriate actions then.					*/

     sprintf(s,"<FocusIn>    : display-caret(on) \n\
                <FocusOut>   : display-caret(off) \n\
                <EnterWindow>: display-caret(on) \n\
                <LeaveWindow>: display-caret(off) VvalueCall(%d) \n\
                <Key>Return  : end-of-line() VvalueCall(%d)", vis,vis);
     XtOverrideTranslations(vis->vs_value, XtParseTranslationTable(s));
    }; /* end of if (button callbacks) */


    /* Add action procedures for from/to enty widgets (if range in use) */

    if (vis->vuserange) {
     /* Create our own action record for V{from,to}Call and add it to the*/
     /* existing action table						*/

     arec.string = "VfromCall";
     arec.proc   = VfromCall;
     XtAppAddActions(XtWidgetToApplicationContext(vis->vparent), &arec, 1);
     arec.string = "VtoCall";
     arec.proc   = VtoCall;
     XtAppAddActions(XtWidgetToApplicationContext(vis->vparent), &arec, 1);

     /* Make <Return> go to end of line instead creating new line	*/
     /* and call V{from,to}Call(vis)					*/
     /* Later, we get the address of "vis" from the parameter and can	*/
     /* take appropriate actions then.					*/

     sprintf(s,"<FocusIn>   : display-caret(on) \n\
                <FocusOut>  : display-caret(off) \n\
                <EnterWindow>: display-caret(on) \n\
                <LeaveWindow>: display-caret(off) VfromCall(%d) \n\
                <Key>Return : end-of-line() VfromCall(%d)", vis,vis);
     XtOverrideTranslations(vis->vft_from, XtParseTranslationTable(s));
     sprintf(s,"<FocusIn>   : display-caret(on) \n\
                <FocusOut>  : display-caret(off) \n\
                <EnterWindow>: display-caret(on) \n\
                <LeaveWindow>: display-caret(off) VtoCall(%d) \n\
                <Key>Return : end-of-line() VtoCall(%d)", vis,vis);
     XtOverrideTranslations(vis->vft_to, XtParseTranslationTable(s));
    }; /* end of if (action procedures) */


    /* Display a section along the first & second dimension initially */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

};



/*
 *  void do_visualization(VIStype *VIS)
 *
 *  Does the final initialization and the visualization itself,
 *  after "vis" has been initialized by "Visualize()", "VisualizeRange()"
 *  or "VisualizeUserFn()" etc.
 *  (vis is VIS->VISvis so we need only one parameter)
 *
 */
void do_visualization(VIS)
VIStype        *VIS;
{
    XEvent 		event;
    XDestroyWindowEvent	*ep;
    visualizertype	*vis = (visualizertype *)VIS->VISvis;


    
    /* Initialize the widgets used by the Visualizer */
    
    do_vis_widgets(VIS,FALSE);		/* Normal init, not just Widgets  */
    

    vis->vrunning = TRUE;		/* Set flag to "running"	  */
    vis->vreadoffset = NULL;		/* Init offset for reading socket */


    /* If background process wanted, fork() one */

    if (vis->vforked) {
        debuginfo(("++ forking visualizer...\n"));

        /* Generate unique socket file name from PID & time (usec) */

        sprintf(vis->vdatasockname,"/tmp/visD%06d%07d", getpid(),time(NULL));
        sprintf(vis->vcmdsockname, "/tmp/visC%06d%07d", getpid(),time(NULL));
        debuginfo(("++ socketnames: data:%s commands:%s\n",
		   vis->vdatasockname, vis->vcmdsockname));
        setup_read(VIS);		/*   Set up socket for reading	*/

	
        /* Start background process */ /* !!!!!!! error check ?? !!!!!! */
        
        if (vis->vchildproc = fork()) {	/* We're the parent:		*/
	    
            debuginfo(("++ parent: PID = %d\n",getpid()));
            setup_write(VIS);		/*   Set up socket for writing	*/
            debuginfo(("++ parent: return\n"));
            return;			/*   Continue foreground work	*/
	    
        } else {			/* We're the child:		*/
	    
            debuginfo(("** child: PID= %d\n",getpid()));
            debuginfo(("** child: setup accept\n"));
            setup_accept(VIS);
	    
        };
        debuginfo(("** child: continue\n"));


        /* Add timeout to make sure that the event handler is called	*/
        /*  at least once a second (necessary to get updates through	*/
        /*  the socket even if user is idle!				*/

        XtAppAddTimeOut(VIS->VISapp_con, UPDATE_INTERVAL, VTimestep, VIS);

    };


    /* Let X Windows do the display etc */

    
    do {				/* Repeat until stopped (quit button)*/
	
        if (vis->vforked) {		/* If in background:		*/
            handle_socket(VIS);		/*   Look at socket for reading	*/
        };
        XtAppNextEvent(VIS->VISapp_con,&event);
        XtDispatchEvent(&event);	/* Get Event and handle it	*/
        ep = (XDestroyWindowEvent*) &event;
	
    } while ((ep->type != DestroyNotify) ||   /* Wait until parent destroyed */
             (ep->window != XtWindow(VIS->VISparent)) );

    /* NO MORE VIS->VRUNNING HERE */


    
    /* Done, exit the visualizer and free its resources			   */


    if (vis->vdispmode != ACT_DISPLAY)
        destroy_counter(vis->vcounter);
    destroy_display(vis->vdisplay);
    VIS->VISvis = NULL;			/* Mark VIS as useable again       */

    if (vis->vforked) {			/* If forked, don't return!!!	   */
	
        unlink(vis->vdatasockname);	/* Remove the sockets		   */
        unlink(vis->vcmdsockname);
        XtFree(vis);			/* (terminate whole child process) */
        debuginfo(("** child %d exits...\n",getpid()));
        exit(0);
	
    };

    XtFree(vis);			/* Else return to caller	   */

    return;

}



/********************* WIDGET CREATION FOR USER with own event loop etc ****/


/*
 * VIStype *VisWidgetCommon(Widget parent)
 *
 * Initializes a new Visualizer type and creates a popup window
 * as new parent for the Visualizer (but as child of the given
 * parent widget "parent").
 * Returns the Visualizer type.
 *
 */
VIStype *VisWidgetCommon(parent)
Widget parent;
{
    Widget       popupW;		/* Popup Widget to appear	*/
    VIStype	*VIS;
    
    
    /* Check if given parent is OK */

    if (parent==NULL) {
        printf(/*stderr,*/ "VISUALIZER: Parent is NULL! \n");
        exit(-1);
    };

    /* Create popup shell as new parent for the Visualizer */

    popupW = XtCreatePopupShell("Visualizer",topLevelShellWidgetClass,
				parent,NULL,NULL);
    
    
    /* Create a new visualizer data structure and init the parent part	*/

    VIS = (VIStype*)XtMalloc(sizeof(VIStype));
    VIS->VISparent = popupW;			/* Pass the popup, not	*/
    VIS->VISvis = NULL;				/*  the given parent!	*/

    return(VIS);

}



/*
 * VIStype *VisWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Creates a normal Visualizer Widget with the given parent Widget,
 * returns the complete Visualizer data structure associated with it.
 * Same parameters as Visualize() but no fork/syncupdate flags, no
 * VIS parameter, however an additional "parent" (which must be
 * already initialized!)
 *
 */
VIStype *VisWidget(parent, name, namelen, adr, dims, numdims,
               type, condition, comparison)
Widget	 parent;		/* Parent widget for this Visualizer	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
int      condition;		/* e.g. V_GREATER			*/
void    *comparison;		/* Value to be compared to		*/
{
    VIStype *VIS;
    

    /* Init Visualizer, get parent popup window for it */
    
    VIS = VisWidgetCommon(parent);

    PrepareVis(VIS, name, namelen, adr, dims, numdims,
               type, condition, comparison,
               0,0);				/* No sync, no fork */

    
    /* Create all the Visualizer's widgets */
    
    do_vis_widgets(VIS,TRUE);			/* Use tame Quit function */
    XtPopup(VIS->VISparent,XtGrabNone);
    
    return(VIS);

}



/*
 * VIStype *VisRangeWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, fromvalue, tovalue)
 *
 * Dito with Visualizer type "Range"
 *
 */
VIStype *VisRangeWidget(parent, name, namelen, adr, dims, numdims,
                    type, fromvalue, tovalue)
Widget	 parent;		/* Parent widget for this Visualizer	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
void    *fromvalue;		/* Range "from"				*/
void    *tovalue;		/* Range "to"				*/
{
    VIStype *VIS;
    

    /* Init Visualizer, get parent popup window for it */
    
    VIS = VisWidgetCommon(parent);

    PrepareVisRange(VIS, name, namelen, adr, dims, numdims,
                    type, fromvalue, tovalue,
		    0,0);			/* No sync, no fork */

    
    /* Create all the Visualizer's widgets */
    
    do_vis_widgets(VIS,TRUE);			/* Use tame Quit function */
    XtPopup(VIS->VISparent,XtGrabNone);

    return(VIS);

}



/*
 * VIStype *VisUserFnWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Dito with Visualizer type "User Function"
 *
 */
VIStype *VisUserFnWidget(parent, name, namelen, adr, dims, numdims, type,
                     cmptext, cmptextlen, compare, access, envp, envlen)
Widget	 parent;		/* Parent widget for this Visualizer	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
char     *cmptext;		/* Text for comparison description	*/
ulong    cmptextlen;		/* Length of this text			*/
Boolean  (*compare)();		/* Comparison function			*/
void     *(*access)();		/* Access function			*/
void     *envp;			/* Pointer to user data for function	*/
ulong    envlen;		/* Length of user data			*/
{
    VIStype *VIS;
    

    /* Init Visualizer, get parent popup window for it */
    
    VIS = VisWidgetCommon(parent);

    PrepareVisUserFn(VIS, name, namelen, adr, dims, numdims, type,
                     cmptext, cmptextlen, compare, access, envp, envlen,
		     0,0,0);		/* No sync, no fork, no sync-fn */

    
    /* Create all the Visualizer's widgets */
    
    do_vis_widgets(VIS,TRUE);			/* Use tame Quit function */
    XtPopup(VIS->VISparent,XtGrabNone);

    return(VIS);

}



/*
 * VIStype *VisValuesWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Dito with Visualizer type "Range"
 *
 */
VIStype *VisValuesWidget(parent, name, namelen, adr, dims, numdims,type)
Widget	 parent;		/* Parent widget for this Visualizer	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
ulong    numdims;		/* Number of dimensions			*/
int      type;			/* Type of array elements (int,char,..)	*/
{
    VIStype *VIS;
    

    /* Init Visualizer, get parent popup window for it */
    
    VIS = VisWidgetCommon(parent);

    PrepareVisValues(VIS, name, namelen, adr, dims, numdims,type,
		     0,0);			/* No sync, no fork */

    
    /* Create all the Visualizer's widgets */
    
    do_vis_widgets(VIS,TRUE);			/* Use tame Quit function */
    XtPopup(VIS->VISparent,XtGrabNone);

    return(VIS);

}



/*
 * VIStype *VisActivationWidget(parent, name, namelen, adr, dims,
 *			 	comparison, callfn)
 *
 * Dito with Visualizer type "Activations"
 *
 */
VIStype *VisActivationWidget(parent, name, namelen, adr, dims,
			 comparison, callfn)
Widget	 parent;		/* Parent widget for this Visualizer	*/
char    *name;			/* Name of displayed Array		*/
ulong    namelen;		/* Length of name			*/
void    *adr;			/* Address of Array			*/
int     *dims;			/* Dimensions of Array			*/
void    *comparison;		/* Value to be compared to		*/
void	(*callfn)();		/* Function to call when clicked	*/
{
    VIStype *VIS;
    

    /* Init Visualizer, get parent popup window for it */
    
    VIS = VisWidgetCommon(parent);

    PrepareVisActivation(VIS, name, namelen, adr, dims,
			      comparison, callfn,0,0);	/* No sync, no fork */

    
    /* Create all the Visualizer's widgets */
    
    do_vis_widgets(VIS,TRUE);			/* Use tame Quit function */
    XtPopup(VIS->VISparent,XtGrabNone);

    return(VIS);

}



/********************* UPDATE CALLS ETC ************************************/



/*
 * void UpdateVisWidget(VIStype *VIS)
 *
 * Updates the display of the given Visualizer.
 * Use when the data the Visualizer is displaying have changed.
 *
 */
void UpdateVisWidget(VIS)
VIStype *VIS;
{
    visualizertype *vis = (visualizertype *)VIS->VISvis;

    
    /* Redisplay the Visualizer using the new function & data	*/
    /* Recalculate the statistics if Value Visualizer used	*/
    
    update_shown_data(vis);
    update_shown_condition(vis);
    if (vis->vdispmode == GRAY_DISPLAY) {
	do_statistics(vis);
    };
    display_layer(vis);
}

/*
 *
 * void UpdateForkedVisualizer(VIStype *VIS)
 *
 * Transmit the new contents of the visualized array to the Visualizer
 * running in the background
 *
 */
void UpdateForkedVisualizer(VIS)
VIStype *VIS;
{
    visualizertype *vis = (visualizertype *)VIS->VISvis;
    int  i;
    long l;		/* Length of data to transmit	*/
    long sync;		/* Return of synchronization	*/


    debuginfo(("++ update: called\n"));

    /* Check if there's a Visualizer to be updated */

    if ( !(vis) || !(vis->vforked) ) {
        printf(/*stderr,*/
               "VISUALIZER: No visualizer running in the background!\n");
        return;
    };

/*    printf("++ update: sending magic on socket %d\n",vis->vdatawritesock);
/*
/*
/*    /* Announce incoming data */
/*
/*!!!!!!!    write(vis->vdatawritesock,"sthMaGiC\0",9);*/
/**/


    /* Compute length of array in bytes and send it off */

    l=SizeOfType[vis->vtype];
    for (i=0; i<vis->vnumdims; i++) {
        l = l*(vis->vdims[i]);
    };
    debuginfo(("++ update: sending %d bytes\n",l));
/*!!!!!!!    write(vis->vdatawritesock,l,sizeof(long));*/
    write(vis->vdatawritesock,vis->vadr,l);

    debuginfo(("++ update: done\n"));


    /* Wait for synchronization if flag set (blocking read call!) */

    if (vis->vsyncupdate) {
        debuginfo(("++ update: syncing...\n"));
        read(vis->vcmdwritesock,&sync,sizeof(sync));
        debuginfo(("++ update: sync done, return value = %d\n",sync));
    };

    return;
}



/*
 *
 * void ChangeUserFn(VIStype *VIS, Boolean (*newfn)(), char* str, ulong strl,
 *                   void *dat, long dlen)
 *
 * Transmit new data for the user function of the Visualizer in the background.
 *   The new function to be called is pointed to by "fn",
 * the string describing the function is "str", "dat" points to the
 * user data to be transmitted and "dlen" is the length of the user data.
 * "str" has to contain one occurence of "%s" which is automatically replaced
 * by the name of the array being visualized. Its length is given in "strl"
 * which is ignored if it is less than one.
 *
 * If "fn" is NULL then no function is given and the old one is used further.
 * If "str" is NULL the string isn't changed and if "dat" is NULL, no
 * data is transmitted.
 *
 * This function requires that the Visualizer uses a User defined Functio
 * (i.e. was called using VisualizeUserFn()) and is running in the background. 
 *
 */
void ChangeUserFn(VIS, newfn, str, strl, dat, dlen)
VIStype  *VIS;		/* Our Visualizer				*/
Boolean (*newfn)(); 	/* New user Function to be called or NULL	*/
char     *str;		/* String describing the function		*/
ulong     strl;		/* Length of this string			*/
void     *dat;		/* User Data					*/
long      dlen;		/* Length of User Data				*/
{
    visualizertype *vis  = (visualizertype *)VIS->VISvis;
    long            sync;		/* Return of synchronization	*/
    long            slen;		/* Internal string length	*/


    /* Check if there's a Visualizer to be updated */

    if ( !(vis) || !(vis->vforked) || !(vis->vusefn) ) {
        printf(/*stderr,*/
               "VISUALIZER: No user function visualizer running in the background!\n");
        return;
    };


    /* Check if string given and remember its length */

    if (str == NULL) {
        strl = 0;
    } else if (strl <1) {
            strl = strlen(str);
    };
    slen = (long)strl;


    debuginfo(("VISUALIZER: Changing user function to %d, description '%s', %d bytes of user data\n",newfn,str,dlen));


    /* Send fn-address, length of string, string, length of data, user data */

    write(vis->vcmdwritesock,&newfn,sizeof(void*));
    write(vis->vcmdwritesock,&slen,sizeof(slen));
    if (slen) write(vis->vcmdwritesock,str,slen);
    write(vis->vcmdwritesock,&dlen,sizeof(dlen));
    if (dlen) write(vis->vcmdwritesock,dat,dlen);

    debuginfo(("++ change: done\n"));


    /* Wait for synchronization if flag set (blocking read!) */

    if (vis->vsyncchfn) {
        debuginfo(("++ change: syncing...\n"));
        read(vis->vcmdwritesock,&sync,sizeof(sync));
        debuginfo(("++ change: sync done, return value = %d\n",sync));
    };

    return;
}



/*
 * void display_layer(vizualizertype *vis)
 *
 * Display a section through the array along the dimensions that
 * are found in vis->vactive{1,2}  (special case for one-dimensional)
 */

void display_layer(vis)
visualizertype *vis;
{
    int x[MAXDIMS];
    int i;
    int zoom;
    void    *(*access)() = vis->vaccfn;	/* Access method for array	  */
    Boolean (*compare)() = vis->vcmpfn;	/* Comparison method for array	  */
    int     *vdims       = vis->vdims;	/* Local copies of vis-variables  */
    void    *vadr        = vis->vadr;	/* (for speed)			  */
    int      vnumdims    = vis->vnumdims;
    int      d1          = vis->vactive1;/* Dimensions along which to cut */
    int      d2          = vis->vactive2;
    void    *secondparam;		/* Second parameter for comparsion*/


    debuginfo(("VISUALIZER: Displaying layer along x%d and x%d\n",d1,d2));


    /* Create a new display suitable for the new cut			*/
    /* If one-dimensional, use vonewidth instead of vdims[{d1,d2}]	*/

    zoom = get_zoom(vis->vdisplay);
    if (vis->vnumdims == 1) {
        if (vis->vdispmode == ACT_DISPLAY) { 	/* Act.Disp: square!	*/
            resize_display(vis->vdisplay,
                           vis->vonewidth, vis->vonewidth,
                           zoom);
        } else {				/* Else: not necessarily...*/
            resize_display(vis->vdisplay,
                           vis->vonewidth,
			   vis->vdims[0] / vis->vonewidth,
                           zoom);
        }
    } else {
        resize_display(vis->vdisplay, vdims[d1],vdims[d2],zoom);
    };


    /* Set those dimension cuts to the counter values (if no activation vis.)*/

    if (vis->vdispmode != ACT_DISPLAY) {
        for (i=0; i<vnumdims; i++) {
            x[i] = get_counter(vis->vcounter, i);
        };
    } else {
        x[0]= *((long*)(vis->venvp));
    };


    /* If using user functions, set the second parameter of the	*/
    /* compare function to point to the user environment	*/
    /* If grayscales are visualized, take the visualizer as arg.*/
    /* Else give the comparison value as second argument	*/

    if (vis->vusefn) {
        secondparam = vis->venvp;
    } else if (vis->vdispmode == GRAY_DISPLAY) {
        secondparam = vis;
    } else {
        secondparam = vis->vcomparison;
    };




    /* Set the array display pixels according to the values in the array */
    /* For one dimensional arrays, display them as nearly-square 2d-	 */
    /* arrays								 */
    /* Note: For a Value Visualizer, the compare function does the	 */
    /* mapping of values to grayscale					 */

    if (vnumdims == 1) {/* One dimensional: display as 2-dim modulo vonewidth*/
        int x;
        register int m = vis->vonewidth;
        register int l = vdims[0];

        for (x=0; x<l; x++) {
             set_pixel(vis->vdisplay, x % m, x / m,
                       (*compare)((*access)(vadr,&x,vdims,vnumdims),
                                  vis->vcondition,
                                  secondparam)
                      );
        }; /* REM: that's no problem with an activation visualizer since */
           /*      we don't access the stuffing pixels anyway, just <= l */

    } else {           /* At-least-2-dimensional: display as usual	*/

        for (x[d1]= 0; x[d1]<vdims[d1]; x[d1]++) {
            for (x[d2]= 0; x[d2]<vdims[d2]; x[d2]++) {

            /* Set a pixel in the 2D Visualization array according	*/
            /* to the value of the corresponding element in the array	*/

                set_pixel(vis->vdisplay,x[d1],x[d2],
                          (*compare)((*access)(vadr,x,vdims,vnumdims),
                                     vis->vcondition,
                                     secondparam)
                         );
            };
        };
    };

    update_image(vis->vdisplay);
}



/*
 * void VShowPos(displaytype *d, int x, int y);
 *
 * Called at mouse event to show position of pointer in display
 * (which is given in x and y)
 */

void VShowPos(d,x,y)
displaytype *d;
int x,y;
{
    int  i;
    Boolean first = True;

    /* Get the visualizer to which this display belongs */
    visualizertype *vis = (visualizertype *)d->userdata;


    /* Set the counters responsible for the active dimensions	*/
    /* to the new values.					*/
    /* If one-dimensional, compute the "real" index instead	*/
    /* If Activation Visualizer, use venvp instead of counter	*/

    if (vis->vnumdims == 1) { /* Just one counter */
        if (vis->vdispmode != ACT_DISPLAY) {
            set_counter(vis->vcounter,0, x + y*vis->vonewidth);	
        } else {
            long l = x + y*vis->vonewidth;
            if (l>=vis->vdims[0])
                l=vis->vdims[0]-1;	/* Check boundaries		*/
            *((long*)vis->venvp) = l;	/* Remember position in venvp	*/
        };
    } else {
        set_counter(vis->vcounter,vis->vactive1,x);
        set_counter(vis->vcounter,vis->vactive2,y);
    };

    update_shown_data(vis);
    
}


/*
 * void VButtonClick(displaytype *d)
 *
 * Called when button of array display clicked
 */

void VButtonClick(d)
displaytype *d;
{
    /* Get the visualizer to which this display belongs */
    visualizertype *vis = (visualizertype *)d->userdata;
    fd_set  fdset;		/* File descriptors for selection	*/
    struct  timeval tv;		/* Timeout interval			*/
    long    sync;		/* For synchronization with callee	*/


    debuginfo(("VISUALIZER: ButtonClick\n"));


    /* If synchronization is wanted, send a sync message back	*/
    /* (Provided that one is awaited)				*/
    /* Remember that our read socket is connected to the	*/
    /* write socket of the parent program (which is waiting)!	*/

    /*!!!!!!! determine which function is waiting (update/changefn) ?? */

    if (vis->vsyncchfn || vis->vsyncupdate) {
        tv.tv_usec = 0;
        tv.tv_sec  = 0;		/* Add timeout for selection -> poll! */
        FD_ZERO(&fdset);
        FD_SET(vis->vcmdreadsock,&fdset);

        /* Look if socket ready for writing */

        if (select(32, NULL, &fdset, NULL, &tv) > 0) {
            debuginfo(("** button: write awaited\n"));
            if (FD_ISSET(vis->vcmdreadsock,&fdset)) {
                sync = 1;			/*!!!!!!! meaningful value? */
                debuginfo(("** button: sending sync, value = %d\n",sync));
                write(vis->vcmdreadsock,&sync,sizeof(sync));
                debuginfo(("** button: sync sent\n"));
            } else {
                debuginfo(("** button: but not for us\n"));
            };
        };
    };

}


/*
 * void VMouseClick(displaytype *d, int x, int y, int b)
 *
 * Called at mouse button event to zoom in/out
 * (Coordinates in x and y, pressed button number in b)
 * If Activation Display, also call user defined function (if any)
 */

void VMouseClick(d,x,y,b)
displaytype *d;
int x,y,b;
{
    char pixel;
    int  zoom;
    Arg a[2];
    /* Get the visualizer to which this display belongs */
    visualizertype *vis = (visualizertype *)d->userdata;


    switch (b) {
    case Button1:		/* Zoom in at the given point		*/
             zoom = get_zoom(vis->vdisplay);
             zoom++;
             set_zoom(vis->vdisplay,zoom);
             center_at(vis->vdisplay,x,y);
             update_image(vis->vdisplay);
             debuginfo(("VISUALIZER: MouseClick: Button 1 clicked at (%d,%d), zoomed in to %d\n",x,y,zoom));
         break;
    case Button2:
             if (vis->vdispmode == ACT_DISPLAY) {
                 if (vis->vcallfn) {			/* Call fn defined? */
                     long l = y*(vis->vonewidth)+x;	/* Compute position */
                     if (l<vis->vdims[0])		/* Inside array?    */
                         (*(vis->vcallfn))(vis->vVIS,l);/* Call it!         */
                 };
             };
             debuginfo(("VISUALIZER: MouseClick: Button 2 clicked at (%d,%d), (NO ACTION) \n",x,y));
         break;
    case Button3:		/* Zoom out at the given point		*/
             zoom = get_zoom(vis->vdisplay);
             zoom--; if (zoom==0) zoom=1;    /* Zoom factor must be >0	*/
             set_zoom(vis->vdisplay,zoom);
             center_at(vis->vdisplay,x,y);
             update_image(vis->vdisplay);
             debuginfo(("VISUALIZER: MouseClick: Button 3 clicked at (%d,%d), zoomed out to %d\n",x,y,zoom));
         break;
    };

}


/* Called when mouse leaves array */
void VLeavePos(d)
displaytype *d;
{
/* do nothing... */
}


/* Quit the visualization */

void VQuit(w, vis_id, call_data)
Widget w;
XtPointer vis_id, call_data;
{
    visualizertype *vis = (visualizertype *)(vis_id);

    vis->vrunning = FALSE;		/* Flag as "no more running" */
    XtDestroyWidget(vis->vparent);	/* Close the widget hierarchy*/
    debuginfo(("VISUALIZER: Quit called\n"));

}


/*
 * void VQuitWidget(Widget w, XtPointer vis_id, call_data)
 *
 * Quit the visualization,
 * pop down the parent (popup!) for Visualizer Widget, deallocate all
 * (except the VIS struct itself, so it's possible to tell that the
 * Visualizer has been quit by checking if VIS->VISvis==NULL)
 *
 */
void VQuitWidget(w, vis_id, call_data)
Widget w;
XtPointer vis_id, call_data;
{
    visualizertype *vis = (visualizertype *)(vis_id);
    VIStype        *VIS = vis->vVIS;

    vis->vrunning = FALSE;		/* Flag as "no more running"	*/
    XtPopdown(vis->vparent);		/* Pop the widget down		*/
    if (vis->vdispmode != ACT_DISPLAY)	
        destroy_counter(vis->vcounter);	/* Free resources		*/
    destroy_display(vis->vdisplay);
    XtDestroyWidget(vis->vparent);	/* Destroy Widget hierarchy	*/
    XtFree(vis);
    VIS->VISvis = NULL;			/* Mark VIS as "exited"		*/

}


/*
 * void VChangedState(countertype *ct, int n_set, int n_reset)
 *
 * Called when state of counter changes. Number of newly set dimension
 * (counter) in n_set, number of reset dimension in n_reset.
 */

void VChangedState(ct, n_set, n_reset)
countertype *ct;
int n_set,n_reset;
{
    Boolean states[MAXDIMS];
    int t;
    /* Get the visualizer to which this counter belongs */
    visualizertype *vis = (visualizertype *)ct->userdata;

    /* Update the information about the active layers			*/
    /* (This uses the fact that only 2 layers are active at once!	*/

    if (n_reset == vis->vactive1) {	/* active1 changed -> set new value */
        vis->vactive1 = n_set;
    } else {				/* else: active2 changed*/
        vis->vactive2 = n_set;
    };
    if (vis->vactive1 > vis->vactive2) {/* Sort dimensions to prevent	*/
        t=vis->vactive2;		/* cases like (1,3) being shown	*/
        vis->vactive2=vis->vactive1;	/* as (3,1) [gets confusing]	*/
        vis->vactive1=t;
    };

    debuginfo(("VISUALIZER: ChangedState: x%d and x%d now active\n", vis->vactive1, vis->vactive2));

    display_layer(vis);

}



/*
 * void VChangedValue(countertype *ct, int nr, int old, int new)
 * 
 * Called when value of counter changes. Updates the displayed
 * information accordingly (just the value if the section is already
 * being displayed / a new section if necessary).
 * Number of the affected dimension in nr, old and new value in (guess).
 */

void VChangedValue(ct, nr, old, new)
countertype *ct;
int nr,old,new;
{
    Boolean states[MAXDIMS];
    int i, x, y;
    /* Get the visualizer to which this counter belongs */
    visualizertype *vis = (visualizertype *)ct->userdata;


    debuginfo(("VISUALIZER: ChangedValue: Value of counter %d is now %d, was %d\n",nr, new, old));

    /* If this counter is inactive, it means we're just displaying	*/
    /* the dimension that it counts. So let's show the new slice	*/

    if ( ! get_state(vis->vcounter,nr) ) {
        debuginfo(("VISUALIZER:               That's a new layer to be shown\n"));
        display_layer(vis);
        update_shown_data(vis);		/* Update value & position */
    } else {
        debuginfo(("VISUALIZER:               We're already in that layer\n"));
        update_shown_data(vis);		/* Update value & position */
    }

}



/*
 * void update_shown_data(visualizertype *vis)
 *
 * Updates the data (i.e. the current position in the array and the
 * value at that position) in the data window
 *
 */
void update_shown_data(vis)
visualizertype *vis;
{
    char s[128];
    Arg  a[1];
    int  xx[MAXDIMS];
    int  i;
    long   val_int;	/* Values (for integer and float data) */
    double val_float;

    if (vis->vdispmode != ACT_DISPLAY) {    
        get_all_counters(vis->vcounter,xx);	/* Get values of all counters*/
    } else {
        xx[0] = *((long*)vis->venvp);		/* Act.Disp: Value in venvp! */
    };

    /* Get array element using the appropriate function */
    switch (vis->vtype) {
    case V_CHAR: val_int = *(char*)
                       (*(vis->vaccfn))(vis->vadr,xx,vis->vdims,vis->vnumdims);
                 break;
    case V_INT:  val_int = *(int*)
                       (*(vis->vaccfn))(vis->vadr,xx,vis->vdims,vis->vnumdims);
                 break;
    case V_LONG: val_int = *(long*)
                       (*(vis->vaccfn))(vis->vadr,xx,vis->vdims,vis->vnumdims);
                 break;
    case V_FLOAT:val_float = *(float*)
                       (*(vis->vaccfn))(vis->vadr,xx,vis->vdims,vis->vnumdims);
                 break;
    case V_DOUBLE:val_float= *(double*)
                       (*(vis->vaccfn))(vis->vadr,xx,vis->vdims,vis->vnumdims);
                 break;
    };


    /* Construct the string to be displayed from position & value	*/
    /* Use different display modes for integer (char,int,long) and	*/
    /* floating point (float,double) data!				*/

    if (vis->vtype <= V_LONG) {	/* (this uses the order of V_xxx !!) */
        switch (vis->vnumdims) {
        case 1: sprintf(s,"%s[%d] = %d", vis->vname,
                          xx[0], val_int);
                break;
        case 2: sprintf(s,"%s[%d][%d] = %d", vis->vname,
                          xx[0], xx[1],val_int);
                break;
        case 3: sprintf(s,"%s[%d][%d][%d] = %d", vis->vname,
                          xx[0], xx[1], xx[2],val_int);
                break;
        case 4: sprintf(s,"%s[%d][%d][%d][%d] = %d", vis->vname,
                          xx[0], xx[1], xx[2], xx[3],val_int);
                break;
        };
    } else {
        switch (vis->vnumdims) {
        case 1: sprintf(s,"%s[%d] = %g", vis->vname,
                          xx[0], val_float);
                break;
        case 2: sprintf(s,"%s[%d][%d] = %g", vis->vname,
                          xx[0], xx[1],val_float);
                break;
        case 3: sprintf(s,"%s[%d][%d][%d] = %g", vis->vname,
                          xx[0], xx[1], xx[2],val_float);
                break;
        case 4: sprintf(s,"%s[%d][%d][%d][%d] = %g", vis->vname,
                          xx[0], xx[1], xx[2], xx[3],val_float);
                break;
        }
    };

    XtSetArg(a[0], XtNlabel, s);	/* Display the result */
    XtSetValues(vis->visshow,a,1);

};


/*
 * void update_shown_condition(visualizertype *vis)
 *
 * Updates the condition after which data is highlit
 *
 */
void update_shown_condition(vis)
visualizertype *vis;
{
    char s[256];	/* Resulting string to be shown	*/
    char s_cond[32];	/* Condition ("<" etc)		*/
    char s_comp[32];	/* Value to be compared to	*/
    char s_from[32];	/* From-value in range		*/
    char s_to[32];	/* To-value in range		*/
    Arg  a[3];


    if ((vis->vdispmode != BW_DISPLAY) &&
        (vis->vdispmode != ACT_DISPLAY) ) {/* Don't bother with this if	*/
        return;				   /* value Visualizer is used	*/
    };

    /* Use a text corresponding to the display useage */

    if (vis->vusefn) {

     /* Just show "vcmptext" & array name if user defined functions active */
     sprintf(s,vis->vcmptext,vis->vname);

    } else if (vis->vuserange) {

     /* Show active range of numbers being diplayed (if range used) */
     switch (vis->vtype) {
     case V_CHAR: sprintf(s_from,"%d",*(char*)(vis->vfromval));
                  sprintf(s_to  ,"%d",*(char*)(vis->vtoval));
                  break;
     case V_INT:  sprintf(s_from,"%d",*(int*)(vis->vfromval));
                  sprintf(s_to  ,"%d",*(int*)(vis->vtoval));
                  break;
     case V_LONG: sprintf(s_from,"%d",*(long*)(vis->vfromval));
                  sprintf(s_to  ,"%d",*(long*)(vis->vtoval));
                  break;
     case V_FLOAT:sprintf(s_from,"%g",*(float*)(vis->vfromval));
                  sprintf(s_to  ,"%g",*(float*)(vis->vtoval));
                  break;
     case V_DOUBLE:sprintf(s_from,"%g",*(double*)(vis->vfromval));
                   sprintf(s_to  ,"%g",*(double*)(vis->vtoval));
                  break;
     };
     if (vis->vfrominfinite) sprintf(s_from,"-inf");	/* Handle..	*/
     if (vis->vtoinfinite)   sprintf(s_to  ,"+inf");	/* .."infinite"	*/

     sprintf(s,"Displayed Range: %s <= %s[...] <= %s",s_from,vis->vname,s_to);

     /* Also set text in value entry widgets! */

     XtSetArg(a[0], XtNstring, s_from);
     XtSetValues(vis->vft_from,a,1);
     XtSetArg(a[0], XtNstring, s_to);
     XtSetValues(vis->vft_to,a,1);

    } else if (vis->vdispmode == ACT_DISPLAY) {

        /* Activation Visualizer: Just show comparison with "=" & value */
        sprintf(s,"Active: %s[] = %d",vis->vname,*(long*)(vis->vcomparison));

    } else {

     /* Show whole condition if conditions used */
     /* Select the array access method according to the element type */
     switch (vis->vtype) {
     case V_CHAR: sprintf(s_comp,"%d",*(char*)(vis->vcomparison));
                  break;
     case V_INT:  sprintf(s_comp,"%d",*(int*)(vis->vcomparison));
                  break;
     case V_LONG: sprintf(s_comp,"%d",*(long*)(vis->vcomparison));
                  break;
     case V_FLOAT:sprintf(s_comp,"%g",*(float*)(vis->vcomparison));
                  break;
     case V_DOUBLE:sprintf(s_comp,"%g",*(double*)(vis->vcomparison));
                  break;
     };
     switch (vis->vcondition) {
     case V_GREATER: sprintf(s_cond,">"); break;
     case V_EQUAL  : sprintf(s_cond,"="); break;
     case V_LESS   : sprintf(s_cond,"<"); break;
     };
     sprintf(s,"Displayed Condition: %s[...] %s %s",vis->vname,s_cond,s_comp);

    };

    XtSetArg(a[0], XtNlabel, s);	/* Display the result */
    XtSetValues(vis->viscond,a,1);

};



/*
 * <type> * access_<type>(void *vadr, int *x, int *d, ulong numdims)
 * 
 * Access functions for each type of array element (array address in vadr,
 * coordinates in x, dimension sizes in d, number of dimensions in numdims).
 * Return int value for display
 */


void *access_char(vadr,x,d,numdims)
void *vadr;		/* Address of array start		*/
int  *x;		/* Coordinates				*/
int  *d;		/* Sizes of dimensions			*/
ulong numdims;		/* Number of dimensions			*/
{
    char *array = (char*)vadr;

    switch(numdims) {
    case 1: { return(&(array[ACCESS1(x[0],d[0])]));
            };
    case 2: { return(&(array[ACCESS2(x[0],d[0],x[1],d[1])]));
            };
    case 3: { return(&(array[ACCESS3(x[0],d[0],x[1],d[1],x[2],d[2])]));
            };
    case 4: { return(&(array[ACCESS4(x[0],d[0],x[1],d[1],x[2],d[2],x[3],d[3])]));
            };
    };
};


void *access_int(vadr,x,d,numdims)
void *vadr;		/* Address of array start		*/
int  *x;		/* Coordinates				*/
int  *d;		/* Sizes of dimensions			*/
ulong numdims;		/* Number of dimensions			*/
{
    int *array = (int *)vadr;

    switch(numdims) {
    case 1: { return(&(array[ACCESS1(x[0],d[0])]));
            };
    case 2: { return(&(array[ACCESS2(x[0],d[0],x[1],d[1])]));
            };
    case 3: { return(&(array[ACCESS3(x[0],d[0],x[1],d[1],x[2],d[2])]));
            };
    case 4: { return(&(array[ACCESS4(x[0],d[0],x[1],d[1],x[2],d[2],x[3],d[3])]));
            };
    };
};


void *access_long(vadr,x,d,numdims)
void *vadr;		/* Address of array start		*/
int  *x;		/* Coordinates				*/
int  *d;		/* Sizes of dimensions			*/
ulong numdims;		/* Number of dimensions			*/
{
    long *array = (long*)vadr;

    switch(numdims) {
    case 1: { return(&(array[ACCESS1(x[0],d[0])]));
            };
    case 2: { return(&(array[ACCESS2(x[0],d[0],x[1],d[1])]));
            };
    case 3: { return(&(array[ACCESS3(x[0],d[0],x[1],d[1],x[2],d[2])]));
            };
    case 4: { return(&(array[ACCESS4(x[0],d[0],x[1],d[1],x[2],d[2],x[3],d[3])]));
            };
    };
};


void *access_float(vadr,x,d,numdims)
void *vadr;		/* Address of array start		*/
int  *x;		/* Coordinates				*/
int  *d;		/* Sizes of dimensions			*/
ulong numdims;		/* Number of dimensions			*/
{
    float *array = (float*)vadr;

    switch(numdims) {
    case 1: { return(&(array[ACCESS1(x[0],d[0])]));
            };
    case 2: { return(&(array[ACCESS2(x[0],d[0],x[1],d[1])]));
            };
    case 3: { return(&(array[ACCESS3(x[0],d[0],x[1],d[1],x[2],d[2])]));
            };
    case 4: { return(&(array[ACCESS4(x[0],d[0],x[1],d[1],x[2],d[2],x[3],d[3])]));
            };
    };
};


void *access_double(vadr,x,d,numdims)
void *vadr;		/* Address of array start		*/
int  *x;		/* Coordinates				*/
int  *d;		/* Sizes of dimensions			*/
ulong numdims;		/* Number of dimensions			*/
{
    double *array = (double*)vadr;

    switch(numdims) {
    case 1: { return(&(array[ACCESS1(x[0],d[0])]));
            };
    case 2: { return(&(array[ACCESS2(x[0],d[0],x[1],d[1])]));
            };
    case 3: { return(&(array[ACCESS3(x[0],d[0],x[1],d[1],x[2],d[2])]));
            };
    case 4: { return(&(array[ACCESS4(x[0],d[0],x[1],d[1],x[2],d[2],x[3],d[3])]));
            };
    };
};


/*
 * Boolean compare_<type>(void *a, int cond, void *b)
 *
 * Comparison functions for each type of array element.
 * Typecast the void* back to <type>* and dereference it.
 * Return Boolean value depending on condition (V_GREATER,...)
 */


Boolean compare_char(a,cond,b)
void *a;
int  cond;
void *b;
{
    switch(cond) {
    case V_GREATER: return((Boolean)(*(char*)a > *(char*)b));
    case V_EQUAL:   return((Boolean)(*(char*)a== *(char*)b));
    case V_LESS:    return((Boolean)(*(char*)a < *(char*)b));
    };
}

Boolean compare_int(a,cond,b)
void *a;
int  cond;
void *b;
{
    switch(cond) {
    case V_GREATER: return((Boolean)(*(int*)a > *(int*)b));
    case V_EQUAL:   return((Boolean)(*(int*)a== *(int*)b));
    case V_LESS:    return((Boolean)(*(int*)a < *(int*)b));
    };
}

Boolean compare_long(a,cond,b)
void *a;
int  cond;
void *b;
{
    switch(cond) {
    case V_GREATER: return((Boolean)(*(long*)a > *(long*)b));
    case V_EQUAL:   return((Boolean)(*(long*)a== *(long*)b));
    case V_LESS:    return((Boolean)(*(long*)a < *(long*)b));
    };
}


Boolean compare_float(a,cond,b)
void *a;
int  cond;
void *b;
{
    switch(cond) {
    case V_GREATER: return((Boolean)(*(float*)a > *(float*)b));
    case V_EQUAL:   return((Boolean)(*(float*)a== *(float*)b));
    case V_LESS:    return((Boolean)(*(float*)a < *(float*)b));
    };
}


Boolean compare_double(a,cond,b)
void *a;
int  cond;
void *b;
{
    switch(cond) {
    case V_GREATER: return((Boolean)(*(double*)a > *(double*)b));
    case V_EQUAL:   return((Boolean)(*(double*)a== *(double*)b));
    case V_LESS:    return((Boolean)(*(double*)a < *(double*)b));
    };
}


/*
 * Boolean compare_range_<type>(void *a, int cond, void *b)
 *
 * Comparison functions for each type of array element for use with ranges
 * Typecast the void* back to <type>* and dereference it.
 * Return Boolean value depending on condition "vfromvalue <= a <= vtovalue"
 * Handle infinite value using "v{from,to}infinite"
 * [ignore parameter "cond"]
 * TRICK: b here is a pointer to the visualizer struct (to make it possible
 *        to access the from and to values!!!)
 */


Boolean compare_range_char(a,cond,b)
void *a;
int  cond;
visualizertype *b;
{
    char f = *(char*)(b->vfromval);
    char x = *(char*)a;
    char t = *(char*)(b->vtoval);;

    return( ( (f <= x) || (b->vfrominfinite) )  &&
            ( (x <= t) || (b->vtoinfinite) )
          );
}

Boolean compare_range_int(a,cond,b)
void *a;
int  cond;
visualizertype *b;
{
    int f = *(int*)(b->vfromval);
    int x = *(int*)a;
    int t = *(int*)(b->vtoval);;

    return( ( (f <= x) || (b->vfrominfinite) )  &&
            ( (x <= t) || (b->vtoinfinite) )
          );

}

Boolean compare_range_long(a,cond,b)
void *a;
int  cond;
visualizertype *b;
{
    long f = *(long*)(b->vfromval);
    long x = *(long*)a;
    long t = *(long*)(b->vtoval);;

    return( ( (f <= x) || (b->vfrominfinite) )  &&
            ( (x <= t) || (b->vtoinfinite) )
          );

}


Boolean compare_range_float(a,cond,b)
void *a;
int  cond;
visualizertype *b;
{
    float f = *(float*)(b->vfromval);
    float x = *(float*)a;
    float t = *(float*)(b->vtoval);;

    return( ( (f <= x) || (b->vfrominfinite) )  &&
            ( (x <= t) || (b->vtoinfinite) )
          );

}


Boolean compare_range_double(a,cond,b)
void *a;
int  cond;
visualizertype *b;
{
    double f = *(double*)(b->vfromval);
    double x = *(double*)a;
    double t = *(double*)(b->vtoval);;

    return( ( (f <= x) || (b->vfrominfinite) )  &&
            ( (x <= t) || (b->vtoinfinite) )
          );

}



/*
 * char value_<type>(void *a, int cond, void *b)
 *
 * Mapping between array element value and display pixel value.
 * Typecast the void* back to <type>* and dereference it.
 * Ignore condition (not needed for this)
 * TRICK: b here is a pointer to the visualizer struct (to make it possible
 *        to access the statistical values)
 */


char value_char(a,cond,vis)
void *a;
int  cond;
visualizertype *vis;
{
    return((char) ( 5 * vis->vfactorval * (*(char*)a - *(char*)vis->vminval)));
}

char value_int(a,cond,vis)
void *a;
int  cond;
visualizertype *vis;
{
    return((char) ( 5 * vis->vfactorval * (*(int*)a - *(int*)vis->vminval)));
}

char value_long(a,cond,vis)
void *a;
int  cond;
visualizertype *vis;
{
    return((char) ( 5 * vis->vfactorval * (*(long*)a - *(long*)vis->vminval)));
}


char value_float(a,cond,vis)
void *a;
int  cond;
visualizertype *vis;
{
    return((char) ( 5.0 * vis->vfactorval * (*(float*)a - *(float*)vis->vminval)));
}


char value_double(a,cond,vis)
void *a;
int  cond;
visualizertype *vis;
{
    return((char) ( 5.0 * vis->vfactorval * (*(double*)a - *(double*)vis->vminval)));
}



/*
 * V{gt,eq,lt}ButtonCall(Widget w, XtPointer cmp_number, call_data)
 *
 * Callbacks for the comparison operator setting buttons.
 * Change the comparison operator accordingly
 */

void VgtButtonCall(w, vis_id, call_data)
Widget w;
XtPointer vis_id, call_data;
{
    /* Get the appropriate "vis" structure from the	*/
    /* user call data and set the condition		*/		

    visualizertype *vis = (visualizertype *)(vis_id);
    vis->vcondition = V_GREATER;

    debuginfo(("VISUALIZER: Comparison operator '>' selected\n"));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}

void VeqButtonCall(w, vis_id, call_data)
Widget w;
XtPointer vis_id, call_data;
{
    /* Get the appropriate "vis" structure from the	*/
    /* user call data and set the condition		*/		

    visualizertype *vis = (visualizertype *)(vis_id);
    vis->vcondition = V_EQUAL;

    debuginfo(("VISUALIZER: Comparison operator '=' selected\n"));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}

void VltButtonCall(w, vis_id, call_data)
Widget w;
XtPointer vis_id, call_data;
{
    /* Get the appropriate "vis" structure from the	*/
    /* user call data and set the condition		*/		

    visualizertype *vis = (visualizertype *)(vis_id);
    vis->vcondition = V_LESS;

    debuginfo(("VISUALIZER: Comparison operator '<' selected\n"));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}


/*
 * VvalueCall(Widget w, XEvent *e, String *s, Cardinal *c)
 *
 * Gets called as action procedure by the "vs_value" widget
 * whenever the return key is pressed there.
 * Updates the comparison value.
 */

void VvalueCall(w, e, s, c)
Widget w;
XEvent *e;
String *s;
Cardinal *c;
{
    Arg a[1];
    char *val_string;
    visualizertype *vis;


    /* Convert the string we got from our translation table call entry	*/
    /* to the address of the appropriate "vis" structure.		*/
    /* Get the text string from the widget				*/

    vis = (visualizertype *)(atol(s[0]));
    XtSetArg(a[0], XtNstring, &val_string);
    XtGetValues(vis->vs_value,a,1);


    /* Set the new comparison element accordingly, casting the void*	*/
    /* to the right type						*/

    switch (vis->vtype) {
    case V_CHAR: *((char*)(vis->vcomparison)) = (char) atoi(val_string);
                 break;
    case V_INT:  *((int*) (vis->vcomparison)) = (int)  atoi(val_string);
                 break;
    case V_LONG: *((long*)(vis->vcomparison)) = (long) atol(val_string);
                 break;
    case V_FLOAT:*((float*)(vis->vcomparison))  = (float) atof(val_string);
                 break;
    case V_DOUBLE:*((double*)(vis->vcomparison))= (double)atof(val_string);
                 break;
    };

    debuginfo(("VISUALIZER: Comparison value '%s' selected\n",val_string));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}


/*
 * V{from,to}Call(Widget w, XEvent *e, String *s, Cardinal *c)
 *
 * Gets called as action procedure by the "vft{from,to}" widget
 * whenever the return key is pressed there.
 * Updates the range. [Functions the same way as VvalueCall, see there]
 */

void VfromCall(w, e, s, c)
Widget w;
XEvent *e;
String *s;
Cardinal *c;
{
    Arg a[1];
    char *val_string;
    char *vs;
    visualizertype *vis;

    vis = (visualizertype *)(atol(s[0]));
    XtSetArg(a[0], XtNstring, &val_string);
    XtGetValues(vis->vft_from,a,1);

    /* Check if infinity entered, i.e. zero-length entry or "inf" */

    vs=val_string;

    vis->vfrominfinite = (vs[0] == '\0');

    while ( (*vs) && (!(vis->vfrominfinite)) ) {
        if (!(strncasecmp(vs,"inf",3))) vis->vfrominfinite = TRUE;
        vs++;
    };


    /* Set the new comparison element accordingly, casting the void*	*/
    /* to the right type  (in case of infinity this data is just	*
    /* ignored in the comparison later on)				*/

    switch (vis->vtype) {
    case V_CHAR: *((char*)(vis->vfromval))= (char) atoi(val_string);
                 break;
    case V_INT:  *((int*) (vis->vfromval))= (int)  atoi(val_string);
                 break;
    case V_LONG: *((long*)(vis->vfromval))= (long) atol(val_string);
                 break;
    case V_FLOAT:*((float*)(vis->vfromval))  = (float) atof(val_string);
                 break;
    case V_DOUBLE:*((double*)(vis->vfromval))= (double)atof(val_string);
                 break;
    };

    debuginfo(("VISUALIZER: Range 'from' set to '%s'\n",val_string));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}

void VtoCall(w, e, s, c)
Widget w;
XEvent *e;
String *s;
Cardinal *c;
{
    Arg a[1];
    char *val_string;
    char *vs;
    visualizertype *vis;

    vis = (visualizertype *)(atol(s[0]));
    XtSetArg(a[0], XtNstring, &val_string);
    XtGetValues(vis->vft_to,a,1);

    /* Check if infinity entered, i.e. zero-length entry or "inf" */

    vs=val_string;

    vis->vtoinfinite = (vs[0] == '\0');

    while ( (*vs) && (!(vis->vtoinfinite)) ) {
        if (!(strncasecmp(vs,"inf",3))) vis->vtoinfinite = TRUE;
        vs++;
    };


    /* Set the new comparison element accordingly, casting the void*	*/
    /* to the right type  (in case of infinity this data is just	*
    /* ignored in the comparison later on)				*/

    switch (vis->vtype) {
    case V_CHAR: *((char*)(vis->vtoval))= (char) atoi(val_string);
                 break;
    case V_INT:  *((int*) (vis->vtoval))= (int)  atoi(val_string);
                 break;
    case V_LONG: *((long*)(vis->vtoval))= (long) atol(val_string);
                 break;
    case V_FLOAT:*((float*)(vis->vtoval))  = (float) atof(val_string);
                 break;
    case V_DOUBLE:*((double*)(vis->vtoval))= (double)atof(val_string);
                 break;
    };

    debuginfo(("VISUALIZER: Range 'to' set to '%s'\n",val_string));

    /* Update the shown information */

    update_shown_data(vis);
    update_shown_condition(vis);
    display_layer(vis);

}


/*
 * void  VTimestep()
 * Called every UPDATE_INTERVAL / 1000 sec  to check the read socket
 * for new date
 */

void VTimestep(VIS, timer)
VIStype *VIS;
XtIntervalId *timer;
{

    /* Just generate the next timeout */

    XtAppAddTimeOut(VIS->VISapp_con, UPDATE_INTERVAL, VTimestep, VIS);
    handle_socket(VIS);
}


/*
 * void setup_write(VIStype *VIS
 *
 * Sets up the sockets for a write connection to the visualizer
 * that was just fork()ed  (data and command connection)
 */

void setup_write(VIS)
VIStype *VIS;
{
    struct sockaddr_un datasockadr;	/* Socket address for system (data) */
    struct sockaddr_un cmdsockadr;	/* Dito for commands		    */
    visualizertype *vis = (visualizertype *)VIS->VISvis;


    /* Create a new pair of sockets */

    debuginfo(("++ setup write: called\n"));
    if ( (vis->vdatawritesock = socket( AF_UNIX, SOCK_STREAM, 0 )) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Write socket (data) initialization failed !\n");
        exit(-1);
    };
    if ( (vis->vcmdwritesock = socket( AF_UNIX, SOCK_STREAM, 0 )) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Write socket (command) initialization failed !\n");
        exit(-1);
    };

    debuginfo(("++ setup write: sockets %d (data) and %d (commands) created, connecting to %s and %s...\n", vis->vdatawritesock, vis->vcmdwritesock, vis->vdatasockname,  vis->vcmdsockname ));


    /* Connect to the reading sockets of the child */

    datasockadr.sun_family = AF_UNIX;
    strcpy(datasockadr.sun_path, vis->vdatasockname);
    if ( connect( vis->vdatawritesock, (struct sockaddr *) &datasockadr,
                  sizeof(datasockadr) ) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Connect to read socket (data) failed!\n");
        exit(-1);
    };
    debuginfo(("++ setup write: socket connected\n"));
    cmdsockadr.sun_family = AF_UNIX;
    strcpy(cmdsockadr.sun_path, vis->vcmdsockname);
    if ( connect( vis->vcmdwritesock, (struct sockaddr *) &cmdsockadr,
                  sizeof(cmdsockadr) ) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Connect to read socket (commands) failed!\n");
        exit(-1);
    };
    debuginfo(("++ setup write: sockets connected\n"));

    return;

};


/*
 * void setup_read(VIStype *VIS)
 *
 * Sets up the sockets for a read connection (data & commands)
 * to the visualizer that was just fork()ed
 */

void setup_read(VIS)
VIStype *VIS;
{
    struct sockaddr_un datasockadr;	/* Data socket address for system */
    struct sockaddr_un cmdsockadr;	/* Dito Command socket		  */
    visualizertype     *vis = (visualizertype *)VIS->VISvis;

    debuginfo(("** setup read: called\n"));


    /* Create a new pair of sockets */

    if ( (vis->vdatabindsock = socket( AF_UNIX, SOCK_STREAM, 0 )) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Bind socket (data) initialization failed!\n");
        exit(-1);
    };
    if ( (vis->vcmdbindsock = socket( AF_UNIX, SOCK_STREAM, 0 )) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Bind socket (command) initialization failed!\n");
        exit(-1);
    };

    debuginfo(("** setup read: socket %d (data) and %d (command) ok\n",vis->vdatabindsock, vis->vcmdbindsock));


    /* Bind sockets to names (also known to writer !)  and listen to them */

    datasockadr.sun_family = AF_UNIX;
    strcpy(datasockadr.sun_path, vis->vdatasockname);
    if ( bind(vis->vdatabindsock,(struct sockaddr *) &datasockadr,
         sizeof(datasockadr)) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Binding of data socket failed!\n");
        exit(-1);
    };
    cmdsockadr.sun_family = AF_UNIX;
    strcpy(cmdsockadr.sun_path, vis->vcmdsockname);
    if ( bind(vis->vcmdbindsock,(struct sockaddr *) &cmdsockadr,
         sizeof(cmdsockadr)) < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Binding of command socket failed!\n");
        exit(-1);
    };

    debuginfo(("** setup read: binding to '%s' (data) and '%s' (command) ok, listening\n", vis->vdatasockname, vis->vcmdsockname));

    listen(vis->vdatabindsock, 1 );		/* Just 1 request possible */
    listen(vis->vcmdbindsock, 1 );


    /* Now we've got the sockets ready for use */

    return;

};


/*
 * void setup_accept(VIStype *VIS)
 *
 * Accepts a connection on the sockets that were created using setup_read()
 */

void setup_accept(VIS)
VIStype        *VIS;
{
    visualizertype *vis = (visualizertype *)VIS->VISvis;


    debuginfo(("** setup accept: called\n"));


    /* Get connections from writer */

    debuginfo(("** setup accept: accepting from bind socket %d (data) and %d (command)...\n",vis->vdatabindsock, vis->vcmdbindsock));
    vis->vdatareadsock = accept(vis->vdatabindsock,
                              (struct sockaddr *) NULL, (int *) NULL);
    if (vis->vdatareadsock < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Accept of data socket failed!\n");
        exit(-1);
    };
    vis->vcmdreadsock = accept(vis->vcmdbindsock,
                              (struct sockaddr *) NULL, (int *) NULL);
    if (vis->vcmdreadsock < 0 ) {
        printf(/*stderr,*/"VISUALIZER: Accept of command socket failed!\n");
        exit(-1);
    };

    debuginfo(("** setup accept: accepted connections %d (data) and %d (command)\n", vis->vdatareadsock, vis->vcmdreadsock));


    /* Make sockets non-blocking */

/*    fcntl(vis->vdatareadsock,F_SETFL,O_NDELAY);*/
/*    fcntl(vis->vcmdreadsock ,F_SETFL,O_NDELAY);*/


    /* Now we've got a communication line for use with read/write */
    /* based on v{cmd,data}readsock and v{cmd,data}writesock	  */

    return;

};


/*
 * void handle_socket(VIStype *VIS)
 *
 * Scan the read socket for input, update array data if there is any
 */

void handle_socket(VIS)
VIStype        *VIS;
{
    visualizertype *vis = (visualizertype *)VIS->VISvis;
    fd_set  fdset;		/* File descriptors for selection	*/
    int     read_ret;		/* Number of bytes read			*/
    char   *p;			/* Pointer to data buffer (array)	*/
    long    l;			/* Length of incoming data		*/
    long    lr;			/* Number of bytes received yet		*/
    char    magic[9];		/* Start sequence			*/
    char    s[BUFSIZ];		/* String for general use		*/
    struct  timeval tv;		/* Timeout interval			*/
    Boolean redisplay;		/* True if we have got new data		*/
    void   *fn;			/* Pointer to new user-function		*/
    long    slen;		/* Length of received string		*/
    char   *str;		/* String describing the new function	*/
    long    dlen;		/* Length of user data			*/
    char   *dat;		/* Pointer to user data (allocated)	*/
    long    sync;		/* For synchronization with caller	*/
    int     i;



    /* Check if data pending on our sockets (and if it's for us) */

    tv.tv_usec = 0;
    tv.tv_sec  = 0;		/* Add timeout for selection -> poll! */
    FD_ZERO(&fdset);
    FD_SET(vis->vdatareadsock,&fdset);
    FD_SET(vis->vcmdreadsock,&fdset);
    redisplay = 0;		/* Don't redisplay data as default */
    if (select(32,&fdset,NULL,NULL,&tv) > 0) {
        debuginfo(("** handle socket: selected\n"));
        if (FD_ISSET(vis->vdatareadsock,&fdset)) {

            /* Get data from data socket, use previous offset */

            p = (char *)(vis->vadr) + (long)(vis->vreadoffset);
            debuginfo(("** handle socket: something to read for us (data), start adress = %d, offset = %d, effective address = %d !\n",vis->vadr, vis->vreadoffset,p));

/*            /* Get magic number and check it *
 *
 *            if (!(read_ret = read(vis->vdatareadsock, &(magic[0]), 9))) {
 *                printf("** handle socket: couldn't get magic number\n");
 *                return;
 *            };
 *            if (!strncmp(magic,"sthMaGiC\0",9)) {
 *                printf("** handle socket: wrong magic number\n");
 *                return;
 *            };
 *
 *
 *            /* Get length of data *
 *
 *            if (!(read_ret = read(vis->vdatareadsock, &l, sizeof(long)))) {
 *                printf("** handle socket: couldn't get length of data\n");
 *                return;
 *            };
 */

            /* Compute size of incoming data */

            l=SizeOfType[vis->vtype];
            for (i=0; i<vis->vnumdims; i++) {
                l = l*(vis->vdims[i]);
            };


            /* Get data itself */

            lr = (long) vis->vreadoffset;
            while ((read_ret = read(vis->vdatareadsock, p, l-lr)) > 0) {
                lr = lr + read_ret;		/* Collect data length	*/
                debuginfo(("** handle socket: read %d bytes of data to address %d, remaining = %d\n", read_ret,p,l-lr ));
                p = p + read_ret;		/* Increment pointer	*/
                redisplay = 1;			/* Redisplay new data!	*/
            };

            vis->vreadoffset = (char*) lr;	/* New offset for reading */
            if ( read_ret < 0 ) {
                sprintf(s,"** handle socket: error during reading from data socket");
                perror(s);
            };
            debuginfo(("** handle socket: read a total of %d bytes of data \n",lr));



        } else if (FD_ISSET(vis->vcmdreadsock,&fdset)) {


            /* Get data from command socket 				*/
            /* Format of data:						*/
            /* <adr> <strlen> <string> <len> <data>			*/
            /*   |      |        |       |     ^			*/
            /*   |      |        |       |    data for user function	*/
            /*   |      |        |      length of user data		*/
            /*   |      |      string describing the user function	*/
            /*   |    length of description string			*/
            /*  address of (new) user function to be used or NULL	*/


            debuginfo(("** handle socket: something to read for us (command) !\n"));


            /* Get address of user function */

            if ((read_ret = read(vis->vcmdreadsock, &fn, sizeof(fn))) > 0) {
                debuginfo(("** handle socket: function = %d\n", l));
            };
            if ( read_ret < 0 ) {
                sprintf(s,"** handle socket: error during reading from command socket");
                perror(s);
            };


            /* Get length of string */

            if ((read_ret=read(vis->vcmdreadsock, &slen, sizeof(slen))) > 0) {
                debuginfo(("** handle socket: string length = %d\n", slen));
            };
            if ( read_ret < 0 ) {
                sprintf(s,"** handle socket: error during reading from command socket");
                perror(s);
            };


            /* Allocate buffer for string read slen bytes */

            str = &(s[0]);		/*!!!!!!! assumes <256 bytes!	*/
            lr=0;			/* No data bytes read yet	*/


            /* Read data until we have received all slen bytes,		*/
            /* appending new data to existing one in buffer		*/

            while ((lr<slen) &&
                   ((read_ret=read(vis->vcmdreadsock, str+lr,slen-lr)) > 0)) {
                debuginfo(("** handle socket: read %d bytes of string\n",read_ret));
                lr = lr+read_ret;	/* Add size of read data	*/
            };
            str[slen]='\0';		/* Terminate string to be sure	*/
            debuginfo(("** handle socket: read string '%s' (%d bytes)\n",str,slen));


            /* Get user data set size and allocate buffer */

            if ((read_ret=read(vis->vcmdreadsock, &dlen, sizeof(dlen))) > 0) {
                debuginfo(("** handle socket: data length = %d\n", dlen));
            };
            if ( read_ret < 0 ) {
                sprintf(s,"** handle socket: error during reading from command socket");
                perror(s);
            };


            /* Allocate buffer for user data read dlen bytes */

            dat = (char*)(XtMalloc(dlen));
            lr=0;			/* No data bytes read yet	*/

            /* Read data until we have received all dlen bytes,		*/
            /* appending new data to existing one in buffer		*/

            while ((lr<dlen) &&
                   ((read_ret=read(vis->vcmdreadsock, dat+lr,dlen-lr)) > 0)) {
                debuginfo(("** handle socket: read %d bytes of user data\n", read_ret));
                lr = lr+read_ret;	/* Add size of read data	*/
            };
            debuginfo(("** handle socket: read %d bytes of user data in all\n",dlen));

            /* Now the command is completely read:			*/
            /* Set the new function (if any), new string (if any),	*/
            /* fill in the data for the new user environment and	*/
            /* free the old one						*/

            if (fn) {			/* Update if given, else leave it */
                vis->vcmpfn = fn;
            };
            if (slen) {
                strncpy(vis->vcmptext,str,256);	/* Copy string to us	*/
            };
            if (vis->venvp) {
                XtFree(vis->venvp);	/* Free old environment data	*/
            };
            if (dlen) {
                vis->venvp   = dat;	/* Set new data	if present	*/
                vis->venvlen = dlen;
            } else {
                vis->venvp   = NULL;	/* Else just clear pointer	*/
            };

            debuginfo(("** handle socket: set function to %d, string '%s', user data @ %d (%d bytes)\n", fn, str, vis->venvp, dlen));


            /* Redisplay the Visualizer using the new function & data	*/
            /* Recalculate the statistics if Value Visualizer used	*/

            update_shown_data(vis);
            update_shown_condition(vis);
            if (vis->vdispmode == GRAY_DISPLAY) {
                debuginfo(("** handle socket: recalculate statistics\n"));
                do_statistics(vis);
            };
            display_layer(vis);


        } else {
            debuginfo(("** handle socket: something to read but not for us\n"));
        };
    } else {
/*        printf("** handle socket: nothing to read\n");*/
    };


    /* Display the newly received data if there was any */

    if (redisplay) {
        if (lr >= l) {

            /* Data complete, mark "being read" as false and update display */

            debuginfo(("** handle socket: data complete, redisplaying...\n"));
            vis->vreadoffset = NULL;
            update_shown_data(vis);
            update_shown_condition(vis);
            if (vis->vdispmode == GRAY_DISPLAY) {
                debuginfo(("** handle socket: recalculate statistics\n"));
                do_statistics(vis);
            };
            display_layer(vis);

        } else {

            /* Data incomplete, don't display yet but wait for rest of data */

            debuginfo(("** handle socket: %d bytes of data still missing\n",l-lr));
        };

    };

};


/*
 * do_statistics(visualizertype *vis)
 *
 * Computes the statistics (max, min, average, variance) and grayscale steps
 * for Value Visualizer. Eliminates values that are way over/under the average
 * and would distort the picture
 *
 */
void do_statistics(vis)
visualizertype *vis;
{
    /*****!!!!!!!!! dangerous - doesn't use "legal" access functions !!!!!!!*/

    char  s[512];
    Arg	  args[10];
    int   arraysize = vis->vdims[0];
    int   i;
    double sum;			/* Sum of all array elements		*/
    double av;			/* Average of elements			*/
    double var;			/* Variance of elements			*/
    double av2;			/* Average squared			*/
    double t;			/* Temp					*/
    char *pc;			/* Pointers to go through the array	*/
    int  *pi;
    long *pl;
    float*pf;
    double*pd;
    char  tc;			/* Temporary variables			*/
    int   ti;
    long  tl;
    float tf;
    double td;
    char  minc,maxc;		/* Min & Max values			*/
    int   mini,maxi;
    long  minl,maxl;
    float minf,maxf;
    double mind,maxd;
    void *minp,*maxp;		/* Pointer to Values			*/


    for (i=1; i<vis->vnumdims; i++) {		/* Get array size */
        arraysize = arraysize * (vis->vdims[i]);
    };

    /* Initialize statistics to 1st array element */

    vis->vminval = vis->vmaxval = minp = maxp = vis->vadr;
    vis->vavval = vis->vfactorval = 0.0;
    sum = var = av = 0.0;

    /* Gather the statistics, according to element type 	*/
    /* Average : av  = 1/n * sum(i=1..n) x(i)			*/
    /* Variance: var = 1/(n-1) * sum(i=1..n) (x(i)-av)**2	*/

    /*!!!!!!! will generate overflows if arrays large & numbers large */

    switch (vis->vtype) {
    case V_CHAR: pc = (char*) vis->vadr;
                 minc = maxc = *pc;
                 for (i=0; i<arraysize; i++) {
                     tc = *pc;
                     if (tc > maxc) {
                          maxp = (void*)pc;
                          maxc = tc;
                     }
                     if (tc < minc) {
                          minp = (void*)pc;
                          minc = tc;
                     }
                     sum = sum + (double) tc;
                     pc++;
                 };
                 /* Now we have the sum, compute the average	*/
                 vis->vavval  = av = sum / (double)arraysize;
                 av2 = av*av;
                 vis->vminval = minp;
                 vis->vmaxval = maxp;
                 /* And now the variance			*/
                 pc = (char*) vis->vadr;
                 for (i=0; i<arraysize; i++) {
                     t = (double) (*pc);
                     var = var + (t*t - 2*av*t + av2);
                     pc++;
                 };
                 var = sqrt( (1/(double)(arraysize-1)) * var);
                 vis->vvarval = var;
                 sprintf(s,"min=%d max=%d av=%f var=%f",
                         minc,maxc,av,var);
                 tc = (char) av + 2*(char)var;
                 if (tc<maxc)		/* Limit max to be within 95%	*/
                     maxc=tc;
                 tc = (char) av - 2*(char)var;
                 if (tc>minc)		/* Limit min			*/
                     minc=tc;
                 vis->vfactorval = 1.0 / (double) (maxc - minc);
                 break;
    case V_INT:  pi = (int*) vis->vadr;
                 mini = maxi = *pi;
                 for (i=0; i<arraysize; i++) {
                     ti = *pi;
                     if (ti > maxi) {
                          maxp = (void*)pi;
                          maxi = ti;
                     }
                     if (ti < mini) {
                          minp = (void*)pi;
                          mini = ti;
                     }
                     sum = sum + (double) ti;
                     pi++;
                 };
                 /* Now we have the sum, compute the average	*/
                 vis->vavval  = av = sum / (double)arraysize;
                 av2 = av*av;
                 vis->vminval = minp;
                 vis->vmaxval = maxp;
                 /* And now the variance			*/
                 pi = (int*) vis->vadr;
                 for (i=0; i<arraysize; i++) {
                     t = (double) (*pi);
                     var = var + (t*t - 2*av*t + av2);
                     pi++;
                 };
                 var = sqrt( (1/(double)(arraysize-1)) * var);
                 vis->vvarval = var;
                 sprintf(s,"min=%d max=%d av=%f var=%f",
                         mini,maxi,av,var);
                 ti = (int) av + 2*(int)var;
                 if (ti<maxi)		/* Limit max to be within 95%	*/
                     maxi=ti;
                 ti = (int) av - 2*(int)var;
                 if (ti>mini)		/* Limit min			*/
                     mini=ti;
                 vis->vfactorval = 1.0 / (double) (maxi - mini);
                 break;
    case V_LONG: pl = (long*) vis->vadr;
                 minl = maxl = *pl;
                 for (i=0; i<arraysize; i++) {
                     tl = *pl;
                     if (tl > maxl) {
                          maxp = (void*)pl;
                          maxl = tl;
                     }
                     if (tc < minl) {
                          minp = (void*)pl;
                          minl = tl;
                     }
                     sum = sum + (double) tl;
                     pl++;
                 };
                 /* Now we have the sum, compute the average	*/
                 vis->vavval  = av = sum / (double)arraysize;
                 av2 = av*av;
                 vis->vminval = minp;
                 vis->vmaxval = maxp;
                 /* And now the variance			*/
                 pl = (long*) vis->vadr;
                 for (i=0; i<arraysize; i++) {
                     t = (double) (*pl);
                     var = var + (t*t - 2*av*t + av2);
                     pl++;
                 };
                 var = sqrt( (1/(double)(arraysize-1)) * var);
                 vis->vvarval = var;
                 sprintf(s,"min=%d max=%d av=%f var=%f",
                         minl,maxl,av,var);
                 tl = (long) av + 2*(long)var;
                 if (tl<maxl)		/* Limit max to be within 95%	*/
                     maxl=tl;
                 tl = (long) av - 2*(long)var;
                 if (tl>minl)		/* Limit min			*/
                     minl=tl;
                 vis->vfactorval = 1.0 / (double) (maxl - minl);
                 break;
    case V_FLOAT:pf = (float*) vis->vadr;
                 minf = maxf = *pf;
                 for (i=0; i<arraysize; i++) {
                     tf = *pf;
                     if (tf > maxf) {
                          maxp = (void*)pf;
                          maxf = tf;
                     }
                     if (tf < minf) {
                          minp = (void*)pf;
                          minf = tf;
                     }
                     sum = sum + (double) tf;
                     pf++;
                 };
                 /* Now we have the sum, compute the average	*/
                 vis->vavval  = av = sum / (double)arraysize;
                 av2 = av*av;
                 vis->vminval = minp;
                 vis->vmaxval = maxp;
                 /* And now the variance			*/
                 pf = (float*) vis->vadr;
                 for (i=0; i<arraysize; i++) {
                     t = (float) (*pf);
                     var = var + (t*t - 2*av*t + av2);
                     pf++;
                 };
                 var = sqrt( (1/(double)(arraysize-1)) * var);
                 vis->vvarval = var;
                 sprintf(s,"min=%g max=%g av=%f var=%f",
                         minf,maxf,av,var);
                 tf = (float) av + 2*(float)var;
                 if (tf<maxf)		/* Limit max to be within 95%	*/
                     maxf=tf;
                 tf = (float) av - 2*(float)var;
                 if (tf>minf)		/* Limit min			*/
                     minf=tf;
                 vis->vfactorval = 1.0 / (double) (maxf - minf);
                 break;
    case V_DOUBLE:pd = (double*) vis->vadr;
                 mind = maxd = *pd;
                 for (i=0; i<arraysize; i++) {
                     td = *pd;
                     if (td > maxd) {
                          maxp = (void*)pd;
                          maxd = td;
                     }
                     if (td < mind) {
                          minp = (void*)pd;
                          mind = td;
                     }
                     sum = sum + td;
                     pd++;
                 };
                 /* Now we have the sum, compute the average	*/
                 vis->vavval  = av = sum / (double)arraysize;
                 av2 = av*av;
                 vis->vminval = minp;
                 vis->vmaxval = maxp;
                 /* And now the variance			*/
                 pd = (double*) vis->vadr;
                 for (i=0; i<arraysize; i++) {
                     t = *pd;
                     var = var + (t*t - 2*av*t + av2);
                     pd++;
                 };
                 var = sqrt( (1/(double)(arraysize-1)) * var);
                 vis->vvarval = var;
                 sprintf(s,"min=%g max=%g av=%f var=%f",
                         mind,maxd,av,var);
                 td =  av + 2*var;
                 if (td<maxd)		/* Limit max to be within 95%	*/
                     maxd=td;
                 ti =  av - 2*var;
                 if (td>mind)		/* Limit min			*/
                     mind=td;
                 vis->vfactorval = 1.0 / (maxd - mind);
                 break;
    };

    /* Set the (former) condition display to show the Statistics 	*/

    XtSetArg(args[0], XtNlabel, s);
    XtSetValues(vis->viscond,args,1);

};
