/*
 * display.h
 *
 * Definitions used in Array Display
 *
 *
 * Stefan Haenssgen 05-11-91
 *
 * Changes:	06-18-91 Reformatted code
 *			 Added comments
 *		06-26-91 Added function "resize_display"
 *			 Added detailed description functions
 *			 Added function "set_zoom"
 *		07-02-91 Added array "pixels"
 *			 Added Display *"disp" as lookup (saves many fn-calls)
 *			 Added function "get_pixel"
 *		07-09-91 Changed color to char
 *		09-28-91 Added function "center_at" to faciliate zooming in etc
 *		10-03-91 Added new widgets (view, hbar, vbar, canvas, dbutton)
 *			  to replace the existing viewport & canvas
 *			 Added new variables (cx, cy, hsize, vsize etc)
 *		10-05-91 Replaced pixmap by buffer
 *			 Made hsize,vsize int (and typecast Dimension)
 *		12-28-91 Added user data for user defineable data
 *		02-05-92 Added "buttonfn" to set_display_functions()
 *		04-16-92 Added "mode" for b&w/gray/color display and definition
 *			  of modes {BW,GRAY,COLOR}_DISPLAY
 *			 Added "colornumber" for number of colors used
 *			 Updated the description of functions
 *		08-27-92 Added support for VisualizeActivation:
 *			 - defined ACT_DISPLAY
 *			 - Documented this new feature
 *			 - introduced "xrest" to make incomplete last lines
 *			    possible in the array
 *			 - added set_xrest() to set this value, with docu
 *		08-28-92 Changed xrest to restl (max length to be accessed)
 *		10-24-92 size = n means (0..n-1) as range!
 *
 */



/* Structure of one Array Display, just the graphics 			*/

typedef struct {
    Widget  parent;	/* parent widget of this display structure	*/
    Widget  view;	/* Parent for the array display (scrollbars..)	*/
    Widget  hbar, vbar;	/* Scrollbars for display			*/
    Widget  dbutton;	/* Button in display				*/
    Widget  canvas;	/* For display of actual array graphics		*/
    GC      clear_gc,	/* GraphicContext for this window, one for..	*/
            set_gc;	/* ..clearing pixels and one for setting them	*/
    Display *disp;	/* Lookup instead of every XtDisplay(d->parent)	*/
    int     mode;	/* Display mode (b/w,gray,color)		*/
    int     colornumber;/* Number of colors used (2,5,...) for mode	*/
    int     zoom;	/* Zoom factor					*/
    int     xptr,yptr;	/* Position of mouse pointer in array		*/
    int     xoff,yoff;	/* Offset of array display in window		*/
    int     cx, cy;	/* Pixel around which the display is centered	*/
    int     xfrom,yfrom;/* First momentarily visible pixel of array	*/
    int     xto,yto;	/* Last dito (giving currently visible region)	*/
    int     xsize,ysize;/* Size of displayed array			*/
    long    restl;	/* From where to ignore last values of array	*/
    int     hsize,vsize;/* Physical size of canvas			*/
    Pixmap  buffer;	/* Offscreen buffer for display			*/
    char    *pixels;	/* array [xsize][ysize] for own representation	*/
    void    (*showposfn)(); /* called when mouse coordinates change	*/
			/* parameters: (displaytype *d,			*/
			/*              Postion x, Position y)		*/
			/* Only called when pointer in effective	*/
			/* array, i.e. not in border etc		*/
    void    (*enterfn)();  /* called when mouse enters array display	*/
			/* Parameter: (displaytype *d)			*/
    void    (*leavefn)();  /* called when mouse leaves display		*/
			/* Parameter: (displaytype *d)			*/
    void    (*clickfn)();  /* called when mouse button is pressed down	*/
			/* parameters: (displaytype *d,			*/
			/*              Postion x, Position y,		*/
			/*              int buttonmask)			*/
			/* Only called when pointer in effective	*/
			/* array, i.e. not in border etc		*/
    void    (*buttonfn)(); /* called when display button is pressed	*/
			/* Parameter: (displaytype *d, int buttonmask)	*/
    void    *userdata;	/* Pointer to user defineable data structures	*/
} displaytype;


/* Display modes */

#define BW_DISPLAY	0
#define GRAY_DISPLAY	1
#define COLOR_DISPLAY	2
#define ACT_DISPLAY	3


/* Public functions in array.c */

#ifndef THIS_IS_ARRAY_C
extern displaytype *create_display();	/* Create a display structure	*/
extern void         init_display();	/* Initialize it (bitmaps etc)	*/
extern void         destroy_display();	/* Destroy one instance		*/
extern void         resize_display();	/* Clear and resize the bitmap	*/
extern void         clear_all();	/* Clear the bitmap		*/
extern void         set_pixel();	/* Set one Pixel in the bitmap	*/
extern char    	    get_pixel();	/* Returns the pixel value	*/
extern void         update_image();	/* Redisplay the bitmap		*/
extern void         get_cursor_xy();	/* Return Position of cursor	*/
extern void         set_display_functions(); /* Set event handling fn's */
extern void         set_zoom();		/* Set zoom factor		*/
extern int          get_zoom();		/* Get zoom factor		*/
extern void	    set_xrest();	/* Set the xrest value (s.above)*/
extern void         center_at();	/* Center display at (x,y)	*/
extern void         set_display_userdata();/* Set pointer to user data	*/
extern void        *get_display_userdata();/* Return ptr to user data	*/
#endif

#define THICK 15		/* Thickness of scrollbars */


/*  Detailed description of functions:
 *  ==================================
 *
 *  displaytype *create_display(int xsize, int ysize, int zoom,
 *                              Widget parent, int mode)
 *
 *  Creates a new array display as child widget of "parent" with width "xsize"
 *  and height "ysize", but doesn't do any initialization yet.
 *  (NOTE, this means the resulting array is [0..xsize-1]x[0..ysize-1] !!!)
 *  The zoom factor determines the size of a displayed pixel.
 *  The mode (BW_DISPLAY, GRAY_DISPLAY, COLOR_DISPLAY) determines how pixel
 *  values are displayed, i.e. as b&w pixels / using different bitmaps for
 *  gray / using real colors. ACT_DISPLAY is used for the special case of
 *  activation value visualization which is a tuned up B&W display.
 *
 *
 *  void init_display(displaytype *d)
 *
 *    Initialized an already created display window after it has been
 *    realized (before that we don't know anything about its display
 *    and screen etc)
 *
 *
 *  void destroy_display(displaytype *d)
 *
 *    Destroys the given display, freeing all of its resources
 *
 *
 *  void resize_display(displaytype *d, int xsize_new, int ysize_new, int zoom)
 *
 *    Alter the size of the displayed bitmap. The old bitmap is lost.
 *    (Avoids having to re-create a new display every time the size,
 *    i.e. the diplayed dimensions of the array, changes)
 *    (NOTE, this means the resulting array is [0..xsize-1]x[0..ysize-1] !!!)
 *    The zoom factor is also changed.
 *
 *
 *  void clear_all(displaytype *d)
 *
 *    Sets all of the display's pixels to zero
 *
 *
 *  void set_pixel(displaytype *d, int x, int y, char color)
 *
 *    Sets a pixel (x,y) in the given array-display "d" to some color
 *    Depending on the display mode, the possible range of colors
 *    is:
 *         0..1   BW_DISPLAY	(black&white)
 *         0..4   GRAY_DISPLAY	(shades of gray using different pixmaps)
 *         0..?   COLOR_DISPLAY	(real colors)
 *
 *    (NB: Drawing is done internally - to display the result use
 *         update_image() !)
 *
 *
 *  char get_pixel(displaytype *d, int x, int y)
 *
 *    Returns the color value of the pixel at (x,y)
 *
 *
 *  void update_image(displaytype *d)
 *
 *    Updates the image of the given Display in the corresponding window
 *
 *
 *  void get_cursor_xy(displaytype *d, int *x, int *y)
 *
 *    Returns in x,y the mouse cursor's position, given as coordinates in
 *    the displayed array
 *    Returns (-1,-1) if the cursor is not in the correspoding window
 *
 *
 * void set_display_functions(d,showposfn, enterfn, leavefn, clickfn, buttonfn)
 *
 *   Sets the functions that are called when a corresponding event
 *   occurs:
 *      showposfn - mouse pointer has canged position within array display
 *                  Parameters: (displaytype *d, Position x, Position y)
 *      enterfn   - mouse pointer has entered display
 *                  Parameter: (displaytype *d)
 *      leavefn   - mouse pointer has left display
 *                  Parameter: (displaytype *d)
 *      clickfn   - mouse button pressed down within array display
 *                  Parameter: (displaytype *d, Position x, y, int Btnmask)
 *      buttonfn  - button of display clicked
 *                  Parameter: (displaytype *d)
 *   showposfn is only called when the pointer is within the display of
 *   the array (i.e. it's not called when the pointer is inside the border
 *   of the display etc). Dito clickfn
 *
 *
 * void set_zoom(displaytype *d, int zoom)
 *
 *   Sets the zoom factor of the display,  calculates a new display
 *   accordingly and re-displays it.
 *
 *
 * int get_zoom(displaytype *d)
 *
 *   Returs the zoom factor of the display
 *
 *
 * void set_restl(displaytype *d, long restl)
 *
 *   Sets a new value for restl, which determines the position from which
 *    the rest of the array should not be accessed (i.e. drawn as background)
 *    Useful for displaying an array as square even when its dimensions are not
 *    square roots - just stuff it until it's square and ignore the fillers.
 *
 *
 * void center_at(displaytype *d, int x, int y)
 *
 *   Centers the display around the given coordinates (as well as
 *   possible, e.g. concerning edges of the array etc)
 *
 *
 * void set_display_userdata(displaytype *d, void *userdata)
 *
 *   Stores a user defineable pointer in the display data structure
 *   (not used by the display functions themselves, just user data)
 *
 *
 * void *get_displayuserdata(displaytype *d)
 *
 *   Returns the user defineable pointer
 */
