/*
 *  display.c
 *
 *  Graphical display of array elements and interactive pointing
 *  via mouse
 *
 *  Stefan Haenssgen 04-27-91
 *
 *  Updates:	04-27-91  Specifications, function headers
 *		05-11-91  Implementation based on experimental "pixel.c"
 *		05-20-91  Further Debugging
 *			  Event handler functions externally accessible
 *			   and settable via set_functions(d,s,e,l)
 *		06-18-91  Source reformatted
 *			  DEBUG_OUTPUT introduced
 *			  Renamed set_functions to set_display_functions
 *			  Implemented event function for mouse clicks
 *		06-26-91  Added function "resize_display"
 *			  Added zoom factor and "{set,get}_zoom"
 *		07-02-91  Added array "pixels" to represent the bitmap
 *			   internally and to make resizing etc easier
 *			  Added struct element "disp" for lookup of
 *			   X-Display instead of recalculating it always
 *			  Swapped GXclear and GXset to make "1" represent
 *			   a black dot consistently
 *			  Added function "get_pixel" to read a pixel value
 *			  Increased debugging output
 *			  Use crosshair cursor instead of usual arrow
 *		07-09-91  Changed pixel type from short to char
 *		07-17-91  Disabled resizing of display window by the parent
 *			  Set to new size after changing zoom or a resize
 *			  Changed it back to the state before... doesn't
 *			   look too nice :-/
 *		09-28-91  Added zooming-support using "center_at"
 *			   (to let the user zoom into a specific point that
 *			   is to be centered in the actual display)
 *			  Debugged the position indication (didn't
 *			   consistently take zoom into account)
 *			  Changed debugging output to be consistent
 *			  Also update xoff,yoff when zoom changes
 *		10-03-91  Implemented the viewport Widget functionality
 *			   by hand, eliminating the need for an offscreen
 *			   bitmap:
 *			  - Added new Widgets (view, hbar, vbar, canvas,
 *			     dbutton) and did the layout in the form Widget
 *			  - Added scrollbar event handling procedures
 *			     D_JumpProc() and D_ScrollProc()
 *			  - Added new variables (cx, cy, hsize, vsize)
 *			  - Rewrote update_image() to directly draw onto
 *			     the canvas window
 *			  - Removed all bitmap-oriented code
 *		10-05-91  Cleaned things up:
 *			  - Changed variables that are necessary for
 *			     display management ({x,y}{from,to}, {h,v}size,
 *			     c{x,y})
 *			  - Added D_ComputeOffsets to easily update
 *			     x/yfrom/to and h/vsize *
 *			  - Renamed old D_GetOffset to D_GetEnter to reflect
 *			     its changed functionality
 *			  - Added D_RedrawAll() to handle complete
 *			     off-screen redraw
 *			  - Added D_SetScrollbars() to update scrollbars
 *			  - Defined side-effects of procedures (e.g.
 *			     redrawing on screen, updating scrollbars etc)
 *			  - Debugging
 *			  - Made h/vsize an int, avoiding trouble with
 *			     type Dimension
 *		10-09-91  Added event handlers for redrawing etc:
 *			  - Added D_RedrawEvent() to handle exposure
 *			     and visibility changes
 *			  - Made canvas widget resizeable
 *			  - Added D_ResizeEvent() to handle resizing
 *			  - Made size of view-Widget more flexible
 *		12-28-91  Added user data for user defineable data
 *			   (access via {set,get}_display_userdata() )
 *		01-09-92  Changed the display "canvas" to make resizing
 *			   look nicer
 *		09-Jan-92  Fill background with gray (black/white mixed)
 *			    pattern to make boundaries of array visible,
 *			    introducing "gray_bits" and "gray_gc"
 *			   Eliminated flickering (no unnecessary clearing
 *			    of the canvas)
 *		19-Jan-92  Use "CreatePixmapFromBitmapData()" for background
 *			    Tile (Bitmap alone created problems!)
 *		05-Feb-92  Added functionality to button of Array Display:
 *			     New userfunction "buttonfn" for 
 *			     set_display_functions()
 *			   Added BUTTONLABEL definition
 *		15-Apr-92  Added grayscale pixels (with zoom >= 2) by
 *			    using pixmaps to copy the bits from
 *			   Created bitmaps "gray[0-4].map" for five
 *			    different shades of gray (white...black)
 *			   Limited zoom factor to size of bitmaps
 *			   Modified set_pixel() and D_RedrawAll()
 *			    to use the bitmaps
 *		16-Apr-92  Added display modes {BW,GRAY,COLOR}_DISPLAY
 *			   Modified create_display() to accept modes
 *			   Rewrote set_pixel() and D_RedrawAll() to utilize
 *			    different drawing routines for different modes
 *			   Introduced "colornumber" for the number of different
 *			    available display pixel representations
 *		01-Jul-92  Limit color value to boundaries 0..colornumber-1
 *			    instead of complaining
 *		27-Aug-92  Added support for VisualizeActivation:
 *			   - ACT_DISPLAY mode as special case of B&W
 *			   - introduced "xrest" to ignore the rest of
 *			      the last line in an array. Always init
 *			      to xsize, so normally without effect
 *			   - added new function set_xrest to adjust this value
 *			   - modified D_RedrawAll() and set_pixel() to
 *			      handle the last line seperately when ACT_DISPLAY
 *		28-Aug-92  Changed xrest to restl because more than one line
 *			    could be affected. Now: y*xsize+x <= restl  -> OK
 *		24-Oct-92  Size of N implies range 0..N-1 now!!!
 *			    (so, also y*xsize+x < restl etc pp)
 *
 */


/* Include headers, not defining the external functions we're	*/
/* about to supply in this program!				*/

#define THIS_IS_ARRAY_C
#include "X.h"
#include "display.h"


/* Label of button in lower left corner, "S" like "Step"	*/

#define BUTTONLABEL "S"


/* Include cursor bitmaps for display cursor			*/

#include "cursor.map"
#include "cursor_mask.map"


/* Bitmap for grey background and different shades of gray pixels (global) */

#define  GRAY_COLORNUMBER 5
#include "gray_bits.map"
#include "gray0.map"
#include "gray1.map"
#include "gray2.map"
#include "gray3.map"
#include "gray4.map"
Pixmap    gray_maps[5];
Pixmap    gray_bits;
GC        gray_gc;
#define COLOR_COLORNUMBER 256	/* !!! to be implemented !!! */


/* Forward declarations of functions (for event handlers etc) */

void clear_all();		/* Clear display internally		*/
void D_GetMotion();		/* Mouse pointer motion			*/
void D_GetClick();		/* Mouse button pressed			*/
void D_GetEnter();		/* Enter widget event -> resize		*/
void D_GetLeave();		/* Leave widget event			*/
void D_GetButton();		/* Button click callback		*/
void D_JumpProc();		/* Scrollbar jump event			*/
void D_ScrollProc();		/* Scrollbar scroll event		*/
void D_ComputeOffsets();	/* Recalculate offsets for display	*/
void D_RedrawAll();		/* Complete off-screen redraw		*/
void D_RedrawScrollbars();	/* Update scrollbar positions & sizes   */
void D_RedrawEvent();		/* Redraw the window to handle event	*/
void D_ResizeEvent();		/* Handle resizing of the display	*/




/*
 *  displaytype *create_display(int xsize, int ysize, int zoom,
 *                              Widget parent, int mode)
 *
 *  Creates a new array display as child widget of "parent" with width "xsize"
 *  and height "ysize", but doesn't do any initialization yet.
 *  (NOTE, this means the resulting array is [0..xsize-1]x[0..ysize-1] !!!)
 *  The zoom factor determines the size of a displayed pixel.
 *  The mode (BW_DISPLAY, GRAY_DISPLAY, COLOR_DISPLAY) determines how pixel
 *  values are displayed, i.e. as b&w pixels / using different bitmaps for
 *  gray / using real colors. ACT_DISPLAY is used for the special case of
 *  activation value visualization which is a tuned up B&W display.
 *
 *  (ugly, but necessary because the canvas widget will complain if it
 *  is created in a realized state (and it doesn't have a child yet, so
 *  it thinks all sizes are 0, bah!), and we don't have any information
 *  concerning displays, windows, etc before that)
 */

displaytype *create_display(xsize, ysize, zoom, parent,mode)
int    xsize, ysize, zoom;
Widget parent;
int    mode;
{
    displaytype *d;
    Arg a[10];
    int n;
    XtCallbackRec *jumpcb;		/* Callback records for scrollbar */
    XtCallbackRec *scrollcb;
    Dimension dh, dv;			/* For size of canvas (no int)	  */


    /* Allocate space for the display struct */

    d = (displaytype *) XtMalloc(sizeof(displaytype));
    d->parent = parent;


    /* Check the mode and set the number of allowed colors accordingly */

    if (mode!=BW_DISPLAY && mode!=GRAY_DISPLAY && mode!=COLOR_DISPLAY &&
        mode != ACT_DISPLAY ) {
        fprintf(stderr,"DISPLAY: Unknown mode number %d, assuming B&W\n",mode);
        d->mode=BW_DISPLAY;
    } else {
        d->mode = mode;
    };
    if (mode == BW_DISPLAY || mode == ACT_DISPLAY) d->colornumber=2;
    else if (mode == GRAY_DISPLAY) d->colornumber=GRAY_COLORNUMBER;
    else                           d->colornumber=COLOR_COLORNUMBER;


    /* Create the widget family */

    n=0;
    XtSetArg(a[n], XtNdefaultDistance, 0); n++;	/* Pack children tightly  */
    XtSetArg(a[n], XtNwidth, xsize*zoom+THICK); n++;	/* Geometry	  */
    XtSetArg(a[n], XtNheight, ysize*zoom+THICK); n++;
/*    XtSetArg(a[n], XtNpreferredPaneSize, True); n++; /* For paned widget! */
/*    XtSetArg(a[n], XtNallowResize, False); n++;	*/
/*    XtSetArg(a[n], XtNmin, ysize*zoom+THICK); n++;
/*    XtSetArg(a[n], XtNmax, ysize*zoom+THICK); n++;	/*resizing OK! */
    d->view = XtCreateManagedWidget("view", formWidgetClass, parent, a, n);


    /* Create callback information for jump/scroll events	*/
    /* Pass the counter's displaytype* as client data		*/

    jumpcb = (XtCallbackRec *) XtMalloc(2*sizeof(XtCallbackRec));
    jumpcb[0].callback = (XtCallbackProc) D_JumpProc;
    jumpcb[0].closure = (XtPointer) d;
    jumpcb[1].callback = (XtCallbackProc) NULL;
    jumpcb[1].closure = (XtPointer) NULL;
    scrollcb = (XtCallbackRec *) XtMalloc(2*sizeof(XtCallbackRec));
    scrollcb[0].callback = (XtCallbackProc) D_ScrollProc;
    scrollcb[0].closure = (XtPointer) d;
    scrollcb[1].callback = (XtCallbackProc) NULL;
    scrollcb[1].closure = (XtPointer) NULL;

    /* Create vertical scrollbar */
    n=0;
    XtSetArg(a[n], XtNminimumThumb, 7);	n++;	/* Thumb >= 7 pixels high */
    XtSetArg(a[n], XtNjumpProc, jumpcb); n++;	/* Register callbacks	  */
    XtSetArg(a[n], XtNscrollProc, scrollcb); n++;
/***    XtSetArg(a[n], XtNleft, XtChainLeft); n++;	/* Stay at upper left..	  */
/***    XtSetArg(a[n], XtNtop, XtChainTop); n++;	/* ..corner of view	  */
    XtSetArg(a[n], XtNthickness, THICK); n++;
    XtSetArg(a[n], XtNlength, ysize*zoom); n++;		   /* Size	  */
    XtSetArg(a[n], XtNorientation, XtorientVertical); n++; /* Vertical	  */
    d->vbar = XtCreateManagedWidget("vbar", scrollbarWidgetClass, d->view,a,n);

    /* Create canvas for drawing on */
    n=0;
/***    XtSetArg(a[n], XtNtop, XtChainTop); n++;	/* Stay at topof view	  */
    XtSetArg(a[n], XtNfromHoriz, d->vbar); n++; /* Place right of vbar	  */
    XtSetArg(a[n], XtNlabel, ""); n++;
    XtSetArg(a[n], XtNwidth, xsize*zoom); n++;	     /* Geometry       */
    XtSetArg(a[n], XtNheight, ysize*zoom); n++;
    d->canvas = XtCreateManagedWidget("canvas", labelWidgetClass,
                        d->view, a, n);

    /* Create button in lower left corner */
    n=0;
    XtSetArg(a[n], XtNlabel, BUTTONLABEL); n++;
/***    XtSetArg(a[n], XtNleft, XtChainLeft); n++;	/* Stay at left edge	  */
    XtSetArg(a[n], XtNfromVert, d->vbar); n++;	/* Place under vbar	  */
    XtSetArg(a[n], XtNwidth, THICK); n++;	/* Geometry		  */
    XtSetArg(a[n], XtNheight, THICK); n++;
    d->dbutton = XtCreateManagedWidget("dbutton", commandWidgetClass,
			d->view,a,n);

    /* Create horizontal scrollbar (using the callback data from above)	*/
    n=0;
    XtSetArg(a[n], XtNminimumThumb, 7);	n++;	/* Thumb >= 7 pixels high */
    XtSetArg(a[n], XtNjumpProc, jumpcb); n++;	/* Register callbacks	  */
    XtSetArg(a[n], XtNscrollProc, scrollcb); n++;
    XtSetArg(a[n], XtNfromHoriz, d->dbutton);n++; /* Place right of button*/
    XtSetArg(a[n], XtNfromVert, d->canvas);n++; /* Place below canvas	  */
    XtSetArg(a[n], XtNthickness, THICK); n++;
    XtSetArg(a[n], XtNlength, xsize*zoom); n++;		     /* Size	  */
    XtSetArg(a[n], XtNorientation, XtorientHorizontal); n++; /* Horizontal*/
    d->hbar = XtCreateManagedWidget("hbar", scrollbarWidgetClass, d->view,a,n);


    /* Initialize variables						  */
 
    d->xsize = xsize;
    d->ysize = ysize;
    d->restl = xsize*ysize-1; /* Last position which is ok to access	  */
    d->cx = xsize / 2;				/* Center at center (wow) */
    d->cy = ysize / 2;
    if (zoom>gray0_width) {			/* Limit zoom factor to	*/
        d->zoom = gray0_width;			/*  bitmap size		*/
        fprintf(stderr,"DISPLAY: zoom factor limited to %d\n",d->zoom);
    } else {
        d->zoom  = zoom;
    };
    d->showposfn = NULL;			/* Functions not active   */
    d->clickfn = NULL;
    d->enterfn = NULL;
    d->leavefn = NULL;
    d->buttonfn= NULL;

    n=0;					/* Get physical size of.. */
    XtSetArg(a[n], XtNwidth, &(dh));n++;	/* ..canvas		  */
    XtSetArg(a[n], XtNheight, &(dv));n++;
    XtGetValues(d->canvas, a, n);
    d->hsize = (int) dh;
    d->vsize = (int) dv;


    /* Allocate array for internal representation of the display	  */
    /* and clear it							  */
    d->pixels = (char *) XtMalloc(sizeof(char)*xsize*ysize);
    memset(d->pixels, sizeof(char)*xsize*ysize, 0);

#ifdef DEBUG_OUTPUT
printf("DISPLAY: create_display(): display created, %dx%d zoom %d\n",xsize,ysize,d->zoom);
#endif

    return(d);
}



/*
 *  void init_display(displaytype *d)
 *
 *  Initialized an already created display window after it has been
 *  realized (before that we don't know anything about its display
 *  and screen etc)
 *  Sets the display window to white and sets initial scrollbar values
 */

void *init_display(d)
displaytype *d;
{
    Arg a[5];
    int n;
    XColor fg,bg;
    int    scrn;
    Display *disp;

    d->disp = disp = XtDisplay(d->parent);	/* Get display for window */
    scrn = DefaultScreen(d->disp);		/* Get screen		  */


    /* Set up GC for creating a gray background 	*/
    /* Set up all bitmaps for different shades of gray	*/

    gray_gc =  XCreateGC(d->disp, XtWindow(d->parent), (unsigned long)0, NULL);
/*    gray_bits = XCreateBitmapFromData(d->disp,XtWindow(d->parent),*/
/*                gray_bits_bits,gray_bits_width, gray_bits_height);*/
    gray_bits = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray_bits_bits,gray_bits_width, gray_bits_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    gray_maps[0] = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray0_bits,gray0_width, gray0_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    gray_maps[1] = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray1_bits,gray1_width, gray1_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    gray_maps[2] = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray2_bits,gray2_width, gray2_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    gray_maps[3] = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray3_bits,gray3_width, gray3_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    gray_maps[4] = XCreatePixmapFromBitmapData(d->disp,XtWindow(d->parent),
                gray4_bits,gray4_width, gray4_height,
                BlackPixel(disp,scrn),WhitePixel(disp,scrn),
                DefaultDepthOfScreen(XtScreen(d->parent)) );
    XSetState(d->disp,gray_gc, WhitePixel(disp,scrn),BlackPixel(disp,scrn),
              GXcopy,AllPlanes);
    XSetTile(d->disp,gray_gc,gray_bits);
    XSetFillStyle(d->disp,gray_gc,FillTiled);
  


    /* Set up GCs for the array display, clear canvas & init variables	*/

    d->clear_gc = XCreateGC(d->disp,XtWindow(d->canvas),
                (unsigned long) 0, NULL);  	/* GC for clearing	*/
/*              GCForeground | GCBackground, NULL);  /* GC for clearing	*/
    XSetState(d->disp,d->clear_gc,
              WhitePixel(disp,scrn),BlackPixel(disp,scrn),GXcopy,AllPlanes);
    XSetFillStyle(d->disp,d->clear_gc,FillSolid);
    d->set_gc = XCreateGC(d->disp,XtWindow(d->canvas),
                (unsigned long) 0, NULL);  	/* GC for setting	*/
    XSetState(d->disp,d->set_gc,
              BlackPixel(disp,scrn),WhitePixel(disp,scrn),GXcopy,AllPlanes);
    XSetFillStyle(d->disp,d->set_gc,FillSolid);
    XFillRectangle(d->disp,XtWindow(d->canvas),gray_gc,
                   0,0,d->hsize,d->vsize);
    d->xptr = d->yptr = -1;		/* "Not valid" flag		*/
    d->xoff = d->yoff = 0;


    /* White border around canvas and button				*/
    n=0;
    XtSetArg(a[n], XtNborderColor, WhitePixel(disp,scrn)); n++;
    XtSetValues(d->canvas, a, n);
    XtSetValues(d->dbutton, a, n);

    /* Prepare a background buffer for drawing				*/

    d->buffer = XCreatePixmap(disp, XtWindow(d->canvas),
                             (int)d->hsize, (int)d->vsize,
                             DefaultDepth(disp,scrn));
    XFillRectangle(d->disp,d->buffer,gray_gc,0,0,d->hsize,d->vsize);


    /* Add event handling routines with "d" as calling info		*/
    /* Add Callback for display button					*/

    XtAddEventHandler(d->canvas,EnterWindowMask,FALSE,D_GetEnter,d);
    XtAddEventHandler(d->canvas,LeaveWindowMask,FALSE,D_GetLeave,d);
    XtAddEventHandler(d->canvas,PointerMotionMask,FALSE,D_GetMotion,d);
    XtAddEventHandler(d->canvas,ButtonPressMask,FALSE,D_GetClick,d);
    XtAddEventHandler(d->canvas,VisibilityChangeMask,FALSE,D_RedrawEvent,d);
    XtAddEventHandler(d->canvas,VisibilityChangeMask,FALSE,D_RedrawEvent,d);
    XtAddEventHandler(d->canvas,ExposureMask,FALSE,D_RedrawEvent,d);
    XtAddEventHandler(d->canvas,ResizeRedirectMask,FALSE,D_ResizeEvent,d);
    XtAddEventHandler(d->canvas,StructureNotifyMask,FALSE,D_ResizeEvent,d);
    XtAddCallback(d->dbutton, XtNcallback, D_GetButton, d);


    /* Prepare fore/background and use a self defined crosshair cursor	*/

    fg.pixel = 0;
    fg.red = fg.green = fg.blue = 0;		/* Black foreground	*/
    fg.flags = 0;
    bg.pixel = 1;
    bg.red = bg.green = bg.blue = 65535;	/* White background	*/
    bg.flags = 0;
    XDefineCursor(d->disp,XtWindow(d->canvas),
                  XCreatePixmapCursor(d->disp,
                      XCreateBitmapFromData(d->disp, XtWindow(d->parent),
                          cursor_bits, cursor_width,
                          cursor_height),
                      XCreateBitmapFromData(d->disp, XtWindow(d->parent),
                          cursor_mask_bits, cursor_mask_width,
                          cursor_mask_height),
                      &fg,&bg,
                      cursor_x_hot, cursor_y_hot)
                 );

    D_ComputeOffsets(d);		/* Recalculate display variables */
    D_RedrawScrollbars(d);		/* Update scrollbars		 */

#ifdef DEBUG_OUTPUT
printf("DISPLAY: init_display(): OK\n");
#endif

}


/*
 *  void destroy_display(displaytype *d)
 *
 *  Destroys the given display, freeing all of its resources
 */

void destroy_display(d)
displaytype *d;
{
    XFreeGC(d->disp,d->set_gc);
    XFreeGC(d->disp,d->clear_gc);
    XFreePixmap(d->disp,d->buffer);
    XtDestroyWidget(d->view);	/* also destroys child "canvas"	*/
    XtFree(d->pixels);
    XtFree(d);

#ifdef DEBUG_OUTPUT
printf("DISPLAY: destroy_display(): OK\n");
#endif

}



/*
 *  void resize_display(displaytype *d, int xsize_new, int ysize_new, int zoom)
 *
 *  Alter the size of the displayed array.
 *  (Avoids having to re-create a new display every time the size,
 *  i.e. the diplayed dimensions of the array, changes)
 *  (NOTE, this means the resulting array is [0..xsize-1]x[0..ysize-1] !!!)
 *  The zoom factor is also changed.
 *  Clears the internal display buffer AND the window
 */

void resize_display(d, xsize_new, ysize_new, zoom)
displaytype *d;
int xsize_new, ysize_new, zoom;
{
    Arg a[10];
    int n;


#ifdef DEBUG_OUTPUT
printf("DISPLAY: resize_display(): Resizing display from %dx%d to %dx%d\n",
       d->xsize, d->ysize, xsize_new, ysize_new);
#endif

    if (zoom>gray0_width) {			/* Limit zoom factor to	*/
        zoom = gray0_width;			/*  bitmap size		*/
        fprintf(stderr,"DISPLAY: zoom factor limited to %d\n",zoom);
    };
    D_ComputeOffsets(d);	/* Recalculate display information	*/


    /* Delete the old pixels array and buffer and create new ones	*/

    XtFree(d->pixels);
    d->pixels = (char *) XtMalloc(sizeof(char)*xsize_new*ysize_new);
    memset(d->pixels, sizeof(char)*xsize_new*ysize_new, 0);
    XFreePixmap(d->disp,d->buffer);
    d->buffer = XCreatePixmap(d->disp, XtWindow(d->canvas),
                             (int)d->hsize, (int)d->vsize,
                             DefaultDepth(d->disp,DefaultScreen(d->disp)));
    XFillRectangle(d->disp,d->buffer,gray_gc,0,0,d->hsize,d->vsize);

    d->xsize = xsize_new;
    d->ysize = ysize_new;
    d->xptr = d->yptr = -1;             /* Old pointer values not valid */
    d->xoff = d->yoff = 0;
    d->zoom = zoom;
    d->cx = xsize_new / 2;              /* Center at center (wow)       */
    d->cy = ysize_new / 2;

    n=0;
    XtSetArg(a[n], XtNwidth, xsize_new*zoom); n++;	/* Geometry    */
    XtSetArg(a[n], XtNheight, ysize_new*zoom); n++;
/*    XtSetArg(a[n], XtNmin, ysize_new*zoom); n++;
    XtSetArg(a[n], XtNmax, ysize_new*zoom); n++;*/
    XtSetValues(d->canvas,a,n);

    D_ComputeOffsets(d);
    clear_all(d);
/****    XFillRectangle(d->disp,XtWindow(d->canvas),gray_gc, /* Clear canvas */
/****                   0,0,d->hsize,d->vsize);*/
};



/*
 *  void clear_all(displaytype *d)
 *
 *  Sets all of the display's pixels to zero, also clear internal buffer
 *  (not on-screen!)
 */

void clear_all(d)
displaytype *d;
{

    XFillRectangle(d->disp,d->buffer,gray_gc,
                   0,0,d->hsize,d->vsize);
    memset(d->pixels, sizeof(char)*d->xsize*d->ysize, 0);

#ifdef DEBUG_OUTPUT
printf("DISPLAY: clear_display(): OK\n");
#endif

}


/*
 *  void set_pixel(displaytype *d, int x, int y, char color)
 *
 *    Sets a pixel (x,y) in the given array-display "d" to some color
 *    Depending on the display mode, the possible range of colors
 *    is:
 *         0..1   BW_DISPLAY	(black&white)
 *         0..4   GRAY_DISPLAY	(shades of gray using different pixmaps)
 *         0..?   COLOR_DISPLAY	(real colors)
 *
 *    (NB: Drawing is done internally - to display the result use
 *         update_image() !)
 */

void set_pixel(d, x, y, color)
displaytype *d;
int x,y;
char color;
{
    char cmax = d->colornumber-1;	/* Maximum color number allowed	*/


    /* Check if point lies within boundaries */

    if ( (x<0) || (x >= d->xsize) || (y<0) || (y >= d->ysize) ) {
        fprintf(stderr,"DISPLAY: set_pixel() called with illegal values (%d,%d) color %d\n        Maximum is %dx%d with %d colors\n",x,y,color,d->xsize,d->ysize,d->colornumber);
    } else {

        if (color>cmax)			/* Restrict color to boundaries	*/
            color = cmax;
        if (color<0)
            color = 0;
 
        /* Update the pixels array to have state info   	*/

        d->pixels[x + d->xsize*y] = (char) color;

        /* Draw in buffer if pixel visible			*/
        /* using appropriate drawing mode			*/

        if ( (x >= d->xfrom) && (x <= d->xto) &&
             (y >= d->yfrom) && (y <= d->yto) ) {

            if (d->mode == BW_DISPLAY) {		/* B&W display	   */
                if (d->zoom == 1) {
                    XDrawPoint(d->disp,d->buffer,
                               color ? d->set_gc : d->clear_gc,
                               x + d->xoff - d->xfrom,
                               y + d->yoff - d->yfrom);
                } else {
                     XFillRectangle(d->disp,d->buffer,
                               color ? d->set_gc : d->clear_gc,
                               (x - d->xfrom) * d->zoom + d->xoff,
                               (y - d->yfrom) * d->zoom + d->yoff,
                               d->zoom, d->zoom);
                };

            } else if (d->mode == ACT_DISPLAY) {	/* Activation display*/
              if ( (y*(d->xsize) + x <= d->restl) &&
		   (y < d->ysize) ) {			/* Ignore last parts */
                if (d->zoom == 1) {
                    XDrawPoint(d->disp,d->buffer,
                               color ? d->set_gc : d->clear_gc,
                               x + d->xoff - d->xfrom,
                               y + d->yoff - d->yfrom);
                } else {
                     XFillRectangle(d->disp,d->buffer,
                               color ? d->set_gc : d->clear_gc,
                               (x - d->xfrom) * d->zoom + d->xoff,
                               (y - d->yfrom) * d->zoom + d->yoff,
                               d->zoom, d->zoom);
                };
              }
            } else if (d->mode == GRAY_DISPLAY) {	/* Graymap display */
                XCopyArea(d->disp,gray_maps[color],d->buffer,d->set_gc,
                          0,0,d->zoom+1,d->zoom+1,
                          (x - d->xfrom) * d->zoom + d->xoff,
                          (y - d->yfrom) * d->zoom + d->yoff
                         );

            } else {					/* Color display   */
                fprintf(stderr,"DISPLAY: Color mode not yet implemented!\n");
            };
        };
/*#ifdef DEBUG_OUTPUT		TOO MUCH OUTPUT */
/*printf("DISPLAY: set_pixel(): pixel at (%d,%d) set to %d\n",x,y,color);*/
/*#endif*/
    };
}



/*  char get_pixel(displaytype *d, int x, int y)
 *
 *    Returns the color value of the pixel at (x,y)
 */

char get_pixel(d, x, y)
displaytype *d;
int x,y;
{
    return( d->pixels[x + d->xsize*y] );
}



/*
 *  void update_image(displaytype *d)
 *
 *    Redraws the display on-screen and updates the scrollbars.
 *    Updates the image of the given Display in the corresponding window.
 *    Also updates the scrollbars.
 */

void update_image(d)
displaytype *d;
{

/* ??? ALSO UPDATE DISPLAY VARIABLES BY CALLING D_ComputeOffsets ??? */


    /* Copy buffer to display window */

    XCopyArea(d->disp,d->buffer,XtWindow(d->canvas),
              d->set_gc, 0, 0, d->hsize, d->vsize, 0, 0);
#ifdef DEBUG_OUTPUT
printf("DISPLAY: update_image():  center at (%d,%d) phys.size = %dx%d\n", d->cx, d->cy, d->hsize, d->vsize);
#endif

    D_RedrawScrollbars(d);	/* Update scrollbars */

}


/*
 *  void get_cursor_xy(displaytype *d, int *x, int *y)
 *
 *  Returns in x,y the mouse cursor's position, given as coordinates in
 *  the displayed array
 *  Returns (-1,-1) if the cursor is not in the correspoding window
 */

void get_cursor_xy(d,x,y)
displaytype *d;
int *x;
int *y;
{

    /* Get x,y and cry "error" if cursor position not up to date */

    *x = d->xptr;
    *y = d->yptr;
    if ( (*x<0) || (*y<0) ) {
        fprintf(stderr,"DISPLAY: get_cursor_xy() called and cursor out of array\n");
    } else {
    };
}


/*
 * void set_display_functions(d,showposfn, enterfn, leavefn, clickfn, buttonfn)
 *
 * Sets the functions that are called when a corresponding event
 * occurs:
 *    showposfn - mouse pointer has canged position within array display
 *                Parameters: (displaytype *d, Position x, Position y)
 *    enterfn   - mouse pointer has entered display
 *                Parameter: (displaytype *d)
 *    leavefn   - mouse pointer has left display
 *                Parameter: (displaytype *d)
 *    clickfn	- mouse button pressed down within array display
 *		  Parameter: (displaytype *d, Position x, y, int Btnmask)
 *    buttonfn  - button of display clicked
 *                Parameter: (displaytype *d)
 * showposfn is only called when the pointer is within the display of
 * the array (i.e. it's not called when the pointer is inside the border
 * of the display etc). Dito clickfn
 */

void set_display_functions(d,sfn,efn,lfn,cfn,bfn)
displaytype *d;
void (*sfn)();
void (*efn)();
void (*lfn)();
void (*cfn)();
void (*bfn)();
{
    d->showposfn = sfn;
    d->enterfn   = efn;
    d->leavefn   = lfn;
    d->clickfn   = cfn;
    d->buttonfn  = bfn;

#ifdef DEBUG_OUTPUT
printf("DISPLAY: set_display_functions(): functions set to %d, %d, %d, %d, %d\n",sfn,efn,lfn,cfn,bfn);
#endif

}



/*
 * void set_zoom(displaytype *d, int zoom)
 *
 *   Sets the zoom factor of the display.
 *   redraws the image (only internally !!! window not affected!)
 */

void set_zoom(d, zoom)
displaytype *d;
int zoom;
{
    int   x,y;
    Arg   a[10];
    int   n;
    int   depth; 


    if (zoom == d->zoom) return;	/* nothing changed, let it be	*/

    if (zoom>gray0_width) {			/* Limit zoom factor to	*/
        zoom = gray0_width;			/*  bitmap size		*/
        fprintf(stderr,"DISPLAY: zoom factor limited to %d\n",zoom);
    };
#ifdef DEBUG_OUTPUT
printf("DISPLAY: set_zoom(): Zoom factor changed from %d to %d\n",d->zoom, zoom);
#endif

    d->zoom = zoom;

/* FLICKERS TOO MUCH
/*   XFillRectangle(d->disp,XtWindow(d->canvas),gray_gc,0,0,
/*                  d->hsize,d->vsize);	/* Clear display window		   */
    D_ComputeOffsets(d);		/* Update offsets!		   */
    D_RedrawAll(d);			/* Redraw whole display internally */

}


/*
 * int get_zoom(displaytype *d)
 *
 *   Returs the zoom factor of the display
 */

int get_zoom(d)
displaytype *d;
{
    return(d->zoom);
}



/*  void center_at(displaytype *d, int x, int y)
 *
 *   Centers the display around the given coordinates (as well as
 *   possible, e.g. concerning edges of the array etc)
 *   Updates the image and the scrollbars on the screen (not just
 *   internally)
 */

void center_at(d,x,y)
displaytype *d;
int x;
int y;
{

#ifdef DEBUG_OUTPUT
printf("DISPLAY: center_at(): Centered at (%d,%d)\n", x,y);
#endif

    if ( (x >= 0) && (x < d->xsize) &&
         (x >= 0) && (x < d->xsize) ) {
        d->cx = x;
        d->cy = y;
        D_ComputeOffsets(d);	/* Compute actual offsets for drawing	*/
        D_RedrawAll(d);		/* Draw the new area internally		*/
        update_image(d);	/* Display the result immediately	*/
				/* (also updates scrollbars)		*/
    } else {
	fprintf(stderr,
            "DISPLAY: center_at() called with illegal values (%d,%d)\n",x,y);
    };
}


/*
 * void set_restl(displaytype *d, long restl)
 *
 *   Sets a new value for restl, which determines the position from which
 *    the rest of the array should not be accessed (i.e. drawn as background)
 *    Useful for displaying an array as square even when its dimensions are not
 *    square roots - just stuff it until it's square and ignore the fillers.
 *
 */
void set_restl(d, restl)
displaytype *d;
long restl;
{

#ifdef DEBUG_OUTPUT
printf("DISPLAY: set_restl(): set restl to %d\n", restl);
#endif

    if ( (restl >= (d->xsize)*(d->ysize)) || (restl <= 0) ) {
        fprintf(stderr,
            "DISPLAY: set_restl() called with illegal value %d, must be <= %d (%dx%d)\n",restl,(d->xsize)*(d->ysize),d->xsize,d->ysize);
    } else {
        d->restl = restl;
    };
}


/******************************************/
/*                                        */
/* INTERNAL routines, event handlers, etc */
/*                                        */
/******************************************/



/*
 * void D_GetMotion(widget, call_data, event)
 *
 * Event handler for motion event in one canvas
 * Updates the pointer's position in the corresponding display stuct
 * (which is given as call_data in the event dispatcher)
 */

void D_GetMotion(widget, d, event)
Widget widget;
XEvent *event;
displaytype *d;
{
    /* return position with offset taken into consideration */
    /* return -1 if pointer is not inside the array display */
    /* (take zoom factor in account)			    */

    d->xptr = ( (((XMotionEvent*) event)->x) - d->xoff) / d->zoom + d->xfrom;
    d->yptr = ( (((XMotionEvent*) event)->y) - d->yoff) / d->zoom + d->yfrom;
    if ( (d->xptr < 0) || (d->xptr >= d->xsize) ||
         (d->yptr < 0) || (d->yptr >= d->ysize) )  {
        d->xptr = -1;
        d->yptr = -1;
    };
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetMotion(): pointer at (%d,%d) with offset (%d,%d)\n",d->xptr,d->yptr,d->xoff,d->yoff);
#endif


    /* call user specified function if available and cursor in array */
    if ((d->xptr > -1) && (d->yptr > -1)) {
        if (d->showposfn) {
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetMotion(): showposfn at %d called\n",d->showposfn);
#endif
            (*(d->showposfn))(d, d->xptr, d->yptr);
        }
    };

}


/*
 * void D_GetClick(widget, call_data, event)
 *
 * Event handler for button click event in one canvas
 * Also updates the pointer's position in the corresponding display stuct
 * (which is given as call_data in the event dispatcher)
 * Calls user defied event function, if available, to notify of button press
 */

void D_GetClick(widget, d, event)
Widget widget;
XEvent *event;
displaytype *d;
{
    /* return position with offset taken into consideration */
    /* return -1 if pointer is not inside the array display */

    d->xptr = ( (((XMotionEvent*) event)->x) - d->xoff) / d->zoom + d->xfrom;
    d->yptr = ( (((XMotionEvent*) event)->y) - d->yoff) / d->zoom + d->yfrom;
    if ( (d->xptr < 0) || (d->xptr >= d->xsize) ||
         (d->yptr < 0) || (d->yptr >= d->ysize) )  {
        d->xptr = -1;
        d->yptr = -1;
    };
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetClick(): button (mask %d) pressed at (%d,%d) with offset (%d,%d)\n",
       ((XButtonEvent *) event) -> button, d->xptr,d->yptr,d->xoff,d->yoff);
#endif


    /* call user specified function if available and cursor in array */

    if ((d->xptr > -1) && (d->yptr > -1)) {
        if (d->clickfn) {
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetClick(): clickfn at %d called\n",d->clickfn);
#endif
            (*(d->clickfn))(d, d->xptr, d->yptr,
			      ((XButtonEvent *) event) -> button);
        }
    };

}


/*
 * void D_GetEnter(widget, d, event)
 *
 * Event handler for Enter events in one canvas.
 * Computes a new offset when the pointer enters the widget,
 * Display struct given as call_data.
 */

void D_GetEnter(widget,d,event)
Widget widget;
displaytype *d;
XEvent *event;
{


    /* Compute new offset just in case the widget has been resized */

    D_ComputeOffsets(d);


#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetEnter(): size of canvas = %dx%d, offset = (%d,%d)\n",
        d->hsize,d->vsize,d->xoff,d->yoff);
#endif

    /* call user defined enterfn if available */

    if (d->enterfn) {
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetEnter(): Enterfn at %d called\n",d->enterfn);
#endif
        (*(d->enterfn))(d);
    };
}


/*
 * void D_GetLeave(widget, d, event)
 *
 * Event handler for Leave events.
 * Sets the position to "not up to date" as the mouse pointer has
 * left the window.
 * Display struct given as call_data.
 */

void D_GetLeave(widget,d,event)
Widget widget;
displaytype *d;
XEvent *event;
{
    d->xptr = d->yptr = -1;


    /* call user defined leavefn if available */

    if (d->leavefn) {
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetLeave(): leavefn at %d called\n",d->leavefn);
#endif
        (*(d->leavefn))(d);
    };
}


/*
 * void D_GetButton(Widget w, displaytype *d, XtPointer call_data)
 *
 * Callback for display button.
 * Calls user-defined buttonfn if available.
 */

void D_GetButton(w, d, call_data)
Widget w;
displaytype *d;
XtPointer call_data;
{

    if (d->buttonfn) {
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_GetButton(): buttonfn at %d called\n",d->buttonfn);
#endif
        (*(d->buttonfn))(d);
    }


};


/*
 * void D_JumpProc(Widget sb, displaytype *d, float *percent)
 *
 * Callback for scrollbar jump events, i.e. when the user positions
 * the thumb directly.
 * Redraws the display on-screen and updates the scrollbars.
 */

void D_JumpProc(sb, d, percent)
Widget sb;
displaytype *d;
float *percent;
{
    Boolean is_hbar = (sb == d->hbar);	/* True if called by horiz.scrollbar */


    if (is_hbar) {				/* Set center accordingly */
        d->cx = (int) ((float) d->xsize * (*percent));
    } else {
        d->cy = (int) ((float) d->ysize * (*percent));
    };
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_JumpProc(): %s jumped to %f\%\n",is_hbar?"hbar":"vbar",
       *percent);
#endif

    center_at(d,d->cx,d->cy);

}


/*
 * void D_ScrollProc(Widget sb, displaytype *d, int pos)
 *
 * Callback for scrollbar scroll events, i.e. when the user clicks the
 * right or left mouse button to move the thumb.
 * Redraws the display on-screen and updates the scrollbars.
 */

void D_ScrollProc(sb, d, pos)
Widget sb;
displaytype *d;
int pos;
{

    Arg a[5];
    int n;
    Dimension len;	/* Scrollbar's length (to compare with position)*/
    float top;		/* Actual position of thumb (%)			*/
    Boolean is_hbar = (sb == d->hbar);	/* True if called by horiz.scrollbar */


    /* Get length of scrollbar and use it to compute the percentage	*/
    /* we want to scroll from out current position (top)		*/
    n=0;
    XtSetArg(a[n], XtNlength, &len); n++;
    XtSetArg(a[n], XtNtopOfThumb, &top); n++;
    XtGetValues(sb, a, n);

    top = (float)pos/(float)(len)/3.0 + top;
    top = top<0.0 ? 0.0 : top;			/* Limit to 0.0 .. 1.0	*/
    top = top>1.0 ? 1.0 : top;

    if (is_hbar) {				/* Set center accordingly */
        d->cx = (int) ((float) d->xsize * (top));
    } else {
        d->cy = (int) ((float) d->ysize * (top));
    };
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_ScrollProc():  %s scrolled to to %f\%\n",
       is_hbar?"hbar":"vbar", top);
#endif

    center_at(d, d->cx, d->cy);

}

/*
 * void D_ComputeOffsets(displaytype *d)
 *
 * Recalculate Offsets for display (xfrom, xto, yfrom, yto, hsize, vsize)
 *
 */
void D_ComputeOffsets(d)
displaytype *d;
{
    int    n;
    Arg    a[5];
    int    cx = d->cx;
    int    cy = d->cy;
    int    xmax,ymax;
    Dimension dh, dv;			/* For size on canvas, no int*/

    n=0;				/* Get physical size of..    */
    XtSetArg(a[n], XtNwidth, &dh);n++;	/* ..canvas		     */
    XtSetArg(a[n], XtNheight, &dv);n++;
    XtGetValues(d->canvas, a, n);
    d->hsize = (int) dh;
    d->vsize = (int) dv;
    xmax = d->hsize / d->zoom;		/* Max. pixels that fit..    */
    ymax = d->vsize / d->zoom;		/* ..on phys. display	     */

    /* Compute visible region and limit it to array size 	     */

    d->xfrom = cx - xmax/2;
    if (d->xfrom < 0) {		/* Edge of screen, display max. rest */
        d->xfrom = 0;
    };
    d->yfrom = cy - ymax/2;
    if (d->yfrom < 0) {
        d->yfrom = 0;
    };
    d->xto   = d->xfrom + xmax;
    if (d->xto > d->xsize-1) {
        d->xto = d->xsize-1;
    };
    d->yto   = d->yfrom + ymax;
    if (d->yto > d->ysize-1) {
        d->yto = d->ysize-1;
    };

    /* Compute offset of array in display window */

    d->xoff = (d->hsize - d->xsize * d->zoom) / 2;
    d->yoff = (d->vsize - d->ysize * d->zoom) / 2;
    if (d->xoff < 0) d->xoff=0;
    if (d->yoff < 0) d->yoff=0;

#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_ComputeOffsets(): off = (%d,%d)  from (%d,%d) to (%d,%d)\n",d->xoff, d->yoff, d->xfrom, d->yfrom, d->xto, d->yto);
printf("                             max = (%d,%d)\n",xmax, ymax);
#endif

}


/*
 * void D_RedrawAll(displaytype *d)
 *
 * Redraw the whole display centered at (cx,cy) internally
 * (i.e. without displaying it on the screen)
 * Needs actual offsets etc, so call D_ComputeOffsets before
 * or assert that the offsets are up to date
 *
 */
void D_RedrawAll(d)
displaytype *d;
{
    register int x,y;
    int		x0 = d->xfrom;
    int		y0 = d->yfrom;
    Display	*disp = d->disp;	/* For speed (not many references!) */
    Pixmap	buf = d->buffer;
    GC		gc = d->set_gc;
    int		dx = d->xsize;
    int		z  = d->zoom;
    int		xoff = d->xoff;
    int		yoff = d->yoff;


    /* Clear buffer to gray, marking the border between array and background */
    /* (Do it only if background is visible!				     */

    if ( !( (xoff <= 0) && (yoff <= 0) &&
            ((d->xto - x0)*z + xoff >= d->hsize) &&
            ((d->yto - y0)*z + yoff >= d->vsize)
          ))  {
              XFillRectangle(disp,buf,gray_gc,0,0,d->hsize,d->vsize);
    };


    /* Redraw all visible dots, depending on the zoom factor (performance!) */
    /* One drawing routine for each mode for reasons of speed		    */

    if (d->mode == BW_DISPLAY) {
        if (z == 1) {
            for (x=x0; x <= d->xto; x++)
                for (y=y0; y <= d->yto; y++)
                    XDrawPoint(disp,buf,
                               d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                               x+xoff-x0,y+yoff-y0);
        } else {
            for (x=x0; x <= d->xto; x++)
                for (y=y0; y <= d->yto; y++)
                    XFillRectangle(disp,buf,
                                   d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                                   (x-x0)*z+xoff,(y-y0)*z+yoff,z,z);
        };
    } else if (d->mode == ACT_DISPLAY) { /* Special handling of last line! */
        int ylast = ((d->restl+1) / d->xsize) -1;
        int xlast = ((d->restl+1) % d->xsize) -1;
        if (z == 1) {
            for (y=y0; y <= ylast; y++)		/* All but last line	*/
                for (x=x0; x < d->xto; x++)
                    XDrawPoint(disp,buf,
                               d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                               x+xoff-x0,y+yoff-y0);
            x=x0; y=ylast+1;
            while (x<=xlast) {			/* Part of last line	*/
                XDrawPoint(disp,buf,
                           d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                           x+xoff-x0,y+yoff-y0);
                x++;
            };
        } else {
            for (y=y0; y <= ylast; y++)	/* All but last line	*/
                for (x=x0; x <= d->xto; x++)
                    XFillRectangle(disp,buf,
                                   d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                                   (x-x0)*z+xoff,(y-y0)*z+yoff,z,z);
            x=x0; y=ylast+1;
            while (x<=xlast) {			/* Part of last line	*/
                XFillRectangle(disp,buf,
                               d->pixels[x+dx*y] ? d->set_gc : d->clear_gc,
                               (x-x0)*z+xoff,(y-y0)*z+yoff,z,z);
                x++;
            };
        };
    } else if (d->mode == GRAY_DISPLAY) {
            for (x=x0; x <= d->xto; x++)
                for (y=y0; y <= d->yto; y++)
                    XCopyArea(disp,gray_maps[d->pixels[x+dx*y]],buf,
                              d->set_gc,  0,0, d->zoom+1,d->zoom+1,
                              (x - x0) * z + xoff,
                              (y - y0) * z + yoff
                             );
    } else {
       fprintf(stderr,"DISPLAY: Color mode not yet implemented!\n");
    };

/******* OPTIMIZE THE "+xoff" PART? IF xoff == 0 IT'S NOT NECESSARY!!! ******/

#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_RedrawAll(): %dx%d pixel display from (%d,%d) to (%d,%d) redrawn\n", d->hsize, d->vsize, d->xfrom, d->yfrom, d->xto, d->yto);
#endif

};



/*
 * void D_RedrawScrollbars(displaytype *d)
 *
 * Redraw the scrollbars according to the display information,
 * updating positions & sizes
 *
 */
void D_RedrawScrollbars(d)
displaytype *d;
{
    int		z = d->zoom;			/* Zoom factor		  */
    float	h_percent, v_percent;		/* Scrollbar positions	  */
    float	h_show, v_show;			/* Scroll-Thumb size	  */
    float	h_off, v_off;			/* Offset of center	  */


    /* Compute percentage for centering X */

    if (d->xsize*z <= d->hsize) {
        /* Bitmap X smaller than viewport, no positioning necessary	*/
        h_percent = 0.0;
        h_show = 1.0;
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_RedrawScrollbars(): no X positioning, \t\t");
#endif
    } else {
        /* Bitmap X doesn't fit in viewport, compute offset to center x	*/
        h_off = (float) d->cx / (float) d->xsize;
        h_show = (float) d->hsize / (float)(d->xsize*z);
        h_percent = h_off;
#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_RedrawScrollbars(): h_perc = %f, h_show = %f,\t", h_percent,h_show);
#endif
    };


    /* Compute percentage for centering Y */

    if (d->ysize*z <= d->vsize) {
        /* Bitmap Y smaller than viewport, no positioning necessary	*/
        v_percent = 0.0;
        v_show = 1.0;
#ifdef DEBUG_OUTPUT
printf("no Y positioning necessary\n");
#endif
    } else {
        /* Bitmap Y doesn't fit in viewport, compute offset to center y	*/
        v_off = (float) d->cy / (float) d->ysize;
        v_show = (float) d->vsize / (float)(d->ysize*z);
        v_percent = v_off;		/* was: (1.0-v_show)*v_off */
#ifdef DEBUG_OUTPUT
printf("v_perc = %f, v_show = %f\n", v_percent, v_show);
#endif
    };

    /* Now set the scrollbars accordingly */

    XawScrollbarSetThumb(d->hbar, h_percent, h_show);
    XawScrollbarSetThumb(d->vbar, v_percent, v_show);

};


/*
 * void D_RedrawEvent(widget, call_data, event)
 *
 * Event handler for events that make a redrawing of the display necessary
 */

void D_RedrawEvent(widget, d, event)
Widget widget;
XEvent *event;
displaytype *d;
{

    /* Redraw everything by simply calling update_image() */

#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_RedrawEvent(): OK\n");
#endif
    update_image(d);

}


/*
 * void D_ResizeEvent(widget, call_data, event)
 *
 * Event handler for events that make a resizing of the display necessary
 */

void D_ResizeEvent(widget, d, event)
Widget widget;
XEvent *event;
displaytype *d;
{
    int     n;
    Arg     a[5];
    int     h,v;
    Boolean doit = FALSE;

#ifdef DEBUG_OUTPUT
printf("DISPLAY: D_ResizeEvent():");
#endif

    /* Depending on type of event, get the new size information		*/

    if (event->type == ResizeRequest) {
        h = ((XResizeRequestEvent*)event)->width;
        v = ((XResizeRequestEvent*)event)->height;
        doit = TRUE;
    } else if (event->type == ConfigureNotify) {
        h = ((XConfigureEvent*)event)->width;
        v = ((XConfigureEvent*)event)->height;
        doit = TRUE;
    };

    if (!doit) {		/* No relevant event, return		*/
#ifdef DEBUG_OUTPUT
printf(" irrelevant\n");
#endif
        return;
    };

    n=0;
    XtSetArg(a[n], XtNwidth, h);  n++;	/* Set new canvas size		*/
    XtSetArg(a[n], XtNheight, v); n++;
    XtSetArg(a[n], XtNmin, v);    n++;
    XtSetArg(a[n], XtNmax, v);    n++;
    XtSetValues(d->canvas, a, n);

    XtSetArg(a[0], XtNlength, h); n++;	/* Set new scrollbar length	*/
    XtSetValues(d->hbar, a, 1);
    XtSetArg(a[0], XtNlength, v); n++;
    XtSetValues(d->vbar, a, 1);
    
#ifdef DEBUG_OUTPUT
printf(" new size = %dx%d\n",h,v);
#endif

    D_ComputeOffsets(d);	/* Get new size of canvas		*/

    /* Delete the old buffer and create a newly sized one		*/

    XFreePixmap(d->disp,d->buffer);
    d->buffer = XCreatePixmap(d->disp, XtWindow(d->canvas),
                             (int)d->hsize, (int)d->vsize,
                             DefaultDepth(d->disp,DefaultScreen(d->disp)));
    XFillRectangle(d->disp,d->buffer,gray_gc,0,0,d->hsize,d->vsize);

    D_RedrawScrollbars(d);		/* Update scrollbars		*/
    D_RedrawAll(d);			/* Redraw display internally	*/
    update_image(d);			/* Display the new result	*/

}


/*
 * void set_display_userdata(displaytype *d, void *userdata)
 *
 *   Stores a user defineable pointer in the display data structure
 *   (not used by the display functions themselves, just user data)
 */

void set_display_userdata(d,userdata)
displaytype *d;
void *userdata;
{
    d->userdata = userdata;
};


/*
 * void *get_display_userdata(displaytype *d)
 *
 *   Returns the user defineable pointer
 */
void *get_display_userdata(d)
displaytype *d;
{
    return(d->userdata);
};
