/*
 *  counter.c
 *
 *  Management of counters for array boundaries
 *
 *  Stefan Haenssgen 05-20-91
 *
 *  Updates:	05-20-91  Specifications, implementation of create_counter
 *			  Bitmaps drawn
 *			  debugging of first implementation
 *		05-30-91  Bitmaps included into source code
 *			  init_counter for initializing bitmaps etc
 *		06-18-91  Source reformatted
 *			  DEBUG_OUTPUT introduced
 *			  User defined event handling functions
 *			   changedvalue, changedstate  introduced,
 *			   function set_counter_functions used to set them
 *			  New action procedure C_Entered that gets called
 *			   from the action table of the "center" widget
 *			   when <return> is pressed, to parse the new value
 *		06-26-91  Debugged state change handler
 *			  Added function "set_counter_text"
 *		07-02-91  Increased debugging output
 *		07-13-91  Experimental implementation of scrollbars
 *			   instead of all those buttons:
 *			  - changed widgets and added scrollbar inits
 *			  - Added  C_SetScrollbarFromCounter  and
 *			     C_SetValueFromScrollbar  to make things easier
 *			  - Added callbacks C_JumpProc and C_ScrollProc
 *			     that get called by the scrollbar
 *		12-28-91  Added user data for user defineable data
 *			   (access via {set,get}_counter_userdata() )
 *			  Also set new scrollbar position if value changes
 *		01-09-92  Made resizing of counters look nicer
 *			   by changing the "counterbox" to "counterform"
 *			   and making it a form Widget
 *		01-10-92  Changed C_ChangeActive() to ignore call_data 
 *			   (used to dump core)
 *		01-19-92  Initialize lastactive !!! (argh)
 *		02-Sep-93 Also set new value when leaving entry widget
 *
 */


/* Include headers, not defining the external functions we're	*/
/* about to supply in this program!				*/

#define THIS_IS_COUNTER_C
#include "X.h"
#include "counter.h"


/* Include bitmaps for counter buttons */

#include "up.map"
#include "upfar.map"
#include "down.map"
#include "downfar.map"


/* Private Callback routines */

void C_PlusMinus();		/* Counting			*/
void C_ChangeActive();		/* Active button toggling	*/
void C_Entered();		/* Value entered event		*/
void C_JumpProc();		/* Scrollbar jump event		*/
void C_ScrollProc();		/* Scrollbar scroll event	*/
void C_SetScrollbarFromValue();	/* Update scrollbar using value	*/
void C_SetValueFromScrollbar();	/* Update value using scrollbar	*/

/*
 * countertype *create_counter(int num_counters, int *c_min, int *c_max,
 *                             Widget parent)
 *
 * Creates a set of counters and initializes them. Boundaries of
 * possible counter values are passed in the c_min and c_max arrays.
 */

countertype *create_counter(num_counters, c_min, c_max, parent)
int num_counters;
int *c_min;
int *c_max;
Widget parent;
{
    countertype *c;
    Arg a[10];
    int n,i;
    char *s = XtMalloc(512);
    XtActionsRec arec;


    /* allocate space for the counter struct & create the widget family */

    c = (countertype *) XtMalloc(sizeof(countertype));
    c->parent = parent;
    c->num_counters = num_counters;
    c->changedvalue = NULL;		/* No event handler functions yet    */
    c->changedstate = NULL;
    c->lastactive   = 0;

    n=0;
    XtSetArg(a[n], XtNorientation, "vertical"); n++;	/* Tile side by side */
    XtSetArg(a[n], XtNshowGrip, False); n++;		/* No grips	     */
    XtSetArg(a[n], XtNhSpace, 5); n++;	 		/* leave some space..*/
    XtSetArg(a[n], XtNvSpace, 5); n++;	 		/* ..between counters*/
    c->counterform = XtCreateManagedWidget("counterform", formWidgetClass,
					parent, a, n);
    /* Create our own action record for C_Entered and add it to the	*/
    /* existing action table					   	*/
    arec.string = "C_Entered";
    arec.proc   = C_Entered;
    XtAppAddActions(XtWidgetToApplicationContext(parent), &arec, 1);



    /* Create one counter Widget for each dimension */

    for (i=0; i<num_counters; i++) {
        onecountertype *counter;
        callcountertype *callt;
        Widget tsource;			/* to find textSource of AsciiText   */
        XtCallbackRec *jumpcb;		/* Callback records for scrollbar    */
        XtCallbackRec *scrollcb;


        counter = (onecountertype *) XtMalloc(sizeof(countertype));
        (c->counters)[i] = counter;	/* link from parent to this	     */
	/* this is a void* just to avoid cylces in the declaration, sorry :) */
        counter->counterset = c;	/* link to parent		     */
        counter->my_number = i;		/* own index			     */
        counter->min = c_min[i];
        counter->max = c_max[i];
        counter->value = c_min[i];	/* Init boundaries & value	*/
        counter->active = (i<2);	/* First 2 buttons active	*/

	/* vertical box widget in which buttons are stacked */
        /* arrange the boxes in the parent form widget! */
        n=0;
        XtSetArg(a[n], XtNshowGrip, False); n++;
        if (i>0) {
            XtSetArg(a[n],XtNfromHoriz,((c->counters)[i-1])->cpaned);n++;
            XtSetArg(a[n],XtNhorizDistance,1);n++;
        }
        XtSetArg(a[n], XtNshowGrip, False); n++;
        counter->cpaned = XtCreateManagedWidget("cpaned", panedWidgetClass,
					c->counterform, a, n);

        /* count up - button */
        n=0;
        XtSetArg(a[n], XtNlabel,""); n++;
        counter->cup = XtCreateManagedWidget("cup", commandWidgetClass,
					counter->cpaned, a, n);

        /* counter scrollbar */

        /* Create callback information for jump/scroll events	*/
        /* Pass the counter's onecountertype* as client data	*/
        jumpcb = (XtCallbackRec *) XtMalloc(2*sizeof(XtCallbackRec));
        jumpcb[0].callback = (XtCallbackProc) C_JumpProc;
        jumpcb[0].closure = (XtPointer) counter;
        jumpcb[1].callback = (XtCallbackProc) NULL;
        jumpcb[1].closure = (XtPointer) NULL;
        scrollcb = (XtCallbackRec *) XtMalloc(2*sizeof(XtCallbackRec));
        scrollcb[0].callback = (XtCallbackProc) C_ScrollProc;
        scrollcb[0].closure = (XtPointer) counter;
        scrollcb[1].callback = (XtCallbackProc) NULL;
        scrollcb[1].closure = (XtPointer) NULL;
        n=0;
        XtSetArg(a[n],XtNminimumThumb, 7);	n++;	/* >= 7 pixels high*/
/* ARGH!!! DOESN'T WORK - we HATE Xaw !!!        XtSetArg(a[n],XtNorientation, "vertical");n++;/* Orientation 	   */
        XtSetArg(a[n],XtNlength, SCROLLBARLENGTH);n++;	/* Minimum length    */
        XtSetArg(a[n],XtNheight, SCROLLBARLENGTH);n++;
        XtSetArg(a[n],XtNjumpProc, jumpcb); n++;	/* Register callbacks*/
        XtSetArg(a[n],XtNscrollProc, scrollcb); n++;
        counter->cscroll = XtCreateManagedWidget("cscroll",
                                scrollbarWidgetClass, counter->cpaned, a, n);

        /* count down - button */
        n=0;
        XtSetArg(a[n], XtNlabel,""); n++;
        counter->cdown = XtCreateManagedWidget("cdown", commandWidgetClass,
					counter->cpaned, a, n);

        /* counter display & entry button */
        n=0;
        sprintf(s,"%d",counter->value);
        XtSetArg(a[n], XtNstring, s); n++;		/* Text = value	   */
/*        XtSetArg(a[n], XtNjustify, "center"); n++;	/* Centered text   */
        XtSetArg(a[n], XtNeditType, "edit"); n++;	/* Text entry OK   */
        XtSetArg(a[n], XtNdisplayCaret, False); n++;	/* No text cursor  */
        counter->center = XtCreateManagedWidget("center", asciiTextWidgetClass,
					counter->cpaned, a, n);

        /* toggle button for active widgets */
        n=0;
        XtSetArg(a[n], XtNlabel, "on"); n++;
        XtSetArg(a[n], XtNstate, (i < 2) ); n++;	/* active if <2 btns */
/*        XtSetArg(a[n], XtNshapeStyle, "oval"); n++;*/
        counter->ctoggle = XtCreateManagedWidget("ctoggle", toggleWidgetClass,
					counter->cpaned, a, n);


        /* Add corresponding callbacks to inc/decrement buttons		*/
        /* Depending on the button pressed, the counter is		*/
        /* incremented by the corresponding value (or decremented)	*/
        /* Also add a callback for the "active" toggle button		*/

/* !!!! TODO: das muss auch eleganter gehen ohne diese ganze redundanz.
   !!!! aber wie kriegen wie im callback sonst diese information wieder raus?
   !!!! und: wie werden wir diese Speicherstueckchen wieder los??? bei loeschen
*/

        callt = (callcountertype *) XtMalloc(sizeof(callcountertype));
        callt->ct = c;		/* To which counter set the call belongs */
        callt->ot = counter ;	/* Which exact counter			 */
        callt->nr = i;		/* Which number				 */
	/* One single step						 */
        callt->howmuch = 1;
        XtAddCallback(counter->cup, XtNcallback, C_PlusMinus, callt);

        callt = (callcountertype *) XtMalloc(sizeof(callcountertype));
        callt->ct = c;		/* To which counter set the call belongs */
        callt->ot = counter ;	/* Which exact counter			 */
        callt->nr = i;		/* Which number				 */
        callt->howmuch = -1;
        XtAddCallback(counter->cdown, XtNcallback, C_PlusMinus, callt);

        callt = (callcountertype *) XtMalloc(sizeof(callcountertype));
        callt->ct = c;		/* To which counter set the call belongs */
        callt->ot = counter ;	/* Which exact counter			 */
        callt->nr = i;		/* Which number				 */
        /* Don't change, just update value that was newly entered	 */
        callt->howmuch = 0;

        /* Make <Return> go to end of line instead creating new line	    */
        /* and call C_Entered(callt)					    */
        /* Later, we get the address of callt from the parameter and can    */
        /* take appropriate actions then.				    */
        sprintf(s,"<FocusIn>   : display-caret(on) \n\
                   <FocusOut>  : display-caret(off) \n\
                   <EnterWindow>: display-caret(on) \n\
                   <LeaveWindow>: display-caret(off) C_Entered(%d) \n\
                   <Key>Return : end-of-line() C_Entered(%d)", callt,callt);
        XtOverrideTranslations(counter->center, XtParseTranslationTable(s));

        callt = (callcountertype *) XtMalloc(sizeof(callcountertype));
        callt->ct = c;		/* To which counter set the call belongs */
        callt->ot = counter ;	/* Which exact counter			 */
        callt->nr = i;		/* Which number				 */
        /* Don't care which decrement - we need this callback just for	 */
        /* toggle actions in  C_ChangeActive				 */
        callt->howmuch = 0;
        XtAddCallback(counter->ctoggle, XtNcallback, C_ChangeActive, callt);

    }

    XtFree(s);

#ifdef DEBUG_OUTPUT
printf("COUNTER: %d counters created\n",num_counters);
#endif

    return(c);

}


/*
 * void init_counter(countertype *ct)
 *
 * Initializes bitmaps of counter buttons
 * (Impossible before parent widget is realized - we need its X-display
 *  to create a bitmap! So this function MUST be called AFTER the parent
 *  of the counter widget is realized)
 */

void init_counter(ct)
countertype *ct;
{
    Display *display;   /* temporary display, faster than many X-calls	*/
    Window window;	/* dito						*/
    Arg a[10];
    int n,i;


    display = XtDisplay(ct->parent);
    window = XtWindow(ct->parent);


    /* Init bitmaps for up/down buttons */

    for (i=0; i< ct->num_counters; i++) {
        int len;
        int thick;

	/* Get dimensions of up/down buttons */
        n=0;
        XtSetArg(a[n], XtNwidth, &thick);n++;
        XtSetArg(a[n], XtNheight, &len); n++;
        XtGetValues((ct->counters)[i]->cup,a,n);

        /* Set up-button bitmap */
        n=0;
        XtSetArg(a[n], XtNbitmap, XCreateBitmapFromData(
                 display, window, up_bits, up_width, up_height)
                ); n++;
        XtSetValues((ct->counters)[i]->cup,a,n);

        /* Set size of scrollbar and initial values:	*/
	/* start at button, thumbsize = 10% of scrollbar*/
        n=0;
	/* Make sure the thumb doesn't disappear when at minimum*/
        /* value by resizing it to 93%	of it's former length,	*/
        /* thus always showing the thumb			*/
        XtSetArg(a[n], XtNlength, (Dimension) (93*SCROLLBARLENGTH/100)); n++;
        XtSetArg(a[n], XtNthickness, thick); n++;
        XtSetValues((ct->counters)[i]->cscroll,a,n);
	C_SetScrollbarFromValue((ct->counters)[i]);
        XtCallActionProc((ct->counters)[i]->center, "end-of-line",
                         NULL, NULL, ZERO);		/* Set cursor to EOL */

        /* Set down-button bitmap */
        n=0;
        XtSetArg(a[n], XtNlength, len);
        n=0;
        XtSetArg(a[n], XtNbitmap, XCreateBitmapFromData(
                 display, window, down_bits, down_width, down_height)
                ); n++;
        XtSetValues((ct->counters)[i]->cdown,a,n);

        /* select a smaller font for the number displaying buttons */
        n=0;
        XtSetArg(a[n], XtNfont, XLoadQueryFont(display,"fixed")); n++;
/*        XtSetArg(a[n], XtNjustify, "center"); n++;*/
        XtSetValues((ct->counters)[i]->center,a,n);
        XtSetValues((ct->counters)[i]->ctoggle,a,n);
    };

#ifdef DEBUG_OUTPUT
printf("COUNTER: counters initialized\n");
#endif

}


/* 
 * void destroy_counter(countertype *ct)
 *
 * Destroys an instance of a counter set
 */

void destroy_counter(ct)
countertype *ct;
{
    int i;

    /* destroy all counters */
    /* PROBLEM: callback information not referenced by any leftover pointer */
    /*          so how can i free it? automagically on destroy_widget?	    */
    for (i=0; i<ct->num_counters; i++) {
        XtFree((ct->counters)[i]);
    };
    XtDestroyWidget(ct->counterform);	/* also destroys all children	    */
    XtFree(ct);

#ifdef DEBUG_OUTPUT
printf("COUNTER: counters destroyed\n");
#endif

}


/*
 * void clear_all_counters(countertype *ct)
 *
 * Sets all counters in a counter set to the minimum bound
 */

void clear_all_counters(ct)
countertype *ct;
{
    int i;
    Arg a[1];
    String s = XtMalloc(512);


    /* Reset all values and displayed strings to corresp. lower boundary */

    for (i=0; i< ct->num_counters; i++) {
        onecountertype *ot = (ct->counters)[i];

        ot->value = ot->min;
        sprintf(s,"%d",ot->value);
        XtSetArg(a[0], XtNstring, s);
        XtSetValues(ot->center,a,1);
    };
#ifdef DEBUG_OUTPUT
printf("COUNTER: counters cleared\n");
#endif
        
    XtFree(s);
}


/*
 * int get_number(countertype *ct)
 *
 * Returns number of counters in one counter structure
 */

int get_number(ct)
countertype *ct;
{
    return(ct->num_counters);
}


/*
 * void set_counter(countertype *ct, int number, int value)
 *
 * Sets one counter of a set to a specified value. Checking against
 * boundaries is done, and values are truncated if necessary.
 */

void set_counter(ct, number, value)
countertype *ct;
int number;
int value;
{
    Arg a[1];
    String s = XtMalloc(512);
    onecountertype *ot = (ct->counters)[number];


    /* Check boundaries */

    if (value > ot->max) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: set: value %d exceeded maximum of %d, truncated!\n",value,ot->max);
#endif
        value = ot->max;
    };
    if (value < ot->min) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: set: value %d exceeded minimum of %d, truncated!\n",value,ot->min);
#endif
        value = ot->min;
    };


    /* Set value and displayed string */

    ot->value = value;
    sprintf(s,"%d",ot->value);
    XtSetArg(a[0], XtNstring, s);
    XtSetValues(ot->center,a,1);
    C_SetScrollbarFromValue(ot);	/* Update the scrollbar, too */
        
#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter set to %d\n",number,value);
#endif

    XtFree(s);
}


/*
 * void set_all_counters(countertype *ct, int *values)
 *
 * Sets all counters of a set to a specified value. Boundaries are checked.
 * Values are passed in the array "values".
 */

void set_all_counters(ct, values)
countertype *ct;
int *values;
{
    Arg a[1];
    String s = XtMalloc(512);
    int i;


    for (i=0; i< ct->num_counters; i++) {
        onecountertype *ot = (ct->counters)[i];

        if (values[i] > ot->max) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: setall: value %d exceeded maximum of %d, truncated!\n",
       values[i],ot->max);
#endif
            values[i] = ot->max;
        };
        if (values[i] < ot->min) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: setall: value %d exceeded minimum of %d, truncated!\n",
        values[i],ot->min);
#endif
            values[i] = ot->min;
        };
        ot->value = values[i];
        sprintf(s,"%d",ot->value);
        XtSetArg(a[0], XtNstring, s);
        XtSetValues(ot->center,a,1);
        C_SetScrollbarFromValue(ot);	/* Update the scrollbar, too */
    };

#ifdef DEBUG_OUTPUT
printf("COUNTER: all counters set\n");
#endif

}


/*
 * int get_counter(countertype *ct, int number);
 *
 * Returns the value of the number-th counter in the counter set
 */

int get_counter(ct, number)
countertype *ct;
int number;
{
    return((ct->counters)[number]->value);
}


/*
 * void get_all_counters(countertype *ct, int *values)
 *
 * Writes the actual values of all pointers into the array "values"
 * (that must be allocated!!)
 */

void get_all_counters(ct, values)
countertype *ct;
int *values;
{
    int i;

    for (i = 0; i < ct->num_counters; i++) {
        values[i] = (ct->counters)[i]->value;
    };
}


/*
 * Boolean get_state(countertype *ct, int number)
 *
 * Returns the state of the number-th counter in the set.
 * "True" means the counter is active. Only up to two counters in
 * a set may be active at once
 */

Boolean get_state(ct,number)
countertype *ct;
int number;
{
    return( (ct->counters)[number]->active);
}


/*
 * void get_all_states(countertype *ct, Boolean *states)
 *
 * Returns the states of all counters in "states" (which must be
 * already allocated!).
 * "True" means the counter is active. Only up to two counters in
 * a set may be active at once
 */

void get_all_states(ct, states)
countertype *ct;
Boolean *states;
{
    int i;


    for (i=0; i< ct->num_counters; i++) {
        states[i] = (ct->counters)[i]->active;
    };

}


/*
 * void set_counter_functions(countertype *ct,void (*chval)(),void (*chsta)() )
 *
 * Sets event handling functions:
 * chval(countertype *c, int number, int oldvalue, int newvalue)
 *	is called when the value of one counter changes. Parameters:
 *	number		= which of the single counters
 *	{old,new}value	= value of that counter
 * chsta(countertype *c, int number, Boolean newstate)
 *	is called when the state (on/off) of one counter changes. Parameters:
 *	number		= which of the single counters
 *	newstate	= new state of that counter
 * If chval of chsta is NULL then no corresponding function is called
 */

void set_counter_functions(ct, chval, chsta)
countertype *ct;
void (*chval)();
void (*chsta)();
{
    ct->changedvalue = chval;
    ct->changedstate = chsta;

#ifdef DEBUG_OUTPUT
printf("COUNTER: functions set to %d, %d\n", chval, chsta);
#endif
}


/*
 * void set_counter_text(countertype *ct, int nr, char *text)
 *
 * Sets the label of the active-button of the nr-th counter to "text"
 */

void set_counter_text(ct, nr, text)
countertype *ct;
int nr;
char *text;
{
    Arg a[1];


    XtSetArg(a[0], XtNlabel, text);
    XtSetValues((ct->counters)[nr]->ctoggle,a,1);

#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter's label set to `%s`\n",nr,text);
#endif

}




/************************/
/*			*/
/* Internal functions	*/
/*			*/
/************************/


/*
 * void C_PlusMinus(widget w, callcountertype *callt, XtPointer call_data)
 *
 * This callback function gets called any time the user presses one of
 * the counter increment/decrement button.
 * All information necessary (i.e. which counter, ammount, etc) is
 * passed in "callt" pointing to a callcountertype struct.
 * If a user defined event function is available, it is called at the end.
 */

void C_PlusMinus(widget, callt, call_data)
Widget widget;
callcountertype *callt;
XtPointer call_data;
{


/* !!! TODO: effizienter. muss besser gehen als mit stringwandeln! */

    Arg a[1];
    String s = XtMalloc(512);
    String t = XtMalloc(30);
    countertype *ct = callt->ct;
    int val, oldval;


    /* Get string containing the current counter value from Widget	*/
    /* (text widget indirectly known through "parent" of callt)		*/

    XtSetArg(a[0], XtNstring, &s);
    XtGetValues( (callt->ot)->center,a,1);

#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter incremented by %d\n",callt->nr,callt->howmuch);
printf("         current value as string was %s\n",s);
#endif


    /* convert to number and increment by given ammount */
    /* watching out for the boundaries */

    oldval = ( ((callt->ct)->counters)[callt->nr] )-> value;
    val = atol(s);
    val = val + callt->howmuch;
    if (val > (callt->ot)->max) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: value %d exceeded maximum of %d, truncated!\n",
        val,(callt->ot)->max);
#endif
        val = (callt->ot)->max;
    };
    if (val < (callt->ot)->min) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: value %d exceeded minimum of %d, truncated!\n",
        val,(callt->ot)->min);
#endif
        val = (callt->ot)->min;
    };
    sprintf(t,"%d",val);
#ifdef DEBUG_OUTPUT
printf("COUNTER: new value as string is %s\n",t);
#endif
    XtFree(s);


    /* Enter new value into Widget and counter set*/

    XtSetArg(a[0], XtNstring, t);
    XtSetValues((callt->ot)->center,a,1);

    (callt->ot)->value = val;

    XtFree(t);


    /* Set cursor to end of line (looks nicer ans is more consistent	*/
    /* with direct value entry followed by <return> )			*/

    XtCallActionProc((callt->ot)->center, "end-of-line", NULL,
                     NULL, ZERO);

    /* Update the scrollbar */
    C_SetScrollbarFromValue(callt->ot);

    /* Call user defined event function if avaliable	*/
    /* and value has really changed			*/

    if ( (ct->changedvalue) && (val != oldval) ) {
        (*(ct->changedvalue))(ct, callt->nr, oldval, val);
#ifdef DEBUG_OUTPUT
printf("COUNTER: changedvalue(%d, %d, %d, %d) at %d called\n",
ct, callt->nr, oldval, val, ct->changedvalue);
#endif
    };


}



/*
 * void C_ChangeActive(Widget w, callcountertype *callt, XtPointer call_data)
 *
 * This callback function gets called when one of the "active" buttons
 * in a counter set is pressed.
 * It decides which counters in the set to activate and which to
 * desactivate accordingly.
 * If a user defined event funcion is available, it is called at the end.
 */

void C_ChangeActive(w, callt)/*, call_data)*/
Widget w;
callcountertype *callt;
/*XtPointer call_data;*/
{
    int i;
    int otheractive;
    int old_lastactive;
    countertype *ct = callt->ct;


    /* Ignore clicks on active buttons				*/
    /* Ignore everything if there are less than 3 buttons	*/
    /* If inactive button activated, desactivate least recently */
    /* activated button						*/

    if ( ((callt->ot)->active) || ct->num_counters<3) {
        XtCallActionProc((callt->ot)->ctoggle,"set",NULL,NULL,ZERO);
    } else {

        /* Find the least recently pressed button and the other one	*/
        /* (only 2 buttons are active at once, one of them is the	*/
        /* "lastactive", the other one we're looking for)		*/

        otheractive = -1;
        for(i=0; i< ct->num_counters; i++) {
            if ( ((ct->counters)[i]->active) && (i != ct->lastactive) )  {
                otheractive = i;
            };
        };
        if (otheractive == -1) {
#ifdef DEBUG_OUTPUT
printf("COUNTER: no other active button found while toggling - SHOULDN'T HAPPEN!!\n");
#endif
            return;
        };

        /* Now unset the least recently pressed button, mark the	*/
        /* other one as new "lastactive" and set the pressed button	*/

#ifdef DEBUG_OUTPUT
printf("COUNTER: pressed %d, lastactive %d, other %d\n",
       callt->nr, ct->lastactive, otheractive);
#endif
        old_lastactive = ct->lastactive;
        XtCallActionProc((ct->counters)[ct->lastactive]->ctoggle,
                         "unset",NULL,NULL,ZERO);
        (ct->counters)[ct->lastactive]->active = False;
        ct->lastactive = otheractive;
        XtCallActionProc((callt->ot)->ctoggle,"set",NULL,NULL,ZERO);
        (callt->ot)->active = True;


        /* Call user defined event function if avaliable */

        if (ct->changedstate) {
            (*(ct->changedstate))(ct, callt->nr, old_lastactive);
#ifdef DEBUG_OUTPUT
printf("COUNTER: changedstate(%d, %d, %d) at %d called\n",
ct, callt->nr, old_lastactive, ct->changedstate);
#endif
        };
    };



}


/*
 * C_Entered(Widget w, XEvent *e, String *s, Cardinal *c)
 *
 * Gets called as action procedure by the "center" widget in every
 * counter whenever the return key is pressed there.
 * Calls the C_PlusMinus with the appropriate parameters in turn.
 */

void C_Entered(w, e, s, c)
Widget w;
XEvent *e;
String *s;
Cardinal *c;
{
    /* Convert the string we got from our translation table call entry	*/
    /* to the address of the appropriate "callt" structure and call the	*/
    /* PlusMinus function accordingly					*/

    callcountertype *ct = (callcountertype *)(atol(s[0]));

#ifdef DEBUG_OUTPUT
printf("COUNTER: value entered\n");
#endif

    C_PlusMinus(w, ct, NULL);    
}


/*
 * C_SetScrollbarFromValue(onecountertype *ot)
 *
 * Sets the scrollbar of the given counter widget to represent the
 * value given in that counter.
 */

void C_SetScrollbarFromValue(ot)
onecountertype *ot;
{
    float percentage;

    /* Compute new percentage and set the thumb accordingly	*/
    /* (don't change its' size, so use "-1.0" as new size)	*/
    percentage = (float) (ot->value - ot->min) / (float) (ot->max - ot->min);
#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter's scrollbar set to %f%%, value is %d\n",
	ot->my_number,(1.0-percentage)*100.0,ot->value);
#endif
    /* (reverse percentage - top should be 100%, bottom 0%!	*/
    XawScrollbarSetThumb(ot->cscroll, 1.0 - percentage, -1.0);
    
}

/*
 * C_SetValueFromScrollbar(onecountertype *ot)
 * Sets the value of the counter according to the scrollbar's position.
 * Similiar to C_PlusMinus, but acts on absolute values instead, so
 * we just have to set the value-text, not read it before.
 */

void C_SetValueFromScrollbar(ot)
onecountertype *ot;
{
    Arg a[1];
    float percentage;				/* Scrollbar value (%)	*/
    String s = XtMalloc(30);			/* Value as string	*/
    countertype *ct = ot->counterset;		/* Parent counter set	*/
    int val, oldval;				/* New and old value	*/


    /* Get percentage from scrollbar and convert it to a value	*/
    /* (reverse percentage - top should be 100%, bottom 0%!	*/

    XtSetArg(a[0], XtNtopOfThumb, &percentage);
    XtGetValues(ot->cscroll, a, 1);
    val = (int) ( (1.0 - percentage) * (float) (ot->max - ot->min) ) + ot->min;
#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter's value set to %d, scrollbar is at %f%%\n",
	ot->my_number,val,(1.0-percentage)*100.0);
#endif


    /* If the new value is different from he former one, change	*/
    /* the text in the center widget accordingly		*/
    /* Call the event procedure if the user has defined one	*/

    oldval = ot->value;
    if (oldval != val) {
        sprintf(s,"%d",val);
        XtSetArg(a[0], XtNstring, s);
        XtSetValues(ot->center,a,1);
        ot->value = val;

        /* Set cursor to end of line (looks nicer ans is more consistent    */
        /* with direct value entry followed by <return> )                   */

        XtCallActionProc(ot->center, "end-of-line", NULL,
                         NULL, ZERO);
        if (ct->changedvalue) {
            (*(ct->changedvalue))(ct, ot->my_number, oldval, val);
#ifdef DEBUG_OUTPUT
printf("COUNTER: changedvalue(%d, %d, %d, %d) at %d called\n",
ct, ot->my_number, oldval, val, ct->changedvalue);
#endif
        };
    };

    XtFree(s);

}


/*
 * void C_JumpProc(Widget sb, onecountertype *ot, float *percent)
 *
 * Callback for scrollbar jump events, i.e. when the user positions
 * the thumb directly.
 * Updates the counter's value and redisplays the scrollbar.
 */

void C_JumpProc(sb, ot, percent)
Widget sb;
onecountertype *ot;
float *percent;
{

    /* Simply call the function to take care of the new position */
    C_SetValueFromScrollbar(ot);

}


/*
 * void C_ScrollProc(Widget sb, onecountertype *ot, int pos)
 *
 * Callback for scrollbar scroll events, i.e. when the user clicks the
 * right or left mouse button to move the thumb.
 * Updates the counter's value and redisplays the scrollbar.
 */

void C_ScrollProc(sb, ot, pos)
Widget sb;
onecountertype *ot;
int pos;
{

    Arg a[5];
    int n;
    Dimension len;	/* Scrollbar's length (to compare with position)*/
    float top;		/* Actual position of thumb (%)			*/

    /* Get length of scrollbar and use it to compute the percentage	*/
    /* we want to scroll from out current position (top)		*/
    n=0;
    XtSetArg(a[n], XtNlength, &len); n++;
    XtSetArg(a[n], XtNtopOfThumb, &top); n++;
    XtGetValues(ot->cscroll, a, n);
    top = (float)pos/(float)len/2.0 + top;
    top = top<0.0 ? 0.0 : top;			/* Limit to 0.0 .. 1.0	*/
    top = top>1.0 ? 1.0 : top;

#ifdef DEBUG_OUTPUT
printf("COUNTER: %d. counter's scrollbar scrolled at position %d (length %d)\n", ot->my_number, pos);
printf("COUNTER:     New position = %f%%\n",top);
#endif

    XawScrollbarSetThumb(ot->cscroll, top, -1.0);

    /* Now let's make a new value out of this new position		*/
    C_SetValueFromScrollbar(ot);

}



/*
 * void set_counter_userdata(countertype *ct, void *userdata)
 *
 *   Stores a user defineable pointer in the counter data structure
 *   (not used by the counter functions themselves, just user data)
 */

void set_counter_userdata(ct,userdata)
countertype *ct;
void *userdata;
{
    ct->userdata = userdata;
};


/*
 * void *get_counter_userdata(countertype *ct)
 *
 *   Returns the user defineable pointer
 */
void *get_counter_userdata(ct)
countertype *ct;
{
    return(ct->userdata);
};
