/*
 * Vis.h
 *
 * Public definitions for Array Visualizer
 *
 * Stefan Haenssgen 28-Nov-91
 *
 * Updates :	28-Nov-91  Basic Implementation
 *		28-Dec-91  Structured redesign, formed an ADT out of
 *			    the single data structures
 *		30-Dec-91  Added new Widgets "visselect" and "vs_*"
 *			    for user selection of conditions
 *			   Added "vactive{1,2}" for easy access to
 *			    currently active layers
 *		09-Jan-92  Enhanced the documentation
 *			   Changed the interface: Just one call to
 *			    "visualize()" instead of two calls & X Windows init
 *		10-Jan-92  Added generic comparison function "vcmpfn()" and
 *			    array access function "vaccfn()"
 *			   Added new interface "visualize_userfn()"
 *			   Added vrunning for termination check
 *			   Added "{open,close}_visualizer()" and "VIStype"
 *		16-Jan-92  Renamed functions for easier Modula interfacing:
 *				open_visualizer	   -> OpenVisualizer()
 *				visualize()	   -> Visualize()
 *				visualize_userfn() -> VisualizeUserFn()
 *				close_visualizer   -> CloseVisualizer()
 *		19-Jan-92  Added Widgets for "from-to range": vftomto, vft_*
 *			   Added Values for range: vfromval, vtoval
 *		28-Jan-92  Added support for sockets and fork()ing:
 *			    VISvis  in VIS,
 *			    vforked, vsocketname, vwritesocket  in visual.type
 *			    also: vbindsocket, vreadsocket
 *		29-Jan-92  Updated documentation
 *		30-Jan-92  Also include time.h for selection timeouts
 *			   Added UPDATE_INTERVAL for regual checks on socket
 *		04-Feb-92  Split visual.h into Vis.h (public) and VisP.h
 *			    (private)
 *			   Made VIS.VISvis void* (instead of visualtype *)
 *		05-Feb-92  Added example useage for library libVis.a
 *			   Added ChangeUserFn()
 *		08-Feb-92  Added coordinates as parameter to OpenVisualizer()
 *			   Added note for one-dimensional arrays
 *		09-Feb-92  libm.a necessary for compilation!
 *		10-Feb-92  Second sync flag for function changes
 *		16-Apr-92  New Visualizer for array values (not only boolean)
 *			   Updated function documentation
 *		03-Jul-92  Added V_FLOAT and V_DOUBLE for floating point
 *			    support
 *		07-Aug-92  Introduced Vis_header to make double includes OK
 *		27-Aug-92  Added VisualizeActivation()
 *		24-Oct-92  Made dimensions (0..n-1 == n) clearer
 *		24-Nov-92  Introduced new calls to utilize Visualizers
 *			    as widgets, not as 100% standalone X applications
 *			    (VisRangeWidget() etc)
 *
 *
 * Example for usage for libVis.a:
 *
 *	cc -o foo foo.o libVis.a -lXaw -lXmu -lXt -lXext -lX11 -lm
 *
 */


/*
 *  Description of functions:
 *  -------------------------
 *
 *
 *  VIStype *OpenVisualizer(int x, int y)
 *
 *  Initializes the X Window basics to make further visualization possible.
 *  The return value is later needed for visualization and finally for
 *  "CloseVisualizer()"
 *  "x" and "y" specify the coordinates of the upper left corner of the
 *  Visualizer Window when it appears. If "x" or "y" are less than zero,
 *  the Visualizer will be positioned with user interaction (unless the
 *  user's window manager places windows automatically, anyway).
 *
 *
 *  void CloseVisualizer(VIStype *VIS)
 *
 *  Destroys the X Window basics connected with VIS.
 *  Also kill off background process if there is one.
 * 
 * 
 *
 *  void  Visualize(  VIStype *VIS,
 *                    char    *name,
 *                    ulong    namelen,		( unsigned long )
 *                    void    *adr,
 *                    int     *dims,
 *                    ulong    numdims,		( unsigned long )
 *                    int      type,
 *                    int      condition,
 *                    void    *comparison,
 *                    Boolean  forkflag,
 *                    Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements fulfilling the comparison defined by "condition"
 *    and "comparison" are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    Visualize(VIS,"a", 1, dims, numdims,
 *              V_INT, V_GREATER, &value, FALSE, FALSE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements that are greater than "value", running in the foreground
 *    (i.e. main program being suspended until Visualizer is terminated)
 *  Of course the sync-flag is not set here, because there is no background
 *    process at all.
 *
 *
 *  void  VisualizeRange(  VIStype *VIS,
 *                         char    *name,
 *                         ulong    namelen,		( unsigned long )
 *                         void    *adr,
 *                         int     *dims,
 *                         ulong    numdims,		( unsigned long )
 *                         int      type,
 *                         void    *fromvalue,
 *                         void    *tovalue,
 *		           Boolean  forkflag,
 *                         Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements in the range "fromvalue" to "tovalue"
 *    are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *  Example call:
 *
 *    VisualizeRange(VIS,"a", 0, a, dims, numdims, V_INT, &i, &j,
 *                   TRUE, FALSE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements in the range [i..j] running in the background (i.e. the
 *    main program continues and is able to update the array using
 *    UpdateForkedVisualizer(VIS) ), not synchronizing the update calls.
 *    Then length of the name (i.e. "a") is computed internally since the
 *    value given (i.e. 0) is less than 1.
 *
 *
 *  void  VisualizeUserFn(  VIStype *VIS,
 *                          char    *name,
 *                          ulong    namelen,		( unsigned long )
 *                          void    *adr,
 *                          int     *dims,
 *                          ulong    numdims,		( unsigned long )
 *                          int      type,
 *                          char    *cmptext,
 *                          ulong    cmptextlen,	( unsigned long )
 *                          Boolean  (*compare)(),
 *                          void    *(*access)(),
 *                          void    *envp,
 *                          ulong    envlen,		( unsigned long )
 *			    Boolean  forkflag,
 *                          Boolean  syncupdateflag,
 *                          Boolean  syncfunctionflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *    (the same applies to "cmptextlen")
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  "envp" points to some user data that may be used in the compare
 *    function later on. The length of the data is given in "envlen".
 *  The user data is newly allocated and copied so changing it after-
 *    wards has no effect. Use ChangeUserFn() to alter it.
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *  If "syncfunctionflag" is TRUE, a call to ChangeUserFn() will block.
 *
 *  The "cmptext" describes the comparison. It has to contain one
 *    occurence of "%s" which is automagically replaced by the array name
 *    and thea ctual coordinates.
 *  The user defined comparison and array access functions are used as
 *    follows:
 *
 *  Boolean compare(void *a, int cond, void *envp)
 *
 *      Compares the value of the array element (given in "a") to some
 *        other value, possibly given by the environment pointer "envp"
 *        (which is passed at the 1st call to VisualizeUserFn() and may
 *        be changed using ChangeUserFn() ). The comparison condition
 *        (V_EQUAL, V_LESS etc) is passed in "cond".
 *      The user function is free to ignore "cond" as well as "envp"
 *        and perform whatever operation it likes.
 *
 *  void   *access(void *adr, int *x, int *dims, ulong numdims)
 *
 *      Returns a pointer to the array element at adr[x[0],x[1],...]
 *        where the number of dimensions is in numdims and the size of
 *        each dimension is given in dims[].
 *
 *  All array elements fulfilling the comparison defined by the user
 *    supplied functions are displayed in black, the rest in white.
 *
 *  Example call:
 *
 *    VisualizeUserFn(VIS, "a", 0, a, dims, numdims, V_INT,
 *                    "10 <= %s <= 20", 0, my_compare, my_access,
 *                    &data, sizeof(data), TRUE, TRUE, TRUE)
 *
 *  will display the visualization of the integer array a and highlight
 *    all elements that fulfill the comparison function "my_compare"
 *    using the data in "data" and are accessed via "my_access",
 *    running in the background (i.e. the main program continues and is able
 *    to update the array using UpdateForkedVisualizer(VIS))
 *  The Visualizer is synchronized with the foreground process for both
 *    UpdateForkedVisualizer() and ChangeUserFn() calls.
 *  The length of both the name of the array and the string describing
 *    the user function are computed internally since they are both <1.
 *
 *
 *
 *  void  VisualizeValues(  VIStype *VIS,
 *                          char    *name,
 *                          ulong    namelen,		( unsigned long )
 *                          void    *adr,
 *                          int     *dims,
 *                          ulong    numdims,		( unsigned long )
 *                          int      type,
 *                          Boolean  forkflag,
 *                          Boolean  syncupdateflag)
 *
 *  Creates a visualization of the array with given starting address
 *    "adr" and name "name" of the dimensions [0..dims[i]-1] (whose
 *    number is given in "numdims"). The array is of type "type".
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  The elements of the array are displayed in using grayscale bitmaps,
 *    thus giving a quick overview of the distribution of values in
 *    the array. However, no boolean operators are used.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    VisualizeValues(VIS,"a", 1, a, dims, numdims, V_INT, FALSE, FALSE)
 *
 *  will display the visualization of the integer array a and display
 *    all elements as different shades of gray according to their value.
 *    No background operation (and thus no sync) is initiated.
 *
 *
 *
 *  void  VisualizeActivation(  VIStype *VIS,
 *                              char    *name,
 *                              ulong    namelen,
 *                              void    *adr,
 *                              int     *dims,
 *                              void    *comparison,
 *                              void     (*callfn)(),
 *                              Boolean  forkflag,
 *                              Boolean  syncupdateflag)
 *
 *  Creates a visualization of the 1-dimensional array of activation values
 *    (longs) with given starting address "adr" and name "name" of the
 *    size [0..dims[0]-1]. The length of the array is adjusted so that
 *    it fits into a square (but the additional elements are never referenced
 *    so no need to worry :-).
 *    The length of the name string is passed in "namelen" for compati-
 *    bility with Modula calls. It is ignored if it is less than 1.
 *  All array elements equal to the long value passed in "*comparison"
 *    are displayed in black, the rest in white.
 *  "VIS" is the VIStype* returned by "OpenVisualizer".
 *  "callfn" is a function that gets called whenever the user presses
 *    the middle mouse button inside the Visualizer. If NULL is given,
 *    nothing is called. The function has the following syntax:
 *
 *        callfn(VIStype *VIS, long nr)  [where nr = array index clicked on]
 *
 *  If "forkflag" is TRUE, the Visualizer will be put to work in the
 *    background; the displayed data can be updated using a call to
 *    UpdateForkedVisualizer()
 *  If "syncupdateflag" is TRUE, a call to UpdateForkedVisualizer() will block
 *    until the user presses the sync-button of the Visualzier.
 *
 *
 *  Example call:
 *
 *    VisualizeActivation(VIS,"activation for a", 0, act_bits, &size,
 *              &act_value, NULL, FALSE, FALSE)
 *
 *  will display the visualization of the activation array a and highlight
 *    all elements that are active (i.e. equal to "act_value"), running
 *    in the foreground (i.e. main program being suspended until Visualizer
 *    is terminated). When the mouse is clicked, nothing will be done.
 *  Of course the sync-flag is not set here, because there is no background
 *    process at all.
 *
 *
 *
 * void ChangeUserFn(VIStype *VIS, Boolean (*newfn)(), char* str, ulong strl,
 *                   void *dat, long dlen)
 *
 * Transmit new data for the user function of the Visualizer in the background.
 *   The new function to be called is pointed to by "fn",
 * the string describing the function is "str", "dat" points to the
 * user data to be transmitted and "dlen" is the length of the user data.
 * "str" has to contain one occurence of "%s" which is automatically replaced
 * by the name of the array being visualized. Its length is given in "strl"
 * which is ignored if it is less than one.
 *
 * If "fn" is NULL then no function is given and the old one is used further.
 * If "str" is NULL the string isn't changed and if "dat" is NULL, no
 * data is transmitted.
 *
 * This function requires that the Visualizer uses a User defined Function
 * (i.e. was called using VisualizeUserFn()) and is running in the background. 
 *
 *
 *
 * void UpdateForkedVisualizer(VIStype *VIS)
 *
 * Transmit the new contents of the visualized array to the Visualizer
 * running in the background
 *
 *
 **
 **   NOTES
 **   =====
 **
 ** - One-dimensional arrays are now supported, too. To visualize them
 **   they are shown as 2-dimensional arrays with the dimension (m,n)
 **   where m*n = l and l = size of the one-dimensional array. The Visualizer
 **   tries to keep m and n as close to sqrt(l) as possible. However, in
 **   the case of a prime length the display may degenerate to 1*l.
 **   Note also that an array with n elements uses indices 0..n-1 so
 **   m*n will try to fit n-1 ! (??? is that ok ???)
 **   Exception: VisualizeActivation() always generates a square display
 **              by stuffing the given array till it fits
 **
 ** - One VIS struct can only be used once at a time, so to use multiple
 **   concurrent Visualizers, you'll have to do an "OpenVisualizer()" for
 **   each of them. After a Visualizer has exited, however, its VIS can
 **   be reused, i.e. sequentially used Visualizers (not in the background)
 **   can use the same VIS.
 **
 ** - Since the Visualizer does its own X Event Loop, it only returns
 **   when the "quit" button is pressed (if it isn't put in the background)
 **
 ** - If you have problems with "undefined stdout" etc, include <stdio.h>
 **   before Vis.h
 *
 *
 * New standalone calls for Widget creation:
 *
 *
 * VIStype *VisWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Creates a normal Visualizer Widget with the given parent Widget,
 * returns the complete Visualizer data structure associated with it.
 * Same parameters as Visualize() but no fork/syncupdate flags, no
 * VIS parameter, however an additional "parent" (which must be
 * already initialized!)
 *
 *
 * VIStype *VisRangeWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, fromvalue, tovalue)
 *
 * Dito with Visualizer type "Range"
 *
 *
 * VIStype *VisUserFnWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Dito with Visualizer type "User Function"
 *
 *
 * VIStype *VisValuesWidget(parent, name, namelen, adr, dims, numdims,
 *                    type, condition, comparison)
 *
 * Dito with Visualizer type "Range"
 *
 *
 * VIStype *VisActivationWidget(parent, name, namelen, adr, dims,
 *			 	comparison, callfn)
 *
 * Dito with Visualizer type "Activations"
 *
 *
 * void UpdateVisWidget(VIStype *VIS)
 *
 * Update a Visualizer's display when the data it's showing have changed
 *
 *
 * NOTE: If one of these Visualizer Widgets is quit, its VIS->VISvis
 *       gets changed to NULL, so it's possible to tell if it has exited
 */

#ifndef Vis_header
#define Vis_header

/* Necessary definitions for X Windows & Maths*/

#include <X11/Intrinsic.h>


/* Definitions of type-tokens */

#define V_CHAR		0
#define V_INT		1
#define V_LONG		2
#define V_FLOAT		3
#define V_DOUBLE	4


/* Definitions of comparison operators */

#define V_GREATER	0
#define V_EQUAL		1
#define V_LESS		2


/* Common context for visualizer */

typedef struct {
  XtAppContext	VISapp_con;	/* Application Context (X Windows)	*/
  Widget	VISparent;	/* Main parent Widget			*/
  void		*VISvis;	/* Visualizer being used (or NULL)	*/
} VIStype;


/* For shorter headers :-) */

#ifndef ulong
#    define ulong unsigned long
#endif


/* Declarations of functions that the Visualizer offers */

#ifndef VISUAL_ITSELF
extern VIStype		*OpenVisualizer();
extern void		 Visualize();
extern void		 VisualizeUserFn();
extern void		 VisualizeRange();
extern void		 VisualizeValues();
extern void		 VisualizeActivation();
extern void		 CloseVisualizer();
extern void		 UpdateForkedVisualizer();
extern VIStype		*VisWidget();
extern VIStype		*VisRangeWidget();
extern VIStype		*VisUserFnWidget();
extern VIStype		*VisValuesWidget();
extern VIStype		*VisActivationWidget();
extern void		 UpdateVisWidget();
#endif

#endif /* Vis_header */
