/*
 * DebugFnV.c
 *
 * Implementation of the debug functions for Modula-to-C programs
 * (Additional graphic stuff: Visualizer, Windows etc)
 *
 * Stefan Haenssgen  10-Jul-92
 *
 * Updates:
 *
 *	26-Oct-92	Split into DebugFn.c (basic functions without graphics)
 *			 and DebugFnV.c (additional Visualizer / X11 stuff)
 *			Here: X11 / Visulizer specific stuff
 *	27-Oct-92	Modified to use the new ADT DebugTree (global DTree)
 *	28-Oct-92	Added Quit button for tree, root node no longer exits
 *	29-Oct-92	Improved resizing of Tree (directly resize popup)
 *			Made size of Profile Window depend on Font size
 *	30-Oct-92	Debugging: Name of newly created Tree in UpdateTreeW.
 *			 must be the same (or it won't be found later!)
 *	31-Oct-92	Renamed "ShowProfileCallback" to "S.P.Quit" and
 *			 "ShowTreeCallback" to "ShowTreeClick"
 *			Show RunCount of DebugNode instead of Count taken
 *			 from Profile Table!
 *	01-Nov-92	Modernized calls - use global DTree & PTable (no
 *			 more node information as parameters)
 *			Quit ShowTree() when either profile or tree exits
 *	08-Nov-92	Wrote ProfileGraphWidget() to show a graphical view
 *			 of the profile distribution
 *			Wrote ProfileGraphQuit() to handle popdown & quit
 *	09-Nov-92	Added resizing of Viewport
 *			Added title for profile table
 *	29-Nov-92	Added UpdateProfileGraph() to simply replace the
 *			 old contents with a new graph
 *			Packed common code into ProfileGraphInnerpane()
 *			Renamed ProfileGraphWidget to ShowGraphWidget
 *	10-Jan-93	Display JoinInfo if any
 *	13-Jan-93	Display PE if no JoinInfo (i.e. PE is determ.)
 *
 */

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include "DebugFnV.h"


/* Geometry of Profile Information Widget & Profile Graph*/
/*!!!!!!! SHOULD BE COMPUTED (more flexible)	*/

#define PROFILE_MAX_HEIGHT	500
#define PROFILE_MAX_WIDTH	400
#define GRAPH_HEIGHT		350

/*
 * void DebugVisualizeActives(long *AVector, long AValue, long from, long to,
 *			      char *name, ulong namelen)
 *
 * Displays a Visualization of the currently active forall-variables.
 * "AVector" is the vector of activation values, "AValue" is the value of
 * the active PEs, "from" & "to" are the range of the indices and
 * "name" is the name of the forall variable (with the length namelen
 * which can be ignored by setting it to 0)
 *
 */
void DebugVisualizeActives(AVector,AValue,from,to,name,namelen)
long *AVector;		/* Vector with activation values	*/
long  AValue;		/* Currently active Value		*/
long  from;		/* Lower bound of FORALL variable	*/
long  to;		/* Upper bound				*/
char *name;		/* Name of FORALL variable		*/
ulong namelen;		/* Length of name (if >0)		*/
{
    VIStype *vis;
    int      dims[1];
    char     myname[256];


    dims[0] = to-from+1;/*!!!!!!! allow for arbitrary ranges not just 0..x */

    if ((namelen==0) &&(name != NULL))	/* Make a local copy of the name */
        namelen=strlen(name);
    strncpy(myname,name,namelen);
    myname[namelen]='\0';
    
    vis = OpenVisualizer(-1,-1);
    VisualizeActivation(vis, myname,namelen,/* Visualize activations	*/
              AVector,dims,		/* Just the AB-array		*/
              &AValue,NULL,		/* Which ones are active	*/
              FALSE,FALSE);		/* No background/sync		*/
    CloseVisualizer(vis);

}


/*
 * ShowProfileQuit(Widget w, Widget popupW, XtPointer dummy)
 *
 * Callback procedure for click on Quit button of Profile Info.
 * Pops down the information widget and destroys it.
 *
 */
void ShowProfileQuit(w,popupW,dummy)
Widget w;
Widget popupW;
XtPointer dummy;
{

    XtPopdown(popupW);			/* Pop the widget down..	*/
    XtDestroyWidget(popupW);		/* ..and destroy it		*/
    profile_active--;			/* One profile view less	*/

}



/*
 * char *MakeProfileText(int *lines)
 *
 * Prints the contents of the profile table nicely into the given string s.
 * Returns the number of lines generated in &lines.
 * Note: The caller has to deallocate the returned string using free()
 *       it it's no longer needed!
 *
 */
char *MakeProfileText(lines)
int *lines;
{
    char *s;			/* String to output to			*/
    char *sorig;		/* Original address of s		*/
    char r[256];		/* Temporary string			*/
    long i;


    /* Output the profiling information into a string */

    s=(char*)malloc(256*P_TABLE_SIZE);  /* YUCK!!! but what else?	*/
    s[0]='\0';				/* Init String			*/
    sorig=s;
    sprintf(r,"Profiling information table\n\n");
    strcat(s,r);
    sprintf(r," Line Type \t: Count  [...]\n");
    strcat(s,r);
    sprintf(r,"----- -----------------------------------\n");
    strcat(s,r);
    (*lines)=4;				/* Output profile info table:	*/

    /* Go through whole table using line numbers 1..maxProfileLine	*/
    /* and looking at the information at the hashed position, if any	*/
    for (i=1; i<=PTable->maxProfileLine; i++) {
        long	     l     = phash(i);		/* Hashed line number	 */
        ProfileNode *p     = PTable->ProfileTable[l]; /* Entry to look at*/
        Boolean	     first = True;		/* 1st entry in line?	 */

        while (p!=NULL) {		/* Print only non-empty entries	*/
            if (p->MLine == i) {	/* Correct line, output it:	*/
                if (first) {		
                    sprintf(r,"%5d ",i);/* Print line number if 1st time*/
                    strcat(s,r);
                    first=False;
                } else {
                    sprintf(r," \t/ ");	/* Else print separator		*/
                    strcat(s,r);
                };
                sprintf(r,"%s\t: %d ",	/* Print the information	*/
                           _D_TypeString[p->Type], p->RunCount);
                strcat(s,r);
            };
            p = p->NextNode;	       /* Examine whole chain of entries*/
        };	
        if (!first) {			/* End line if there was any	*/
            sprintf(r,"\n");
            strcat(s,r);
            (*lines)++;			/* Update our line count	*/
        };
    };
    return(sorig);
}



/*
 * Widget ShowProfileWidget(Widget parent)
 *
 * Displays the profiling information in ProfileTable[].
 * Returns the popup Widget that it creates.
 *
 */
Widget ShowProfileWidget(parent)
Widget parent;
{
    Arg         a[10];		/* Arguments for Widget creation	*/
    int         n;
    Widget 	popupW;		/* Popup Widget to appear		*/
    Widget	paneW;		/* Container for button & text		*/
    Widget	quitW;		/* Quit button				*/
    Widget 	profileW;	/* Tree Widget to layout the Node Tree	*/
    int		lines;		/* Line count of profile output		*/
    Dimension	wi,he;		/* Width & height of the Widget		*/
    Dimension	fonthe;		/* Height of font			*/
    XFontStruct *FontS;		/* Font of Widget			*/
    char	*s;

    s = MakeProfileText(&lines);	/* Generate the text to be shown*/


    /* Initialize the basic Widgets using "parent" as parent Widget	*/
    /* (Pane contains quit button & profile-info text)			*/

    popupW = XtCreatePopupShell("Profile Information",topLevelShellWidgetClass,
                                parent,NULL,ZERO);
    n=0;
    XtSetArg(a[n], XtNorientation, XtorientVertical); n++;
    paneW = XtCreateManagedWidget("profilepane",panedWidgetClass,
                                  popupW,a,n);
    n=0;
    XtSetArg(a[n], XtNlabel, "Quit"); n++;
    XtSetArg(a[n], XtNshowGrip, False); n++;
    quitW = XtCreateManagedWidget("profilequit",commandWidgetClass,
                                  paneW,a,n);


    wi = PROFILE_MAX_WIDTH;		/* Set useful dimensions of	*/
    he = PROFILE_MAX_HEIGHT;		/*  Text Widget			*/

    XtVaGetValues(quitW,XtNfont,&FontS,NULL);	/* Examine height of font */
    if (FontS)
      fonthe = FontS->max_bounds.ascent +  FontS->max_bounds.descent +2;
    else
      fonthe=16;			/* Default if no font given	*/
    
    if (lines*fonthe < he)		/* Use up to P_M_H lines	*/
        he = lines*fonthe;		/* (or less if small enough)	*/

    n=0;
    XtSetArg(a[n], XtNtype, XawAsciiString); n++;
    XtSetArg(a[n], XtNstring, s); n++;	/* Use text generated above	*/
    XtSetArg(a[n], XtNscrollHorizontal, XawtextScrollWhenNeeded); n++;
    XtSetArg(a[n], XtNscrollVertical, XawtextScrollWhenNeeded); n++;
/*  XtSetArg(a[n], XtNresize, XawtextResizeBoth);n++;/* try to grow with text*/
    XtSetArg(a[n], XtNeditType, XawtextRead); n++;
    XtSetArg(a[n], XtNwidth, wi); n++;
    XtSetArg(a[n], XtNheight, he); n++;
    XtSetArg(a[n], XtNshowGrip, True); n++;
    XtSetArg(a[n], XtNallowResize, True); n++;
    profileW = XtCreateManagedWidget("profiletext",asciiTextWidgetClass,
                                  paneW,a,n);

    XtAddCallback(quitW, XtNcallback, ShowProfileQuit, popupW);


    /* Pop up the Text Widget */

    profile_active++;	/* One more profile being shown now */
    XtPopup(popupW,XtGrabNone);

    free(s);

    return(popupW);	/* Return our popup window */
}



/*
 * void UpdateProfileWidget(Widget profilePopupW)
 *
 * Updates the displayed profile information.
 * Note: profilePopupW has to exist! uses global profile table!
 *       uses name of profile widget to find it from the popup shell
 *
 */
void UpdateProfileWidget(profilePopupW)
Widget profilePopupW;
{
    Widget	profileW;	/* The profile text widget itself	*/
    Widget	quitW;		/* Quit button				*/
    Arg         a[10];		/* Arguments for Widget creation	*/
    int         n;
    int		lines;		/* Line count of profile output		*/
    Dimension	wi,he;		/* Width & height of the Widget		*/
    Dimension	fonthe;		/* Height of font			*/
    XFontStruct *FontS;		/* Font of Widget			*/
    XtWidgetGeometry prefer_quit;
    char	*s;


    /* Find the right widget starting fromthe popup shell given	*/

    profileW = XtNameToWidget(profilePopupW,"*profiletext");
/*printf("Found profile text widget %d\n",profileW);*/


    s = MakeProfileText(&lines);	/* Generate the text to be shown*/


    /* Set useful dimensions of Text Widget */

    wi = PROFILE_MAX_WIDTH;		/* Set useful dimensions of	*/
    he = PROFILE_MAX_HEIGHT;		/*  Text Widget			*/

    XtVaGetValues(profileW,XtNfont,&FontS,NULL); /* Examine height of font */
    if (FontS)
      fonthe = FontS->max_bounds.ascent +  FontS->max_bounds.descent +2 ;
    else
      fonthe=16;			/* Default if no font given	*/
    
    if (lines*fonthe < he)		/* Use up to P_M_H lines	*/
        he = lines*fonthe;		/* (or less if small enough)	*/


/*  XtMakeResizeRequest(profileW,wi,he,NULL,NULL); /* Try to get right size */

    n=0;
    XtSetArg(a[n], XtNstring, s); n++;		/* Use text generated	*/
    XtSetValues(profileW,a,n);			/*  above		*/

    quitW = XtNameToWidget(profilePopupW,"*profilequit");
    XtQueryGeometry(quitW,NULL,&prefer_quit);	/* Resize Widget	*/
    XtResizeWidget(profilePopupW,		/*  to fit (with text	*/
		   wi + prefer_quit.width,	/*  and button)		*/
		   he + prefer_quit.height+1, 1);

    free(s);

}



/*
 * ShowInfoCallback(Widget w, Widget showpopW, XtPointer dummy)
 *
 * Callback procedure for click on information display.
 * Pops down the information widget and destroys it.
 *
 */
void ShowInfoCallback(w,showpopW,dummy)
Widget w;
Widget showpopW;
XtPointer dummy;
{

    XtPopdown(showpopW);		/* Pop the widget down..	*/
    XtDestroyWidget(showpopW);		/* ..and destroy it		*/

}



/*
 * ShowTreeQuit(Widget w, Widget popupW, XtPointer dummy)
 *
 * Callback procedure for Quit button. Pops all down.
 *
 */
void ShowTreeQuit(w,popupW,dummy)
Widget w;
Widget popupW;
XtPointer dummy;
{

    XtPopdown(popupW);
    XtDestroyWidget(popupW);
    tree_active--;			/* One tree less		*/
    return;

};



/*
 * ShowTreeClick(Widget w, DebugNode *node, XtPointer dummy)
 *
 * Callback procedure for clicks on elements of the tree.
 * Pops up a widget showing more information on the selected element.
 *
 */
void ShowTreeClick(w,node,dummy)
Widget w;
DebugNode *node;
XtPointer dummy;
{
    Arg		a[10];
    int		n;
    Widget	showpopW;	/* Popup for the info widget		*/
    Widget	showinfoW;	/* Information diplay & exit button	*/
    char	s[2048];	/* Information string			*/
    Position	x,y;		/* Where to put the widget		*/
    Widget	top;		/* Topwindow (parent of parent of w)	*/
				/* i.e. our popup shell			*/
    JoinInfo	*ji;		/* Info about formerly joined nodes	*/

    if (node == NULL)
        return;
    top = XtParent(XtParent(w));


    /* Position the info widget above the button used to called it */

    XtTranslateCoords(w,-3,-3,&x,&y);


    /* Write the node information into a button text and create the widgets */

    n=0;
    XtSetArg(a[n], XtNx, x); n++;
    XtSetArg(a[n], XtNy, y); n++;
    showpopW = XtCreatePopupShell("NodeInfo",transientShellWidgetClass,
                                  top,a,n);
    sprintf(s,"Type    : %s \nContext : %s \nLines   : %d..%d \nActVal  : %d \nRunCount: %d",
              _D_TypeString[node->Type],_D_ContextString[node->Context],
              node->MLine, node->MToLine,node->ActValue,node->RunCount);
    
    /* Add info about joined nodes, if any - else add PE this node's on	*/

    if (ji = node->Joined) {
	char r[2048];
	sprintf(r,"%s\nJoinInfo: ",s);
	strcpy(s,r);
	while (ji) {
	    sprintf(r,"%s\n  ID %d RC %d (PE %d) ",
		      s,ji->ID,ji->RunCount, ji->PE);
	    strcpy(s,r);
	    ji = ji->Next;
	};
    } else {
	char r[2048];
	sprintf(r,"%s\nPE      : %d",s,node->PE);
	strcpy(s,r);
    };

    n=0;
    XtSetArg(a[n], XtNlabel, s); n++;
    XtSetArg(a[n], XtNhighlightThickness, 0); n++;
    showinfoW = XtCreateManagedWidget("showinfo",commandWidgetClass,
                                      showpopW,a,n);


    /* Add callback to pop the widget down when clicked */

    XtAddCallback(showinfoW, XtNcallback, ShowInfoCallback, showpopW);


    /* Show the Widget and focus all input on it (disable rest) */

    XtPopup(showpopW,XtGrabExclusive);

    /* Resize the widget that all text fits inside!		*/
    /* (necessary because of fixed button size in xdbx)		*/
    {
        XtWidgetGeometry prefgeo;

        XtQueryGeometry(showinfoW,NULL,&prefgeo);	/* whaddaya like? */
        XtResizeWidget(showpopW,prefgeo.width,prefgeo.height, /* gimme!   */
                       prefgeo.border_width);
    }

}



/*
 * ShowTreeAddNode(DebugNode *node, Widget parentW, Widget treeW)
 *
 * Recursively adds all children of the node to the tree widget
 * (parentW == NULL -> root widget in tree widget)
 *
 */
void ShowTreeAddNode(node,parentW,treeW)
DebugNode *node;
Widget parentW;
Widget treeW;
{
    Widget	thisW;		/* The Widget we're creating		*/
    Arg		a[10];		/* Arguments for creation		*/
    int		n;		/* Argument count			*/
    char	s[256];		/* Widget label				*/
    DebugNode	*p;		/* For traversing the DebugNode tree	*/


    if (node == NULL)		/* End at leafs				*/
        return;

    p = node->Parent;		/* Recursively generate widget tree	*/
    do {
				/* Generate name of node		*/
        sprintf(s,"%s", _D_TypeString[node->Type]);
        n=0;
        XtSetArg(a[n], XtNlabel, s); n++;
        XtSetArg(a[n], XtNborderWidth, 0); n++;
        XtSetArg(a[n], XtNhighlightThickness, 1); n++;
        XtSetArg(a[n], XtNtreeParent, parentW); n++;	/* Set Parent	*/
        thisW = XtCreateManagedWidget(s, commandWidgetClass, treeW,
                                      a,n);		/* Place in tree*/
				/* Set callback function to show infos	*/
        XtAddCallback(thisW, XtNcallback, ShowTreeClick, node);

        if (node->Child != NULL)        /* Add children of this node	*/
            ShowTreeAddNode(node->Child,thisW,treeW);
        node = node->RBrother;          /* Add brothers			*/
    } while ((p!=NULL) && (p->Child!=node)); /* Until at start node again */

}



/*
 * Widget ShowTreeWidget(Widget parent)
 *
 * Display the gloabl DebugNode DTree, using X Windows
 * to generate a graphical view. The parent widget is "parent".
 * Returns the popup Widget that it creates.
 *
 */
Widget ShowTreeWidget(parent)
Widget parent;
{
    Arg         a[10];		/* Arguments for Widget creation	*/
    int         n;
    Widget 	popupW;		/* Popup Widget to appear		*/
    Widget	paneW;		/* Paned Widget for button & tree	*/
    Widget 	quitW;		/* Quit button				*/
    Widget 	treeW;		/* Tree Widget to layout the Node Tree	*/
    DebugNode	*node;		/* Start node				*/


    node = DTree->RootNode;	/* Show whole Tree			*/


    /* Initialize the basic Widgets using "parent" as parent */

    n=ZERO;
    popupW = XtCreatePopupShell("Structure Tree",topLevelShellWidgetClass,
                                parent,NULL,ZERO);

    n=0;
    XtSetArg(a[n], XtNorientation, XtorientVertical); n++;
    paneW = XtCreateManagedWidget("treepane",panedWidgetClass,
                                  popupW,a,n);
    n=0;
    XtSetArg(a[n], XtNlabel, "Quit"); n++;
    XtSetArg(a[n], XtNshowGrip, False); n++;
    XtSetArg(a[n], XtNallowResize, False); n++;
    quitW = XtCreateManagedWidget("treequit",commandWidgetClass,
                                  paneW,a,n);
    n=0;
    XtSetArg(a[n], XtNautoReconfigure, FALSE); n++;/* For faster construction*/
    XtSetArg(a[n], XtNgravity, NorthGravity); n++; /* Top-to-bottom arrangemt*/
    XtSetArg(a[n], XtNhSpace, 10); n++;		   /* Spacing for nodes	     */
    XtSetArg(a[n], XtNvSpace, 10); n++;
    XtSetArg(a[n], XtNshowGrip, False); n++;	   /* For paned widget	     */
    XtSetArg(a[n], XtNallowResize, True); n++;
    XtSetArg(a[n], XtNresizeToPreferred, True); n++;
/*    XtSetArg(a[n], XtNskipAdjust, True); n++;*/
    treeW = XtCreateManagedWidget("treetree",treeWidgetClass,
                                  paneW,a,n);


    /* Recursively add all children DebugNodes of "node" */

    ShowTreeAddNode(node,NULL,treeW);

    
    XtAddCallback(quitW, XtNcallback, ShowTreeQuit, popupW);

    
    /* Do the tree layout and pop it all up */

    tree_active++;		/* One more tree being shown now	*/
    XawTreeForceLayout(treeW);
    XtPopup(popupW,XtGrabNone);

    return(popupW);	/* Return our popup window */
}



/*
 * void UpdateTreeWidget(Widget treePopupW)
 *
 * Updates the display of the structure tree (in the global DTree)
 * Note: treeW has to exist already! looks for "real" tree widget starting from
 *       given treePopupW, searching by its name.
 *
 */
void UpdateTreeWidget(treePopupW)
Widget treePopupW;
{
    Widget	treeW;		/* Real tree widget			*/
    Widget	quitW;		/* Quit button				*/
    Widget	parentW;	/* Parent widget of real tree widget	*/
    DebugNode 	*node;		/* Start node				*/
    Arg         a[10];		/* Arguments for Widget creation	*/
    int         n;
    XtWidgetGeometry prefer_tree;	/* Preferred geometry of widgets*/
    XtWidgetGeometry prefer_quit;


    node = DTree->RootNode;	/* Use whole global DTree		*/


    /* Find the right widget starting from the popup shell given	*/

    treeW = XtNameToWidget(treePopupW,"*treetree");
    parentW = XtParent(treeW);			/* Remember parent	*/
    XtDestroyWidget(treeW);			/* Kill the old widget	*/
/*printf("Found & killed old tree widget %d\n",treeW);*/

    /* Create a new tree widget with the same parent */
    n=0;
    XtSetArg(a[n], XtNautoReconfigure, FALSE); n++;/* For faster construction*/
    XtSetArg(a[n], XtNgravity, NorthGravity); n++; /* Top-to-bottom arrangemt*/
    XtSetArg(a[n], XtNhSpace, 10); n++;		   /* Spacing for nodes	     */
    XtSetArg(a[n], XtNvSpace, 10); n++; 
    XtSetArg(a[n], XtNshowGrip, False); n++;	   /* For paned widget	     */
    XtSetArg(a[n], XtNallowResize, True); n++;
    XtSetArg(a[n], XtNresizeToPreferred, True); n++;
/*    XtSetArg(a[n], XtNskipAdjust, True); n++;*/
    treeW = XtCreateManagedWidget("treetree",treeWidgetClass,
                                  parentW,a,n);

    /* Recursively add all new children DebugNodes of "node" */

    ShowTreeAddNode(node,NULL,treeW);

    /* Re-do the tree layout and resize the parent to fit the new tree */

    XawTreeForceLayout(treeW);
    XtQueryGeometry(treeW,NULL,&prefer_tree);
    quitW = XtNameToWidget(treePopupW,"*treequit");
    XtQueryGeometry(quitW,NULL,&prefer_quit);
/* XtMakeResizeRequest(treeW,prefer_tree.width,prefer_tree.height,NULL,NULL);*/
    XtResizeWidget(treePopupW,
		   prefer_tree.width+prefer_quit.width,
		   prefer_tree.height+prefer_quit.height+1, 1);
/*  hehe, nice too:  for (n=0; n<300; n++) XtMoveWidget(treePopupW,n*2,n);*/

}



/*
 * ProfileGraphQuit(Widget w, Widget popupW, XtPointer dummy)
 *
 * Callback procedure for click on Quit button of Profile Graph Display.
 * Pops down the profile graph widget and destroys it.
 *
 */
void ProfileGraphQuit(w,popupW,dummy)
Widget w;
Widget popupW;
XtPointer dummy;
{

    XtPopdown(popupW);			/* Pop the widget down..	*/
    XtDestroyWidget(popupW);		/* ..and destroy it		*/

}


/*
 * Widget ProfileGraphLine(Widget parent, char *type, char *line,
 *			   char *count, long length)
 *
 * Creates a Box Widget containing one line of profile information
 * with a bar graph. A bar length of 0 means "no bar graph" for this
 * line, e.g. for title lines etc.
 *
 */
Widget ProfileGraphLine(parent,type,line,count,length)
Widget parent;
char  *type;
char  *line;
char  *count;
long   length;
{
    Arg         args[10];	/* Arguments for Widget creation	*/
    int         n;
    Widget	boxW,nameW,	/* Widgets forming one graph line	*/
    		lineW,countW,
    		graphW;
    
    if (parent == NULL)
      return(NULL);

    /* Create box to hold the labels & the graph */
    n=0;
    XtSetArg(args[n], XtNorientation, XtorientHorizontal); n++;
    XtSetArg(args[n], XtNshowGrip, False); n++;
    XtSetArg(args[n], XtNallowResize, True); n++;
    XtSetArg(args[n], XtNvSpace, 1); n++;
    boxW = XtCreateManagedWidget("pgraphbox", boxWidgetClass, parent,
				args,n);
    n=0;
    XtSetArg(args[n], XtNlabel, type); n++;		/* Construct type   */
    XtSetArg(args[n], XtNborderWidth, 0); n++;
    XtSetArg(args[n], XtNjustify, XtJustifyLeft); n++;
    nameW = XtCreateManagedWidget("pgraphname", labelWidgetClass, boxW,
				 args,n);
    n=0;
    XtSetArg(args[n], XtNlabel, line); n++;		/* Source code line */
    XtSetArg(args[n], XtNborderWidth, 0); n++;
    XtSetArg(args[n], XtNjustify, XtJustifyLeft); n++;
    lineW = XtCreateManagedWidget("pgraphline", labelWidgetClass, boxW,
				 args,n);
    n=0;
    XtSetArg(args[n], XtNlabel, count); n++;		/* Run Count	    */
    XtSetArg(args[n], XtNborderWidth, 0); n++;
    XtSetArg(args[n], XtNjustify, XtJustifyLeft); n++;
    countW = XtCreateManagedWidget("pgraphcount", labelWidgetClass, boxW,
				  args,n);

    if (length > 0) {	     /* Add black bar with given length (if needed) */
	n=0;
	XtSetArg(args[n], XtNborderWidth, 0); n++;
	XtSetArg(args[n], XtNbackground,		/* Paint it black   */
		          XtDefaultForeground); n++;
	XtSetArg(args[n], XtNwidth,			/* Right length	    */
		          (Dimension)length); n++;
	graphW = XtCreateManagedWidget("pgraphgraph", labelWidgetClass, boxW,
				      args,n);
    };
    
    return(boxW);		/* Return the resulting Widget */
    
}


/*
 * Widget ProfileGraphInnerpane(Widget parent, ProfileNode **pstat,
 *				long numentries)
 *
 * Creates the inner pane of the Profile Graph, containing all
 * profile bars according to the profile statistics "pstat" (with
 * "numentries" elements).
 *
 */
Widget ProfileGraphInnerpane(parent,pstat,numentries)
Widget parent;
ProfileNode **pstat;
long numentries;
{
    Arg         args[10];	/* Arguments for Widget creation	*/
    int         n;
    Widget 	viewW;		/* Viewport to contain graphic		*/
    Widget	innerpaneW;	/* Paned Widget inside viewport		*/
    double	rs;		/* Relative size compared to max RunCnt	*/
    int		i;
    int		maxlen = 200;	/* MAXIMUM LENGTH OF ONE GRAPH		*/
    Widget	onebox;		/* Typical profile line box		*/

    
    if ((parent==NULL) || (pstat==NULL))
      return;

    n=0;
    XtSetArg(args[n], XtNallowVert, True); n++;
    XtSetArg(args[n], XtNshowGrip, False); n++;
    XtSetArg(args[n], XtNheight, GRAPH_HEIGHT); n++;
    XtSetArg(args[n], XtNforceBars, True); n++;
    viewW = XtCreateManagedWidget("pgraphview", viewportWidgetClass,
				  parent, args, n);
    
    n=0;
    XtSetArg(args[n], XtNinternalBorderWidth, 0); n++;
    XtSetArg(args[n], XtNshowGrip, False); n++;
    innerpaneW = XtCreateManagedWidget("pgraphinnerpaned", panedWidgetClass,
				       viewW, args, n);

    /* Generate the contents of the viewport: Name, Line, Count, Graph	*/
    /*  for each Statistics entry					*/

    onebox = ProfileGraphLine(innerpaneW,"Construct  ", /* Line with expla-  */
			      "  Line","  Run Count",0);/*  nations	     */
    
    rs = (double)maxlen / (double)pstat[0]->RunCount;	/* For sizing graphs */
    for (i=0; i<numentries; i++) {
	char	s1[256],s2[256],s3[265];
	long	length;
	
	sprintf(s1,"%-11s",_D_TypeString[pstat[i]->Type]);	/* Set text  */
	sprintf(s2,"%6d",pstat[i]->MLine);
	sprintf(s3,"%11d",pstat[i]->RunCount);
	length = 1+(Dimension) (rs * (double)pstat[i]->RunCount);
	
	(void) ProfileGraphLine(innerpaneW,s1,s2,s3,length);	/* Make line */

    };

    return(innerpaneW);
}


/*
 * Widget ShowGraphWidget(Widget parent, ProfileTable *ptable)
 *
 * Display the given profile table graphically (with bars representing
 * the number of runs through a function etc).
 * The parent widget is "parent".
 * Returns the popup Widget that it creates.
 *
 */
Widget ShowGraphWidget(parent,ptable)
Widget parent;
ProfileTable *ptable;
{
    Arg         args[10];	/* Arguments for Widget creation	*/
    int         n;
    Widget 	popupW;		/* Popup Widget to appear		*/
    Widget	paneW;		/* Paned Widget for button & tree	*/
    Widget 	viewW;		/* Viewport to contain graphic		*/
    Widget 	quitW;		/* Quit button				*/
    ProfileNode **pstat;	/* Array with sorted ProfileNodes	*/

    
    if (ptable == NULL)			/* Check for bad input		*/
      return(NULL);

    
    pstat = MakeProfileStat(ptable);	/* Generate sorted statistics	*/


    /* Initialize the basic Widgets using "parent" as parent */

    n=ZERO;
    popupW = XtCreatePopupShell("ProfileGraph",topLevelShellWidgetClass,
                                parent,NULL,ZERO);

    n=0;
    XtSetArg(args[n], XtNorientation, XtorientVertical); n++;
    paneW = XtCreateManagedWidget("pgraphpane",panedWidgetClass,
                                  popupW,args,n);
    n=0;
    XtSetArg(args[n], XtNlabel, "Quit"); n++;
    XtSetArg(args[n], XtNshowGrip, False); n++;
    XtSetArg(args[n], XtNallowResize, False); n++;
    quitW = XtCreateManagedWidget("pgraphquit",commandWidgetClass,
                                  paneW,args,n);
    
    XtAddCallback(quitW, XtNcallback, ProfileGraphQuit, popupW);

    /* Create Viewport Widget and Paned Widget (for Graphics) inside it */
    /* and fill in graph lines						*/
    
    viewW = ProfileGraphInnerpane(paneW,pstat,ptable->NumEntries);
    free(pstat);

    XtPopup(popupW,XtGrabNone);

    return(popupW);	/* Return our popup window */
}


/*
 * void UpdateProfileGraph(Widget pgraph, ProfileTable *ptable)
 *
 * Updates the graphic display containing the profile information.
 * (Quick&Dirty implementation - redo whole interior from scratch!)
 *
 */
void UpdateGraphWidget(pgraph,ptable)
Widget pgraph;
ProfileTable *ptable;
{
    Arg         args[10];	/* Arguments for Widget creation	*/
    int         n;
    Widget	viewW;		/* Viewport for graph			*/
    Widget	oldviewW;	/* Old viewport for graph		*/
    Widget	paneW;		/* Parent of view			*/
    ProfileNode **pstat;	/* Array with sorted ProfileNodes	*/

    
    if ((ptable == NULL) || (pgraph==NULL))	/* Check for bad input	*/
      return;
    oldviewW = XtNameToWidget(pgraph,"*pgraphview");
    if (oldviewW==NULL)
      return;
    paneW = XtParent(oldviewW);

    
    pstat = MakeProfileStat(ptable);	/* Generate sorted statistics	*/

    viewW=ProfileGraphInnerpane(paneW,pstat,ptable->NumEntries);

    XtDestroyWidget(oldviewW);

    free(pstat);

}



/*
 * ShowTree()
 *
 * Display the global DebugNode tree "DTree" and the global Profile Table
 * "PTable". Exits if either of the displays is quit.
 *
 */
void ShowTree()
{
    XtAppContext ac;		/* Our own Application Context		*/
    Widget	topW;		/* Top window 				*/
    Widget 	treeW,profileW;	/* Popup Widgets to appear		*/
    Widget	graphW;
    XEvent	event;		/* For event loop			*/
    XDestroyWindowEvent *ep;
    int		done;		/* True if done, exit this function	*/
    int		n=0;


    /* Initialize the basic Widget  */

    topW = XtAppInitialize(&ac,"ShowTreeTop",NULL,ZERO,&n,NULL,NULL,NULL,ZERO);


    /* Generate the Tree & Profile & Graph displays and pop them up */

    treeW    = ShowTreeWidget(topW);

    profileW = ShowProfileWidget(topW);

    graphW   = ShowGraphWidget(topW);


    /* Do our own equivalent to XtAppMainLoop. End if Popup Window destroyed */

    done=FALSE;
    while (done==FALSE) { 
        XtAppNextEvent(ac,&event);
        XtDispatchEvent(&event);
        ep = (XDestroyWindowEvent*) &event;
        if ( (ep->type == DestroyNotify) &&
             ( (ep->window == XtWindow(treeW))    ||
	       (ep->window == XtWindow(profileW)) ||
	       (ep->window == XtWindow(graphW))
	     )
	   )
           done = TRUE;		/* Done if tree window destroyed */
    };

}
