/*
 * XDW			Simple X Windows Interface for Graphics
 *
 *
 * Stefan Haenssgen	09-Mar-92  started it
 *			10-Mar-92  got Images to work :-)
 *			11-Mar-92  Debugging (now also likes color displays)
 *				   Improved comments and documentation
 *			12-Mar-92  Still better docu :-)
 *				   FastDrawPixel()
 *				   Nicer colors
 *				...holidays...
 *			08-Apr-92  Experiments with color
 *				   Added colors to OpenDisplayWindow()
 *			09-Apr-92  Added RGB definition of colors
 *			10-Apr-92  Added window title
 *				   Added ClearDisplayWindow
 *			13-Apr-92  Added GetMouseData() for coordinates &
 *				    Buttons, added event handling
 *				   Made B&W default if no color data given
 *				   Added auto/manual positioning of window
 *			05-May-92  Added B&W FastDrawMonoPixel()
 *			12-May-92  Added SetDisplayWindowFont(), DrawString()
 *				   Modified parameters of OpenDW() [tlen]
 *				   Added DrawLine() and DrawRectangle()
 *			13-May-92  Debugging, mainly interfacing between
 *				    ZPixmap and XImages (yuck!)
 *				   Added colors to Drawing routines
 *			16-May-92  Added filled figures
 *				   Added circles & ellipses
 *			25-May-92  Use own colormap
 *			19-Jun-92  Use private colormap
 *			25-mar-93  New FastDrawBWPixel
 *
 *
 * Functions:
 * ---------
 *
 * OpenDisplayWindow(x,y,width,heigth,numcolors,colors,wtitle,tlen)
 *		Opens a display window of the given size at (x,y)
 *		If x or y are negative, the window is manually positioned
 *
 *		int         numcolors = number of wanted colors
 *		singleColor colors[]  = array with RGB values
 *				        (i.e. colors[0].red = red value
 *				         colors[0].green, colors[0].blue dito)
 *		char        *wtitle   = title of window
 *		unsigned long tlen    = length of title string
 *
 * DrawPixel(x,y,color)
 *		Draws a pixel of the given color internally (!)
 *
 * FastDrawPixel(x,y,color)
 *		Draws a pixel of the given color internally (!), faster
 *
 * FastDrawMonoPixel(x,y,color)
 *		Draws a pixel of the given color internally (!), faster
 *		*** for monochrome displays, use this function!!! ***
 *
 * FastDrawBWPixel(x,y,color)
 *		Draws a pixel of the given color internally (!), faster
 *		(like ..Mono.. but no color lookup, just 0 or !=0)
 *
 * SetDisplayWindowFont(fontname)
 *		Sets the font to use with text output, e.g. "fixed"
 *
 * DrawString(x,y,color,string,slen)
 *		Draws a text string of slen characters at (x,y)
 *
 * DrawLine(x1,y1,x2,y2,color)
 *		Draws a line between the given coordinates
 *
 * DrawRectangle(x,y,width,height,color,filled)
 *		Draws a rectangle (filled if True)
 *
 * DrawCircle(x,y,r,color,filled)
 *		Draws a circle (filled if True)
 *
 * DrawEllipse(x,y,r1,r2,color,filled)
 *		Draws an ellipse (filled if True)
 *
 * GetMouseData(&x,&y,&button)
 *		Get most recent mouse coordinates & button status:
 *		button = 1*left + 2*middle + 4*right  (if l/m/r button pressed)
 *		x = y = -1  if mouse not in window
 *
 * ClearDisplayWindow()
 *		Quickly clears the drawing area internally (!)
 *
 * UpdateDisplayWindow()
 *		Updates displayed window, makes recent changes visible
 *
 * CloseDisplayWindow()
 *		Destroys the window and closes the connection to X
 *
 * WARNING: All Drawing Routines have SERIOUS OVERHEAD because of
 *	    conversion of different picture formats. Beware :-)
 *
 * History:
 *	- tried multiple GCs and drawing directly... too slow
 *	- dito with off-screen drawing and copying all at once... not better
 *	- dito with pixmaps instead of multiple GCs... nothing
 *	- use XImages - faster, ok...
 *	- use XImages and direct memset - TURBO!!!!
 *	- note: just (char)s in the buffer! ulong not necessary!
 */


#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdio.h>

#include "msXDW.SUN4.h"


#define BORDER_WIDTH 2   	/* Window border width */
#define EV_MASK PointerMotionMask | ButtonPressMask | ButtonReleaseMask | KeyPressMask | LeaveWindowMask	/* Events we want to monitor	*/


Display *disp;  		/* Display			*/
int     screen; 		/* Screen of Display		*/
unsigned long blackPixel;	/* Black and white pixel values	*/
unsigned long whitePixel;
int	XD_depth;		/* Depth (in bits) of display	*/
int	bits_per_pixel;		/* Pixmap information		*/
int	scanline_pad;		/* Padding for the pixelbuffer	*/
int	maxcolors;		/* Number of colors we got	*/
int	width,height;		/* Size of window		*/
char	*pixelbuffer;		/* Off-screen drawing area	*/
int     numcolors;		/* Number of colors user wants	*/
int	mouseX,mouseY,mouseButton=0; /* Mouse position data	*/
Colormap cmap;


unsigned long foregrounds[256];	/* Pixel values for colors */
unsigned long backgrounds[256];

char setmask[] = {		/* Masks for setting and clearing pixels */
         0x80,
         0x40,
         0x20,
         0x10,
         0x08,
         0x04,
         0x02,
         0x01,
     };
char clearmask[] = {
         0x7f,
         0xaf,
         0xdf,
         0xef,
         0xf7,
         0xfa,
         0xfd,
         0xfe,
     };




GC		gc;		/* Graphic Context for drawing		*/
XGCValues	gcval;		/* For modification of GC		*/
Window		win;		/* Our Window				*/
XImage 		*img;		/* Our off-screen buffer		*/
Pixmap		pmap;		/* Pixmap for sophisticated operations	*/
XFontStruct	*font=NULL;	/* Font to use				*/




GC MakeGC(win,fore,back)	/* Create new GC		*/
Window win;
unsigned long fore;
unsigned long back;
{
  XGCValues gcv;

  gcv.foreground = fore;
  gcv.background = back;
  gcv.graphics_exposures=False;
  return(XCreateGC(disp,win,(GCForeground | GCBackground | GCGraphicsExposures),&gcv));

}

int createGC(win,gc)		/* Create GC with default pixels */
Window win;
GC *gc;
{
  XGCValues GCValues;
  int fore = blackPixel;
  int back = whitePixel;

  if (!(*gc = MakeGC(win,blackPixel, whitePixel)))
    return(0);
  else 
    return(1);
}


initcolors(numcolors,colors)		/* Allocate wanted colors	*/
int numcolors;
singleColor *colors;
{
  int i;
  int cells;
  XColor col;
  unsigned long dummy[256];


  /* Default is B&W if no colors given */

  if (numcolors == 0 || colors == NULL) {
      maxcolors=2;
      foregrounds[0] = backgrounds[1] = blackPixel;
      backgrounds[0] = foregrounds[1] = whitePixel;
      cmap = DefaultColormap(disp,screen);
/*      printf("No colors given, using B&W as default\n");  */
      return(0);
  };


  /* Else define colors as given by user */

    cmap = XCreateColormap(disp,
                           RootWindowOfScreen(DefaultScreenOfDisplay(disp)),
                           DefaultVisualOfScreen(DefaultScreenOfDisplay(disp)),
                           AllocNone);

   i=XAllocColorCells(disp,cmap,True,                   /* Contiguous    */
                           dummy,0,                     /* No planes     */
                           foregrounds,numcolors);      /* Our Colors    */

  maxcolors=2;
  for (i = 0 ; i < numcolors; i++) {
      unsigned long fore = blackPixel;
      unsigned long back = whitePixel;

      cells = DisplayCells(disp,screen);/* Number of entries in color map */

      if (cells > 2) {

/*	  if (cells > numcolors) {
/*	      col.red = col.green = col.blue = 30000; /* Gray for rest */
/*              col.flags = DoRed | DoGreen | DoBlue;
/*	  } else {*/
              col.red   = colors[i].red;	/* Get new color as wanted */
              col.green = colors[i].green;
              col.blue  = colors[i].blue;
              col.flags = DoRed | DoGreen | DoBlue;
              col.pixel = foregrounds[i];
/*	  }*/
/*	  if (XAllocColor(disp,cmap,&col) == 0)  {*/
	  XStoreColor(disp,cmap,&col);
	  fore = col.pixel;
/*	  printf("alloced color (%5d,%5d,%5d) as no.%5d\n",
	  colors[i].red,colors[i].green,colors[i].blue,fore);*/
          maxcolors++;

      } else {
          /* Get black/white on monochrome systems depending on brightness */

/*          printf("mapping (%5d,%5d,%5d) to ",colors[i].red,colors[i].green,
                 colors[i].blue);*/
          if (colors[i].red + colors[i].blue + colors[i].green > 65535*3/2) { 
              back = whitePixel;
              fore = blackPixel;
/*              printf("black\n");*/
          } else {
              back = blackPixel;
              fore = whitePixel;
/*              printf("white\n");*/
          }
          foregrounds[i] = fore;	/* Fuer color hat er schon pixels! */
          backgrounds[i] = back;
      }
  }
/*  printf("got %d colors\n",maxcolors);*/
}


void SetDisplayWindowFont(FontName,flen)		/* Select new font to use	*/
char *FontName;
unsigned long flen;
{
    XFontStruct *fontStruct;

    if (font != NULL) {			/* Free old font...		*/
        XFreeFont(disp,font);		/* ...if there was one 		*/
    };
    if (font = XLoadQueryFont(disp,FontName)) {	/* Load new font	*/
        gcval.font = font->fid;
        XChangeGC(disp,gc,GCFont,&gcval);
/*        printf("Set Font to %s\n",FontName);*/
    } else {					/* Complain if error	*/
        fprintf(stderr,"Couldn`t open font %s!\n",FontName);
    };
}


int initX(displayname,numcolors,colors)		/* Init X Server connection */
char *displayname;
int numcolors;
singleColor *colors;
{ 
    if (!(disp = XOpenDisplay(displayname))) {
        fprintf(stderr,"Cannot establish a connection to the X Server %s\n",
              XDisplayName(displayname));
        exit(1);
    }

    /* Get default screen and pixel values */

    screen = DefaultScreen(disp);	
    XD_depth=DefaultDepth(disp,screen);

    blackPixel = WhitePixel(disp,screen);
    whitePixel = BlackPixel(disp,screen);	

    initcolors(numcolors,colors);
    return(1);
}


Window OpenWindow(x,y,w,h,gc,wtitle)		/* Open the window itself */
int x, y, w, h; 
GC *gc;
char *wtitle;
{
  XSetWindowAttributes	attribs;
  XSizeHints		hints;
  unsigned long		mask;
  Window		win;
  XWMHints		WMhints;	/* Window Manager Hints	*/
  int			max_width,max_height;
  int			i;
  char			auto_position;
  XSetWindowAttributes  attr;		/* What we want to monitor */


  /* Automatically position window using given coordinates (x,y) */
  /* if x and y are positive, else let user position manually	 */

  auto_position = (x>=0 && y>=0);
 
  width  = w;				/* Set global dimensions */
  height = h;
  max_width=DisplayWidth(disp,screen);
  max_height=DisplayHeight(disp,screen);
/*  printf("Screen is %dx%d pixels\n",max_width,max_height);*/

  attribs.border_pixel = blackPixel;
  attribs.background_pixel = whitePixel;
  attribs.override_redirect= auto_position;
  attribs.colormap = cmap;
  
  mask = CWBackPixel | CWBorderPixel | CWOverrideRedirect | CWColormap;

  win = XCreateWindow(disp,		/* Create Window for Display */
	 RootWindow(disp,screen),
         x,y,
         width, height,
         BORDER_WIDTH,
         CopyFromParent,
         InputOutput,
	 CopyFromParent,
	 mask,
 	&attribs);

  WMhints.initial_state	= NormalState;
  WMhints.flags		= StateHint;

  XSetWMHints(disp,win,&WMhints);

  hints.flags	= PPosition | PSize | PMinSize | PMaxSize;
  hints.x	= x;
  hints.y	= y;
  hints.max_width=width;
  hints.min_width=width;		/* Set Window Manager Parameters */
  hints.max_height=height;
  hints.min_height=height;
  hints.width	= width;
  hints.height	= height;

  XSetNormalHints(disp,win,&hints);

  if (!(createGC(win,gc))) {
    XDestroyWindow(disp,win);
    return( (Window) 0);
  }

  XMapWindow(disp,win);			/* Show Window on Screen */

  XFlush(disp);

  XStoreName(disp,win,wtitle);		/* Set window title	 */
  XMapRaised(disp,win);

  attr.event_mask = EV_MASK;		/* Define which events we want to get*/
  XChangeWindowAttributes(disp,win,CWEventMask,&attr);

  SetDisplayWindowFont("fixed",5);	/* Set Font to default	 */
  XInstallColormap(disp,cmap);

  return(win);
}


int dummypredicate(disp, event, arg)	/* For event handling	*/
Display *disp;
XEvent *event;
char *arg;
{
  return 1;
}



/* ------------------------------------- Public interface routines */



int OpenDisplayWindow(x,y,width,height,	/* Open the display window */
                      numcolors,colors,wtitle,tlen)
int x,y,width,height;
int numcolors;				/* Number of colors wanted */
singleColor *colors;			/* RGB-Values of colors	   */
char *wtitle;				/* Title of window	   */
unsigned long tlen;			/* Length of title (!!! IGNORED !!!) */
{
    int i;
    int scanline_pad = 32;		/* Bits to pad scanline */


    initX(NULL,numcolors,colors);
    win=OpenWindow(x,y,width,height,&gc,wtitle);

/*    def_vis = DefaultVisual(disp,screen);*/
/*    printf("default visual = %d\n",def_vis);*/


    /* Allocate Pixel Buffer and Create Image */

    pixelbuffer = (char*) malloc(width*height+1);
    img = XCreateImage(disp,DefaultVisual(disp,screen),
                       DefaultDepth(disp,screen),ZPixmap,0,
                       pixelbuffer, (unsigned int)width,
                       (unsigned int) height, scanline_pad, 0);

    pmap = XCreatePixmap(disp,win,width,height,DefaultDepth(disp,screen));

    mouseX = mouseY = -1;	/* Mark mouse "not in window"	*/
    mouseButton = 0;		/* and "no button pressed"	*/
}


void DrawPixel(x,y,color)	/* Draw a point in the given color	*/
int x,y,color;
{

    XPutPixel(img,x,y,foregrounds[color]);

}


void FastDrawPixel(x,y,color)	/* Draw a point via direct access	*/
int x,y,color;
{

    *(pixelbuffer +x+y*width) = (char)foregrounds[color];

}


void FastDrawBWPixel(x,y,color) /* Draw a point via direct access	*/
int x,y,color;
{
    register int  offset    = x+y*width;
    register char byte      = *(pixelbuffer + offset/8);

    if (color) {			/* Set the corresponding bit	*/ 
        *(pixelbuffer + offset/8) = byte | setmask[offset % 8];
    } else {				/* Clear it			*/
        *(pixelbuffer + offset/8) = byte & clearmask[offset % 8]; 
    };
}

void FastDrawMonoPixel(x,y,color) /* Draw a point via direct access	*/
int x,y,color;
{
    register int  offset    = x+y*width;
    register char byte      = *(pixelbuffer + offset/8);

    if (foregrounds[color]) {		/* Set the corresponding bit	*/ 
        *(pixelbuffer + offset/8) = byte | setmask[offset % 8];
    } else {				/* Clear it			*/
        *(pixelbuffer + offset/8) = byte & clearmask[offset % 8]; 
    };

    /* !!!!!!! WARNING: this approach assumes that the pixel buffer	*/
    /* !!!!!!! is one complete area in memory that is mapped 1:1 onto	*/
    /* !!!!!!! the display screen. Furthermore, it assumes that the	*/
    /* !!!!!!! display is organized as bytes with the least significant	*/
    /* !!!!!!! bit in a position determined by the setmask[] and	*/
    /* !!!!!!! clearmask[] arrays.					*/
    /*         The possibility of scanline padding (e.g. rounding each  */
    /*         line of pixels to 32 bit length) etc is ignored.		*/
    /*         Use multiples of 32 for sizes to avoid conflicts.	*/

}


void GetMouseData(x,y,button)	/* Get mouse position & button status	*/
int *x;
int *y;
int *button;
{
    XEvent event;
    static int buttonvalues[] = {1,2,4}; /* 2^i for i=0,1,2 */


    /* Go through all pending events */

    while (XCheckIfEvent(disp, &event, dummypredicate, ""))
      switch (event.type) {
        case MotionNotify :	/* New mouse position	*/
          mouseX = event.xmotion.x;
          mouseY = event.xmotion.y;
          break;
        case ButtonPress :	/* Button pressed	*/
          mouseButton=mouseButton + buttonvalues[event.xbutton.button-1];
          break;
        case ButtonRelease :	/* Button released	*/
          mouseButton=mouseButton - buttonvalues[event.xbutton.button-1];
          if (mouseButton < 0) mouseButton=0;
          break;
        case LeaveNotify :	/* Mouse no longer in / newly in window	*/
          mouseX = mouseY = -1;
          mouseButton=0;
          break;
      };

    /* Return latest positions & button status */

    *x = mouseX;
    *y = mouseY;
    *button = mouseButton;

}


void WaitForMouseClick()	/* Wait until any mouse button was pressed */
{
    int x, y, button;
    
    do { GetMouseData(&x, &y, &button); } while (button==0);
}


void DrawString(x,y,color,string,slen)/* Draw a text string	*/
int x,y;			/* Coordinates			*/
int color;			/* Color to use			*/
char *string;			/* String to draw		*/
unsigned long slen;		/* String length (!!! IGNORED !!!)*/
{

    /* Set the right color */

    gcval.foreground = foregrounds[color];
    XChangeGC(disp,gc,GCForeground,&gcval);

    /* Put the Image into a Pixmap, draw text and get the result back */

    XPutImage(disp,pmap,gc,img,0,0,0,0,width,height);
    XDrawString(disp,pmap,gc,x,y,string,strlen(string));
    XGetSubImage(disp,pmap,0,0,width,height,AllPlanes,ZPixmap,img,0,0);
}


void DrawLine(x,y,x2,y2,color)	/* Draw a line			*/
int x,y,x2,y2;			/* Coordinates			*/
int color;			/* Color to use			*/
{

    /* Set the right color */

    gcval.foreground = foregrounds[color];
    XChangeGC(disp,gc,GCForeground,&gcval);

    /* Put the Image into a Pixmap, draw line and get the result back */

    XPutImage(disp,pmap,gc,img,0,0,0,0,width,height);
    XDrawLine(disp,pmap,gc,x,y,x2,y2);
    XGetSubImage(disp,pmap,0,0,width,height,AllPlanes,ZPixmap,img,0,0);
}


void DrawRectangle(x,y,dx,dy,color,filled) /* Draw a rectangle		*/
int x,y;			/* Coordinates			*/
int dx,dy;			/* Size				*/
int color;			/* Color to use			*/
Bool filled;			/* True if filled wanted	*/
{

    /* Set the right color */

    gcval.foreground = foregrounds[color];
    XChangeGC(disp,gc,GCForeground,&gcval);

    /* Put the Image into a Pixmap, draw rectangle and get the result back */

    XPutImage(disp,pmap,gc,img,0,0,0,0,width,height);
    filled ? 
        XFillRectangle(disp,pmap,gc,x,y,dx,dy) :
        XDrawRectangle(disp,pmap,gc,x,y,dx,dy);
    XGetSubImage(disp,pmap,0,0,width,height,AllPlanes,ZPixmap,img,0,0);
}


void DrawCircle(x,y,r,color,filled) /* Draw a rectangle		*/
int x,y;			/* Coordinates			*/
int r;				/* Radius			*/
int color;			/* Color to use			*/
Bool filled;			/* True if filled wanted	*/
{

    /* Set the right color */

    gcval.foreground = foregrounds[color];
    XChangeGC(disp,gc,GCForeground,&gcval);

    /* Put the Image into a Pixmap, draw rectangle and get the result back */

    XPutImage(disp,pmap,gc,img,0,0,0,0,width,height);
    filled ? 
        XFillArc(disp,pmap,gc,x-r,y-r,2*r,2*r,0,360*64) :
        XDrawArc(disp,pmap,gc,x-r,y-r,2*r,2*r,0,360*64);
    XGetSubImage(disp,pmap,0,0,width,height,AllPlanes,ZPixmap,img,0,0);
}


void DrawEllipse(x,y,r1,r2,color,filled) /* Draw a rectangle		*/
int x,y;			/* Coordinates			*/
int r1,r2;			/* Radii (x & y)		*/
int color;			/* Color to use			*/
Bool filled;			/* True if filled wanted	*/
{

    /* Set the right color */

    gcval.foreground = foregrounds[color];
    XChangeGC(disp,gc,GCForeground,&gcval);

    /* Put the Image into a Pixmap, draw rectangle and get the result back */

    XPutImage(disp,pmap,gc,img,0,0,0,0,width,height);
    filled ? 
        XFillArc(disp,pmap,gc,x-r1,y-r2,2*r1,2*r2,0,360*64) :
        XDrawArc(disp,pmap,gc,x-r1,y-r2,2*r1,2*r2,0,360*64);
    XGetSubImage(disp,pmap,0,0,width,height,AllPlanes,ZPixmap,img,0,0);
}


void UpdateDisplayWindow()	/* Update real window on screen	*/
{				/* Also process pending events	*/
   int foo;

    XPutImage(disp,win,gc,img,0,0,0,0,
              (unsigned int)width,(unsigned int)height);
    GetMouseData(&foo,&foo,&foo);
}


void CloseDisplayWindow()	/* Remove window, free memory	*/
{
    int i;

    XFreeGC(disp,gc);
    free(pixelbuffer);
    XUninstallColormap(disp,cmap);
    XDestroyImage(img);
    XCloseDisplay(disp);

}


void ClearDisplayWindow()	/* Clear the display internally	*/
{
  memset(pixelbuffer,(int) 0,(width*height+1));
}


void ClearScreen()
{
    ClearDisplayWindow();
}
