
/********************************************************************
 ********************************************************************
 ******                                                        ******
 ******  Scan Library for Modula-2*                            ******
 ******                                                        ******
 ******  allows for efficient prefix scan operations on        ******
 ******  arbitrarily large and distributed arrays              ******
 ******                                                        ******
 ******  C implementation by Ernst A. Heinz & Juergen Fritsch  ******
 ******                                                        ******
 ******  Last change: March 25, 1994                           ******
 ******                                                        ******
 ********************************************************************
 ********************************************************************/

#include "msScan.SUN4.h"


/********************************************************************
 ********************************************************************
 *******             NORMAL PREFIX SCAN ROUTINES              *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template of normal prefix scan routines
 */

#define ScanFunc(FuncName,elemType,GlobalScan,accu,initVal,SCAN_OP)	\
    void								\
    FuncName(x, nx, lx, y, ny, ly)					\
    register elemType *x;						\
    register LONGCARD nx;						\
    LONGCARD lx;							\
    register elemType *y;						\
    LONGCARD ny, ly;							\
    {									\
      register elemType accu;						\
									\
      if (nx!=ny) return;						\
									\
      GlobalScan(elemType,accu,initVal,SCAN_OP);			\
    }

/*
 *  generic template of normal prefix scan code with accumulation operator INC_OP
 */

#define GenericScan1(elemType,accu,initVal,INC_OP)	\
    accu = initVal;					\
    for (; nx>0; nx--, x++, y++) {			\
      accu INC_OP *x;					\
      *y = accu;					\
    };

/*
 *  generic template of normal min/max prefix scan code with compare operator CMP_OP
 */

#define GenericScan2(elemType,accu,initVal,CMP_OP)	\
    accu = initVal;					\
    {							\
      register elemType tmp;				\
							\
      for (; nx>0; nx--, x++, y++) {			\
        tmp = *x;					\
        if (tmp CMP_OP accu)				\
	  accu = tmp;					\
        *y = accu;					\
      }							\
    };


/******************************************************************
 *****  normal routines with prefix scan operator "addition"  *****
 ******************************************************************/

#define ScanAddFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan1,sum,initVal,+=)

ScanAddFunc( SScanAddCh, CHAR,      '\0' )
ScanAddFunc( SScanAddSC, SHORTCARD, 0    )
ScanAddFunc( SScanAddC,  CARDINAL,  0    )
ScanAddFunc( SScanAddSI, SHORTINT,  0    )
ScanAddFunc( SScanAddI,  INTEGER,   0    )
ScanAddFunc( SScanAddR,  REAL,      0.0  )
ScanAddFunc( SScanAddLR, LONGREAL,  0.0  )


/*********************************************************************
 *****  normal routines with prefix scan operator "logical AND"  *****
 *********************************************************************/

#define ScanAndFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan1,and,initVal,&=)

ScanAndFunc( SScanAndCh, CHAR,      '\377'     )
ScanAndFunc( SScanAndSC, SHORTCARD, 0xFFFF     )
ScanAndFunc( SScanAndC,  CARDINAL,  0xFFFFFFFF )
ScanAndFunc( SScanAndSI, SHORTINT,  0xFFFF     )
ScanAndFunc( SScanAndI,  INTEGER,   0xFFFFFFFF )


/*****************************************************************
 *****  normal routines with prefix scan operator "maximum"  *****
 *****************************************************************/

#define ScanMaxFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan2,max,initVal,>)

ScanMaxFunc( SScanMaxCh, CHAR,      MIN_CHAR      )
ScanMaxFunc( SScanMaxSC, SHORTCARD, MIN_SHORTCARD )
ScanMaxFunc( SScanMaxC,  CARDINAL,  MIN_LONGCARD  )
ScanMaxFunc( SScanMaxSI, SHORTINT,  MIN_SHORTINT  )
ScanMaxFunc( SScanMaxI,  INTEGER,   MIN_LONGINT   )
ScanMaxFunc( SScanMaxR,  REAL,      MIN_REAL      )
ScanMaxFunc( SScanMaxLR, LONGREAL,  MIN_LONGREAL  )


/*****************************************************************
 *****  normal routines with prefix scan operator "minimum"  *****
 *****************************************************************/

#define ScanMinFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan2,min,initVal,<)

ScanMinFunc( SScanMinCh, CHAR,      MAX_CHAR      )
ScanMinFunc( SScanMinSC, SHORTCARD, MAX_SHORTCARD )
ScanMinFunc( SScanMinC,  CARDINAL,  MAX_LONGCARD  )
ScanMinFunc( SScanMinSI, SHORTINT,  MAX_SHORTINT  )
ScanMinFunc( SScanMinI,  INTEGER,   MAX_LONGINT   )
ScanMinFunc( SScanMinR,  REAL,      MAX_REAL      )
ScanMinFunc( SScanMinLR, LONGREAL,  MAX_LONGREAL  )


/************************************************************************
 *****  normal routines with prefix scan operator "multiplication"  *****
 ************************************************************************/

#define ScanMulFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan1,prod,initVal,*=)

ScanMulFunc( SScanMulCh, CHAR,      '\001' )
ScanMulFunc( SScanMulSC, SHORTCARD, 1      )
ScanMulFunc( SScanMulC,  CARDINAL,  1      )
ScanMulFunc( SScanMulSI, SHORTINT,  1      )
ScanMulFunc( SScanMulI,  INTEGER,   1      )
ScanMulFunc( SScanMulR,  REAL,      1.0    )
ScanMulFunc( SScanMulLR, LONGREAL,  1.0    )


/********************************************************************
 *****  normal routines with prefix scan operator "logical OR"  *****
 ********************************************************************/

#define ScanOrFunc(FuncName,elemType,initVal) \
    ScanFunc(FuncName,elemType,GenericScan1,or,initVal,|=)

ScanOrFunc( SScanOrCh, CHAR,      '\0' )
ScanOrFunc( SScanOrSC, SHORTCARD, 0    )
ScanOrFunc( SScanOrC,  CARDINAL,  0    )
ScanOrFunc( SScanOrSI, SHORTINT,  0    )
ScanOrFunc( SScanOrI,  INTEGER,   0    )


/********************************************************************
 ********************************************************************
 *******             MASKED PREFIX SCAN ROUTINES              *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template for masked prefix scan routines
 */

#define MaskScanFunc(FuncName,elemType,GlobalMaskScan,accu,initVal,SCAN_OP)	\
    void									\
    FuncName(mask, nm, lm, x, nx, lx, y, ny, ly)				\
    register BOOLEAN *mask;							\
    LONGCARD nm, lm;								\
    register elemType *x;							\
    register LONGCARD nx;							\
    LONGCARD lx;								\
    register elemType *y;							\
    LONGCARD ny, ly;								\
    {										\
      register elemType accu;							\
										\
      if (nx!=nm || nx!=ny) return;                     			\
										\
      GlobalMaskScan(elemType,accu,initVal,SCAN_OP);				\
    }

/*
 *  generic template of masked prefix scan code with accumulation operator INC_OP
 */

#define GenericMaskScan1(elemType,accu,initVal,INC_OP)	\
    accu = initVal;					\
    for (; nx>0; nx--, mask++, x++, y++)		\
      if (*mask) {					\
        accu INC_OP *x;					\
        *y = accu;					\
      };

/*
 *  generic template of masked min/max prefix scan code with compare operator CMP_OP
 */

#define GenericMaskScan2(elemType,accu,initVal,CMP_OP)	\
    accu = initVal;					\
    {							\
      register elemType tmp;				\
							\
      for (; nx>0; nx--, mask++, x++, y++) {		\
        if (*mask) {					\
          tmp = *x;					\
          if (tmp CMP_OP accu)				\
	    accu = tmp;					\
          *y = accu;					\
	}						\
      }							\
    };


/******************************************************************
 *****  masked routines with prefix scan operator "addition"  *****
 ******************************************************************/

#define MaskScanAddFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan1,sum,initVal,+=)

MaskScanAddFunc( SMaskScanAddCh, CHAR,      '\0' )
MaskScanAddFunc( SMaskScanAddSC, SHORTCARD, 0    )
MaskScanAddFunc( SMaskScanAddC,  CARDINAL,  0    )
MaskScanAddFunc( SMaskScanAddSI, SHORTINT,  0    )
MaskScanAddFunc( SMaskScanAddI,  INTEGER,   0    )
MaskScanAddFunc( SMaskScanAddR,  REAL,      0.0  )
MaskScanAddFunc( SMaskScanAddLR, LONGREAL,  0.0  )


/*********************************************************************
 *****  masked routines with prefix scan operator "logical AND"  *****
 *********************************************************************/

#define MaskScanAndFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan1,and,initVal,&=)

MaskScanAndFunc( SMaskScanAndCh, CHAR,      '\377'     )
MaskScanAndFunc( SMaskScanAndSC, SHORTCARD, 0xFFFF     )
MaskScanAndFunc( SMaskScanAndC,  CARDINAL,  0xFFFFFFFF )
MaskScanAndFunc( SMaskScanAndSI, SHORTINT,  0xFFFF     )
MaskScanAndFunc( SMaskScanAndI,  INTEGER,   0xFFFFFFFF )


/*****************************************************************
 *****  masked routines with prefix scan operator "maximum"  *****
 *****************************************************************/

#define MaskScanMaxFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan2,max,initVal,>)

MaskScanMaxFunc( SMaskScanMaxCh, CHAR,      MIN_CHAR      )
MaskScanMaxFunc( SMaskScanMaxSC, SHORTCARD, MIN_SHORTCARD )
MaskScanMaxFunc( SMaskScanMaxC,  CARDINAL,  MIN_LONGCARD  )
MaskScanMaxFunc( SMaskScanMaxSI, SHORTINT,  MIN_SHORTINT  )
MaskScanMaxFunc( SMaskScanMaxI,  INTEGER,   MIN_LONGINT   )
MaskScanMaxFunc( SMaskScanMaxR,  REAL,      MIN_REAL      )
MaskScanMaxFunc( SMaskScanMaxLR, LONGREAL,  MIN_LONGREAL  )


/*****************************************************************
 *****  masked routines with prefix scan operator "minimum"  *****
 *****************************************************************/

#define MaskScanMinFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan2,min,initVal,<)

MaskScanMinFunc( SMaskScanMinCh, CHAR,      MAX_CHAR      )
MaskScanMinFunc( SMaskScanMinSC, SHORTCARD, MAX_SHORTCARD )
MaskScanMinFunc( SMaskScanMinC,  CARDINAL,  MAX_LONGCARD  )
MaskScanMinFunc( SMaskScanMinSI, SHORTINT,  MAX_SHORTINT  )
MaskScanMinFunc( SMaskScanMinI,  INTEGER,   MAX_LONGINT   )
MaskScanMinFunc( SMaskScanMinR,  REAL,      MAX_REAL      )
MaskScanMinFunc( SMaskScanMinLR, LONGREAL,  MAX_LONGREAL  )


/************************************************************************
 *****  masked routines with prefix scan operator "multiplication"  *****
 ************************************************************************/

#define MaskScanMulFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan1,prod,initVal,*=)

MaskScanMulFunc( SMaskScanMulCh, CHAR,      '\001' )
MaskScanMulFunc( SMaskScanMulSC, SHORTCARD, 1      )
MaskScanMulFunc( SMaskScanMulC,  CARDINAL,  1      )
MaskScanMulFunc( SMaskScanMulSI, SHORTINT,  1      )
MaskScanMulFunc( SMaskScanMulI,  INTEGER,   1      )
MaskScanMulFunc( SMaskScanMulR,  REAL,      1.0    )
MaskScanMulFunc( SMaskScanMulLR, LONGREAL,  1.0    )


/********************************************************************
 *****  masked routines with prefix scan operator "logical OR"  *****
 ********************************************************************/

#define MaskScanOrFunc(FuncName,elemType,initVal) \
    MaskScanFunc(FuncName,elemType,GenericMaskScan1,or,initVal,|=)

MaskScanOrFunc( SMaskScanOrCh, CHAR,      '\0' )
MaskScanOrFunc( SMaskScanOrSC, SHORTCARD, 0    )
MaskScanOrFunc( SMaskScanOrC,  CARDINAL,  0    )
MaskScanOrFunc( SMaskScanOrSI, SHORTINT,  0    )
MaskScanOrFunc( SMaskScanOrI,  INTEGER,   0    )


/********************************************************************
 ********************************************************************
 *******            SEGMENTED PREFIX SCAN ROUTINES            *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template of segmented prefix scan routines
 */

#define SegScanFunc(FuncName,elemType,GlobalSegScan,accu,initVal,SCAN_OP)	\
    void									\
    FuncName(segs, ns, ls, x, nx, lx, y, ny, ly)				\
    BOOLEAN  *segs;								\
    LONGCARD ns, ls;								\
    register elemType *x;							\
    register LONGCARD nx;							\
    LONGCARD lx;								\
    elemType *y;								\
    LONGCARD ny, ly;								\
    {										\
      register elemType accu;							\
      register BOOLEAN *segp;							\
      register elemType *yp;							\
      register BOOLEAN seg_flag;						\
										\
      if (nx!=ns || nx!=ny) return;						\
										\
      GlobalSegScan(elemType,accu,initVal,SCAN_OP);				\
    }

/*
 *  generic template of segmented prefix scan code with accumulation operator INC_OP
 */

#define GenericSegScan1(elemType,accu,initVal,INC_OP)	\
    accu = initVal;					\
    seg_flag = FALSE;					\
    segp = segs;					\
    yp = y;						\
    for (; nx>0; nx--, segp++, x++, yp++) {		\
      accu INC_OP *x;					\
      *yp = accu;					\
      if (*segp) {					\
        accu = initVal;					\
        seg_flag = TRUE;				\
      }							\
    };							\
							\
    if (!seg_flag) return;				\
							\
    segp = segs;					\
    yp = y;						\
    seg_flag = (accu!=initVal);				\
    for (; seg_flag; segp++, yp++) {			\
      *yp INC_OP accu;					\
      seg_flag = !(*segp);				\
    };

/*
 *  generic template of segmented min/max prefix scan code with compare operator CMP_OP
 */

#define GenericSegScan2(elemType,accu,initVal,CMP_OP)	\
    accu = initVal;					\
    seg_flag = FALSE;					\
    segp = segs;					\
    yp = y;						\
    {							\
      register elemType tmp;				\
							\
      for (; nx>0; nx--, segp++, x++, yp++) {		\
        tmp = *x;					\
        if (tmp CMP_OP accu)				\
          accu = tmp;					\
        *yp = accu;					\
        if (*segp) {					\
          accu = initVal;				\
          seg_flag = TRUE;				\
        }						\
      }							\
    };							\
							\
    if (!seg_flag) return;				\
							\
    segp = segs;					\
    yp = y;						\
    for (; seg_flag; segp++, yp++) {			\
      seg_flag = (accu CMP_OP *yp);			\
      if (seg_flag)					\
        *yp = accu;					\
      seg_flag &= !(*segp);				\
    };


/*********************************************************************
 *****  segmented routines with prefix scan operator "addition"  *****
 *********************************************************************/

#define SegScanAddFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan1,sum,initVal,+=)

SegScanAddFunc( SSegScanAddCh, CHAR,      '\0' )
SegScanAddFunc( SSegScanAddSC, SHORTCARD, 0    )
SegScanAddFunc( SSegScanAddC,  CARDINAL,  0    )
SegScanAddFunc( SSegScanAddSI, SHORTINT,  0    )
SegScanAddFunc( SSegScanAddI,  INTEGER,   0    )
SegScanAddFunc( SSegScanAddR,  REAL,      0.0  )
SegScanAddFunc( SSegScanAddLR, LONGREAL,  0.0  )


/************************************************************************
 *****  segmented routines with prefix scan operator "logical AND"  *****
 ************************************************************************/

#define SegScanAndFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan1,and,initVal,&=)

SegScanAndFunc( SSegScanAndCh, CHAR,      '\377'     )
SegScanAndFunc( SSegScanAndSC, SHORTCARD, 0xFFFF     )
SegScanAndFunc( SSegScanAndC,  CARDINAL,  0xFFFFFFFF )
SegScanAndFunc( SSegScanAndSI, SHORTINT,  0xFFFF     )
SegScanAndFunc( SSegScanAndI,  INTEGER,   0xFFFFFFFF )


/********************************************************************
 *****  segmented routines with prefix scan operator "maximum"  *****
 ********************************************************************/

#define SegScanMaxFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan2,max,initVal,>)

SegScanMaxFunc( SSegScanMaxCh, CHAR,      MIN_CHAR      )
SegScanMaxFunc( SSegScanMaxSC, SHORTCARD, MIN_SHORTCARD )
SegScanMaxFunc( SSegScanMaxC,  CARDINAL,  MIN_LONGCARD  )
SegScanMaxFunc( SSegScanMaxSI, SHORTINT,  MIN_SHORTINT  )
SegScanMaxFunc( SSegScanMaxI,  INTEGER,   MIN_LONGINT   )
SegScanMaxFunc( SSegScanMaxR,  REAL,      MIN_REAL      )
SegScanMaxFunc( SSegScanMaxLR, LONGREAL,  MIN_LONGREAL  )


/********************************************************************
 *****  segmented routines with prefix scan operator "minimum"  *****
 ********************************************************************/

#define SegScanMinFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan2,min,initVal,<)

SegScanMinFunc( SSegScanMinCh, CHAR,      MAX_CHAR      )
SegScanMinFunc( SSegScanMinSC, SHORTCARD, MAX_SHORTCARD )
SegScanMinFunc( SSegScanMinC,  CARDINAL,  MAX_LONGCARD  )
SegScanMinFunc( SSegScanMinSI, SHORTINT,  MAX_SHORTINT  )
SegScanMinFunc( SSegScanMinI,  INTEGER,   MAX_LONGINT   )
SegScanMinFunc( SSegScanMinR,  REAL,      MAX_REAL      )
SegScanMinFunc( SSegScanMinLR, LONGREAL,  MAX_LONGREAL  )


/***************************************************************************
 *****  segmented routines with prefix scan operator "multiplication"  *****
 ***************************************************************************/

#define SegScanMulFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan1,prod,initVal,*=)

SegScanMulFunc( SSegScanMulCh, CHAR,      '\001' )
SegScanMulFunc( SSegScanMulSC, SHORTCARD, 1      )
SegScanMulFunc( SSegScanMulC,  CARDINAL,  1      )
SegScanMulFunc( SSegScanMulSI, SHORTINT,  1      )
SegScanMulFunc( SSegScanMulI,  INTEGER,   1      )
SegScanMulFunc( SSegScanMulR,  REAL,      1.0    )
SegScanMulFunc( SSegScanMulLR, LONGREAL,  1.0    )


/***********************************************************************
 *****  segmented routines with prefix scan operator "logical OR"  *****
 ***********************************************************************/

#define SegScanOrFunc(FuncName,elemType,initVal) \
    SegScanFunc(FuncName,elemType,GenericSegScan1,or,initVal,|=)

SegScanOrFunc( SSegScanOrCh, CHAR,      '\0' )
SegScanOrFunc( SSegScanOrSC, SHORTCARD, 0    )
SegScanOrFunc( SSegScanOrC,  CARDINAL,  0    )
SegScanOrFunc( SSegScanOrSI, SHORTINT,  0    )
SegScanOrFunc( SSegScanOrI,  INTEGER,   0    )


/********************************************************************
 ********************************************************************
 *******            UNIVERSAL PREFIX SCAN ROUTINES            *******
 ********************************************************************
 ********************************************************************/

/*
 *  generic template of universal prefix scan routines
 */

#define UnivScanFunc(FuncName,elemType,GlobalUnivScan,accu,initVal,SCAN_OP)	\
    void									\
    FuncName(mask, nm, lm, segs, ns, ls, x, nx, lx, y, ny, ly)			\
    BOOLEAN  *mask;								\
    LONGCARD nm, lm;								\
    BOOLEAN  *segs;								\
    LONGCARD ns, ls;								\
    register elemType *x;							\
    register LONGCARD nx;							\
    LONGCARD lx;								\
    elemType *y;								\
    LONGCARD ny, ly;								\
    {										\
      register elemType accu;							\
      register BOOLEAN *maskp, *segp;						\
      register elemType *yp;							\
      register BOOLEAN seg_flag;						\
										\
      if (nx!=nm || nx!=ns || nx!=ny) return;					\
										\
      GlobalUnivScan(elemType,accu,initVal,SCAN_OP);				\
    }

/*
 *  generic template of universal prefix scan code with accumulation operator INC_OP
 */

#define GenericUnivScan1(elemType,accu,initVal,INC_OP)	\
    accu = initVal;					\
    seg_flag = FALSE;					\
    maskp = mask;					\
    segp = segs;					\
    yp = y;						\
    for (; nx>0; nx--, maskp++, segp++, x++, yp++)	\
      if (*maskp) {					\
        accu INC_OP *x;					\
        *yp = accu;					\
        if (*segp) {					\
          accu = initVal;				\
          seg_flag = TRUE;				\
        }						\
      };						\
							\
    if (!seg_flag) return;				\
							\
    maskp = mask;					\
    segp = segs;					\
    yp = y;						\
    seg_flag = (accu!=initVal);				\
    for (; seg_flag; maskp++, segp++, yp++)		\
      if (*maskp) {					\
        *yp INC_OP accu;				\
        seg_flag = !(*segp);				\
      };

/*
 *  generic template of universal min/max prefix scan code with compare operator CMP_OP
 */

#define GenericUnivScan2(elemType,accu,initVal,CMP_OP)	\
    accu = initVal;					\
    seg_flag = FALSE;					\
    maskp = mask;					\
    segp = segs;					\
    yp = y;						\
    {							\
      register elemType tmp;				\
							\
      for (; nx>0; nx--, maskp++, segp++, x++, yp++)	\
        if (*maskp) {					\
          tmp = *x;					\
          if (tmp CMP_OP accu)				\
            accu = tmp;					\
          *yp = accu;					\
          if (*segp) {					\
            accu = initVal;				\
            seg_flag = TRUE;				\
          }						\
        }						\
    };							\
							\
    if (!seg_flag) return;				\
							\
    maskp = mask;					\
    segp = segs;					\
    yp = y;						\
    seg_flag = (accu!=initVal);				\
    for (; seg_flag; maskp++, segp++, yp++)		\
      if (*maskp) {					\
        seg_flag = (accu CMP_OP *yp);			\
        if (seg_flag)					\
          *yp = accu;					\
        seg_flag &= !(*segp);				\
      };


/*********************************************************************
 *****  universal routines with prefix scan operator "addition"  *****
 *********************************************************************/

#define UnivScanAddFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan1,sum,initVal,+=)

UnivScanAddFunc( SUnivScanAddCh, CHAR,      '\0' )
UnivScanAddFunc( SUnivScanAddSC, SHORTCARD, 0    )
UnivScanAddFunc( SUnivScanAddC,  CARDINAL,  0    )
UnivScanAddFunc( SUnivScanAddSI, SHORTINT,  0    )
UnivScanAddFunc( SUnivScanAddI,  INTEGER,   0    )
UnivScanAddFunc( SUnivScanAddR,  REAL,      0.0  )
UnivScanAddFunc( SUnivScanAddLR, LONGREAL,  0.0  )


/************************************************************************
 *****  universal routines with prefix scan operator "logical AND"  *****
 ************************************************************************/

#define UnivScanAndFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan1,and,initVal,&=)

UnivScanAndFunc( SUnivScanAndCh, CHAR,      '\377'     )
UnivScanAndFunc( SUnivScanAndSC, SHORTCARD, 0xFFFF     )
UnivScanAndFunc( SUnivScanAndC,  CARDINAL,  0xFFFFFFFF )
UnivScanAndFunc( SUnivScanAndSI, SHORTINT,  0xFFFF     )
UnivScanAndFunc( SUnivScanAndI,  INTEGER,   0xFFFFFFFF )


/********************************************************************
 *****  universal routines with prefix scan operator "maximum"  *****
 ********************************************************************/

#define UnivScanMaxFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan2,max,initVal,>)

UnivScanMaxFunc( SUnivScanMaxCh, CHAR,      MIN_CHAR      )
UnivScanMaxFunc( SUnivScanMaxSC, SHORTCARD, MIN_SHORTCARD )
UnivScanMaxFunc( SUnivScanMaxC,  CARDINAL,  MIN_LONGCARD  )
UnivScanMaxFunc( SUnivScanMaxSI, SHORTINT,  MIN_SHORTINT  )
UnivScanMaxFunc( SUnivScanMaxI,  INTEGER,   MIN_LONGINT   )
UnivScanMaxFunc( SUnivScanMaxR,  REAL,      MIN_REAL      )
UnivScanMaxFunc( SUnivScanMaxLR, LONGREAL,  MIN_LONGREAL  )


/********************************************************************
 *****  universal routines with prefix scan operator "minimum"  *****
 ********************************************************************/

#define UnivScanMinFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan2,min,initVal,<)

UnivScanMinFunc( SUnivScanMinCh, CHAR,      MAX_CHAR      )
UnivScanMinFunc( SUnivScanMinSC, SHORTCARD, MAX_SHORTCARD )
UnivScanMinFunc( SUnivScanMinC,  CARDINAL,  MAX_LONGCARD  )
UnivScanMinFunc( SUnivScanMinSI, SHORTINT,  MAX_SHORTINT  )
UnivScanMinFunc( SUnivScanMinI,  INTEGER,   MAX_LONGINT   )
UnivScanMinFunc( SUnivScanMinR,  REAL,      MAX_REAL      )
UnivScanMinFunc( SUnivScanMinLR, LONGREAL,  MAX_LONGREAL  )


/***************************************************************************
 *****  universal routines with prefix scan operator "multiplication"  *****
 ***************************************************************************/

#define UnivScanMulFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan1,prod,initVal,*=)

UnivScanMulFunc( SUnivScanMulCh, CHAR,      '\001' )
UnivScanMulFunc( SUnivScanMulSC, SHORTCARD, 1      )
UnivScanMulFunc( SUnivScanMulC,  CARDINAL,  1      )
UnivScanMulFunc( SUnivScanMulSI, SHORTINT,  1      )
UnivScanMulFunc( SUnivScanMulI,  INTEGER,   1      )
UnivScanMulFunc( SUnivScanMulR,  REAL,      1.0    )
UnivScanMulFunc( SUnivScanMulLR, LONGREAL,  1.0    )


/***********************************************************************
 *****  universal routines with prefix scan operator "logical OR"  *****
 ***********************************************************************/

#define UnivScanOrFunc(FuncName,elemType,initVal) \
    UnivScanFunc(FuncName,elemType,GenericUnivScan1,or,initVal,|=)

UnivScanOrFunc( SUnivScanOrCh, CHAR,      '\0' )
UnivScanOrFunc( SUnivScanOrSC, SHORTCARD, 0    )
UnivScanOrFunc( SUnivScanOrC,  CARDINAL,  0    )
UnivScanOrFunc( SUnivScanOrSI, SHORTINT,  0    )
UnivScanOrFunc( SUnivScanOrI,  INTEGER,   0    )
