
/*
 *  Reduction Library for Modula-2*
 *
 *  allows for efficient reduction operations on
 *  arbitrarily large and distributed arrays
 *
 *  MPL implementation by Ernst A. Heinz
 *
 *  Last change: 16.10.1992
 */

#include "msReduce.MASP.h"
#include <stdio.h>


#define ReduceExitValue -88

#define SCalcReduce(elemType)	\
    l = lx / sizeof(elemType);	\
    k = nx % l;			\
    pe = nx / l;		\
    if (k==0) {			\
      k = l;			\
      pe--;			\
    };
/*
    printf("nx = %u  l = %i  k = %i  pe = %i \n",nx,l,k,pe);
*/


/*
 *  normal reduction routines
 *  (singular versions)
 */

#define ReduceFunc(funcName,elemType,infoCalc,localReduce,globalReduce,accu) \
    elemType				\
    funcName(x, nx, lx)			\
    register plural elemType *x;	\
    LONGCARD nx, lx;			\
    {					\
      register plural elemType accu;	\
      elemType result;			\
      register int j, k, l, pe;		\
					\
      infoCalc(elemType);		\
      all {				\
        localReduce(elemType);		\
        result = globalReduce(accu);	\
      }					\
      return (result);			\
    }

/*
 *  normal singular routines with reduction operator "addition"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceAdd(elemType)	\
    sum = 0;				\
    if (iproc<=pe)			\
      for (j=1; j<=k; j++, x++)		\
        sum += *x;			\
    if (k<l)				\
      if (iproc<pe)			\
        for (j=k+1; j<=l; j++, x++)	\
          sum += *x;

#define SReduceAddFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceAdd,globalReduce,sum)

SReduceAddFunc( SReduceAddCh, CHAR,    	 reduceAdd8u  )
SReduceAddFunc( SReduceAddSC, SHORTCARD, reduceAdd16u )
SReduceAddFunc( SReduceAddC,  CARDINAL,	 reduceAdd32u )
SReduceAddFunc( SReduceAddSI, SHORTINT,	 reduceAdd16  )
SReduceAddFunc( SReduceAddI,  INTEGER,	 reduceAdd32  )
SReduceAddFunc( SReduceAddR,  REAL,	 reduceAddf   )
SReduceAddFunc( SReduceAddLR, LONGREAL,	 reduceAddd   )

/*
 *  normal singular routines with reduction operator "logical AND"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceAnd(elemType)	\
    and = 0xFFFFFFFF;			\
    if (iproc<=pe)			\
      for (j=1; j<=k; j++, x++)		\
        and &= *x;			\
    if (k<l)				\
      if (iproc<pe)			\
        for (j=k+1; j<=l; j++, x++)	\
          and &= *x;

#define SReduceAndFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceAnd,globalReduce,and)

SReduceAndFunc( SReduceAndCh, CHAR,    	 reduceAnd8u  )
SReduceAndFunc( SReduceAndSC, SHORTCARD, reduceAnd16u )
SReduceAndFunc( SReduceAndC,  CARDINAL,	 reduceAnd32u )
SReduceAndFunc( SReduceAndSI, SHORTINT,	 reduceAnd16  )
SReduceAndFunc( SReduceAndI,  INTEGER,	 reduceAnd32  )

/*
 *  normal singular routines with reduction operator "maximum"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceMax(elemType)	\
    result = proc[0].*x;		\
    max = result;			\
    {					\
      register plural elemType tmp;	\
					\
      if (iproc<=pe)			\
        for (j=1; j<=k; j++, x++) {	\
          tmp = *x;			\
          if (max<tmp)			\
	    max = tmp;			\
        };				\
      if (k<l)				\
        if (iproc<pe)			\
          for (j=k+1; j<=l; j++, x++) {	\
            tmp = *x;			\
            if (max<tmp)		\
	      max = tmp;		\
	  };				\
    }

#define SReduceMaxFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceMax,globalReduce,max)

SReduceMaxFunc( SReduceMaxCh, CHAR,    	 reduceMax8u  )
SReduceMaxFunc( SReduceMaxSC, SHORTCARD, reduceMax16u )
SReduceMaxFunc( SReduceMaxC,  CARDINAL,	 reduceMax32u )
SReduceMaxFunc( SReduceMaxSI, SHORTINT,	 reduceMax16  )
SReduceMaxFunc( SReduceMaxI,  INTEGER,	 reduceMax32  )
SReduceMaxFunc( SReduceMaxR,  REAL,	 reduceMaxf   )
SReduceMaxFunc( SReduceMaxLR, LONGREAL,	 reduceMaxd   )

/*
 *  normal singular routines with reduction operator "minimum"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceMin(elemType)	\
    result = proc[0].*x;		\
    min = result;			\
    {					\
      register plural elemType tmp;	\
					\
      if (iproc<=pe)			\
        for (j=1; j<=k; j++, x++) {	\
          tmp = *x;			\
          if (min>tmp)			\
	    min = tmp;			\
        };				\
      if (k<l)				\
        if (iproc<pe)			\
          for (j=k+1; j<=l; j++, x++) {	\
            tmp = *x;			\
            if (min>tmp)		\
	      min = tmp;		\
	  };				\
    }

#define SReduceMinFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceMin,globalReduce,min)

SReduceMinFunc( SReduceMinCh, CHAR,    	 reduceMin8u  )
SReduceMinFunc( SReduceMinSC, SHORTCARD, reduceMin16u )
SReduceMinFunc( SReduceMinC,  CARDINAL,	 reduceMin32u )
SReduceMinFunc( SReduceMinSI, SHORTINT,	 reduceMin16  )
SReduceMinFunc( SReduceMinI,  INTEGER,	 reduceMin32  )
SReduceMinFunc( SReduceMinR,  REAL,	 reduceMinf   )
SReduceMinFunc( SReduceMinLR, LONGREAL,	 reduceMind   )

/*
 *  normal singular routines with reduction operator "multiplication"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceMul(elemType)	\
    prod = 1;				\
    if (iproc<=pe)			\
      for (j=1; j<=k; j++, x++)		\
        prod *= *x;			\
    if (k<l)				\
      if (iproc<pe)			\
        for (j=k+1; j<=l; j++, x++)	\
          prod *= *x;

#define SReduceMulFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceMul,globalReduce,prod)

SReduceMulFunc( SReduceMulCh, CHAR,    	 reduceMul8u  )
SReduceMulFunc( SReduceMulSC, SHORTCARD, reduceMul16u )
SReduceMulFunc( SReduceMulC,  CARDINAL,	 reduceMul32u )
SReduceMulFunc( SReduceMulSI, SHORTINT,	 reduceMul16  )
SReduceMulFunc( SReduceMulI,  INTEGER,	 reduceMul32  )
SReduceMulFunc( SReduceMulR,  REAL,	 reduceMulf   )
SReduceMulFunc( SReduceMulLR, LONGREAL,	 reduceMuld   )

/*
 *  normal singular routines with reduction operator "logical OR"
 *  for 1-dimensional spread arrays
 */

#define SLocalReduceOr(elemType)	\
    or = 0x0;				\
    if (iproc<=pe)			\
      for (j=1; j<=k; j++, x++)		\
        or |= *x;			\
    if (k<l)				\
      if (iproc<pe)			\
        for (j=k+1; j<=l; j++, x++)	\
          or |= *x;

#define SReduceOrFunc(funcName,elemType,globalReduce) \
    ReduceFunc(funcName,elemType,SCalcReduce,SLocalReduceOr,globalReduce,or)

SReduceOrFunc( SReduceOrCh, CHAR,      reduceOr8u  )
SReduceOrFunc( SReduceOrSC, SHORTCARD, reduceOr16u )
SReduceOrFunc( SReduceOrC,  CARDINAL,  reduceOr32u )
SReduceOrFunc( SReduceOrSI, SHORTINT,  reduceOr16  )
SReduceOrFunc( SReduceOrI,  INTEGER,   reduceOr32  )


/*
 *  masked reduction routines
 *  (singular versions)
 */

#define MaskReduceFunc(funcName,elemType,infoCalc,localReduce,globalReduce,accu) \
    elemType				\
    funcName(mask, nm, lm, x, nx, lx)	\
    register plural BOOLEAN *mask;	\
    LONGCARD nm, lm;			\
    register plural elemType *x;	\
    LONGCARD nx, lx;			\
    {					\
      register plural elemType accu;	\
      elemType result;			\
      register int j, k, l, pe;		\
					\
      if (nx!=nm) return;		\
					\
      infoCalc(elemType);		\
      all {				\
        localReduce(elemType);		\
        result = globalReduce(accu);	\
      }					\
      return (result);			\
    }

/*
 *  masked singular routines with reduction operator "addition"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceAdd(elemType)		\
    sum = 0;					\
    if (iproc<=pe)				\
      for (j=1; j<=k; j++, mask++, x++)		\
        if (*mask)				\
          sum += *x;				\
    if (k<l)					\
      if (iproc<pe)				\
        for (j=k+1; j<=l; j++, mask++, x++)	\
          if (*mask)				\
	    sum += *x;

#define SMaskReduceAddFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceAdd,globalReduce,sum)

SMaskReduceAddFunc( SMaskReduceAddCh, CHAR,    	 reduceAdd8u  )
SMaskReduceAddFunc( SMaskReduceAddSC, SHORTCARD, reduceAdd16u )
SMaskReduceAddFunc( SMaskReduceAddC,  CARDINAL,	 reduceAdd32u )
SMaskReduceAddFunc( SMaskReduceAddSI, SHORTINT,	 reduceAdd16  )
SMaskReduceAddFunc( SMaskReduceAddI,  INTEGER,	 reduceAdd32  )
SMaskReduceAddFunc( SMaskReduceAddR,  REAL,	 reduceAddf   )
SMaskReduceAddFunc( SMaskReduceAddLR, LONGREAL,	 reduceAddd   )

/*
 *  masked singular routines with reduction operator "logical AND"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceAnd(elemType)		\
    and = 0xFFFFFFFF;				\
    if (iproc<=pe)				\
      for (j=1; j<=k; j++, mask++, x++)		\
        if (*mask)				\
	  and &= *x;				\
    if (k<l)					\
      if (iproc<pe)				\
        for (j=k+1; j<=l; j++, mask++, x++)	\
	  if (*mask)				\
	    and &= *x;

#define SMaskReduceAndFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceAnd,globalReduce,and)

SMaskReduceAndFunc( SMaskReduceAndCh, CHAR,    	 reduceAnd8u  )
SMaskReduceAndFunc( SMaskReduceAndSC, SHORTCARD, reduceAnd16u )
SMaskReduceAndFunc( SMaskReduceAndC,  CARDINAL,	 reduceAnd32u )
SMaskReduceAndFunc( SMaskReduceAndSI, SHORTINT,	 reduceAnd16  )
SMaskReduceAndFunc( SMaskReduceAndI,  INTEGER,	 reduceAnd32  )

/*
 *  masked singular routines with reduction operator "maximum"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceMax(elemType)	\
    result = proc[0].*x;		\
    max = result;			\
    {					\
      register plural elemType tmp;	\
					\
      if (iproc<=pe)			\
        for (j=1; j<=k; j++, x++)	\
          if (*mask) {			\
            tmp = *x;			\
            if (max<tmp)		\
	      max = tmp;		\
          };				\
      if (k<l)				\
        if (iproc<pe)			\
          for (j=k+1; j<=l; j++, x++)	\
            if (*mask) {		\
              tmp = *x;			\
              if (max<tmp)		\
	        max = tmp;		\
	    };				\
    }

#define SMaskReduceMaxFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceMax,globalReduce,max)

SMaskReduceMaxFunc( SMaskReduceMaxCh, CHAR,    	 reduceMax8u  )
SMaskReduceMaxFunc( SMaskReduceMaxSC, SHORTCARD, reduceMax16u )
SMaskReduceMaxFunc( SMaskReduceMaxC,  CARDINAL,	 reduceMax32u )
SMaskReduceMaxFunc( SMaskReduceMaxSI, SHORTINT,	 reduceMax16  )
SMaskReduceMaxFunc( SMaskReduceMaxI,  INTEGER,	 reduceMax32  )
SMaskReduceMaxFunc( SMaskReduceMaxR,  REAL,	 reduceMaxf   )
SMaskReduceMaxFunc( SMaskReduceMaxLR, LONGREAL,	 reduceMaxd   )

/*
 *  masked singular routines with reduction operator "minimum"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceMin(elemType)	\
    result = proc[0].*x;		\
    min = result;			\
    {					\
      register plural elemType tmp;	\
					\
      if (iproc<=pe)			\
        for (j=1; j<=k; j++, x++)	\
          if (*mask) {			\
            tmp = *x;			\
            if (min>tmp)		\
	      min = tmp;		\
          };				\
      if (k<l)				\
        if (iproc<pe)			\
          for (j=k+1; j<=l; j++, x++)	\
            if (*mask) {		\
              tmp = *x;			\
              if (min>tmp)		\
	        min = tmp;		\
	    };				\
    }

#define SMaskReduceMinFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceMin,globalReduce,min)

SMaskReduceMinFunc( SMaskReduceMinCh, CHAR,    	 reduceMin8u  )
SMaskReduceMinFunc( SMaskReduceMinSC, SHORTCARD, reduceMin16u )
SMaskReduceMinFunc( SMaskReduceMinC,  CARDINAL,	 reduceMin32u )
SMaskReduceMinFunc( SMaskReduceMinSI, SHORTINT,	 reduceMin16  )
SMaskReduceMinFunc( SMaskReduceMinI,  INTEGER,	 reduceMin32  )
SMaskReduceMinFunc( SMaskReduceMinR,  REAL,	 reduceMinf   )
SMaskReduceMinFunc( SMaskReduceMinLR, LONGREAL,	 reduceMind   )

/*
 *  masked singular routines with reduction operator "multiplication"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceMul(elemType)		\
    prod = 1;					\
    if (iproc<=pe)				\
      for (j=1; j<=k; j++, mask++, x++)		\
        if (*mask)				\
          prod *= *x;				\
    if (k<l)					\
      if (iproc<pe)				\
        for (j=k+1; j<=l; j++, mask++, x++)	\
          if (*mask)				\
            prod *= *x;

#define SMaskReduceMulFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceMul,globalReduce,prod)

SMaskReduceMulFunc( SMaskReduceMulCh, CHAR,    	 reduceMul8u  )
SMaskReduceMulFunc( SMaskReduceMulSC, SHORTCARD, reduceMul16u )
SMaskReduceMulFunc( SMaskReduceMulC,  CARDINAL,	 reduceMul32u )
SMaskReduceMulFunc( SMaskReduceMulSI, SHORTINT,	 reduceMul16  )
SMaskReduceMulFunc( SMaskReduceMulI,  INTEGER,	 reduceMul32  )
SMaskReduceMulFunc( SMaskReduceMulR,  REAL,	 reduceMulf   )
SMaskReduceMulFunc( SMaskReduceMulLR, LONGREAL,	 reduceMuld   )

/*
 *  masked singular routines with reduction operator "logical OR"
 *  for 1-dimensional spread arrays
 */

#define SLocalMaskReduceOr(elemType)		\
    or = 0x0;					\
    if (iproc<=pe)				\
      for (j=1; j<=k; j++, mask++, x++)		\
        if (*mask)				\
          or |= *x;				\
    if (k<l)					\
      if (iproc<pe)				\
        for (j=k+1; j<=l; j++, mask++, x++)	\
          if (*mask)				\
            or |= *x;

#define SMaskReduceOrFunc(funcName,elemType,globalReduce) \
    MaskReduceFunc(funcName,elemType,SCalcReduce,SLocalMaskReduceOr,globalReduce,or)

SMaskReduceOrFunc( SMaskReduceOrCh, CHAR,      reduceOr8u  )
SMaskReduceOrFunc( SMaskReduceOrSC, SHORTCARD, reduceOr16u )
SMaskReduceOrFunc( SMaskReduceOrC,  CARDINAL,  reduceOr32u )
SMaskReduceOrFunc( SMaskReduceOrSI, SHORTINT,  reduceOr16  )
SMaskReduceOrFunc( SMaskReduceOrI,  INTEGER,   reduceOr32  )


/*
 *  normal reduction routines
 *  (plural versions)
 */

#define pSReduceFunc(funcName,funcNameString,elemType,globalReduce) \
    plural elemType				\
    funcName(x, nx, lx)				\
    plural elemType *plural x;			\
    plural LONGCARD nx, lx;			\
    {						\
      printf(funcNameString);			\
      printf(": not implemented yet\n");	\
      exit(ReduceExitValue);			\
    }

pSReduceFunc( p_0_SReduceAddCh, "p_0_SReduceAddCh", CHAR,      reduceAdd8u  )
pSReduceFunc( p_0_SReduceAddSC, "p_0_SReduceAddSC", SHORTCARD, reduceAdd16u )
pSReduceFunc( p_0_SReduceAddC,	"p_0_SReduceAddC",  CARDINAL,  reduceAdd32u )
pSReduceFunc( p_0_SReduceAddSI,	"p_0_SReduceAddSI", SHORTINT,  reduceAdd16  )
pSReduceFunc( p_0_SReduceAddI,	"p_0_SReduceAddI",  INTEGER,   reduceAdd32  )
pSReduceFunc( p_0_SReduceAddR,	"p_0_SReduceAddR",  REAL,      reduceAddf   )
pSReduceFunc( p_0_SReduceAddLR, "p_0_SReduceAddLR", LONGREAL,  reduceAddd   )

pSReduceFunc( p_0_SReduceAndCh, "p_0_SReduceAndCh", CHAR,      reduceAnd8u  )
pSReduceFunc( p_0_SReduceAndSC, "p_0_SReduceAndSC", SHORTCARD, reduceAnd16u )
pSReduceFunc( p_0_SReduceAndC,	"p_0_SReduceAndC",  CARDINAL,  reduceAnd32u )
pSReduceFunc( p_0_SReduceAndSI, "p_0_SReduceAndSI", SHORTINT,  reduceAnd16  )
pSReduceFunc( p_0_SReduceAndI,	"p_0_SReduceAndI",  INTEGER,   reduceAnd32  )

pSReduceFunc( p_0_SReduceMaxCh, "p_0_SReduceMaxCh", CHAR,      reduceMax8u  )
pSReduceFunc( p_0_SReduceMaxSC, "p_0_SReduceMaxSC", SHORTCARD, reduceMax16u )
pSReduceFunc( p_0_SReduceMaxC,	"p_0_SReduceMaxC",  CARDINAL,  reduceMax32u )
pSReduceFunc( p_0_SReduceMaxSI, "p_0_SReduceMaxSI", SHORTINT,  reduceMax16  )
pSReduceFunc( p_0_SReduceMaxI,	"p_0_SReduceMaxI",  INTEGER,   reduceMax32  )
pSReduceFunc( p_0_SReduceMaxR,	"p_0_SReduceMaxR",  REAL,      reduceMaxf   )
pSReduceFunc( p_0_SReduceMaxLR, "p_0_SReduceMaxLR", LONGREAL,  reduceMaxd   )

pSReduceFunc( p_0_SReduceMinCh, "p_0_SReduceMinCh", CHAR,      reduceMin8u  )
pSReduceFunc( p_0_SReduceMinSC, "p_0_SReduceMinSC", SHORTCARD, reduceMin16u )
pSReduceFunc( p_0_SReduceMinC,	"p_0_SReduceMinC",  CARDINAL,  reduceMin32u )
pSReduceFunc( p_0_SReduceMinSI, "p_0_SReduceMinSI", SHORTINT,  reduceMin16  )
pSReduceFunc( p_0_SReduceMinI,	"p_0_SReduceMinI",  INTEGER,   reduceMin32  )
pSReduceFunc( p_0_SReduceMinR,	"p_0_SReduceMinR",  REAL,      reduceMinf   )
pSReduceFunc( p_0_SReduceMinLR, "p_0_SReduceMinLR", LONGREAL,  reduceMind   )

pSReduceFunc( p_0_SReduceMulCh, "p_0_SReduceMulCh", CHAR,      reduceMul8u  )
pSReduceFunc( p_0_SReduceMulSC, "p_0_SReduceMulSC", SHORTCARD, reduceMul16u )
pSReduceFunc( p_0_SReduceMulC,	"p_0_SReduceMulC",  CARDINAL,  reduceMul32u )
pSReduceFunc( p_0_SReduceMulSI, "p_0_SReduceMulSI", SHORTINT,  reduceMul16  )
pSReduceFunc( p_0_SReduceMulI,	"p_0_SReduceMulI",  INTEGER,   reduceMul32  )
pSReduceFunc( p_0_SReduceMulR,	"p_0_SReduceMulR",  REAL,      reduceMulf   )
pSReduceFunc( p_0_SReduceMulLR, "p_0_SReduceMulLR", LONGREAL,  reduceMuld   )

pSReduceFunc( p_0_SReduceOrCh, "p_0_SReduceOrCh", CHAR,      reduceOr8u  )
pSReduceFunc( p_0_SReduceOrSC, "p_0_SReduceOrSC", SHORTCARD, reduceOr16u )
pSReduceFunc( p_0_SReduceOrC,  "p_0_SReduceOrC",  CARDINAL,  reduceOr32u )
pSReduceFunc( p_0_SReduceOrSI, "p_0_SReduceOrSI", SHORTINT,  reduceOr16  )
pSReduceFunc( p_0_SReduceOrI,  "p_0_SReduceOrI",  INTEGER,   reduceOr32  )


/*
 *  masked reduction routines
 *  (plural versions)
 */

#define pSMaskReduceFunc(funcName,funcNameString,elemType,globalReduce) \
    plural elemType				\
    funcName(mask, nm, lm, x, nx, lx)		\
    plural BOOLEAN *plural mask;		\
    plural LONGCARD nm, lm;			\
    plural elemType *plural x;			\
    plural LONGCARD nx, lx;			\
    {						\
      printf(funcNameString);			\
      printf(": not implemented yet\n");	\
      exit(ReduceExitValue);			\
    }

pSMaskReduceFunc( p_0_SMaskReduceAddCh, "p_0_SMaskReduceAddCh", CHAR,      reduceAdd8u  )
pSMaskReduceFunc( p_0_SMaskReduceAddSC, "p_0_SMaskReduceAddSC", SHORTCARD, reduceAdd16u )
pSMaskReduceFunc( p_0_SMaskReduceAddC,	"p_0_SMaskReduceAddC",  CARDINAL,  reduceAdd32u )
pSMaskReduceFunc( p_0_SMaskReduceAddSI,	"p_0_SMaskReduceAddSI", SHORTINT,  reduceAdd16  )
pSMaskReduceFunc( p_0_SMaskReduceAddI,	"p_0_SMaskReduceAddI",  INTEGER,   reduceAdd32  )
pSMaskReduceFunc( p_0_SMaskReduceAddR,	"p_0_SMaskReduceAddR",  REAL,      reduceAddf   )
pSMaskReduceFunc( p_0_SMaskReduceAddLR, "p_0_SMaskReduceAddLR", LONGREAL,  reduceAddd   )

pSMaskReduceFunc( p_0_SMaskReduceAndCh, "p_0_SMaskReduceAndCh", CHAR,      reduceAnd8u  )
pSMaskReduceFunc( p_0_SMaskReduceAndSC, "p_0_SMaskReduceAndSC", SHORTCARD, reduceAnd16u )
pSMaskReduceFunc( p_0_SMaskReduceAndC,	"p_0_SMaskReduceAndC",  CARDINAL,  reduceAnd32u )
pSMaskReduceFunc( p_0_SMaskReduceAndSI, "p_0_SMaskReduceAndSI", SHORTINT,  reduceAnd16  )
pSMaskReduceFunc( p_0_SMaskReduceAndI,	"p_0_SMaskReduceAndI",  INTEGER,   reduceAnd32  )

pSMaskReduceFunc( p_0_SMaskReduceMaxCh, "p_0_SMaskReduceMaxCh", CHAR,      reduceMax8u  )
pSMaskReduceFunc( p_0_SMaskReduceMaxSC, "p_0_SMaskReduceMaxSC", SHORTCARD, reduceMax16u )
pSMaskReduceFunc( p_0_SMaskReduceMaxC,	"p_0_SMaskReduceMaxC",  CARDINAL,  reduceMax32u )
pSMaskReduceFunc( p_0_SMaskReduceMaxSI, "p_0_SMaskReduceMaxSI", SHORTINT,  reduceMax16  )
pSMaskReduceFunc( p_0_SMaskReduceMaxI,	"p_0_SMaskReduceMaxI",  INTEGER,   reduceMax32  )
pSMaskReduceFunc( p_0_SMaskReduceMaxR,	"p_0_SMaskReduceMaxR",  REAL,      reduceMaxf   )
pSMaskReduceFunc( p_0_SMaskReduceMaxLR, "p_0_SMaskReduceMaxLR", LONGREAL,  reduceMaxd   )

pSMaskReduceFunc( p_0_SMaskReduceMinCh, "p_0_SMaskReduceMinCh", CHAR,      reduceMin8u  )
pSMaskReduceFunc( p_0_SMaskReduceMinSC, "p_0_SMaskReduceMinSC", SHORTCARD, reduceMin16u )
pSMaskReduceFunc( p_0_SMaskReduceMinC,	"p_0_SMaskReduceMinC",  CARDINAL,  reduceMin32u )
pSMaskReduceFunc( p_0_SMaskReduceMinSI, "p_0_SMaskReduceMinSI", SHORTINT,  reduceMin16  )
pSMaskReduceFunc( p_0_SMaskReduceMinI,	"p_0_SMaskReduceMinI",  INTEGER,   reduceMin32  )
pSMaskReduceFunc( p_0_SMaskReduceMinR,	"p_0_SMaskReduceMinR",  REAL,      reduceMinf   )
pSMaskReduceFunc( p_0_SMaskReduceMinLR, "p_0_SMaskReduceMinLR", LONGREAL,  reduceMind   )

pSMaskReduceFunc( p_0_SMaskReduceMulCh, "p_0_SMaskReduceMulCh", CHAR,      reduceMul8u  )
pSMaskReduceFunc( p_0_SMaskReduceMulSC, "p_0_SMaskReduceMulSC", SHORTCARD, reduceMul16u )
pSMaskReduceFunc( p_0_SMaskReduceMulC,	"p_0_SMaskReduceMulC",  CARDINAL,  reduceMul32u )
pSMaskReduceFunc( p_0_SMaskReduceMulSI, "p_0_SMaskReduceMulSI", SHORTINT,  reduceMul16  )
pSMaskReduceFunc( p_0_SMaskReduceMulI,	"p_0_SMaskReduceMulI",  INTEGER,   reduceMul32  )
pSMaskReduceFunc( p_0_SMaskReduceMulR,	"p_0_SMaskReduceMulR",  REAL,      reduceMulf   )
pSMaskReduceFunc( p_0_SMaskReduceMulLR, "p_0_SMaskReduceMulLR", LONGREAL,  reduceMuld   )

pSMaskReduceFunc( p_0_SMaskReduceOrCh, "p_0_SMaskReduceOrCh", CHAR,      reduceOr8u  )
pSMaskReduceFunc( p_0_SMaskReduceOrSC, "p_0_SMaskReduceOrSC", SHORTCARD, reduceOr16u )
pSMaskReduceFunc( p_0_SMaskReduceOrC,  "p_0_SMaskReduceOrC",  CARDINAL,  reduceOr32u )
pSMaskReduceFunc( p_0_SMaskReduceOrSI, "p_0_SMaskReduceOrSI", SHORTINT,  reduceOr16  )
pSMaskReduceFunc( p_0_SMaskReduceOrI,  "p_0_SMaskReduceOrI",  INTEGER,   reduceOr32  )
