#include <stdio.h>

#ifdef _AIX_
#include <unistd.h>
#else
#include <sys/types.h>
#endif

#define MAX_SIZE_RADICAL  30
#define MAX_SIZE_PARAFFIN 60

#define NUM_SUB_RADICALS  3
#define MAX_NUM_RADICALS  4

enum kind_radical { H, C };
typedef enum kind_radical KIND_RADICAL;

enum kind_paraffin { BCP, CCP };
typedef enum kind_paraffin KIND_PARAFFIN;

/***************************************************************
 * Structures for Radicals and Paraffins
 ***************************************************************/

typedef
struct radical
{
  KIND_RADICAL   kind;
  struct radical *sub_radicals[NUM_SUB_RADICALS];
  struct radical *next;
}
RADICAL;

#define RADNULL ((RADICAL *) NULL)

typedef
struct paraffin
{
  KIND_PARAFFIN   kind;
  RADICAL         *radicals[MAX_NUM_RADICALS];
  struct paraffin *next;
}
PARAFFIN;

#define PARNULL ((PARAFFIN *) NULL)

typedef
struct tuple
{
  PARAFFIN     *bcp;
  PARAFFIN     *ccp;
}
TUPLE;

RADICAL  *radicals[MAX_SIZE_RADICAL];
int       radicals_counts[MAX_SIZE_RADICAL];

PARAFFIN *BCP_array[MAX_SIZE_PARAFFIN];
PARAFFIN *CCP_array[MAX_SIZE_PARAFFIN];
TUPLE    *paraffins_array[MAX_SIZE_PARAFFIN];

int       BCP_counts[MAX_SIZE_PARAFFIN];
int       CCP_counts[MAX_SIZE_PARAFFIN];
int       paraffins_counts[MAX_SIZE_PARAFFIN];

/***************************************************************
 * User-Level Heap Stuff ...
 ***************************************************************/
#define MEGABYTE (1<<20)
#define HEAPSIZE (120*MEGABYTE)

static char *heapbase, *heapfree, *heaplimit;

void init_heap()
{
  int p;

  p = sbrk(HEAPSIZE);
  if (p == -1) {
    printf("Error (init_heap) cannot alloc %d bytes.\n", HEAPSIZE);
    exit(0);
  }
  heapbase = (char *) p;
  heaplimit = heapbase + HEAPSIZE - 1;
}

inline void reset_heap()
{
  heapfree = heapbase;
}

inline char *halloc(int n)
{
  char *p;

  p = heapfree;
  heapfree += n;

  if (heapfree > heaplimit) {
    printf("Error (halloc) No space for %d bytes.\n", n);
    exit(0);
  }

  return p;
}

#define U_NEW_STRUCT(name) ((name *) (halloc(sizeof(name))))

/***************************************************************
 * Printout Functions
 ***************************************************************/
void print_radical_counts(int n)
{
  int i;
  int nrads = 0;

  printf("\nRadical Counts\n");
  for (i = 1; i <= n; i++)
    {
      printf("\tradicals[%2d] : %d\n", i, radicals_counts[i]);
      nrads += radicals_counts[i];
    }
  printf("\nRadicals memory: %d\n", nrads*sizeof(RADICAL));
}

void print_paraffin_counts(int n)
{
  int i;
  int npars = 0;

  printf("\nParaffin Counts\n");
  for (i = 1; i <= n; i++)
    {
      printf("\tparaffins[%2d] : %d\n", i, paraffins_counts[i]);
      npars += paraffins_counts[i];
    }
  printf("\nParaffin memory: %d\n", npars*sizeof(PARAFFIN));
}

/***************************************************************
 * Radical Generation
 ***************************************************************/
inline void make_radical(int m, RADICAL *r1, RADICAL *r2, RADICAL *r3)
{
  RADICAL *rad = U_NEW_STRUCT(RADICAL);
		
  rad->kind = C;
  rad->sub_radicals[0] = r1;
  rad->sub_radicals[1] = r2;
  rad->sub_radicals[2] = r3;

  /* Link it into the radicals list array */
  rad->next = radicals[m];
  radicals[m] = rad;
}

void rads_of_size_n_with_no_copying (int m)
{
  int nc1, nc2;
  int n = m-1;
  int num_rads = 0;
  
  for (nc1 = 0; nc1 <= n/3; nc1++)
    for (nc2 = nc1; nc2 <= (n-nc1)/2; nc2++)
      {
	int nc3 = n - (nc1 + nc2);
	RADICAL *r1, *r2, *r3;
	
	for (r1 = radicals[nc1]; r1 != RADNULL; r1 = r1->next)
	  for (r2 = ((nc1 == nc2) ? r1 : radicals[nc2]); r2 != RADNULL; r2 = r2->next)
	    for (r3 = ((nc2 == nc3) ? r2 : radicals[nc3]); r3 != RADNULL; r3 = r3->next)
	      {
		make_radical(m, r1, r2, r3);
		num_rads++;
	      } 
      }
  radicals_counts[m] = num_rads;
#ifdef DBG
  printf("\tradicals[%d] computed: %d\n", m, num_rads);
#endif
}

void radical_generator (int n)
{
  RADICAL *h = U_NEW_STRUCT(RADICAL);
  int i;

#ifdef DBG
  printf("Begin RADICALS computation\n");
#endif
  
  h->kind = H;
  h->next = RADNULL;
  radicals[0] = h;
  
  for (i = 1; i <= n; i++)
    {
      radicals[i] = RADNULL;
      rads_of_size_n_with_no_copying(i);
    }
}

/***************************************************************
 * Paraffins Generation
 ***************************************************************/
inline void make_BCP(int n, RADICAL *r1, RADICAL *r2)
{
  PARAFFIN *p = U_NEW_STRUCT(PARAFFIN);
	
  p->kind = BCP;
  p->radicals[0] = r1;
  p->radicals[1] = r2;

  /* Link it into the BCP list array */  
  p->next = BCP_array[n];
  BCP_array[n] = p;
}

void BCP_generator (int n)
{
  int half_n = n/2;
  int num_BCPs = 0;

  BCP_array[n] = PARNULL;
  /* Only for even n's */
  if (! (n & 1))
    {
      RADICAL  *r1, *r2;
      
      for (r1 = radicals[half_n]; r1 != RADNULL; r1 = r1->next)
	for (r2 = r1; r2 != RADNULL; r2 = r2->next)
	  {
	    make_BCP(n, r1, r2);
	    num_BCPs++;
	  }
    }
  
  BCP_counts[n] = num_BCPs;
#ifdef DBG
  printf("\tBCP[%d] computed: %d\n", n, num_BCPs);
#endif
}

inline void make_CCP(int n, RADICAL *r1, RADICAL *r2, RADICAL *r3, RADICAL *r4)
{
  PARAFFIN *p = U_NEW_STRUCT(PARAFFIN);
		      
  p->kind = CCP;
  p->radicals[0] = r1;
  p->radicals[1] = r2;
  p->radicals[2] = r3;
  p->radicals[3] = r4;
  
  p->next = CCP_array[n];
  CCP_array[n] = p;
}

void CCP_generator_with_no_copying (int n)
{
  int m = n-1;
  int half_m = m/2;
  int nc1, nc2;
  int num_CCPs = 0;

  CCP_array[n] = PARNULL;
  
  for (nc1 = 0; nc1 <= m/4; nc1++)
    for (nc2 = nc1; nc2 <= (m-nc1)/3; nc2++)
      {
	int k;
	int nc3;
	
	if (m == half_m*2)
	  k = half_m-nc1-nc2;
	else 
	  k = half_m+1-nc1-nc2;
	
	if (k < nc2)
	  k = nc2;
	
	for (nc3 = k; nc3 <= (m-nc1-nc2)/2; nc3++)
	  {
	    RADICAL  *r1, *r2, *r3, *r4;
	    int nc4 = m - nc1 - nc2 - nc3;

	    for (r1 = radicals[nc1]; r1 != RADNULL; r1 = r1->next)
	      for (r2 = ((nc1 == nc2) ? r1 : radicals[nc2]);
		   r2 != RADNULL; r2 = r2->next)
		for (r3 = ((nc2 == nc3) ? r2 : radicals[nc3]);
		     r3 != RADNULL; r3 = r3->next)
		  for (r4 = ((nc3 == nc4) ? r3 : radicals[nc4]);
		       r4 != RADNULL; r4 = r4->next)
		    {
		      make_CCP(n, r1, r2, r3, r4);
		      num_CCPs++;
		    }
	  }
      }
  CCP_counts[n] = num_CCPs;
#ifdef DBG
  printf("\tCCP[%d] computed: %d\n", n, num_CCPs);
#endif
}

void paraffins_until (int n)
{
  int i;

  radical_generator(n/2);
#ifdef DBG
  printf("Begin BCP computation\n");
  printf("Begin CCP computation\n");
#endif
  
  for (i = 1; i <= n; i++)
    {
      TUPLE *t = U_NEW_STRUCT(TUPLE);
      
      BCP_generator(i);
      CCP_generator_with_no_copying(i);
      
      t->bcp = BCP_array[i];
      t->ccp = CCP_array[i];    
      paraffins_array[n] = t;
      paraffins_counts[i] = BCP_counts[i] + CCP_counts[i];
    }
}

main (int argc, char *argv[])
{
  int i, n, iters;
  int doprint;
  
  if (argc != 4) {
    printf("Usage: paraffins size iterations doprint\n");
    exit(0);
  }
  
  n = atoi(argv[1]);
  iters = atoi(argv[2]);
  doprint = atoi(argv[3]);

  if (n > MAX_SIZE_PARAFFIN) {
    printf("Your input size %d is larger than the maximum paraffin size %d\n",
	   MAX_SIZE_PARAFFIN);
    exit(0);
  }
  
  init_heap();

  for (i = 0; i < iters; i++)
    {
      reset_heap();
      paraffins_until(n);
    }
  
  /* print the information about results */
  if (doprint) {
    print_radical_counts(n/2);
    print_paraffin_counts(n);
  }
}
