#include <stdio.h>
#include <math.h>

/*
 * GAMTEB -- Monte Carlo photon transport algorithm from LANL
 * 
 * This is snarfed from Agarwal's group, where someone ported
 * our "original" Id version, which was based on a LANL Fortran
 * version.  The Id version was written by Olaf Lubeck or 
 * someone who works for him.
 *
 * This version has been heavily revised, cleaned up, and
 * checked against the Id version.  Some bugs were found; some
 * more have probably been left in ... it's hard to tell.
 *
 * Here's a nice paper describing the Gamteb application:
 *
 * "A parallel Monte Carlo transport algorithm using a 
 *  pseudo-random tree to guarantee reproducibility"
 * Frederickson, Hiromoto & Larson, _Parallel_Computing_ 4(1987)
 * p 281-290
 * 
 * Current C version by:
 *
 * Andy Shaw
 * shaw@abp.lcs.mit.edu
 * Mon Nov 28 23:36:28 EST 1994
 *
 */

#define TRUE 1
#define FALSE 0

#define BSCAT_COUNTER    2
#define ESC_COUNTER      0
#define TRANS_COUNTER    1
#define ROUL_DIE_COUNTER 6
#define COL_COUNTER      7
#define NOCOL_COUNTER    8
#define ECUT_COUNTER     4
#define WTCUT_COUNTER    5
#define THREE_COUNTER    3

/* Global Counter for Adds */
int counters[9] = {0,0,0,0,0,0,0,0,0};

inline void bump_counter(int counter)
{
  counters[counter]++;
}

inline void bump_counter_inc(int counter, int inc)
{
  counters[counter] += inc;
}

/* These tables have a dummy value in the zero slot to simulate 1-based
   arrays (a la Fortran and Id) */
double energies[36] =
{0.0,				/*dummy value*/
   .001,.0015,.002,.003,.004,.005,.006,.008,.01,.015,.02,.03,
   .04,.05,.06,.08,.1,.15,.2,.3,.4,.5,.6,.8,1.0,1.5,2.0,3.0,
   4.0,5.0,6.0,8.0,10.0,15.0,20.0
};

double xc[36] =
{0.0,				/*dummy value*/
   .015,.0296,.0451,.0717,.0913,.105,.115,.128,.137,.152,.160,
   .165,.165,.163,.160,.153,.146,.133,.122,.106,.0953,.0867,.0802,
   .0707,.0637,.0516,.0440,.0346,.0289,.025,.0221,.0181,.0154,.0114,.00913
};

double xp[36] =
{0.0,				/*dummy value*/
   0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,
   0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,.0000792,.000316,.000923,
   .00153,.00208,.00256,.00343,.00414,.00547,.00652
};

double xpe[36] =
{0.0,				/*dummy value*/
   2010.0,632.0,280.0,87.7,37.3,18.9,10.4,4.01,1.91,.489,.192,.0491,
   .0186,.00887,.00481,.00179,.000862,.000234,.0000918,0.0,0.0,0.0,
   0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0,0.0};

/* These are close to the eventual values, but to match with the Id version, we
   recalculate them to stick them into the arrays */
double xcompton_table[] =
{
  -30.000000000000,-3.411247717516,-2.731523557288,-2.310415672109,-1.846807171012,
  -1.605147131017,-1.465337568460,-1.374365790255,-1.267267654698,-1.199316992790,
  -1.095417397772,-1.044124103384,-1.013352444717,-1.013352444717,-1.025547717811,
  -1.044124103384,-1.088859997225,-1.135691296910,-1.228948790396,-1.315276873885,
  -1.455858824506,-1.562268107958,-1.656844034831,-1.734774403745,-1.860852345715,
  -1.965113356040,-2.175776246130,-2.335108284700,-2.575444236554,-2.755456323499,
  -2.900422093750,-3.023720310094,-3.223385980346,-3.384930409198,-3.685684563217,-3.907732224011,
};

double xpair_table[] =
{
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-8.655076898780,-7.271310984013,-6.199423963097,-5.694030183214,
  -5.386930024905,-5.179290660126,-4.886737657440,-4.698602130781,-4.420019302184,-4.244423542679,
};

double ergs_table[] =
{
  -30.000000000000,-6.907755278982,-6.502290170874,-6.214608098422,-5.809142990314,
  -5.521460917862,-5.298317366548,-5.115995809754,-4.828313737302,-4.605170185988,
  -4.199705077880,-3.912023005428,-3.506557897320,-3.218875824868,-2.995732273554,
  -2.813410716760,-2.525728644308,-2.302585092994,-1.897119984886,-1.609437912434,
  -1.203972804326,-0.916290731874,-0.693147180560,-0.510825623766,-0.223143551314,
  0.000000000000,0.405465108108,0.693147180560,1.098612288668,1.386294361120,
  1.609437912434,1.791759469228,2.079441541680,2.302585092994,2.708050201102,
  2.995732273554,
};

double xphoto_table[] =
{
  -30.000000000000,8.394347361417,7.237346754511,6.423246963534,5.262379259742,
  4.407450687014,3.727619282430,3.130263166512,2.177248601683,1.435560602423,
  0.073064570857,-0.861802546590,-2.225438883817,-3.196136337899,-3.936623122296,
  -4.548600834500,-5.537082298765,-6.267797926936,-7.571732082242,-8.507440899974,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
  -30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,-30.000000000000,
};

/* These are histograms */
double escape_profile[36];
double bscatter_profile[36];
double transit_profile[36];

/* Tally statistic into histograms */
inline void tally(double wt, double stat[], int index)
{
  stat[index]+=wt;
}

void print_histogram(double *array, double height)
{
  int i;
  
  for(i=1; i<=35; i++) 
    printf("[%d] %f\n", i, array[i]);
  printf("\n");
}

/* This is not very accurate ... should probably call the log10 library function */
inline double bounded_log10(double x)
{
  if (x<1.0e-13)
    return -30.0;
  else
    return log(2.2 * x);
}

/* Bounded Natural Log */
inline double bounded_ln(double x)
{
  if (x<1.0e-13)
    return -30.0;
  else
    return log(x);
}

struct Sparticle
{
  double x,y,z;
  double u,v,w;
  double wt, e;
  int e_bin, cell;
  double seed;
  /* These probabilities? used to be in a separate data structure */
  double pcompton;
  double ppair;
  double pphoto;
  double ptotal;
};

/* Code to alloc and return particles to free list */
int *free_list = 0;
int alloced = 0;

inline struct Sparticle *alloc_particle()
{
  struct Sparticle *new_particle;
  alloced++;

  if (free_list != 0)
    {
      new_particle = (struct Sparticle *) free_list;
      free_list = (int *) *free_list;
    }
  else
    new_particle = (struct Sparticle *) malloc(sizeof(struct Sparticle));

  return new_particle;
}

inline void dealloc_particle(struct Sparticle *particle)
{
  int *ptr = (int *) particle;

  alloced--;
  *ptr = (int) free_list;
  free_list = ptr;
}

inline void translate_particle(double x, double y, double z,
			       double u, double v, double w, double d,
			       double *nx, double *ny, double *nz
			       )
{
  *nx = x+u*d;
  *ny = y+v*d;
  *nz = z+w*d;
}

/* Definition which is called recursively by several procedures */
void transport_particle(struct Sparticle *);

/* "Lehmer tree" Random Number Generator from reference above */
inline void grand (double seed, double *nextseed, double *rand)
{
  /* We want to do a floor here, but a conversion to int from
     double does a truncate, which is the same as a float for
     positive numbers, and seed is always a positive number less
     than 1, so the int won't overflow */
  double r1 = (((int)(3145257.0*seed))+271829)/655357.0;
  double r2 = (((int)(2713829.0*seed))+314557)/511267.0;

  *nextseed = r1 - (int) r1;
  *rand     = r2 - (int) r2;
}

/* Linear Search -- very stupid */
inline int find_energy_bin(double e)
{
  int i = 1;
  double ee = log(e);
  
  while (ee > ergs_table[i])
    i++;
  return i;
}

/* Interpolate from the table entries what the probability
   values are for a particle */
void set_particle_probabilities(struct Sparticle *particle)
{
  double e  = particle->e;
  int e_bin = particle->e_bin;
  /* f is the distance from the last bin */
  double f = (log(e) - ergs_table[e_bin-1]) /
    (ergs_table[e_bin] - ergs_table[e_bin-1]);
  
  particle->pcompton = exp (xcompton_table[e_bin-1] +
			    (f*(xcompton_table[e_bin]-xcompton_table[e_bin-1])));
  particle->ppair    = exp (xpair_table[e_bin-1] +
			    (f*(xpair_table[e_bin]-xpair_table[e_bin-1])));
  particle->pphoto   = exp (xphoto_table[e_bin-1] +
			    (f*(xphoto_table[e_bin]-xphoto_table[e_bin-1])));
  particle->ptotal   = particle->pcompton+particle->ppair+particle->pphoto;
}

/* Exponential distribution over random time to collision */
inline double dist_to_collision(double ptotal, double rand)
{
  return (-log(rand)/ptotal);
}

/* Ugly, and quite possibly wrong ... */
inline double dist_to_cylinder(double x, double y, double z, double u, double v, double w)
{
  double m,b,s,r,xi,yi,zi,t;

  t = u*u + w*w;
  if (t == 0.0) 
    /* If velocity vector is parallel to axis of cylinder,
       intersection is at infinity */
    return 100.0;
  else
    {
      m=w/u;
      b=z-m*x;
      s=m*m+1.0;
      r=sqrt(fabs(s-b*b));
      if (u>0.0)
	xi=(-m*b+r)/s;
      else
	xi=(-m*b-r)/s;
      return (xi-x)/u;
    }
}

inline double dist_to_plane(double y, double y_plane, double v)
{
  double t;

  if (v != 0.0)
    t = (y_plane - y) / v;
  else
    t = 100.0;

  if (t <= 0.0)
    return 100.0;
  else
    return t;
}

/* This function supposedly calculates the closest intersecting surface
   for this particle, given a position and velocity.  However, there
   are some bugs in it, as far as I can tell ... I'm keeping it this
   way to be compatible with the Id version */
inline void dist_to_surface(double x, double y, double z, double u, double v, double w,
			    double *distance, int *surface
			    )
{
  double dist_cylinder = dist_to_cylinder( x, y, z, u, v, w);
  int closest_plane = ((int) (y/10.0)) + 1;
  double dist_plane;

  /* if the particle is moving forward, it's the next plane,
     otherwise, it's the previous one (i.e. this one). */
  if (v>0.0)
    closest_plane++;

  if (closest_plane == 2) /* can't be, because we don't have plane 2 ... */
    {
      if (v>0.0)
	closest_plane++;
      else
	closest_plane--;
    }

  /* ??? This should _really_ go after the
     dist_plane = ...  statement ... */
  if (closest_plane > 11)
    closest_plane = 11;

  dist_plane = dist_to_plane( y, (closest_plane-1)*10.0, v);

  if (dist_cylinder <= dist_plane)
    {
      *distance = dist_cylinder + 1.0e-13;
      *surface  = 2;
    }
  else
    {
      *distance = dist_plane + 1.0e-13;
      *surface  = closest_plane;
    }
}

void russian_roulette(struct Sparticle *part, double d_surf)
{
  double seed1, rand1;
  struct Sparticle *new_particle = alloc_particle();

  /* Flip a coin to see if you survive */
  grand(part->seed, &rand1, &seed1);

  /* 50/50 chance that particle dies on going back ... */
  if (0.5 < rand1)
    bump_counter(ROUL_DIE_COUNTER);
  else
    {
      double new_seed, temp;
      /* Find the new seed for the particle */
      grand(seed1, &new_seed, &temp);
      translate_particle(part->x, part->y, part->z,
			 part->u, part->v, part->w, d_surf,
			 &new_particle->x, &new_particle->y, &new_particle->z
			 );
  
      new_particle->u = part->u;
      new_particle->v = part->v;
      new_particle->w = part->w;

      /* ??? Isn't this weight supposed to double? -Andy */
      new_particle->wt    = 0.5*part->wt;
      new_particle->e     = part->e;
      new_particle->e_bin = part->e_bin;
      new_particle->cell  = part->cell - 1;
      new_particle->seed  = new_seed;

      new_particle->pcompton = part->pcompton;
      new_particle->ppair    = part->ppair;
      new_particle->pphoto   = part->pphoto;
      new_particle->ptotal   = part->ptotal;

      transport_particle(new_particle);
    }
  dealloc_particle(new_particle);
}

void splitting(struct Sparticle *particle, double d_surf)
{
  double x  = particle->x;
  double y  = particle->y;
  double z  = particle->z;
  double u  = particle->u;
  double v  = particle->v;
  double w  = particle->w;
  double wt = particle->wt;
  double e  = particle->e;
  int e_bin = particle->e_bin;
  int cell  = particle->cell;
  double seed = particle->seed;
  int n_ways;
  int i;

  if (cell == 7)
    n_ways = 3;
  else if (cell == 8)
    n_ways = 4;
  else
    n_ways = 2;

  for (i=0; i<n_ways; i++)
    {
      struct Sparticle *d_particle = alloc_particle();
      double rnd;
      grand(seed, &seed, &rnd);

      translate_particle(x, y, z, u, v, w, d_surf,
			 &d_particle->x, &d_particle->y, &d_particle->z
			 );
      d_particle->u = u;
      d_particle->v = v;
      d_particle->w = w;
      d_particle->wt = wt/n_ways;
      d_particle->e = e;
      d_particle->e_bin = e_bin;
      d_particle->cell = cell+1;
      d_particle->seed = rnd;
      
      d_particle->pcompton = particle->pcompton;
      d_particle->ppair    = particle->ppair;
      d_particle->pphoto   = particle->pphoto;
      d_particle->ptotal   = particle->ptotal;

      transport_particle(d_particle);
      dealloc_particle(d_particle);
    }

  bump_counter_inc (THREE_COUNTER, (n_ways-1));
}

int photo_elect(struct Sparticle *particle, double d_coll,
		double pphoto, double ptotal, struct Sparticle **new_particle)
{
  double rand1, rand2, new_seed, rand3;
  double tmp_wt,new_wt;
  int cell = particle->cell;
  int kill;
  
  grand(particle->seed, &new_seed, &rand2);
  grand(rand2, &rand1, &rand3);

  tmp_wt = particle->wt*(1.0-(pphoto/ptotal));
  if (tmp_wt>0.25)
    {
      new_wt = tmp_wt;
      kill = FALSE;
    }
  else if ((tmp_wt * (1 << (cell-1)))<(rand1*0.5))
    {
      new_wt = tmp_wt;
      kill = TRUE;
    }
  else
    {
      new_wt = 0.5/(1 << (cell-1));
      kill = FALSE;
    }

  if (kill == FALSE) {
    struct Sparticle *n_particle = alloc_particle();
    translate_particle(particle->x, particle->y, particle->z,
		       particle->u, particle->v, particle->w, d_coll,
		       &n_particle->x, &n_particle->y, &n_particle->z
		       );
    /* Same velocity */
    n_particle->u     = particle->u;
    n_particle->v     = particle->v;
    n_particle->w     = particle->w;
  
    n_particle->wt    = new_wt;
    n_particle->e     = particle->e;
    n_particle->e_bin = particle->e_bin;
    n_particle->cell  = particle->cell;
    n_particle->seed  = new_seed;

    n_particle->pcompton = particle->pcompton;
    n_particle->ppair    = particle->ppair;
    n_particle->pphoto   = particle->pphoto;
    n_particle->ptotal   = particle->ptotal;
    *new_particle = n_particle;
  } 

  return kill;
}

#define klein_f1(e,x2,x5,x6)  2.0*e*(1.0+e)*x5*x5+4.0*x2+(1.0-2.0*x2*(1.0+x2))*x6;
#define klein_f2(x2,x3,x4,x5,x7)  1.0+x2*(x2*(2.0*x7+x4-x3+x2*(x5-x7-x4))-x7);
#define klein_csa(e,ne) 1.0+1.0/e-1.0/ne;

/* How does anyone read this stuff?? */
void klein_nishina(double e, double r, double *new_e, double *scat_angle)
{
  double t1,t2,t3,t4,t5,t6,t7;
  double tt3,tt4,tt7;
  double xt1,xt2,xt3,xt4,xt5,xt6,xt7;
  double yt1,yt2,yt3,yt4,yt5,yt6,yt7;
  double x;
  double zt5;

  t2 = 1.0/e;
  t4 = 2.0*e+1.0;
  t5 = 1.0/t4;
  t6 = log(t4);
  t3 = klein_f1(e,t2,t5,t6);
  if (e>1.16666667) 
    {
      t7 = 1.65898+t2*(.62537*t2-1.00796);
      tt3 = t7/t3;
      if (r>tt3) 
	{
	  tt4 = (t6-1.20397)/(1.0-tt3);
	  tt7 = .3*exp (tt4*(tt3-r));
	  *new_e = tt7*e*.51108;
	  *scat_angle = klein_csa(e,(tt7*e));
	}
      else
	{
	  xt4 = t7/(3.6333+t2*(5.44444*t2-4.66667));
	  xt7 = .5*t7;
	  xt2 = r/tt3;
	  xt3 = 2.1;
	  xt5 = 1.4;
	  x = klein_f2(xt2,xt3,xt4,xt5,xt7);
	  *new_e = e*x*.51108;
	  *scat_angle = klein_csa(e,(e*x));
	}
    }
  else
    {
      yt4 = t3/(t4+t5);
      yt7 = .5*t3;
      yt2 = r;
      yt5 = 1.0-t5;
      yt3 = 3.0*yt5;
      zt5 = 2.0*yt5;
      x = klein_f2(yt2,yt3,yt4,zt5,yt7);
      *new_e = e*x*.51108;
      *scat_angle = klein_csa(e,(e*x))
    };
}

/* This is some really wacked out random number generator */
void rotas(double u, double v, double w,
	   double scat_angle, double rand,
	   double *nu, double *nv, double *nw
	   )
{
  double rs = 2.0;
  double r1 = 2.0;
  double r2 = 2.0;
  double rsq,t1,t2;
  double r,ww;

  /* Who knows how long this loop runs ... */
  while ((rs>1.0) || (rs<1.0e-9))
    {
      double rand1,rand2,rand3;
      
      grand(rand, &rand1, &rand2);
      grand(rand2, &rand, &rand3);
		    
      rs = r1*r1 + r2*r2;
      r1 = 2.0*rand1-1.0;
      r2 = 2.0*rand3-1.0;
    }

  rsq = rs;
  t1 = r1;
  t2 = r2;

  r = sqrt ((1.0-scat_angle*scat_angle)/rsq);
  ww = 1.0-w*w;

  if (ww<1.0e-9)
    {
      *nu = t1*r;
      *nv = t2*r;
      *nw = w*scat_angle;
    }
  else
    {
      double s = sqrt(ww);
      
      *nu = u*scat_angle + (t1*r*u*w-t2*r*v)/s,
      *nv = v*scat_angle + (t1*r*v*w-t2*r*u)/s,
      *nw = w*scat_angle-t1*r*s;
    }
}

/* Handle Compton Scattering Effect */
void compton(struct Sparticle *particle,double d_coll)
{
  struct Sparticle *new_particle = alloc_particle();
  double new_seed,rand2,rand3,rand4;
  double new_e;
  double scat_angle;

  grand(particle->seed, &new_seed, &rand2);
  grand(rand2, &rand3, &rand4);

  klein_nishina((1.956917*particle->e), rand3, &new_e, &scat_angle);

  if (new_e < 1.0e-3)
    bump_counter(ECUT_COUNTER);
  else
    {
      /* Move the particle */
      translate_particle(particle->x, particle->y, particle->z,
			 particle->u, particle->v, particle->w, d_coll,
			 &new_particle->x, &new_particle->y, &new_particle->z
			 );
      /* calculate the new velocity */
      rotas(particle->u, particle->v, particle->w,
	    scat_angle, rand4,
	    &new_particle->u, &new_particle->v, &new_particle->w
	    );
      new_particle->wt    = particle->wt;
      new_particle->e     = new_e;
      new_particle->e_bin = find_energy_bin(new_e);
      new_particle->cell  = particle->cell;
      new_particle->seed  = new_seed;
      set_particle_probabilities(new_particle);

      transport_particle(new_particle);
    }
  
  dealloc_particle(new_particle);
}

/* This is some wacky random number thingy */
void isotropic(double rand, double *nx, double *ny, double *nz)
{
  double rs=2.0;
  double r1=2.0;
  double r2=2.0;
  double randi=rand;
  double rsq, t1, t2, t3, rand0,rand2,rand3,rand4;

  /* This looks like a really bad way to generate random numbers */
  while (rs>1.0)
    {
      grand(randi, &rand0, &rand2);
      grand(rand2, &rand3, &rand4);
      rs = r1*r1 + r2*r2;
      r1 = 2.0*rand4-1.0;
      r2 = 2.0*rand3-1.0;
      randi=rand0;
    }

  rsq = rs;
  t1 = r1;
  t2 = r2;

  *nx = 2.0*rsq-1.0;
  t3 = sqrt((1.0-(*nx)*(*nx))/rsq);
  *ny = t1*t3;
  *nz = t2*t3;
}

void pair_prod(struct Sparticle *particle, double d_coll)
{
  struct Sparticle *new_particle = alloc_particle();
  /* I don't have any clue where they got this new_e from ... */
  double new_e = 0.51008; 
  double new_seed, rand2, rand3, rand4;

  grand(particle->seed, &new_seed, &rand2);
  grand(rand2, &rand3, &rand4);

  translate_particle(particle->x,particle->y,particle->z,
		     particle->u,particle->v,particle->w,d_coll,
		     &new_particle->x, &new_particle->y, &new_particle->z
		     );
  isotropic(rand3, &new_particle->u, &new_particle->v, &new_particle->w);

  new_particle->e     = new_e;
  new_particle->e_bin = find_energy_bin(new_e);
  /* I thought that this weight halves, and in Russian Roulette
     should double ... */
  new_particle->wt    = 2.0*particle->wt;
  new_particle->cell  = particle->cell;
  new_particle->seed  = new_seed;
  
  set_particle_probabilities(new_particle);

  /* Send it on its way ... */
  transport_particle(new_particle);
  dealloc_particle(new_particle);
}

void transport_particle(struct Sparticle *particle)
{
  double distance_to_surface;
  int surface_number;
  double rand,rand1,d_coll;
  double pnphoto = particle->ptotal - particle->pphoto;
  
  dist_to_surface(particle->x,particle->y,particle->z,
		  particle->u,particle->v,particle->w,
		  &distance_to_surface, &surface_number
		  );
  grand(particle->seed, &rand, &rand1);
  d_coll = dist_to_collision(particle->ptotal,rand);

  /* At this point, the particle either collides with something
     and changes direction, or else it hits a surface and we
     tally its energy count and bump the counter for that surface */     
  if (d_coll >= distance_to_surface)
    {
      switch (surface_number)
	{
	case 1:
	  tally(particle->wt,bscatter_profile,particle->e_bin);
	  bump_counter(BSCAT_COUNTER);
	  bump_counter(NOCOL_COUNTER);
	  break;
	case 2:
	  tally(particle->wt,escape_profile,particle->e_bin);
	  bump_counter(ESC_COUNTER);
	  bump_counter(NOCOL_COUNTER);
	  break;
	case 11:
	  tally(particle->wt,transit_profile,particle->e_bin);
	  bump_counter(TRANS_COUNTER);
	  bump_counter(NOCOL_COUNTER);
	  break;
	default:
	  if (particle->cell != (surface_number - 2))
	    russian_roulette(particle, distance_to_surface);
	  else
	    splitting(particle, distance_to_surface);
	  bump_counter(NOCOL_COUNTER);
	}
    }
  else
    /* Hits an electron and does something ... */
    {
      struct Sparticle *new_particle;
      int kill;
      double p_compton = particle->pcompton/pnphoto;
      
      /* photo_elect creates a new particle unless kill
	 is true ... */
      kill = photo_elect(particle,d_coll,particle->pphoto,
			 particle->ptotal, &new_particle);

      if (kill == TRUE)
	bump_counter(WTCUT_COUNTER);
      else
	{
	  if (rand1 < p_compton)
	    compton(new_particle, d_coll);
	  else
	    pair_prod(new_particle, d_coll);
	  dealloc_particle(new_particle);
	}
      bump_counter(COL_COUNTER);
    } 
}

struct Sparticle * create_initial_particle(double seed, double e)
{
  struct Sparticle *new_particle = alloc_particle();

  /* Stick it at the center of the base of the cylinder */
  new_particle->x=0.0;
  new_particle->y=0.0;
  new_particle->z=0.0;

  /* Aim it at the other end, right down the axis of the cylinder */
  new_particle->u=0.0;
  new_particle->v=1.0;
  new_particle->w=0.0;

  /* Stick in the weight, energy, and the random seed for the particle */
  new_particle->wt=1.0;
  new_particle->e=e;
  new_particle->e_bin = find_energy_bin(e);
  new_particle->cell=1;
  new_particle->seed=seed;
  
  return new_particle;
}

inline void trans_source(double rnd, double e0)
{
  struct Sparticle *particle;

  particle = create_initial_particle(rnd, e0);
  set_particle_probabilities(particle);
  transport_particle(particle);
  dealloc_particle(particle);
}

void gamteb_main(int n_sources, double seed, double e0)
{
  int i;

  for (i=0; i<n_sources; i++)
    {
      double rnd, nextseed;
      
      grand(seed, &nextseed, &rnd);
      trans_source(rnd, e0);
      seed = nextseed;
    }
}

void main (int argc, char *argv[])
{
  int i;
  int n;
  double seed = .5;
  double initial_energy = 6.0;

  if (argc != 3)
    {
      printf ("Usage: %s nparticles [print?]\n", argv[0]);
      exit(1);
    }
  
  n = atoi(argv[1]);

  for (i=0; i<=35; i++)
    {
      xcompton_table[i] = bounded_log10(xc[i]);
      xpair_table[i]    = bounded_log10(xp[i]);
      xphoto_table[i]   = bounded_log10(xpe[i]);
      ergs_table[i]     = bounded_ln(energies[i]);
    }

  /* Do the Work */
  gamteb_main(n, seed, initial_energy);

  if (atoi(argv[2]) != 0) {
    /* Print the Answers */
    for (i=0;i<=8;i++)
      printf("%d ",counters[i]);
    printf("\n");

    printf("escape\n");
    print_histogram(escape_profile,10.0);
    printf("\ntransit\n");
    print_histogram(transit_profile,10.0);
    printf("\nback scatter\n");
    print_histogram(bscatter_profile,10.0);
    printf("\n");
    printf("*** LEFTOVER PARTICLES: %d\n", alloced);
  }
}
