#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include "fvector.h"
  
/*
  Barnes-Hut -- Based on Xiao-Wei Shen's Id code.
  Andy Shaw
  17 November 1994
  */

#define SDIM    8
#define THETA   0.6
#define EPSILON 0.05
#define DTIME   0.025 
  
/* Definition of a particle */
typedef struct {
  float   bmass;
  fvector bpos;
  fvector bvel;
  fvector bacc;
} Tbody;

/* Definition of the oct-tree internal node */
typedef struct {
  fvector cbase;
  float   csize;
  float   cmass;
  fvector cpos;
  struct _tree *cnext[SDIM];
} Tcell;

/* Tags to describe oct-trees */
#define EMPTYTREE 0

#define CELL      1
#define BODY      2

/* Tagged union of an internal node and particle and empty */
typedef struct _tree {
  int type;
  union {
    Tbody *tbody;
    Tcell tcell;
  } entry;
} Ttree;

int allocs = 0;
int freelist = 0;
int mallocs = 0;

int *tree_free_list = 0;

inline Ttree *alloc_tree_node()
{
  allocs++;
  if (tree_free_list != NULL)
    {
      Ttree *node = (Ttree *) tree_free_list;
      tree_free_list = (int *) *tree_free_list;

/*
      freelist--;
      printf("Returning from free list %d\n", freelist);
*/
      return node;
    }
  else
    {
      char *ptr;

/*
      mallocs++;
      printf("About to Malloc %d\n", mallocs);
*/
      ptr = malloc(sizeof(Ttree));
      
      if (ptr == NULL)
	{
	  printf("Ran out of memory in malloc, alloced %d nodes\n", allocs);
	  exit(1);
	}
      return ((Ttree *) ptr);
    }
}

inline void dealloc_tree_node(Ttree *node)
{
  *((int **) node) = tree_free_list;
  tree_free_list = (int *) node;
  allocs--;
  freelist++;
}

int nbodies;

inline float MIN (float a, float b)
{
  if (a < b)
    return a;
  else
    return b;
}

inline float MAX (float a, float b)
{
  if (a > b)
    return a;
  else
    return b;
}

int print_answer = 1;

void print_particles(Tbody particles[])
{
  int i;

  for (i=0; i<nbodies; i++)
    {
      fvector position = particles[i].bpos;
      fvector velocity = particles[i].bvel;
      printf("particle[%3d]: pos = %f %f %f\n", i, position.x, position.y, position.z);
      printf("             : vel = %f %f %f\n", velocity.x, velocity.y, velocity.z);
    }
}

inline void print_n_spaces(int n)
{
  int i;
  for (i=0; i<n; i++) printf(" ");
}

void print_tree(Ttree *tree, int indent)
{
  int i;
  print_n_spaces(indent);
  if (tree==EMPTYTREE)
    printf("EMPTY\n");
  else
    switch (tree->type)
      {
      case BODY:
	printf("BODY mass=%f bpos= ", tree->entry.tbody->bmass);
	VPRINT(&tree->entry.tbody->bpos);
	printf("\n");
	break;
      case CELL:
	printf("CELL mass=%f cpos= ", tree->entry.tcell.cmass);
	VPRINT(&tree->entry.tcell.cpos);
	printf("\n");
	for (i=0; i<SDIM; i++)
	  print_tree(tree->entry.tcell.cnext[i], indent+4);
	break;
      }
}

/* This is the inner loop, so it has to be fast ... */
void Gravitation (fvector *pos1, fvector *pos2, float mass2, fvector *dest)
{
  float xdif = pos2->x - pos1->x;
  float ydif = pos2->y - pos1->y;
  float zdif = pos2->z - pos1->z;
  float distance = sqrt(xdif*xdif+ydif*ydif+zdif*zdif);

/*
  printf("Gravitation: Mass = %f\n", mass2);
  printf("           : v1 = ");
  VPRINT(pos1);
  printf("\n");
  printf("           : v2 = ");
  VPRINT(pos2);
  printf("\n");
*/
  
  if (distance > EPSILON)
    {
      float scaling = (mass2/(distance*distance*distance));

      dest->x = scaling * xdif;
      dest->y = scaling * ydif;
      dest->z = scaling * zdif;
    }
  else
    VINIT(dest, 0.0);
  
/*
  printf("           : answer = ");
  VPRINT(dest);
  printf("\n");
*/
}

void AdvanceNbody (Tbody nbody_tab[], Tbody new_nbody_tab[])
{
  int i;
  
  for (i=0; i<nbodies; i++)
    {
      fvector delta_pos;
      fvector delta_vel;

      /* copy over the mass */
      new_nbody_tab[i].bmass = nbody_tab[i].bmass;
      
      /* compute new position */
      VMUL(&nbody_tab[i].bvel, DTIME, &delta_pos);
      VADD(&nbody_tab[i].bpos, &delta_pos, &new_nbody_tab[i].bpos);
      
      /* compute new velocity */
      VMUL(&nbody_tab[i].bacc, DTIME, &delta_vel);
      VADD(&nbody_tab[i].bvel, &delta_vel, &new_nbody_tab[i].bvel);
    }
}

void InitSimple (Tbody nbody_tab[])
{
  int side = 0;
  int i;
  
  /* Initialize side to be in a cube of minimal size to fit all
     the particles */
  while ((side*side*side) < nbodies)
    side++;
  
  for (i=0; i<nbodies; i++)
    {
      /* First, initialize the mass */
      nbody_tab[i].bmass = 1.0;
      /* Next, initialize the position */
      nbody_tab[i].bpos.x = (float) (i%side);
      nbody_tab[i].bpos.y = (float) ((i/side)%side);
      nbody_tab[i].bpos.z = (float) ((i/(side*side))%side);
      /* Finally, initialize the velocity */
      VINIT(&nbody_tab[i].bvel, 1.0);
    }
}

/* Is the particle far enough from this cell to use the approximation? */
inline int FarEnough(fvector *body_pos, fvector *cell_pos, float cell_size)
{
  float distance = VDISTANCE(body_pos, cell_pos);

  return ((distance * THETA) > cell_size);
}

/* Acceleration of one body */
void BodyAcceleration(Tbody *particle, Ttree *BH_tree, fvector *answer_acc)
{
  switch (BH_tree->type)
    {
    case BODY:
      {
	Tbody *body = BH_tree->entry.tbody;

	Gravitation(&particle->bpos, &body->bpos, body->bmass, answer_acc);
      }
      break;
    case CELL:
      {
	Tcell *cell = &BH_tree->entry.tcell;

	if (FarEnough(&particle->bpos, &cell->cpos, cell->csize))
	  {
	    Gravitation(&particle->bpos, &cell->cpos, cell->cmass, answer_acc);
	  }
	else
	  {
	    int i;
	    Ttree **subtrees = cell->cnext;

	    VINIT(answer_acc, 0.0);
	    for (i=0; i<SDIM; i++)
	      {
		Ttree *subtree = subtrees[i];
		
		if (subtree != EMPTYTREE)
		  {
		    fvector temp;

		    BodyAcceleration(particle, subtree, &temp);
		    VADD(answer_acc, &temp, answer_acc);
		  }
	      }
	  }
      }
      break;
    default:
      printf("should never get here\n");
      exit(1);
    }
}

void ComputeAcceleration(Tbody *nbody_tab, Ttree *BH_tree)
{
  int i;
  
  for (i=0; i<nbodies; i++)
    {
      BodyAcceleration(&nbody_tab[i],
		       BH_tree, &nbody_tab[i].bacc);
/*
      printf("acceleration: ");
      VPRINT(&nbody_tab[i].bacc);
      printf("\n\n");
*/
    }
}

int SubIndex(fvector *base, float size, fvector *pos, fvector *corner)
{
  int index = 0;
  int p = 1;
  float half = size/2.0;

  VCOPY(base, corner);
  if (pos->x >= (base->x + half))
    {
      index += 1;
      corner->x = (base->x + half);
    }
  
  if (pos->y >= (base->y + half))
    {
      index += 2;
      corner->y = (base->y + half);
    }
    
  if (pos->z >= (base->z + half))
    {
      index += 4;
      corner->z = (base->z + half);
    }
  return index;
}

void ComputeMassCenter(Ttree *BH_tree)
{
  Tcell *cell = &BH_tree->entry.tcell;
  Ttree **subtrees = cell->cnext;
  int i;
  float Mass = 0.0;
  fvector Center;

  VINIT(&Center, 0.0);

  for (i=0; i<SDIM; i++)
    {
      Ttree *subtree = subtrees[i];
      if (subtree != EMPTYTREE)
	switch (subtree->type)
	  {
	  case BODY:
	    {
	      Tbody *body = subtree->entry.tbody;
	      fvector weighted;
	    
	      Mass += body->bmass;
	      VMUL(&body->bpos, body->bmass, &weighted);
	      VADD(&Center, &weighted, &Center);
	    }
	    break;
	  case CELL:
	    {
	      Tcell *subcell = &subtree->entry.tcell;
	      fvector weighted;
	    
	      ComputeMassCenter(subtree);
	      Mass += subcell->cmass;
	      VMUL(&subcell->cpos, subcell->cmass, &weighted);
	      VADD(&Center, &weighted, &Center);
	    }
	    break;
	  }
    }

  cell->cmass = Mass;
  VDIV(&Center, Mass, &cell->cpos);
}

void LoadBody(Ttree *BH_tree, Tbody *particle)
{
  int index;
  fvector base;
  Tcell *cell = &BH_tree->entry.tcell;
  Ttree *subtree;

  index = SubIndex(&cell->cbase, cell->csize, &particle->bpos, &base);

  subtree = cell->cnext[index];
  if (subtree == EMPTYTREE)
    {
      Ttree *newtree = alloc_tree_node();
      /* Change this from EMPTY to a particle */
      newtree->type = BODY;
      newtree->entry.tbody = particle;
      cell->cnext[index] = newtree;
    }
  else
    switch (subtree->type)
      {
      case CELL:
	{
	  /* Insert the particle recursively into the subtree */
	  LoadBody(subtree, particle);
	}
	break;
      case BODY:
	{
	  /* Create a new subtree, and insert it between this
	     level of the tree and the leaves of the tree */
	  Tbody *thisbody = subtree->entry.tbody;
	  float newsize = cell->csize/2.0;
	  fvector temp;
	  int subindex;
	  Ttree *newtree = alloc_tree_node();
	  Tcell *newcell = &newtree->entry.tcell;
	  int i;

	  newtree->type = CELL;
	
	  subindex = SubIndex(&base, newsize, &thisbody->bpos, &temp);

	  /* Copy over information for cell */
	  VCOPY(&base, &newcell->cbase);
	  newcell->csize = newsize;
	  for (i=0; i<SDIM; i++)
	    newcell->cnext[i] = EMPTYTREE;
	  newcell->cnext[subindex] = subtree;

/*
	  printf("Splitting Tree, base is: ");
	  VPRINT(&base);
	  printf(", size is %f\n", newsize);
*/	  
	  LoadBody(newtree, particle);
	  cell->cnext[index] = newtree;
	}
	break;      
      }
}

/* Find the bounding box for the particles */ 
float SetBound(Tbody *nbody_tab, fvector *base)
{
  fvector max, min;
  int i;
  float length = 0.0;
  
  VINIT(&max, -1e99);
  VINIT(&min,  1e99);
  for (i=0; i<nbodies; i++)
    {
      fvector *position = &nbody_tab[i].bpos;

      min.x = MIN(min.x, position->x);
      min.y = MIN(min.y, position->y);
      min.z = MIN(min.z, position->z);

      max.x = MAX(max.x, position->x);
      max.y = MAX(max.y, position->y);
      max.z = MAX(max.z, position->z);
    }

  for (i=0; i<nbodies; i++)
    {
      fvector span;
      VSUB(&max, &min, &span);
      length = MAX(length,
		   MAX(span.x,
		       MAX(span.y, span.z)));
    }
  base->x = min.x;
  base->y = min.y;
  base->z = min.z;
/*
  printf("Length is %f, Base is:", length);
  VPRINT(base);
  printf("\n");
*/
  return length;
}

Ttree *MakeTree(Tbody *nbody_tab)
{
  fvector base;
  Ttree *BH_tree = alloc_tree_node();
  Tcell *rootcell = &BH_tree->entry.tcell;
  int i;

  BH_tree->type = CELL;
  rootcell->csize = SetBound(nbody_tab, &rootcell->cbase);

  /* Initialize with empty leaves */
  for (i=0; i<SDIM; i++)
    BH_tree->entry.tcell.cnext[i] = EMPTYTREE;

  /* Insert each particle into the tree */
  for (i=0; i<nbodies; i++)
    LoadBody(BH_tree, &nbody_tab[i]);
  
  return BH_tree;
}

void DeallocateTree(Ttree *BH_tree)
{
  int i;
  switch (BH_tree->type)
    {
    case BODY:
      dealloc_tree_node(BH_tree);
      break;
    case CELL:
      for (i=0; i<SDIM; i++)
	{
	  Ttree *subtree = BH_tree->entry.tcell.cnext[i];
	  if (subtree != EMPTYTREE)
	    DeallocateTree(subtree);
	}
      break;
    }
}

void StepSystem (Tbody *nbody_tab, Tbody *new_nbody_tab)
{
  Ttree *BH_tree;

  BH_tree = MakeTree(nbody_tab);
  
  ComputeMassCenter(BH_tree);
  ComputeAcceleration(nbody_tab, BH_tree);

  DeallocateTree(BH_tree);

  AdvanceNbody(nbody_tab, new_nbody_tab);

  /* Here, we can free the BH_tree */
}

Tbody *Simulation (int n, int ntime)
{
  Tbody *nbody_tab;
  Tbody *new_nbody_tab;
  Tbody *temp;
  int i;
  
  nbodies = n;
  
  nbody_tab = (Tbody *) malloc(nbodies*sizeof(Tbody));
  new_nbody_tab = (Tbody *) malloc(nbodies*sizeof(Tbody));

  InitSimple(nbody_tab);
  for (i=0; i<ntime; i++)
    {
      StepSystem(nbody_tab, new_nbody_tab);

      /* Swap the old and new nbody table ... */
      temp = nbody_tab;
      nbody_tab = new_nbody_tab;
      new_nbody_tab = temp;
    }
  return nbody_tab;
}

void main (int argc, char *argv[])
{
  Tbody *answer;
  int n;
  int time;

  if (argc != 4)
    {
      printf ("Usage: %s nparticles ntimesteps [print?]\n", argv[0]);
      exit(1);
    }

  n = atoi(argv[1]);
  time = atoi(argv[2]);

  answer = Simulation(n, time);

  if (atoi(argv[3]) != 0)
    {
      print_particles(answer);
      printf("Alloced %d nodes\n", allocs);
    }
}
