/*************************************************************************
*                                                                        *
*  Name : genlist.c                                                      *
*                                                                        *
*  Purpose : Collects listings of the ADAPTOR translation tool and       *
*            merges their contents to a new listing containing all the   *
*            messages sorted to the line numbers of the related source   *
*            code.                                                       *
*            The name of the generated listing is either the name given  *
*            after -list at call time: genlist -list <messagefilename>   *
*            or - by default - the root of the first found fadapt input  *
*            filename followed by .lst instead of .hpf: The listing of   *
*            file <source.hpf> will be put into file <source.lst> .      *
*                                                                        *
*  exit-codes:                                                           *
*   8: error at open file or close file                                  *
*   9: error due to overflow of some table                               *
*  99: internal error (invalid format of some line)                      *
*                                                                        *
*                                                                        *
*  Author : Resi Hoever-KLier, GMD, I1.HR                                *
*                                                                        *
*  created :     Feb. 1997                                               *
*                                                                        *
*  Last Update :                                                         *
*                                                                        *
*************************************************************************/

# include <stdio.h>   /* stderr */
# include <stdlib.h>
# include <string.h>
# include <limits.h>
# include <time.h>
# include <errno.h>

# include "listsglobal.h"       /* including ratc.h */
# include "global.h"             /* including ratc.h */
 
# ifdef alliant
# include <fcntl.h>             /* for testing fileaccess */
# else
# include <unistd.h>            /* for testing fileaccess */
# endif
 
extern int errno;             /*should be defined in unistd.h*/

/* external variables defined in this module */
 
int interactive = 0;
char dbgfile[] = "dbg";    /* file for DEBUG info */

 
void release_workspace (void);
void error_exit (int);
void coll_error_exit (int);

#ifdef DEBUG
FILE *dbg;
#endif

bool message_file_known;

FILE *extracts;
FILE *sortfile;
FILE *msgfile;
FILE *sourcefile;

long message_start, message_end;
bool msg_started;
bool warning_started;

/* line-no for #MSG at unknown pos. (#MSG will never start in line 0 of a prot-file) */
#define UNKNOWN 0


/*************************************************************************
*                                                                        *
*  int eval_genlist_arg (argc, argv)                                      *
*                                                                        *
*   - evaluates command line arguments                                   *
*   - returns number of not evaluated arguments                          *
*                                                                        *
*************************************************************************/
int eval_genlist_arg (argc, argv)
int argc;
char **argv;
 
{ int i, j, no_arg;
  char *comm;
 
#ifdef DEBUG
  printf ("eval_genlist_arg:\n");
#endif

  no_arg = 0;           /* number of not evaluated arguments */
 
  i=1;
  j = 0;                /* index for input_file_names */
 
  while (i<argc)
    {
      comm = argv[i];
 
       if (strcmp(comm,"-list") == 0)
         {
             if ((argc > (i+1)) && (sscanf(argv[i+1],"%s",message_file_name)))
               { message_file_known = true;
                 i++;
               }
             else
             {
                fprintf(stderr,"argument missing or argument type wrong\n");
                fprintf(stderr,"needed for argument -list\n");
             }
         }
 
      else { /* not recognized option */
             no_arg += 1;
             argv[no_arg] = argv[i];
           }
      i++;
 
    } /*while */
 
    return (no_arg);   /* not evaluated arguments */
 
} /* eval_genlist_arg */
 

/************************************************************************
*                                                                       *
*  int open_prot_files (void)                                           *
*                                                                       *
*  - opens all the protocol files of fadapt                             *
*                                                                       *
*  - returns 0 if                            no error occured           *
*            8 if a prot_file couldn't be opened                        *
*                                                                       *
************************************************************************/
int open_prot_files (void)
 
{
  int i;
  char errormess1[256];

#ifdef DEBUG
  fprintf (dbg,"open_prot_files:\n");
#endif
 
  for (i =0; i < NO_PROT_FILES; i++)
     {
      if ((prot_file_ptr[i] = fopen(prot_file_names[i],"r")) == NULL)
        { sprintf (errormess1,"open_prot_files: Can't open %s\n",prot_file_names[i]);
          strcat(errormessage,errormess1);
          return(8);
        }
#ifdef DEBUG
      else
        {
         fprintf (dbg," >>> OPENED %s\n",prot_file_names[i]);
        }
#endif
     }

  return (0);

} /* int open_prot_files (void) */


/************************************************************************
*                                                                       *
*  int close_prot_files (void)                                          *
*                                                                       *
*  - closes all the protocol files of fadapt                            *
*                                                                       *
*  - returns 0 if                            no error occured           *
*            8 if a prot_file could not be closed                       *
*                                                                       *
************************************************************************/
int close_prot_files (void)
 
{
int i,j;
char errormess1[256];
 
#ifdef DEBUG
  fprintf (dbg,"close_prot_files; NO_PROT_FILES: %d\n",NO_PROT_FILES);
#endif
 
  for (i =0; i < NO_PROT_FILES; i++)
     {
      if ((j = fclose(prot_file_ptr[i])) == EOF)
        { sprintf (errormess1,"close_prot_files: Can't close %s\n",prot_file_names[i]);
          strcat(errormessage,errormess1);
          return (8);
        }
#ifdef DEBUG
      else
        {
         fprintf (dbg," >>> CLOSED %s\n",prot_file_names[i]);
        }
#endif
     }

   return (0);
 
} /* int close_prot_files (void) */
 

/************************************************************************
*                                                                       *
*  bool is_eq_line (line)                                               *
*                                                                       *
*  - tests whether or not line contains only character =                *
*                                                                       *
*  - returns true  if line contains only character =                    *
*            false if line contains any other character or has length 0 *
*                                                                       *
************************************************************************/
bool is_eq_line (line)
char *line;

{
int i;
 
 
  if (strlen(line) == 0) return (false);

  for (i = 0; i < (strlen(line)-1); i++)  /*last char is \n*/
      if (line[i] != '=')   return (false);

  return (true);

} /* is_eq_line */ 

  
/************************************************************************
*                                                                       *
*  bool is_pu_start(line)                                               *
*                                                                       *
*  - tests whether or not line contains the headline of a program unit  *
*                                                                       *
*  - returns true  if line contains the headline of a program unit      *
*            false else                                                 *
*                                                                       *
************************************************************************/
bool is_pu_start(line)
char *line;
 
{
int i;
 
 
  if ((i=strncmp(line,"PROGRAM",7)) == 0) return (true);
  if ((i=strncmp(line,"FUNCTION",8)) == 0) return (true);
  if ((i=strncmp(line,"SUBROUTINE",10)) == 0) return (true);
  if ((i=strncmp(line,"BLOCKDATA",9)) == 0) return (true);
  if ((i=strncmp(line,"MODULE",6)) == 0) return (true);

  return (false);
 
} /* is_pu_start*/
 
 

/************************************************************************
*                                                                       *
*  bool is_message_start(line)                                          *
*                                                                       *
*  - tests whether or not line contains heading of a message            *
*                                                                       *
*  - returns true  if line contains the heading of a message            *
*            false else                                                 *
*                                                                       *
************************************************************************/
bool is_message_start(line)
char *line;
 
{
int i;
 
 
  if ((i=strncmp(line,"#MSG ",5)) == 0) return (true);

  return (false);
 
} /* is_message_start */
 
 
/************************************************************************
*                                                                       *
*  bool is_message_end(line)                                            *
*                                                                       *
*  - tests whether or not line contains trailer of a message            *
*                                                                       *
*  - returns true  if line contains the trailer of a message  	        *
*            false else                                                 *
*                                                                       *
************************************************************************/
bool is_message_end(line)
char *line;
 
{
 
  if (strncmp(line,"#MSGEND",7) == 0) return (true);

  return (false);
 
} /* is_message_end */


/************************************************************************
*                                                                       *
*  bool is_warning_start(line)                                          *
*                                                                       *
*  - tests whether or not line contains heading of a warning            *
*                                                                       *
*  - returns true  if line contains the heading of a warning            *
*            false else                                                 *
*                                                                       *
************************************************************************/
bool is_warning_start(line)
char *line;
 
{
int i;
 
 
  if ((i=strncmp(line,"#WARNING ",9)) == 0) return (true);
 
  return (false);
 
} /* is_warning_start */
 
 
/************************************************************************
*                                                                       *
*  bool is_warning_end(line)                                            *
*                                                                       *
*  - tests whether or not line contains trailer of a warning            *
*                                                                       *
*  - returns true  if line contains the trailer of a warning            *
*            false else                                                 *
*                                                                       *
************************************************************************/
bool is_warning_end(line)
char *line;
 
{
 
  if (strncmp(line,"#WARNINGEND",11) == 0) return (true);
 
  return (false);
 
} /* is_warning_end */



/************************************************************************
*                                                                       *
*  int  complete_actual_message(void)                                   *
*                                                                       *
*  - generates the next entry in file messages if reasonable message got*
*                                                                       *
*  - returns 0 if no error occurred                                     *
*  -         9 if max. no of messages exceeded                          *
*                                                                       *
************************************************************************/
int  complete_actual_message(void)
 
{
char errormess1[256];

 
#ifdef DEBUG
  fprintf (dbg,"\tcomplete_actual_message for prot_f %d, line: %d; pu: %d:\n",act_prot_file,protf_lineno,act_pu);
#endif

  if (act_source_file_line == UNKNOWN)
    {
     msg_started = false;
     warning_started = false;

#ifdef DEBUG
     fprintf (dbg,"completed unknown message (prot_f: %d, lineno: %d; act_pu: %d)\n",act_prot_file,protf_lineno,act_pu);
#endif
     
return (0);
    }


/* got reasonable message */

  if (no_messages > MAX_MESSAGES) 
    { sprintf(errormess1,"complete_actual_message: Max. number of messages already collected.\n");
      strcat(errormessage,errormess1);
      return (9);
    }

#ifdef DEBUG
   fprintf(dbg,"%d %d %d %d %ld %ld (msg_nr:%d, prot_file_lineno:%d)\n",act_pu,act_source_file,act_source_file_line,act_prot_file,message_start,message_end,no_messages,protf_lineno);
#endif

   fprintf(extracts,"%d %d %d %d %ld %ld (msg_nr:%d, prot_file_lineno:%d)\n",act_pu,act_source_file,act_source_file_line,act_prot_file,message_start,message_end,no_messages,protf_lineno);

/*
   messages[no_messages].pu        = act_pu;
   messages[no_messages].file      = act_source_file;
   messages[no_messages].line_nr   = act_source_file_line;
   messages[no_messages].prot_file = act_prot_file;
   messages[no_messages].msg_start = message_start;
   messages[no_messages].msg_end   = message_end;
  
#ifdef DEBUG
   fprintf (dbg,"COMPLETED MESSAGE[%d]: pu:%d; file:%d; line:%d; prot_f:%d (line:%d); start:%d; end:%d\n",no_messages,messages[no_messages].pu,messages[no_messages].file,messages[no_messages].line_nr,messages[no_messages].prot_file,protf_lineno,messages[no_messages].msg_start,messages[no_messages].msg_end);
#endif
*/

   no_messages++;
 
   msg_started = false;
   warning_started = false;

   return (0); 

} /*complete_actual_message */


/*************************************************************************
*                                                                        *
*  char *generate_new_pu_header(header)                                  *
*                                                                        *
*  places header (: already added by get_index_of_unitheader() in array  *
*         generated_pu_heads for later use.                              *
*  - returns pointer to header in array generated_pu_heads               *
*  -         0    if no more space for new pu_header                     *
*                                                                        *
*************************************************************************/
char *generate_new_pu_header (header) 
char * header;
 
{ char *ptr;
  char errormess1[256];

 
#ifdef DEBUG
  fprintf (dbg,"\tgenerate_new_pu_header:\n");
#endif
 
  if ((offset_of_next_pu_head + strlen(header)) > max_size_of_pu_heads)
     { sprintf(errormess1,"generate_new_pu_header: Max. space for pu_headers exceeded.\n");
       strcat(errormessage,errormess1);
       return (0);
     }
 
  ptr = generated_pu_heads + offset_of_next_pu_head;
  strcpy(ptr , header);
  offset_of_next_pu_head += strlen(header);
 
  return (ptr);
 
} /* generate_new_pu_header */


/**************************************************************************
*                                                                         *
*  int  get_index_of_unitheader(line)                                     *
*                                                                         *
*  - inspects arrary of unitheaders (pu_heads) whether line is            *
*             already put on that array.                                  *
*             If not found, generates a new entry for line in pu_heads    *
*                           if possible.                                  *
*  - returns 0  if no error occurred                                      *
*            9  if array pu_heads would overflow                          *
*                                                                         *
**************************************************************************/
int  get_index_of_unitheader(line)
char *line;
 
{
int i,j;
char header[MAX_PU_HEAD_LEN];
char errormess1[256];

 
#ifdef DEBUG
  fprintf (dbg,"\tget_index_of_unitheader for line <%s>:",line); /*\n included*/
#endif

 
/* extract pu_header out of line */

  strcpy(header,line);

  for (i = 0; i < no_pus; i++)
     {
       if ((j = strncmp(pu_heads[i],header,strlen(header))) == 0)
         {
#ifdef DEBUG
  fprintf (dbg,"\tfound in pu_heads[%d]:%s\n",i,header);
#endif
           act_pu = i;
           return (0);
         }
     }

/* line not found in pu_heads: generate new entry for line */

  if (no_pus > MAX_PU_S)
    {sprintf(errormess1,"get_index_of_unitheader: Max. number of pu_headers exeeded.\n");
     strcat(errormessage,errormess1);
     return (9);
    }

  if ((pu_heads[no_pus] = generate_new_pu_header(header)) == 0) return (9);

  act_pu = no_pus;
  no_pus++;

#ifdef DEBUG
fprintf(dbg,">>>generated new entry in pu_heads[%d] for %s\n",no_pus-1,pu_heads[no_pus-1]);
#endif

  return (0);

} /* get_index_of_unitheader */

 
/*************************************************************************
*                                                                        *
*  char *generate_new_source_file_name(source)                           *
*                                                                        *
*  places source in array generated_source_file_names for later use      *
*  - returns pointer to filename in array generated_source_file_names    *
*            0 if no more space for new source_file_name                 *
*                                                                        *
*************************************************************************/
char *generate_new_source_file_name (source)
char * source;
 
{ char *ptr;
char errormess1[256];
 
#ifdef DEBUG
 fprintf (dbg,"\tgenerate_new_source_file_name:\n");
#endif
 
  if ((offset_of_next_source_file_name + strlen(source)) > max_size_of_source_file_names)
     { sprintf(errormess1,"generate_new_source_file_name: Max. size for source file names exceeded.\n");
       strcat(errormessage,errormess1);
       return (0);
     }
 
  ptr = generated_source_file_names + offset_of_next_source_file_name;
  strcpy(ptr , source);
  offset_of_next_source_file_name += strlen(source);

#ifdef DEBUG
  fprintf (dbg,"gen.source.f.names: %s; new source.f.name: %s\n",generated_source_file_names,ptr);
#endif 

  return (ptr);
 
} /* generate_new_source_file_name */

 
/*************************************************************************
*                                                                        *
*  bool  split_filename(filename,main_name,suffix)                       *
*                                                                        *
*   - splits filename into suffix (chars behind last point) and          *
*                          main_name (chars before last point)           *
*   - returns false if no point found in filename                        *
*             true  else                                                 *
*                                                                        *
*************************************************************************/
 
bool  split_filename(filename,main_name,suffix)
  char *main_name, *suffix;
  char *filename;
 
{
  char *last_point;
  int   length,main_len;
 
#ifdef DEBUG
  printf ("split_filename:\n");
#endif
 
  last_point = strrchr(filename,'.');
 
  if (last_point == 0)  return (false);  /* no point in filename */
 
  length = strlen(filename);
  main_len = last_point - filename;
 
  strcpy(suffix,last_point+1);
 
  strcpy (main_name, filename);
  main_name[main_len] = '\0';
 
  return(true);
 
} /*split_filename*/
 
 
/************************************************************************
*                                                                       *
*  void compute_message_file_name(source)                               *
*                                                                       *
*  - if sfile_name has suffix .hpf  comput_message_file_name  generates *
*       message_file_name from the root of source followed by suffix    *
*       .lst  and  sets message_file_known to true.                     *
*                                                                       *
************************************************************************/
void compute_message_file_name(source)
char *source;
 
{
int i;
char main_name[MAXPATHLEN], suffix[MAXPATHLEN];
bool res;
 
#ifdef DEBUG
 fprintf (dbg,"\tcompute_message_file_name: from %s\n",source);
#endif

    if ((res = split_filename(source,main_name,suffix)) == false) return;

#ifdef DEBUG
 fprintf (dbg,"testing suffix >%s<\n",suffix);
#endif
    if ((i = strcmp(suffix,"hpf ")) != 0)
      {
#ifdef DEBUG   
 fprintf (dbg,"\t\t...  suffix >%s< not >hpf <\n",suffix);
#endif
        return;
      }
   
    strcpy(message_file_name,main_name);
    strcat(message_file_name,".lst");

    message_file_known = true;

#ifdef DEBUG
 fprintf (dbg,"\tcomputed message file name: >%s<\n",message_file_name);
#endif
 
    return;

} /* compute_message_file_name */
 
 
/************************************************************************
*                                                                       *
*  int  get_source_file_and_line(line)                                 *
*                                                                       *
*  - extracts name of source file and line the message refers to        *
*             from the first line of the error message looking like:    *
*             #MSG at (LINE = nnn, File = tralala.hpf)    or            *
*             #WARNING at (LINE = nnn, File = tralala.hpf)              *
*    if "#MSG at unknown position" encountered: generates               *
*       UNKNOWN for line-nr and for file-nr                             *
*                                                                       *
*    generates the name of the listfile message_file_name if not yet    *
*              known.                                                   *
*  - returns 0  if no error occurred                                    *
*            9  if table of source_file_names would overflow            *
*           99  if invalid format of message line found.                *
*                                                                       *
************************************************************************/
int  get_source_file_and_line(line)
char *line;
 
{
int i;
char sfile_name[MAXPATHLEN];
char errormess1[256];
 
#ifdef DEBUG
 fprintf (dbg,"\tget_source_file_and_line for prot_f %d, line: %d; pu: %d.\n",act_prot_file,protf_lineno,act_pu);
#endif

/* got MESSAGE AT UNKNOWN POSITION? -> generate valid message line */
  if ((i = strncmp(line,"#MSG at unknown position",24)) == 0)
      strcpy(line,"#MSG at (Line = 0, File = >UNKNOWN<)");


  if ((i = sscanf(line,"#MSG at (Line = %d, File = %s",&act_source_file_line,sfile_name)) != 2)         /* valid msg line? */

    { if ((i = sscanf(line,"#WARNING at (Line = %d, File = %s",&act_source_file_line,sfile_name)) != 2)         /* valid warning line? */

        { sprintf(errormess1,"get_source_file_and_line: Internal error: invalid message line: %s.\n return(sscanf): %d !=2.\n", line,i);
          strcat(errormessage,errormess1);
          return (99);
        }
     }


  sfile_name[strlen(sfile_name)-1] = ' ';      /* eleminate trailing ")" */

/* find position of sfile_name in array source_file_names; if not yet included,
   generate a new entry - if array would not overflow */

 
  for (i = 0; i < no_source_files; i++)
     {
      if (strncmp(source_file_names[i],sfile_name,strlen(sfile_name)) == 0)
        {
          act_source_file = i;
          return (0);
        }
     }
 
  if (no_source_files > MAX_SOURCE_FILE_NAMES)
    { sprintf(errormess1,"get_source_file_and_line: Max. no of source_file_names exeeded.\n");
      strcat(errormessage,errormess1);
      return (9);
    }
 
  if ((source_file_names[no_source_files] = generate_new_source_file_name(sfile_name)) == 0) return (9);

  act_source_file = no_source_files;
  no_source_files++;

  if (message_file_known == false)  compute_message_file_name(sfile_name);


#ifdef DEBUG
fprintf(dbg,">>>generated new entry in source_file_names[%d] for %s\n",no_source_files-1,source_file_names[no_source_files-1]);
#endif

  return (0);
  
} /* get_source_file_and_line */

 
/************************************************************************
*                                                                       *
*  int collect_messages (void)                                          *
*                                                                       *
*  - for each of the protocol files:                                    *
*           for each of the mentioned program units:                    *
*                collects the messages starting with #MSG or #WARNING   *
*                                      or with #<anything>              *
*                         in the array messages                         *
*                         (maybe later on the preceeding messages and   *
*                          the trailing messages too)                   *
*                                                                       *
*  - returns  0 if                            no error occured          *
*             8 if extractsfile cannot be opned                         *
*            99 if error at ftell or fgets                              *
*            returncode  of called routines                             *
*                                                                       *
************************************************************************/
int collect_messages (void)
 
{
int i;
char *ptr;
long act_pos;
char lineact[PROT_FILE_LINE_LEN], lineb[PROT_FILE_LINE_LEN];
char errormess1[256];
 
#ifdef DEBUG
  fprintf (dbg,"\tcollect_messages:\n");
#endif

 
  tmpnam(extractsfile);
 
  if ((extracts = fopen(extractsfile,"w")) == NULL)
    { sprintf(errormess1,"collect_messages: Error at open of extractsfile %s.\n",extractsfile);
      strcat (errormessage,errormess1);
      return (8);
    }
 
  for (act_prot_file = 0; act_prot_file < NO_PROT_FILES; act_prot_file++)
     {
 
#ifdef DEBUG
  fprintf (dbg,">>>NEW PROT_FILE no: %d; name: %s\n",act_prot_file,prot_file_names[act_prot_file]);
#endif
 
      msg_started = false;
      warning_started = false;
      protf_lineno = 0;

      if ((ptr = fgets(lineact,PROT_FILE_LINE_LEN,prot_file_ptr[act_prot_file])) == NULL)
        { sprintf (errormess1,"collect_messages: EOF or error at reading line1 of %s\n",prot_file_names[act_prot_file]);
          strcat (errormessage,errormess1);
          return (99);
        }

      protf_lineno++;

      if ((act_pos = ftell(prot_file_ptr[act_prot_file])) == -1L) /* behind read in line */
        { sprintf(errormess1,"collect_messages: Error at ftell.\n");
          strcat (errormessage,errormess1);
          return (99);
        }
 
      strcpy (lineb,lineact);

 
      while ((ptr = fgets(lineact,PROT_FILE_LINE_LEN,prot_file_ptr[act_prot_file])) != NULL)       /* for each line in prot_file */
        { 
         protf_lineno++;
 
         if ((strlen(lineact) == strlen(lineb)) && is_eq_line(lineact) && is_pu_start(lineb))   /* got  pu_header? */
           {
#ifdef DEBUG
fprintf(dbg,">>> GOT PU_HEAD in line nr. %d: %s",protf_lineno-1,lineb);
#endif
            if (msg_started == true || warning_started == true)
              {
               message_end = act_pos; 
               if ((i = complete_actual_message()) != 0) return (i);
              }
            if ((i = get_index_of_unitheader(lineb)) != 0) return (i);
           }

         else if (is_message_start(lineact))
             {
              if (msg_started == true || warning_started == true)
                {
                 message_end = act_pos;
                 if ((i = complete_actual_message()) != 0) return (i);
                }

              if ((i = get_source_file_and_line(lineact)) != 0) return (i);

/* to avoid header line of msg: */
         if ((act_pos = ftell(prot_file_ptr[act_prot_file])) == -1L)
           { sprintf(errormess1,"collect_messages: Error at ftell\n");
             strcat (errormessage,errormess1);
             return (99);
           }

              message_start = act_pos;
              msg_started = true;
             }

         else if (is_message_end(lineact))
             {
              if (msg_started == false )
                { sprintf(errormess1,"collect_messages: Message end found but no corresponding message start.\n");
                  strcat(errormessage,errormess1);
                  return (99);
                }

              message_end = act_pos;
              if ((i = complete_actual_message()) != 0) return (i);
             }

         else if (is_warning_start(lineact))
             {
              if (warning_started == true || msg_started == true)
                {
                 message_end = act_pos;
                 if ((i = complete_actual_message()) != 0) return (i);
                }
 
              if ((i = get_source_file_and_line(lineact)) != 0) return (i);
 
/* header line of warning must be printed */
 
              message_start = act_pos;
              warning_started = true;
             }
 
         else if (is_warning_end(lineact))
             {
              if (warning_started == false )
                { sprintf(errormess1,"collect_messages: Warning end found but no corresponding warning start.\n");
                  strcat(errormessage,errormess1);
                  return (99);
                }

/* trailer line of warning must be printed */ 
              if ((act_pos = ftell(prot_file_ptr[act_prot_file])) == -1L)
                { sprintf(errormess1,"collect_messages: Error at ftell\n");
                  strcat (errormessage,errormess1);
                  return (99);
                }
 
              message_end = act_pos;
              if ((i = complete_actual_message()) != 0) return (i);
             }

         else   /* any line */
           {
           }

         if ((act_pos = ftell(prot_file_ptr[act_prot_file])) == -1L) /*behind read in line*/
           { sprintf(errormess1,"collect_messages: Error at ftell\n");
             strcat (errormessage,errormess1);
             return (99);
           }


         strcpy (lineb,lineact);

        }/*while*/


/* dont forget last message of act prot file */
     if (msg_started == true || warning_started == true)
       {
        message_end = act_pos;
        if ((i = complete_actual_message()) != 0) return (i);
       }

#ifdef DEBUG
fprintf(dbg,">>>completed handling of prot_file %s after %d lines; %d collected messages up to now\n",prot_file_names[act_prot_file],protf_lineno,no_messages);
#endif

    }/*for*/

  if ((i = fclose (extracts)) != 0)
    { sprintf(errormess1,"collect_messages: Error at fclose extractsfile %s\n",extractsfile);
      strcat(errormessage,errormess1);
      return (8);
    }

  return (0);

} /*int collect_messages (void)*/

 
/******************************************************************************
*                                                                             *
* int sort_messages (void)                                                    *
*                                                                             *
* - sorts extracted messages (in extracts file) due to sort criteria:         *
*         pu nr.                                                              *
*         file nr.                                                            *
*         line nr.                                                            *
*         prot_file nr.                                                       *
*         msg_start (byte_position in prot_file)                              *
*         msg_end (byte_position in prot_file)                                *
*                                                                             *
*   sorts numerically ignoring leading blank characters                       *
*   submitting sort-cmd:                                                      *
*sort -o sortfilename -T TEMPPATH extractsfile -b -n -k 1 -k 2 -k 3 -k 4 -k 5 *
*   if that fails, try old fashioned form:                                    *
*sort -o sortfilename -T TEMPPATH extractsfile -b -n +0 -1 +1 -2 +2 -3 +3 -4  *
*                                                    +4 -5                    *
*                                                                             *
* - returns  0 if                            no error occured                 *
*           99 if error at sort-cmd                                           *
*                                                                             *
******************************************************************************/
int sort_messages (void)
 
{
int i;
char sortcmd[2 * MAXPATHLEN];
char sortcmd1[2 * MAXPATHLEN];
char errormess1[256];
 
#ifdef DEBUG
  fprintf (dbg,"sort_messages:\n");
#endif

  tmpnam(sortfilename);

  strcpy (sortcmd1,"sort -o "); 
  strcat (sortcmd1,sortfilename);
  
  strcpy (sortcmd,sortcmd1);
  strcat (sortcmd," -b -n -k 1 -k 2 -k 3 -k 4 -k 5");
  strcat (sortcmd," ");
  strcat (sortcmd,extractsfile);
 
#ifdef DEBUG
  printf ("submitting sort-comand:\n%s\n",sortcmd);
#endif
  
  if ((i = system(sortcmd)) != 0)
    { 
#ifdef DEBUG
      printf ("sortcmd:\n%s\nreturned error: %d.\nTrying old fashioned form... \n",sortcmd,errno);
#endif
  
      strcpy (sortcmd,sortcmd1);
      strcat (sortcmd," -b -n +0 -1 +1 -2 +2 -3 +3 -4 +4 -5");
      strcat (sortcmd," ");
      strcat (sortcmd,extractsfile);

#ifdef DEBUG
  printf ("submitting sort-comand:\n%s\n",sortcmd);
#endif

      if ((i = system(sortcmd)) != 0)
        { 
         sprintf (errormess1,"sortcmd:\n%s\nreturned error: %d.\n",sortcmd,errno);
         strcat(errormessage,errormess1);
         return (99);
        }
    }
 
#ifdef DEBUG
  fprintf (dbg,"sort-comand completed\n");
#endif
 
  return (0); 
 
} /* sort_messages */

 
/************************************************************************
*                                                                       *
*  int print_source_listings(void)                                      *
*                                                                       *
*  - prints the contents of the adapted source-files with leading       *
*    line numbers.                                                      *
*                                                                       *
*  - returns  0 if                            no error occured          *
*             8 if error at fopen / fclose                              *
*            99 if error at scanning array source_file_names for        *
*                           next source_file_name                       *
*                                                                       *
************************************************************************/
int print_source_listings(void)
 
{
int i;
char *ptr;
char line[PROT_FILE_LINE_LEN];
int infileno;
int lineno;
char errormess1[256];
char sourcefilename[MAXPATHLEN];
 
 
#ifdef DEBUG
  fprintf (dbg,"print_source_listings:\n");
#endif
 

  for (infileno = 0; infileno < no_source_files; infileno++)
     {
      if ((sscanf(source_file_names[infileno],"%s",sourcefilename)) != 1)
        { sprintf(errormess1,"print_source_listings: Error at sscanf of source_file_name[[%d] (%s)\n",infileno,source_file_names[infileno]);
          strcat(errormessage,errormess1);
          return (99);
        }

      if ((i = strcmp(sourcefilename,">UNKNOWN<")) == 0) continue;
 
      if ((sourcefile = fopen(sourcefilename,"r")) == NULL)
        { sprintf(errormess1,"print_source_listings: Error at open of sourcefile[%d] (%s)\n",infileno,sourcefilename);
          strcat(errormessage,errormess1);
          return (8);
        }
 
/*                               generate header line */
      if (infileno == 0)
        fprintf(msgfile,"Listing of source file %s\n<Line> Contents\n\n",sourcefilename);
      else
        fprintf(msgfile,"\fListing of source file %s\n<Line> Contents\n\n",sourcefilename);
 
      lineno = 0;
 
      while ((ptr = fgets(line,MAX_LINE_LENGTH,sourcefile)) != NULL)
        {
         lineno++;
         fprintf(msgfile,"%6d %s",lineno,line); /* line contains \n */
        } /*while*/
 
      if ((i = fclose (sourcefile)) != 0)
        { sprintf(errormess1,"print_source_listings: Error at fclose sourcefile[%d] (%s)\n",infileno,sourcefilename);
          strcat(errormessage,errormess1);
          return (8);
        }
 
     } /*for*/

  return (0);

 
}/*print_source_listings*/


/************************************************************************
*                                                                       *
*  int print_text_of_message(msg_no,prot_file_no,msgstart,msgend)       *
*                                                                       *
*  - prints text of message#msg_no into messagefile                     *
*                                                                       *
*  - returns  0 if                            no error occured          *
*            99 if error at reading / positioning in sort- or protfiles *
*                           or at writing into msgfile                  *
*                                                                       *
************************************************************************/
int print_text_of_message(msg_no,prot_file_no,msgstart,msgend)
int msg_no;
int prot_file_no;
long msgstart,msgend;
 
{
int i;
char *ptr;
long position;
char line[PROT_FILE_LINE_LEN];
char errormess1[256];
 
#ifdef DEBUG
  fprintf (dbg,"print_text_of_message:\n");
#endif
 
  position = msgstart;  /*1rst line of msg text*/
 
  while (position < msgend)
       {
        if ((i = fseek(prot_file_ptr[prot_file_no],position,SEEK_SET)) != 0)
          { sprintf(errormess1,"print_text_of_message: Error %d at fseek %s for pos. %ld.\n",i,prot_file_names[prot_file_no],position);
            strcat(errormessage,errormess1);
            return (99);
          }
 
   
        if ((ptr = fgets(line,PROT_FILE_LINE_LEN,prot_file_ptr[prot_file_no])) == NULL)
          { sprintf(errormess1,"print_text_of_message: Error at fgets msg# %d from prot_file %s pos %ld.\n",msg_no,prot_file_names[prot_file_no],position);
            strcat (errormessage,errormess1);
            return (99);
          }
 
        if ((i = fputs (line,msgfile)) == EOF)
          { sprintf(errormess1,"print_text_of_message: Error at fputs line %s in msgfile %s (msg_no: %d).\n",line,message_file_name,msg_no);
            strcat(errormessage,errormess1);
            return (99);
          }
 
        if ((position = ftell(prot_file_ptr[prot_file_no])) == -1L)
          { sprintf (errormess1,"print_text_of_message: Error at ftell %s (msg_no :%d).\n",prot_file_names[prot_file_no],msg_no);
            strcat(errormessage,errormess1);
            return (99);
          }
 
       } /* while */

  return (0);

} /*print_text_of_message*/

 

/************************************************************************
*                                                                       *
*  int print_messages (void)                                            *
*                                                                       *
*  - prints text of sorted messages in printfile                        *
*                                                                       *
*  - returns  0 if                            no error occured          *
*             8 if error at fopen / fclose                              *
*            99 if error at reading / positioning in sort- or protfiles *
*                           or at writing into msgfile                  *
*                                                                       *
************************************************************************/
int print_messages (void)
 
{
int i;
int msgcnt;
int pu,source_file,source_file_line,prot_file;
long msg_start,msg_end;
char line[PROT_FILE_LINE_LEN];
char errormess1[256];
bool firstfile;
char sourcefilename[MAXPATHLEN];
char puhead[MAXPATHLEN];
char *ptr;
int len;

 
#ifdef DEBUG
  fprintf (dbg,"print_messages:\n");
#endif

  strcpy(sourcefilename,"undef");  /*init; unused*/

  if (message_file_known == false)  strcpy(message_file_name,"fadapt.lst");

  fprintf(stderr,"fadapt listfile: %s\n",message_file_name);

  if ((msgfile = fopen(message_file_name,"w")) == NULL)
    { sprintf(errormess1,"print_messages: Error at open of msgfile %s.\n",message_file_name);
      strcat(errormessage,errormess1);
      return (8);
    }

  if (( i = print_source_listings()) != 0) return (i);

  if ((sortfile = fopen(sortfilename,"r")) == NULL)
    { sprintf(errormess1,"print_messages: Error at open of sortfile %s.\n",sortfilename);
      strcat(errormessage,errormess1);
      return (8);
    }

/* settings for start of listing of messages */
  act_source_file = -1;
  act_pu = -1;
  firstfile = true;

 
#ifdef DEBUG
  fprintf (dbg,">>>print_messages: starting printing of messages.\n");
#endif
 
  for (msgcnt = 0; msgcnt < no_messages; msgcnt++)
     {
      if ((i =  fscanf(sortfile,"%d %d %d %d %ld %ld %*s %*s\n",&pu,&source_file,&source_file_line,&prot_file,&msg_start,&msg_end)) != 6)
       { sprintf(errormess1,"print_messages: Error at reading msg# %d from sortfile %s.\n",msgcnt,sortfilename);
         strcat(errormessage,errormess1);
         return (99);
       }

#ifdef DEBUG
  fprintf (dbg,">>>looking for msg no %d: pu:%d,src_f:%d,src_f_l:%d,prot_f:%d,msg_st:%ld\n",msgcnt,pu,source_file,source_file_line,prot_file,msg_start);
#endif

      if (source_file != act_source_file)         /* sourcefile changed? */
        {
#ifdef DEBUG
  fprintf (dbg,">>>new sourcefile: %d (old:%d)\n",source_file,act_source_file);
#endif

         if (!firstfile)
           {
            if ((i = fclose(sourcefile)) != 0)
              { sprintf (errormess1,"print_messages: Error at fclose sourcefile %d (%s)\n",act_source_file,sourcefilename);
                strcat (errormessage,errormess1);
                return (8);
              }
            else 
              firstfile = false;
           }

         if ((sscanf(source_file_names[source_file],"%s",sourcefilename)) != 1)
           { sprintf(errormess1,"print_messages: Error at sscanf2 of source_file_names[%d] (%s)\n",source_file,source_file_names[source_file]);
             strcat(errormessage,errormess1);
             return (99);
           }
 
         if ((sourcefile = fopen(sourcefilename,"r")) == NULL)
           { sprintf(errormess1,"print_messages: Error at reopen of sourcefile[%d] (%s)\n",source_file,sourcefilename);
             strcat(errormessage,errormess1);
             return (8);
           }

#ifdef DEBUG
  fprintf (dbg,">>>opened src_f: %d: %s\n",source_file,sourcefilename);
#endif

          act_source_file_line = 0;         /*s_f_lines count from 1*/
         } /*source_file != act_source_file */


      if (pu != act_pu)
        {                        /* get pu name: from pu_heads[pu] until \n */
          if ((ptr = strchr(pu_heads[pu],'\n')) == NULL)
            { sprintf(errormess1,"print_messages: Error at strchr(pu_heads[%d],nl)\n",pu);
              strcat(errormessage,errormess1);
              return (99);
            }

          strcpy(puhead,"");
          len = ptr - pu_heads[pu];
          strncat(puhead,pu_heads[pu],len);

#ifdef DEBUG
  fprintf (dbg,"\t>>>got pu_name:>%s<(len:%d; ptr:%d,pu_heads[pu]:%d)\n",puhead,len,ptr,pu_heads[pu]);
#endif
  
         } /* pu != act_pu */


      if ((source_file != act_source_file) || (pu != act_pu)) /*new page req.*/
        {
         fprintf(msgfile,"\fMessages for %s in sourcefile %s\n",puhead,sourcefilename);
#ifdef DEBUG
fprintf(dbg,"\t>>>Messages for %s in sourcefile %s\n",puhead,sourcefilename);
#endif

         act_source_file = source_file;
         act_pu = pu;      
        }


      if (source_file_line != act_source_file_line)/*src_file_line changed?*/
        {                       /* get source_file_line for actual message */
#ifdef DEBUG
  fprintf (dbg,"   \tmoving from line %d to line %d in src_file %s\n",act_source_file_line,source_file_line,sourcefilename);
#endif

         while ( act_source_file_line < source_file_line )
              {
               if ((ptr = fgets(line,MAX_LINE_LENGTH,sourcefile)) == NULL)
                 { sprintf(errormess1,"print_messages: Error at fgets line %d of source file %d (%s) - looking for line %d\n",act_source_file_line,source_file,sourcefilename,source_file_line);
                   strcat(errormessage,errormess1);
                   perror(strerror(errno));
                   return (99);
                 }
               act_source_file_line++;
              }
#ifdef DEBUG
  fprintf (dbg,"   \tgot line %d in src_file %s: %s",act_source_file_line,sourcefilename,line);
#endif

        fprintf(msgfile,"\n%6d %s",act_source_file_line,line); /*\n included*/
       }
 
      if ((i= print_text_of_message(msgcnt,prot_file,msg_start,msg_end)) != 0) 
         return (i);


    }/*for*/  /* message completed ; perform new one */


   if ((i = fclose(msgfile)) == EOF)
     { sprintf(errormess1,"print_messages: Error at fclose of msgfile %s\n",message_file_name);
       strcat(errormessage,errormess1);
       return (8);
     }

  return (0);

 
} /* print_messages */

 
/************************************************************************
*                                                                       *
*  void create_workspace (void)                                         *
*                                                                       *
*  - creates workspace for                                              *
*           array of messages   --- by now done statically ---          *
*           array of names of adapted source files                      *
*           array of headers of adapted pu's                            *
*                                                                       *
************************************************************************/
 
void create_workspace (void)
 
{
#ifdef DEBUG
  fprintf (dbg,"create_workspace:\n");
#endif
 

/* series of pointers to names placed in the arrays below */
 
  source_file_names = (char **) malloc (sizeof(char*) * MAX_SOURCE_FILE_NAMES);
  no_source_files = 0;

  pu_heads            = (char **) malloc (sizeof(char*) * MAX_PU_S);
  no_pus = 0;

/* space for generated names */

  generated_source_file_names = (char *) malloc(max_size_of_source_file_names);
  offset_of_next_source_file_name = 0;

  generated_pu_heads  = (char *) malloc (max_size_of_pu_heads);
  offset_of_next_pu_head = 0;

/* possible workspace for messages: (by now created statically)
  messages = (msg *) malloc (sizeof(msg) * ((NO_PROT_FILES + 10) \
                                           * NO_LINES_OF_ADAPTED_SOURCE_FILES);
*/
 
} /* create_workspace */
 
 
/************************************************************************
*                                                                       *
*  void release_workspace ()                                            *
*                                                                       *
*  - releases workspace for                                             *
*           array of messages   --- by now done statically ---          *
*           array of names of adapted source files                      *
*           array of headers of adapted pu's                            *
*                                                                       *
************************************************************************/
 
void release_workspace ()
{
 
#ifdef DEBUG
  fprintf (dbg,"release_workspace:\n");
#endif
 
  free (source_file_names);
  free (pu_heads);
  free (generated_source_file_names);
  free (generated_pu_heads);
/*free (messages);*/
 
} /* release_workspace */
 
 
/************************************************************************
*                                                                       *
*  void coll_error_exit (int:number)                                    *
*                                                                       *
*  - prints positional message for phase collect_messages               *
*  - closes prot_files                                                  *
*  - closes dbg and extracts file                                       *
*  - calls error_exit (number)                                          *
*                                                                       *
************************************************************************/
void coll_error_exit (number)
int number;
{
int i;
char errormess1[256];
 
#ifdef DEBUG
  printf ("coll_error_exit:\n");
#endif
 
   sprintf(errormess1,"(At handling prot_file %s (line: %d))\n",prot_file_names[act_prot_file],protf_lineno);

   strcat (errormessage,errormess1);

   fflush (NULL);  /* write out all buffered data */

   close_prot_files();

   if ((i = fclose(extracts)) != 0)
     { sprintf(errormess1,"Unable to close extractsfile %s\n",extractsfile);
       strcat(errormessage,errormess1);
     }

#ifdef DEBUG
   if ((i = fclose (dbg)) != 0)
     { sprintf(errormess1,"Unable to close file dbg %s\n",dbgfile);
       strcat(errormessage,errormess1);
     }
#endif

   error_exit (number);

} /* coll_error_exit */

 
/************************************************************************
*                                                                       *
*  void error_exit (int:number)                                         *
*                                                                       *
*  - releases workspace  and  exits with number number                  *
*                                                                       *
************************************************************************/
 
void error_exit (number)
int number;
{
 
#ifdef DEBUG
  printf ("error_exit:\n");
#endif

   fprintf(stderr,"%s",errormessage);
  
   release_workspace();
   exit (number);
 
} /* error_exit */
 
 
/************************************************************************
*                                                                       *
*           main                                                        *
*               (evtl.spaeter  fadapt source files als input)           *
*                                                                       *
*  exits (0) if no error occurred                                       *
*        (8) if error at open of file                                   *
*        (9) if error due to overflow of some internal table            *
*       (99) if internal error (invalid format of some generated line)  *
*       (80) if error at scanning of input param                        *
*                                                                       *
************************************************************************/
 
main(argc,argv)
int argc;
char **argv;
 
{int i,no_arg;
 int returnvalue;
   
 
#ifdef DEBUG
  printf ("main:\n");
#endif

#ifdef DEBUG
  if ((dbg = fopen(dbgfile,"w")) == 0) 
    { fprintf(stderr," error at open of file dbg. exit(8)\n");
      exit (8);
    }
#endif
  
  strcpy (errormessage,"");   /* clear errormessage */
 
  create_workspace ();

  message_file_known = false;    /* compute it later on */

  if ((no_arg = eval_genlist_arg(argc, argv)) > 0)
 
    { fprintf (stderr, "Unrecognized options\n");
        for (i=1; i<=no_arg; i++)
           fprintf (stderr, "invalid option = %s\n", argv[i]);
        error_exit (80);
    }


  if ((returnvalue = open_prot_files()) != 0)
    {
     printf ("genlist: exit due to error %d in open_prot_files\n",returnvalue);
     error_exit (returnvalue);
    }

  act_prot_file = 0;    /* initial act_prot_file for exit messages */
  protf_lineno = 0;     /* in. act lineno of act prot_file for exit messages */ 
  no_messages = 0;      /* no. of messages already collected */
  no_pus =0;            /* no. of pu's already found */
  no_source_files =0;   /* no. of source_file_names already found */

  if ((returnvalue = collect_messages() ) != 0)
    { 
     printf ("genlist: exit due to error %d in collect_messages\n",returnvalue);
     coll_error_exit (returnvalue);
    }

  if ((returnvalue = sort_messages() ) != 0)
    {
     printf ("genlist: exit due to error %d in sort_messages\n",returnvalue);
     error_exit (returnvalue);
    }

  if ((returnvalue = print_messages() ) != 0)
    {
     printf ("genlist: exit due to error %d in print_messages\n",returnvalue);
     error_exit (returnvalue);
    }

  if ((returnvalue = close_prot_files()) != 0)
    {
     printf ("genlist: exit due to error %d in close_prot_files\n",returnvalue);
     error_exit (returnvalue);
    }
 
  release_workspace ();

#ifndef DEBUG
  if ((i = remove (extractsfile)) != NULL)
    fprintf(stderr,"unable to remove extractsfile %s\n",extractsfile);
  if ((i = remove (sortfilename)) != NULL)
    fprintf(stderr,"unable to remove sortfile %s\n",sortfilename);
#endif

  exit (0);
}

