/*************************************************************************
*                                                                        *
*  Name : translate.c                                                    *
*                                                                        *
*  Purpose : Translation of one file                                     *
*                                                                        *
*  Author : Dr. Thomas Brandes, GMD, SCAI.LAB                            *
*                                                                        *
*  Last Update : Aug  1993                                               *
*                                                                        *
*************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#ifdef WIN32
#include <process.h>       /* getpid           */
#else
#include <unistd.h>        /* getpid           */
#endif

#include "global.h"
#include "protocol.h"      /* protocol_errors  */
#include "translate.h"

#include "MySource.h"
#include "Scanner.h"       /* BeginFile        */
#include "Parser.h"
#include "Tree.h"
#include "MakeDefs.h"      /* SEMANTIC Phase 1 */
#include "Semantic.h"      /* SEMANTIC Phase 2 */
#include "ControlFlow.h"   /* SEMANTIC Phase 3 */
#include "Modules.h"       /* WriteModules     */
#include "Unparse.h"

#include "CallGraphFns.h"
#include "Calling.h"
#include "MakeInterface.h"

#include "ImplMapping.h"     /* PARALLEL Phase 1 */
#include "Normalization.h"   /* PARALLEL Phase 2 */
#include "LooPo.h"           /* PARALLEL Phase 3 */
#include "AnaOverlap.h"      /* PARALLEL Phase 4 */
#include "InterProc.h"       /* PARALLEL Phase 5 */
#include "AdaptAnalysis.h"   /* PARALLEL Phase 6 */

/* include main routines of ADAPTOR PHASEs */

#include "MakeArguments.h"
#include "MakeParLoops.h"
#include "MakeHome.h"
#include "MakeLocal.h"
#include "Classification.h"
#include "Optimization.h"
#include "CodeGeneration.h"
#include "Addressing.h"
#include "FinalCode.h"

/*************************************************************************
*                                                                        *
*  strip_file (infile_name, outfile_name, strip_length)                  *
*                                                                        *
*  Purpose : Splitting Fortran Source programs with longer ines          *
*                                                                        *
*  Author : James Cownie, Meiko, Bristol
*                                                                        *
*  Last Update : June 1993                                               *
*                                                                        *
*                                                                        *
*   The behaviour of this code was not actually safe. It will break      *
*   if it has to split lines with long character string definitions,     *
*   as it is not aware that that is what it is doing.                    *
*                                                                        *
*   It would actually be safer not to try to choose a good place to      *
*   split, but always split at the correct column, and ASSUME that the   *
*   Fortran compiler will re-assemble the lines at the same width.       *
*                                                                        *
*************************************************************************/

#define MAXLEN 200

void strip_file (infile_name, strip_length)

char * infile_name;
int strip_length;

{ char buffer [MAXLEN+100];
  char outfile_name [50];
  FILE *infile, *outfile;
  int c;
  int buflen;
  int error;
   
  /* return; */

  /* get the file name for the outfile */

  sprintf (outfile_name, "xwyvz_%d.tmp", getpid());

  if ((strip_length < 50) || (strip_length > 150))
    { 
      fprintf (stderr, "strip_length must be in range [50 - 150]\n");
      exit (-1);
    }
    
  infile  = fopen (infile_name, "r");
    
  if (infile == NULL)
    { fprintf (stderr, "Input File %s could not be opened\n", infile_name);
      exit (-1);
    } 
    
  /* outfile will be a temporary file */

  outfile = fopen (outfile_name, "w");
    
  if (outfile == NULL)
    { fprintf (stderr, "Output File %s could not be opened\n", outfile_name);
      exit (-1);
    } 
    
  buflen = 0;  /* actual buffer length */
    
  while ((c = fgetc(infile)) != EOF)

  { int i;

    if (c == '\n')

      { /* new line, print current line */

        for (i=0; i<buflen; i++) putc (buffer[i], outfile);
        putc ('\n', outfile);
	buflen = 0;
      }

     else 

      { /* set new character */

        buffer[buflen++] = c;

        if (buflen == strip_length)

           { /* print out buffer */

             for (i=0; i<buflen; i++) putc (buffer[i], outfile);
	     putc ('\n', outfile);

             /* make continuation line, but attention to compiler 
                directives that require same first characters      */

             if (   (buffer[0] == ' ')
                 || ((buffer[0] >= '0') && (buffer[0] <= '9')) )

                { for (i=0;i<5;i++) buffer[i] = ' '; }

             buffer[5] = '&';

             buflen = 6;

           }
     }

  }  /* while not EOF */
    
  /* print rest of the buffer (last line if no \n in input) */

  if (buflen > 0)

    { int i;

      for (i=0; i<buflen; i++) putc (buffer[i], outfile);
      putc ('\n', outfile);

    }
    
  fclose (infile);
  fclose (outfile);

#ifdef WIN32
  sprintf (buffer, "copy %s %s", outfile_name, infile_name);
  error = system (buffer);
  sprintf (buffer, "del %s", outfile_name);
  error = system (buffer);
#else
  sprintf (buffer, "mv -f %s %s", outfile_name, infile_name);
  error = system (buffer);
#endif

  if (error != 0)

    { fprintf (stderr, "ATTENTION: first attempt to replace %s failed\n",
                        infile_name);
      error = system ("sleep 10");
      error = system (buffer);
      if (error != 0)
       { fprintf (stderr, "ERROR: could not replace old file %s after strip\n",
                           infile_name);
          exit (-1);
       }
    }

  if (verbose_flag)

     printf ("Fortran file %s has been generated (strip = %d)\n", 
              infile_name, strip_length);
     
} /* strip_file */

/*************************************************************************
*                                                                        *
*  UnparseTree (filename, root)                                          *
*                                                                        *
*   - unparse the tree 'root' to file with name 'filename'               *
*   - make a strip on the file                                           *
*                                                                        *
*************************************************************************/

void UnparseTree (filename, root)
char  * filename;
tTree root;

{ FILE *f;

  f = fopen (filename, "w");

  if (f == (FILE *) NULL)
   { printf ("Error: could not open file %s for unparsing\n", filename);
     printf ("Please check permissions\n");
     exit (-1);
   }

  FileUnparse (f, root);

  fclose (f);

  strip_file (filename, strip_length);

}  /* UnparseTree */

/*************************************************************************
*                                                                        *
*  int SourceParser (char *file_name)                                    *
*                                                                        *
*    - parses fortran file file_name, returns number of syntax errors    *
*    - -1 if file could not be opened                                    *
*    - syntax tree is globally available in TreeRoot                     *
*                                                                        *
*************************************************************************/
 
int SourceParser (fname)
char *fname;

{  int Errors;

   filename = fname;    /* for makefile write, will know the file name */
   BeginFile (fname);
   Errors = Parser () + SourceFileErrors;
   if (Errors == 0) 
       sprintf (last_message, "File %s succesfully parsed", fname);
     else
       sprintf (last_message, "File %s parsed with %d Errors", fname, Errors);
   return (Errors);
} /* SourceParser */

int SourceCheck ()
{  if (!CheckTree (TreeRoot))
     { sprintf (last_message, "Tree is not okay");
       return 1;
     }
    else
     { sprintf (last_message, "Tree is good");
       return 0;
     }
}

/*************************************************************************
*                                                                        *
*                                                                        *
*                                                                        *
*************************************************************************/

int DoPhase (route, ph_id, ph_name, ph_routine, suffix, print_flag)

char route[];
int  ph_id;
char ph_name [];
void ph_routine ();
char suffix [];
int  print_flag;

{ int errors;
  char filename [30];

  sprintf (filename, "adaptor.%s", suffix);

  if (verbose_flag)

    printf ("%s PHASE %d : %s\n", route, ph_id, ph_name);

  open_protocol (filename);
  ph_routine (TreeRoot);
  close_protocol ();

  errors = protocol_errors ();

  if (errors > 0)

    { sprintf (last_message, "%d Errors in %s %d : %s (see adaptor.%s)",
               errors, route, ph_id, ph_name, suffix);
      return errors;
    }

  if (!CheckTree (TreeRoot))

    { sprintf (last_message, "Illegal Tree after %s %d : %s",
               route, ph_id, ph_name);
      return 1;
    }

  if (debug_flag && print_flag)

     { sprintf (filename, "%s_unparse.f", suffix);
       UnparseTree (filename, TreeRoot);
     }

  return 0;  /* no errors */

}  /* DoPhase */

/*************************************************************************
*                                                                        *
*                                                                        *
*                                                                        *
*************************************************************************/

int SourceSemantic ()

{ int phase;
  int errors;

  phase = 1;
  errors = DoPhase ("SEMANTIC", phase, "MakeDefs", MakeDefs, "def", 0);
  if (errors) goto Ende;

  phase = 2;
  errors = DoPhase ("SEMANTIC", phase, "SemChecks", Semantic, "sem", 0);
  if (errors) goto Ende;

  phase = 3;
  errors = DoPhase ("SEMANTIC", phase, "ControlFlow", ControlFlow, "cf", 1);
  if (errors) goto Ende;

  sprintf (last_message, "Semantic Analysis was successful");

  WriteModules (TreeRoot);

  phase = 0;

Ende: return (phase);

}

void SourceWrite ()

{  FILE *myFile;
   printf ("Write Tree on File test.out \n");
   myFile = fopen ("test.out","w");
   if (myFile == (FILE *) NULL)
    { printf ("Adaptor failed to open file test.out\n");
      printf ("Please check permissions\n");
      exit (-1);
    }
   WriteTree (myFile, TreeRoot);
   fclose (myFile);
   printf ("Writing Tree is ready\n");
   sprintf (last_message, "Abtract Tree written to file test.out");
}

void SourceCalling (filename)
char *filename;

{  BeginCalling ();
   printf ("Starting Calling Analysis \n");
   Calling (TreeRoot); 
   printf ("Ending Calling Analysis\n");

   /* import CGFile from Calling.h */

   printf ("Start Writing CallGraph to file %s\n", filename);

   CGFile = fopen (filename,"w");

   if (CGFile == (FILE *) NULL)
    { printf ("Adaptor failed to open file %s\n", filename);
      printf ("Please check permissions\n");
      exit (-1);
    }
   OutCallGraph (TheCallGraph);
   fclose (CGFile);
   printf ("End Writing CallGraph to File %s\n", filename);
   sprintf (last_message, "Call Graph has been generated");
}

void SourceUnparse ()

{ BeginUnparse ();
  Unparse (TreeRoot);
  CloseUnparse ();
}

int SourceInterface (interface_file)
char * interface_file;

{  int errors;

   if (verbose_flag) printf ("MAKE INTERFACE\n");
   MakeInterface (TreeRoot);

   errors = protocol_errors ();

   if (errors > 0)
     { sprintf (last_message,
          "%d Errors for Interface (see adaptor.intf)", errors);
       goto Ende;
     }
   if (!CheckTree (TreeRoot))
     { sprintf (last_message, "Illegal Tree after Making Interface");
       goto Ende;
     }

   UnparseTree (interface_file, TreeRoot);

   sprintf (last_message, "Interface File %s created\n", interface_file);

   Ende: return (errors);
}

int SourceParallelization ()

{ int phase;
  int errors;

  phase = 1;
  errors = DoPhase ("PARALLELIZATION", phase, "Distribution",
                     ImplMapping, "dist", 0);
  if (errors) goto Ende;

  phase = 2;
  errors = DoPhase ("PARALLELIZATION", phase, "Normalization",
                     Normalization, "normal", 1);
  if (errors) goto Ende;

  phase = 3;
  errors = DoPhase ("PARALLELIZATION", phase, "Loop Parallelization",
                     LooPo, "auto", 0);
  if (errors) goto Ende;

  phase = 4;
  errors = DoPhase ("PARALLELIZATION", phase, "Shadows",
                     OverlapAnalysis, "ovl", 0);
  if (errors) goto Ende;

  phase = 6;
  errors = DoPhase ("PARALLELIZATION", phase, "ADAPTOR Analysis",
                     AdaptAnalysis, "ana");
  if (errors) goto Ende;

  sprintf (last_message, "Parallelization without errors");
  phase = 0;   /* no errors happened */

Ende: return(phase);

} /* SourceParallelization */

/*************************************************************************
*                                                                        *
*                                                                        *
*                                                                        *
*************************************************************************/

int SourceAdapt ()

{ int errors;
  int phase;

  phase = 1;
  errors = DoPhase ("ADAPTOR", phase, "Arguments",
                     MakeArguments, "arg", 1);
  if (errors) goto Ende;

  phase = 2;
  errors = DoPhase ("ADAPTOR", phase, "Make Loops",
                     MakeParLoops, "loop", 1);
  if (errors) goto Ende;

  phase = 3;
  errors = DoPhase ("ADAPTOR", phase, "Find Homes",
                     MakeHome, "home", 1);
  if (errors) goto Ende;

  phase = 4;
  errors = DoPhase ("ADAPTOR", phase, "Extract Communication",
                     MakeLocal, "local", 1);
  if (errors) goto Ende;

  phase = 5;
  errors = DoPhase ("ADAPTOR", phase, "Classification",
                     Classification, "class", 0);
  if (errors) goto Ende;

  phase = 1;
  errors = DoPhase ("OPTIMIZATION", phase, "Intraprocedural",
                     Optimization, "opt", 1);
  if (errors) goto Ende;

  phase = 2;
  errors = DoPhase ("OPTIMIZATION", phase, "Interprocedural Analysis",
                     Interprocedural, "ipa", 0);
  if (errors) goto Ende;

  sprintf (last_message, "Translation without errors");
  phase = 0;   /* no errors happened */

Ende: return(phase);
}

/*************************************************************************
*                                                                        *
*                                                                        *
*                                                                        *
*************************************************************************/

int SourceCodeGen (node_file)
char * node_file;

{ int errors;
  int phase;

  phase = 1;
  errors = DoPhase ("CODEGEN", phase, "Code Generation",
                     NodeAdapt, "code", 1);
  if (errors) goto Ende;

  phase = 2;
  errors = DoPhase ("CODEGEN", phase, "Addressing",
                     Addressing, "addr", 1);
  if (errors) goto Ende;

  phase = 3;
  errors = DoPhase ("CODEGEN", phase, "Final Code",
                     FinalCode, "final", 0);
  if (errors) goto Ende;

  UnparseTree (node_file, TreeRoot);

  sprintf (last_message,
           "CodeGeneration successful, %s generated", node_file);
  phase = 0;

Ende: return(phase);

} /* Code

/*************************************************************************
*                                                                        *
*  translate_file (source_file, target_file)                             *
*                                                                        *
*************************************************************************/

void translate_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  char node_file [10];  /* name of file for node/cube/node1 program */

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (!CheckTree (TreeRoot))
    { sprintf (last_message, "Illegal Tree after Parsing");
      exit(-1);
    }

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceParallelization ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceAdapt ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)

   {  strcpy (node_file, model_srcfile [target_model]);
      ErrorCount = SourceCodeGen (node_file);
   }

   else
 
      ErrorCount = SourceCodeGen (target_filename);

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}

/*************************************************************************
*                                                                        *
*  interface_file (source_file, target_file)                             *
*                                                                        *
*************************************************************************/

void interface_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)
      ErrorCount = SourceInterface ("interface.h");
   else
      ErrorCount = SourceInterface (target_filename);

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}

void callgraph_file (source_filename, target_filename)

char *source_filename;
char *target_filename;

{ int ErrorCount;

  ErrorCount = SourceParser (source_filename);
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  ErrorCount = SourceSemantic ();
 
  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

  if (strcmp (target_filename, "") == 0)
      SourceInterface ("interface.h");
   else
      SourceCalling ("test.call");

  if (ErrorCount > 0)
    { printf ("%s\n", last_message);
      exit (-1);
    }
  else if (verbose_flag)
      printf ("%s\n", last_message);

}
