/*********************************************************************
*                                                                    *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                   *
*  Date        : Dec 91                                              *
*  Last Update : Nov 96                                              *
*                                                                    *
*  Module      : protocol (C source module)                          *
*                                                                    *
*  Function    : write protocol files and error messages             *
*                                                                    *
*  Export :                                                          *
*                                                                    *
*********************************************************************/

#include <stdio.h>

#ifdef WIN32
#include <process.h>   /* getpid                  */
#else
#include <unistd.h>    /* getpid                  */
#endif

#include <stdlib.h>    /* system                  */

#include "global.h"    /* verbose_flag, list_flag */
#include "MySource.h"  /* GetSourceName */
#include "protocol.h" 
#include "Unparse.h"
#include "ShowDefs.h"

#undef DEBUG

static char protocol_name[200];
static int  open_file = 0;           /* flag is rtrue if file is opened */
static char unit_name [100];
static FILE *protocol;
static int error_counter;
static int warning_counter;
static tTree last_stmt;     /* is always ACF_NODE (label, line, ...)
                               or        DECL_NODE (..) */

/*********************************************************************
*                                                                    *
*  void open_protocol (char name[])                                  *
*                                                                    *
*  - opens file 'name' for writing protocol                          *
*                                                                    *
*********************************************************************/

void open_protocol (name)
char name[];

{ if (open_file)
     { fprintf (stderr, "open_protocol: there is already a file opened\n");
       exit (-1);
     }

  if (list_flag)
     strcpy (protocol_name, name);
   else
     sprintf (protocol_name, "%s_%d", name, getpid());

  protocol = fopen (protocol_name,"w");

  if (protocol == (FILE *) NULL)
    { printf ("Adaptor failed to open protocol file %s\n", name);
      printf ("Please check permissions\n");
      exit (-1);
    }

  last_stmt = NoTree;
  error_counter = 0;
  warning_counter = 0;
  open_file = 1;

} /* open_protocol */

/*********************************************************************
*                                                                    *
*  void remove_protocol ()                                           *
*                                                                    *
*  - deletes last protocol file                                      *
*                                                                    *
*********************************************************************/

void remove_protocol ()
{ char command[200];
  int error;
#ifdef WIN32
  sprintf (command, "del %s", protocol_name);
#else
  sprintf (command, "rm -f %s", protocol_name);
#endif
  error = system (command);
  if (error != 0)
   { fprintf (stderr, 
              "Protocol file %s could not be removed\n", protocol_name);
     fprintf (stderr, "ERROR CODE = %d\n", error);
     error = system ("sleep 5");
     if (error != 0)
        fprintf (stderr, "ERROR CODE for sleep = %d\n", error);
     error = system (command);
     if (error == 0)
        fprintf (stderr, 
              "Protocol file %s has now been removed\n", protocol_name);
   }

} /* remove_protocol */

       /*************************************************
       *                                                *
       *  close_protocol ()                             *
       *                                                *
       *************************************************/
 
void close_protocol ()

{ rbool error_flag;

  if (!open_file)
     { fprintf (stderr, "close_protocol: there was no file open\n");
       exit (-1);
     }
 
  if (error_counter + warning_counter == 0)
     { if (list_flag)
         printf ("Protocol written in File %s\n", protocol_name);
     }
  else if (error_counter == 0)
     printf ("Protocol written in File %s (%d warnings)\n", 
              protocol_name, warning_counter);
  else
     printf ("Protocol written in File %s (%d errors)\n", 
              protocol_name, error_counter);
  fclose (protocol);

  open_file = 0;

  error_flag = rfalse;
  if (error_counter > 0) error_flag = rtrue;
  if (warning_counter > 0) error_flag = rtrue;  /* what about -w */

  if (!error_flag && !list_flag) 
     remove_protocol ();
}

       /*************************************************
       *                                                *
       *  protocol_errors ()                            *
       *                                                *
       *************************************************/
 
int protocol_errors ()
{
  return (error_counter);
}

       /*************************************************
       *                                                *
       *  set_protocol_unit (Tree unit)                 *
       *                                                *
       *************************************************/
 
void set_protocol_unit (t)
tTree t;
{ tIdent Name;
  char s[100];
  int i;

  last_stmt = NoTree;    /* make sure there is no statement from last unit */

  sprintf (unit_name, "");

  if (t->Kind == kPROGRAM_DECL)
    { Name = t->PROGRAM_DECL.Ident; 
      GetString (Name, s);
      sprintf (unit_name, "PROGRAM %s", s);
    }
  if (t->Kind == kPROC_DECL)
    { Name = t->PROC_DECL.Ident; 
      GetString (Name, s);
      sprintf (unit_name, "SUBROUTINE %s", s);
    }
  if (t->Kind == kFUNC_DECL)
    { Name = t->FUNC_DECL.Ident; 
      GetString (Name, s);
      sprintf (unit_name, "FUNCTION %s", s);
    }
  if (t->Kind == kMODULE_DECL)
    { Name = t->MODULE_DECL.Ident; 
      GetString (Name, s);
      sprintf (unit_name, "MODULE %s", s);
    }
  if (t->Kind == kBLOCK_DATA_DECL)
    { Name = t->BLOCK_DATA_DECL.Ident; 
      GetString (Name, s);
      sprintf (unit_name, "BLOCK DATA %s", s);
    }
  fprintf (protocol, "%s\n", unit_name);
#ifdef DEBUG
  printf ("protocol of %s\n", unit_name);
#endif
  for (i=0; i < strlen (unit_name); i++)
     fprintf (protocol, "=");
  fprintf (protocol, "\n\n");
}

       /*************************************************
       *                                                *
       *  set_protocol_stmt (t : ACF_NODE,DECL_NODE)    *
       *                                                *
       *************************************************/
 
void set_protocol_stmt (t)
tTree t;
{
  last_stmt = t;
}

int get_protocol_line ()

{ int line;
  line = 0;
  if (Tree_IsType (last_stmt, kACF_NODE)) 
     line = last_stmt->ACF_NODE.Line;
   else if (Tree_IsType (last_stmt, kDECL_NODE))
     line = last_stmt->DECL_NODE.Line;
  return (line);
} /* get_protocol_line */
 
void print_protocol (s)
char s[];
{ fprintf (protocol, "%s\n", s);
#ifdef DEBUG
  printf ("%s\n", s);
#endif
}

void tree_protocol (s, t)
char s[];
tTree t;
{ fprintf (protocol, "%s",s);
  FileUnparse (protocol, t);
  if (Tree_IsType (t, kACF_NODE))
    { /* no new line necessary */ }
   else if (Tree_IsType (t, kDECL_NODE))
    { /* no new line necessary */ }
   else
    fprintf (protocol, "\n");
}

void obj_protocol (s, o)
char s[];
tObject o;
{ fprintf (protocol, "%s\n",s);
  SemFile = protocol;
  ShowDeclarations (o);
  fprintf (protocol, "\n");
}

void simple_error_protocol (s)
char s[];
{ error_counter += 1;
  fprintf (protocol, "ERROR : %s\n",s);
#ifdef DEBUG
  printf ("ERROR : %s\n",s);
#endif
}

void simple_warning_protocol (s)
char s[];
{ if (no_warnings) return;
  warning_counter += 1;
  fprintf (protocol, "WARNING : %s\n",s);
#ifdef DEBUG
  printf ("WARNING : %s\n",s);
#endif
}

void tree_error_protocol (s, t)
char s[];
tTree t;
{ error_counter += 1;
  fprintf (protocol, "ERROR : %s\n",s);
  FileUnparse (protocol, t);
  fprintf (protocol, "\n");
#ifdef DEBUG
  printf ("ERROR : %s\n",s);
  FileUnparse (stdout, t);
  printf ("\n");
#endif
}

void obj_error_protocol (s, o)
char s[];
tObject o;
{ error_counter += 1;
  fprintf (protocol, "ERROR : %s\n",s);
  SemFile = protocol;
  ShowDeclarations (o);
  fprintf (protocol, "\n");
#ifdef DEBUG
  printf ("ERROR : %s\n",s);
  SemFile = stdout;
  ShowDeclarations (o);
  printf ("\n");
#endif
}

void tree_warning_protocol (s, t)
char s[];
tTree t;
{ if (no_warnings) return;
  warning_counter += 1;
  fprintf (protocol, "WARNING : %s",s);
  FileUnparse (protocol, t);
  fprintf (protocol, "\n");
#ifdef DEBUG
  printf ("WARNING : %s",s);
  FileUnparse (stdout, t);
  printf ("\n");
#endif
}

       /******************************************************
       *                                                     *
       *  void out_last_stmt (char kind[], char message [])  *
       *                                                     *
       ******************************************************/

void out_last_stmt (kind, message)

char kind[], message[];

{ int line;

  if (!open_file)
     { fprintf (stderr, "out_last_stmt: there is no protocol file open");
       exit (-1);
     }

  line = get_protocol_line ();

  if (line != 0)

     fprintf (protocol, "%s at (Line = %d, File = %s)\n", 
              kind, DecodeLine(line), DecodeFile(line));

    else

     fprintf (protocol, "%s at unknown position\n", kind);

  if (last_stmt != NoTree)
     FileUnparse (protocol, last_stmt);

  fprintf (protocol, "<%s>\n", message);
  fprintf (protocol, "%sEND\n", kind);

} /* out_last_stmt */

       /*************************************************
       *                                                *
       *  stderr_error_message (message)                *
       *  stderr_waring_message (message)               *
       *  stderr_info_message (message)                 *
       *                                                *
       *************************************************/
 
void stderr_message (message, kind)
char message[];
char kind[];
 
{ int line;
 
  line = get_protocol_line ();
  if (line > 0)
     fprintf (stderr, "%s at line %d (%s) : %s\n", kind,
                      DecodeLine(line), DecodeFile(line), message);
   else
     fprintf (stderr, "%s in unit %s : %s\n", kind, unit_name, message);

} /* stderr_message */

void stderr_warning_message (message)
char message[];
 
{ int line;
 
  line = get_protocol_line ();
  if (line > 0)
     fprintf (stderr, "WARNING at line %d (%s) : %s\n",
                      DecodeLine(line), DecodeFile(line), message);
   else
     fprintf (stderr, "WARNING in unit %s : %s\n", unit_name, message);

} /* stderr_warning_message */

void stderr_info_message (message)
char message[];
 
{ int line;
 
  line = get_protocol_line ();
  if (line > 0)
     fprintf (stderr, "INFO at line %d (%s) : %s\n",
                      DecodeLine(line), DecodeFile(line), message);
   else
     fprintf (stderr, "INFO in unit %s : %s\n", unit_name, message);

} /* stderr_info_message */

void error_protocol (message)
char message[];

{ int line;

  error_counter += 1;
  out_last_stmt ("ERROR", message);

  stderr_message (message, "ERROR");
}

void warning_protocol (message)
char message[];

{ if (no_warnings) return;

  warning_counter += 1;
  out_last_stmt ("WARNING", message);
} /* warning_protocol */

void serious_warning_protocol (message)
char message[];

{ if (no_warnings) return;

  warning_counter += 1;
  out_last_stmt ("#WARNING", message);
  stderr_message (message, "WARNING");
} /* warning_protocol */

void info_protocol (message)
char message[];

{ if (no_messages) return;

  out_last_stmt ("#INFO", message);
  stderr_message (message, "INFO");
} /* info_protocol */

void stmt_protocol (message)
char message[];

{
  out_last_stmt ("#MSG" , message);
} /* stmt_protocol */

void kill_in_protocol ()
{ int a;
  printf ("INTERNAL ERROR, will stop\n");
  close_protocol ();
  a = 5;
  a = a / (a - a);
  exit (-1);
}

/*********************************************************************
*                                                                    *
*  void failure_protocol (char module[], char proc[], Tree t)        *
*                                                                    *
*   - internal error in module, routine 'proc', due to Tree t        *
*                                                                    *
*********************************************************************/

void failure_protocol (module, proc, t)

char module[];
char proc[];
tTree t;

{ int a;

  printf ("INTERNAL ERROR: failure in module  = %s\n", module);
  printf ("                procedure/function = %s\n", proc);
  printf ("Tree = "); FileUnparse (stdout, t); printf ("\n");
  WriteTree (stdout, t);
  error_protocol ("stop for internal ERROR");
  out_last_stmt ("ERROR", " last statement/decl");
  close_protocol ();
  a = 5;
  a = a / (a - a);
  exit (-1);

} /* failure_protocol */

/*********************************************************************
*                                                                    *
*  void failure2_protocol (char module[], char proc[], Tree t1, t2)  *
*                                                                    *
*   - same as failure_protocol, but print two Trees, t1 and t2       *
*                                                                    *
*********************************************************************/

void failure2_protocol (module, proc, t1, t2)

char module[];
char proc[];
tTree t1, t2;

{ int a;

  printf ("INTERNAL ERROR: failure in module  = %s\n", module);
  printf ("                procedure/function = %s\n", proc);
  printf ("Arg Tree 1 = "); FileUnparse (stdout, t1); printf ("\n");
  printf ("Arg Tree 2 = "); FileUnparse (stdout, t2); printf ("\n");
  WriteTree (stdout, t1);
  WriteTree (stdout, t2);
  error_protocol ("stop for internal ERROR");
  close_protocol ();
  a = 5;
  a = a / (a - a);
  exit (-1);

} /* failure2_protocol */

/*********************************************************************
*                                                                    *
*   void hint_protocol (int nr)                                      *
*                                                                    *
*********************************************************************/

void hint_protocol (nr)

int nr;

{ fprintf (protocol, "HINT for avoiding this error :\n");

  fprintf (protocol, "   ");

  if (nr == HINT_HOME)
    fprintf (protocol, "use a variable with linear combination of loop ids");

  fprintf (protocol, "\n");

} /* hint_protocol */

