/*********************************************************************
*                                                                    *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                   *
*  Date        : Mar 94                                              *
*  Last Update : Mar 94                                              *
*                                                                    *
*  Module      : SrcLines.c                                          *
*                                                                    *
*  Function    : Handling of Source Lines                            *
*                                                                    *
*  Unsolved Problems:                                                *
*                                                                    *
*       - continuation lines for directives                          *
*       - test read buffer for assignment  IDENT [ (...) ] =         *
*       - blanks are still significant                               *
*       - no clear distinction between fixed and free                *
*            - signs in col 1 - 5 -> fix                             *
*            - & -> free                                             *
*            - ignore continuation if free is assumed                *
*                                                                    *
*********************************************************************/

# include "ratc.h"           /* needed for rbool */
# include "global.h"
# include "SrcLines.h"

# undef DEBUG

# define is_EOF(x)   ((x == -1) || (x == 255))

/*******************************************************************
*                                                                  *
*   fix_length == 0   stands for free format                       *
*                                                                  *
*******************************************************************/

# define  IsFreeFormat  (fix_length == 0)
 
/*******************************************************************
*                                                                  *
*   Global BUFFER data                                             *
*                                                                  *
*******************************************************************/

  /* global data */

#define BUFFER_NORMAL     0
#define BUFFER_STRING1    1   /* string with '         */
#define BUFFER_STRING2    2   /* string with "         */
#define BUFFER_COMMENT    3   /* ending comment with ! */

  static char *Buffer_Data;
  static int  Buffer_MaxSize;
  static int  Buffer_Counter;
  static int  Buffer_NewLines;
  static int  Buffer_State;
  static int  Buffer_ParNest;        /* nesting of parentheses ) and ( */
  static rbool Buffer_Continue;       /* rtrue if Buffer ends with &     */


#define BUFFER_NO_ERROR      0
#define BUFFER_STRING_ERROR  1
#define BUFFER_PARENTH_ERROR 2
#define BUFFER_FIXED_ERROR   3
#define BUFFER_FULL_ERROR    4

  static int  Buffer_Error;

  /* exported data */

  int  is_assignment = 0;        /* will be exported */

/*******************************************************************
*                                                                  *
*   GetSourceLine (f, line)                                        *
*                                                                  *
*   - read a complete line from a Source File                      *
*                                                                  *
*******************************************************************/

void GetSourceLine (f, line)

FILE *f;
SourceLine *line;

{ int counter, stop;
  char c;
  char *ptr;

  counter = 0;
  stop    = 0;
  ptr     = line->content;

  while (!stop) 
    { c = fgetc (f);

      if (is_EOF (c))

         { stop = 1;
         }

      else if (c == '\n')
         { stop = 1;
           *ptr++ = c;
           counter ++;
         }

      else if (IsFreeFormat || (counter < fix_length))

           /* fix_length is imported from global.c */

         { /* c is a relevant charachter */

           *ptr++ = c;
           counter ++;
         }
    }

#ifdef DEBUG
  printf ("GetSourceLine returns line with %d characters\n", counter);
#endif

  line->counter = counter;

} /* GetSourceLine */

/*******************************************************************
*                                                                  *
* int my_compare (char *str1, *str2)                               *
* char my_upper (char c)                                           *
*******************************************************************/

static char my_upper (c)
char c;
{ if ((c >= 'a') && (c <= 'z'))
    return (c + ('A' - 'a'));
   else
    return c;
}

static rbool my_compare (str1, str2)
char *str1, *str2;

{  rbool equal;

   equal = rtrue;

   while (equal && *str2)
     equal = (my_upper (*str1++) == my_upper (*str2++));

   return (equal);
}

/*******************************************************************
*                                                                  *
* int ClassifyLine (NewLine)                                       *
*                                                                  *
*  - this classification is applied to a single source line        *
*                                                                  *
*******************************************************************/

#define EMPTY_LINE        0
#define COMMENT_LINE      1
#define CONTINUATION_LINE 2
#define BLANK_LINE        3
#define STMT_LINE         4
#define DIRECTIVE_LINE    5
#define CONT_DIR_LINE     6

static rbool IsAnyCharacter (c)
char c;

{ if (c == ' ')  return (rfalse);
  if (c == '\t') return (rfalse);
  if (c == '\n') return (rfalse);
  return (rtrue);

} /* IsAnyCharacter */

static rbool IsDigit (c)
char c;
{ if (c < '0') return (rfalse);
  if (c > '9') return (rfalse);
  return (rtrue);
}

static rbool IsFixedComment (c)
char c;
{ if (c == 'c') return (rtrue);
  if (c == 'C') return (rtrue);
  if (c == '!') return (rtrue);
  if (c == '$') return (rtrue);
  if (c == '*') return (rtrue);
}

/* the following predicate returns rtrue if the line does not
   satisfy the conditions of a fixed line                     */

static rbool IsFixedLine (NewLine)
SourceLine *NewLine;

{  int  pos, nr;
   char *ptr, c;

   ptr = NewLine->content;
   nr = NewLine->counter;

   if (nr > 6) nr = 6;      /* only first 6 characters are considered */

   if (nr > 1)
     { if (IsFixedComment (*ptr))
          return (rtrue);
     }

   pos = 0;
   while (pos < nr)

    {  c = *ptr++;
       if (IsAnyCharacter (c) && !IsDigit(c))
          return (rfalse);
       if (c == '\t')
          pos += 8;
       pos ++;
    }
   return (rtrue);

} /* IsFixedLine */

/*******************************************************************
*                                                                  *
* rbool IsCommentLine (SourceLine *NewLine)                        *
*                                                                  *
*******************************************************************/

static rbool IsCommentLine (NewLine)
SourceLine *NewLine;

{  int  pos, nr;
   char *ptr, c;
   rbool stop;

   ptr = NewLine->content;
   nr = NewLine->counter;

   if (nr <= 0) return (rfalse);

   c = *ptr;

   if (!IsFreeFormat)
     { if (c == 'c') return (rtrue);
       if (c == 'C') return (rtrue);
       if (c == '*') return (rtrue);
     }

   /* now look at the first non blank character */

   pos = 0; stop = rfalse;

   while (!stop) 

    {  c = *ptr++;
       stop = (c != ' ');
       if (pos < nr) 
          { if (!stop) pos++; }
        else
          stop = rtrue;
    }
   
   /* case 1 : no non-blank character found */

   if (pos >= nr) return (rfalse);

   /* case 2 : first character is !  */

   if (c == '!') return (rtrue);

   /* case 3 : first charcter is not ! */

   return (rfalse);

} /* IsCommentLine */

/*******************************************************************
*                                                                  *
*  int ClassifyLine (SourceLine *NewLine)                          *
*                                                                  *
*******************************************************************/

int ClassifyLine (NewLine)
SourceLine *NewLine;

{ int pos, nr;
  char *ptr, c;
  int i, is;

  nr  = NewLine->counter;
  ptr = NewLine->content;

  if (nr == 0)
     return (EMPTY_LINE);

  /* at least one character */

  c = *ptr;

  if (IsCommentLine (NewLine))

    { /* directive or comment */

      while (*ptr == ' ') { ptr++; nr--; }

      if (nr > 4)
        { if (my_compare (ptr+1, "HPF$"))
            { *ptr = 'C';                  /* no confusion with ending comm */
              if (!IsFreeFormat && (nr > 5))
                 { if (IsAnyCharacter (ptr[5]))
                      return (CONTINUATION_LINE);
                 }
              return (DIRECTIVE_LINE);
            }
        }

      if (nr > 4)
        { if (my_compare (ptr+1, "ADP$"))
            { *ptr = 'C';                  /* no confusion with ending comm */
              if (!IsFreeFormat && (nr > 5))
                 { if (IsAnyCharacter (ptr[5]))
                      return (CONTINUATION_LINE);
                 }
              return (DIRECTIVE_LINE);
            }
        }

      return (COMMENT_LINE);
    }

  /* check for continuation line */

  if ((!IsFreeFormat) && (nr >= 6))
    { is = IsAnyCharacter (ptr[5]);
      for (i=0; i<5; i++)
        is = (is && (ptr[i] == ' '));
      if (is)
         return (CONTINUATION_LINE);
    }

  /* check for blank line,  also if only ! ending comment  */

  is = 0;   /* is is true if line contains relevant characters */

  i = 0;
  while ((!is) && (i < nr))
    { if (ptr[i] == '!') 
         i = nr;  /* ending comment, is blank line */
       else if (IsAnyCharacter (ptr[i]))
         is = 1;  /* is relevant character         */
       else
         i++;
    } 

  /* for (i=0; i<nr; i++) is = (is || IsAnyCharacter (ptr[i])); */

  if (!is) return (BLANK_LINE);

  return (STMT_LINE);

}

/*******************************************************************
*                                                                  *
*   Shrinking of continuations lines                               *
*                                                                  *
*   FIXED :   '     +abc'  ->  'abc'                               *
*   FREE  :   '  &abc'     ->  'abc'                               *
*                                                                  *
*******************************************************************/

void ShrinkFixedContinuationLine (Line)
SourceLine *Line;

{ int i, n;

  n = Line->counter;

  for (i=0; i<n-6; i++)
    Line->content[i] = Line->content[i+6];

  Line->counter -= 6;

} /* ShrinkFixedContinuationLine */

void ShrinkFreeContinuationLine (Line)
SourceLine *Line;

{ int  i;
  int  pos;        /* pos will be position of first relevant character */
  int  nr;
  int  stop;       /* stop becomes rtrue at first relevant character    */
  char *ptr;
  char c;

  pos  = 0;
  stop = 0;

  nr  = Line->counter;
  ptr = Line->content;

  /* check for !HPF continuation line */

  if (nr > 4)

     { if (my_compare (ptr, "CHPF$"))

         { pos += 5;  
           ptr += 5; 
         }
     }

  while ((pos < nr) && (!stop))

    { c = *ptr++;
      if ((c == '\t') || (c == ' '))
         pos++;
       else if (c == '&')
         { pos++; stop = 1; }
       else
         stop = 1;
    }

  for (i=0; i<nr-pos; i++)
    Line->content[i] = Line->content[i+pos];

  Line->counter -= pos;

} /* ShrinkFreeContinuationLine */


/*******************************************************************
*                                                                  *
*   CheckAssignment (buffer, len)                                  *
*                                                                  *
*   - set is_assignment to 1 globally if buffer is an assignment   *
*                                                                  *
*   Attention : logic for assignment is still too simple           *
*                                                                  *
*   Now  :                                                         *
*                                                                  *
*      ... =  ...         is an assignment                         *
*                                                                  *
*   But  :                                                         *
*                                                                  *
*     PARAMETER (A=50)                                             *
*     CALL SUB (X=VALUE)                                           *
*     "....=...."                                                  *
*                                                                  *
*******************************************************************/

static void CheckAssignment (buffer, len)

char *buffer;
int  len;

{ int i, state;

  state = 0;

  for (i=0; i<len; i++)

    { if (buffer[i] == '=')
         state = 1;
    }

  if (state == 1)
     is_assignment = 1;
   else
     is_assignment = 0;

  /* printf ("Check Assignment, is assign = %d\n", is_assignment); */

} /* CheckAssignment */

/*******************************************************************
*                                                                  *
*   BUFFER handling                                                *
*                                                                  *
*******************************************************************/

      /******************************************************
      *                                                     *
      *  OpenBuffer (char *Buffer, int MaxSize)             *
      *                                                     *
      ******************************************************/

void OpenBuffer (Buffer, MaxSize)
char *Buffer;
int  MaxSize;

{ Buffer_Data     = Buffer;
  Buffer_MaxSize  = MaxSize;
  Buffer_Counter  = 0;
  Buffer_NewLines = 0;
  Buffer_ParNest  = 0;
  Buffer_Continue = rfalse;
  Buffer_State    = BUFFER_NORMAL;
  Buffer_Error    = BUFFER_NO_ERROR;
}

      /******************************************************
      *                                                     *
      *  AppendLineToBuffer (SourceLine *NewLine)           *
      *                                                     *
      ******************************************************/

void AppendLineToBuffer (NewLine)
SourceLine *NewLine;

{ char *ptr, c;
  int i, n;
  rbool found;
  int  last_pos;

  ptr  = NewLine->content;
  n    = NewLine->counter;

#ifdef DEBUG
  printf ("Append line (%d chars) to buffer (%d chars)\n", n, Buffer_Counter);
  if (n == 1) printf ("1 char is %d\n", *ptr);
  c = '\n'; printf ("new line = %d\n", c);
  c = '\r'; printf ("return = %d\n", c);
#endif

  if ( (Buffer_Counter == 0) && (n > 0) && (!IsFreeFormat) )

     { /* now the buffer is filled for the first time */

       if (!IsFixedLine (NewLine))
          Buffer_Error = BUFFER_FIXED_ERROR;
     }

  for (i=0; i<n; i++)
    { c = *(ptr++);

      if (IsAnyCharacter(c))
         Buffer_Continue = rfalse;   /* continuation is satisfied */

      if (c == '\n')
         Buffer_NewLines ++;
       else if (Buffer_Counter == Buffer_MaxSize)
         Buffer_Error = BUFFER_FULL_ERROR;
       else
         Buffer_Data[Buffer_Counter++] = c;

      /* update Buffer State */

      if (Buffer_State == BUFFER_NORMAL)
        { if (c == '\'')
            Buffer_State = BUFFER_STRING1;
          if (c == '\"')
            Buffer_State = BUFFER_STRING2;
          if (c == '!')
            { Buffer_State = BUFFER_COMMENT;
              Buffer_Counter --;
            }
          if (c == '(')  Buffer_ParNest ++;
          if (c == ')')  Buffer_ParNest --;
        }

      else if (Buffer_State == BUFFER_STRING1)
        { if (c == '\'')
            Buffer_State = BUFFER_NORMAL;
        }

      else if (Buffer_State == BUFFER_STRING2)
        { if (c == '\"')
            Buffer_State = BUFFER_NORMAL;
        }

      else if (Buffer_State == BUFFER_COMMENT)
        { if (c == '\n')
             Buffer_State = BUFFER_NORMAL;
           else
             Buffer_Counter --;     /* comments will be deleted */
        }

    }  /* for */

    /* check for Continuation '&    ' */

    if ((!Buffer_Continue) && (IsFreeFormat))
     {  found    = rfalse;
        last_pos = Buffer_Counter-1;
        while ((last_pos >= 0) && !found)
          { found = (IsAnyCharacter (Buffer_Data[last_pos]));
            if (!found) last_pos--;
          }
        if (found) 
           Buffer_Continue = (Buffer_Data[last_pos] == '&');
          else
           Buffer_Continue = rfalse;
        if (Buffer_Continue)
           Buffer_Counter = last_pos;
     }

#ifdef DEBUG
    printf ("AppendLineToBuffer, final state is now : %d\n", Buffer_State);
    printf ("Buffer has now %d characters\n", Buffer_Counter);
    if (Buffer_Continue)
       printf ("Buffer needs continuation line\n");
#endif

} /* AppendLineToBuffer */

      /******************************************************
      *                                                     *
      *  int CloseBuffer ()                                 *
      *                                                     *
      ******************************************************/

rbool LineError ()
{
  return (Buffer_Error != BUFFER_NO_ERROR);
}

char *LineErrorMsg ()

{ if (Buffer_Error == BUFFER_STRING_ERROR)
     return ("unterminated string");
  if (Buffer_Error == BUFFER_PARENTH_ERROR)
     return ("unbalanced parentheses");
  if (Buffer_Error == BUFFER_FIXED_ERROR)
     return ("not fixed format");
  if (Buffer_Error == BUFFER_FULL_ERROR)
     return ("too many characters");
  return ("");
}

int BufferIsFormat ()

{ char *data;

  /* note : this routine is safe as Buffer contains '(' or ')' */

  data = Buffer_Data;

  while (*data == '\n') data++;
  while (*data == ' ') data++;
  while (IsDigit (*data)) data++;
  while (*data == ' ') data++;
  if ((data[0] != 'F') && (data[0] != 'f')) return (0);
  if ((data[1] != 'O') && (data[1] != 'o')) return (0);
  if ((data[2] != 'R') && (data[2] != 'r')) return (0);
  if ((data[3] != 'M') && (data[3] != 'm')) return (0);
  if ((data[4] != 'A') && (data[4] != 'a')) return (0);
  if ((data[5] != 'T') && (data[5] != 't')) return (0);
  return (1);
}

int CloseBuffer ()

{ int i;

  /* check for un-terminated strings */

  if (Buffer_State == BUFFER_STRING1)
     Buffer_Error = BUFFER_STRING_ERROR;

  if (Buffer_State == BUFFER_STRING2)
     Buffer_Error = BUFFER_STRING_ERROR;

  if (Buffer_ParNest != 0)
     if (!BufferIsFormat())  /* might be still correct */
        Buffer_Error = BUFFER_PARENTH_ERROR;

  if (Buffer_Error != BUFFER_NO_ERROR)   /* skip line for an error */
     Buffer_Counter = 0;

  /* append the new line characters counted so far */

  if (Buffer_Counter + Buffer_NewLines >= Buffer_MaxSize)
     Buffer_Error = BUFFER_FULL_ERROR;
   else
     { for (i=0; i< Buffer_NewLines; i++)
          Buffer_Data [Buffer_Counter++] = '\n';
     }

  /* check buffer for an assignment */

  CheckAssignment (Buffer_Data, Buffer_Counter);

#ifdef DEBUG
  printf ("Buffer has been filled with %d characters\n", Buffer_Counter);
  for (i=0; i<Buffer_Counter; i++)
    printf ("%c", Buffer_Data[i]);
#endif

  return (Buffer_Counter);

}  /* CloseBuffer */

/*******************************************************************
*                                                                  *
*   GetCompleteLine (f, Buffer, Size, LastLine)                    *
*                                                                  *
*   - returns in Buffer a complete new line from FILE f            *
*   - LastLine contains already read but unused characters         *
*                                                                  *
*******************************************************************/

int GetCompleteLine (f, Buffer, Size, LastLine, lines)
FILE *f;
char *Buffer; int Size;
SourceLine *LastLine;
int *lines;

{  int i, n, counter; 
   char *ptr;
   int stop;
   int line_class;

   /* nothing read until now */

   /* fill LastLine if not existing */

   if (LastLine->counter == 0)

      { GetSourceLine (f, LastLine);
        line_class = ClassifyLine (LastLine);
        if ((line_class == COMMENT_LINE) || (line_class == BLANK_LINE))
           { LastLine->counter = 1;
             LastLine->content[0] = '\n';
           }
      }

   OpenBuffer (Buffer, Size);

   stop    = 0;

   while (!stop)

     {  /* copy the LastLine into the buffer */

        AppendLineToBuffer (LastLine);

        /* get a new last line */

        GetSourceLine (f, LastLine);

        line_class = ClassifyLine (LastLine);

#ifdef DEBUG
        printf ("classification is %d\n", line_class);
#endif

        if (line_class == EMPTY_LINE)   /* eof */
           stop = 1;
         else if ((line_class == COMMENT_LINE) || (line_class == BLANK_LINE))
           { LastLine->counter = 1;
             LastLine->content[0] = '\n';
           }
         else if (line_class == CONTINUATION_LINE)
           { ShrinkFixedContinuationLine (LastLine);
           }
         else
           { stop = (!Buffer_Continue);
             if ((!stop) && IsFreeFormat)
                ShrinkFreeContinuationLine (LastLine);
           }

     } /* while not completely new line */

   *lines = Buffer_NewLines;

   return (CloseBuffer ());

} /* GetCompleteLine */
