/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 96                                                   *
*  Last Update : Feb 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : rpc_sp.m4                                                *
*                                                                         *
*  Function: machine dependent remote procedure calls                     *
*                                                                         *
*      SP  version       (realization)                                    *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
*   void rpc_init ()                                                      *
*                                                                         *
*     - starts the server mode which allows recvncalls                    *
*                                                                         *
*   void rpc_exit ()                                                      *
*                                                                         *
*     - shutdown the server mode                                          *
*                                                                         *
*   void rpc_call (int pid, char *query, int query_size,                  *
*                           char *answer, int answer_size)                *
*                                                                         *
*    - remote call of dalib_answer_question                               *
*                                                                         *
*  Note: first 4 bytes of query must not be 0  (halt request)             *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"
#include "mpproto.h"

/**************************************************************************
*                                                                         *
*                                                                         *
**************************************************************************/

static int rpc_server_is_running = 0;

static volatile int servers;

extern int mperrno;

#define RPC_QUERY_TYPE  1051
#define RPC_ANSWER_TYPE 2051

#define MAX_QUESTION_SIZE 16

typedef int question_type[MAX_QUESTION_SIZE];

static question_type question[MAXP];
static int           client [MAXP];
static int           type   [MAXP];
static int           msgid  [MAXP];

void rpc_handler ();
void dalib_answer_question ();

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void rpc_start_server (server)

int server;

{ int sid;
  size_t length;
  int rc;

  sid = server-1;

  client[sid] = sid;
  type  [sid] = RPC_QUERY_TYPE + sid;

  length = sizeof(question_type);
 
  rc = mpc_rcvncall (question + sid, length, client + sid,
                     type + sid, msgid + sid, rpc_handler);
 
#ifdef DEBUG
      printf ("%d: recv for %d, tag = %d, msgid = %d\n",
               pcb.i, server, type[sid], msgid[sid]);
#endif
 
  if (rc)

   { dalib_internal_error ("rpc_init: could not start server mode");
     printf ("%d: failed to start server for %d\n", pcb.i, server);
     dalib_exit ();
   }
 
} /* rpc_start_server */

/**************************************************************************
*                                                                         *
*  void rpc_send_question (int target_pid, char *question, int size)      *
*                                                                         *
**************************************************************************/

void rpc_send_question (target_pid, question, size)

int target_pid;
int size;
unsigned char *question;


{  int tag;
   int rc;

   tag = RPC_QUERY_TYPE + pcb.i - 1;  /* call my server */
   rc  = mpc_bsend (question, size, target_pid-1, tag);

   if (rc)
     { dalib_internal_error ("rpc_send_question failed");
       printf ("mpc_bsend error code = %d\n", mperrno);
       dalib_stop ();
     }

} /* rpc_send_question */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void rpc_send_answer (pid, answer, answer_size)

void *answer;
int answer_size;
int pid;
 
{ int answer_tag;
  int source;
  int rc;

  size_t length;

  /* send answer back */
 
  source     = pid - 1;
  answer_tag = RPC_ANSWER_TYPE + source;
  length     = answer_size;
 
#ifdef DEBUG
  printf ("%d: answer (tag=%d) the request of client %d\n",
          pcb.i, answer_tag, pid);
#endif
 
  rc = mpc_bsend (answer, length, source, answer_tag);
 
  if (rc)
 
    { dalib_internal_error ("rpc_send_answer answer failed");
      printf ("%d: mpc_bsend error code = %d\n", pcb.i, mperrno);
      dalib_stop ();
    }

#ifdef DEBUG
  printf ("%d: send the answer data to processor %d\n", pcb.i, pid);
#endif
 
} /* send answer */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void rpc_recv_answer (pid, answer, answer_size)

void *answer;
int answer_size;
int pid;

{ int tag;
  int source;
  int mstat;
  int rc;

  size_t nbytes;

   /* IMPORTANT : do not call the blocking receive before the message
                  is available; otherwise this processor is blocked
                  for requests from other processors                  */
 
   tag    = RPC_ANSWER_TYPE + pcb.i - 1;
   source = pid-1;
 
#ifdef DEBUG
   printf ("%d: wait for answer (tag=%d) of processor %d\n",
            pcb.i, tag, pid);
#endif
 
   for(;;)
       { rc = mpc_probe(&source,&tag,&mstat);
         if(mstat >= 0)  break;
       }
 
   /* now we know that the message is available */
 
   rc = mpc_brecv (answer, answer_size, &source, &tag, &nbytes);
 
   if (rc)
 
     { dalib_internal_error ("dalib_rpc_get answer failed");
       printf ("%d: mpc_brecv error code = %d\n", pcb.i, mperrno);
       dalib_stop ();
     }
 
#ifdef DEBUG
   printf ("%d: got the answer of processr %d\n", pcb.i, pid);
#endif
 
} /* rpc_recv_answer */

/**************************************************************************
*                                                                         *
*  rpc_call (int pid, char *query, int query_size,                        *
*                     char *answer, int answer_size)                      *
*                                                                         *
*    - remote call of dalib_answer_question                               *
*                                                                         *
**************************************************************************/

void rpc_call (pid, query, query_size, answer, answer_size)

int pid;
unsigned char *query, *answer;
int query_size, answer_size;

{ rpc_send_question (pid, query, query_size);

  if (answer_size > 0)
     rpc_recv_answer (pid, answer, answer_size);

} /* rpc_call */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void rpc_handler (int *inmsg)

{ /* answering request */

  int rc;
  int my_client;

  size_t nbytes;

  size_t length;
  int    answer_tag;

  int i;

#ifdef DEBUG
  printf ("%d: rpc_handler activated, msg_id = %d\n", pcb.i, *inmsg);
#endif

  my_client = -1;

  for (i=0; i<pcb.p; i++)

    if (i+1 != pcb.i)

       if (*inmsg == msgid[i]) my_client = i;

  rc = mpc_wait (inmsg, &nbytes);

  if (rc)
    { dalib_internal_error ("rpc_handler: message not available");
      dalib_stop ();
    }

#ifdef DEBUG
  printf ("%d: got a request of client %d, size = %d\n", 
           pcb.i, my_client+1, nbytes);
#endif

  if (question[my_client][0] == 0)

     {  /* I can stop the server for this processor */

        servers = servers - 1;

#ifdef DEBUG
  printf ("%d: stop service for processor %d, %d servers running\n",
           pcb.i, my_client+1, servers);
#endif

     }

   else

     { unsigned char *answer;
       int size;

       /* now we can answer the request */

       dalib_answer_question (question+my_client, &answer, &size);

       /* send answer back */

       rpc_send_answer (my_client+1, answer, size);

       free (answer);   /* give memory of answer free */

       /* restart server mode */

       rpc_start_server (my_client + 1);
 
     }

} /* rpc_handler */

/**************************************************************************
*                                                                         *
*   void rpc_init ()                                                      *
*                                                                         *
*     - starts the server mode which allows recvncalls                    *
*                                                                         *
**************************************************************************/

void rpc_init ()

{ int rc;
  int pid;

  if (rpc_server_is_running) 
     return;

#ifdef DEBUG
  printf ("%d: will start the server mode\n", pcb.i);
#endif 

  /* make sure that msgid is not wrongly initialized */

  for (pid=0; pid<pcb.p; pid++) msgid[pid] = -1;

  for (pid=1; pid<=pcb.p; pid++)

   if (pid != pcb.i) rpc_start_server (pid);

#ifdef DEBUG
  printf ("%d: server mode should run now forever\n", pcb.i);
#endif 

  /* note : server mode will run forever as rpc_handler starts again */

  rpc_server_is_running = 1;

  servers = pcb.p - 1;

#ifdef DEBUG
  printf ("%d: %d servers are running\n", pcb.i, servers);
#endif 

} /* rpc_init */

/**************************************************************************
*                                                                         *
*   void rpc_exit ()                                                      *
*                                                                         *
*     - shutdown the server mode                                          *
*                                                                         *
**************************************************************************/

void rpc_exit ()

{ /* shutdown the server */

  int pid;

  int send_data[2];  /* pid, source */

          /********************************************
          *                                           *
          *   send the final message to all others    *
          *                                           *
          ********************************************/

   for (pid = 0; pid < pcb.p; pid++)

   if (pid+1 != pcb.i)

    { send_data[0]   = 0;           /* I am ready */
      send_data[1]   = pcb.i;

#ifdef DEBUG
      printf ("%d: call of rpc_end on %d\n", pcb.i, pid+1);
#endif
 
      rpc_send_question (pid+1, send_data, 2 * sizeof (int));

    } /* sending final request to all processors */

          /********************************************
          *                                           *
          *   wait for all final message of others    *
          *                                           *
          ********************************************/

#ifdef DEBUG
   printf ("%d: still %d services running\n", pcb.i, servers);
#endif

   while (servers) {

     /* busy wait for final request of all processors */

   }

#ifdef DEBUG
   printf ("%d: all services finished\n", pcb.i);
#endif

} /* rpc_exit */
