/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 94                                                   *
*  Last Update : Aug 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : pt2pt.m4                                                 *
*                                                                         *
*  Function: machine dependent point-to-point communication               *
*                                                                         *
*      MPI version       (realization)                                    *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#ifdef CENJU3
#include "/vol/local/mpich/include/mpi.h"
#else
#include "mpi.h"
#endif

#undef MSG_DEBUG

typedef struct

  { int full, msg_length;
    MPI_Request request;
    char *msg;
  } local_mailbox;

static local_mailbox mailboxes[MAXP1];

     /*************************************************************
     *                                                            *
     *  void machine_mpi_init ()                                  *
     *                                                            *
     *************************************************************/

void machine_mpi_init ()

{ int i;

  for (i=0; i<=pcb.p; i++) mailboxes[i].full = 0;

} /* machine_mpi_init */

     /*************************************************************
     *                                                            *
     *  void machine_wait_send (int pid)                          *
     *                                                            *
     *    - waiting for completion of last sending to pid         *
     *                                                            *
     *************************************************************/

void machine_wait_send (pid)

int pid;

{ MPI_Status status;

  /* wait until the last message has been sent and deallocate send buffer */

  MPI_Wait (&(mailboxes[pid].request), &status);
      
  dalib_free (mailboxes[pid].msg, mailboxes[pid].msg_length);

  mailboxes[pid].full = 0;

} /* machine_wait_send */

     /*************************************************************
     *                                                            *
     *  void machine_mpi_exit ()                                  *
     *                                                            *
     *************************************************************/

void machine_mpi_exit ()

{ int pid;

  for (pid=0; pid<=pcb.p; pid++) 

     if (mailboxes[pid].full) 

        machine_wait_send (pid);

} /* machine_mpi_exit */

     /*************************************************************
     *                                                            *
     *  machine_mpi_send (int to, char *msg, int len, int kind)   *
     *                                                            *
     *  BLOCKING send                                             *
     *                                                            *
     *  - kind = 0 : really blocking until receive                *
     *  - kind = 1 : blocking until msg copied                    *
     *  - kind = 2 : msg buffer can directly be used              *
     *                                                            *
     *************************************************************/

void machine_mpi_send (to, message, length, kind)

int to, length, kind;
char *message;

{ int target;
  int tag;

  char *buffer;

#ifdef MSG_DEBUG
  printf ("%d: call of send to = %d, length = %d\n", pcb.i, to, length);
#endif

  target = to - 1;
  tag    = pcb.i;

  if (kind == 0)

     { /* call a blocking send, buffering not necessary */

       MPI_Send (message, length, MPI_BYTE, target, tag, MPI_COMM_WORLD);

       return;

     }

  /* wait for the last non-blocking send */

  if (mailboxes[to].full)

    machine_wait_send (to);

  if (kind != 2)

    { /* we have to copy the message for the non-blocking send */

      buffer = (char *) dalib_malloc (length, "machine_mpi_send");
      dalib_memcopy (buffer, message, length);
    }

   else  /* no copy required */

      buffer = message;

  MPI_Isend (buffer, length, MPI_BYTE, target, tag, MPI_COMM_WORLD,
             &(mailboxes[to].request));

  mailboxes[to].full       = 1;
  mailboxes[to].msg        = buffer;
  mailboxes[to].msg_length = length;

} /* machine_mpi_send */

     /*********************************************************
     *                                                        *
     *  machine_mpi_receive (int from, char *msg, int len)    *
     *                                                        *
     *  BLOCKING receive                                      *
     *                                                        *
     *  - blocks really until message has been received       *
     *                                                        *
     *********************************************************/

void machine_mpi_receive (from, message, length)

int  from, length;
char *message;

{ int tag, source;
  MPI_Status status;

#ifdef MSG_DEBUG
  printf ("%d: will recv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  if (from == -1)

     { /* receive any message */

       source = MPI_ANY_SOURCE;
       tag    = MPI_ANY_TAG;
     }

    else

     { /* receive specific message */

       source = from - 1;
       tag    = from;
     }

  MPI_Recv (message, length, MPI_BYTE, source, tag,
            MPI_COMM_WORLD, &status);

#ifdef MSG_DEBUG
  printf ("%d: has recvd from = %d , length = %d\n",
           pcb.i, from, length);
#endif

} /* machine_mpi_receive */

     /*********************************************************
     *                                                        *
     *  machine_mpi_sendreceive (...)                         *
     *                                                        *
     *   - blocking send/receive in one step                  *
     *                                                        *
     *********************************************************/

void machine_mpi_sendreceive (to, out_msg, out_len,
                              from, in_msg, in_len)

int to, from;
char *out_msg, *in_msg;
int in_len, out_len;

{
  dalib_internal_error ("machine_mpi_sendreceive not supported");

} /* machine_mpi_sendreceive */


     /*********************************************************
     *                                                        *
     *  int machine_mpi_isend (int to, char *msg, int len)            *
     *                                                        *
     *  NON BLOCKING send                                     *
     *                                                        *
     *********************************************************/

int machine_mpi_isend (to, message, length)

int to, length;
char *message;

{ int tag;
  int msg_id;

  /* non blocking send */

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, length = %d\n",
           pcb.i, to, length);
#endif

  tag = pcb.i;   /* own processor id is the tag */

  msg_id = 0;

  dalib_internal_error ("NON BLOCKING mode not available for MPI");

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, msg_id = %d\n", pcb.i, to, msg_id);
#endif

  return (msg_id);

} /* machine_mpi_isend */


     /*************************************************************
     *                                                            *
     *  int machine_mpi_ireceive (int from, char *msg, int len)   *
     *                                                            *
     *  NON BLOCKING receive                                      *
     *                                                            *
     *************************************************************/

int machine_mpi_ireceive (from, message, length)

int  from, length;
char *message;

{ int tag, msg_id;

  tag = from;

#ifdef MSG_DEBUG
  printf ("%d: will irecv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  /* msg_id = irecv (tag, message, length); */

  dalib_internal_error ("NON BLOCKING mode not available for MPI");

#ifdef MSG_DEBUG
  printf ("%d: irecv got msg_id = %d\n", pcb.i, msg_id);
#endif

  return (msg_id);

} /* machine_mpi_receive */


     /*********************************************************
     *                                                        *
     *  int machine_mpi_done (int msg_id)                     *
     *                                                        *
     *  - ask status of an immediate send/receive             *
     *  - msg_id is free if machine_mpi_done == 1             *
     *                                                        *
     *********************************************************/

int machine_mpi_done (msg_id)

int msg_id;    /* has been given by immediate send or receive */

{ int done;

  dalib_internal_error ("NON BLOCKING mode not available for MPI");

  return (done);

} /* machine_mpi_done */

     /*********************************************************
     *                                                        *
     *  void machine_mpi_wait (int msg_id)                    *
     *                                                        *
     *  - wait for completion of an immediate send/receive    *
     *                                                        *
     *********************************************************/

void machine_mpi_wait (msg_id)
int msg_id;

{
#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d will be waited\n",
           pcb.i, msg_id);
#endif

  /* msgwait (msg_id); */

  dalib_internal_error ("NON BLOCKING mode not available for MPI");

#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d is finished\n",
           pcb.i, msg_id);
#endif

} /* machine_mpi_wait */
