/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jun 94                                                   *
*  Last Update : Jun 94                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : pt2pt.c                                                  *
*                                                                         *
*  Function: machine dependent point-to-point communication               *
*                                                                         *
*      SP1 version       (realization)                                    *
*                                                                         *
*  Export :  internal Interface                                           *
*  ============================                                           *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#define DONTCARE -1

#undef MSG_DEBUG

typedef struct
  { int full, msg_length, msg_id;
    char *msg;
  } local_mailbox;

static local_mailbox mailboxes[MAXP1];
static int all_processors;

void machine_mpi_init ()

{  int i;
   all_processors = pcb.p;
   for (i=0; i<=pcb.p; i++)
      mailboxes[i].full = 0;
}

void machine_mpi_exit ()

{ int i;
  for (i=0; i<=all_processors; i++)

    if (mailboxes[i].full )

      { dalib_free (mailboxes[i].msg, mailboxes[i].msg_length);
        mailboxes[i].full = 0;
      }
} /* machine_mpi_exit */

     /**************************************************************
     *                                                             *
     *  machine_mpi_send (int to, char *msg, int len, int kind)    *
     *                                                             *
     *  BLOCKING send                                              *
     *                                                             *
     *  - kind = 0 : really blocking until receive                 *
     *  - kind = 1 : blocking until msg copied                     *
     *  - kind = 2 : msg buffer can directly be used               *
     *                                                             *
     **************************************************************/

void machine_mpi_send (to, message, length, kind)

int to, length, kind;
char *message;

{ int tag, target_pid;
  int rc, msg_id;
  int nbytes;
  char *buffer;

  /* blocking send */

#ifdef MSG_DEBUG
  printf ("%d: call of send to = %d, length = %d\n",
           pcb.i, to, length);
#endif

  tag = pcb.i;   /* own processor id is the tag */
  target_pid = to - 1;

  if (mailboxes[to].full)

    { /* there is already a pending message */

      msg_id = mailboxes[to].msg_id;
      nbytes = mailboxes[to].msg_length;
      rc = mpc_wait ( &msg_id, &nbytes);
      if (rc)
         printf ("%d: ERROR WAIT\n", pcb.i);
      dalib_free (mailboxes[to].msg, nbytes);
      mailboxes[to].full = 0;
    }

  /* now a new message can be send, type will be from */


  if (kind != 2)

    { /* at first make a copy of the message */

      buffer = (char *) dalib_malloc (length, "asend");
      dalib_memcopy (buffer, message, length);
    }

   else  /* no copy required */
  
      buffer = message;

  rc = mpc_send (buffer, length, target_pid, tag, &msg_id);
  if (rc)
     printf ("%d: ERROR SEND (rc=%d), to = %d, length = %d\n",
             pcb.i, rc, to, nbytes);

  mailboxes[to].full       = 1;
  mailboxes[to].msg        = buffer;
  mailboxes[to].msg_length = length;
  mailboxes[to].msg_id     = msg_id;

} /* machine_mpi_send */

     /*********************************************************
     *                                                        *
     *  machine_mpi_receive (int from, char *msg, int len)    *
     *                                                        *
     *  BLOCKING receive                                      *
     *                                                        *
     *  - blocks really until message has been received       *
     *                                                        *
     *********************************************************/

void machine_mpi_receive (from, message, length)

int  from, length;
char *message;


{ /* from becomes message type */

  int rc, source, type, nbytes;

  /* receiving process knows the full specification */

  nbytes = length;

  if (from == -1)

     { /* receiving from any processor */

       type   = DONTCARE;
       source = DONTCARE;
     }

   else

     { /* receiving process knows the full specification */

       type   = from;
       source = from-1;
     }

#ifdef MSG_DEBUG
  printf ("%d: will recv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  rc = mpc_brecv (message, length, &source, &type, &nbytes);

  if (rc)
     printf ("%d: RECEIVE ERROR, from = %d, len = %d, type = %d, rc = %d\n",
              pcb.i, from, length, type, rc);

  if (nbytes != length)
    { printf ("%d: expected msg from %d with length = %d, but got %d\n",
              pcb.i, from, length, nbytes);
    }

#ifdef MSG_DEBUG
  printf ("%d: has recvd from = %d length = %d\n",
           pcb.i, from, length);
#endif

} /* machine_mpi_receive */

     /*********************************************************
     *                                                        *
     *  machine_mpi_sendreceive (...)                         *
     *                                                        *
     *   - blocking send/receive in one step                  *
     *                                                        *
     *********************************************************/

void machine_mpi_sendreceive (to, out_msg, out_len,
                              from, in_msg, in_len)

int to, from;
char *out_msg, *in_msg;
int in_len, out_len;

{ int type;
  int nbytes;
  int source, target;
  int rc;

  /* machine_mpi_send (to, out_msg, out_len, 0);
     machine_mpi_receive (from, in_msg, in_len); */

  source = from - 1;
  target = to   - 1;
  type   = 1391;

  rc = mpc_bsendrecv (out_msg, out_len, source, type,
                      in_msg, in_len, &target, &nbytes);

} /* machine_mpi_sendreceive */

     /*********************************************************
     *                                                        *
     *  int machine_mpi_isend (int to, char *msg, int len)            *
     *                                                        *
     *  NON BLOCKING send                                     *
     *                                                        *
     *********************************************************/

int machine_mpi_isend (to, message, length)

int to, length;
char *message;

{ int tag;
  int target_pid;
  int msg_id;
  int rc;

  extern int mperrno;

  /* non blocking send */

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, length = %d\n",
           pcb.i, to, length);
#endif

  tag        = pcb.i;   /* own processor id is the tag */
  target_pid = to - 1;

  rc = mpc_send (message, length, target_pid, tag, &msg_id);

  if (rc)
    { printf ("%d: mpc_send has fatal error, mperrno = %d\n", pcb.i, mperrno);
      dalib_internal_error ("machine_mpi_isend");
    }

#ifdef MSG_DEBUG
  printf ("%d: call of isend : to = %d, msg_id = %d\n", pcb.i, to, msg_id);
#endif

  return (msg_id);

} /* machine_mpi_isend */


     /*********************************************************
     *                                                        *
     *  int machine_mpi_ireceive (int from, char *msg, int len)       *
     *                                                        *
     *  NON BLOCKING receive                                  *
     *                                                        *
     *********************************************************/

int machine_mpi_ireceive (from, message, length)

int  from, length;
char *message;

{ int msg_id;
  int source, type;
  int rc;

  type = from;
  source = from - 1;

#ifdef MSG_DEBUG
  printf ("%d: will irecv from = %d length = %d\n",
           pcb.i, from, length);
#endif

  rc = mpc_recv (message, length, &source, &type, &msg_id);
  if (rc)
     dalib_internal_error ("machine_mpi_irecv");

#ifdef MSG_DEBUG
  printf ("%d: irecv got msg_id = %d\n", pcb.i, msg_id);
#endif

  return (msg_id);

} /* machine_mpi_receive */

     /*********************************************************
     *                                                        *
     *  int machine_mpi_done (int msg_id)                             *
     *                                                        *
     *  - ask status of an immediate send/receive             *
     *  - msg_id is free if mpi_done == 1                     *
     *                                                        *
     *********************************************************/

int machine_mpi_done (msg_id)

int msg_id;    /* has been given by immediate send or receive */

{ int status;
  int rc;

  status = mpc_status (msg_id); /* returns number of bytes */

  if (status < 0)
     return (0);
   else
     return (1);

} /* machine_mpi_done */

     /*********************************************************
     *                                                        *
     *  void machine_mpi_wait (int msg_id)                            *
     *                                                        *
     *  - wait for completion of an immediate send/receive    *
     *                                                        *
     *********************************************************/

void machine_mpi_wait (msg_id)
int msg_id;

{ int nbytes, help_msg_id;
  int rc;

#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d will be waited\n",
           pcb.i, msg_id);
#endif

  help_msg_id = msg_id;

  rc = mpc_wait (&help_msg_id, &nbytes);
  if (rc)
     dalib_internal_error ("machine_mpi_wait");

#ifdef MSG_DEBUG
  printf ("%d: isend/irecv for msg_id %d is finished\n",
           pcb.i, msg_id);
#endif

} /* machine_mpi_wait */
