/*******************************************************************
*                                                                  *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                 *
*  Copyright   : GMD St. Augustin, Germany                         *
*  Date        : Feb 94                                            *
*  Last Update : Feb 94                                            *
*                                                                  *
*  This Module is part of the DALIB                                *
*                                                                  *
*  INITIALIZATION OF PROGRAM for PVM (Version 3.0)                 *
*                                                                  *
*  MODULE : init.c                                                 *
*                                                                  *
*  Function: Realization of System Dependent Operations            *
*                                                                  *
*  INTERFACE                                                       *
*  =========                                                       *
*                                                                  *
*    void dalib_machine_exit ()                                    *
*                                                                  *
*    void dalib_machine_enroll (int *AllNP)                        *
*                                                                  *
*      - makes machine dependent initialization                    *
*      - return in pcb.i the relative processor position           *
*      - return in pcb.p the number of real existing processes     *
*      - Process 0 or 1 returns in AllNP number of processes       *
*        that will be used really (from environment, command line) *
*                                                                  *
*******************************************************************/

#include <stdio.h>
#include <string.h>
#include "pvm3.h"
#include "dalib.h"

#undef DEBUG

/*******************************************************************
*                                                                  *
*  Task Ids for PVM version 3.0                                    *
*                                                                  *
*******************************************************************/

int tids[MAXP1];

int NP;   /* number of processes that are really used */

/*******************************************************************
*                                                                  *
*  Machine dependent functions                                     *
*                                                                  *
*******************************************************************/

# define INIT_MSGID  9999

          /***********************************************
          *                                              *
          *  dalib_machine_exit ()                       *
          *                                              *
          ***********************************************/

void dalib_machine_exit ()

{

  pvm_exit ();

} /* dalib_machine_exit */

/*********************************************************************
*                                                                    *
*  static void dalib_spawn (name, number, offset)                    *
*                                                                    *
*********************************************************************/

static void dalib_spawn (task_name, no_procs, offset)

char *task_name;
int no_procs;
int offset;

{ int new_processes;  /* number of processes really spawned */

  if (no_procs < 1) return;   /* nothing to do */

  new_processes = pvm_spawn (task_name, (char**)0, PvmTaskDefault, "", 
                             no_procs, tids+offset);

#ifdef DEBUG
  printf ("called pvm_spawn of %s, no_procs = %d, %d spawned\n",
           task_name, no_procs, new_processes);
#endif

  /* check that all tasks have been created */

  if (new_processes != no_procs)

    { if (new_processes < 0)
        printf ("pvm_spawn of %s has error code %d\n", 
                 task_name, new_processes);
       else
        printf ("pvm_spawn of %s creates only %d of %d tasks\n", 
                 task_name, new_processes, no_procs);
      exit(-1);
    }

} /* dalib_spawn */

/*********************************************************************
*                                                                    *
*  static void dalib_initiate ()                                     *
*                                                                    *
*    - this subroutine is only called by the first process           *
*    - this process spawns all the other processes                   *
*                                                                    *
*********************************************************************/

static void dalib_initiate ()

{ int answer;          /* answer number of created tasks */
  int ntasks;          /* number of created tasks        */
  int i;
  int offset;          /* start of new task ids          */
  int rc;              /* return code                    */
  int no_tasks;

  pvm_config (&NP, (int*) 0, (struct pvmhostinfo**) 0);

#ifdef DEBUG
  printf ("configuration has %d hosts\n", NP);
#endif

  /* default number of processors : NP  (number of hosts)
     maximal number of processors : MAXP (arbitrary for PVM */

  eval_arg (NP, MAXP);

  pcb.trace_flag = 0;   /* not realized yet */

  /* first now will now spawn all the other processes  */

  if (no_tasks = pcb.mpmd_flag)

     { NP = 1;  /* we have now one process */

       for (i=1; i <= no_tasks; i++)

        { char *task_name;  
          int  task_procs;

          dalib_get_mpmd_task_info (i, &task_name, &task_procs);

          if (i == 1)

           { /* check that this process is part of the first task */

             if (strcmp (task_name, dalib_program_name) != 0)

               { printf ("starting process %s must belong to first task %s\n",
                          dalib_program_name, task_name);
                 exit (-1);
               }
                
               task_procs -= 1;

             }

          dalib_spawn (task_name, task_procs, NP+1);

          NP += task_procs;
        }

#ifdef DEBUG
        printf ("MPMD mode, spawned %d tasks with %d processes\n",
                 no_tasks, NP);
#endif

        pcb.p = NP;

     }

   else 

    { offset = 2;
      NP = pcb.p;

#ifdef DEBUG
  printf ("SPMD mode, will spawn %d new processes\n", NP - 1);
#endif

      dalib_spawn (dalib_program_name, NP - 1, offset);
    }

  /* check the task ids of the new created processes */

  for (i=2; i<=NP; i++)
    if (tids[i] < 0)
      { pvm_perror ("spawn failure");
        pvm_exit ();
        exit(-1);
      }

  /* send initial message */

  if (NP > 1)

    { pvm_initsend (PvmDataRaw);
      pvm_pkbyte ((char *) &(pcb.p), 6 * sizeof(int), 1);
      pvm_pkbyte ((char *) tids, (NP+1)* sizeof(int), 1);
      rc = pvm_mcast  (tids+2, NP-1, INIT_MSGID);
      if (rc < 0)
        { printf ("pvm_mcast has error code %d\n", rc);
          dalib_internal_error ("stop");
        }
    }

#ifdef DEBUG
  printf ("have sent initial message\n");
#endif

  /* wait for response of nodes */

  for (i=2; i<=NP; i++)
    { pvm_recv (-1, INIT_MSGID-1);
      pvm_upkbyte ((char *) &answer, sizeof(answer), 1);
#ifdef DEBUG
      printf ("Node process %d responded\n", answer);
#endif
    }
}

/*********************************************************************
*                                                                    *
*    enter: gets all relevant information                            *
*                                                                    *
*    called by created processes                                     *
*                                                                    *
*********************************************************************/

void dalib_enter (my_tid)
int my_tid;

{ int i;

  /* recv initial message */

  pvm_recv (-1, INIT_MSGID);
  pvm_upkbyte ((char *) &(pcb.p), 6 * sizeof(int), 1);
  NP = pcb.p;
  pvm_upkbyte ((char *) tids, (NP+1)*sizeof(int), 1);

#ifdef DEBUG
  printf ("process %d got initial message\n", my_tid);
#endif

  /* try to find my_id */

  for (i=2; i<=NP ; i++)
    if (my_tid == tids[i]) { pcb.i = i; break; }

#ifdef DEBUG
  printf ("process %d has the dalib id %d\n", my_tid, pcb.i);
#endif

  /* tell parent that I am ready */

  pvm_initsend (PvmDataRaw);
  pvm_pkbyte ((char *) (&(pcb.i)), sizeof(int), 1);
  pvm_send (tids[1], INIT_MSGID -1);

}

/*********************************************************************
*                                                                    *
*    enroll: starts and makes initializations                        *
*                                                                    *
*********************************************************************/

void dalib_machine_enroll (AllNP)

int *AllNP;

{ int my_tid, parent_tid;

  /* enroll in PVM */

  my_tid = pvm_mytid();

  if (my_tid < 0)

      { /* there is an error , illegal tid */
        if (my_tid == PvmSysErr)
            printf ("pvmd not responding\n");
          else
            printf ("illegal task id for calling process\n");
        exit (-1);
      }

  parent_tid = pvm_parent ();

#ifdef DEBUG
  printf ("process with taskid %d started (parent_tid = %d)\n", 
           my_tid, parent_tid);
#endif

  if (parent_tid < 0)

     { /* I am the first process and have to start the other processes */

       tids [1] = my_tid;

       pcb.i = 1;  /* I am the first node */

       dalib_initiate ();

     }

    else

     { /* I AM a created process */

       dalib_enter (my_tid);

     }

  /* definition of the process control block */

  *AllNP     = NP;
}
