/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.WR                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Jul 97                                                   *
*  Last Update : Aug 98                                                   *
*                                                                         *
*  This Module is part of the UNILIB                                      *
*                                                                         *
*  Module      : dummy                                                    *
*                                                                         *
*  Function    : operations for descriptors dummy - actual                *
*                                                                         *
*  Changes:                                                               *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/**************************************************************************
*                                                                         *
*  PREDICATE dalib_array_same_dim (DimInfo *d_dim, DimInfo *a_dim)        *
*                                                                         *
*  - d_dim and a_dim must be have same global and local size              *
*  - local part must be relatively the same                               *
*  - a_dim (actual) must have bigger overlap than d_dim (dummy)           *
*                                                                         *
**************************************************************************/

int dalib_array_same_dim (d_dim, a_dim)

DimInfo *d_dim, *a_dim;

{ int extent, offset;

  /* same global size */

#ifdef DEBUG
  printf ("%d: same dim, size = (%d:%d-%d:%d), shdw = (%d:%d-%d:%d)\n",
           pcb.i, d_dim->global_size[0], d_dim->global_size[1],
           a_dim->global_size[0], a_dim->global_size[1],
           d_dim->overlap[0], d_dim->overlap[1],
           a_dim->overlap[0], a_dim->overlap[1]);
#endif

  extent = d_dim->global_size[1] - d_dim->global_size[0];
  if (a_dim->global_size[1] - a_dim->global_size[0] != extent) return (0);

  /* same local size */

  extent = d_dim->local_size[1] - d_dim->local_size[0];
  if (a_dim->local_size[1] - a_dim->local_size[0] != extent) return (0);
  if (d_dim->local_size[2] != a_dim->local_size[2]) return (0);

#ifdef DEBUG
  printf ("local (%d:%d:%d = %d:%d:%d), map (%d:%d)\n",
           a_dim->local_size[0], a_dim->local_size[1],
           a_dim->local_size[2], 
           d_dim->local_size[0], d_dim->local_size[1],
           d_dim->local_size[2],
           a_dim->map_flag,
           d_dim->map_flag);
#endif

  /* if it is a mapped dimension, they must have the same distribution */

  if (a_dim->map_flag != d_dim->map_flag) return (0);

  if (d_dim->map_flag)

     { /* both dimensions are mapped, so they should have same distrib. */

       return (0);
     }
  
  /* offset between global and local size must be the same */

  offset = d_dim->global_size[0] - a_dim->global_size[0];
  if (d_dim->local_size[0] - a_dim->local_size[0] != offset) return (0);

  /* check overlap size, d_dim must not have bigger one than a_dim */

  if (d_dim->overlap[0] > a_dim->overlap[0]) return (0);
  if (d_dim->overlap[1] > a_dim->overlap[1]) return (0);

  /* now everything is equal */

  return (1);

} /* dalib_array_same_dim */

/**************************************************************************
*                                                                         *
*  PREDICATE dalib_array_same_shape (dummy_id, actual_id)                 *
*                                                                         *
*   - returns 1 if dummy_id and actual_id are globally/locally conform    *
*     and if actual_id has enough overlap (as expected for dummy)         *
*                                                                         *
**************************************************************************/

int dalib_array_same_shape (dummy_id, actual_id)

array_info dummy_id, actual_id;

{ DimInfo *d_dim, *a_dim;
  int rank, size;
  int i, equal;

#ifdef DEBUG
  printf ("%d: dalib_array_same_shape of %d (rank=%d) and %d (rank=%d)\n", 
           pcb.i, dummy_id, dummy_id->rank, actual_id, actual_id->rank);
#endif

  rank = dummy_id->rank;
  if (rank != actual_id->rank) return (0);

  size = dummy_id->size;
  if (size != actual_id->size) return (0);

  d_dim = dummy_id->dimensions;
  a_dim = actual_id->dimensions;

  equal = 1;

  for (i=0; i<rank; i++)        /* comparison of every dimension */

     if (!dalib_array_same_dim (d_dim+i,a_dim+i)) 

       {
#ifdef DEBUG
         printf ("dim %d is different\n", i+1);
#endif
         equal = 0;
       }

#ifdef DEBUG
  printf ("%d: same shape, result = %d\n", pcb.i, equal);
#endif

  return (equal);

} /* dalib_array_same_shape */

/**************************************************************************
*                                                                         *
*   int dalib_use_actual_data (dummy_id, actual_id)                       *
*                                                                         *
*    - returns true if data of actual can be used directly                *
*    - condition 1 : actual_id has same local shape as dummy_id           *
*    - condition 2 : if dummy_id is shared, actual_id must be shared too  *
*                                                                         *
**************************************************************************/
 
int dalib_use_actual_data (dummy_id, actual_id)

array_info dummy_id, actual_id;

{ if (!dalib_array_same_shape (dummy_id, actual_id))

      return (0);

  /* if dummy is not shared but array should be shared */

  if (dummy_id->SharedInfo != NO_SHARED)

     { if (actual_id->SharedInfo == NO_SHARED)
          return (0);
     }

  /* if dummy needs remote access but array should be shared */

  if (dummy_id->RemoteInfo != NO_REMOTE)

     { if (actual_id->RemoteInfo == NO_REMOTE)
          return (0);
     }

  return (1);

}  /* dalib_use_actual_data */

/**************************************************************************
*                                                                         *
*  int dalib_match_global_shape (array_info dsp1, array_info dsp2)        *
*                                                                         *
*    -1  : dsp1 and dsp2 do not match at all                              *
*     0  : same shape and same dimensions                                 *
*     1  : same shape, but different dimensions                           *
*                                                                         *
**************************************************************************/
 
int dalib_match_global_shape (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dim1, *dim2;
  int i, rank, size;

  int result;

#ifdef DEBUG
  printf ("%d: dalib_match_global_shape of %p (rank=%d) and %p (rank=%d)\n", 
           pcb.i, dsp1, dsp1->rank, dsp2, dsp2->rank);
#endif

  /* Step 1 : verification of the same rank  */

  rank = dsp1->rank;
  if (rank != dsp2->rank) return (-1);

  size = dsp1->size;
  if (size != dsp2->size) return (-1);

  dim1 = dsp1->dimensions;
  dim2 = dsp2->dimensions;

  result = 0;

  for (i=0; i<rank; i++, dim1++, dim2++)   /* comparison of every dimension */

       { int lb1, ub1, lb2, ub2;

         lb1 = dim1->global_size[0];
         ub1 = dim1->global_size[1];
         lb2 = dim2->global_size[0];
         ub2 = dim2->global_size[1];

         if ((ub1 - lb1) != (ub2 - lb2))

            { result = -1; break; }

         if (lb1 != lb2) result = 1;

       }

  return (result);

} /* dalib_match_global_shape */

/**************************************************************************
*                                                                         *
*  int dalib_sufficient_shadow (array_info dsp1, array_info dsp2)         *
*                                                                         *
*     1  : dsp2 has more or equal shadow than expeected by dsp1           *
*     0  : dsp2 has less shadow than expected here                        *
*                                                                         *
**************************************************************************/
 
int dalib_sufficient_shadow (dsp1, dsp2)

array_info dsp1, dsp2;

{ DimInfo *dim1, *dim2;
  int i, rank;

  int result;

#ifdef DEBUG
  printf ("%d: dalib_sufficient_shadow of %p (rank=%d) and %p (rank=%d)\n", 
           pcb.i, dsp1, dsp1->rank, dsp2, dsp2->rank);
#endif

  /* Step 1 : verification of the same rank, otherwise problems  */

  rank = dsp1->rank;
  if (rank != dsp2->rank) return (0);

  dim1 = dsp1->dimensions;
  dim2 = dsp2->dimensions;

  result = 0;

  for (i=0; i<rank; i++, dim1++, dim2++)   /* comparison of every dimension */

    { if (dim1->overlap[0] > dim2->overlap[0]) return (0);
      if (dim1->overlap[1] > dim2->overlap[1]) return (0);
    }

  return (1);

} /* dalib_sufficient_shadow */

/**************************************************************************
*                                                                         *
*   int dalib_is_dummy_data (array_id, dummy_id)                          *
*                                                                         *
*    - returns true if array_id has still same data as dummy              *
*                                                                         *
**************************************************************************/
 
int dalib_is_dummy_data (array_id, dummy_id)

array_info array_id, dummy_id;

{ if (array_id->f_data != dummy_id->f_data) return (0);  /* cannot be used */

  if (array_id->dsp_status_flag == DSP_OWN_DATA)

     return (0);  /* has new data allocated */

  return (1);   /* now we are sure that it is the same data */

} /* dalib_is_dummy_data */

/*******************************************************************
*                                                                  *
*  void dalib_get_actual_info (array_info dummy_dsp, actual_dsp)   *
*                                                                  *
*  - dummy inherits the data directly from the actual argument     *
*  - dummy inherits also the overlap size of the actual argument   *
*                                                                  *
*******************************************************************/

void dalib_get_actual_info (dummy_dsp, actual_dsp)

array_info dummy_dsp, actual_dsp;

{ DimInfo *local_dims, *dummy_dims;
  int i, rank;
 
  local_dims = dummy_dsp->dimensions;
  dummy_dims = actual_dsp->dimensions;
 
  rank = dummy_dsp->rank;   /* == actual_dsp->rank */

  /* inherit data pointer */

  dummy_dsp->f_data          = actual_dsp->f_data;
  dummy_dsp->dsp_status_flag = actual_dsp->dsp_status_flag;

  if (actual_dsp->dsp_status_flag == DSP_OWN_DATA)
    dummy_dsp->dsp_status_flag = DSP_PTR_DATA;  /* is default, but make sure */

  /* inherit size of overlap area */

  for (i=0; i<rank; i++, local_dims++, dummy_dims++)

     { local_dims->overlap[0] = dummy_dims->overlap[0];
       local_dims->overlap[1] = dummy_dims->overlap[1];
     }

} /* dalib_get_actual_info */

/**************************************************************************
*                                                                         *
*  void dalib_copy_info (array_info dsp1, array_info dsp2)                *
*                                                                         *
*  - copy the valid info from descriptor dsp2 to dsp1                     *
*                                                                         *
**************************************************************************/

void dalib_copy_valid_info (dsp1, dsp2, destroy_flag)

array_info dsp1, dsp2;

int destroy_flag;

{ array_info a1, a2;

  if (dalib_is_section_info (dsp1))
     a1 = ((section_info) dsp1)->array_id;
    else
     a1 = dsp1;
    
  if (dalib_is_section_info (dsp2))
     a2 = ((section_info) dsp2)->array_id;
    else
     a2 = dsp2;
    
  if (a2->trace)

     { if (a1->trace)

         a1->reuse_bits = a2->reuse_bits;
   
        else if (destroy_flag)

         FUNCTION(dalib_array_set_dirty) (&a2);

     }

} /* dalib_copy_valid_info */

/*******************************************************************
*                                                                  *
*  ERROR message for COMMON conflicts                              *
*                                                                  *
*******************************************************************/

void dalib_common_mismatch (local_dsp, common_dsp, msg)

array_info local_dsp, common_dsp;
char msg[];

{ char msg1[256];

  sprintf (msg1, "COMMON mismatch : %s\n", msg);

  dalib_internal_error (msg1);

  printf ("%d: descriptor found in COMMON = \n", pcb.i);
  dalib_print_array_info (common_dsp);

  printf ("%d: descriptor expected = \n", pcb.i);
  dalib_print_array_info (local_dsp);

  dalib_stop ();

} /* common_mismatch_error */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_attach_in) (local_dsp, global_dsp)               *
*                                                                  *
*   - attach to a global array (COMMON) given by global_dsp        *
*   - assumed specification is given by local_dsp                  *
*   - verification of same size/type                               *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_attach_in) (local_dsp, global_dsp)

array_info *local_dsp;
array_info *global_dsp;

{ int info;

  /* check correct descriptors */

  if (!dalib_is_array_info (*global_dsp))

     { char msg[128];

       /* global_dsp not defined in main / or error */

       sprintf (msg, "first array descriptor for common array %s",
                (*local_dsp)->name);

       dalib_internal_error (msg);

       *global_dsp = *local_dsp;

       return;
     }

  if (!dalib_is_array_info (*local_dsp))

     { /* that must be indeed a serious error */

       dalib_internal_error ("attach in: illegal local descriptor");
       dalib_stop ();

     }

  /* make some verifications with the defined local descriptor */

  info = dalib_match_global_shape (*local_dsp, *global_dsp);

  if (info != 0) 

      dalib_common_mismatch (*local_dsp, *global_dsp, 
                             "not same shape/rank/dims");

  if (!dalib_sufficient_shadow (*local_dsp, *global_dsp))

      dalib_common_mismatch (*local_dsp, *global_dsp, 
                             "insufficient shadow (use !HPF$ SHADOW)");

   /* verifications okay, so free the local descriptor and set global one */

#ifdef DEBUG
   printf ("%d attach in : local dsp = %p, global dsp = %p\n",
            pcb.i, *local_dsp, *global_dsp);
#endif

   dalib_free_descriptor (*local_dsp);

   *local_dsp = *global_dsp; 

} /* dalib_attach_in */

/**********************************************************************
*                                                                     *
*  bool dalib_are_dynamic (array_info actual_id, array_info dummy_id) *
*                                                                     *
**********************************************************************/

int dalib_are_dynamic (actual_id, dummy_id)

array_info actual_id, dummy_id;

{ if (!dalib_is_array_info (actual_id)) return (0);

  if (!actual_id->dynamic) return (0);
  if (!dummy_id->dynamic) return (0);

  return (1);

} /* dalib_are_dynamic */
