/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Sep 93                                                   *
*  Last Update : Mar 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : topology                                                 *
*                                                                         *
*                A topology is an ordering of a group in a one-, two-     *
*                or three-dimensional array                               *
*                                                                         *
*  group:    1, 3, 5, 7, 9 , 11                                           *
*                                                                         *
*  topology:    1   5    9      shape : [2, 3]                            *
*               3   7   11                                                *
*                                                                         *
*  Function: Definition of Processor Topologies                           *
*                                                                         *
*    - handles many different torus specifications                        *
*    - supports one-, two- or three-dimensional tori                      *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
*  void FUNCTION(dalib_top_create) (int *topid, int *first, int *rank,    *
*                                   int *n1, int *n2, int *n3)            *
*                                                                         *
*   - all the routines define new topologies that will be identified      *
*     with the integer vaule of 'topid'                                   *
*                                                                         *
*   - note: some values can be non-positive for free choices              *
*                                                                         *
*  Export :  local to DALIB                                               *
*  ========================                                               *
*                                                                         *
*  int dalib_top_make (int group, int rank, int shape[])                  *
*                                                                         *
*     - defines a topology, described by rank and shape, for the          *
*       processors of the group identified by group                       *
*                                                                         *
*     - returns an internal identification (integer) by which the         *
*       topology is identified for query operations, topid                *
*                                                                         *
*  void dalib_top_info (int topid, int dim, int *N, int *I)               *
*                                                                         *
*    - number of processors and relative position for dimension 'dim'     *
*      in processors topology specified by 'topid'                        *
*                                                                         *
*  int dalib_top_size (int topid)                                         *
*                                                                         *
*    - returns number of processors in topology topid                     *
*                                                                         *
*  int dalib_top_elem (int topid, int k)                                  *
*                                                                         *
*    - processor id of kth processor in topology (0 <= k < top_size)      *
*                                                                         *
*  int dalib_in_topology (topid)                                          *
*                                                                         *
*   - returns true if processor is in topology topid                      *
*                                                                         *
* Changes:                                                                *
* ========                                                                *
*                                                                         *
*  03/96 : all routines are extended to deal with arbitrary dimensions    *
*                                                                         *
**************************************************************************/

#include <stdio.h> 
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
* Definition of the Data Structures for Topologie/Processor (Sub-)Groups  *
*                                                                         *
**************************************************************************/

     /*********************************************************
     *                                                        *
     *  Definition of a processor group / topology            *
     *                                                        *
     *  RESTRICTION !!!   must always be contiguous           *
     *                                                        *
     *********************************************************/

typedef struct

  { /* global information: consistent between all processors            */

    int group;            /* subset of processors belonging to topology */
    int rank;             /* rank of the topology                       */
    int shape [MAX_RANK]; /* shape of the topology                      */

    /* local information: about position of me in this group            */

    int within;                    /* 0 : processor itself is not in 
                                          this topololgy                */

    int relpos    [MAX_RANK];      /* my own coordinates in this top    */

    int sub_group [MAX_RANK];      /* subgroup for every row, col, ...
                                      has shape[i] processors           */

  } topology_entry;

     /*********************************************************
     *                                                        *
     *  All available topologies (ever defined)               *
     *                                                        *
     *********************************************************/

static topology_entry *topologies [MAX_TOPS];
static int            topology_top = 0;

/**************************************************************************
*                                                                         *
*  ERROR Messages (if CHECK is defined)                                   *
*                                                                         *
*   void dalib_top_check_valid (int topid)                                *
*   void dalib_top_check_rank (int topid, int dim)                        *
*                                                                         *
**************************************************************************/

#ifdef CHECK

        /******************************************************
        *                                                     *
        ******************************************************/

void dalib_top_check_valid (topid)

int topid;

{  char msg[80];
   if ((topid < 1) || (topid > topology_top))
   { sprintf (msg,"dalib_top_check_valid, %d is illegal (must be in 1:%d)\n",
       topid, topology_top);
     dalib_internal_error (msg);
     dalib_stop ();
   }

} /* dalib_top_check_valid */

        /******************************************************
        *                                                     *
        ******************************************************/

void dalib_top_check_rank (topid, dim)

int topid, dim;

{  int trank;
   char msg[150];
   dalib_top_check_valid(topid);
   trank = topologies[topid-1]->rank;
   if ((dim <= 0) || (dim > trank))
     { sprintf (msg, "topology - rank : dim (%d) of top %d not in 1:%d\n",
       dim, topid, trank);
       dalib_internal_error (msg);
       dalib_stop ();
     }

} /* dalib_top_check_rank */

        /******************************************************
        *                                                     *
        ******************************************************/

void dalib_top_rank_error (message, must_rank, is_rank)
char message[];
int must_rank, is_rank;
{ char msg[80];
  sprintf (msg, "(topology, at %s): rank is %d, but must be %d\n",
       message, is_rank, must_rank);
  dalib_internal_error (msg);
} /* dalib_top_rank_error */

#endif 

     /*********************************************************
     *                                                        *
     *  Printing out all topologies                           *
     *                                                        *
     *********************************************************/

void dalib_top_print ()

{ topology_entry *entry;

  int k, i, rank;
  int kind, first, step, size;
  int *elem;

  for (k=0; k < topology_top; k++)

   { printf ("%d: Topology %d: ", pcb.i, k+1);

     entry = topologies[k];

     dalib_group_info (entry->group, &size, &kind, &first, &step, &elem);

     if (kind == 0)

        { if (step == 1)
             printf ("group = %d (%d:%d),", entry->group,
                      first, first+size-1);
           else
             printf ("group = %d (%d:%d:%d),", entry->group,
                      first, first+(size-1)*step, step);
        }

       else

        { printf ("group = %d (", entry->group);
          for (i=0; i < size-1; i++) printf ("%d,", elem[i]);
          printf ("%d),", elem[size-1]);
        }

     printf (" shape = [");

     rank  = entry->rank;
     for (i = 1; i<=rank; i++)
        printf ("%d ", entry->shape[i-1]);
     printf ("], ");

     if (entry->within == 0)
        printf (" --  me not within");
      else
        { printf ("my pos = ");
          for (i = 1; i<=rank; i++)
            printf ("%d ", entry->relpos[i-1]);
        }
     printf ("\n");
   }
}

/**************************************************************************
*                                                                         *
*  IMPORT of module 'linear':  transformation linear - array coordinates  *
*                                                                         *
*  void dalib_linear1_relpos (int pos, int rank, int shape[rank],         *
*                             int relpos[rank]                   )        *
*                                                                         *
*  int dalib_linear1_abspos (int rank, int shape[], int relpos[])         *
*                                                                         *
**************************************************************************/

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_top_get_dim_elements (entry, dim, size, elements)

topology_entry *entry;

int dim;
int *size;        /* number of processors with the same coordinates */
int elements[];   /* will contain the absolute processor ids        */

{ int first, step;

  int my_pos[MAX_RANK];

  int i, rank;

  rank  = entry->rank;
  *size = entry->shape[dim-1];

  for (i=0; i<rank; i++)
      my_pos[i] = entry->relpos[i];

  my_pos[dim-1] = 1;
  first = dalib_linear1_abspos (rank, entry->shape, my_pos);

  my_pos[dim-1] = 2;   /* must not really exist, but works */
  step  = dalib_linear1_abspos (rank, entry->shape, my_pos);
  step  = step - first;

  dalib_group_get_elements (entry->group, *size, first, step, elements);

#ifdef DEBUG
  printf ("%d: top_get_dim_elements (gid=%d, size=%d, first=%d, step= %d)\n",
           pcb.i, entry->group, *size, first, step);
#endif

} /* dalib_top_get_dim_elements */

     /*************************************************************
     *                                                            *
     *  defining groups for every dimension of the topology       *
     *                                                            *
     *************************************************************/

static void dalib_define_groups (topid)

int topid;

{ int rank, dim;
  topology_entry *entry;

  entry = topologies[topid-1];

  rank = entry->rank;   /* rank of the topology */

  /* every processor will belong to a group for one dimension */

  for (dim=0; dim<rank; dim++)

   {  int size;
      int elements [MAXP];
      int gid;

      dalib_top_get_dim_elements (entry, dim+1, &size, elements);

      gid    = dalib_group_create (size, elements);

      entry->sub_group [dim] = gid;

   }

} /* dalib_define_groups */

         /****************************************************
         *                                                   *
         *  dalib_top_setentry (topology)                    *
         *                                                   *
         *   - put entry on the big stack                    *
         *                                                   *
         ****************************************************/

void dalib_top_setentry (entry)
topology_entry *entry;

{ /* set the new entry at the top of the table */

  if (topology_top >= MAX_TOPS)
    { fprintf (stderr, "too many topologies, limit is %d\n", MAX_TOPS);
      exit(-1);
    }

  topologies[topology_top] = entry;
  topology_top++;

  dalib_define_groups (topology_top);

}

         /****************************************************
         *                                                   *
         *  dalib_top_set (entry, group, rank, shape[])      *
         *                                                   *
         ****************************************************/
 
static void dalib_top_set (entry, group, rank, shape)

topology_entry *entry;
int group, shape[];
 
{ int i;       /* counter variable                          */
  int pid;     /* processor id in group, 1<=pid<=group_size */
 
  entry->rank = rank;

  for (i=0; i<rank; i++)
     entry->shape[i] = shape[i];

  entry->group = group;

  /* get my own position in this new topology */

  pid = dalib_group_position (group, pcb.i);   /* get my position in group */

  if (pid)

     { entry->within = 1;
       dalib_linear1_relpos (pid, rank, shape, entry->relpos);
     }

    else

     entry->within   = 0;
 
} /* dalib_top_set */

     /*********************************************************
     *                                                        *
     *  Make a new entry at the end of the topology table     *
     *                                                        *
     *  dalib_top_define (group, rank, shape)                 *
     *                                                        *
     *********************************************************/

/**************************************************************************
*                                                                         *
*  void dalib_top_define (int group, int rank, int shape[])               *
*                                                                         *
*    - define a new topology for processors in group                      *
*                                                                         *
**************************************************************************/

static void dalib_top_define (group, rank, shape)

int group, rank, shape[];

{ topology_entry *entry;
  int i;

  entry = (topology_entry *) dalib_malloc (sizeof (topology_entry),
                                           "dalib_top_define");

#ifdef DEBUG
  printf ("%d: dalib_top_define, group = %d, rank = %d, shape = ",
           pcb.i, group, rank);
  for (i=0; i<rank; i++) printf ("%d ", shape[i]);
  printf ("\n");
#endif

  dalib_top_set (entry, group, rank, shape);
  dalib_top_setentry (entry);

} /* dalib_top_define */

     /*********************************************************
     *                                                        *
     *  int dalib_find_topology (group, rank, int shape[])    *
     *                                                        *
     *  Searching a topology entry in the table               *
     *                                                        *
     *  (pos == topology_top) stands for not found            *
     *                                                        *
     *  - called by dalib_top_create                          *
     *                                                        *
     *********************************************************/

static int dalib_find_topology (group, rank, shape)

int rank, group;
int shape[];

{ int found, pos;
  topology_entry *entry;
  int i;

  pos = 0; found = 0;

  while ((pos < topology_top) && (!found))

   { entry = topologies[pos];

     found = (entry->rank == rank);

     if (found)
        found = (entry->group == group);

     for (i=0; i<rank; i++)
        if (found) 
           found = (entry->shape[i] == shape[i]);

     if (!found) pos++;
   }

   return (pos);

} /* dalib_find_topology */

/**************************************************************************
*                                                                         *
*  int dalib_top_make (int group, int rank, int shape[])                  *
*                                                                         *
*     - defines a topology, described by rank and shape, for the          *
*       processors of the group identified by group                       *
*                                                                         *
*     - returns an internal identification (integer) by which the         *
*       topology is identified for query operations                       *
*                                                                         *
**************************************************************************/

int dalib_top_make (group, rank, shape)

int group, rank, shape[];

{ int id;

  id = dalib_find_topology (group, rank, shape);

  /* if not found, then define a new topology */

  if (id == topology_top)

     dalib_top_define (group, rank, shape);

  return (id+1);

} /* dalib_top_make */

     /*********************************************************
     *                                                        *
     *  Initialization of Topogies                            *
     *                                                        *
     *   Topology 1 :    NP                                   *
     *   Topology 2 :    N1 x N2                              *
     *   Topology 3 :    N1 x N2 x N3                         *
     *                                                        *
     *  Every processor is contained (within) the topology    *
     *                                                        *
     *********************************************************/

void dalib_top_exit ()

{ int i;

  for (i=0; i<topology_top; i++)
     dalib_free (topologies[i], sizeof(topology_entry));
     
  dalib_pop_context ();

} /* dalib_top_exit */

     /*********************************************************
     *                                                        *
     *  local export for DALIB                                *
     *                                                        *
     *********************************************************/

void dalib_top_info (topid, dim, N, I)

int topid, dim;
int *N, *I;

{ topology_entry *entry;

  /* topology 0 is topology 1 + host processor */

  if (topid == -1)
     { *N = 1;
       if (pcb.i == 0) 
          *I = 1;
         else
          *I = 0;
       return;
     }

  if (topid == 0)
     { dalib_top_info (1, dim, N, I);
       return;
     }

#ifdef CHECK
   dalib_top_check_rank (topid, dim);
#endif

  entry = topologies[topid-1];
  *N = entry->shape[dim-1];
  if (entry->within)
    *I = entry->relpos[dim-1];
   else
    *I = 0;
}

int dalib_top_rank (topid)
int topid;

{ 
  if (topid == 0) return (0);
  if (topid == -1) return (0);
  dalib_top_check_valid (topid);
  return (topologies[topid-1]->rank);
}

int dalib_top_group (topid)
int topid;

{ 
  if (topid == 0) return (0);
  if (topid == -1) return (0);
  dalib_top_check_valid (topid);
  return (topologies[topid-1]->group);
}

void dalib_top_shape (topid, shape)
int shape[];

{ int i;
  topology_entry *entry;

  if (topid == 0) return;
  if (topid == -1) return;
  dalib_top_check_valid (topid);
  if (topid == 0) return;
  if (topid == -1) return;
  entry = topologies[topid-1];
  for (i=0; i<entry->rank; i++) shape[i] = entry->shape[i];
}

int dalib_top_first (topid)

int topid;

{ 
  if (topid == 0) return (1);
  if (topid == -1) return (1);

#ifdef CHECK
  dalib_top_check_valid (topid);
#endif

  return dalib_group_first (topologies[topid-1]->group);

} /* dalib_top_first */

         /****************************************************
         *                                                   *
         *  int dalib_top_size (int topid)                   *
         *                                                   *
         *  - returns number of processors in topology       *
         *                                                   *
         ****************************************************/

int dalib_top_size (topid)
int topid;

{ topology_entry *entry;
  int i, size, rank;

  if (topid == 0) return (pcb.p);  /* ATTENTION: is important to say */
  if (topid == -1) return (1);

  dalib_top_check_valid (topid);
  entry = topologies[topid-1];
  size = 1;
  rank = entry->rank;
  for  (i=0; i<rank; i++)
     size *= entry->shape[i];
  return (size);
} /* dalib_top_size */

/**************************************************************************
*                                                                         *
*   int dalib_top_elem (int topid, int k)                                 *
*                                                                         *
*   - returns k-th element of a topology (0 <= k < size(topid))           *
*                                                                         *
**************************************************************************/

int dalib_top_elem (topid, k)
int topid, k;

{ /* no check for 0 <= k < top_size (topid) */

  if (topid == 0) return (1+k);   
  if (topid == -1) return (1);

  dalib_top_check_valid (topid);
  return dalib_group_element (topologies[topid-1]->group, k+1);

} /* dalib_top_elem */

/**************************************************************************
*                                                                         *
*   int dalib_top_subgroup (int topid, int topdim)                        *
*                                                                         *
*   - returns subgroup of topology topid for all processors with          *
*     same coordinates except in dimension topdim                         *
*                                                                         *
*   - calling processor must be in this subgroup                          *
*                                                                         *
**************************************************************************/

int dalib_top_subgroup (topid, topdim)

int topid, topdim;

{ topology_entry *entry;
  int gid;

#ifdef CHECK
  dalib_top_check_rank (topid, topdim);
#endif

  entry = topologies[topid-1];
  gid   = entry->sub_group[topdim-1];

  return (gid);

}  /* dalib_top_subgroup */

/**************************************************************************
*                                                                         *
*  FUNCTION(dalib_top_create) (int *topid, array_info *top_dsp,           *
*                              int *first)                                *
*                                                                         *
*   - returns in topid an internal id for the new defined topology        *
*   - shape is needed for building subsections                            *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_top_create) (top_id, top_dsp, first)

int        *top_id;     /* out: ADAPTORs internal topology identification   */
array_info *top_dsp;    /* in : array descriptor for processor array of HPF */
int *first;             /* in : first processor in current context          */

{ int rank;
  int current_group;
  int new_group;

  int N, P;

  int i;
  int shape[MAX_RANK];

  char msg[256];

  /* get shape / rank of the processor array */

  dalib_secarray_shape (top_dsp, &rank, shape);

  if (rank > MAX_RANK)

    { sprintf (msg, "top_create with rank = %d, maximal rank is %d",
                     rank, MAX_RANK);
      dalib_internal_error (msg);
      dalib_stop ();
    }

  /* with the array descriptor we can get the shape to define topology */

  /* step 1 : make useful values of topology */

  current_group = dalib_context_group ();

  N  = dalib_group_size (current_group);

  P  = N - *first + 1;   /* available processors */

  dalib_make_torus (P, rank, shape);

  /* now we have found the final value of shape */

  P = 1;
  for (i=0; i<rank; i++) P *= shape[i];

  /* step 2 : verify that there are sufficient processors */

  if ((P <= 0) || (P + *first - 1 > N))
    { fprintf (stderr, 
       "DALIB: top_create (P=%d, first = %d) fails (only %d processors)\n",
         P, *first, N);
      exit(-1);
    }

  /* define a new group of processors for the new topology */

  new_group = dalib_subgroup1_create (current_group, P, *first, 1);

  /* make the new topology (defines it if not available) */

  *top_id = dalib_top_make (new_group, rank, shape);

  /* hide the topology id also in the array descriptor */

  (*top_dsp)->size = *top_id;

#ifdef DEBUG
  dalib_top_print ();
#endif

} /* dalib_top_create */ 

/**************************************************************************
*                                                                         *
*  int dalib_subgroup_make (int group_id, int first, int rank,            *
*                           int rank, int n[], int inc[])                 *
*                                                                         *
*  - make a subgroup within group_id by the specified processors          *
*                                                                         *
**************************************************************************/

int dalib_subgroup_make (group_id, first, rank, n, inc)

int group_id;
int first;
int rank;
int n[];
int inc[];

{ int new_rank;
  int new_n[MAX_DIMENSIONS];
  int new_inc [MAX_DIMENSIONS];

  dalib_linear_compress (rank, n, inc, &new_rank, new_n, new_inc);

  if (new_rank == 0)

     return dalib_subgroup1_create (group_id, 1, first, 1);

   else if (new_rank == 1)

     return dalib_subgroup1_create (group_id, new_n[0], first, new_inc[0]);

   else

     { int p_ids [MAXP];
       int i, size;

       dalib_linear_set (first, new_rank, new_n, new_inc, &size, p_ids);
  
       return dalib_subgroup_create (group_id, size, p_ids);
     }

} /* dalib_subgroup_make */

static void add_shape (st_shape, st_rank, dim_size)

int st_shape[];
int *st_rank;
int dim_size;

{ int i;

  for (i=(*st_rank); i>0; i--) st_shape[i] = st_shape[i-1];
  st_shape [0] = dim_size;
  (*st_rank)++;

} /* add_shape */

/**************************************************************************
*                                                                         *
*  void dalib_sg_set (int *sg_size, int sg_elems[],                       *
*                                                                         *
*    - add new dimension for subtopology/subgroup specification           *
*                                                                         *
**************************************************************************/

static void dalib_sg_set (sg_size, sg_elems, st_rank, st_shape,
                          full_dim, kind, lb, ub, str, elems)

int *sg_size;
int sg_elems[];

int *st_rank;
int st_shape[];

DimInfo *full_dim;
int kind;
int lb, ub, str;
int elems[];

{ int N;
  int N1, N2;

  int i, j;

#ifdef DEBUG
  printf ("%d: dalib_sg_set, sg_size = %d\n", pcb.i, *sg_size);
#endif

  N1 = full_dim->global_size[0];
  N2 = full_dim->global_size[1];
  N  = N2 - N1 + 1;

  if (kind == 0)

    { /* single element specified */

      if ((lb < N1) || (lb > N2))

        { char msg[128];
          sprintf (msg, "subtop_create: %d out of range %d:%d", lb, N1, N2);
          dalib_internal_error (msg);
          dalib_stop ();
        }

      for (i=0; i<(*sg_size); i++) 
         sg_elems[i] = (sg_elems[i]-1) * N + lb + 1 - N1;

    }

   else if (kind == 1)

     { /* range specified */

       int new_size;
       int dim_size;
       int old_elems [MAXP];
       
       /* transfer the old positions to new ones */

       for (i=0; i<(*sg_size); i++) old_elems[i] = (sg_elems[i] - 1) * N;

       new_size = 0;
       dim_size = 0;

       if ((lb < N1) || (lb > N2) || (ub < N1) || (ub > N2) || (str == 0)) 

        { char msg[128];
          sprintf (msg, "subtop_create: %d:%d:%d out of range %d:%d", 
                   lb, ub, str, N1, N2);
          dalib_internal_error (msg);
          dalib_stop ();
        }

       for (i=0; i<(*sg_size); i++)

         { if (str > 0)
             { for (j=lb; j<=ub; j=j+str)
                  sg_elems[new_size++] = old_elems[i] + j + 1 - N1;
             }

            else if (str < 0)
             { for (j=lb; j>=ub; j=j+str)
                  sg_elems[new_size++] = old_elems[i] + j + 1 - N1;
             }
           if (i==0) dim_size = new_size;
         }

       add_shape (st_shape, st_rank, dim_size);

       *sg_size = new_size;

     }

   else if (kind == 2)

    { /* indirection specified */

      int new_size;
      int old_elems [MAXP];
      
      for (i=0; i<(*sg_size); i++) old_elems[i] = (sg_elems[i] - 1) * N;

      for (i=0; i<ub; i++)

      { int val;
        int k;

        val = elems[i];

        if ((val < N1) || (val > N2))

          { char msg[128];
            sprintf (msg, "subtop_create: %d out of range %d:%d", 
                           val, N1, N2);
            dalib_internal_error (msg);
            dalib_stop ();
          }

        for (k=0; k<i; k++)
          if (val == elems[k])
            { char msg[128];
              sprintf (msg, "subtop_create: %d double value", val);
              dalib_internal_error (msg);
              dalib_stop ();
            }

     }  /* for all elements */

#ifdef DEBUG
       printf ("%d: dalib_sg_set (irregular, size=%d)\n", pcb.i, ub);
#endif

       new_size = 0;

       for (i=0; i<(*sg_size); i++)
         for (j=0; j<ub; j++)
            sg_elems[new_size++] = old_elems[i] + elems[j] + 1 - N1;

       *sg_size = new_size;

       add_shape (st_shape, st_rank, ub);

     }

   else

     { dalib_internal_error ("subtop_create, illegal kind");
       dalib_stop ();
     }

} /* dalib_sg_set */

/**************************************************************************
*                                                                         *
*  Functions for defining sub topologies                                  *
*                                                                         *
*  void FUNCTION(dalib_subtop_create) (subtopid, topid,                   *
*                               kind1, lb1, ub1, inc1,                    *
*                               ...                                       *
*                               kind7, lb7, ub7, inc7)                    *
*                                                                         *
*    - returns an internal descriptor for a subtopology                   *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_subtop_create) (subtop_id, top_dsp,
                            kind1, lb1, ub1, str1, kind2, lb2, ub2, str2,
                            kind3, lb3, ub3, str3, kind4, lb4, ub4, str4,
                            kind5, lb5, ub5, str5, kind6, lb6, ub6, str6,
                            kind7, lb7, ub7, str7)

int        *subtop_id;
array_info *top_dsp;

int *kind1, *lb1, *ub1, *str1, *kind2, *lb2, *ub2, *str2,
    *kind3, *lb3, *ub3, *str3, *kind4, *lb4, *ub4, *str4,
    *kind5, *lb5, *ub5, *str5, *kind6, *lb6, *ub6, *str6,
    *kind7, *lb7, *ub7, *str7;

{ int top_id;
  int top_rank;
  int top_group;

  int st_rank;
  int st_shape[MAX_RANK];
  int st_group;

  int sg_size;
  int sg_elems[MAXP];

  int i;

  DimInfo *top_dims;

  top_id    = (*top_dsp)->size;    /* topology id was hidden in descriptor */
  top_rank  = dalib_top_rank (top_id);
  top_group = dalib_top_group (top_id);
  top_dims  = (*top_dsp)->dimensions;

  st_rank     = 0;
  sg_size     = 1;
  sg_elems[0] = 1;

  switch (top_rank) {

     case 7 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 6, *kind7, *lb7, *ub7, *str7, lb7);
     case 6 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 5, *kind6, *lb6, *ub6, *str6, lb6);
     case 5 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 4, *kind5, *lb5, *ub5, *str5, lb5);
     case 4 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 3, *kind4, *lb4, *ub4, *str4, lb4);
     case 3 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 2, *kind3, *lb3, *ub3, *str3, lb3);
     case 2 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 1, *kind2, *lb2, *ub2, *str2, lb2);
     case 1 : dalib_sg_set (&sg_size, sg_elems, &st_rank, st_shape,
                            top_dims + 0, *kind1, *lb1, *ub1, *str1, lb1);

  } /* switch */

#ifdef DEBUG
  printf ("%d: subtop_create, new group of group (id=%d): ",
          pcb.i, top_group);
  for (i=0; i<sg_size; i++)
     printf ("%d ", sg_elems[i]);
  printf ("\n");
  printf ("%d: subtop_create, shape : ", pcb.i);
  for (i=0; i<st_rank; i++)
     printf ("%d ", st_shape[i]);
  printf ("\n");
#endif

  /* create the new processor subgroup */

  st_group = dalib_subgroup_create (top_group, sg_size, sg_elems);

  /* special case : single processor -> make one-dimensional proc array */

  if (st_rank == 0)

     { st_rank     = 1;
       st_shape[0] = 1;
     }

  *subtop_id = dalib_top_make (st_group, st_rank, st_shape);

#ifdef DEBUG
  printf ("%d: subtop (group=%d) of top(id=%d,rank=%d,group=%d) -> %d\n",
           pcb.i, st_group, top_id, top_rank, top_group,
           *subtop_id);
#endif

#ifdef DEBUG
  dalib_top_print ();
#endif

} /* dalib_subtop_create */

     /*************************************************************
     *                                                            *
     *  Getting absolute processor ids in topology                *
     *                                                            *
     *  Example (n1=2, n2=3)                                      *
     *                                                            *
     *            1        2        3                             *
     *          |----------------------|                          *
     *       1  | first    3        5  |                          *
     *          |                      |                          *
     *       2  | 2        4        6  |                          *
     *          ------------------------                          *
     *                                                            *
     *   p  =  (i2 - 1) * n1 + i1                                 *
     *   p  =  (i3 - 1) * n1 * n2  + (i2 - 1) * n1 + i1           *
     *                                                            *
     *************************************************************/

int dalib_top_abspos (topid, pos)

int topid, pos[];
 
{ topology_entry *entry;
  int i, rank;

  int g_pos;
  int a_pos;

  if (topid <= 0)  return (1);   /* return processor 1 as representant */
 
  entry = topologies[topid-1];

  rank  = entry->rank;

  g_pos = dalib_linear1_abspos (rank, entry->shape, pos);
  a_pos = dalib_group_element  (entry->group, g_pos);

#ifdef DEBUG
  printf ("%d: top_abspos (top=%d,group=%d), ", pcb.i, topid, entry->group);
  for (i=0; i<rank; i++) printf ("%d ", pos[i]);
  printf (" of ");
  for (i=0; i<rank; i++) printf ("%d ", entry->shape[i]);
  printf (" is g_pos=%d, a_pos=%d\n", g_pos, a_pos);
#endif

  return a_pos;

} /* dalib_top_abspos */
 
     /*************************************************************
     *                                                            *
     *  int dalib_in_topology (topid)                             *
     *                                                            *
     *   - returns true if processor is in topology topid         *
     *                                                            *
     *************************************************************/

int dalib_in_topology (topid)
int topid;

{  if (topid < 0)

      { /* host topology */

        if (pcb.i == 0)   
           return (1);
         else
           return (0);
      }

    else if (topid == 0)

      return (1);    /* replicated */

    else

      return (topologies[topid-1]->within);
}

int FUNCTION(dalib_is_in_procs) (topology)

int *topology;

{ 

#ifdef DEBUG
   printf ("%d: dalib_is_in_procs (top=%d)\n", pcb.i, *topology);
   dalib_top_print ();
#endif

   return dalib_in_topology (*topology);

} /* dalib_is_in_procs */

     /*************************************************************
     *                                                            *
     *  int dalib_top_dimpos (topid, dim, pos)                    *
     *                                                            *
     *   - returns the process id of the processor in the         *
     *     same topology only in dimension dim at position pos    *
     *                                                            *
     *   This function is needed for global communication         *
     *   in processor topologies (reductions, broadcasts)         *
     *                                                            *
     *************************************************************/

int dalib_top_dimpos (topid, dim, pos) 
int topid, dim, pos;

{ topology_entry *entry;
  int my_pos[MAX_RANK];
  int i, rank;
  int pid;

  entry = topologies[topid-1];
  rank  = entry->rank;

  for (i=0; i<rank; i++)
      my_pos[i] = entry->relpos[i];

  my_pos[dim-1] = pos;

  pid = dalib_top_abspos (topid, my_pos);

#ifdef DEBUG
  printf ("dalib_top_dimpos, topid = %d, dim = %d, pos = %d -> pid = %d\n",
           topid, dim, pos, pid);
#endif

  return (pid);

} /* dalib_top_dimpos */

     /*************************************************************
     *                                                            *
     *  int dalib_top_relpos (topid, dim, pos)                    * 
     *                                                            *
     *   - returns the process id of the processor in the         *
     *     same topology but +pos relative positions              *
     *                                                            *
     *   This function is needed for local neighbored communic.   *
     *                                                            *
     *************************************************************/
 
int dalib_top_relpos (topid, dim, pos)
int topid, dim, pos;
 
{ topology_entry *entry;
  int my_pos[MAX_RANK];
  int i, rank;
  int *dimpos;
  int n;
 
  entry = topologies[topid-1];
  rank  = entry->rank;
 
  for (i=0; i<rank; i++)
      my_pos[i] = entry->relpos[i];
 
  dimpos = my_pos + dim - 1;
 
  n = entry->shape[dim-1];

  *dimpos += pos;
  if (*dimpos < 1) *dimpos += n;
  if (*dimpos > n) *dimpos -= n;

  return (dalib_top_abspos (topid, my_pos));
 
} /* dalib_top_relpos */
 
 
     /*************************************************************
     *                                                            *
     *  void dalib_top_neighbors (p_left, p_right, topid, dim)    *
     *                                                            *
     *************************************************************/
 
void dalib_top_neighbors (p_left, p_right, topid, dim) 
int *p_left, *p_right;
int topid, dim;

{ int pos;

  *p_left  = dalib_top_relpos (topid, dim, -1);
  *p_right = dalib_top_relpos (topid, dim, 1);

#ifdef DEBUG
  printf ("%d: top_neighbors (topid=%d,dim=%d) left = %d, right = %d\n", 
          pcb.i, topid, dim, *p_left, *p_right);
#endif

} /* dalib_top_neighbors */ 

/**************************************************************************
*                                                                         *
*  int dalib_top_distance (int topid, int dim)                            *
*                                                                         *
*   - topid must be valid topology identifier, 1 <= dim <= dalib_top_rank *
*   - returns difference between two neighbored processors                *
*                                                                         *
**************************************************************************/

int dalib_top_distance (topid, topdim)
int topid;
int topdim;

{ topology_entry *entry;
  int i, dist;

  entry = topologies[topid-1];

  dist = 1;

  /* topdim == 1 : dist = 1
     topdim == 2 : dist = entry->shape[0];
     topdim == 3 : dist = entry->shape[0] * entry->shape[1];
  */

  for (i=1; i<topdim; i++)
     dist = dist * entry->shape[i-1];

  return (dist);

} /* dalib_top_distance */

/**************************************************************************
*                                                                         *
*  void dalib_top_position (int topid, int global_pid,                    *
*                           *int is_in_top, int relpos [])                *
*                                                                         *
*   - find the relative position of a global process id in topology       *
*                                                                         *
**************************************************************************/

void dalib_top_position (topid, global_pid, is_in_top, relpos)

int topid;
int global_pid;
int *is_in_top;
int relpos [];

{ topology_entry *entry;

  int group;
  int pos;
  int i;

#ifdef CHECK
  dalib_top_check_valid (topid);
#endif 

  entry = topologies[topid-1];

  pos = dalib_group_position (entry->group, global_pid);  

  /* 1 <= pid <= size */

  if (!pos)   /* global_pid does not belong to the topology */

     { *is_in_top = 0;
       return;
     }

  *is_in_top = 1;

  dalib_linear1_relpos (pos, entry->rank, entry->shape, relpos);

#ifdef DEBUG
  printf ("%d: top_position of global pid %d, in group %d = %d, in top %d = ",
           pcb.i, global_pid, entry->group, pos, topid);
  for (i=0; i<entry->rank; i++) printf ("%d ", relpos[i]);
  printf ("\n");
#endif

} /* dalib_top_position */

     /*************************************************************
     *                                                            *
     *  int dalib_top_distinct (int topid1, int topid2)           *
     *                                                            *
     *   - returns 1 if topologies have no common processor       *
     *                                                            *
     *************************************************************/

int dalib_top_distinct (topid1, topid2)

{ int size1, size2;
  int first1, first2;

  size1 = dalib_top_size (topid1);
  size2 = dalib_top_size (topid2);

  first1 = dalib_top_first (topid1);
  first2 = dalib_top_first (topid2);

  if (first1 <= first2)

    { if (first1 + size1 -1 < first2) return 1;
      return 0;
    }

   else

    { if (first2 + size2 -1 < first1) return 1;
      return 0;
    }

} /* dalib_top_distinct */

/**************************************************************************
*                                                                         *
*  int dalib_top_restrict (int topology, int n, int dims[n])              *
*                                                                         *
*  - creates a new topology where dims are fixed to my dimensions         *
*                                                                         *
**************************************************************************/

int dalib_top_restrict (topology, n, dims)

int topology;
int n;
int dims[];

{ topology_entry *entry;

  int group;
  int rank;

  int full      [MAX_RANK];
  int new_shape [MAX_RANK];
  int new_inc   [MAX_RANK];

  int new_group;
  int new_rank;
  int first;
  int new_top;

  int i;

  if (topology == 0) return (topology);
  if (topology == -1) return (topology);

#ifdef CHECK
  dalib_top_check_valid (topology);
#endif

  entry = topologies[topology-1];

  rank  = entry->rank;
  group = entry->group;

  /* fix dimensions according dims       */

  for (i=0; i<rank; i++) full[i] = 1;
  for (i=0; i<n; i++)    full[dims[i]-1] = 0;
 
  dalib_linear1_restrict (rank, entry->shape, entry->relpos, full,
                          &new_rank, &first, new_shape, new_inc);

  new_group = dalib_subgroup_make (group, first, new_rank, new_shape, new_inc);

  if (new_rank == 0)

     { new_rank     = 1;
       new_shape[0] = 1;
     }

  new_top = dalib_top_make (new_group, new_rank, new_shape);

  return (new_top);

} /* dalib_top_restrict */

/**************************************************************************
*                                                                         *
*  void dalib_pack_topology (char buffer[], int top_id => int length)     *
*                                                                         *
*  void dalib_unpack_topology (char buffer[] => int top_id, int length)   *
*                                                                         *
*  - pack/unpack information of topology (first, rank, shape)             *
*  - returns number of bytes needed for the coding in buffer              *
*                                                                         *
*  Note: exchanging topology information might result in different        *
*        topology identifiers on the processors                           *
*                                                                         *
**************************************************************************/

     /*************************************************************
     *   dalib_pack_topology (buffer, top_id => length)           *
     *      [group, rank, shape]                             *
     *************************************************************/

void dalib_pack_topology (buffer, top_id, length)

char buffer[];
int  top_id;
int  *length;

{ topology_entry *entry;

  int rank;
  int top_len;

  dalib_top_check_valid (top_id);

  entry = topologies[top_id-1];

  dalib_pack_group (buffer, entry->group, &top_len);

  rank = entry->rank;

  dalib_memcopy (buffer+top_len, &rank, sizeof(int));
  top_len += sizeof (int);

  dalib_memcopy (buffer+top_len, entry->shape, rank * sizeof(int));
  top_len += rank *sizeof (int);

  *length += top_len;

#ifdef DEBUG
  printf ("%d: packed top %d (group=%d, rank=%d) into buffer (%d bytes)\n",
          pcb.i, top_id, entry->group, entry->rank, *length);
#endif

} /* dalib_pack_topology */

     /*************************************************************
     *   dalib_unpack_topology (buffer => top_id, length)         *
     *************************************************************/

void dalib_unpack_topology (buffer, top_id, length)

char buffer[];
int  *length;
int  *top_id;

{ int group;
  int rank;
  int shape [MAX_RANK];
  int glen;

  dalib_unpack_group (buffer, &group, &glen);

  dalib_memcopy (&rank, buffer+glen, sizeof(int));
  glen += sizeof(int);

#ifdef DEBUG
  printf ("%d: unpack topology, group = %d, rank = %d\n",
           pcb.i, group, rank);
#endif

  dalib_memcopy (shape, buffer+glen, rank * sizeof(int));

  *length = glen + rank * sizeof(int);
  *top_id = dalib_top_make (group, rank, shape);

#ifdef DEBUG
  printf ("%d: unpacked top %d (group=%d, rank=%d) from buffer (%d bytes)\n",
          pcb.i, *top_id, group, rank, *length);
#endif

} /* dalib_unpack_topology */

