/**************************************************************************
*                                                                         *
*  Author      : Stephan Springstubbe, GMD, SCAI.WR                       *
*                Thomas Brandes, GMD, SCAI.WR                             *
*                Resi Hoever-Klier, GMD, SCAI.WR                          *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : May 95                                                   *
*  Last Update : Aug 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : shm.m4                                                   *
*                                                                         *
*  Function: allocation/deallocation of shared memory segments            *
*                                                                         *
*   EXPORT for DALIB                                                      *
*   ================                                                      *
*                                                                         *
*   void dalib_shm_allocate (int shm_size, char **shm_ptr, int *shm_id)   *
*                                                                         *
*   - allocation of shared/global memory in the current context           *
*   - memory area must have at least shm_size bytes                       *
*   - returns the address and an internal identification id               *
*                                                                         *
*   void dalib_shm_deallocate (int shm_id, char *shm_ptr)                 *
*                                                                         *
*   Currently two versions are supported:                                 *
*                                                                         *
*   #define SHM          System V shared memory segments                  *
*   #define GM           Global memory of SX/4 machines                   *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h" 

#include <stdio.h>          /* printf */
 
#if defined(SHM)

/**************************************************************************
*                                                                         *
*   SYSTEM V Support of Shared Memory Segments                            *
*                                                                         *
*   int shm_allocate (int size)                                           *
*   char *shm_attach (int shmid)                                          *
*   void shm_detach (char *shm_address)                                   *
*   void shm_deallocate (int shmid)                                       *
*                                                                         *
**************************************************************************/

#include <sys/types.h>      /* shmget */
#include <sys/ipc.h>        /* shmget */
#include <sys/shm.h>        /* shmget */
#include <sys/errno.h>      /* error numbers */
 
#define PAGE_SIZE           4096

/**************************************************************************
*                                                                         *
*   int shm_allocate (int size)                                           *
*                                                                         *
*    - one processor defines a shared memory segment with at leat         *
*      size bytes, and returns its id                                     *
*                                                                         *
**************************************************************************/
 
int shm_allocate (size)

int size;    /* number of bytes for the shared memory */

{
  int shmid;
  extern int errno;
  int memsize;
  int shmflag;

  memsize = ( (size + PAGE_SIZE - 1) / PAGE_SIZE ) * PAGE_SIZE;

  shmflag = 255*256 + 255;  /* set last 9 lower bits all to true */

  shmid = shmget (IPC_PRIVATE, memsize, shmflag); 

  if (shmid < 0) {
    printf ("error in shm_allocate operation: <%d>!\n", errno);
    printf ("size was : %d, chosen memsize : %d\n", size, memsize);
    dalib_internal_error ("shm_allocate");
    if (errno == EINVAL)
       printf ("invalid size for shared memory segment\n");
    if (errno == ENOMEM)
       printf ("not enough physical memory\n");
    if (errno == ENOSPC)
       printf ("system imposed limit exceeded\n");
    dalib_stop ();
  }

#ifdef DEBUG
  printf ("%d: SM - allocate (%d bytes), shmid = %d\n", 
          pcb.i, size, shmid);
#endif

  return (shmid);
 
} /* shm_allocate */
 
/**************************************************************************
*                                                                         *
*   char *shm_attach (int shmid)                                          *
*                                                                         *
*    - attach to a shared memory segment (given by shmid)                 *
*    - returns the address for accessing the shared memory                *
*                                                                         *
**************************************************************************/

char *shm_attach (int shmid)

{ char *shm_address;
  extern int errno;
 
#ifdef DEBUG
  printf ("%d: SM - attach to shmid = %d\n", pcb.i, shmid);
#endif

  /* SHM_RND stands for random attach, included from shm.h */

  shm_address = (char *) shmat (shmid, (char *) 0, SHM_RND);

  if ((long) shm_address == -1) {
    printf ("error in shmat operation, error number = %d\n", errno);
    dalib_internal_error ("shm_attach failed");
    if (errno == EACCES)
       printf ("no permission to access shmid %d\n", shmid);
    if (errno == EINVAL)
       printf ("%d is not a valid shared memory id to attach\n", shmid);
    if (errno == EMFILE)
       printf ("exceed system limit\n");
    if (errno == ENOMEM)
       printf ("not enough memory to attach to shared memory %d\n", shmid);
    dalib_stop ();
  }

#ifdef DEBUG
  printf ("%d: SM - attached to shmid = %d, addr = %p\n", 
           pcb.i, shmid, shm_address);
#endif

  return (shm_address);
 
} /* shm_attach */
 
/**************************************************************************
*                                                                         *
*  void shm_detach (char *shm_address)                                    *
*                                                                         *
*  - process will no longer use the shared memory (start at shm_address)  *
*                                                                         *
**************************************************************************/

void shm_detach (shm_address)

char *shm_address;

{ extern int errno;
  char msg[128];
  int shmrtn;
 
#ifdef DEBUG
  printf ("%d: SM - detach from shared memory, addr = %p\n", 
           pcb.i, shm_address);
#endif

  shmrtn = shmdt (shm_address);

  if (shmrtn < 0) 

   { sprintf (msg, "shm_detach fails, errno = %d\n", errno);
     dalib_internal_error (msg);
     if (errno == EINVAL)
        printf ("not address of a shared memory segment\n");
     dalib_stop ();
   }
 
} /* end shm_detach */
 
/**************************************************************************
*                                                                         *
*  void shm_deallocate (int shm_id)                                       *
*                                                                         *
*  - gives shared memory region free (only called by one proc)            *
*                                                                         *
**************************************************************************/

void shm_deallocate (shm_id)

int shm_id;

{ extern int errno;
  char msg[128];
  int shmrtn;
 
  shmrtn = shmctl (shm_id, IPC_RMID, NULL);

#ifdef DEBUG
    printf ("%d: SM deallocate (shm_id=%d),  operation: <%d>!\n", 
            pcb.i, shm_id, shmrtn);
#endif 

  if (shmrtn < 0)

   { sprintf (msg, "shm_deallocate fails, errno = %d\n", errno);
     dalib_internal_error (msg);
     dalib_stop ();
   }
 
}  /* shm_deallocate */

#elif defined(GM)

/**************************************************************************
*                                                                         *
*   GM Global Memory Support for Shared Memory                            *
*                                                                         *
*   int shm_allocate (size)                                               *
*   char *shm_attach (shmid)                                              *
*   void shm_detach (char *shm_address)                                   *
*   void shm_deallocate (int shmid)                                       *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/mppg.h>
#include <sys/types.h>
#include <unistd.h>
#include <mpi.h>
 
#define PAGE_SIZE           4096
 
/**************************************************************************
*                                                                         *
*   void gm_allocate (int gm_size, int *p_id, int **gm_ptr)               *
*                                                                         *
*     - one processor defines a global memory region,                     *
*       returns the address and the process id (UNIX)                     *
*                                                                         *
**************************************************************************/

void gm_allocate (gm_size, p_id, gm_ptr)

int gm_size;
int *p_id;
int **gm_ptr;

{ int *adr;
  int *gm_address;

  char err_msg [32];

  adr = (int *) 0;

  gm_address = (int *) dp_xmalloc(adr, gm_size);

  if (gm_address == (void *) -1)

     { printf ("(%d) dp_xmalloc returned errno %d; meaning: ",pcb.i,errno);

       switch (errno) {
           case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
           case ENOMEM:{ strcpy (err_msg,"ENOMEM");break;}
           case EAGAIN:{ strcpy (err_msg,"EAGAIN");break;}
           default: strcpy (err_msg,"strange");
               }
          printf (" %s\n",err_msg);
        dalib_internal_error ("GM ERROR");
        dalib_stop ();
     }

#ifdef DEBUG
     printf ("(%d) dp_xmalloc returned: gm_address = %p; gm_size %d\n",
             pcb.i, gm_address, gm_size);
#endif

  *gm_ptr = gm_address;
  *p_id   = getpid();

}  /* gm_allocate */

/**************************************************************************
*                                                                         *
*   char *gm_attach (int p_id, int *gm_ptr)                               *
*                                                                         *
*    - attach to global memory (on p_id at addresss gm_ptr)               *
*                                                                         *
**************************************************************************/

char *gm_attach (p_id, gm_ptr)

int p_id;
int *gm_ptr;

{ char *gm_address;
  int *adr;

  char err_msg [32];

  adr = (int *) 0;

  gm_address = (char *) dp_xmatt(p_id, gm_ptr, adr);

  if (gm_address ==  (void *) -1)

       { printf ("%d: error %d occurred at dp_xmatt. meaning: ",
                  pcb.i, errno);

         switch (errno)
               {
                case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
                case ESRCH:{ strcpy (err_msg,"ESRCH");break;}
                default: strcpy (err_msg,"strange");
               }
          printf (" %s\n",err_msg);
          dalib_stop ();
        }

  return (gm_address);

} /* gm_attach */
 
/**************************************************************************
*                                                                         *
*  void gm_free (int *gm_ptr)                                             *
*                                                                         *
*  - gives global memory region free (must be called by all processors)   *
*                                                                         *
**************************************************************************/

void gm_free (gm_ptr)

int *gm_ptr;

{ int rc;

  char err_msg [32];

#ifdef DEBUG
  printf ("%d: gm_free of gm_ptr = %p\n", pcb.i, gm_ptr);
#endif

  rc = dp_xmfree (gm_ptr);

  if (rc == -1)

      { printf ("(%d) dp_xmfree returned errno: %d meaning: ", 
                 pcb.i, errno);
        switch (errno)
              {
               case EINVAL:{ strcpy (err_msg,"EINVAL");break;}
               default: strcpy (err_msg,"strange");
              }
        printf (" %s\n",err_msg);
        dalib_stop ();
      }

#ifdef DEBUG
   printf ("%d: dp_xmfree successful!\n", pcb.i);
#endif

}  /* gm_free */

#else

/**************************************************************************
*                                                                         *
*   NO SUPPORT OF SYSYEM_V FEATURES !!!!!!!!!!!!!!!!!!                    *
*                                                                         *
**************************************************************************/

#endif

/**************************************************************************
*                                                                         *
*   void dalib_shm_error (char *routine)                                  *
*                                                                         *
**************************************************************************/

void dalib_shm_error (routine)
char *routine;

{ dalib_internal_error (routine);
  printf ("do not use SHARED directive or -shared flag\n");
  dalib_stop ();

} /* dalib_shm_error */

/**************************************************************************
*                                                                         *
*   ALLOCATING shared/global MEMORY                                       *
*                                                                         *
*   void dalib_shm_allocate (int shm_size, char **shm_ptr, int *shm_id)   *
*                                                                         *
*   - allocation of shared/global memory in the current context           *
*   - memory area must have at least shm_size bytes                       *
*   - returns the address and an internal identification id               *
*                                                                         *
**************************************************************************/

void dalib_shm_allocate (shm_size, shm_address, shm_id)

int  shm_size;
char **shm_address;
int  *shm_id;
 
{ int group_id;     /* group identification of the current context */
  int group_pos;    /* my own position in this group               */

  struct {

    int  id;          /* identification of the memory                */
    char *address;    /* address of the allocated/attached memory    */

  } dsp;

  /* identify the group and my postition in this group */

  group_id  = dalib_context_group ();
  group_pos = dalib_group_position (group_id, pcb.i);

#ifdef DEBUG
  printf ("%d: dalib_shm_allocate, size = %d\n", pcb.i, shm_size);
#endif

  /* allocation of the global memory region */
 
  if (group_pos == 1)  /* I am repsonsible  */

    {
#if defined(SHM)
       dsp.id = shm_allocate (shm_size); dsp.address = (char *) 0;
#elif defined(GM)
       gm_allocate (shm_size, &(dsp.id), &(dsp.address));
#else
       dsp.id = 0; dsp.address = (char *) 0;
#endif
    }
 
  /* broadcast the descriptor from first one to all processors */
 
  dalib_group_bcast (&dsp, sizeof(dsp), 1, group_id);

#ifdef DEBUG
  printf ("%d: dalib_shm_allocate, attach to id = %d, addr = %p\n", 
            pcb.i, dsp.id, dsp.address);
#endif

  /* attaching the shared/global memory region */

#if defined(SHM)

  dsp.address = shm_attach (dsp.id);

#elif defined(GM)

  /* for global memory the address is already correct on first proc */

  if (group_pos != 1) 
     dsp.address = gm_attach (dsp.id, dsp.address);

#else

  dalib_shm_error ("shm_allocate: no support of shared/global memory");

#endif

  *shm_address = dsp.address;
  *shm_id      = dsp.id;
 
} /* dalib_shm_allocate */

/**************************************************************************
*                                                                         *
*   DEALLOCATING shared/global MEMORY                                     *
*                                                                         *
*   void dalib_shm_deallocate (int shm_id, char *shm_ptr)                 *
*                                                                         *
**************************************************************************/

void dalib_shm_deallocate (shm_id, shm_ptr)

int shm_id;
int *shm_ptr;

{ int group_id;     /* group identification of the current context */
  int group_pos;    /* my own position in this group               */

  group_id  = dalib_context_group ();
  group_pos = dalib_group_position (group_id, pcb.i);

#ifdef DEBUG
  printf ("%d: dalib_shm_deallocate from shm_id = %d, addr = %p\n",
           pcb.i, shm_id, shm_ptr);
#endif


#if defined(SHM)

  shm_detach (shm_ptr);

  dalib_group_barrier (group_id);     /* barrier before deallocation */

  if (group_pos == 1) shm_deallocate (shm_id);

#elif defined (GM)

  /* barrier before memory is really freed  */

  dalib_group_barrier (group_id);     /* barrier before deallocation */

  gm_free (shm_ptr);

  dalib_group_barrier (group_id);     /* wait for complete release   */

#else

  dalib_shm_error ("shm_deallocate: no support of shared/global memory");

#endif

} /* dalib_shm_deallocate */


