/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : May 95                                                   *
*  Last Update : Aug 97                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : shared.m4                                                *
*                                                                         *
*  Function: Creating descriptors for (distributed) shared data           *
*                                                                         *
*  EXPORT:                                                                *
*                                                                         *
*   a) FORTRAN interface                                                  *
*                                                                         *
*      void dalib_array_shared (array_info array_id)                      *
*                                                                         *
*        - defines descriptor for shared array (no allocate)              *
*                                                                         *
*   b) DALIB interface                                                    *
*                                                                         *
*      void dalib_array_shared_malloc (array_info array_id)               *
*                                                                         *
*        - allocates shared data                                          *
*                                                                         *
*      void dalib_array_shared_free (array_info array_id)                 *
*                                                                         *
*        - deallocates shared data                                        *
*                                                                         *
*      void dalib_shared_free (Shared shared_ptr)                         *
*                                                                         *
*       - frees only descriptor (data can remain)                         *
*                                                                         *
*  Attention: this module should be available in any case                 *
*                                                                         *
*  Changes:                                                               *
*                                                                         *
*     08/97  : shared completely independent of RMA (remote mem access)   *
*                                                                         *
**************************************************************************/
 
#undef DEBUG

#ifdef GM

/* in this module we make no difference between GM and SHM */

#define SHM

#endif

#include "dalib.h"

     /***********************************************
     *                                              *
     *   record describing how data is shared       *
     *                                              *
     ***********************************************/

struct SharedRecord
 
 { 
    int   shm_defined;        /* shared memory has been allocated     */
    int   shm_id;             /* identifier for shared memory segment */

    unsigned char *shm_addr;  /* address of shared memory, but is     */
                              /* only valid if shm_defined == 1       */
 };

/******************************************************************
*                                                                 *
*  FUNCTION(dalib_array_shared) (array_info array_id)             *
*                                                                 *
*   - defines an array to be shared (via shm)                     *
*                                                                 *
******************************************************************/

void FUNCTION(dalib_array_shared) (array_id)
 
array_info *array_id;
 
{ Shared shared_ptr;

  shared_ptr  = (Shared) dalib_malloc (sizeof (struct SharedRecord), 
                                               "dalib_array_shared");

  shared_ptr->shm_defined = 0;   /* no shm defined */

#ifdef SHM
#else
  dalib_internal_error ("SHM shared not available");
  dalib_stop ();
#endif

#ifdef DEBUG
  printf ("%d: array %d has been defined shared\n", pcb.i, *array_id);
#endif 

  (*array_id)->SharedInfo        = shared_ptr;
  (*array_id)->global_addressing = 1;

} /* dalib_array_shared */

/**************************************************************************
*                                                                         *
*   void dalib_array_shared_malloc (array_info array_id)                  *
*                                                                         *
**************************************************************************/

void dalib_array_shared_malloc (array_id)
 
array_info array_id;
 
{ Shared shared_ptr;
  
  int rank;
  int size;             /* number of bytes for one element */

  int first;
  int total [MAX_DIMENSIONS+1];
  int elems;

  int shmid;
 
  unsigned char *dummy, *a_ptr; 

  shared_ptr = array_id->SharedInfo;
  
  if (shared_ptr == NO_SHARED)
 
     { dalib_array_malloc (array_id);
       return;
     }
 
  if (shared_ptr->shm_defined != 0)

     { dalib_internal_error ("array_shared_malloc (already allocated)");
       dalib_stop ();
     }

  size = array_id->size;
  rank = array_id->rank;

  /* creating global shared data */

  dalib_array_addressing (array_id, pcb.i, &dummy, &first, total);

  elems = total[rank] * size + size;
 
#ifdef SHM
  dalib_shm_allocate (elems, &a_ptr, &shmid);

  shared_ptr->shm_defined = 1;
  shared_ptr->shm_id      = shmid;
  shared_ptr->shm_addr    = a_ptr;

#ifdef DEBUG
  printf ("%d: shm %d allocated/attached\n", pcb.i, shared_ptr->shm_id);
#endif

  /* that is pointer to the allocated memory */
 
  array_id->n_data          = elems;
  array_id->c_data          = a_ptr;
  array_id->f_data          = NO_DATA;
  array_id->dsp_status_flag = DSP_OWN_DATA;

#else
  dalib_internal_error ("SHM shared not available");
  dalib_stop ();
#endif

}  /* dalib_array_shared_malloc */

/*******************************************************************
*                                                                  *
*   void dalib_array_shared_free (array_info array_id)             *
*                                                                  *
*    - free shared data (but do not free the descriptor)           *
*                                                                  *
*******************************************************************/

void dalib_array_shared_free (array_id)

array_info array_id;

{ Shared shared_ptr;
  unsigned char *shared_data;

  shared_ptr  = array_id->SharedInfo;
  shared_data = array_id->c_data;

  /* detach/deallocate shared memory region */

#ifdef DEBUG
  printf ("%d: %d shared free (dsp_status = %d), shm_id = %d\n",
          pcb.i, array_id, array_id->dsp_status_flag, shared_ptr->shm_id);
#endif

#ifdef SHM
  if (array_id->dsp_status_flag == DSP_OWN_DATA)

     { dalib_shm_deallocate (shared_ptr->shm_id, shared_data);

       /* make sure that deallocate is not called twice */

       array_id->dsp_status_flag = DSP_DEFINED;
       array_id->n_data = 0;
       array_id->c_data = NO_DATA;
     }
#else
  dalib_internal_error ("SHM shared not available");
  dalib_stop ();
#endif

} /* dalib_array_shared_free */

/*******************************************************************
*                                                                  *
*   void dalib_shared_free (Shared shared_ptr)                     *
*                                                                  *
*    - frees only descriptor (data can remain)                     *
*                                                                  *
*******************************************************************/

void dalib_shared_free (shared_ptr)

Shared shared_ptr;

{ dalib_free (shared_ptr, sizeof(struct SharedRecord));

} /* dalib_shared_free */

/*******************************************************************
*                                                                  *
*  dalib_shared_read (int owner, array_info array_id,              *
*                     char *data, int offset)                      *
*                                                                  *
*******************************************************************/

void dalib_shared_read (owner, array_id, data, offset)

int owner;
array_info *array_id;
char *data;
int offset;

{ Shared shared_ptr;
  int size;

#ifdef DEBUG
  printf ("%d: dalib_shared_read on owner = %d (offset = %d)\n",
           pcb.i, owner, offset);
#endif

  shared_ptr = (*array_id)->SharedInfo;

  if (shared_ptr == NO_SHARED)
 
    { dalib_internal_error ("shared read on non-shared array");
      dalib_stop ();
    }

  size = (*array_id)->size;

  dalib_memcopy (data, (*array_id)->f_data + offset * size, size);

} /* dalib_shared_read */

/*******************************************************************
*                                                                  *
*  dalib_shared_update (int owner, array_info array_id,            *
*                       char *data, int offset)                    *
*                                                                  *
*******************************************************************/

void dalib_shared_update (owner, array_id, data, offset)

int owner;
array_info *array_id;
char *data;
int offset;

{ Shared shared_ptr;
  int size;

#ifdef DEBUG
  printf ("%d: dalib_shared_update on owner = %d (offset = %d)\n",
           pcb.i, owner, offset);
#endif

  shared_ptr = (*array_id)->SharedInfo;
 
  if (shared_ptr == NO_SHARED)
 
    { dalib_internal_error ("shared write on non-shared array");
      dalib_stop ();
    }
 

  size = (*array_id)->size;

  /* note : offsets will be global offsets */

  dalib_memcopy ((*array_id)->f_data + offset * size, data, size);

} /* dalib_shared_update */

/*******************************************************************
*                                                                  *
*  dalib_shared_nread (int owner, array_info *array_id,            *
*                      char *data, int offsets[], int n)           *
*                                                                  *
*******************************************************************/

static void dalib_shared_nread (owner, array_id, data, offsets, n)

int owner, n;
array_info *array_id;
char *data;
int offsets[];

{ Shared shared_ptr;
  array_info id;
  unsigned char *ptr;
  int size;
  int i;

  id = *array_id;

#ifdef DEBUG
  printf ("%d: dalib_shared_nread on owner = %d (%d values)\n",
           pcb.i, owner, n);
#endif

  shared_ptr = id->SharedInfo;

  if (shared_ptr == NO_SHARED)
 
    { dalib_internal_error ("shared multiple read on non-shared array");
      dalib_stop ();
    }

  size = id->size;

  /* note : offsets will be global offsets */

  ptr = id->f_data;

#ifdef DEBUG
  printf ("%d: offsets are ", pcb.i);
  for (i=0; i<n; i++)
      printf (" %d", offsets[i]);
  printf ("\n");
#endif 

  dalib_memget (data, ptr, offsets, n, size);

} /* dalib_shared_nread */

/*******************************************************************
*                                                                  *
*  dalib_shared_nwrite (int owner, array_info array_id,            *
*                       char *data, int offsets[], int n)          *
*                                                                  *
*******************************************************************/

static void dalib_shared_nwrite (owner, array_id, data, offsets, n)

int owner, n;
array_info *array_id;
char *data;
int offsets[];

{ Shared shared_ptr;
  array_info id;
  unsigned char *ptr;
  int *hd;
  int size;
  int i;

#ifdef DEBUG
  printf ("%d: dalib_shared_nwrite on owner = %d (%d values)\n",
           pcb.i, owner, n);
#endif

  id = *array_id;

  shared_ptr = id->SharedInfo;

  if (shared_ptr == NO_SHARED)
 
    { dalib_internal_error ("shared multiple write on non-shared array");
      dalib_stop ();
    }
 
  size = id->size;

  hd = (int *) data;   /* printing data values */

  /* note : offsets will be global offsets */

  ptr = id->f_data;

#ifdef DEBUG
  printf ("%d: offsets are ", pcb.i);
  for (i=0; i<n; i++)
     printf (" %d (val = %d)", offsets[i], hd[i]);
  printf ("\n");
#endif 

  dalib_memset (ptr, offsets, data, n, size);

} /* dalib_shared_nwrite */

/**************************************************************************
*                                                                         *
*   void dalib_array_shm_remote_malloc (array_info array_id,              *
*                                       Shared *SharedInfo  )             *
*                                                                         *
**************************************************************************/

void dalib_array_shm_remote_malloc (array_id, SharedInfo)
 
array_info array_id;
Shared *SharedInfo;
 
{ Shared shared_ptr;
  
  int group_id;            /* current context group           */
  int group_pos;
  int group_size;

  int rank;
  int size;             /* number of bytes for one element */

  int first;
  int total [MAX_DIMENSIONS+1];

  int local_size;           /* size in bytes for my local part of array */
  int all_sizes [MAXP];     /* sizes in bytes for all local parts       */
  int global_size;          /* total size in bytes                      */
  int offset;               /* my offset to calculate my address        */
  int i;                    /* loop variable                            */

  unsigned char *a_ptr;
  unsigned char *dummy;

  int shmid;

  shared_ptr  = (Shared) dalib_malloc (sizeof (struct SharedRecord), 
                                      "dalib_array_shm_remote_malloc");

  size = array_id->size;
  rank = array_id->rank;

  /* calculate the local size needed for me */

  dalib_array_addressing (array_id, pcb.i, &dummy, &first, total);
  local_size = (total[rank] + 1) * size;
 
  /* identify the group and my postition (1<=pos<=size) in this group */

  group_id   = dalib_context_group ();
  group_pos  = dalib_group_position (group_id, pcb.i);
  group_size = dalib_group_size (group_id);

  /* get all local sizes of all relevant processors */

  all_sizes [group_pos-1] = local_size;
  dalib_group_concat (all_sizes, sizeof(int), group_id);

  /* calculate the total size of all processors and my offset */

  offset = 0;
  global_size = 0;

  for (i=1; i<=group_size; i++)

     { if (i < group_pos) offset += all_sizes[i-1];
       global_size += all_sizes[i-1];
     }

#ifdef DEBUG
  printf ("%d: shm_remote_malloc, all_sizes = ", pcb.i);
  for (i=0; i<group_size; i++) printf ("%d ", all_sizes[i]);
  printf ("\n");
  printf ("%d: shm_remote_malloc (local = %d, global = %d, offset = %d)\n",
           pcb.i, local_size, global_size, offset);
#endif

#ifdef SHM
  dalib_shm_allocate (global_size, &a_ptr, &shmid);

  shared_ptr->shm_id      = shmid;
  shared_ptr->shm_defined = 1;
  shared_ptr->shm_addr    = a_ptr;

#ifdef DEBUG
  printf ("%d: shm %d allocated/attached\n", pcb.i, shared_ptr->shm_id);
#endif

  a_ptr += offset;   /* that is now my data */
 
  array_id->n_data          = local_size;
  array_id->c_data          = a_ptr;
  array_id->f_data          = NO_DATA;
  array_id->dsp_status_flag = DSP_OWN_DATA;

#else
  dalib_internal_error ("SHM shared not available");
  dalib_stop ();
#endif

  *SharedInfo = shared_ptr;

} /* dalib_array_shm_remote_malloc */

/*******************************************************************
*                                                                  *
*   void dalib_shared_data_free (SharedInfo shared_ptr)            *
*                                                                  *
*******************************************************************/

void dalib_shared_data_free (SharedInfo)

Shared SharedInfo;

{ unsigned char *shm_addr;
  int shm_id;

  if (SharedInfo->shm_defined)

    { shm_addr = SharedInfo->shm_addr;
      shm_id   = SharedInfo->shm_id;

      dalib_shm_deallocate (shm_id, shm_addr);
    }

  dalib_free (SharedInfo, sizeof(struct SharedRecord));

} /* dalib_shared_data_free */
