/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*                                                                         *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Feb 95                                                   *
*  Last Update : Feb 95                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : secarray                                                 *
*                                                                         *
*  Function:  Operations working on sections and array                    *
*                                                                         *
*  Export :  FORTRAN Interface                                            *
*  ===========================                                            *
*                                                                         *
**************************************************************************/

#include <stdio.h>
#include "dalib.h"

#undef  DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  void dalib_make_secarray_ddt (new_ddt, secarray_id)                    *
*                                                                         *
*   - gets a derived data type for array or section                       *
*                                                                         *
**************************************************************************/

void dalib_make_secarray_ddt (new_ddt, section_id)
 
dd_type      *new_ddt;
section_info section_id;
 
{ /* makes ddt independent whether it is a section or an array */
 
#ifdef DEBUG
   printf ("%d: make secarray ddt\n", pcb.i);
   if (dalib_is_section_info (section_id)) 
      dalib_print_section_info (section_id);
#endif 

  if (dalib_is_array_info (section_id))
 
      dalib_make_array_ddt (new_ddt, section_id);

   else if (dalib_is_section_info (section_id))

      dalib_make_perm_section_ddt (new_ddt, section_id, (int *) 0);

   else

    { dalib_internal_error ("make_secarray_ddt: no section_id/array_id");
      dalib_stop ();
    }
 
} /* dalib_make_secarray_ddt */
 
void dalib_make_secarray_ddt1 (new_ddt, section_id, section_size, elem_size)
dd_type *new_ddt;
section_info section_id;
int *section_size, *elem_size;
 
{ if (dalib_is_array_info (section_id))
 
    { *elem_size = ((array_info) section_id)->size;
      dalib_make_array_ddt (new_ddt, section_id, *elem_size);
    }
 
   else if (dalib_is_section_info (section_id))
 
    { *elem_size =  section_id->array_id->size;
      dalib_make_section_ddt (new_ddt, section_id, *elem_size);
    }
 
   else
    { dalib_internal_error ("cmf_random, int, not a section/array");
      dalib_stop ();
    }
 
  dalib_ddt_get_size (*new_ddt, section_size);
  *section_size = *section_size / *elem_size;
 
} /* dalib_make_secarray_ddt1 */

/**************************************************************************
*                                                                         *
*    ---- sections can be real sections or arrays itselft -----           *
*                                                                         *
*  void dalib_secarray_copy (target_section, source_section)              *
*                                                                         *
*  void dalib_secarray_perm_copy (target_section, source_section, perm)   *
*                                                                         *
*  void dalib_secarray_send (pid, send_section)                           *
*                                                                         *
*  void dalib_secarray_perm_send (pid, send_section, int perm[])          *
*                                                                         *
*  void dalib_secarray_recv (pid, send_section)                           *
*                                                                         *
*  void dalib_secarray_pack (char **mem, section)                         *
*                                                                         *
*  void dalib_secarray_unpack (section, char *mem)                        *
*                                                                         *
**************************************************************************/

       /************************************************
       *                                               *
       *   dalib_secarray_copy (target, source)        *
       *                                               *
       ************************************************/

void dalib_secarray_copy (target_section, source_section)

section_info target_section;
section_info source_section;

{ dd_type source_ddt, target_ddt;

  dalib_make_secarray_ddt (&source_ddt, source_section);
  dalib_make_secarray_ddt (&target_ddt, target_section);

  dalib_ddt_move (target_ddt, source_ddt);
  dalib_ddt_free (target_ddt);
  dalib_ddt_free (source_ddt);

} /* dalib_section_copy */
 
       /****************************************************
       *                                                   *
       *   dalib_secarray_perm_copy (target, source, perm) *
       *                                                   *
       *   - apply permutation on target (for RESHAPE)     *
       *                                                   *
       ****************************************************/

void dalib_secarray_perm_copy (target_section, source_section, perm)

section_info target_section;
section_info source_section;
int perm[];

{ dd_type source_ddt, target_ddt;
  section_info section_id;

  /* permute for array does not exist, so make a section in any case */

  dalib_make_secarray_ddt (&source_ddt, source_section);

  dalib_section_full (&section_id, target_section);
  dalib_make_perm_section_ddt (&target_ddt, section_id, perm);

  dalib_ddt_move (target_ddt, source_ddt);
  dalib_ddt_free (target_ddt);
  dalib_ddt_free (source_ddt);

  if (dalib_is_array_info (target_section))
     FUNCTION(dalib_section_free) (&section_id);

} /* dalib_section_copy */
 
       /************************************************
       *                                               *
       *   dalib_secarray_send (pid, section)          *
       *                                               *
       ************************************************/

void dalib_secarray_send (pid, send_section)

section_info send_section;

{ dd_type send_ddt;
 
  dalib_make_secarray_ddt (&send_ddt, send_section);
  dalib_send_ddt (pid, send_ddt);
  dalib_ddt_free (send_ddt);

}  /* dalib_secarray_send */
 
       /*****************************************************
       *                                                    *
       *   dalib_secarray_perm_send (pid, secarray, perm)   *
       *                                                    *
       *****************************************************/

void dalib_secarray_perm_send (pid, send_section, perm)

section_info send_section;

int pid;    /* absolute processor id to which data is sent */
int perm[]; /* permutation vector */

{ dd_type      send_ddt;
  section_info section_id;
 
#ifdef DEBUG
  printf ("%d: section perm send to %d\n", pcb.i, pid);
#endif

  /* permute for array does not exist, so make a section in any case */

  dalib_section_full (&section_id, send_section);

  dalib_make_perm_section_ddt (&send_ddt, section_id, perm);
  dalib_send_ddt (pid, send_ddt);
  dalib_ddt_free (send_ddt);

  if (dalib_is_array_info (send_section))
     FUNCTION(dalib_section_free) (&section_id);

}  /* dalib_secarray_perm_send */

       /************************************************
       *                                               *
       *   dalib_section_recv (pid, section)           *
       *                                               *
       ************************************************/

void dalib_secarray_recv (pid, recv_section)

section_info recv_section;

{ dd_type recv_ddt;
 
  dalib_make_secarray_ddt (&recv_ddt, recv_section);
  dalib_recv_ddt_op (pid, recv_ddt, 0);
  dalib_ddt_free (recv_ddt);

}  /* dalib_secarray_recv */
 
       /************************************************
       *                                               *
       *   dalib_section_pack (mem, section)           *
       *                                               *
       ************************************************/

void dalib_secarray_pack (mem, section)

section_info section;
char **mem;

{ dd_type ddt;
  int mem_size;
 
  dalib_make_secarray_ddt (&ddt, section);
  dalib_ddt_get_size (ddt, &mem_size);
  *mem = (char *) dalib_malloc (mem_size, "dalib_secarray_pack");
  dalib_ddt_pack (*mem, ddt);
  dalib_ddt_free (ddt);

}  /* dalib_secarray_pack */
 
/*******************************************************************
*                                                                  *
*   void dalib_secaray_unpack (section_info section, char *mem)    *
*                                                                  *
*    - copies contiguous data into non-contiguous section          *
*    - mem will be freed afterwards (got by secarray_pack)         *
*                                                                  *
*******************************************************************/

void dalib_secarray_unpack (section, mem)

section_info section;
char *mem;

{ dd_type ddt;
  int size;
 
  dalib_make_secarray_ddt (&ddt, section);
  dalib_ddt_get_size (ddt, &size);
  dalib_ddt_unpack (ddt, mem, 0);
  dalib_ddt_free (ddt);
  dalib_free (mem, size);

}  /* dalib_secarray_unpack */

       /***************************************************
       *                                                  *
       *   dalib_secarray_bc (topid, dim, section)        *
       *                                                  *
       ***************************************************/

void dalib_secarray_bc (topid, dim, section)
int topid, dim;
section_info section;

{ dd_type ddt;

#ifdef DEBUG
  printf ("dalib_secarray_bc, topid = %d, dim = %d\n", topid, dim);
#endif

  dalib_make_secarray_ddt (&ddt, section);
  dalib_top_broadcast_ddt (topid, dim, ddt);
  dalib_ddt_free (ddt);

} /* dalib_secarray_bc */

       /***************************************************
       *                                                  *
       *   dalib_secarray_bc (topid, dim, section)        *
       *                                                  *
       ***************************************************/

void dalib_section_fill (section, data)
section_info section;
char *data;

{ dd_type ddt;

  int elems;
  int size;
  int i;
  int mem_size;

  char *mem, *memptr;

#ifdef DEBUG
  printf ("will fill section with data \n");
#endif

  size = section->array_id->size;

  dalib_make_secarray_ddt (&ddt, section);
  dalib_ddt_get_size (ddt, &mem_size);

  mem = (char *) dalib_malloc (mem_size, "dalib_section_fill");

  elems = mem_size / size;

#ifdef DEBUG
  printf ("fill %d values a %d bytes\n", elems, size);
#endif

  memptr = mem;
  for (i=0;i<elems;i++)
    {  dalib_memcopy (memptr, data, size);
       memptr += size;
    }

  dalib_ddt_unpack (ddt, mem, 0);
  dalib_ddt_free (ddt);
  dalib_free (mem, mem_size);

} /* dalib_section_fill */

/**************************************************************************
*                                                                         *
*   int dalib_lbound (section_dsp/array_dsp, dim)                         *
*   int dalib_ubound (section_dsp/array_dsp, dim)                         *
*                                                                         *
**************************************************************************/

int FUNCTION(dalib_lbound) (array_id, dim)
section_info *array_id;
int          *dim;

{ array_info  id;
 
  if (array_id == (section_info *) 0)

     { dalib_internal_error ("lbound: no descriptor");
       dalib_stop();
     }

  if (dalib_is_array_info (*array_id))

     { id = (array_info) *array_id;
       return (id->dimensions[*dim-1].global_size[0]);
     }

  if (dalib_is_section_info (*array_id))

     { int array_dim;
       int *range;
 
       array_dim = dalib_section_array_dim (*array_id, *dim);
       range = (*array_id)->dimensions[array_dim-1].global_range;
       return (range[0]);
     }

  dalib_internal_error ("lbound: no array/section info");
  dalib_stop ();
  return (1);

} /* dalib_lbound */

int FUNCTION(dalib_ubound) (array_id, dim)
section_info *array_id;
int          *dim;
 
{ array_info   id;
 
  if (array_id == (section_info *) 0)
 
     { dalib_internal_error ("ubound: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*array_id))
 
     { id = (array_info) *array_id;
       return (id->dimensions[*dim-1].global_size[1]);
     }
 
  if (dalib_is_section_info (*array_id))
 
     { int array_dim;
       int *range;

       array_dim = dalib_section_array_dim (*array_id, *dim);
       range = (*array_id)->dimensions[array_dim-1].global_range;
       return (range[1]);
     }
 
  dalib_internal_error ("ubound: no array/section info");
  dalib_stop ();
  return (1);
 
} /* dalib_ubound */
 
/**************************************************************************
*                                                                         *
*   int dalib_local_lbound (section_dsp, dim)                             *
*   int dalib_local_ubound (section_dsp, dim)                             *
*                                                                         *
**************************************************************************/
 
int FUNCTION(dalib_local_lbound) (array_id, dim)
array_info *array_id;
int        *dim;
 
{ DimInfo      *dims;
 
  if (dalib_is_array_info (*array_id))

    { dims = (*array_id)->dimensions;
      return (dims[*dim-1].local_size[0]);
    }

  if (dalib_is_section_info (*array_id))

    { int lb, ub, inc;
      dalib_section_local_bounds (*array_id, *dim, &lb, &ub, &inc);
      return (lb);
    }

  dalib_internal_error ("dalib_local_lbound, not array/section info");
  dalib_stop ();
  return (1);
 
} /* dalib_local_lbound */
 
int FUNCTION(dalib_local_ubound) (array_id, dim)
array_info *array_id;
int        *dim;
 
{ DimInfo      *dims;
 
  if (dalib_is_array_info (*array_id))

    { dims = (*array_id)->dimensions;
      return (dims[*dim-1].local_size[1]);
    }

  if (dalib_is_section_info (*array_id))

    { int lb, ub, inc;
      dalib_section_local_bounds (*array_id, *dim, &lb, &ub, &inc);
      return (ub);
    }

  dalib_internal_error ("dalib_local_ubound, not array/section info");
  dalib_stop ();
 
  return (1);

} /* dalib_local_ubound */

/**************************************************************************
*                                                                         *
*   int dalib_global_lbound (section_dsp, dim)                            *
*   int dalib_global_ubound (section_dsp, dim)                            *
*                                                                         *
**************************************************************************/
 
int FUNCTION(dalib_global_lbound) (array_id, dim)

array_info *array_id;
int        *dim;
 
{ DimInfo      *dims;
  array_info   global_id;

  global_id = (*array_id)->GlobalInfo;

  if (global_id == ((array_info) 0))

     { dalib_internal_error ("dalib_global_lbound: no global array");
       dalib_stop ();
       return 1;
     }
 
  dims = global_id->dimensions;
  return (dims[*dim-1].global_size[0]);

} /* dalib_global_lbound */
 
int FUNCTION(dalib_global_ubound) (array_id, dim)

array_info *array_id;
int        *dim;
 
{ DimInfo      *dims;
  array_info   global_id;

  global_id = (*array_id)->GlobalInfo;

  if (global_id == ((array_info) 0))

     { dalib_internal_error ("dalib_global_ubound: no global array");
       dalib_stop ();
       return 1;
     }
 
  dims = global_id->dimensions;
  return (dims[*dim-1].global_size[1]);

} /* dalib_global_ubound */
 
/**************************************************************************
*                                                                         *
*   int dalib_secarray_size (section_dsp)                                 *
*                                                                         *
**************************************************************************/

int dalib_secarray_size (section_id)

section_info *section_id;

{ array_info  id;
 
  if (section_id == (section_info *) 0)
 
     { dalib_internal_error ("secarray_size: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*section_id))
 
     { id = (array_info) *section_id;
       return (id->size);
     }
 
  if (dalib_is_section_info (*section_id))
 
     { id = (*section_id)->array_id;
       return (id->size);
     }
 
  dalib_internal_error ("secarray_size: no array/section info");
  dalib_stop ();
  return (0);
 
} /* dalib_secarray_size */

/**************************************************************************
*                                                                         *
*   int dalib_secarray_rank (section_info *section_id)                    *
*                                                                         *
**************************************************************************/

int dalib_secarray_rank (section_id)

section_info *section_id;

{ array_info  id;
 
  if (section_id == (section_info *) 0)
 
     { dalib_internal_error ("secarray_rank: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*section_id))
 
     { id = (array_info) *section_id;
       return (id->rank);
     }
 
  if (dalib_is_section_info (*section_id))
 
     return (dalib_section_rank (*section_id));
 
  dalib_internal_error ("secarray_rank: no array/section info");
  dalib_stop ();
  return (1);
 
} /* dalib_secarray_rank */

/*********************************************************************
*                                                                    *
*   dalib_secarray_get_data (array_info/section_info dsp,            *
*                            int copy_flag,                          *
*                            int *no, int *size,                     *
*                            char **data, int *is_new)               *
*                                                                    *
*    - get data of a section in a contiguous memory area             *
*                                                                    *
*   dsp (in)    :  descriptor for section or array                   *
*   copy (in)   :  copies section data to data if necessary          *
*                                                                    *
*   no (out)    :  number of elements in section/array               *
*   size (out)  :  number of bytes for one element                   *
*   data (out)  :  pointer to the contiguous data of section/array   *
*   is_new (out):  flag indicates if new memory has been used        *
*                                                                    *
*********************************************************************/

void dalib_secarray_get_data (dsp, copy_flag, no, size, data, is_new)

array_info dsp;
int        copy_flag;

char       **data;
int        *no, *size;
int        *is_new;

{ array_info   array_dsp;
  section_info section_dsp;
  dd_type      ddt;

  int          is_contiguous;
  char         *vals;

  if (dalib_is_array_info (dsp))

     { array_dsp = dsp;
       dalib_section_full (&section_dsp, array_dsp);
       *no = dalib_array_local_size (array_dsp);
     }

   else if (dalib_is_section_info (dsp))

     { section_dsp = (section_info) dsp;
       array_dsp   = section_dsp->array_id;
       *no = dalib_section_local_size (dsp);
     }

   else
 
     { dalib_internal_error ("secarray_get_data : illegal descriptor");
       dalib_stop ();
     }

   *size = array_dsp->size;

   dalib_make_section_ddt (&ddt, section_dsp);

   dalib_ddt_is_contiguous (ddt, &is_contiguous, &vals);

   if (!is_contiguous)
     { vals = dalib_malloc ((*no)*(*size), "dalib_secarray_get_data");
       if (copy_flag) dalib_ddt_pack (vals, ddt);
     }

   dalib_ddt_free (ddt);

   *data = vals;
   *is_new = (!is_contiguous);

   if (dalib_is_array_info (dsp))
      FUNCTION(dalib_section_free) (&section_dsp);

} /* dalib_secarray_get_data */

/**************************************************************************
*                                                                         *
*   int dalib_secarray_is_mapped (section_info *section_id)               *
*                                                                         *
**************************************************************************/

int dalib_secarray_is_mapped (section_id)

section_info *section_id;

{ array_info  id;
 
  if (section_id == (section_info *) 0)
 
     { dalib_internal_error ("secarray_is_mapped: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*section_id))
 
     id = (array_info) *section_id;
 
   else if (dalib_is_section_info (*section_id))
 
     id = (*section_id)->array_id;
 
   else
 
     { dalib_internal_error ("secarray_is_mapped: no array/section info");
       dalib_stop ();
     }

   if (id->AlignInfo != NO_ALIGNMENT) return (1);
   if (id->DistributeInfo != NO_DISTRIBUTION) return (1);
 
   return (0);

} /* dalib_secarray_is_mapped */

/**************************************************************************
*                                                                         *
**************************************************************************/

void dalib_secarray_shape (array_id, rank, shape)

section_info *array_id;    /* descriptor for   section/array */
int *rank;                 /* returns rank of  section/array */
int shape[];               /* returns shape of section/array */

{ array_info   id;
 
  if (array_id == (section_info *) 0)
 
     { dalib_internal_error ("shape: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*array_id))
 
     { int lb[7], ub[7], i;
 
       dalib_array_global_shape (*array_id, rank, lb, ub);
       for (i=0; i<*rank; i++)
          shape [i] = ub[i] - lb[i] + 1;

     }
 
   else if (dalib_is_section_info (*array_id))
 
     { int lb[7], ub[7], str[7], i;
 
       dalib_section_global_shape (*array_id, rank, lb, ub, str);
       for (i=0; i<*rank; i++)
          shape[i] = dalib_range_size (lb[i], ub[i], str[i]);

     }
 
   else
 
     { dalib_internal_error ("dalib_secarray_shape: no array/section info");
       dalib_stop ();
     }
 
} /* dalib_secarray_shape */
 
/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void FUNCTION(dalib_shape) (rank_data, array_data, rank_dsp, array_id)

int rank_data [];
char array_data [];
section_info *rank_dsp, *array_id;

{ array_info   id;
 
  if (array_id == (section_info *) 0)
 
     { dalib_internal_error ("shape: no descriptor");
       dalib_stop();
     }
 
  if (dalib_is_array_info (*array_id))
 
     { int shape[7], lb[7], ub[7], i, rank;
 
       dalib_array_global_shape (*array_id, &rank, lb, ub);
       for (i=0; i<rank; i++)
          shape [i] = ub[i] - lb[i] + 1;

       dalib_replicate_out (shape, sizeof(int)*rank, rank_data, rank_dsp);
     }
 
   else if (dalib_is_section_info (*array_id))
 
     { int shape[7], lb[7], ub[7], str[7], i, rank;
 
       dalib_section_global_shape (*array_id, &rank, lb, ub, str);
       for (i=0; i<rank; i++)
          shape[i] = dalib_range_size (lb[i], ub[i], str[i]);

       dalib_replicate_out (shape, sizeof(int)*rank, rank_data, rank_dsp);
     }
 
   else
 
     { dalib_internal_error ("shape: no array/section info");
       dalib_stop ();
     }
 
} /* dalib_shape */
 
