/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Oct 93                                                   *
*  Last Update : Feb 96                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  Module      : schedule                                                 *
*                                                                         *
*  Function    : Operations for structured schedules                      *
*                                                                         *
*   int dalib_new_schedule (int send_NP, int recv_NP)                     *
*                                                                         *
*      - returns integer sid for a new structured schedule                *
*                                                                         *
*   void dalib_schedule_send (int sid, int pid, int low, high, stride)    *
*   void dalib_schedule_recv (int sid, int pid, int low, high, stride)    *
*                                                                         *
*      - defines the structured schedule                                  *
*                                                                         *
*   for query and release the structured schedule                         *
*                                                                         *
*   void dalib_schedule_psends (int sid, int *p_low, int *p_high)         *
*   void dalib_schedule_precvs (int sid, int *p_low, int *p_high)         *
*                                                                         *
*      - processors that are used for the movement                        *
*                                                                         *
*   int dalib_schedule_nsends (int sid, int p_id)                         *
*   int dalib_schedule_nrecvs (int sid, int p_id)                         *
*                                                                         *
*      - number of slices to be sent/recv from a certain processor        *
*                                                                         *
*   void dalib_schedule_send_range (int sid, int pid, int n,              *
*                                   int *low, *high, *stride, *size)      *
*                                                                         *
*   void dalib_schedule_recv_range (int sid, int pid, int n,              *
*                                   int *low, *high, *stride, *size)      *
*                                                                         *
*      - asking for the section to send or receive from proc *pid         *
*      - *low:*high:*stride is the section with *size elements            *
*                                                                         *
*   void dalib_schedule_send1 (int sid, int *low,                         *
*                              int *high, int *stride, int *n)            *
*                                                                         *
*   void dalib_schedule_recv1 (int sid, int *low,                         *
*                              int *high, int *stride, int *n)            *
*                                                                         *
*   void dalib_free_schedule (int sid)                                    *
*                                                                         *
**************************************************************************/

# undef DEBUG
# define CHECK

# include "dalib.h"
# include <stdio.h>

     /*******************************************************
     *                                                      *
     *  D A T A    S T R U C T U R E S                      *
     *                                                      *
     *  F O R   S C H E D U L E S                           *
     *                                                      *
     *******************************************************/

typedef struct

  { int kind;   /* 0 : not defined, 1 : subsection, 2 : indirect */

    int low, high, stride;   /* valid for kind 1 */

    int size;
    int *indexes;            /* valid for kind 2 */

  } subsection_info;

typedef struct

  { int send_low, send_high, send_NP;
    int recv_low, recv_high, recv_NP;

    int no_send_indexes, no_recv_indexes;

    int *send_indexes, *recv_indexes;

    subsection_info *send_sections, *recv_sections;

  } structured_schedule;

static structured_schedule *global_schedule [10];

static int schedule_counter = 0;

          /***************************************
          *                                      *
          *  ERROR HANDLING                      *
          *                                      *
          ***************************************/

#ifdef CHECK
void dalib_schedule_check_nr (nr, sname)
int nr;
char sname[];

{ if ((nr < 0) || (nr >= schedule_counter))
    { dalib_internal_error ("illegal structured schedule");
      printf ("dalib_schedule_%s, %d is illegal schedule\n",
               sname, nr);
      dalib_stop ();
    }
} /* dalib_schedule_check_nr */
#endif

          /**********************************************
          *                                             *
          *  dalib_new_schedule (send_NP, recv_NP)      *
          *                                             *
          *  - create a structured schedule for         *
          *    sending to receiving processors          *
          *                                             *
          **********************************************/

int dalib_new_schedule (send_NP, recv_NP)

int send_NP, recv_NP;

{ structured_schedule *s;
  int i;

  s = (structured_schedule*) dalib_malloc (sizeof (structured_schedule),
                                           "dalib_new_schedule");

  s->send_low  = send_NP+1;
  s->send_high = 0;
  s->send_NP   = send_NP;
  s->recv_low  = recv_NP + 1;
  s->recv_high = 0;
  s->recv_NP   = recv_NP;

  s->no_send_indexes = 0;
  s->no_recv_indexes = 0;

  s->send_sections = (subsection_info*) 
       dalib_malloc (sizeof (subsection_info) * send_NP, "dalib_new_schedule");
  s->recv_sections = (subsection_info*) 
       dalib_malloc (sizeof (subsection_info) * recv_NP, "dalib_new_schedule");

  for (i=0; i<send_NP; i++)
    s->send_sections[i].kind = 0;
     
  for (i=0; i<recv_NP; i++)
    s->recv_sections[i].kind = 0;

  global_schedule[schedule_counter] = s;

  return (schedule_counter++);
     
} /* dalib_new_schedule */

     /********************************************************
     *                                                       *
     *  dalib_schedule_send (nr, pid, low, high, stride)     *
     *                                                       *
     *  - in schedule nr pid will send low:high:stride       *
     *                                                       *
     ********************************************************/

void dalib_schedule_send (nr, pid, low, high, stride)
int nr;
int low, high, stride;
{ subsection_info *si;
  structured_schedule *s;

#ifdef DEBUG
  printf ("%d: dalib_schedule_send, in schedule(%d) sends %d range %d:%d:%d\n",
           pcb.i, nr, pid, low, high, stride);
#endif

#ifdef CHECK
  dalib_schedule_check_nr (nr, "send");
#endif

  s = global_schedule[nr];

  if ((stride > 0) && (low > high)) return;
  if ((stride < 0) && (low < high)) return;
  si = s->send_sections + (pid-1);
  if (si->kind > 0)
     { fprintf (stderr,"Internal Error: dalib_schedule_send\n");
       exit(-1);
     }
  si->kind = 1;
  si->low = low;
  si->high = high;
  si->stride = stride;
  if (pid < s->send_low) s->send_low = pid;
  if (pid > s->send_high) s->send_high = pid;

} /* dalib_schedule_send */

void dalib_schedule_send_section (nr, pid, section)

int nr, pid;
int section[];

{ dalib_schedule_send (nr, pid, section[0], section[1], section[2]);

} /* dalib_schedule_send_section */

          /***************************************
          *                                      *
          *  dalib_schedule_recv                 *
          *                                      *
          ***************************************/

void dalib_schedule_recv (nr, pid, low, high, stride)
int nr;
int low, high, stride;
{ subsection_info *si;
  structured_schedule *s;

#ifdef DEBUG
  printf ("%d: dalib_schedule_recv, in schedule(%d) recvs %d range %d:%d:%d\n",
           pcb.i, nr, pid, low, high, stride);
#endif

  s = global_schedule[nr];

  if ((stride > 0) && (low > high)) return;
  if ((stride < 0) && (low < high)) return;
  si = s->recv_sections + (pid-1);
  if (si->kind)
     { fprintf (stderr,"Internal Error: dalib_schedule_recv\n");
       exit(-1);
     }
  si->kind = 1;
  si->low = low;
  si->high = high;
  si->stride = stride;
  if (pid < s->recv_low) s->recv_low = pid;
  if (pid > s->recv_high) s->recv_high = pid;

} /* dalib_schedule_recv */

void dalib_schedule_recv_section (nr, pid, section)

int nr, pid;
int section[];

{ dalib_schedule_recv (nr, pid, section[0], section[1], section[2]);

} /* dalib_schedule_send_section */

          /***************************************
          *                                      *
          *  dalib_print_schedule                *
          *                                      *
          ***************************************/

void dalib_print_schedule (nr)
int  nr;

{ int pid;
  structured_schedule *s;
  subsection_info     *info;
  int kind;
  int j;
 
  s = global_schedule[nr];

  for (pid = s->send_low; pid <= s->send_high; pid++)

    { info = &(s->send_sections[pid-1]);

      kind = info->kind;

      if (kind == 1)

         { printf ("%d: sending to %d section (%d:%d:%d)\n",
                    pcb.i, pid,
                    info->low,
                    info->high,
                    info->stride);
         }

      else if (kind == 2)

         { int j, size;
           int *indexes;

           size    = info->size;
           indexes = info->indexes;

           printf ("%d: sending to %d indexes ", pcb.i, pid);
           for (j=0; j<size; j++)
              printf ("%d ", indexes[j]);
           printf ("\n");

         }

    } /* for sending part */

  for (pid = s->recv_low; pid <= s->recv_high; pid++)

    { info = &(s->recv_sections[pid-1]);

      kind = info->kind;

      if (kind == 1)

         { printf ("%d: receiving from %d section (%d:%d:%d)\n",
                    pcb.i, pid, 
                    info->low,
                    info->high,
                    info->stride);
         }

      else if (kind == 2)

         { int j, size;
           int *indexes;

           size    = info->size;
           indexes = info->indexes;

           printf ("%d: receiving from %d indexes ", pcb.i, pid);
           for (j=0; j<size; j++)
              printf ("%d ", indexes[j]);
           printf ("\n");

         }

    } /* for receiving part */

} /* dalib_print_schedule */

          /***************************************
          *                                      *
          *  dalib_schedule_psends               *
          *                                      *
          ***************************************/

void dalib_schedule_psends (sid, p_low, p_high)
int sid, *p_low, *p_high;

{ structured_schedule *s;
 
#ifdef CHECK
  dalib_schedule_check_nr (sid, "psends");
#endif

  s = global_schedule[sid];
  *p_low = s->send_low;
  *p_high = s->send_high;
}

          /***************************************
          *                                      *
          *  dalib_schedule_precvs               *
          *                                      *
          ***************************************/

void dalib_schedule_precvs (sid, p_low, p_high)
int sid, *p_low, *p_high;

{ structured_schedule *s;
 
#ifdef CHECK
  dalib_schedule_check_nr (sid, "precvs");
#endif

  s = global_schedule[sid];
  *p_low  = s->recv_low;
  *p_high = s->recv_high;
}

          /***************************************
          *                                      *
          *  dalib_schedule_nsends/nrecvs        *
          *                                      *
          ***************************************/

int dalib_schedule_nsends (sid, pid)
int sid, pid;
{
  return (1);
} /* dalib_schedule_nsends */

int dalib_schedule_nrecvs (sid, pid)
int sid, pid;
{
  return (1);
} /* dalib_schedule_nrecvs */

/*************************************************************************
*                                                                        *
*  void dalib_schedule_send_range (int sid, int pid,                     *
*                                  int *okay,                            *
*                                  int *low, *high, *stride)             *
*                                                                        *
*  - get the section to be sent to processor pid by schedule sid         *
*  - okay is set to 1 if it is really a range (0 otherwise)              *
*  - low:high:stride is the corresponding range                          *
*                                                                        *
*************************************************************************/

void dalib_schedule_send_range (sid, pid, okay, low, high, stride)

int sid, pid;
int *okay, *low, *high, *stride;

{ structured_schedule *s;
 
  int kind;

#ifdef CHECK
  dalib_schedule_check_nr (sid, "send_range");
#endif

  s = global_schedule[sid];
  
#ifdef CHECK
  if ((pid < 0) || (pid > s->send_NP))
    { dalib_internal_error ("schedule_send_range: illegal pid");
      dalib_stop ();
    }
#endif

  kind = s->send_sections[pid-1].kind;

  if (kind == 0)

    { *low    = 1;
      *high   = 0;
      *stride = 1;
      *okay   = 1;
      return;
    }

  if (kind == 1)

    { *low    = s->send_sections[pid-1].low;
      *high   = s->send_sections[pid-1].high;
      *stride = s->send_sections[pid-1].stride;
      *okay   = 1;
      return;
    }

  *okay = 0;

} /* dalib_schedule_send_range */

/*************************************************************************
*                                                                        *
*  void dalib_schedule_send_values (int sid, int pid,                    *
*                                                                        *
*************************************************************************/

void dalib_schedule_send_values (sid, pid, N, values)

int sid, pid;
int *N, **values;

{ structured_schedule *s;
 
  int kind;

#ifdef CHECK
  dalib_schedule_check_nr (sid, "send_values");
#endif

  s = global_schedule[sid];
  
  kind = s->send_sections[pid-1].kind;

  if (kind == 2)

    { *N      = s->send_sections[pid-1].size;
      *values = s->send_sections[pid-1].indexes;
      return;
    }

  dalib_internal_error ("schedule_send_values: not indirect");
  dalib_stop ();

} /* dalib_schedule_send_values */

/*************************************************************************
*                                                                        *
*  void dalib_schedule_recv_range (int sid, int pid,                     *
*                                  int *okay,                            *
*                                  int *low, *high, *stride)             *
*                                                                        *
*  - get the section to be received from processor pid by schedule sid   *
*  - okay is set to 1 if it is really a range (0 otherwise)              *
*  - low:high:stride is the corresponding range                          *
*                                                                        *
*************************************************************************/

void dalib_schedule_recv_range (sid, pid, okay, low, high, stride)

int sid, pid;
int *okay, *low, *high, *stride;

{ structured_schedule *s;
 
  int kind;

#ifdef CHECK
  dalib_schedule_check_nr (sid, "recv_range");
#endif

  s = global_schedule[sid];
  
#ifdef CHECK
  if ((pid < 0) || (pid > s->recv_NP))
    { dalib_internal_error ("schedule_recv_range: illegal pid");
      dalib_stop ();
    }
#endif

  kind = s->recv_sections[pid-1].kind;

  if (kind == 0)

    { *low    = 1;
      *high   = 0;
      *stride = 1;
      *okay   = 1;
      return;
    }

  if (kind == 1)

    { *low    = s->recv_sections[pid-1].low;
      *high   = s->recv_sections[pid-1].high;
      *stride = s->recv_sections[pid-1].stride;
      *okay   = 1;
      return;
    }

  *okay = 0;

} /* dalib_schedule_recv_range */

/*************************************************************************
*                                                                        *
*  void dalib_schedule_recv_values (int sid, int pid,                    *
*                                                                        *
*************************************************************************/

void dalib_schedule_recv_values (sid, pid, N, values)

int sid, pid;
int *N, **values;

{ structured_schedule *s;
 
  int kind;

#ifdef CHECK
  dalib_schedule_check_nr (sid, "recv_values");
#endif

  s = global_schedule[sid];
  
  kind = s->recv_sections[pid-1].kind;

  if (kind == 2)

    { *N      = s->recv_sections[pid-1].size;
      *values = s->recv_sections[pid-1].indexes;
      return;
    }

  dalib_internal_error ("schedule_recv_values: not indirect");
  dalib_stop ();

} /* dalib_schedule_recv_values */

/*******************************************************************
*                                                                  *
*  dalib_set_any_schedule                                          *
*                                                                  *
*    int send_size [send_NP]                                       *
*    int recv_size [recv_NP]                                       *
*                                                                  *
*******************************************************************/

void dalib_set_any_schedule (nr,
                             send_size, send_indexes,
                             recv_size, recv_indexes)

int nr;
int send_size[];
int send_indexes[];
int recv_size[];
int recv_indexes[];

{ structured_schedule *s;
  subsection_info *info;

  int *ptr;   /* used to traverse send_indexes and recv_indexes */

  int pid, size; 

  s = global_schedule[nr];

  s->send_low  = 1;
  s->send_high = s->send_NP;

  s->recv_low  = 1;
  s->recv_high = s->recv_NP;

  s->send_indexes = send_indexes;
  s->recv_indexes = recv_indexes;

  s->no_send_indexes = 0;
  s->no_recv_indexes = 0;

  ptr = send_indexes;

  for (pid = 0; pid < s->send_NP; pid++)

    { size = send_size[pid];

      s->no_send_indexes += size;

      if (size > 0)

        { info = s->send_sections + pid;

          info->kind = 2;
          info->size = size;
          info->indexes = ptr;

          ptr += size;
        }

    } /* for to send sending values */

  ptr = recv_indexes;

  for (pid = 0; pid < s->recv_NP; pid++)

    { size = recv_size[pid];

      s->no_recv_indexes += size;

      if (size > 0)

        { info = s->recv_sections + pid;

          info->kind = 2;
          info->size = size;
          info->indexes = ptr;

          ptr += size;
        }

    } /* for to send sending values */

  /* set send_indexes and recv_indexes globally in schedule so
     that it can be freed later                                 */

} /* dalib_set_any_schedule */

          /***************************************
          *                                      *
          *  dalib_free_schedule                 *
          *                                      *
          ***************************************/

void dalib_free_schedule (sid)
int sid;

{ structured_schedule *s;
  int size;

  s = global_schedule[sid];

  schedule_counter --;

  if (sid != schedule_counter)

    { fprintf (stderr,"ERROR: illegal release of schedule\n");
      fprintf (stderr,"       expected id = %d\n", schedule_counter);
      fprintf (stderr,"       was sche id = %d\n", sid);
      dalib_stop ();
    }

  size = s->no_send_indexes;
  if (size > 0) dalib_int_free (s->send_indexes, size);

  size = s->no_recv_indexes;
  if (size > 0) dalib_int_free (s->recv_indexes, size);

  size = sizeof(subsection_info);

  dalib_free (s->send_sections, size * s->send_NP);
  dalib_free (s->recv_sections, size * s->recv_NP);
  dalib_free (s, sizeof(structured_schedule));

} /* dalib_free_schedule */

