/**************************************************************************
*                                                                         *
*  Author      : Dr. Thomas Brandes, GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Mar 96                                                   *
*  Last Update : May 96                                                   *
*                                                                         *
*  This Module is only part of the DALIB                                  *
*                                                                         *
*  Module      : redist.m4                                                *
*                                                                         *
*  Function    :                                                          *
*                                                                         *
*  Changes:                                                               *
*                                                                         *
*   05/1996 : problem with block -> replicated if one processor           *
*             contains all the date (use_dummy_data must not give true)   *
*                                                                         *
**************************************************************************/

#undef DEBUG

#include "dalib.h"

/*******************************************************************
*                                                                  *
*  PREDICATE dalib_has_replication (target_dsp, source_dsp)        *
*                                                                  *
*    - returns true if assignment target = source requires         *
*      at least the broadcast of some data                         *
*    - broadcast of source might also require accepting msg        *
*                                                                  *
*******************************************************************/

int dalib_has_replication (target_dsp, source_dsp)

array_info target_dsp, source_dsp;

{ /* case 1 : if target and source are not replicated at all we must
              certainly not copy any data                             */

  if(  (!dalib_is_replicated (source_dsp))
    && (!dalib_is_replicated (target_dsp)) )  return (0);
 
  /* case 2 : if source is full replicated, this processor is not
              involved in any communication                           */

  if (   (source_dsp->AlignInfo == NO_ALIGNMENT)
      && (source_dsp->DistributeInfo == NO_DISTRIBUTION) )

     return (0);   /* source is full replicated */

  return (1);   /* make worse assumption */

} /* dalib_has_replication */

/**********************************************************************
*                                                                     *
*  int dalib_inherit_mapping (array_info dummy_dsp, actual_dsp)       *
*                                                                     *
*  dummy  : !HPF$ DISTRIBUTE (BLOCK(),*) ONTO *                       *
*                                                                     *
*  - dummy inherits mapping of actual (returns 1)                     *
*  - choose an arbitrary mapping for dummy (returns 0)                *
*                                                                     *
*  - returns 1 if mapping has been inherited (no redistribution)      *
*                                                                     *
**********************************************************************/

int dalib_inherit_mapping (dummy_dsp, actual_dsp)

array_info dummy_dsp, actual_dsp;

{ int done;

  if (dalib_is_map_specialization (actual_dsp, dummy_dsp))

     { /* actual matches dummy, so we inherit everything for the dummy */

#ifdef DEBUG
  printf ("%d: actual %d is specialization of dummy %d (inherit mapping)\n",
           pcb.i, actual_dsp, dummy_dsp);
#endif

       dalib_array_inherit_mapping (dummy_dsp, actual_dsp);
       dalib_get_actual_info (dummy_dsp, actual_dsp);
       done = 1;

     }

   else

     { /* choose an arbitrary full distribution for the dummy */

#ifdef DEBUG
  printf ("%d: actual %d is not specialization of dummy %d (choose mapping)\n",
           pcb.i, actual_dsp, dummy_dsp);
#endif

       dalib_array_make_full_mapping (dummy_dsp);
       done = 0;

     }

    /* now we can compute the local sizes of dummy_dsp */

   if (dummy_dsp->DistributeInfo != NO_DISTRIBUTION)
      dalib_dist_local_sizes (dummy_dsp);
    else if (dummy_dsp->AlignInfo != NO_ALIGNMENT)
      dalib_align_local_sizes (dummy_dsp);
    else
      dalib_full_local_sizes (dummy_dsp);

   return (done);

} /* dalib_inherit_mapping */

/*******************************************************************
*                                                                  *
*                                                                  *
*                                                                  *
*******************************************************************/

void dalib_local_copy_in (dummy_dsp, actual_dsp, copy_flag)

array_info dummy_dsp;
array_info actual_dsp;
int        copy_flag;

{ DimInfo *actual_dims;
  DimInfo *dummy_dims;

  int i, rank;

#ifdef DEBUG
  printf ("%d: dalib_local_copy_in (dummy=%d,actual=%d,copy=%d)\n",
          pcb.i, dummy_dsp, actual_dsp, copy_flag);
  dalib_print_array_info (actual_dsp);
#endif

  if (dalib_is_array_info (actual_dsp))

     { /* inherit the local sizes as global sizes */

       actual_dims = actual_dsp->dimensions;
       dummy_dims  = dummy_dsp->dimensions;

       rank = dummy_dsp->rank;

       for (i=0; i<rank; i++)

         { dummy_dims->global_size [0] = actual_dims->local_size[0];
           dummy_dims->global_size [1] = actual_dims->local_size[1];

           dummy_dims->shadow     [0] = actual_dims->shadow[0];
           dummy_dims->shadow     [1] = actual_dims->shadow[1];

           dummy_dims->local_size  [0] = dummy_dims->global_size[0];
           dummy_dims->local_size  [1] = dummy_dims->global_size[1];
           dummy_dims->local_size  [2] = 1;

           dummy_dims->map_flag = 0;

           actual_dims++; dummy_dims++;

         }

       dummy_dsp->f_data = actual_dsp->f_data;
       dummy_dsp->dsp_status_flag = DSP_PTR_DATA;

#ifdef DEBUG
       printf ("%d: dalib_local_copy_in, local dsp = \n", pcb.i);
       dalib_print_array_info (dummy_dsp);
#endif

     }

    else if (dalib_is_section_info (actual_dsp))

     { section_info actual_section;
       SecDimInfo *actual_dims;

       dalib_internal_error (
              "only full arrays can be passed to LOCAL routines");
       dalib_stop ();

       actual_section = (section_info) actual_dsp;

       actual_dims = actual_section->dimensions;
       dummy_dims  = actual_dsp->dimensions;

       rank = actual_section->array_id->rank;

       if (dalib_section_rank (actual_section) != dummy_dsp->rank)

           { dalib_internal_error ("dalib_local_copy_in");
             dalib_stop ();
           }

       for (i=0; i<rank; i++)

         { dummy_dims->global_size [0] = actual_dims->local_range[0];
           dummy_dims->global_size [1] = actual_dims->local_range[1];

           dummy_dims->local_size  [0] = dummy_dims->global_size[0];
           dummy_dims->local_size  [1] = dummy_dims->global_size[1];
           dummy_dims->local_size  [2] = 1;

           dummy_dims->map_flag = 0;
         }

     }

} /* dalib_local_copy_in */

void dalib_local_copy_out (dummy_dsp, actual_dsp, copy_flag)

array_info dummy_dsp;
array_info actual_dsp;
int        copy_flag;

{ /* currently nothing to do as data is always inherited here */

#ifdef DEBUG
  printf ("%d: dalib_local_copy_out (dummy=%d,actual=%d,copy=%d)\n",
          pcb.i, dummy_dsp, actual_dsp, copy_flag);
#endif

} /* dalib_local_copy_out */

/*******************************************************************
*                                                                  *
*  FUNCTION(dalib_redistribute) (array_info *new_dsp,              *
*                                array_info *old_dsp,              *
*                                void       *f_ptr)                *
*                                                                  *
*    call DALIB_array_create_copy (B_NDSP, B_DSP)                  *
*    ...                                                           *
*    ...   ! define new distribution for B_DSP                     *
*    ...                                                           *
*    call DALIB_redistribute (B_DSP,B_NDSP)                        *
*                                                                  *
*    - old_dsp has old mapping, new_dsp contains new mapping       *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_redistribute) (new_dsp, old_dsp, f_ptr)

array_info *new_dsp, *old_dsp;
void *f_ptr;

{ array_info descriptor;

  /* attention: can be that old dsp has no data allocated */

  descriptor = *new_dsp;

  if (!dalib_array_has_data (*old_dsp))

     { /* NO data is given for the ARRAY, makes redistribute simple */

#ifdef DEBUG
       printf ("%d: redistribute of non allocated array\n", pcb.i);
#endif

       descriptor->n_data          = 0;
       descriptor->f_data          = NO_DATA;
       descriptor->c_data          = NO_DATA;
       descriptor->dsp_status_flag = descriptor->dsp_status_flag;

       /* old descriptor will be freed */

       dalib_free_descriptor (*old_dsp); 

       return;

     }

  if (descriptor->DistributeInfo != NO_DISTRIBUTION)
     dalib_dist_local_sizes (descriptor);
   else if (descriptor->AlignInfo != NO_ALIGNMENT)
     dalib_align_local_sizes (descriptor);
   else
     dalib_full_local_sizes (descriptor);

#ifdef DEBUG
   printf ("redistribute: print new and old descriptor\n");
   dalib_print_array_info (descriptor);
   dalib_print_array_info (*old_dsp);
#endif

  if (dalib_use_actual_data (*new_dsp, *old_dsp))
 
     { /* copy just the data pointer */

       Shared shared_ptr;
       Remote remote_ptr;

#ifdef DEBUG
       printf ("%d: redistribute uses existing data\n", pcb.i);
#endif

       /* take over all data information */

       (*new_dsp)->n_data          = (*old_dsp)->n_data;
       (*new_dsp)->c_data          = (*old_dsp)->c_data;
       (*new_dsp)->f_data          = (*old_dsp)->f_data;
       (*new_dsp)->dsp_status_flag = (*old_dsp)->dsp_status_flag;

       if (dalib_has_replication (*new_dsp, *old_dsp))

          FUNCTION(dalib_assign) (new_dsp, old_dsp);

       /* switch the shared info between new and old descriptor */

       shared_ptr = (*new_dsp)->SharedInfo;
       (*new_dsp)->SharedInfo = (*old_dsp)->SharedInfo;
       (*old_dsp)->SharedInfo = shared_ptr;

       /* switch the remote info between new and old descriptor */

       remote_ptr = (*new_dsp)->RemoteInfo;
       (*new_dsp)->RemoteInfo = (*old_dsp)->RemoteInfo;
       (*old_dsp)->RemoteInfo = remote_ptr;

       /* old descriptor has no longer own data */

       (*old_dsp)->n_data = 0;
       (*old_dsp)->c_data = 0;
       (*old_dsp)->dsp_status_flag = DSP_PTR_DATA;

       /* free the old descriptor               */

       dalib_free_descriptor (*old_dsp);  

     }

   else

     {  /* new descriptor, so new data is needed */
 
#ifdef DEBUG
       printf ("%d: redistribute allocates new data\n", pcb.i);
#endif

        /* allocate new memory from HEAP, align it to the old data */

        dalib_array_full_allocate (*new_dsp, f_ptr);

        FUNCTION(dalib_assign) (new_dsp, old_dsp);

        /* free the old descriptor, with all data */

        dalib_free_descriptor (*old_dsp);

     }
 
}  /* dalib_redistribute */

/*******************************************************************
*                                                                  *
*  FORTRAN Interface                                               *
*                                                                  *
*  FUNCTION(dalib_local_create) (local_dsp, global_dsp, pack_flag) *
*                                                                  *
*******************************************************************/

void FUNCTION(dalib_local_create) (local_dsp, global_dsp, pack_flag)

array_info *local_dsp, *global_dsp;
int        *pack_flag;

{ array_info  global, local;

  DimInfo *global_dims;
  DimInfo *local_dims;

  int i, rank;

  if (dalib_is_section_info (*global_dsp))

     { dalib_internal_error ("local_create: cannot localize section");
       dalib_stop ();
     }

   else if (dalib_is_array_info (*global_dsp))

     {  global = *global_dsp;

     }  /* descriptor was for a full array */

    else

     { dalib_internal_error ("local_create: not array/section");
       dalib_stop ();
     }

   local = dalib_array_copy_dsp (global);
   
   local->GlobalInfo = global;

   global_dims = global->dimensions;
   local_dims  = local->dimensions;

   rank = global->rank;

   for (i=0; i<rank; i++)

     { local_dims->global_size [0] = global_dims->local_size[0];
       local_dims->global_size [1] = global_dims->local_size[1];

       local_dims->shadow     [0] = global_dims->shadow[0];
       local_dims->shadow     [1] = global_dims->shadow[1];

       local_dims->local_size  [0] = local_dims->global_size[0];
       local_dims->local_size  [1] = local_dims->global_size[1];
       local_dims->local_size  [2] = 1;

       local_dims->map_flag = 0;

       global_dims++; local_dims++;

     }

   local->f_data = global->f_data;
   local->dsp_status_flag = DSP_PTR_DATA;

   *local_dsp = local;

} /* dalib_local_create */

void FUNCTION(dalib_local_free) (local_dsp)

array_info *local_dsp;

{ 

   dalib_free_descriptor (*local_dsp);  

} /* dalib_local_free */

