/**************************************************************************
*                                                                         *
*  Author      : Resi Hoever-Klier,  GMD, SCAI.LAB                        *
*  Copyright   : GMD St. Augustin, Germany                                *
*  Date        : Oct 98                                                   *
*  Last Update : Nov 98                                                   *
*                                                                         *
*  This Module is part of the DALIB                                       *
*                                                                         *
*  MODULE : mpmd.m4                                                       *
*                                                                         *
*  Function: MPMD tasking                                                 *
*                                                                         *
*    int dalib_mpmd_init ()                                               *
*                                                                         *
*       - initializes mpmd context if different program names exist       *
*       - returns 1 if MPMD modus is used                                 *
*                                                                         *
**************************************************************************/

#include <stdio.h> 
#include "dalib.h"

#undef DEBUG
#define CHECK

/**************************************************************************
*                                                                         *
*  Global data structure for string memory containig all program names    *
*                                                                         *
*   - string_mem will have size string_mem_size                           *
*                                                                         *
**************************************************************************/

#define STRING_MAX_SIZE   1024    /* default value */

static char *string_mem;
static int  string_mem_size   = 0;
static int  string_mem_offset = 0;

static int  no_tasks        = 0;
static char *task_names [MAXP];
static int  task_sizes  [MAXP];

/**************************************************************************
*                                                                         *
*  void dalib_get_all_prg_names (char *program_names [])                  *
*                                                                         *
*    - routine will allocate string_mem corresponding sizes of all names  *
*                                                                         *
*    out:   program_names[i] -> name of code running on processor i+1     *
*                                                                         *
**************************************************************************/

void dalib_get_all_prg_names (program_names)

char *program_names [];

{ int my_size;
  int all_sizes [MAXP];
  int total_size;

  int offset;

  int group_id;
  int group_pos;
  int group_size;

  int i;

  group_id   = dalib_group_all ();
  group_pos  = pcb.i;
  group_size = pcb.p;

  my_size = strlen (dalib_program_name) + 1;

  all_sizes [group_pos-1] = my_size;
  dalib_group_concat (all_sizes, sizeof(int), group_id);

  total_size = 0;
  for (i=0; i<group_size; i++) total_size += all_sizes[i];

#ifdef DEBUG
  printf ("%d: total bytes for program names is %d\n", pcb.i, total_size);
#endif

  /* now allocate string memory for all program names */

  string_mem_size = total_size;
  string_mem = (char *) dalib_malloc (total_size, "mpmd_prg_names");

  /* now broadcast of all strings */

  offset = 0;

  for (i=1; i<=group_size; i++)

     { int j, length;

       length = all_sizes [i-1];

       if (i == group_pos)

          { /* put my string into the string memory */

            for (j=0; j<length; j++) 

               string_mem[offset+j] = dalib_program_name[j];

#ifdef DEBUG
            printf ("%d: set my name %s at offset %d\n",
                     pcb.i, dalib_program_name, offset);
#endif

          }

       dalib_group_bcast (string_mem+offset, length, i, group_id);

       program_names[i-1] = string_mem + offset;

       offset += length;

     }

} /* dalib_get_all_pgr_names */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_bcast_task_info ()

{ int group_id;
  int i;
  int total_size;

  group_id = dalib_group_all();

  /* broadcast the information read in by the task file :

     string_mem_offset : number of bytes used for task ids
     string_mem        : task names                               */

  dalib_group_bcast (&string_mem_offset, sizeof (int), 1, group_id);

  if (string_mem_size == 0)

     { string_mem_size= string_mem_offset;
       string_mem = (char *) dalib_malloc (string_mem_size,
                                           "mpmd_prg_names");
     }

  dalib_group_bcast (string_mem, string_mem_offset, 1, group_id);

  dalib_group_bcast (task_sizes, no_tasks * sizeof (int), 1, group_id);

  /* sum up the task sizes and compare with the number of processors */

  total_size = 0;

  for (i=0; i < no_tasks; i++) total_size += task_sizes[i];

  if (total_size != pcb.p)

    { printf ("number of processors = %d, but %d expected by TASK_FILE\n",
               pcb.p, total_size);
      dalib_stop ();
    }

} /* dalib_bcast_task_info */

/**************************************************************************
*                                                                         *
*                                                                         *
*                                                                         *
**************************************************************************/

void dalib_set_task_names (no_tasks, task_names, string_mem)

int no_tasks;
char *task_names[];
char *string_mem;

{ int task;
  char *string_ptr;

  string_ptr = string_mem;

  for (task = 0; task < no_tasks; task++)

    { task_names[task] = string_ptr;
      while (*string_ptr != '\0') string_ptr++;
      string_ptr ++;  /* points now to the next string */
    }

} /* dalib_set_task_names */

/**************************************************************************
*                                                                         *
*  int dalib_set_task_ids (char *program_names [],                        *
*                          int  task_size[], int task_ids[])              *
*                                                                         *
*   in :   program_names[i] -> name of code running on processor i+1      *
*                                                                         *
*   returns number of different tasks (no_tasks)                          *
*                                                                         *
*   out:   task_size[i]  :   0 <= i < no_tasks   number of processors     *
*                                                for this task            *
*                                                                         *
*          task_ids [i]  :   0 <= i < pcb.p      sorted ids corresponding *
*                                                the tasks                *
*                                                                         *
*   Example: pcb.p = 8                                                    *
*                                                                         *
*     program_names[0] = Stage1                                           *
*     program_names[1] = Stage2a                                          *
*     program_names[2] = Stage2b                                          *
*     program_names[3] = Stage2a                                          *
*     program_names[4] = Stage2b                                          *
*     program_names[5] = Stage2a                                          *
*     program_names[6] = Stage2b                                          *
*     program_names[7] = Stage3                                           *
*                                                                         *
*   ->  no_tasks = 4,  task_size[0] = 1                                   *
*                      task_size[1] = 3                                   *
*                      task_size[2] = 3                                   *
*                      task_size[3] = 1                                   *
*                                                                         *
*     task_ids :  1 | 2 4 6 | 3 5 7 | 8      (numbering from 1 to pcb.p)  *
*                                                                         *
**************************************************************************/

int dalib_set_task_ids (program_names, task_size, task_ids)

char *program_names [];
int task_ids[];
int task_size [];

{ int  no_tasks;
  int  task_offset [MAXP];

  int  pid_in_task [MAXP];
  int  pid;
  int  i, j;

  no_tasks = 0;

  for (pid = 0; pid < pcb.p; pid++)

    { int task_id;
      int found;
      char *pname;

      pname = program_names[pid];

      task_id = 0;
      found   = 0;

      while ((!found) && (task_id < no_tasks))

         { if (strcmp (pname, task_names[task_id]) == 0) found = 1;
           if (!found) task_id += 1;
         }

      if (task_id >= no_tasks)

         { /* is a new task */

#ifdef DEBUG
           if (pcb.i == 1)
              printf ("%s becomes new task %d\n", pname, no_tasks+1);
#endif

           task_names [no_tasks] = pname;
           task_size  [no_tasks] = 0;
           no_tasks += 1;

         }

      pid_in_task [pid] = task_id;
      task_size [task_id] += 1;

#ifdef DEBUG
      if (pcb.i == 1)
         printf ("%d belongs to task %d (%s), size now %d\n",
                 pid+1, task_id+1, task_names[task_id], task_size[task_id]);
#endif

    }

   task_offset [0] = 0;
   for (i=1; i<no_tasks; i++)
      task_offset[i] = task_offset[i-1] + task_size[i-1];

   /* now sort the pids */

   for (pid = 0; pid < pcb.p; pid++)

      { int task_id;
        task_id = pid_in_task[pid];
        task_ids[task_offset[task_id]++] = pid+1;
      }

   return no_tasks;

} /* dalib_set_task_ids */

/**************************************************************************
*                                                                         *
*   int dalib_set_default_task_ids (int no_tasks, int task_sizes[],       *
*                                   int task_ids[]                  )     *
*                                                                         *
*   in :   no_tasks                                                       *
*          task_sizes [no_tasks]   number of procs in one task            *
*                                                                         *
*   ->  no_tasks = 4,  task_sizes[0] = 1                                  *
*                      task_sizes[1] = 3                                  *
*                      task_sizes[2] = 3                                  *
*                      task_sizes[3] = 1                                  *
*                                                                         *
*     task_ids :  1 | 2 3 4 | 5 6 7 | 8      (numbering from 1 to pcb.p)  *
*                                                                         *
**************************************************************************/

static int dalib_set_default_task_ids (no_tasks, task_sizes, task_ids)

int no_tasks;
int task_sizes [];
int task_ids[];

{ /* we have read the task file, so we assume that all tasks 
          are defined in the correct order                         */

   int *task_id_ptr;
   int task;
   int pos;
   int j;
   int my_task_id;

   pos = 0;
   task_id_ptr = task_ids;

   for (task=1; task <= no_tasks; task++) 

     { int no_procs;

       no_procs = task_sizes[task-1];

       for (j=0; j<no_procs; j++)

          { task_id_ptr [j] = pos + 1; 
            pos++; 
            if (pos == pcb.i) my_task_id = task;
          }

       task_id_ptr += no_procs;

     }

#ifdef DEBUG
   printf ("%d: program_name = %s, task_name = %s\n",
            pcb.i, dalib_program_name, task_names [my_task_id-1]);
#endif

   if (strcmp (dalib_program_name, task_names[my_task_id-1]) == 0) 

      return 0;   /* task name fits with my program name */

   printf ("%d: started wrong task, program_name = %s, task_name = %s\n",
            pcb.i, dalib_program_name, task_names [my_task_id-1]);

   return 1;   /* otherwise we have an error */

} /* dalib_set_default_task_ids */

/**************************************************************************
*                                                                         *
*  int dalib_print_tasks (int task_ids [])                                *
*                                                                         *
**************************************************************************/

static void dalib_print_tasks (no_tasks, task_names, task_sizes, task_ids)

int no_tasks;
char *task_names [];
int  task_sizes [];
int  task_ids [];

{ int task;
  int *task_id_ptr;

  task_id_ptr = task_ids;

  for (task = 1; task <= no_tasks; task ++)

    { int p, no_procs;

      no_procs = task_sizes[task-1];

      printf ("TASK %d (%s), size = %d : ", 
               task, task_names[task-1], no_procs);

      for (p=0; p<no_procs; p++) printf (" %d", task_id_ptr[p]);
      printf ("\n");

      task_id_ptr += no_procs;

    }

} /* dalib_print_tasks */

/**************************************************************************
*                                                                         *
*  int dalib_my_task_id (int no_tasks, int task_sizes[], int task_ids[])  *
*                                                                         *
*    - returns my own task id (1 <= id <= no_tasks)                       *
*    - 0 implies an error                                                 *
*                                                                         *
**************************************************************************/

static int dalib_my_task_id (no_tasks, task_sizes, task_ids)

int no_tasks;
int task_sizes [];
int task_ids [];

{ int i, j;
  int my_id;
  int *ids;

  ids = task_ids;  /* pointer for traversing */

  my_id = 0;

  for (i=0; i<no_tasks; i++)

    { int task_size;

      task_size = task_sizes[i];

      for (j=0; j<task_size; j++)

         if (ids[j] == pcb.i) my_id = i+1;

      ids += task_size;
    }

  return my_id;

} /* dalib_my_task_id */

/**************************************************************************
*                                                                         *
*  int dalib_verify_tasking (int no_tasks)                                *
*                                                                         *
**************************************************************************/

static int dalib_verify_tasking (my_task_id, no_tasks, task_names)

int  my_task_id;
int  no_tasks;
char *task_names[];

{ int errors [MAXP];
  int error;

  int NP, NId, group_id;

  error = 0;

  if ((my_task_id < 1) || (my_task_id > no_tasks))

     { error = 1;
       printf ("%d: belong to illegal task %d, (no_tasks = %d)\n",
               pcb.i, my_task_id, no_tasks);
     }

   else if (strcmp (dalib_program_name, task_names[my_task_id-1]) != 0)

     { error = 1;
       printf ("%d: started wrong task, program_name = %s, task_name = %s\n",
                pcb.i, dalib_program_name, task_names [my_task_id-1]);
     }

  NP = pcb.p; NId = pcb.i; group_id = dalib_group_all ();

  errors [NId-1] = error;
  dalib_group_concat (errors, sizeof(int), group_id);

  for (NId=0; NId < NP; NId++)

    if (errors[NId])

       { error = 1;
 
         if (pcb.i == 1)

             printf ("MPMD: error on processor %d\n", NId + 1);
       }

  return error;

} /* dalib_verify_tasking */

/**************************************************************************
*                                                                         *
*   void dalib_mpmd_tasking (int no_tasks, int task_sizes[],              *
*                                          int task_pids[]  )             *
*                                                                         *
*   - initializes hpf_tasking                                             *
*                                                                         *
*   in :   no_tasks      :   number of tasks                              *
*                                                                         *
*   in :   task_size[i]  :   0 <= i < no_tasks   number of processors     *
*                                                for this task            *
*                                                                         *
*   in :   task_ids [i]  :   0 <= i < pcb.p      sorted ids corresponding *
*                                                the tasks                *
*                                                                         *
*   - creates a new group for every task                                  *
*   - find my_group, the task I belong to                                 *
*   - define a topology for the group I belong to                         *
*   - push this topology as the new context where HPF task is executed    *
*                                                                         *
**************************************************************************/

void dalib_mpmd_tasking (no_tasks, task_sizes, task_pids)

int no_tasks;
int task_sizes [];
int task_pids  [];

{ int group_ids [MAXP];

  int my_group;
  int top_id;

  int *pids;
  int shape [1];

  int i;

  pids = task_pids;

  for (i=0; i<no_tasks; i++)

    { int task_size;
      int j;

      task_size = task_sizes[i];

      group_ids[i] = dalib_group_create (task_size, pids);
      
      /* check whether I belong also to this group */

      for (j=0; j<task_size; j++)
         if (pids[j] == pcb.i) my_group = group_ids[i];

      pids += task_size;
    }

  /* Now we are ready to start the tasking of different programs */

  FUNCTION(dalib_start_tasking) ();

  dalib_set_group_tasks (no_tasks, group_ids);

  /* define topology for my group and push it as the new context */

  shape [0] = dalib_group_size (my_group);
  top_id = dalib_top_make (my_group, 1, shape);

#ifdef DEBUG
  printf ("%d: start tasking in group %d (size = %d), top_id %d (size = %d)\n", 
           pcb.i, my_group, dalib_group_size (my_group),
                  top_id, dalib_top_size (top_id));
#endif

  dalib_push_top_context (top_id);

} /* dalib_mpmd_tasking */

/**************************************************************************
*                                                                         *
*  int dalib_mpmd_init (void)                                             *
*                                                                         *
*    - get the names and ids of all processes executed together           *
*    - generate array of mpmd-members                                     *
*                                                                         *
**************************************************************************/

int dalib_mpmd_init ()

{  char *program_names [MAXP];  /* dalib_program_name of every node */
   int  task_ids   [MAXP];      /* task to which node belongs       */

#ifdef DEBUG
  printf("%d: dalib_mpmd_init %s\n", pcb.i, dalib_program_name);
#endif

  if (pcb.mpmd_flag == 0)

     { /* we have not seen a task file until now, so exchange names */

       dalib_get_all_prg_names (program_names);

       no_tasks = dalib_set_task_ids (program_names, task_sizes, task_ids);

     }

    else

     { /* first processor has read the task file, so we assume that all tasks 
          are defined in the correct order                         

          no_tasks    is already defined
          task_sizes  is already defined 
          task_ids    must be set now                              */

       int error;
       int my_task_id;

       no_tasks = pcb.mpmd_flag;

       dalib_bcast_task_info ();

       dalib_set_task_names (no_tasks, task_names, string_mem);

       dalib_set_default_task_ids (no_tasks, task_sizes, task_ids);

       my_task_id = dalib_my_task_id (no_tasks, task_sizes, task_ids);

       error = dalib_verify_tasking (my_task_id, no_tasks, task_names);

       if (error) dalib_stop ();

     }

  /* now we can free the string memory */

  if (string_mem_size > 0)

      dalib_free (string_mem, string_mem_size);

  if (no_tasks == 1) return 0;

  if (pcb.i == 1)    /* only the first processor prints task info */

     dalib_print_tasks (no_tasks, task_names, task_sizes, task_ids);

  /* now define new groups, topologies and tasks */

  dalib_mpmd_tasking (no_tasks, task_sizes, task_ids);

  return 1;

} /* dalib_mpmd_init */

/*******************************************************************
*                                                                  *
*  void dalib_mpmd_exit ()                                         *
*                                                                  *
*   - return to the old context, so pop it and stop tasking        *
*                                                                  *
*******************************************************************/

void dalib_mpmd_exit ()

{ int i;

#ifdef DEBUG
  printf("%d: dalib_mpmd_exit\n", pcb.i);
#endif

  dalib_pop_context ();

  FUNCTION(dalib_stop_tasking) ();

} /* dalib_mpmd_exit */

/*******************************************************************
*                                                                  *
*   void dalib_eval_task_entry (char *buffer, int length)          *
*                                                                  *
*    in :  char buffer[length] is one line of task file            *
*                                                                  *
*    buffer = '<task_name>  <task_size>'                           *
*                                                                  *
*   - increase no_tasks globally                                   *
*   - sets task_names and task_sizes                               *
*                                                                  *
*******************************************************************/

static dalib_eval_task_entry (buffer, length)

char *buffer;
int  length;

{  int no_procs;
   char task_string [100];
   int n;
   int j, slen;
   char *string_ptr;

   buffer[length] = '\0';

   n=sscanf (buffer, "%s%d", &task_string, &no_procs);

   if (n != 2) return;

   /* so this seems to be a valid entry */

   slen = strlen (task_string) + 1;
   string_ptr = string_mem + string_mem_offset;

   for (j=0; j<slen; j++) string_ptr[j] = task_string[j];

   task_names [no_tasks] = string_ptr;
   task_sizes [no_tasks] = no_procs;

   string_mem_offset += slen;
   no_tasks ++;

}  /* dalib_eval_task_entry */

/*******************************************************************
*                                                                  *
*  int dalib_mpmd_read_taskfile (char *filename)                   *
*                                                                  *
*    - reads in the task file specification (env TASK_FILE)        *
*    - returns number of tasks to create                           *
*                                                                  *
*   sets globally:                                                 *
*                                                                  *
*    no_tasks  : number of task                                    *
*    task_names [i] : 0 <= i < no_tasks   name of the task         *
*    task_sizes [i] : 0 <= i < no_tasks   number of procs for task *
*                                                                  *
*   Attention: routine only called on first processor              *
*                                                                  *
*******************************************************************/

int dalib_mpmd_read_taskfile (filename)

char *filename;

{ FILE *task_file;

  char c;
  char buffer [180];
  int  buf_len = 0;
  int  i;

  no_tasks = 0;

  task_file = fopen (filename, "r");

  if (task_file == NULL)

    { printf ("TASK_FILE %s: could not open this file\n", filename);
      printf ("SPMD execution assumed");
      return no_tasks;
    }

  string_mem_size = STRING_MAX_SIZE;
  string_mem = (char *) dalib_malloc (STRING_MAX_SIZE, "mpmd_read_taskfile");
  string_mem_offset = 0;

  buf_len = 0; /* actual buffer length */

  while ((c = fgetc (task_file)) != EOF)

    { if (c == '\n')

         { /* new line, evaluate current line */

           dalib_eval_task_entry (buffer, buf_len);
           buf_len = 0;
         }

       else buffer [buf_len++] = c;
    } 

  if (buf_len > 0)

     dalib_eval_task_entry (buffer, buf_len);

  fclose (task_file);

#ifdef DEBUG
  printf ("read TASK_FILE %s, got %d tasks\n", filename, no_tasks);
  for (i=0; i < no_tasks; i++)
     printf ("   TASK %d, name = :%s:, no_procs = %d\n",
              i+1, task_names[i], task_sizes[i]);
#endif

  return no_tasks;

} /* mpmd_read_task_file */


void dalib_get_mpmd_task_info (task_id, name, procs)

int task_id;
char **name;
int  *procs;

{ *name =  task_names [task_id-1];
  *procs = task_sizes [task_id-1];

} /* dalib_get_mpmd_task_info */
